/*
    Copyright (C) 1996-1999  Ulric Eriksson <ulric@edu.stockholm.se>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston,
    MA 02111-1307, USA.
*/

/* filesel.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <dirent.h>

#include "../config.h"
#ifdef HAVE_FNMATCH
#include <fnmatch.h>
#else
#ifdef HAVE_GMATCH
#include <libgen.h>
#define fnmatch(a,b,c) (!gmatch(b,a))
#else
#define fnmatch(a,b,c) 0		/* match anything */
#endif
#endif

#include <sys/stat.h>
#include <sys/param.h>
#include <unistd.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include <X11/Xaw/Form.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/List.h>

#include "../common/cmalloc.h"
#include "../common/common.h"
#include "TextField.h"
#include "xcommon.h"
#include "dialogs.h"
#include "filesel.h"
#include "Rudegrid.h"
#include "Frame.h"

static Widget fsel_pshell, fsel_form, fsel_topframe, fsel_bottomframe,
	fsel_topbox, fsel_bottombox, fsel_filelabel2, fsel_dirlabel2,
	fsel_fileframe, fsel_dirframe, fsel_textframe,
	fsel_filelabel, fsel_filetext, fsel_fileviewport, fsel_filelist,
	fsel_formatlabel, fsel_formatbutton, fsel_formatmenu,
	fsel_dirlabel, fsel_dirbutton, fsel_dirmenu,
	fsel_dirviewport, fsel_dirlist,
	fsel_okbutton, fsel_cancelbutton, fsel_findbutton,
	fsel_helpbutton, fsel_extra[10], below;

static int nextra;

/* collect directory contents */

static int compar(const void *p, const void *q)
{
	return strcmp(*(const char **)p, *(const char **)q);
}

static int
getdirent(char ***ldirp, int *ndirp, char ***lfilep, int *nfilep,
	char *path, char *pattern)
{
	DIR *dird;
	struct dirent *dire;
	char fn[1024];
	struct stat buf;
	char **ldir = NULL, **lfile = NULL;
	int ndir = 0, mdir = 0, nfile = 0, mfile = 0;

	if ((dird = opendir(path)) == NULL) {
		fprintf(stderr, "Can't open %s\n", path);
		return 1;
	}
	while ((dire = readdir(dird))) {
		sprintf(fn, "%s/%s", path, dire->d_name);
		if ((stat(fn, &buf))) {
			fprintf(stderr, "Can't stat %s\n", fn);
			continue;
		}
		if (S_IFDIR & buf.st_mode) {
			if (ndir >= mdir)
				ldir = crealloc(ldir,
					(mdir += 256)*sizeof(char *));
			ldir[ndir++] = cstrdup(dire->d_name);
		}
		else if (S_IFREG & buf.st_mode && 
				!fnmatch(pattern, dire->d_name, 0)) {
			if (nfile >= mfile)
				lfile = crealloc(lfile,
					(mfile += 256)*sizeof(char *));
			lfile[nfile++] = cstrdup(dire->d_name);
		}
	}
	closedir(dird);

	qsort(ldir, ndir, sizeof(char *), compar);
	qsort(lfile, nfile, sizeof(char *), compar);

	*ndirp = ndir;
	*ldirp = ldir;
	*nfilep = nfile;
	*lfilep = lfile;

	return 0;
}

static void
freedirent(char **ldir, int ndir, char **lfile, int nfile)
{
	int i;

	for (i = 0; i < ndir; i++)
		cfree(ldir[i]);
	cfree(ldir);
	for (i = 0; i < nfile; i++)
		cfree(lfile[i]);
	cfree(lfile);
}

/* and that's all there is to it */


/* Foolproof default format */
static char *fileformats[] = {
	"All (*)",
	NULL};

static char **files, **dirs;
static int nfiles, ndirs;

static void make_files(String files[], Cardinal nfiles)
{
	static String foo[1] = {"<none>"};

	if (nfiles == 0)
		XawListChange(fsel_filelist,
			foo, 1, 0, True);
	else
		XawListChange(fsel_filelist,
			files, nfiles, 0, True);
}

static void make_dirs(String dirs[], Cardinal ndirs)
{
	XawListChange(fsel_dirlist,
		dirs, ndirs, 0, True);
}

static void make_dirmenu(char *);

static void fsel_scan(void)
{
	String string;
	char dir[1024];
	char pattern[1024];
	char *pst;

	if (files != NULL)
		freedirent(dirs, ndirs, files, nfiles);
	
	strcpy(dir, label_get(fsel_dirbutton));
	string = label_get(fsel_formatbutton);
	pst = strchr(string, '(');
	if (pst == NULL) {
		/* apparently no pattern here; use foolproof default */
		strcpy(pattern, "*");
	} else {
		strcpy(pattern, pst+1);
		if ((pst = strchr(pattern, ')')))
			*pst = '\0';
	}
	if (getdirent(&dirs, &ndirs, &files, &nfiles, dir, pattern)) {
		fprintf(stderr, "Grmbl. getdirent() failed\n");
	}
	make_files(files, nfiles);
	make_dirs(dirs, ndirs);
	make_dirmenu(dir);
}

static void format_select(Widget w, XtPointer client_data, XtPointer call_data)
{
	label_set(fsel_formatbutton, (String)client_data);
	fsel_scan();
}

static void file_select(Widget w, XtPointer client_data, XtPointer call_data)
{
	XawListReturnStruct *list_struct = (XawListReturnStruct *)call_data;
	String string = list_struct->string;

	XtVaSetValues(fsel_filetext,
		XtNstring, string, (char *)0);
	XawListUnhighlight(fsel_filelist);
	XawListUnhighlight(fsel_dirlist);
}

static void dir_select(Widget w, XtPointer client_data, XtPointer call_data)
{
	char path[1024];
	char newpath[1024];

	XawListReturnStruct *list_struct = (XawListReturnStruct *)call_data;
	String string = list_struct->string;

	sprintf(path, "%s/%s", label_get(fsel_dirbutton), string);

	if (!realpath(path, newpath))
		fprintf(stderr, "Couldn't realpath %s\n", path);

	label_set(fsel_dirbutton, newpath);

	XawListUnhighlight(fsel_filelist);
	XawListUnhighlight(fsel_dirlist);
	fsel_scan();
}

static void make_menu(char **menu_entry_names, Widget menu)
{
	int i;
	Widget entry;

	for (i = 0; menu_entry_names[i]; i++) {
		String item = menu_entry_names[i];

		if (item[0] == '-')	/* line pane */
			entry = XtCreateManagedWidget(item,
				smeLineObjectClass, menu, NULL, 0);
		else {
			entry = XtCreateManagedWidget(item,
				smeBSBObjectClass, menu, NULL, 0);
			XtAddCallback(entry,
				XtNcallback, format_select,
				menu_entry_names[i]);
		}
	}
}

static int nentry = 0;

static struct {
	Widget w;
	char *d;
} fsel_dir_entry[20];

static void change_dir(Widget w, XtPointer client_data, XtPointer call_data)
{
	char path[1024];
	char newpath[1024];

	strcpy(path, (char *)client_data);

	if (!realpath(path, newpath))
		fprintf(stderr, "Couldn't realpath %s\n", path);

	label_set(fsel_dirbutton, newpath);

	fsel_scan();
}

static void make_dirmenu(char *dir)
{
        int i;
	char *p, *b, *item;

	b = cstrdup(dir);
	for (i = 0; i < nentry; i++) {
		if (fsel_dir_entry[i].w != None) {
			XtDestroyWidget(fsel_dir_entry[i].w);
			cfree(fsel_dir_entry[i].d);
		}
	}
	nentry = 0;

	XtVaSetValues(fsel_dirbutton,
		XtNwidth, 200, (char *)0);
	if (fsel_dirmenu != None)
		XtDestroyWidget(fsel_dirmenu);
	fsel_dirmenu = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, fsel_dirbutton,
		(char *)0);

	while ((p = strrchr(b, '/'))) {
		*p = '\0';
		if (p == b)	/* root dir */
			item = cstrdup("/");
		else
			item = cstrdup(b);
		fsel_dir_entry[nentry].d = item;

		fsel_dir_entry[nentry].w = XtCreateManagedWidget(item,
			smeBSBObjectClass, fsel_dirmenu, NULL, 0);
		XtAddCallback(fsel_dir_entry[nentry].w,
			XtNcallback, change_dir,
			fsel_dir_entry[nentry].d);
		nentry++;
	}
}

static int status;

static void fsel_done(Widget w, XtPointer client_data, XtPointer call_data)
{
	char path[1024], newpath[1024];
	struct stat buf;
	String string;
	XtVaGetValues(fsel_filetext,
		XtNstring, &string, (char *)0);
	if (string[0] == '/') strcpy(path, string);
	else sprintf(path, "%s/%s", label_get(fsel_dirbutton), string);
	realpath(path, newpath);
	if (!stat(newpath, &buf) &&
		(S_IFDIR & buf.st_mode)) {
		label_set(fsel_dirbutton, newpath);
		XtVaSetValues(fsel_filetext,
			XtNstring, "", (char *)0);
		fsel_scan();
	} else {
		XtPopdown(fsel_pshell);
		status = DONE;
        }
}

static void fsel_abort(Widget w, XtPointer client_data, XtPointer call_data)
{
	XtPopdown(fsel_pshell);
	status = ABORT;
}

static void fsel_done_action(Widget w, XEvent *event, String *params, Cardinal *n)
{
	fsel_done(w, NULL, NULL);
}

static void fsel_cancel_action(Widget w, XEvent *event,
		String *params, Cardinal *n)
{
	fsel_abort(w, NULL, NULL);
}

static XtActionsRec actions[] =
{
	{"fsel-done", fsel_done_action},
	{"fsel-cancel", fsel_cancel_action}
};

/* ---
*/
Widget add_button(char *name, char *label,
	void (*cb)(Widget, XtPointer, XtPointer), XtPointer data, Widget pw)
{
	below = XtVaCreateManagedWidget(name,
		commandWidgetClass, pw,
		XtNwidth, 80,
		(char *)0);
	label_set(below, label);
	XtVaSetValues(below, XtNwidth, 80, (char *)0);
	if (cb) XtAddCallback(below, XtNcallback, cb, data);
	return below;
}

static void fsel_find(Widget w, XtPointer client_data, XtPointer call_data)
{
	char fn[1024];
	char cmd[1024];	
	char **files = NULL;
	int nfiles = 0, i;
	FILE *fp;

	fn[0] = '\0';
	if (dialog_input(fsel_pshell, "Find pattern:", fn, NULL)) {
		sprintf(cmd, "find %s -name '%s' -print",
			label_get(fsel_dirbutton), fn);
		if (!(fp = popen(cmd, "r"))) {
			errorbox(fsel_pshell, "Can't exec find");
			return;
		}
		while ((fgets(fn, sizeof fn, fp))) {
			chomp(fn);
			files = crealloc(files, (nfiles+1)*(sizeof(char *)));
			files[nfiles++] = cstrdup(fn);
		}
		pclose(fp);
		if (nfiles) {
			int n = listsel(fsel_pshell,
				"Pick one:", files, nfiles);
			if (n != -1) {
				char *p = strrchr(files[n], '/');
				if (p) {
					*p = '\0';
					XtVaSetValues(fsel_filetext,
						XtNstring, p+1, (char *)0);
					change_dir(w,
						(XtPointer)files[n], NULL);
				} else {
					XtVaSetValues(fsel_filetext,
						XtNstring, files[n], (char *)0);
				}
			}
		} else {
			errorbox(fsel_pshell, "No files found");
		}
		for (i = 0; i < nfiles; i++)
			cfree(files[i]);
		cfree(files);
	}
}

static void fsel_help(Widget w, XtPointer client_data, XtPointer call_data)
{
	char b[1024];

	sprintf(b, "file://localhost%s/xcommon/filesel.html", docdir);
	if (!fork()) {
		execlp(siaghelp, "Siaghelp", b, (char *)0);
		_exit(0);
	}
}

static Atom wm_delete_window;

static void fsel_init(Widget topLevel)
{
	XtAppContext app_context = XtWidgetToApplicationContext(topLevel);
	XtAppAddActions(app_context, actions, XtNumber(actions));

	fsel_pshell = XtVaCreatePopupShell("fsel_pshell",
		transientShellWidgetClass, topLevel,
		XtNtitle, translate("Select File"),
		(char *)0);

	fsel_form = XtCreateManagedWidget("fsel_form",
		rudegridWidgetClass, fsel_pshell, NULL, 0);

	fsel_topframe = XtVaCreateManagedWidget("fsel_topframe",
		frameWidgetClass, fsel_form, (char *)0);
	fsel_topbox = XtVaCreateManagedWidget("fsel_topbox",
		boxWidgetClass, fsel_topframe, (char *)0);
	fsel_bottomframe = XtVaCreateManagedWidget("fsel_bottomframe",
		frameWidgetClass, fsel_form, (char *)0);
	fsel_bottombox = XtVaCreateManagedWidget("fsel_bottombox",
		boxWidgetClass, fsel_bottomframe, (char *)0);
	fsel_filelabel = XtVaCreateManagedWidget("fsel_filelabel",
		labelWidgetClass, fsel_form, (char *)0);
	label_set(fsel_filelabel, "File Name:");

	fsel_textframe = XtVaCreateManagedWidget("fsel_textframe",
		frameWidgetClass, fsel_form,
		(char *)0);
	fsel_filetext = XtVaCreateManagedWidget("fsel_filetext",
		textfieldWidgetClass, fsel_textframe,
		(char *)0);
	XtVaSetValues(fsel_filetext,
		XtNstring, "", (char *)0);

	fsel_fileframe = XtVaCreateManagedWidget("fsel_fileframe",
		frameWidgetClass, fsel_form, (char *)0);
	fsel_filelabel2 = XtVaCreateManagedWidget("fsel_filelabel2",
		labelWidgetClass, fsel_fileframe, (char *)0);
	label_set(fsel_filelabel2, "Files");
	XtVaSetValues(fsel_fileframe,
		XtNtitle, fsel_filelabel2,
		(char *)0);
	fsel_fileviewport = XtVaCreateManagedWidget("fsel_fileviewport",
		viewportWidgetClass, fsel_fileframe, (char *)0);
	fsel_filelist = XtVaCreateManagedWidget("fsel_filelist",
		listWidgetClass, fsel_fileviewport, (char *)0);

	XtAddCallback(fsel_filelist, XtNcallback, file_select, NULL);

	fsel_formatlabel = XtVaCreateManagedWidget("fsel_formatlabel",
		labelWidgetClass, fsel_form, (char *)0);
	label_set(fsel_formatlabel, "Format:");

	fsel_formatbutton = XtVaCreateManagedWidget("fsel_formatbutton",
		menuButtonWidgetClass, fsel_form,
		XtNlabel, fileformats[0],
		(char *)0);

	fsel_dirlabel = XtVaCreateManagedWidget("fsel_dirlabel",
		labelWidgetClass, fsel_form, (char *)0);
	label_set(fsel_dirlabel, "Directory:");

	fsel_dirbutton = XtVaCreateManagedWidget("fsel_dirbutton",
		menuButtonWidgetClass, fsel_form,
		(char *)0);

	fsel_dirframe = XtVaCreateManagedWidget("fsel_dirframe",
		frameWidgetClass, fsel_form, (char *)0);
	fsel_dirlabel2 = XtVaCreateManagedWidget("fsel_dirlabel2",
		labelWidgetClass, fsel_dirframe, (char *)0);
	label_set(fsel_dirlabel2, "Directories");
	XtVaSetValues(fsel_dirframe,
		XtNtitle, fsel_dirlabel2,
		(char *)0);
	fsel_dirviewport = XtVaCreateManagedWidget("fsel_dirviewport",
		viewportWidgetClass, fsel_dirframe, (char *)0);

	fsel_dirlist = XtVaCreateManagedWidget("fsel_dirlist",
		listWidgetClass, fsel_dirviewport, (char *)0);

	XtAddCallback(fsel_dirlist, XtNcallback, dir_select, NULL);

	below = None;

	fsel_okbutton = add_button("fsel_okbutton", "OK",
		fsel_done, NULL, fsel_bottombox);
	fsel_cancelbutton = add_button("fsel_cancelbutton", "Cancel",
		fsel_abort, NULL, fsel_bottombox);
	fsel_findbutton = add_button("fsel_findbutton", "Find",
		fsel_find, NULL, fsel_topbox);
	fsel_helpbutton = add_button("fsel_helpbutton", "Help",
		fsel_help, NULL, fsel_bottombox);

	wm_delete_window = XInternAtom(XtDisplay(fsel_pshell),
				"WM_DELETE_WINDOW", False);
	XtOverrideTranslations(fsel_pshell,
		XtParseTranslationTable(
			"<Message>WM_PROTOCOLS: fsel-cancel()"));
}

static void add_extra(char *extra)
{
	char *p, *q;

	nextra = 0;
	if (!extra) return;
	below = fsel_helpbutton;
	for (p = strtok(extra, ":"); p; p = strtok(NULL, ":")) {
		q = strchr(p, '=');
		if (!q) continue;
		*q++ = '\0';
		fsel_extra[nextra] = add_button("fsel_extra", p,
			change_dir, q, fsel_topbox);
		XtVaSetValues(fsel_extra[nextra++],
			XtNwidth, 80, (char *)0);
	}
}

/* ---
Select a file and directory. The path and name must contain valid strings.
*/

int fsel_input(Widget pw, char *path, char *name, char **patterns,
	char *fmt, char *extra)
{
	String string;
	XtAppContext app_context = XtWidgetToApplicationContext(pw);
	int i;

	if (fsel_pshell == None) fsel_init(pw);

	if (patterns == NULL)
		patterns = fileformats;

	/* First, create all the formats on the menu */
	XtVaSetValues(fsel_formatbutton,
		XtNlabel, patterns[0],
		XtNwidth, 200, (char *)0);
	fsel_formatmenu = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, fsel_formatbutton, (char *)0);

	make_menu(patterns, fsel_formatmenu);

	label_set(fsel_dirbutton, path);
	XtVaSetValues(fsel_filetext, XtNstring, name, (char *)0);
	add_extra(extra);
	status = WAITING;

	center(pw, fsel_pshell);
	XtPopup(fsel_pshell, XtGrabNonexclusive);
	XSetWMProtocols(XtDisplay(fsel_pshell), XtWindow(fsel_pshell),
			&wm_delete_window, 1);
	fsel_scan();

	XtSetKeyboardFocus(fsel_pshell, fsel_filetext);

	while (status == WAITING) {
		XEvent event_return;
		XtAppNextEvent(app_context, &event_return);
		XtDispatchEvent(&event_return);
	}

	/* And get rid of the formats */
	XtDestroyWidget(fsel_formatmenu);
	make_dirmenu("");
	XtDestroyWidget(fsel_dirmenu);
	fsel_dirmenu = None;
	for (i = 0; i < nextra; i++)
		XtDestroyWidget(fsel_extra[i]);

	XtVaGetValues(fsel_filetext, XtNstring, &string, (char *)0);
	strcpy(name, string);
	strcpy(path, label_get(fsel_dirbutton));
	strcpy(fmt, label_get(fsel_formatbutton));
	return status;
}

