/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev, Christian Lupien and Chris Hardy

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#define GATOS_TVOUT_C 1

#include "gatos.h"
#include "tvout.h"
#include "atiregs.h"

#include <stdio.h>
#include <errno.h>
#include <string.h>

/* MPP_CONFIG values */
#define MPPNORMAL	0x80038CB0	/* write */
#define MPPREAD		0x84038CB0	/* prefetch read */
#define MPPNORMALINC	0x80338CB0	/* write, autoincrement MPP_ADDR */
#define MPPREADINC	0x84338CB0	/* prefetch read, MPP_ADDR autoincr */
/* MPP_ADDR autoincrement wraps around after 4 adresses (2 LSB's only) */

#define MPPTRIES	1	/* See comments in mpp_wait() /AA */

/* Private global vars */
static char *ident=NULL, *name=NULL ;
static int id=0, rev=0 ;
static u32 timeouts=0 ;

/* Note:
 * ImpacTV registers are 32 bit, indexed by the 16 bit
 * value in MPP registers 0x0009 (MSB) and 0x0008 (LSB).
 * Value of indexed ImpacTV register is accessed
 * through MPP registers 0x0018 (LSB) to 0x001B (MSB).
 * tvout_read_i2c_cntl8() and tvout_write_i2c_cntl8() assumes that
 * the ImpacTV Register Index points to TV_I2C_CNTL (index 0x0015)
 * and that MPP_ADDR points to low byte of ImpacTV Register Data.
 * All other routines must call tvout_addr(TV_I2C_CNTL) to restore
 * these conditions after accessing other ImpacTV chip registers.
 * AA
 */

/* ------------------------------------------------------------------------ */
/* Initialization routine */

int tvout_init(void) {

  u8 init ; u32 save1, save2, save3 ;

  save1 = DAC_CNTL ; save2 = MPP_CONFIG ; save3 = MPP_STROBE_SEQ ;
  MPP_STROBE_SEQ = 0x00000383 ; DAC_CNTL &= 0xFFFFBFFF ;

  /* Read ImpacTV Revision Code and Initialize value */
  MPP_CONFIG = MPPREAD ; mpp_wait("tvout_init()") ;
  MPP_ADDR = 0x0000000A ; init = MPP_DATA0 ; mpp_wait("tvout_init() 1") ;
  MPP_ADDR = 0x0000000B ; id = MPP_DATA0 ;   mpp_wait("tvout_init() 2") ;
  MPP_CONFIG = MPPNORMAL ;

  switch (id) {
    case IMPACTV:   ident = "ImpacTV" ; rev = 1 ; break ;
    case IMPACTV2:  case IMPACTV2M:  ident = "ImpacTV2" ; rev = 2 ; break ;
    case IMPACTV2P: case IMPACTV2PM: ident = "ImpacTV2+" ; rev = 3 ; break ;
    default: break ; }
  switch (id) {
    case IMPACTV:    name = "ImpacTV" ; break ;
    case IMPACTV2:   name = "ImpacTV2 (Macrovision disabled)" ; break ;
    case IMPACTV2M:  name = "ImpacTV2 (Macrovision enabled)" ; break ;
    case IMPACTV2P:  name = "ImpacTV2+ (Macrovision disabled)" ; break ;
    case IMPACTV2PM: name = "ImpacTV2+ (Macrovision enabled)" ; break ;
    default: break ; }
  if (init != 0x00) rev = 0 ;

  if (rev) {
    if (VERBOSE) { printf("TV Out: %s\n",name) ; fflush(stdout) ; }
    tvout_addr(TV_I2C_CNTL) ; }
  else { /* Initialization failed, restore registers */
    DAC_CNTL = save1 ; MPP_CONFIG = save2 ; MPP_STROBE_SEQ = save3 ; }

  /* Always return success (even if ImpacTV chip not present :-) */
  RETURN0 ; }

/* ------------------------------------------------------------------------ */
/* Public routines */

/* For MPP/ImpacTV I2C Driver */
int tvout_present(void) { return (rev!=0) ; }

/* Read low byte of ImpacTV TV_I2C_CNTL register */
u8 tvout_read_i2c_cntl8(void) {
  u8 data ;
  mpp_wait("tvout_read_i2c_cntl8()") ;
  MPP_CONFIG = MPPREAD ; mpp_wait("tvout_read_i2c_cntl8() 1") ;
  data = MPP_DATA0 ;
  /* Is this really necessary ? /AA */
  if (gatos.bugfix_mppread) data ^= gatos.bugfix_mppread ;
  return data ; }

/* Write low byte of ImpacTV TV_I2C_CNTL register */
void tvout_write_i2c_cntl8(u8 data) {
  mpp_wait("tvout_write_i2c_cntl8()") ;
  MPP_CONFIG = MPPNORMAL ; MPP_DATA0 = data ; }

/* Read ImpacTV register */
u32 tvout_read32(u16 addr) {
  u32 data ;
  tvout_addr(addr) ;
  MPP_CONFIG = MPPREADINC ; mpp_wait("tvout_read32()") ;
  data  = MPP_DATA0 ;       mpp_wait("tvout_read32() 1") ;
  data |= MPP_DATA0 << 8 ;  mpp_wait("tvout_read32() 2") ;
  data |= MPP_DATA0 << 16 ; mpp_wait("tvout_read32() 3") ;
  data |= MPP_DATA0 << 24 ; mpp_wait("tvout_read32() 4") ;
  /* Is this really necessary ? /AA */
  if (gatos.bugfix_mppread) data ^=
    gatos.bugfix_mppread | (gatos.bugfix_mppread<<8) |
    (gatos.bugfix_mppread<<16) | (gatos.bugfix_mppread<<24) ;
  if (addr != TV_I2C_CNTL) tvout_addr(TV_I2C_CNTL) ; return data ; }

/* Write to ImpacTV register */
void tvout_write32(u16 addr, u32 data) {
  tvout_addr(addr) ;
  mpp_wait("tvout_write32()") ;
  MPP_CONFIG = MPPNORMALINC ;
  MPP_DATA0 = data ;       mpp_wait("tvout_write32() 1") ;
  MPP_DATA0 = data >> 8 ;  mpp_wait("tvout_write32() 2") ;
  MPP_DATA0 = data >> 16 ; mpp_wait("tvout_write32() 3") ;
  MPP_DATA0 = data >> 24 ; mpp_wait("tvout_write32() 4") ;
  if (addr != TV_I2C_CNTL) tvout_addr(TV_I2C_CNTL) ; }

/* ------------------------------------------------------------------------ */
/* Private routines */

/* Set ImpacTV register index and return MPP_ADDR
 * to 0x0018 for ImpacTV register data access. */
static void tvout_addr(u16 addr) {
  mpp_wait("tvout_addr()") ;
  MPP_CONFIG = MPPNORMALINC ; MPP_ADDR = 0x00000008 ;
  MPP_DATA0 = addr ;    mpp_wait("tvout_addr() 1") ;
  MPP_DATA0 = addr>>8 ; mpp_wait("tvout_addr() 2") ;
  MPP_CONFIG = MPPNORMAL ; MPP_ADDR = 0x00000018 ; }

/* Returns 1 if ready, 0 if MPP bus timed out (not ready) */
static int mpp_wait(char *str) {
#if 0
  u32 tries=MPPTRIES ;
  /* For some reason, it's *necessary* to have *two* while() poll loops
   * to avoid MPP bus timeouts, but doubling MPPTRIES doesn't help ...
   * Maybe the code optimizer is playing tricks on us ?!?
   * Nah, can't be, problem is there with 'gcc -g' as well ...
   * Also, it seems to make no difference if MPPTRIES is 0x100
   * (the official ATI value) or 1, so I've naturally chosen 1 ...  /AA
   */
  while (tries && (MPP_CONFIG3 & 0x40)) tries-- ; if (tries) return 1 ;
  tries = MPPTRIES ;
  while (tries && (MPP_CONFIG3 & 0x40)) tries-- ; if (tries) return 1 ;
#endif
  /* These two if's seems to work just as
  * fine, but there *must* be two /AA */
  if (!(MPP_CONFIG3 & 0x40)) return 1 ;
  if (!(MPP_CONFIG3 & 0x40)) return 1 ;
  if (!GATOSQUIET) fprintf(stderr,"mpp_wait(): MPP bus timeout in %s\n",str) ;
  timeouts++ ; return 0 ; }

/* ------------------------------------------------------------------------ */
/* Debug and report routines */

#define DUMPREG(NAME)	\
	fprintf(stderr,"%s: %-22s (0x%04X) = 0x%08X\n", \
		ident,#NAME,NAME,tvout_read32(NAME))

void tvout_dumpregs(void) {
  if (!rev) return ;
  DUMPREG(TV_MASTER_CNTL) ;
  DUMPREG(TV_RGB_CNTL) ;
  DUMPREG(TV_CLKOUT_CNTL) ;
  DUMPREG(TV_SYNC_CNTL) ;
  DUMPREG(TV_I2C_CNTL) ;
  DUMPREG(TV_MPP_DATA_CNTL) ;
  DUMPREG(TV_HTOTAL) ;
  DUMPREG(TV_HDISP) ;
  DUMPREG(TV_HSIZE) ;
  DUMPREG(TV_HSTART) ;
  DUMPREG(TV_HCOUNT) ;
  DUMPREG(TV_VTOTAL) ;
  DUMPREG(TV_VDISP) ;
  DUMPREG(TV_VCOUNT) ;
  DUMPREG(TV_FTOTAL) ;
  DUMPREG(TV_FCOUNT) ;
  DUMPREG(TV_FRESTART) ;
  DUMPREG(TV_HRESTART) ;
  DUMPREG(TV_VRESTART) ;
  DUMPREG(TV_SYNC_SIZE) ;
  DUMPREG(TV_TV_PLL_CNTL) ;
  DUMPREG(TV_CRT_PLL_CNTL) ;
  DUMPREG(TV_PLL_CNTL) ;
  DUMPREG(TV_PLL_TEST_CNTL) ;
  DUMPREG(TV_CLOCK_SEL_CNTL) ;
  DUMPREG(TV_FRAME_LOCK_CNTL) ;
  DUMPREG(TV_SYNC_LOCK_CNTL) ;
  DUMPREG(TV_TVO_SYNC_PAT_ACCUM) ;
  DUMPREG(TV_TVO_SYNC_THRESHOLD) ;
  DUMPREG(TV_TVO_SYNC_PAT_EXPECT) ;
  DUMPREG(TV_DELAY_ONE_MAP_A) ;
  DUMPREG(TV_DELAY_ONE_MAP_B) ;
  DUMPREG(TV_DELAY_ZERO_MAP_A) ;
  DUMPREG(TV_DELAY_ZERO_MAP_B) ;
  DUMPREG(TV_TVO_DATA_DELAY_A) ;
  DUMPREG(TV_TVO_DATA_DELAY_B) ;
  DUMPREG(TV_HOST_READ_DATA) ;
  DUMPREG(TV_HOST_WRITE_DATA) ;
  DUMPREG(TV_HOST_RD_WT_CNTL) ;
  DUMPREG(TV_VSCALER_CNTL) ;
  DUMPREG(TV_TIMING_CNTL) ;
  DUMPREG(TV_GAMMA_CNTL) ;
  if (rev == 2) {
    DUMPREG(TV_Y_FALL_CNTL) ;
    DUMPREG(TV_Y_RISE_CNTL) ;
    DUMPREG(TV_Y_SAW_TOOTH_CNTL) ; }
  DUMPREG(TV_MODULATOR_CNTL1) ;
  DUMPREG(TV_MODULATOR_CNTL2) ;
  DUMPREG(TV_PRE_DAC_MUX_CNTL) ;
  DUMPREG(TV_TV_DAC_CNTL) ;
  DUMPREG(TV_CRC_CNTL) ;
  DUMPREG(TV_VIDEO_PORT_SIG) ;
  if (rev == 3) {
    DUMPREG(TV_VBI_20BIT_CNTL) ;
    DUMPREG(TV_VBI_LEVEL_CNTL) ; }
  DUMPREG(TV_UV_ADR) ;
  DUMPREG(TV_FIFO_TEST_CNTL) ;
  /* Report then clear MPP bus timeout stats */
  if (timeouts) fprintf(stderr,"Warning: %d MPP bus timeouts\n",timeouts) ;
  timeouts = 0 ; }
