/*
 * Copyright (C) 1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#define VIRTCONN_INTERNAL_ACCESS
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include "queue.h"
#include "list.h"
#include "eventlog.h"
#include "virtconn.h"


static t_list * virtconn_head=NULL;


extern t_virtconn * virtconn_create(int csd, int ssd, unsigned int udpaddr, unsigned short udpport)
{
    t_virtconn * temp=NULL;
   
    if (csd<0)
    {
        eventlog(eventlog_level_error,"virtconn_create","got bad client socket");
        return NULL;
    }
    if (ssd<0)
    {
        eventlog(eventlog_level_error,"virtconn_create","got bad server socket");
        return NULL;
    }
   
    if (!(temp = malloc(sizeof(t_virtconn))))
    {
        eventlog(eventlog_level_error,"virtconn_create","could not allocate memory for temp");
        return NULL;
    }
   
    temp->csd                   = csd;
    temp->ssd                   = ssd;
    temp->udpport               = udpport;
    temp->udpaddr               = udpaddr;
    temp->class                 = virtconn_class_none;
    temp->state                 = virtconn_state_initial;
    temp->coutqueue             = NULL;
    temp->coutsize              = 0;
    temp->cinqueue              = NULL;
    temp->cinsize               = 0;
    temp->soutqueue             = NULL;
    temp->soutsize              = 0;
    temp->sinqueue              = NULL;
    temp->sinsize               = 0;
    temp->fileleft              = 0;
    
    if (list_prepend_item(&virtconn_head,temp)<0)
    {
        pfree(temp,sizeof(t_virtconn));
        eventlog(eventlog_level_error,"virtconn_create","could not prepend temp");
        return NULL;
    }
    
    return temp;
}


extern void virtconn_destroy(t_virtconn * vc)
{
    t_list * * pos;
    
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_destroy","got NULL virtconn");
        return;
    }
    if (!(pos = list_find_item(&virtconn_head,vc)))
	eventlog(eventlog_level_error,"virtconn_destroy","could not find virtconn in list");
    else
        if (list_remove(pos)<0)
            eventlog(eventlog_level_error,"virtconn_destroy","could not remove item from list");
    
    vc->state = virtconn_state_empty;
    
    close(vc->ssd);
    close(vc->csd);
    
    /* clear out the packet queues */
    queue_clear(&vc->sinqueue);
    queue_clear(&vc->soutqueue);
    queue_clear(&vc->cinqueue);
    queue_clear(&vc->coutqueue);
    
    eventlog(eventlog_level_info,"virtconn_destroy","[%d] closed server connection (%d) class=%d",vc->ssd,vc->csd,(int)vc->class);
    eventlog(eventlog_level_info,"virtconn_destroy","[%d] closed client connection (%d) class=%d",vc->csd,vc->ssd,(int)vc->class);
    
    pfree(vc,sizeof(t_virtconn));
}


extern t_virtconn_class virtconn_get_class(t_virtconn const * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_class","got NULL virtconn");
        return virtconn_class_none;
    }

    return vc->class;
}


extern void virtconn_set_class(t_virtconn * vc, t_virtconn_class class)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_set_class","got NULL virtconn");
        return;
    }
    
    vc->class = class;
}


extern t_virtconn_state virtconn_get_state(t_virtconn const * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"cvirtonn_set_state","got NULL virtconn");
        return virtconn_state_empty;
    }
    
    return vc->state;
}


extern void virtconn_set_state(t_virtconn * vc, t_virtconn_state state)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_set_state","got NULL virtconn");
        return;
    }
    
    vc->state = state;
}


extern unsigned int virtconn_get_udpaddr(t_virtconn const * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_udpaddr","got NULL virtconn");
        return 0;
    }
    
    return vc->udpaddr;
}


extern unsigned short virtconn_get_udpport(t_virtconn const * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_udpport","got NULL virtconn");
        return 0;
    }
    
    return vc->udpport;
}


extern t_queue * * virtconn_get_clientin_queue(t_virtconn * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_clientin_queue","got NULL connection");
        return NULL;
    }

    return &vc->cinqueue;
}


extern int virtconn_get_clientin_size(t_virtconn const * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_clientin_size","got NULL connection");
        return -1;
    }

    return vc->cinsize;
}


extern void virtconn_set_clientin_size(t_virtconn * vc, unsigned int size)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_set_clientin_size","got NULL connection");
        return;
    }

    vc->cinsize = size;
}


extern t_queue * * virtconn_get_clientout_queue(t_virtconn * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_clientout_queue","got NULL connection");
        return NULL;
    }

    return &vc->coutqueue;
}


extern int virtconn_get_clientout_size(t_virtconn const * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_clientout_size","got NULL connection");
        return -1;
    }

    return vc->coutsize;
}


extern void virtconn_set_clientout_size(t_virtconn * vc, unsigned int size)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_set_clientout_size","got NULL connection");
        return;
    }

    vc->coutsize = size;
}


extern int virtconn_get_client_socket(t_virtconn const * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_client_socket","got NULL virtconn");
	return -1;
    }
    return vc->csd;
}


extern t_queue * * virtconn_get_serverin_queue(t_virtconn * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_serverin_queue","got NULL connection");
        return NULL;
    }

    return &vc->sinqueue;
}


extern int virtconn_get_serverin_size(t_virtconn const * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_serverin_size","got NULL connection");
        return -1;
    }

    return vc->sinsize;
}


extern void virtconn_set_serverin_size(t_virtconn * vc, unsigned int size)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_set_serverin_size","got NULL connection");
        return;
    }

    vc->sinsize = size;
}


extern t_queue * * virtconn_get_serverout_queue(t_virtconn * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_serverout_queue","got NULL connection");
        return NULL;
    }

    return &vc->soutqueue;
}


extern int virtconn_get_serverout_size(t_virtconn const * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_serverout_size","got NULL connection");
        return -1;
    }

    return vc->soutsize;
}


extern void virtconn_set_serverout_size(t_virtconn * vc, unsigned int size)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_set_serverout_size","got NULL connection");
        return;
    }

    vc->soutsize = size;
}


extern int virtconn_get_server_socket(t_virtconn const * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_server_socket","got NULL virtconn");
	return -1;
    }
    return vc->ssd;
}


extern void virtconn_set_fileleft(t_virtconn * vc, unsigned int size)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_set_fileleft","got NULL virtconn");
	return;
    }
    vc->fileleft = size;
}


extern unsigned int virtconn_get_fileleft(t_virtconn const * vc)
{
    if (!vc)
    {
        eventlog(eventlog_level_error,"virtconn_get_fileleft","got NULL virtconn");
	return -1;
    }
    return vc->fileleft;
}



extern void virtconnlist_init(void)
{
    /* nothing to do for now */
}


extern t_virtconn * virtconnlist_get_first(t_list const * const * * save)
{
    void * virtconn;

    if (!save)
    {
	eventlog(eventlog_level_error,"virtconnlist_get_first","got NULL save");
        return NULL;
    }

    *save = (t_list const * const *)&virtconn_head; /* avoid warning */

    if (!**save)
    {
        *save = NULL;
        return NULL;
    }
    virtconn = list_get_item(**save);
    *save = list_get_next_const(**save);

    return virtconn;
}


extern t_virtconn * virtconnlist_get_next(t_list const * const * * save)
{
    void * virtconn;

    if (!save)
    {
	eventlog(eventlog_level_error,"virtconnlist_get_next","got NULL save");
        return NULL;
    }

    if (!*save || !**save)
    {
        *save = NULL;
        return NULL;
    }
    virtconn = list_get_item(**save);
    *save = list_get_next_const(**save);

    return virtconn;
}
