// Simple utility to put a file using Darxite.

#include <stdio.h>  // for printf() etc.
#include <getopt.h> // for getopt()
#include <stdlib.h> // for atoi()
#include <string.h>	// for strerror()
#include <unistd.h> // for write()
#include <errno.h>  // for errno

#include <darxite.h>

#define OPTION_STRING	"vhabc:ef:i:prqt:u:"
#define VERSION			"0.1"
#define AUTHORS			"Ashley Montanaro"

BOOL Echo, Quiet, Blocking, FromStdIn;
char *TempName, RemoteHost[256];
char LocalPath[256], User[256], Flags[20];
int RemotePort = 0, Timeout = 10;

char *ReadCmdLine(int argc, char **argv);
void PrintVersion(void);
void Usage(char *prog_name);

void PrintVersion(void)
{
    printf("DarxPut utility v%s, release %s (%s) by %s\n", VERSION,
           RELEASE_VER, RELEASE_NAME, AUTHORS);
}

void Usage(char *prog_name)
{
    printf("Usage: %s [OPTIONS] [-i localfile] URL\n"
           "where options are:\n"
           "-v, --version\t\t\tShow version and exit\n"
           "-h, --help\t\t\tShow some usage information\n"
           "-a, --append\t\t\tAppend the local file to the remote file\n"
           "-b, --blocking\t\t\tBlock until the transfer is complete\n"
           "-c<host>:<port>, --connect\tConnect to the daemon on <host>, "
           "port <port>\n"
           "-e, --echo\t\t\tEcho the command as it's sent\n"
           "-f<flags>, --flags <flags>\tSend <flags> as options\n"
           "-i<path>, --input <path>\tUpload <path> to server\n"
           "-p, --paused\t\t\tDon't start the transfer yet\n"
           "-q, --quiet\t\t\tDon't output anything\n"
           "-r, --recursive\t\t\tUploads a directory recursively\n"
           "-t<secs>, --timeout <secs>\tTime out response after <secs>\n"
           "-u<user>, --user <user>\t\tLog in as <user>\n"
           "If input file is -, the program will read from stdin.\n",
           prog_name);
    exit(0);
}

char *ReadCmdLine(int argc, char **argv)
{
    char opt, *override_flags = NULL;
    int option_index;

    static struct option long_options[] = {
        { "version", 	0, 0, 'v' },
        { "help",		0, 0, 'h' },
        { "append",		0, 0, 'a' },
        { "blocking",	0, 0, 'b' },
        { "connect",	1, 0, 'c' },
        { "directory",  0, 0, 'd' },
        { "echo",		0, 0, 'e' },
        { "flags",		1, 0, 'f' },
        { "input",		1, 0, 'i' },
        { "no-resume",  0, 0, 'n' },
        { "paused",		0, 0, 'p' },
        { "quiet",		0, 0, 'q' },
        { "recursive",	0, 0, 'r' },
        { "timeout",	1, 0, 't' },
        { "user",		1, 0, 'u' },
        { 0, 			0, 0,  0  }
    };
    
    strcpy(LocalPath, "");
    strcpy(Flags, "");
    strcpy(User, "");
    opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                      &option_index);
    while (opt != -1)
    {
        switch (opt)
        {
        case 'v':
            PrintVersion();
            exit(0);
            break;
            
        case 'h':
            Usage(argv[0]);
            break;

        case 'a':
            if (!strchr(Flags, 'c'))
                strcat(Flags, "c");
            break;
            
        case 'b':
            Blocking = TRUE;
            if (!strchr(Flags, 'n'))
                strcat(Flags, "n");
            break;
            
        case 'c':
            if (strchr(optarg, ':'))
            {
                memset(RemoteHost, 0, sizeof(RemoteHost));
                strncpy(RemoteHost, optarg, strchr(optarg, ':') - optarg);
                RemotePort = atoi(strchr(optarg, ':') + 1);
                if ((strlen(RemoteHost) == 0) || (RemotePort <= 0))
                {
                    printf("Invalid host/port to connect to daemon\n");
                    exit(0);
                }
            }
            else
            {
                printf("Syntax: -c <host>:<port>\n");
                exit(0);
            }
            break;
            
        case 'e':
            Echo = TRUE;
            break;

        case 'f':
            if (override_flags == NULL)
                override_flags = strdup(optarg);
            break;

        case 'i':
            // path must end with a space for format
            // of command below
            if (!strcmp(optarg, "-"))
            {
                TempName = tempnam(NULL, "darxput");
                sprintf(LocalPath, "%s ", TempName);
                FromStdIn = TRUE;
            }
            else
            {
                sprintf(LocalPath, "%s ", optarg);
            }
            break;
            
        case 'n':
            strcat(Flags, "b");
            break;
            
        case 'p':
            strcat(Flags, "p");
            break;
            
        case 'q':
            Quiet = TRUE;
            break;
            
        case 'r':
            strcat(Flags, "d");
            break;
            
        case 't':
            Timeout = atoi(optarg);
            break;
            
        case 'u':
            strcpy(User, optarg);
            break;
        }
        opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                          &option_index);        
    }
    // if we need to override the flags
    if (override_flags)
    {
        strcpy(Flags, override_flags);
        free(override_flags);
    }
    else
    {
        if (!strchr(Flags, 'u'))
            strcat(Flags, "u");
    }
    if (optind < argc)
        return argv[optind];
    else
       return NULL;
}

int main(int argc, char *argv[])
{
    int fd, rc;
    char c;
    FILE *local_file;
    char buffer[2048], password[256], *url, *pass;

    url = ReadCmdLine(argc, argv);
    if (url == NULL)
        Usage(argv[0]);
    
    if (strcmp(RemoteHost, "") && (RemotePort > 0))
    {
        sprintf(buffer, "Enter password for daemon on %s: ", RemoteHost);
        fd = DX_ConnectRemoteClient(RemoteHost, RemotePort, getpass(buffer),
                                    "darxget");
    }
    else
    {
        fd = DX_ConnectClient("darxget");
    }

    if (fd < 0)
    {
        fprintf(stderr, "Couldn't connect to daemon: %s\n",
                strerror(DX_errno));
        return 1;
    }

    memset(password, 0, sizeof(password));
    if (strcmp(User, ""))
    {
        sprintf(buffer, "Enter password for %s: ", User);
        pass = getpass(buffer);
        sprintf(password, "%s ", pass);
        strcat(User, " ");
    }
    // read from stdin if we have to
    if (FromStdIn)
    {
        local_file = fopen(TempName, "w");
        if (local_file)
        {
            while ((c = getchar()) != EOF)
                fputc(c, local_file);
            fclose(local_file);
        }
        else
        {
            printf("Couldn't open temporary input file: %s\n",
                   strerror(errno));
            return 1;
        }
    }
    sprintf(buffer, "Get \"%s\" | %s| %s| %s| %s\n", url, LocalPath,
            User, password, Flags);
    
    // Write (and echo to screen?) the command
    if (write(fd, buffer, strlen(buffer)) < 0)
        perror("write");
    else if (Echo)
        printf("%s", buffer);
    
    rc = DX_GetResponse(fd, buffer, sizeof(buffer), Timeout);
    if (rc == DX_LIB_OK)
    {
        if (!Quiet)
            printf("%s", buffer);
    }
    else if (rc == DX_LIB_TIMEOUT)
    {
        printf("There was no response within %d second%s.\n", Timeout,
               (Timeout == 1) ? "" : "s");
    }
    else
    {
        printf("Couldn't get the response: %s", strerror(DX_errno));
    }
    
    // wait for the "file complete" response if there wasn't an error
    if (Blocking)
    {
        if (atoi(buffer) == 900)
        {
            rc = DX_GetResponse(fd, buffer, sizeof(buffer), 3600);
            if (rc == DX_LIB_OK)
            {
                if (!Quiet)
                    printf("%s", buffer);
            }
        }
        else
        {
            DX_DisconnectClient(fd);
            return 0;
        }
    }
    if (FromStdIn)
        remove(TempName);
    DX_DisconnectClient(fd);
    return 0;
}
