// File handling stuff

#include <darxite.h>
#include <wordexp.h>

#include "files.h"
#include "global.h"
#include "network.h"
#include "clients.h"
#include "http.h"

// Find a server in the list of servers visited, or add it if it isn't in
// the list
ServerInfo *FindServer(char *serv_name, BOOL add_to_file)
{
    char buffer[512], alias[256], *ptr;
    ServerInfo *server = FirstServer;

    if (!serv_name || !strcmp(serv_name, ""))
        return NULL;
    
    // we only want to look at the bit before the /
    memclr(buffer);
    if (strchr(serv_name, '/') > serv_name)
        strncpy(buffer, serv_name, strchr(serv_name, '/') - serv_name);
    else
        strcpy(buffer, serv_name);
    // First try and find an alias
    while (server)
    {
        if (!strcasecmp(buffer, server->Alias))
        {
            // If we want to add it to the file now and it
            // hasn't been added already, then add it
            if (add_to_file && !server->AddedToFile)
            {
                AddServerToFile(server);
            }
            return server;
        }
        server = server->Next;
    }
    
    // Now try and find a URL
    server = FirstServer;
    while (server)
    {
        if (!strcasecmp(buffer, server->Name))
        {
            if (add_to_file && !server->AddedToFile)
            {
                AddServerToFile(server);
            }
            return server;
        }
        server = server->Next;
    }
    
    // if it's not there, work out a decent alias for it
    // (but don't bother if the server name's got a space in it)
    if ((strchr(serv_name, '/') &&
         (strchr(serv_name, ' ') < strchr(serv_name, '/'))) ||
        (!strchr(serv_name, '/') && strchr(serv_name, ' ')))
    {
        error(E_TRACE, "Server in \"%s\" has a space in it - not adding",
              serv_name);
        return NULL;
    }
    memclr(alias);
	memclr(buffer);
    safecpy(buffer, serv_name);
    // if the first few letters in the host name are suitable for aliasing
    if ((strncasecmp(buffer, "ftp.", 4) && strncasecmp(buffer, "www.", 4) &&
         strncasecmp(buffer, "web.", 4)))
    {
        ptr = buffer;
        while ((*ptr) && (*ptr != '.') && (*ptr != '/'))
            ptr++;
        strncpy(alias, buffer, ptr - buffer);
        if (!strcmp(alias, serv_name) || (strlen(alias) < 3))
            strcpy(alias, "");
        else
            *alias = toupper(*alias);
        error(E_TRACE, "New alias is \"%s\"", alias);
    }
    // if no luck so far, try the next part of the host name
    if (!strcmp(alias, "") && (strlen(serv_name) > 4))
    {
        memset(alias, 0, sizeof(alias));
        memset(buffer, 0, sizeof(buffer));
        strcpy(buffer, serv_name);
        ptr = buffer + 4;
        while ((*ptr) && (*ptr != '.') && (*ptr != '/'))
            ptr++;
        strncpy(alias, buffer + 4, ptr - (buffer + 4));
        if (strlen(alias) < 3)
            strcpy(alias, "");
        else
            *alias = toupper(*alias);
        error(E_TRACE, "New alias is \"%s\"", alias);
    }
    
    // make sure it doesn't already exist
    server = FirstServer;
    while (server)
    {
        if (!strcasecmp(alias, server->Alias) ||
            !strcasecmp(alias, server->Name))
        {
            strcpy(alias, "");
            break;
        }
        server = server->Next;
    }
    // add it to the list (and maybe the file)
    return AddServer(add_to_file, serv_name, alias,
                     "Server added automatically", "", "", "");
}

BOOL AddServerToFile(ServerInfo *server)
{
    char buffer[512];
    FILE *file;
    int rc;

    sprintf(buffer, "%s/servers", DX_ProgDir);
    chmod(buffer, S_IRUSR | S_IWUSR);
    if ((file = fopen(buffer, "a")))
    {
        sprintf(buffer, "\n\"%s%s\" | %s | %s |%s%s | \"%s\" | |",
                server->Name, server->DefaultPath,
                server->UserName, server->Password,
                (strlen(server->Alias) > 0) ? " " : "",
                server->Alias, server->Comment);
        rc = fwrite(buffer, strlen(buffer), 1, file);
        fclose(file);
        if (rc > 0)
        {
            server->AddedToFile = TRUE;
            return TRUE;
        }
    }
    return FALSE;
}

// Add a server to the array and possibly the file - or alter an existing
// server's attributes
ServerInfo *AddServer(BOOL add_to_file, const char *url, const char *alias,
                      const char *comment, const char *user_name,
                      const char *password, const char *flags)
{
    char protocol[10], server[256], path[256];
    ServerInfo *serv_ptr = NULL, *last_serv = FirstServer;
    BOOL server_exists = FALSE;

    if (!url)
        return NULL;
    error(E_TRACE, "Adding server with URL \"%s\", alias \"%s\", "
          "comment \"%s\", user name \"%s\", password \"%s\", flags \"%s\"",
          url, alias, comment, user_name, "???", flags);
    DX_ParseUrl(url, protocol, server, sizeof(server), path, sizeof(path));
    error(E_TRACE, "Server is %s, path %s", server, path);
    while (last_serv && last_serv->Next)
    {
        // check whether it already exists
        if (!strcasecmp(last_serv->Name, server) &&
            !strcasecmp(last_serv->DefaultPath, path))
        {
            server_exists = TRUE;
            serv_ptr = last_serv;
            break;
        }
        last_serv = last_serv->Next;
    }
    // if the server didn't exist, then reset its info
    if (!server_exists)
    {
        serv_ptr = dxmalloc(sizeof(ServerInfo));
        if (FirstServer == NULL)
        {
            FirstServer = serv_ptr;
        }
        else
        {
            last_serv->Next = serv_ptr;
            serv_ptr->Prev = last_serv;
        }
        serv_ptr->AddedToFile = FALSE;
        serv_ptr->SupportsFileResume = TRUE;
        serv_ptr->MirrorCount = 0;
        serv_ptr->IpAddress = 0;
        serv_ptr->Next = NULL;
    }
    strcpy(serv_ptr->Name, server);
    if (strcmp(path, "/"))
        strcpy(serv_ptr->DefaultPath, path);
    else
        strcpy(serv_ptr->DefaultPath, "");
    strcpy(serv_ptr->Alias, alias);
    strcpy(serv_ptr->Comment, comment);
    strcpy(serv_ptr->UserName,
           strcmp(user_name, "") ? user_name : "anonymous");
    strcpy(serv_ptr->Password,
           strcmp(password, "") ? password : DX_EMailAddress);
    strcpy(serv_ptr->Flags, flags);
    
    if (add_to_file && !server_exists)
        AddServerToFile(serv_ptr);
    ServerCount++;
    return serv_ptr;
}

#if 0
/* backup */
// Read the server info file in
int ReadServersFile(const char *servers_file)
{
    FILE *file;
    ServerInfo *server;
    char buffer[2048], *field_ptr, *buf_ptr;
    char comment[256], user_name[256], password[256];
    char alias[256], flags[256], url[256];

    ServerCount = 0;
    if (!DX_EnableReadingServerDB)
        return 0;
    
    chmod(servers_file, S_IRUSR | S_IWUSR);
    file = fopen(servers_file, "r");
    if (file)
    {
        memset(buffer, 0, sizeof(buffer));
        while (DX_ReadFileLine(buffer, sizeof(buffer), file))
        {
	    strip(buffer);
	
            if (!(buf_ptr = DX_StripComments(buffer)))
                continue;
            
            //error(E_TRACE, "Servers file: Read line %s", buf_ptr);
            
// URL | UserName | Password | Alias | Comment | Flags | Mirrors
            
            strcpy(url, "");
            strcpy(user_name, "");
            strcpy(password, "");
            strcpy(alias, "");
            strcpy(comment, "");
            strcpy(flags, "");
            
            field_ptr = DX_GetNextField(buffer);
            if (*field_ptr)
            {
                // because the GetNextField call does not remove
                // quotes, we have to do it ourselves
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(url, field_ptr);
                if (url[strlen(url) - 1] == '"')
                    url[strlen(url) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;
            
            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(user_name, field_ptr);
                if (user_name[strlen(user_name) - 1] == '"')
                    user_name[strlen(user_name) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;
            
            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(password, field_ptr);
                if (password[strlen(password) - 1] == '"')
                    password[strlen(password) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;
            
            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(alias, field_ptr);
                if (alias[strlen(alias) - 1] == '"')
                    alias[strlen(alias) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;
            
            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(comment, field_ptr);
                if (comment[strlen(comment) - 1] == '"')
                    comment[strlen(comment) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;
            
            field_ptr = DX_GetNextField(field_ptr);
            if (*field_ptr)
            {
                if (*field_ptr == '"')
                    field_ptr++;
                strcpy(flags, field_ptr);
                if (flags[strlen(flags) - 1] == '"')
                    flags[strlen(flags) - 1] = '\0';
            }
            field_ptr += strlen(field_ptr) + 1;
            
            while (*field_ptr && (isspace(*field_ptr) || (*field_ptr == '|')))
                field_ptr++;
            //error(E_TRACE, "Read mirrors string \"%s\"", field_ptr);
            
            if (strcmp(url, ""))
            {
                // don't add it to the file since it's already there
                server = AddServer(FALSE, url, alias,
                                   comment, user_name, password, flags);
                if (!server)
                    return ServerCount;
                server->AddedToFile = TRUE;
                FindServerMirrors(server, field_ptr);
            }
            else
            {
                error(E_TRACE, "Couldn't add server: no URL found");
            }
        }
        fclose(file);
    }
    else
    {
        error(E_TRACE, "Couldn't open servers file %s", servers_file);
    }
    // create a dummy server for file:// transfers etc.
    DummyServer = dxmalloc(sizeof(ServerInfo));
    strcpy(DummyServer->Name, "");
    strcpy(DummyServer->Comment, "This server doesn't really exist");
    DummyServer->Next = NULL;
    DummyServer->Prev = NULL;
    return ServerCount;
}
#endif

// Read the server info file in
int ReadServersFile(const char *servers_file)
{
    FILE *file;
    ServerInfo *server;
    char buffer[2048], *field_ptr, *buf_ptr;
    char comment[256], user_name[256], password[256];
    char alias[256], flags[256], url[256];

    ServerCount = 0;
    if (!DX_EnableReadingServerDB)
        return 0;
   
    /* glenn: Should we do this even when
     * DX_EnableReadingServerDB is false? */
    
    // create a dummy server for file:// transfers etc.
    DummyServer = dxmalloc(sizeof(ServerInfo));
    strcpy(DummyServer->Name, "");
    strcpy(DummyServer->Comment, "This server doesn't really exist");
    DummyServer->Next = NULL;
    DummyServer->Prev = NULL;
    
    chmod(servers_file, S_IRUSR | S_IWUSR);
    file = fopen(servers_file, "r");
    if (file == NULL)
    {
        error(E_TRACE, "Couldn't open servers file %s", servers_file);
	return ServerCount;
    }

    memset(buffer, 0, sizeof(buffer));
    while (DX_ReadFileLine(buffer, sizeof(buffer), file))
    {
	strip(buffer);

	if (!(buf_ptr = DX_StripComments(buffer)))
	    continue;

	//error(E_TRACE, "Servers file: Read line %s", buf_ptr);

	// URL | UserName | Password | Alias | Comment | Flags | Mirrors

	strcpy(url, "");
	strcpy(user_name, "");
	strcpy(password, "");
	strcpy(alias, "");
	strcpy(comment, "");
	strcpy(flags, "");

	field_ptr = DX_GetNextField(buffer);
	if (*field_ptr)
	{
	    // because the GetNextField call does not remove
	    // quotes, we have to do it ourselves
	    if (*field_ptr == '"')
		field_ptr++;
	    strcpy(url, field_ptr);
	    if (url[strlen(url) - 1] == '"')
		url[strlen(url) - 1] = '\0';
	}
	field_ptr += strlen(field_ptr) + 1;

	field_ptr = DX_GetNextField(field_ptr);
	if (*field_ptr)
	{
	    if (*field_ptr == '"')
		field_ptr++;
	    strcpy(user_name, field_ptr);
	    if (user_name[strlen(user_name) - 1] == '"')
		user_name[strlen(user_name) - 1] = '\0';
	}
	field_ptr += strlen(field_ptr) + 1;

	field_ptr = DX_GetNextField(field_ptr);
	if (*field_ptr)
	{
	    if (*field_ptr == '"')
		field_ptr++;
	    strcpy(password, field_ptr);
	    if (password[strlen(password) - 1] == '"')
		password[strlen(password) - 1] = '\0';
	}
	field_ptr += strlen(field_ptr) + 1;

	field_ptr = DX_GetNextField(field_ptr);
	if (*field_ptr)
	{
	    if (*field_ptr == '"')
		field_ptr++;
	    strcpy(alias, field_ptr);
	    if (alias[strlen(alias) - 1] == '"')
		alias[strlen(alias) - 1] = '\0';
	}
	field_ptr += strlen(field_ptr) + 1;

	field_ptr = DX_GetNextField(field_ptr);
	if (*field_ptr)
	{
	    if (*field_ptr == '"')
		field_ptr++;
	    strcpy(comment, field_ptr);
	    if (comment[strlen(comment) - 1] == '"')
		comment[strlen(comment) - 1] = '\0';
	}
	field_ptr += strlen(field_ptr) + 1;

	field_ptr = DX_GetNextField(field_ptr);
	if (*field_ptr)
	{
	    if (*field_ptr == '"')
		field_ptr++;
	    strcpy(flags, field_ptr);
	    if (flags[strlen(flags) - 1] == '"')
		flags[strlen(flags) - 1] = '\0';
	}
	field_ptr += strlen(field_ptr) + 1;

	while (*field_ptr && (isspace(*field_ptr) || (*field_ptr == '|')))
	    field_ptr++;
	//error(E_TRACE, "Read mirrors string \"%s\"", field_ptr);

	if (strcmp(url, ""))
	{
	    // don't add it to the file since it's already there
	    server = AddServer(FALSE, url, alias,
		    comment, user_name, password, flags);
	    if (!server)
		return ServerCount;
	    server->AddedToFile = TRUE;
	    FindServerMirrors(server, field_ptr);
	}
	else
	{
	    error(E_TRACE, "Couldn't add server: no URL found");
	}
    }
    fclose(file);
    
    return ServerCount;
}

void FindServerMirrors(ServerInfo *server, char *mirror_string)
{
    char *field_ptr = mirror_string;
    char mirror[256];
    ServerInfo *mirror_server;
    
    while (*field_ptr)
    {
        field_ptr = DX_GetNextField(field_ptr);
        if (!*field_ptr)
            continue;
        
        if (*field_ptr == '"')
            field_ptr++;
        strcpy(mirror, field_ptr);
        if (mirror[strlen(mirror) - 1] == '"')
            mirror[strlen(mirror) - 1] = '\0';
        
        error(E_TRACE, "Read mirror \"%s\";", mirror);
        mirror_server = FindServer(mirror, FALSE);
        
        error(E_TRACE, "Server comment is \"%s\"", mirror_server->Comment);
        server->Mirrors[server->MirrorCount] = mirror_server;
        server->MirrorCount++;
        field_ptr += strlen(field_ptr) + 1;
    }
}

FileInfo *AddFileToBatch(BOOL add_to_file, char *file_line,
                         struct _clientInfo *client)
{
    char buffer[1024], line_buffer[1024], *field_ptr;
    char url[256], local_path[256], login[256], password[256], flags[256];
    char protocol[10], serv_buf[256], path[256];
    int i, total_size = SIZE_UNKNOWN, dup_count = 0;
    FileInfo *last_file = Batch.FirstFile, *new_file, *file_ptr;
    struct stat stat_buf;
    ServerInfo *server;
    BOOL id_duplicate;
    FILE *file;

    //error(E_TRACE, "aftb: \"%s\"", file_line);
    while (last_file && last_file->Next)
        last_file = last_file->Next;
    
    // Format: url | local_path | login | password | flags | total size
    memclr(url);
    memclr(local_path);
    memclr(login);
    memclr(password);
    memclr(flags);
    memclr(line_buffer);
    safecpy(line_buffer, file_line);
    field_ptr = DX_GetNextField(line_buffer);
    if (*field_ptr)
    {
        // because the GetNextField call does not remove
        // quotes, we have to do it ourselves
        if (*field_ptr == '"')
            field_ptr++;
        safecpy(url, field_ptr);
        if (lastchr(url) == '"')
            lastchr(url) = '\0';
    }
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
    {
        if (*field_ptr == '"')
            field_ptr++;
        safecpy(local_path, field_ptr);
        if (lastchr(local_path) == '"')
            lastchr(local_path) = '\0';
    }
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
    {
        if (*field_ptr == '"')
            field_ptr++;
        safecpy(login, field_ptr);
        if (lastchr(login) == '"')
            lastchr(login) = '\0';
    }
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
    {
        if (*field_ptr == '"')
            field_ptr++;
        safecpy(password, field_ptr);
        if (lastchr(password) == '"')
            lastchr(password) = '\0';
    }
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
    {
        if (*field_ptr == '"')
            field_ptr++;
        safecpy(flags, field_ptr);
        if (lastchr(flags) == '"')
            lastchr(flags) = '\0';
    }
    field_ptr += strlen(field_ptr) + 1;
    
    field_ptr = DX_GetNextField(field_ptr);
    if (*field_ptr)
    {
        total_size = atoi(field_ptr);
    }
    
    error(E_TRACE, "Read file with URL \"%s\", local path \"%s\", "
          "login \"%s\", password \"%s\", flags \"%s\", size %d",
          url, local_path, login, password, flags, total_size);
    
    if (!strcmp(url, ""))
    {
        error(E_WARN, "Couldn't add file - URL was not specified");
        return NULL;
    }
    
    DX_ParseUrl(url, protocol, serv_buf, sizeof(serv_buf), path, sizeof(path));
    
    // do we know about the protocol?
    if (strcasecmp(protocol, "ftp") && strcasecmp(protocol, "http") &&
        strcasecmp(protocol, "file") && strcasecmp(protocol, ""))
    {
        error(E_WARN, "Sorry, don't know about the \"%s\" protocol", protocol);
        return NULL;
    }
    
    // We allow no filename when it's an HTTP download to a known local file
    // (note that this also applies to darxget downloads to stdout). We also
    // allow it when we're uploading.
    if (lastchr(path) == '/')
    {
        // if it's an upload
        if (strchr(flags, 'u'))
        {
            // if we've been told to upload a directory ending in a /,
            // remove it
            if ((lastchr(local_path) == '/') && strchr(flags, 'd'))
                lastchr(local_path) = '\0';
            // if we don't know the local path
            if ((!strcmp(local_path, "") ||
                 (lastchr(local_path) == '/') || !strcmp(local_path, ".") ||
                 !strcmp(local_path, "..")))
            {
                error(E_WARN, "File to upload not specified");
                return NULL;
            }
            else
            {
                strcat(path, BASENAME(local_path));
                error(E_TRACE, "Actually uploading to %s", path);
            }
        }
        // if it's a non-recursive FTP download
        else if (!strcasecmp(protocol, "ftp") && !strchr(flags, 'd'))
        {
            error(E_WARN, "No file specified (path is %s)", path);
            return NULL;
        }
        // if it's an HTTP download without a local path
        else if (!strcasecmp(protocol, "http") && (!strcmp(local_path, "") ||
            (lastchr(local_path) == '/') || !strcmp(local_path, ".") ||
            !strcmp(local_path, "..")))
        {
            // if it's a recursive download, we must fake a filename
            if (strchr(flags, 'd'))
            {
                error(E_TRACE, "Faking local path for recursive download");
                strcat(local_path, "index.html");
            }
            else
            {
                error(E_WARN, "You must specify a local filename for "
                      "%s://%s%s", protocol, serv_buf, path);
                return NULL;
            }
        }
    }
    if (strcasecmp(protocol, "file"))
    {
        // flag "w" is used to enable/disable writing to the server
        // database on disk
        if ((DX_EnableWritingServerDB && !strchr(flags, 'W')) ||
            strchr(flags, 'w'))
        {
            server = FindServer(serv_buf, TRUE);
        }
        else
        {
            server = FindServer(serv_buf, FALSE);
        }
        error(E_TRACE, "found server: %d", server);
        if (server == NULL)
            return NULL;
    }
    else
    {
        // if it's a local "file://" url we do some special magic
        snprintf(buffer, sizeof(buffer), "%s%s", serv_buf, path);
        safecpy(path, buffer);
        server = DummyServer;
    }
    
    // Check for duplicate remote files
    // buf_ptr now points to the remote file path
    error(E_TRACE, "Checking for duplicates");
    file_ptr = Batch.FirstFile;
    while (file_ptr && !strchr(flags, 'v'))
    {
        if ((server == file_ptr->Server) && !strcasecmp(path, file_ptr->Path)
            && !strcasecmp(protocol, file_ptr->Protocol))
        {
            error(E_WARN, "File \"%s://%s%s\" is a duplicate - not adding",
                  protocol, server->Name, path);
            return NULL;
        }
        file_ptr = file_ptr->Next;
    }
    
    // allocate space for the file and clear it
    new_file = dxmalloc(sizeof(FileInfo));
    if (Batch.FirstFile == NULL)
        Batch.FirstFile = new_file;
    else
        last_file->Next = new_file;
    new_file->Prev = last_file;
    new_file->Next = NULL;
    ClearFileInfo(new_file);
    
    // if the local path was specified
    if (strcmp(local_path, ""))
    {
        // check if we're downloading to a special case
        if (!strcmp(local_path, ".") || !strcmp(local_path, ".."))
        {
            sprintf(new_file->LocalPath, "%s/%s", local_path, BASENAME(path));
        }
        else
        {
            ExpandPath(local_path, sizeof(local_path));
            if (*local_path != '/')
            {
                snprintf(new_file->LocalPath, sizeof(new_file->LocalPath),
                         "%s/%s", DX_OutputDir, local_path);
            }
            else
            {
                safecpy(new_file->LocalPath, local_path);
            }
            // if the path's a directory, append the filename
            if (lastchr(new_file->LocalPath) == '/')
            {
                strcat(new_file->LocalPath, BASENAME(path));
            }
        }
        if (strchr(flags, '!'))
        {
            // if we're in pr0n mode, then randomise the end of the path
            srand(time(NULL));
            safecpy(buffer, BASENAME(new_file->LocalPath));
            for (i = 0; i < strlen(buffer); i++)
            {
                buffer[i] = buffer[i] +
                    ((int)(11.0 * rand() / (RAND_MAX+1.0))) - 5;
                if ((buffer[i] == '/') || (buffer[i] == '"'))
                    buffer[i] = '?';
            }
            buffer[0] = '.';
            strcpy(BASENAME(new_file->LocalPath), buffer);
        }
    }
    else
    {
        if (!strchr(flags, '!'))
        {
            // create a suitable-looking local path
            snprintf(new_file->LocalPath, sizeof(new_file->LocalPath) - 1,
                     "%s/%s", DX_OutputDir, BASENAME(path));
            DX_StripControlChars(new_file->LocalPath);
        }
        else
        {
            // if we're in pr0n mode, then randomise the characters
            srand(time(NULL));
            strcpy(buffer, BASENAME(path));
            for (i = 0; i < strlen(buffer); i++)
            {
                buffer[i] = buffer[i] +
                    ((int)(11.0 * rand() / (RAND_MAX+1.0))) - 5;
                if ((buffer[i] == '/') || (buffer[i] == '"'))
                    buffer[i] = '?';
            }
            buffer[0] = '.';
            snprintf(new_file->LocalPath, sizeof(new_file->LocalPath) - 1,
                     "%s/%s", DX_OutputDir, buffer);
        }
        // Avoid duplicate local files by appending a number to the filename
        file_ptr = Batch.FirstFile;
        while (file_ptr)
        {
            if ((file_ptr != new_file) && (!file_ptr->Complete) &&
                !strcmp(new_file->LocalPath, file_ptr->LocalPath))
            {
                dup_count++;
            }
			file_ptr = file_ptr->Next;
        }
        if (dup_count > 0)
        {
            sprintf(buffer, "%s-dx%d", BASENAME(path), dup_count + 1);
            error(E_INFO, "File \"%s://%s%s\" is a duplicate - "
                  "renaming it to \"%s\"", protocol, server->Name, path,
                  buffer);
        }
    }
    
    // flag "s" is used to enable/disable spooling, but we can't spool if we
    // don't know the local path - we never spool for uploads
    if (((DX_EnableSpooling && !strchr(flags, 'S')) || strchr(flags, 's')) &&
        (lastchr(path) != '/') && !strchr(flags, 'u'))
    {
        snprintf(new_file->SpooledPath, sizeof(new_file->SpooledPath) - 1,
                 "%s/%s", DX_SpoolDir, BASENAME(path));
    }
    else
    {
        safecpy(new_file->SpooledPath, new_file->LocalPath);
    }
    ExpandPath(new_file->LocalPath, sizeof(new_file->LocalPath));
    error(E_TRACE, "Local filename is %s", new_file->LocalPath);
    
    if (!strcmp(protocol, ""))
        safecpy(new_file->Protocol, "ftp");
    else
        safecpy(new_file->Protocol, protocol);
    
    strcpy(new_file->Path, path);
    // use defaults if we're given blanks
    if (!strcmp(login, ""))
        safecpy(new_file->LogIn, server->UserName);
    else
        safecpy(new_file->LogIn, login);
    if (!strcmp(password, ""))
        safecpy(new_file->Password, server->Password);
    else
        safecpy(new_file->Password, password);
    safecpy(new_file->Flags, flags);
    if (strchr(flags, 'u'))
        new_file->Upload = TRUE;
    new_file->Server = server;
    new_file->ActualServer = server;
    new_file->NextMirror = -1;
    if (!new_file->Upload)
        new_file->TotalSize = total_size;
    new_file->CurrentSize = 0;
    new_file->Complete = FALSE;
    new_file->Started = FALSE;
    new_file->Starting = FALSE;
    new_file->GotHeader = FALSE;
    
    // Create an ID for the file
    do {
        id_duplicate = FALSE;
        // # is the first valid character we can use
        for (i = 0; i < 3; i++)
        {
            // don't allow a \ in the ID (could confuse clients)
            do {
                new_file->Id[i] = (rand() % 88) + '#';
            } while (new_file->Id[i] == '\\');     
        }
        new_file->Id[3] = '\0';
        file_ptr = Batch.FirstFile;
        while (file_ptr)
        {
            if (!strcmp(file_ptr->Id, new_file->Id) && (file_ptr != new_file))
            {
                id_duplicate = TRUE;
                break;
            }
            file_ptr = file_ptr->Next;
        }
    } while (id_duplicate);
    error(E_TRACE, "New file has ID \"%s\"", new_file->Id);
    
    if (strchr(flags, 'p'))
    {
        new_file->Paused = TRUE;
        strcpy(new_file->Activity, "Paused");
    }
    else
    {
        if ((Batch.FileCount >= DX_MaxSimDownloads) &&
            (DX_MaxSimDownloads > 0) && strcasecmp(new_file->Protocol, "file"))
        {
            new_file->Paused = TRUE;
            new_file->WouldTransfer = TRUE;
            strcpy(new_file->Activity, "Paused");
            error(E_TRACE, "New file paused because there are >%d "
                  "simultaneous transfers", DX_MaxSimDownloads);
        }
        else
        {
            if (!strchr(flags, 'v'))
                Batch.FileCount++;
            Batch.Complete = FALSE;
            new_file->Paused = FALSE;
            WakeUpNow = TRUE;
            strcpy(new_file->Activity, "Initialising...");
            error(E_TRACE, "Setting complete flag to false");
        }
    }
    new_file->Client = client;
    
    // if it's an upload, check whether the local file exists
    if (strchr(flags, 'u'))
    {
        if (stat(new_file->LocalPath, &stat_buf) != -1)
        {
            new_file->TotalSize = stat_buf.st_size;
            error(E_TRACE, "File size: %d", stat_buf.st_size);
            if (strchr(flags, 'd') && !S_ISDIR(stat_buf.st_mode))
                RemoveFlag(new_file->Flags, 'd');
            // FIXME: should remove if try to upload a directory
        }
        else
        {
            error(E_WARN, "Local file \"%s\" does not exist",
                  new_file->LocalPath);
            // FIXME: is this right?
            FileComplete(new_file, "File does not exist");
            return NULL;
        }
    }
    
    // Send an event to any clients that asked for it
    memclr(buffer);
    snprintf(buffer, sizeof(buffer) - 1, "%d \"%s://%s%s\" | \"%s\" | %d | %d"
             " | %d | %d | \"%s\" | %s\n", DX_NEW_FILE_EVENT,
             new_file->Protocol, new_file->Server->Name, new_file->Path,
             new_file->LocalPath, 0, 0, 0, 0, "Sleeping", new_file->Id);
    SendEvent(NEW_FILE_EVENT, buffer);
    
    // only add it to disk if it's got a server
    if (add_to_file && (new_file->Server != DummyServer))
    {
        error(E_TRACE, "Adding file to on-disk batch");
        memclr(buffer);
        snprintf(buffer, sizeof(buffer) - 1, "%s/batch", DX_ProgDir);
        // FIXME: I highly doubt if this is correct!
        LOCK_MUTEX(BatchMutex);
        file = fopen(buffer, "a");
        if (!file)
        {
            error(E_WARN,
                  "Couldn't add file \"%s://%s%s\" to on-disk batch: %s",
                  protocol, server->Name, new_file->Path, strerror(errno));
        }
        else
        {
            sprintf(line_buffer, "\"%s://%s%s\" | \"%s\" | %s | %s | %s | "
                    "%d\n", new_file->Protocol, server->Name,
                    new_file->Path, new_file->LocalPath,
                    new_file->LogIn, new_file->Password,
                    new_file->Flags, new_file->TotalSize);
            fwrite(line_buffer, 1, strlen(line_buffer), file);
            fclose(file);
        }
        chmod(buffer, S_IRUSR | S_IWUSR);
        //error(E_TRACE, "m_count: %d, kind: %d", BatchMutex.__m_count,
        //      BatchMutex.__m_kind);
        //error(E_TRACE, "addr: %d", &BatchMutex);
        UNLOCK_MUTEX(BatchMutex);
    }
    error(E_TRACE, "Returning");
    return new_file;
}

// Wipes all a file's fields back to scratch
void ClearFileInfo(FileInfo *file)
{
    strcpy(file->Id, "AAA");
    memclr(file->Protocol);
    memclr(file->Path);
    memclr(file->LocalPath);
    memclr(file->SpooledPath);
    memclr(file->Activity);
    memclr(file->Flags);
    memclr(file->LogIn);
    memclr(file->Password);
    file->Server = file->ActualServer = NULL;
    file->Complete = TRUE;
    file->Started = file->Starting = file->Paused = FALSE;
    file->Upload = file->WouldTransfer = file->GotHeader = FALSE;
    file->TotalSize = file->CurrentSize = file->RxTxOverall = 0;
    file->StartOffset = file->TimeOfLastRxTx = file->StartTime = 0;
    file->NextMirror = -1;
    file->LocalFile = NULL;
    file->Client = NULL;
    file->ControlSocket = NULL;
    file->DataSocket = NULL;
    file->MassiveMaster = NULL;
    file->MassiveFiles = NULL;
}

// Read the transfer batch file in
int ReadBatchFile(const char *batch_file)
{
    FILE *file;
    char buffer[1024], *buf_ptr;

    Batch.FileCount = 0;
    Batch.Complete = 1;

    file = fopen(batch_file, "r");

    if (file == NULL)
    {
        error(E_TRACE, "Couldn't open batch file \"%s\"", batch_file);
        return Batch.FileCount;
    }

    while (DX_ReadFileLine(buffer, sizeof(buffer), file))
    {
        strip(buffer);
        
        if (!(buf_ptr = DX_StripComments(buffer)))
            continue;
        
        //error(E_TRACE, "Batch file: Read line \"%s\"", buf_ptr);
        AddFileToBatch(FALSE, buf_ptr, NULL);
        error(E_TRACE, "Added OK");
    }
    fclose(file);
    
    // keep it simple to understand!
    if (Batch.FileCount == 0)
        Batch.Complete = TRUE;
    else
        Batch.Complete = FALSE;
    
    return Batch.FileCount;
}

void ReadDataFiles(void)
{
    FileInfo *file = Batch.FirstFile, *last;
    char buffer[256];
    int rc;

    while (file && file->Next)
    {
        last = file;
        file = file->Next;
        dxfree(last);
    }
    Batch.FirstFile = NULL;
    
    sprintf(buffer, "%s/servers", DX_ProgDir);
    rc = ReadServersFile(buffer);
    error(E_TRACE, "%d server(s) read", rc);
    //ReadFileMirrors(buffer);
    sprintf(buffer, "%s/batch", DX_ProgDir);
    rc = ReadBatchFile(buffer);
    error(E_TRACE, "%d file(s) in batch", rc);
}

// Rewrites a file's information in the on-disk batch
void RewriteFileInfo(FileInfo *file, const char *new_line)
{
    char *buf_ptr, line_buffer[1024], buffer[1024], buffer2[256];
    char url[512], protocol[10], serv_buf[256], path[256];
    FILE *batch_file, *new_batch_file;
    
    LOCK_MUTEX(BatchMutex);
    error(E_TRACE, "Rewriting file: looking for %s://%s%s", file->Protocol,
          file->Server->Name, file->Path);
    sprintf(buffer, "%s/batch", DX_ProgDir);
    batch_file = fopen(buffer, "r");
    if (batch_file == NULL)
    {
        error(E_WARN, "Couldn't open batch file %s/batch", DX_ProgDir);
        UNLOCK_MUTEX(BatchMutex);
        return;
    }
    
    sprintf(buffer, "%s/batch.new", DX_ProgDir);
    new_batch_file = fopen(buffer, "w");
    if (new_batch_file == NULL)
    {
        error(E_WARN, "Couldn't open temporary output file %s/batch.new",
              DX_ProgDir);
        UNLOCK_MUTEX(BatchMutex);
        return;
    }
    //error(E_TRACE, "Opened files OK");
    while (DX_ReadFileLine(line_buffer, sizeof(line_buffer), batch_file))
    {
	strip(line_buffer);
        //error(E_TRACE, "Read line %s", line_buffer);
        if (!(buf_ptr = DX_StripComments(line_buffer)))
        {
            fputs(line_buffer, new_batch_file);
            putc('\n', new_batch_file);
            continue;
        }
        
        memset(buffer, 0, sizeof(buffer));
        strcpy(buffer, line_buffer);
        buf_ptr = DX_GetNextField(buffer);
        if (*buf_ptr)
        {
            if (*buf_ptr == '"')
                buf_ptr++;
            strcpy(url, buf_ptr);
            if (lastchr(url) == '"')
                lastchr(url) = '\0';
        }
        DX_ParseUrl(url, protocol, serv_buf, sizeof(serv_buf), path,
                    sizeof(path));
        //error(E_TRACE, "Read server %s, path %s, protocol %s", serv_buf,
        //      path, protocol);
        // add the server's default path if necessary
        if (strncmp(file->ActualServer->DefaultPath, path,
                    strlen(file->ActualServer->DefaultPath)))
        {
            sprintf(buffer, "%s%s", file->ActualServer->DefaultPath, path);
        }
        else
        {
            strcpy(buffer, path);
        }
        // is it the right file?
        if (!strcmp(serv_buf, file->Server->Name) &&
            !strcmp(buffer, file->Path) &&
            !strcmp(protocol, file->Protocol))
        {
                error(E_TRACE, "Found file \"%s%s\": rewriting to \"%s\"...",
                      file->Server->Name, file->Path, new_line);
                if (strcmp(new_line, ""))
                {
                    fputs(new_line, new_batch_file);
                    putc('\n', new_batch_file);
                }
                continue;
        }
        fputs(line_buffer, new_batch_file);
        putc('\n', new_batch_file);
    }
    fclose(batch_file);
    fclose(new_batch_file);
    sprintf(buffer, "%s/batch", DX_ProgDir);
    sprintf(buffer2, "%s/batch.new", DX_ProgDir);
    remove(buffer);
    rename(buffer2, buffer);
    chmod(buffer, S_IRUSR | S_IWUSR);
    UNLOCK_MUTEX(BatchMutex);
}

void RemoveFromBatch(FileInfo *file)
{
    error(E_TRACE, "Removing \"%s\" from batch", file->LocalPath);
    
    // remove the file from tbe linked list
    if (file->Prev)
        file->Prev->Next = file->Next;
    else
        Batch.FirstFile = file->Next;
    if (file->Next)
        file->Next->Prev = file->Prev;
    
    // remove it from the on-disk batch if it isn't a massive child (which
    // are never added to the disk batch)
    error(E_TRACE, "about to rewrite");
    
    if (file->Server && !strchr(file->Flags, 'v'))
        RewriteFileInfo(file, "");
    
    error(E_TRACE, "prev->next: %d, next->prev: %d", file->Next, file->Prev);
    
    dxfree(file);
}

void FileComplete(FileInfo *file, const char *reason)
{
    char buffer[1024], buffer2[256];
    int i, extlen;
    FILE *log_file;
    time_t now;
    struct tm *time_now;
    MassiveFileList *massive_file, *last_massive_file;
    FileInfo *check_file;

    error(E_TRACE, "FileComplete(%s)", reason);
    file->Complete = TRUE;
    file->FinishTime = time(NULL);
    strcpy(file->Activity, "Complete");
    /* FIXME: we probably shouldn't parse the reason */
    if (strcasecmp(reason, "File moved") && (!strchr(file->Flags, 'v')))
    {
        if (!strcasecmp(reason, "File cancelled"))
        {
            sprintf(buffer, "File %s://%s%s was cancelled",
                    file->Protocol, file->Server ? file->Server->Name : "",
                    file->Path);
        }
        else
        {
            sprintf(buffer, "File %s://%s%s complete: %d bytes",
                    file->Protocol, file->Server ? file->Server->Name : "",
                    file->Path, file->TotalSize);
        }
        // output info if we should output to a file or a program
        if (strcmp(DX_OutputToFile, "") || strcmp(DX_OutputToProgram, ""))
            error(E_INFO, buffer);
        
        // do a recursive download if we have to
        if (!strcasecmp(file->Protocol, "http") && strrchr(file->Path, '.') &&
            (strchr(file->Flags, 'd') || strchr(file->Flags, 'l')))
        {
            strcpy(buffer, strrchr(file->Path, '.'));
            if (!strcasecmp(buffer, ".html") || !strcasecmp(buffer, ".htm") ||
                !strcasecmp(buffer, ".shtml") || lastchr(file->Path) == '/')
            {
                if (strchr(file->Flags, 'd'))
                    HttpFindLinks(file, TRUE);
                else
                    HttpFindLinks(file, FALSE);
            }
        }
        
        // notify the client which wanted the file if necessary
        if (strchr(file->Flags, 'n') && file->Client)
        {
            strcat(buffer, "\n");
            write(file->Client->Socket, buffer, strlen(buffer) + 1);
        }
		
        // auto-run a program if necessary
        if (!strcasecmp(reason, "File complete") && !file->Upload &&
            strrchr(file->LocalPath, '.'))
        {
            for (i = 0; i < DX_FileTypeCount; i++)
            {
                extlen = strlen(DX_FileType[0][i]);
                if (extlen > 0)
                {
                    memclr(buffer);
                    strncpy(buffer,
                            file->LocalPath + strlen(file->LocalPath) - extlen,
                            extlen);
                }
                else
                {
                    continue;
                }
                if (!strcasecmp(buffer, DX_FileType[0][i]))
                {
                    snprintf(buffer, sizeof(buffer) - 1, "%s %s &",
                             DX_FileType[1][i], file->LocalPath);
                    system(buffer);
                    error(E_TRACE, "Executed %s on %s",
                          DX_FileType[1][i], file->LocalPath);
                }
            }
        }
        // write into the file log if necessary
        if (strcmp(DX_FileLogName, ""))
        {
            error(E_TRACE, "Writing to file log \"%s\"...", DX_FileLogName);
            log_file = fopen(DX_FileLogName, "a");
            if (log_file != NULL)
            {
                if (ftell(log_file) == 0)
                {
                    fprintf(log_file, "# Darxite file log - a list of "
                            "completed files.\n");
                }
                time(&now);
                time_now = localtime(&now);
                fprintf(log_file, "%d/%d/%d %.2d:%.2d | ", time_now->tm_mday,
                        time_now->tm_mon + 1, time_now->tm_year,
                        time_now->tm_hour, time_now->tm_min);
                if (file->TotalSize >= 0)
                    snprintf(buffer, sizeof(buffer)-1, "%d", file->TotalSize);
                else
                    safecpy(buffer, "?");
                fprintf(log_file, "\"%s://%s%s\" | \"%s\" | %d | %s | %d | %d "
                        "| \"%s\" | %s\n",
                        file->Protocol, file->ActualServer->Name, file->Path,
                        file->LocalPath, file->CurrentSize, buffer,
                        (int)(time(NULL) - file->StartTime),
                        (int)(file->RxTxOverall /
                              (time(NULL) - file->StartTime + 1)), reason,
                        file->Id);
                fclose(log_file);
            }
            else
            {
                error(E_WARN, "Couldn't open log file \"%s\": %s",
                      DX_FileLogName, strerror(errno));
            }
            error(E_TRACE, "Finished writing");
        }
        // notify any clients that requested it
        if (file->TotalSize >= 0)
            sprintf(buffer2, "%d", file->TotalSize);
        else
            strcpy(buffer2, "?");
        memclr(buffer);
//Modified a bit by Manuel Clos to include Id
        snprintf(buffer, sizeof(buffer) - 1, "%d \"%s://%s%s\" | \"%s\" | %d "
                 "| %s | %d | %d | \"%s\" | %s\n", DX_FILE_COMPLETION,
                 file->Protocol, file->ActualServer->Name, file->Path,
                 file->LocalPath, file->CurrentSize, buffer2,
                 (int)(time(NULL) - file->StartTime),
                 (int)(file->RxTxOverall / (time(NULL) - file->StartTime + 1)),
                 reason, file->Id);
        SendEvent(FILE_COMPLETION_EVENT, buffer);
    }
    
    if (!file->Paused)
    {
        Batch.FileCount--;
        // catch excessively weird situations before they get weirder
        if (Batch.FileCount < 0)
            Batch.FileCount = 0;
        if (Batch.FileCount == 0)
            Batch.Complete = TRUE;
    }
    
    // if the file is a massive master and we want to remove it
    if ((file->MassiveMaster == file) && !DX_LeaveCompleteFiles)
    {
        // free the massive file list (but not the first one because
        // it will be removed in a minute)
        massive_file = file->MassiveFiles->Next;
        while (massive_file)
        {
            // remove all cbildren if they're still there
            check_file = Batch.FirstFile;
            while (check_file)
            {
                error(E_TRACE, "\"%s\" = \"%s\"?", check_file->SpooledPath,
                      massive_file->LocalPath);
                if (!strcmp(check_file->SpooledPath, massive_file->LocalPath))
                {
                    RemoveFromBatch(check_file);
                    break;
                }
                check_file = check_file->Next;
            }
            dxfree(massive_file->LocalPath);
            last_massive_file = massive_file;
            massive_file = massive_file->Next;
            dxfree(last_massive_file);
        }
    }
    if (!strcasecmp(reason, "File moved") || !DX_LeaveCompleteFiles)
    {
        RemoveFromBatch(file);
    }
    
    // check if there are any files that would be starting
    if (strcasecmp(reason, "File moved") &&
        (Batch.FileCount < DX_MaxSimDownloads))
    {
        file = Batch.FirstFile;
        while (file)
        {
            if (!file->Complete && file->Paused && file->WouldTransfer)
            {
                file->Paused = FALSE;
                file->Started = FALSE;
                strcpy(file->Activity, "Transferring file");
                // use the last mirror
                file->NextMirror--;
                if (file->NextMirror < -1)
                    file->NextMirror = -1;
                error(E_TRACE, "Transferring %s", file->Path);
                Batch.Complete = FALSE;
                Batch.FileCount++;
                WakeUpNow = TRUE;
                break;
            }
            file = file->Next;
        }
    }
}

// Removes a flag from a file's flags
char *RemoveFlag(char *old_flags, char to_remove)
{
    char *new_flags, *src, *dst;

    error(E_TRACE, "Removing %c from %s", to_remove, old_flags);
    if (!strchr(old_flags, to_remove))
        return old_flags;
    new_flags = dxmalloc(strlen(old_flags + 1));
    memset(new_flags, 0, sizeof(new_flags));
    src = old_flags;
    dst = new_flags;
    while (*src)
    {
        if (*src != to_remove)
            *dst++ = *src++;
        else
            src++;
    }
    strcpy(old_flags, new_flags);
    dxfree(new_flags);
    error(E_TRACE, "Returning %s", old_flags);
    return old_flags;
}



void *ThreadedMoveFile(FileInfo *file)
{
    char buffer[256];

    // if we're renaming the file, then stick a ".darxite" onto the
    // end of the spooled filename
    if ((DX_EnableRenaming && !strchr(file->Flags, 'R')) ||
        strchr(file->Flags, 'r'))
    {
        snprintf(buffer, sizeof(buffer), "%s.darxite", file->SpooledPath);
    }
    else
    {
        strcpy(buffer, file->SpooledPath);
    }
    CopyFile(file->LocalPath, buffer, FALSE);
    remove(buffer);
    FileComplete(file, "File complete");
    return NULL;
}

// the file passed is the one that has just completed, *not* the master
void *ThreadedMassiveComplete(FileInfo *file)
{
    MassiveFileList *massive_file;
    FileInfo *master = file->MassiveMaster;
    char buffer[256];
    
    error(E_TRACE, "Massive download is complete");
    strcpy(master->Activity, "Finishing massive download...");

    // if we're renaming the master file, then stick a ".darxite" onto the
    // end of the spooled filename
    if ((DX_EnableRenaming && !strchr(master->Flags, 'R')) ||
        strchr(master->Flags, 'r'))
    {
        snprintf(buffer, sizeof(buffer), "%s.darxite", master->SpooledPath);
    }
    else
    {
        strcpy(buffer, master->SpooledPath);
    }
    // despool the master file if spooling's enabled
    if (((DX_EnableSpooling && !strchr(master->Flags, 'S')) ||
         strchr(master->Flags, 's')) && (lastchr(master->Path) != '/'))
    {
        CopyFile(master->LocalPath, buffer, FALSE);
        //remove(buffer);
    }
    
    // append the others to it
    massive_file = master->MassiveFiles->Next;
    while (massive_file)
    {
        CopyFile(master->LocalPath, massive_file->LocalPath, TRUE);
        //remove(massive_file->LocalPath);
        massive_file = massive_file->Next;
    }
    FileComplete(master, "File complete");
    
    return NULL;
}

BOOL CopyFile(const char *dst, const char *src, BOOL append)
{
    int input_file, output_file;
    char buffer[2048];
    int bytes_read, bytes_written;

    if (!strcmp(dst, src))
        return TRUE;
    input_file = open(src, O_RDONLY);
 
    if (input_file == -1)
    {
        error(E_WARN, "Couldn't copy file \"%s\" to \"%s\": %s opening "
              "input file", src, dst, strerror(errno));
        return FALSE;
    }
    
    CreatePath(dst);
    if (append)
    {
        output_file = open(dst, O_WRONLY | O_APPEND);
        error(E_TRACE, "Appending file %s to %s", src, dst);
    }
    else
    {
        output_file = creat(dst,
                            S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH);
        //error(E_TRACE, "Copying file %s to %s", src, dst);
    }
    
    if (output_file == -1)
    {
        error(E_WARN, "Couldn't copy file \"%s\" to \"%s\": %s opening "
              "output file", src, dst, strerror(errno));
        return FALSE;
    }
    
    do {
        do {
            bytes_read = read(input_file, buffer, sizeof(buffer));
        } while ((bytes_read == -1) && (errno == EINTR));
        
        if (bytes_read == -1)
        {
            error(E_WARN, "Couldn't copy file \"%s\" to \"%s\": %s",
                  src, dst, strerror(errno));
            return FALSE;
        }
        
        do {
            bytes_written = write(output_file, buffer, bytes_read);
        } while ((bytes_written == -1) && (errno == EINTR));
        
        if (bytes_written == -1)
        {
            error(E_WARN, "Couldn't copy file \"%s\" to \"%s\": %s",
                  src, dst, strerror(errno));
            return FALSE;
        }
    } while (bytes_read > 0);
    close(input_file);
    close(output_file);
    //error(E_TRACE, "Copy of %s to %s complete", src, dst);
    return TRUE;
}

// Performs filename expansion (eg. ~ashley) on path.
BOOL ExpandPath(char *path, int path_max)
{
    struct passwd result_buf, *result_ptr;
    char buffer[1024], name[256];
    int rc;

    if (*path != '~')
        return TRUE;
    // if we've just got a tilde (eg. "~/new")
    if ((*(path + 1) == '/') || (*(path + 1) == '\0'))
    {
        sprintf(buffer, "%s%s", DX_HomeDir, path + 1);
        strcpy(path, buffer);
        return TRUE;
    }
    // if we've got a user name (eg. "~ashley/new")
    memset(name, 0, sizeof(name));
    if (!strchr(path, '/'))
        strcpy(name, path + 1);
    else
        strncpy(name, path + 1, strchr(path, '/') - (path + 1));
    rc = getpwnam_r(name, &result_buf, buffer, sizeof(buffer), &result_ptr);
    if (rc != 0)
        return FALSE;
    if (strchr(path, '/'))
        sprintf(buffer, "%s%s", result_buf.pw_dir, strchr(path, '/'));
    else
        sprintf(buffer, "%s/", result_buf.pw_dir);
    strncpy(path, buffer, path_max);
    return TRUE;
    
    // does wordexp actually exist?
    /*wordexp_t wordvec;
    int rc;

    rc = wordexp(path, &wordvec, 0);
    if (rc != 0)
    {
        error(E_TRACE, "wordexp returned %d", rc);
        return FALSE;
    }
    if (wordvec.we_wordc != 1)
    {
        error(E_TRACE, "wordexp returned %d words", wordvec.we_wordc);
        return FALSE;
    }
    error(E_TRACE, "wordexp found the word %s", wordvec.we_wordv[0]);
    strncpy(path, wordvec.we_wordv[0], path_max - 1);
    path[path_max - 1] = '\0';
    wordfree(&wordvec);*/
    return TRUE;
}

// Creates a path - pass it a file and it will
// ensure that its parent dir exists.
// Path must begin with a /.
BOOL CreatePath(const char *path)
{
    int fd;
    char *path_ptr, path_buf[256];
    char part_path_buf[256];

    // if the path is "." then don't bother
    if ((*path == '.') && ((*(path+1) == '/') || (*(path+1) == '\0')))
        return FALSE;
    // chop the filename off the end - if it ends in a / it's a directory
    strcpy(path_buf, path);
    if (strrchr(path_buf, '/') && (strrchr(path_buf, '/') !=
				   (path_buf + strlen(path_buf))))
    {
        *strrchr(path_buf, '/') = '\0';
    }
    
    fd = open(path_buf, O_RDONLY);
    if ((fd > -1) || (errno != ENOENT))
    {
        close(fd);
        return FALSE;
    }
    // we DO need to create some directories
    strcpy(part_path_buf, "/");
    path_ptr = strtok(path_buf, "/");
    while (path_ptr)
    {
        strcat(part_path_buf, path_ptr);
        fd = open(part_path_buf, O_RDONLY);
        if ((fd == -1) && (errno == ENOENT))
        {
            error(E_TRACE, "Creating directory %s", part_path_buf);
            if (mkdir(part_path_buf, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) <
                0)
            {
                error(E_WARN, "Couldn't create directory \"%s\": %s",
                      part_path_buf, strerror(errno));
                return FALSE;
            }
        }
        else
        {
            close(fd);
        }
        strcat(part_path_buf, "/");
        path_ptr = strtok(NULL, "/");
    }
    return TRUE;
}
