/*****************************************************************************/
/*  gftp.c - the main user interface for the ftp program                     */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"
#include "pix/connect.xpm"
#include "pix/left.xpm"
#include "pix/right.xpm"
#include "pix/stop.xpm"
#include "pix/up.xpm"
#include "pix/down.xpm"
#include "options.h"

char version[] = "gFTP " VERSION;

static void menu_exit (GtkWidget *widget, gpointer data);
static gint delete_event (GtkWidget *widget, GdkEvent *event, gpointer data);
static void doexit (GtkWidget *widget, gpointer data);
static int get_column (GtkCListColumn *col);
static void destroy (GtkWidget *widget, gpointer data);
static void sig_child (int signo);
static void init_gftp (int argc, char *argv[], GtkWidget *parent);
static void usage (void);
static GtkWidget *CreateFTPWindows (GtkWidget *ui);
static GtkWidget *CreateToolbar (GtkWidget *parent);
static GtkWidget *CreateFTPWindow (struct ftp_window_data *wdata,
	GtkItemFactoryEntry *entries, int num_menu_items);
static void setup_column (GtkWidget *listbox, int column, int width);
static void change_setting (struct ftp_window_data *wdata, int menuitem, GtkWidget *checkmenu);
static void selectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data);
static void unselectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data);
static void selectall (struct ftp_window_data *wdata);
static void selectallfiles (struct ftp_window_data *wdata);
static void deselectall (struct ftp_window_data *wdata);
static gint log_click (GtkWidget *widget, GdkEventButton *event, gpointer data);
static gint list_dblclick (GtkWidget *widget, GdkEventButton *event, gpointer data);
static gint list_enter (GtkWidget *widget, GdkEventKey *event, gpointer data);
static void chfunc (struct ftp_window_data *wdata);
static int chdirfunc (struct ftp_window_data *wdata);
static void disconn (struct ftp_window_data *wdata);
static void compare_windows (gpointer data);

int main (int argc, char *argv[]) {
   GtkWidget *window, *ui;

#ifdef HAVE_GETTEXT
   setlocale (LC_ALL, "");
   bindtextdomain ("gftp", LOCALE_DIR);
   textdomain ("gftp");
#endif
   gtk_set_locale ();
   
   gtk_init (&argc, &argv);
   signal (SIGCHLD, sig_child);
   signal (SIGPIPE, SIG_IGN);
   if (argc > 1) {
      if (strcmp (argv[1], "--help") == 0 || strcmp (argv[1], "-h") == 0) {
         usage ();
      }
      else if (strcmp (argv[1], "--version") == 0 || strcmp (argv[1], "-v") == 0) {
         printf ("%s\n", version);
         exit (0);
      }
   }

   read_config_file ();
   window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   gtk_signal_connect (GTK_OBJECT (window), "delete_event", 
   	GTK_SIGNAL_FUNC (delete_event), NULL);
   gtk_signal_connect (GTK_OBJECT (window), "destroy", 
   	GTK_SIGNAL_FUNC (destroy), NULL);
   gtk_window_set_title (GTK_WINDOW (window), version);
   gtk_window_set_policy (GTK_WINDOW (window), TRUE, TRUE, FALSE);
   gtk_widget_realize (window);
   
   ui = CreateFTPWindows (window);
   gtk_container_add (GTK_CONTAINER (window), ui);
   gtk_widget_show (window);

   ftp_log (gftp_logging_misc, NULL, "%s, Copyright (C) 1998-1999 Brian Masney <", version);
   ftp_log (gftp_logging_recv, NULL, "masneyb@seul.org");
   ftp_log (gftp_logging_misc, NULL, _(">. If you have any questions, comments, or suggestions about this program, please feel free to email them to me. You can always find out the latest news about gFTP from my website at http://gftp.seul.org/\n"));
   ftp_log (gftp_logging_misc, NULL, _("gFTP comes with ABSOLUTELY NO WARRANTY; for details, see the COPYING file. This is free software, and you are welcome to redistribute it under certain conditions; for details, see the COPYING file\n"));
   gtk_timeout_add (1000, update_downloads, NULL);
   init_gftp (argc, argv, window);
   add_local_files (&window1); 
   gtk_main ();
   return (0);
}  
/*****************************************************************************/
static void menu_exit (GtkWidget *widget, gpointer data) {
   if (!delete_event (widget, NULL, data)) {
      doexit (widget, data);
   }
}
/*****************************************************************************/
static gint delete_event (GtkWidget *widget, GdkEvent *event, gpointer data) {
   if (file_transfers == NULL) doexit (widget, data);
   else {
      MakeYesNoDialog (_("Exit"), _("There are file transfers in progress.\nAre you sure you want to exit?"), 1, 2,
      	_("Exit"), doexit, NULL, _("Don't Exit"), NULL, NULL);
      return (TRUE);
   }
   return (FALSE);
}
/*****************************************************************************/
static void doexit (GtkWidget *widget, gpointer data) {
   if (save_geometry) {
      listbox_local_width = GTK_WIDGET (local_frame)->allocation.width;
      listbox_remote_width = GTK_WIDGET (remote_frame)->allocation.width;
      listbox_file_height = GTK_WIDGET (remote_frame)->allocation.height;
      log_height = GTK_WIDGET (log_table)->allocation.height;
      transfer_height = GTK_WIDGET (transfer_scroll)->allocation.height;
      
      local_columns[0] = get_column (&GTK_CLIST (window1.listbox)->column[1]);
      local_columns[1] = get_column (&GTK_CLIST (window1.listbox)->column[2]);
      local_columns[2] = get_column (&GTK_CLIST (window1.listbox)->column[3]);
      local_columns[3] = get_column (&GTK_CLIST (window1.listbox)->column[4]);
      local_columns[4] = get_column (&GTK_CLIST (window1.listbox)->column[5]);
      local_columns[5] = get_column (&GTK_CLIST (window1.listbox)->column[6]);
      
      remote_columns[0] = get_column (&GTK_CLIST (window2.listbox)->column[1]);
      remote_columns[1] = get_column (&GTK_CLIST (window2.listbox)->column[2]);
      remote_columns[2] = get_column (&GTK_CLIST (window2.listbox)->column[3]);
      remote_columns[3] = get_column (&GTK_CLIST (window2.listbox)->column[4]);
      remote_columns[4] = get_column (&GTK_CLIST (window2.listbox)->column[5]);
      remote_columns[5] = get_column (&GTK_CLIST (window2.listbox)->column[6]);
      
      file_trans_column = get_column (&GTK_CLIST (dlwdw)->column[0]);
   }
   write_config_file ();
   clear_cache_files ();
   exit (0);
}
/*****************************************************************************/
static int get_column (GtkCListColumn *col) {
   if (col->auto_resize) return (0);
   else if (!col->visible) return (-1);
   else return (col->width);
}
/*****************************************************************************/
static void destroy (GtkWidget *widget, gpointer data) {
   exit (0);
}
/*****************************************************************************/
static void sig_child (int signo) {
   viewedit_process_done = 1;
   signal (SIGCHLD, sig_child);
}
/*****************************************************************************/
void init_gftp (int argc, char *argv[], GtkWidget *parent) {
   struct pix_ext *tempext;
   GtkWidget *sort_wid;
   struct utsname unme;
   struct passwd *pw;
   struct hostent *hent;

   window1.local = window2.local = -1;
   if (emailaddr == NULL || *emailaddr == '\0') {
      /* If there is no email address specified, then we'll just use the
         currentuser@currenthost */
      pw = getpwuid (geteuid ());
      uname (&unme);
      hent = gethostbyname (unme.nodename);
      if (strchr (unme.nodename, '.') == NULL && hent != NULL) {
         emailaddr = g_strconcat (pw->pw_name, "@", hent->h_name, NULL);
      }
      else {
         emailaddr = g_strconcat (pw->pw_name, "@", unme.nodename, NULL);
      }
      write_config_file ();
   }
                        
   open_xpm ("dotdot.xpm", parent, &dotdot_pixmap, &dotdot_mask, 1);
   open_xpm ("dir.xpm", parent, &dir_pixmap, &dir_mask, 1);
   open_xpm ("open_dir.xpm", parent, &open_dir_pixmap, &open_dir_mask, 1);
   open_xpm ("linkdir.xpm", parent, &linkdir_pixmap, &linkdir_mask, 1);
   open_xpm ("linkfile.xpm", parent, &linkfile_pixmap, &linkfile_mask, 1);
   open_xpm ("exe.xpm", parent, &exe_pixmap, &exe_mask, 1);
   open_xpm ("doc.xpm", parent, &doc_pixmap, &doc_mask, 1);
   
   if (window1.sortasds) sort_wid = toolbar_pixmap (window1.listbox, down_xpm);
   else sort_wid = toolbar_pixmap (window1.listbox, up_xpm);
   gtk_clist_set_column_widget (GTK_CLIST (window1.listbox), 0, sort_wid);

   if (window2.sortasds) sort_wid = toolbar_pixmap (window2.listbox, down_xpm);
   else sort_wid = toolbar_pixmap (window2.listbox, up_xpm);
   gtk_clist_set_column_widget (GTK_CLIST (window2.listbox), 0, sort_wid);

   tempext = registered_exts;
   while (tempext != NULL) {
      tempext->pixmap = NULL;
      open_xpm (tempext->filename, parent, &tempext->pixmap, &tempext->mask, 1);
      tempext = tempext->next;
   }
   if (argc > 2) usage ();
   else if (argc == 2) {
      if (gftp_parse_url (window2.hdata->ftpdata, argv[1]) == 0) {
         if (strcmp (GFTP_GET_USERNAME (window2.hdata->ftpdata), ANON_LOGIN) == 0) {
            gftp_set_password (window2.hdata->ftpdata, emailaddr);
         }
         ftp_connect (window2.hdata, 1);
      }
      else usage ();
   }
}
/*****************************************************************************/
static void usage (void) {
   printf (_("usage: gftp [[ftp://][user:pass@]ftp-site[:port][/directory]]\n"));
   exit (0);
}
/*****************************************************************************/
static GtkWidget *CreateFTPWindows (GtkWidget *ui) {
   GtkWidget *box, *dlbox, *winpane, *dlpane, *logpane, *mainvbox, 
   	*tempwid, *button;
   int i, remote_start, remote_len, local_start, local_len, trans_start, 
   	trans_len, log_start, log_len;
   GtkItemFactory *log_factory, *dl_factory;
   GtkAccelGroup *accel_group;
   char *dltitles[2];
   static GtkItemFactoryEntry dummy_item, menu_items[] = {
      {N_("/_FTP"),			NULL,	0,		0,	"<Branch>"},
      {N_("/FTP/tearoff"),		NULL,	0,		0,	"<Tearoff>"},
      {N_("/FTP/Ascii"), 		NULL, 	change_setting,	1,	"<RadioItem>"},
      {N_("/FTP/Binary"), 		NULL, 	change_setting,	2,	N_("/FTP/Ascii")},
      {N_("/FTP/sep"), 			NULL, 	0, 		0,	"<Separator>"},
      {N_("/FTP/_Options..."), 		"<control>O",	options_dialog, 0},
      {N_("/FTP/sep"), 			NULL, 	0, 		0,	"<Separator>"},
      {N_("/FTP/_Quit"), 		"<control>Q", 	menu_exit,	0},
      {N_("/_Local"),			NULL,	0,		0,	"<Branch>"},
      {N_("/Local/tearoff"),		NULL,	0,		0,	"<Tearoff>"},
      {N_("/Local/Change Filespec..."), NULL, change_filespec, 0},
      {N_("/Local/Select All"), 	NULL, 	selectall, 	0},
      {N_("/Local/Select All Files"), 	NULL, 	selectallfiles, 	0},
      {N_("/Local/Deselect All"),	NULL, 	deselectall, 	0},
      {N_("/Local/sep"),		NULL,	0,		0,	"<Separator>"},
      {N_("/Local/Change Directory"), 	NULL,	chfunc, 	0},
      {N_("/Local/Chmod..."),		NULL,	chmod_dialog,	0},
      {N_("/Local/Make Directory..."), 	NULL, mkdir_dialog, 	0},
      {N_("/Local/Rename..."), 		NULL, 	rename_dialog, 	0},
      {N_("/Local/Delete..."), 		NULL, 	delete_dialog, 	0},
      {N_("/Local/Edit..."),		NULL,	edit_dialog,	0},
      {N_("/Local/View..."),		NULL,	view_dialog,	0},
      {N_("/Local/Refresh"), 		NULL, 	refresh, 	0},
      {N_("/_Remote"),			NULL,	0,		0,	"<Branch>"},
      {N_("/Remote/tearoff"),		NULL,	0,		0,	"<Tearoff>"},
      {N_("/Remote/Open _URL..."),	"<control>U",	openurl_dialog,	0},
      {N_("/Remote/_Disconnect"),	"<control>D", 	disconn, 	0},
      {N_("/Remote/sep"),		NULL,	0,		0,	"<Separator>"},
      {N_("/Remote/Change Filespec..."), NULL, change_filespec, 0},
      {N_("/Remote/Select All"), 	NULL, 	selectall, 	0},
      {N_("/Remote/Select All Files"), 	NULL, 	selectallfiles, 	0},
      {N_("/Remote/Deselect All"), 	NULL, 	deselectall, 	0},
      {N_("/Remote/sep"),	NULL,	0,		0,	"<Separator>"},
      {N_("/Remote/Send SITE Command..."), NULL, site_dialog, 0},
      {N_("/Remote/Change Directory"), NULL, chfunc, 	0},
      {N_("/Remote/Chmod..."),	NULL,	chmod_dialog,	0},
      {N_("/Remote/Make Directory..."), NULL, mkdir_dialog,	0},
      {N_("/Remote/Rename..."), 	NULL, 	rename_dialog, 	0},
      {N_("/Remote/Delete..."), 	NULL, 	delete_dialog, 	0},
      {N_("/Remote/Edit..."),		NULL,	edit_dialog,	0},
      {N_("/Remote/View..."),		NULL,	view_dialog,	0},
      {N_("/Remote/Refresh"), 		NULL, 	refresh, 	0},
      {N_("/_Bookmarks"),		NULL,	0,		0,	"<Branch>"},
      {N_("/Bookmarks/tearoff"), 	NULL,	0,		0,	"<Tearoff>"},
      {N_("/Bookmarks/Add bookmark"), 	"<control>A", add_bookmark, 0},
      {N_("/Bookmarks/Edit bookmarks"), NULL, edit_bookmarks,		0},
      {N_("/Bookmarks/sep"),		NULL,	0,		0,	"<Separator>"},
      {N_("/_Transfers"),		NULL,	0,		0,	"<Branch>"},
      {N_("/Transfers/tearoff"),	NULL,	0,		0,	"<Tearoff>"},
      {N_("/Transfers/Start Transfer"), NULL, 	start_transfer, 0},
      {N_("/Transfers/Stop Transfer"), 	NULL, 	stop_transfer,	0},
      {N_("/Transfers/sep"),		NULL,	0,		0,	"<Separator>"},
      {N_("/Transfers/Retrieve Files"), "<control>R",  retrCB,	0},
      {N_("/Transfers/Put Files"),	"<control>P",	putCB,	0},
      {N_("/L_ogging"),			NULL,	0,		0,	"<Branch>"},
      {N_("/Logging/tearoff"),		NULL,	0,		0,	"<Tearoff>"},
      {N_("/Logging/Clear"), 		NULL, 	clearlog, 	0},
      {N_("/Logging/View log..."), 	NULL, 	viewlog, 	0},
      {N_("/Logging/Save log..."), 	NULL, 	savelog, 	0},
      {N_("/Tool_s"),			NULL,	0,		0,	"<Branch>"},
      {N_("/Tools/tearoff"),		NULL,	0,		0,	"<Tearoff>"},
      {N_("/Tools/Compare Windows"), 	NULL, 	compare_windows,  0},
      {N_("/_Help"), 			NULL, 	0,		0,	"<LastBranch>"},
      {N_("/Help/tearoff"),		NULL,	0,		0,	"<Tearoff>"},
      {N_("/Help/About..."), 		NULL, 	about_dialog,	0}};
      
   dltitles[0] = _("Filename");
   dltitles[1] = _("Progress");

   mainvbox = gtk_vbox_new (FALSE, 0);
   gtk_widget_show (mainvbox);

   accel_group = gtk_accel_group_new ();
   factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", accel_group);
   gtk_box_pack_start (GTK_BOX (mainvbox), factory->widget, FALSE, FALSE, 0);
   gtk_widget_show (factory->widget);

   i = 0;
   /* FTP Menu */
   create_item_factory (factory, 8, menu_items, &window2);
   
   i += 8;
   /* Local Menu */
   local_start = i;
   local_len = 15;
   create_item_factory (factory, local_len, menu_items + i, &window1);
   
   i += local_len;
   /* Remote Menu */
   remote_start = i;
   remote_len = 19;
   create_item_factory (factory, remote_len, menu_items + i, &window2);
   
   i += remote_len;
   /* Bookmarks Menu */
   create_item_factory (factory, 5, menu_items + i, &window2);
   
   i += 5;
   /* Transfers Menu */
   trans_start = i;
   trans_len = 7;
   create_item_factory (factory, trans_len, menu_items + i, NULL);
   
   i += trans_len;
   /* Logging Menu */
   log_start = i;
   log_len = 5;
   create_item_factory (factory, log_len, menu_items + i, NULL);
   
   i += log_len;
   /* Tools Menu */
   create_item_factory (factory, 3, menu_items + i, NULL);
   
   i += 3;
   /* Help Menu */
   create_item_factory (factory, 3, menu_items + i, NULL);

   gtk_accel_group_attach (accel_group, GTK_OBJECT (ui));

   tempwid = gtk_item_factory_get_widget (factory, _(menu_items[3].path));
   gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM (tempwid), TRUE);

   build_bookmarks_menu ();

   tempwid = CreateToolbar (ui);
   gtk_box_pack_start (GTK_BOX (mainvbox), tempwid, FALSE, FALSE, 0);
   gtk_widget_show (tempwid);
   
   winpane = gtk_hpaned_new ();
   box = gtk_hbox_new (FALSE, 0);

   window1.local = 1;
   local_frame = CreateFTPWindow (&window1, menu_items + local_start + 2, local_len - 2);
   window1.local = -1;
   gtk_box_pack_start (GTK_BOX (box), local_frame, TRUE, TRUE, 0);
   gtk_widget_show (local_frame);

   dlbox = gtk_vbox_new (FALSE, 0);
   gtk_container_border_width (GTK_CONTAINER (dlbox), 5);
   gtk_box_pack_start (GTK_BOX (box), dlbox, FALSE, FALSE, 0);
   gtk_widget_show (dlbox);

   tempwid = toolbar_pixmap (ui, right_xpm);
   button = gtk_button_new ();
   gtk_box_pack_start (GTK_BOX (dlbox), button, TRUE, FALSE, 0);
   gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (putCB), NULL);
   gtk_container_add (GTK_CONTAINER (button), tempwid);
   gtk_widget_show (button);

   tempwid = toolbar_pixmap (ui, left_xpm);
   button = gtk_button_new ();
   gtk_box_pack_start (GTK_BOX (dlbox), button, TRUE, FALSE, 0);
   gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (retrCB), NULL);
   gtk_container_add (GTK_CONTAINER (button), tempwid);
   gtk_widget_show (button);

   gtk_widget_show (box);
   gtk_paned_pack1 (GTK_PANED (winpane), box, 1, 1);
   
   remote_frame = CreateFTPWindow (&window2, menu_items + remote_start + 2, remote_len - 2);
   gtk_paned_pack2 (GTK_PANED (winpane), remote_frame, 1, 1);
   gtk_widget_show (remote_frame);
   gtk_widget_show (winpane);

   dlpane = gtk_vpaned_new ();
   gtk_paned_pack1 (GTK_PANED (dlpane), winpane, 1, 1);

   transfer_scroll = gtk_scrolled_window_new (NULL, NULL);
   gtk_widget_set_usize (transfer_scroll, 1, transfer_height);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (transfer_scroll),
   	GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

   dl_factory = gtk_item_factory_new (GTK_TYPE_MENU, "<download>", NULL);
   for (i=trans_start + 2; i<trans_start + trans_len; i++) {
      memcpy (&dummy_item, menu_items + i, sizeof (GtkItemFactoryEntry));
      dummy_item.path = strrchr (_(dummy_item.path), '/');
      gtk_item_factory_create_item (dl_factory, &dummy_item, NULL, 1);
   }

   dlwdw = gtk_ctree_new_with_titles (2, 0, dltitles);
   gtk_clist_set_selection_mode (GTK_CLIST (dlwdw), GTK_SELECTION_SINGLE);
   gtk_clist_set_reorderable (GTK_CLIST (dlwdw), 0);
   if (file_trans_column == 0) {
      gtk_clist_set_column_auto_resize (GTK_CLIST (dlwdw), 0, TRUE);
   }
   else {
      gtk_clist_set_column_width (GTK_CLIST (dlwdw), 0, file_trans_column);
   }
   gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (transfer_scroll), dlwdw);
   gtk_signal_connect (GTK_OBJECT (dlwdw), "button_press_event", GTK_SIGNAL_FUNC (dl_click), (gpointer) dl_factory);
   gtk_paned_pack2 (GTK_PANED (dlpane), transfer_scroll, 1, 1);
   gtk_widget_show (dlwdw);
   gtk_widget_show (transfer_scroll);
   gtk_widget_show (dlpane);

   log_factory = gtk_item_factory_new (GTK_TYPE_MENU, "<log>", NULL);
   for (i=log_start + 2; i<log_start + log_len; i++) {
      memcpy (&dummy_item, menu_items + i, sizeof (GtkItemFactoryEntry));
      dummy_item.path = strrchr (_(dummy_item.path), '/');
      gtk_item_factory_create_item (log_factory, &dummy_item, NULL, 1);
   }
   
   logpane = gtk_vpaned_new ();
   gtk_paned_pack1 (GTK_PANED (logpane), dlpane, 1, 1);

   log_table = gtk_table_new (1, 2, FALSE);
   gtk_widget_set_usize (log_table, 1, log_height);
   logwdw = gtk_text_new (NULL, NULL);
   gtk_text_set_editable (GTK_TEXT (logwdw), FALSE);
   gtk_text_set_word_wrap (GTK_TEXT (logwdw), TRUE);
   gtk_table_attach (GTK_TABLE (log_table), logwdw, 0, 1, 0, 1,
   	GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 0);
   gtk_signal_connect (GTK_OBJECT (logwdw), "button_press_event", GTK_SIGNAL_FUNC (log_click), (gpointer) log_factory);
   gtk_widget_show (logwdw);

   tempwid = gtk_vscrollbar_new (GTK_TEXT (logwdw)->vadj);
   gtk_table_attach (GTK_TABLE (log_table), tempwid, 1, 2, 0, 1,
   	GTK_FILL, GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0);
   gtk_widget_show (tempwid);
   gtk_widget_show (log_table);
   
   gtk_paned_pack2 (GTK_PANED (logpane), log_table, 1, 1);
   gtk_widget_show (logpane);      
   gtk_box_pack_start (GTK_BOX (mainvbox), logpane, TRUE, TRUE, 0);

   return (mainvbox);
}   
/*****************************************************************************/
static GtkWidget *CreateToolbar (GtkWidget *parent) {
   const GtkTargetEntry possible_types[] = {
      {"STRING",			0,	0},
      {"text/plain", 			0, 	0},
      {"application/x-rootwin-drop", 	0, 	1}};
   GtkWidget *toolbar, *box, *tempwid, *optionmenu;
   int i;
   
   toolbar = gtk_handle_box_new ();

   box = gtk_hbox_new (FALSE, 4);
   gtk_container_add (GTK_CONTAINER (toolbar), box);
   gtk_container_border_width (GTK_CONTAINER (box), 5);
   gtk_widget_show (box);

   tempwid = toolbar_pixmap (parent, connect_xpm);
   openurl_btn = gtk_button_new ();
   gtk_container_add (GTK_CONTAINER (openurl_btn), tempwid);
   gtk_signal_connect_object (GTK_OBJECT (openurl_btn), "clicked", 
   	GTK_SIGNAL_FUNC (tb_openurl_dialog), NULL);
   gtk_signal_connect (GTK_OBJECT (openurl_btn), "drag_data_received", 
   	GTK_SIGNAL_FUNC (openurl_get_drag_data), NULL);
   gtk_drag_dest_set (openurl_btn, GTK_DEST_DEFAULT_ALL, possible_types, 2, 
   	GDK_ACTION_COPY | GDK_ACTION_MOVE);
   gtk_container_border_width (GTK_CONTAINER (openurl_btn), 1);
   gtk_box_pack_start (GTK_BOX (box), openurl_btn, FALSE, FALSE, 0);
   gtk_widget_show (openurl_btn);

   tempwid = gtk_label_new (_("Host: "));
   gtk_box_pack_start (GTK_BOX (box), tempwid, FALSE, FALSE, 0);
   gtk_widget_show (tempwid);
   
   hostedit = gtk_combo_new ();
   gtk_widget_set_usize (hostedit, 130, -1);
   gtk_signal_connect (GTK_OBJECT (GTK_COMBO (hostedit)->entry), "activate", 
   	GTK_SIGNAL_FUNC (toolbar_hostedit), NULL);
   if (host_history) gtk_combo_set_popdown_strings (GTK_COMBO (hostedit), host_history);
   gtk_combo_disable_activate (GTK_COMBO (hostedit));
   gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (hostedit)->entry), "");
   gtk_box_pack_start (GTK_BOX (box), hostedit, TRUE, TRUE, 0);
   gtk_widget_show (hostedit);

   tempwid = gtk_label_new (_("Port: "));
   gtk_box_pack_start (GTK_BOX (box), tempwid, FALSE, FALSE, 0);
   gtk_widget_show (tempwid);
   
   portedit = gtk_combo_new ();
   gtk_widget_set_usize (portedit, 40, -1);
   gtk_signal_connect (GTK_OBJECT (GTK_COMBO (portedit)->entry), "activate", 
   	GTK_SIGNAL_FUNC (toolbar_hostedit), NULL);
   if (port_history) gtk_combo_set_popdown_strings (GTK_COMBO (portedit), port_history);
   gtk_combo_disable_activate (GTK_COMBO (portedit));
   gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (portedit)->entry), "");
   gtk_box_pack_start (GTK_BOX (box), portedit, FALSE, FALSE, 0);
   gtk_widget_show (portedit);

   tempwid = gtk_label_new (_("User: "));
   gtk_box_pack_start (GTK_BOX (box), tempwid, FALSE, FALSE, 0);
   gtk_widget_show (tempwid);
   
   useredit = gtk_combo_new ();
   gtk_widget_set_usize (useredit, 75, -1);
   gtk_signal_connect (GTK_OBJECT (GTK_COMBO (useredit)->entry), "activate", 
   	GTK_SIGNAL_FUNC (toolbar_hostedit), NULL);
   if (user_history) gtk_combo_set_popdown_strings (GTK_COMBO (useredit), user_history);
   gtk_combo_disable_activate (GTK_COMBO (useredit));
   gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (useredit)->entry), "");
   gtk_box_pack_start (GTK_BOX (box), useredit, TRUE, TRUE, 0);
   gtk_widget_show (useredit);

   tempwid = gtk_label_new (_("Pass: "));
   gtk_box_pack_start (GTK_BOX (box), tempwid, FALSE, FALSE, 0);
   gtk_widget_show (tempwid);

   passedit = gtk_entry_new ();
   gtk_widget_set_usize (passedit, 55, -1);
   gtk_entry_set_visibility (GTK_ENTRY (passedit), FALSE);
   gtk_signal_connect (GTK_OBJECT (passedit), "activate", 
   	GTK_SIGNAL_FUNC (toolbar_hostedit), NULL);
   gtk_box_pack_start (GTK_BOX (box), passedit, FALSE, FALSE, 0);
   gtk_widget_show (passedit);
   
   tempwid = gtk_vbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (box), tempwid, FALSE, FALSE, 0);
   gtk_widget_show (tempwid);
   
   optionmenu = gtk_option_menu_new ();
   gtk_box_pack_start (GTK_BOX (tempwid), optionmenu, TRUE, FALSE, 0);
   gtk_widget_show (optionmenu);

   protocol_menu = gtk_menu_new ();
   for (i=0; gftp_protocols[i].name; i++) {
      tempwid = gtk_menu_item_new_with_label (gftp_protocols[i].name);
      gtk_object_set_user_data (GTK_OBJECT (tempwid), (gpointer) i);
      gtk_menu_append (GTK_MENU (protocol_menu), tempwid);
      gtk_widget_show (tempwid);
   }
   gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu), protocol_menu);
   gtk_option_menu_set_history (GTK_OPTION_MENU (optionmenu), 0);
   
   tempwid = toolbar_pixmap (parent, stop_xpm);   
   stop_btn = gtk_button_new ();
   gtk_container_add (GTK_CONTAINER (stop_btn), tempwid);
   gtk_widget_set_sensitive (stop_btn, 0);
   gtk_signal_connect_object (GTK_OBJECT (stop_btn), "clicked", 
   	GTK_SIGNAL_FUNC (stop_button), (gpointer) &window2);
   gtk_container_border_width (GTK_CONTAINER (stop_btn), 1);
   gtk_box_pack_start (GTK_BOX (box), stop_btn, FALSE, FALSE, 0);
   gtk_widget_show (stop_btn);
   
   return (toolbar);
}
/*****************************************************************************/
static GtkWidget *CreateFTPWindow (struct ftp_window_data *wdata, 
	GtkItemFactoryEntry *entries, int num_menu_items) {
   const GtkTargetEntry possible_types[] = {
      {"STRING",			0,	0},
      {"text/plain", 			0, 	0},
      {"application/x-rootwin-drop", 	0, 	1}};
   GtkWidget *box, *scroll_list, *parent;
   GtkItemFactoryEntry dummy_item;
   char *titles[7];
   int i;

   titles[0] = "";
   titles[1] = _("Filename");
   titles[2] = _("Size");
   titles[3] = _("User");
   titles[4] = _("Group");
   titles[5] = _("Date");
   titles[6] = _("Attribs");
   wdata->filespec = g_malloc (2);
   strcpy (wdata->filespec, "*");
   wdata->hdata = new_hdata ();
   wdata->hdata->wdata = wdata;
   wdata->sortcol = 1;
   wdata->sortasds = 1;
   wdata->hdata->totalfiles = wdata->numselected = 0;

   parent = gtk_frame_new (NULL);
   gtk_widget_set_usize (parent, wdata->local == 1 ? listbox_local_width : listbox_remote_width,
   	listbox_file_height);
   gtk_container_border_width (GTK_CONTAINER (parent), 5);
   gtk_widget_show (parent);

   box = gtk_vbox_new (FALSE, 0);
   gtk_container_border_width (GTK_CONTAINER (box), 5);
   gtk_container_add (GTK_CONTAINER (parent), box);
   
   wdata->combo = gtk_combo_new ();
   gtk_box_pack_start (GTK_BOX (box), wdata->combo, FALSE, FALSE, 0);
   gtk_signal_connect (GTK_OBJECT (GTK_COMBO (wdata->combo)->entry), "activate", GTK_SIGNAL_FUNC (chdiredit), (gpointer) wdata);
   if (wdata->history) gtk_combo_set_popdown_strings (GTK_COMBO (wdata->combo), wdata->history);
   gtk_combo_disable_activate (GTK_COMBO (wdata->combo));
   gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry), "");
   gtk_widget_show (wdata->combo);
   
   wdata->hoststxt = gtk_label_new (_("Not connected"));
   gtk_misc_set_alignment (GTK_MISC (wdata->hoststxt), 0, 0);
   gtk_box_pack_start (GTK_BOX (box), wdata->hoststxt, FALSE, FALSE, 0);
   gtk_widget_show (wdata->hoststxt);

   scroll_list = gtk_scrolled_window_new (NULL, NULL);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll_list),
   	GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   wdata->listbox = gtk_clist_new_with_titles (7, titles);
   gtk_container_add (GTK_CONTAINER (scroll_list), wdata->listbox);
   gtk_drag_source_set (wdata->listbox, GDK_BUTTON2_MASK, possible_types, 3,
   	GDK_ACTION_COPY | GDK_ACTION_MOVE);
   gtk_drag_dest_set (wdata->listbox, GTK_DEST_DEFAULT_ALL, possible_types, 2,
   	GDK_ACTION_COPY | GDK_ACTION_MOVE);
   gtk_clist_set_selection_mode (GTK_CLIST (wdata->listbox), GTK_SELECTION_EXTENDED);
   gtk_clist_set_column_width (GTK_CLIST (wdata->listbox), 0, 16);
   gtk_clist_set_column_justification (GTK_CLIST (wdata->listbox), 0, GTK_JUSTIFY_CENTER);
   setup_column (wdata->listbox, 1, wdata->local ? local_columns[0] : remote_columns[0]);
   gtk_clist_set_column_justification (GTK_CLIST (wdata->listbox), 2, GTK_JUSTIFY_RIGHT);
   setup_column (wdata->listbox, 2, wdata->local ? local_columns[1] : remote_columns[1]);
   setup_column (wdata->listbox, 3, wdata->local ? local_columns[2] : remote_columns[2]);
   setup_column (wdata->listbox, 4, wdata->local ? local_columns[3] : remote_columns[3]);
   setup_column (wdata->listbox, 5, wdata->local ? local_columns[4] : remote_columns[4]);
   setup_column (wdata->listbox, 6, wdata->local ? local_columns[5] : remote_columns[5]);
   gtk_box_pack_start (GTK_BOX (box), scroll_list, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (wdata->listbox), "click_column", GTK_SIGNAL_FUNC (sortrows), (gpointer) wdata);
   gtk_signal_connect_after (GTK_OBJECT (wdata->listbox), "select_row", GTK_SIGNAL_FUNC (selectrow), (gpointer) wdata);
   gtk_signal_connect_after (GTK_OBJECT (wdata->listbox), "unselect_row", GTK_SIGNAL_FUNC (unselectrow), (gpointer) wdata);
   gtk_signal_connect_after (GTK_OBJECT (wdata->listbox), "button_press_event", GTK_SIGNAL_FUNC (list_dblclick), (gpointer) wdata);
   gtk_signal_connect (GTK_OBJECT (wdata->listbox), "key_press_event", GTK_SIGNAL_FUNC (list_enter), (gpointer) wdata);
   gtk_signal_connect (GTK_OBJECT (wdata->listbox), "drag_data_get", GTK_SIGNAL_FUNC (listbox_drag), (gpointer) wdata);
   gtk_signal_connect (GTK_OBJECT (wdata->listbox), "drag_data_received", GTK_SIGNAL_FUNC (listbox_get_drag_data), (gpointer) wdata);
   gtk_widget_show (wdata->listbox);
   gtk_widget_show (scroll_list);

   gtk_widget_show (box);

   wdata->ifactory = gtk_item_factory_new (GTK_TYPE_MENU, wdata->local == 1 ? "<local>" : "<remote>", NULL);
   for (i=0; i<num_menu_items; i++) {
      memcpy (&dummy_item, entries + i, sizeof (GtkItemFactoryEntry));
      dummy_item.path = strchr (_(dummy_item.path) + 1, '/');
      if (dummy_item.item_type) {
         dummy_item.item_type = _(entries[i].item_type);
      }
      gtk_item_factory_create_item (wdata->ifactory, &dummy_item, wdata, 1);
   }
   return (parent);
}
/*****************************************************************************/
static void setup_column (GtkWidget *listbox, int column, int width) {
   if (width == 0) {
      gtk_clist_set_column_auto_resize (GTK_CLIST (listbox), column, TRUE);
   }
   else if (width == -1) {
      gtk_clist_set_column_visibility (GTK_CLIST (listbox), column, FALSE);
   }
   else {
      gtk_clist_set_column_width (GTK_CLIST (listbox), column, width);
   }
}
/*****************************************************************************/
static void change_setting (struct ftp_window_data *wdata, int menuitem, GtkWidget *checkmenu) {
   switch (menuitem) {
      case 1 : if (wdata->hdata) gftp_set_data_type (wdata->hdata->ftpdata, GFTP_TYPE_ASCII);
               break;
      case 2 : if (wdata->hdata) gftp_set_data_type (wdata->hdata->ftpdata, GFTP_TYPE_BINARY);
               break;
   }
}
/*****************************************************************************/
static void selectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data) {
   struct ftp_window_data *wdata;
   struct ftp_file_data *tempfle;
   GList *templist;
   int i;
   
   wdata = (struct ftp_window_data *) data;
   i = 0;
   templist = wdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->shown) {
         if (i == row) {
            if (!tempfle->selected) {
               wdata->numselected++;
               tempfle->selected = 1;
            }
            break;
         }
         i++;
      }
      templist = templist->next;
   }
}
/*****************************************************************************/
static void unselectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data) {
   struct ftp_window_data *wdata;
   struct ftp_file_data *tempfle;
   GList *templist;
   int i;
   
   wdata = (struct ftp_window_data *) data;
   i = 0;
   templist = wdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->shown) {
         if (i == row) {
            if (tempfle->selected) {
               wdata->numselected--;
               tempfle->selected = 0;
            }
            break;
         }
         i++;
      }
      templist = templist->next;
   }
}
/*****************************************************************************/
static void selectall (struct ftp_window_data *wdata) {
   struct ftp_file_data *tempfle;
   GList *templist;
   int i = 0;
   
   if (!check_status (_("Select all"), wdata, 0, 0, 1)) return;
   templist = wdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->shown) {
         if (strcmp (tempfle->file, "..") != 0) {
            gtk_clist_select_row (GTK_CLIST (wdata->listbox), i++, 0);
         }
         else {
            gtk_clist_unselect_row (GTK_CLIST (wdata->listbox), i++, 0);
         }
      }
      templist = templist->next;
   }
}
/*****************************************************************************/
static void selectallfiles (struct ftp_window_data *wdata) {
   struct ftp_file_data *tempfle;
   GList *templist;
   int i = 0;
   
   if (!check_status (_("Select all files"), wdata, 0, 0, 1)) return;
   templist = wdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->shown) {
         if (tempfle->isdir)  {
            gtk_clist_unselect_row (GTK_CLIST (wdata->listbox), i++, 0);
         }
         else {
            gtk_clist_select_row (GTK_CLIST (wdata->listbox), i++, 0);
         }
      }
      templist = templist->next;
   }
}
/*****************************************************************************/
static void deselectall (struct ftp_window_data *wdata) {
   struct ftp_file_data *tempfle;
   GList *templist;
   int i = 0;
   
   if (!check_status (_("Deselect all"), wdata, 0, 0, 1)) return;
   templist = wdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->shown) {
         gtk_clist_unselect_row (GTK_CLIST (wdata->listbox), i++, 0);
      }
      templist = templist->next;
   }
}
/*****************************************************************************/
static gint log_click (GtkWidget *widget, GdkEventButton *event, gpointer data) {
   GtkItemFactory *log_factory;
   
   log_factory = (GtkItemFactory *) data;
   if (event->button == 3) {
      gtk_item_factory_popup (log_factory, (guint) event->x_root, (guint) event->y_root, 1, 0);
   }
   return (FALSE);
}
/*****************************************************************************/
static gint list_dblclick (GtkWidget *widget, GdkEventButton *event, gpointer data) {
   struct ftp_window_data *wdata;
   struct ftp_file_data *tempfle;
   int success, dir;
   GList *templist;
   
   wdata = (struct ftp_window_data *) data;
   if (event->button == 3) {
      gtk_item_factory_popup (wdata->ifactory, (guint) event->x_root, (guint) event->y_root, 1, 0);
   }
   else if (wdata->local == -1 || wdata->numselected != 1) return (FALSE);

   if (event->type == GDK_2BUTTON_PRESS && event->button == 1) {
      if ((templist = get_next_selected_filename (wdata->hdata->files)) == NULL) {
         ftp_log (gftp_logging_misc, NULL, "Internal gFTP Error: Could not find a selected file. This is probably a bug. Please email masneyb@seul.org about it\n");
         return (TRUE);
      }
      tempfle = (struct ftp_file_data *) templist->data;
      dir = tempfle->isdir;
      success = 0;
      if (tempfle->isdir || (wdata->local != 1 && tempfle->islink) || tempfle->possible_dir) {
            success = chdirfunc (data);
      }
      if (!dir && !success) view_dialog (data);
      return (FALSE);
   }
   return (TRUE);
}
/*****************************************************************************/
static gint list_enter (GtkWidget *widget, GdkEventKey *event, gpointer data) {
   struct ftp_window_data *wdata;
   struct ftp_file_data *tempfle;
   int success, dir;
   GList *templist;
   
   wdata = (struct ftp_window_data *) data;
   if (wdata->local == -1 || wdata->numselected == 0) return (TRUE);
   if (wdata->numselected == 1 && event->type == GDK_KEY_PRESS && event->keyval == GDK_Return) {
      if ((templist = get_next_selected_filename (wdata->hdata->files)) == NULL) {
         ftp_log (gftp_logging_misc, NULL, "Internal gFTP Error: Could not find a selected file. This is probably a bug. Please email masneyb@seul.org about it\n");
         return (FALSE);
      }
      tempfle = (struct ftp_file_data *) templist->data;
      dir = tempfle->isdir;
      success = 0;
      if (tempfle->isdir || (wdata->local != 1 && tempfle->islink) || tempfle->possible_dir) {
            success = chdirfunc (data);
      }
      if (!dir && !success) view_dialog (data);
   }
   else if (event->type == GDK_KEY_PRESS && (event->keyval == GDK_KP_Delete || event->keyval == GDK_Delete)) {
      delete_dialog (data);
   }
   else return (TRUE);
   return (FALSE);
}
/*****************************************************************************/
static void chfunc (struct ftp_window_data *wdata) {
   chdirfunc (wdata);
}
/*****************************************************************************/
static int chdirfunc (struct ftp_window_data *wdata) {
   char *newdir, *tempstr, resp;
   struct ftp_file_data *tempfle;
   GList *templist;
   
   if (!check_status (_("Chdir"), wdata, 1, 0, wdata->hdata->ftpdata->chdir != NULL)) return (0);
   if ((templist = get_next_selected_filename (wdata->hdata->files)) == NULL) {
      ftp_log (gftp_logging_misc, NULL, "Internal gFTP Error: Could not find a selected file. This is probably a bug. Please email masneyb@seul.org about it\n");
      return (0);
   }
   tempfle = (struct ftp_file_data *) templist->data;
   if (wdata->local == 1) {
      if (chdir(tempfle->file) == 0) {
         ftp_log (gftp_logging_misc, NULL, _("Successfully changed local directory to %s\n"), tempfle->file);
         update_ftp_info (wdata);
         gtk_clist_freeze (GTK_CLIST (wdata->listbox));
         delete_ftp_file_info (wdata);
         add_local_files (wdata);
         gtk_clist_thaw (GTK_CLIST (wdata->listbox));
         return (1);
      }
      else {
         if (wdata->local == 1) {
            ftp_log (gftp_logging_misc, NULL, _("Could not change local directory to %s: %s\n"),
            	tempfle->file, g_strerror (errno));
         }
         else {
            ftp_log (gftp_logging_misc, NULL, _("Could not change remote directory to %s: %s\n"), 
            	tempfle->file, g_strerror (errno));
         }
      }
   }
   else {
      newdir = g_strconcat (GFTP_GET_DIRECTORY (wdata->hdata->ftpdata), "/", tempfle->file, NULL);
      remove_double_slashes (newdir);
      tempstr = expand_path (newdir);
      g_free (newdir);
      resp = gftp_set_directory (wdata->hdata->ftpdata, tempstr);
      g_free (tempstr);
      if (resp == 0) {
         if (wdata->local == 2 && !ftp_connect (wdata->hdata, 0)) return (FALSE);
         wdata->hdata->getdir = 1;
         update_ftp_info (wdata);
         gtk_clist_freeze (GTK_CLIST (wdata->listbox));
         delete_ftp_file_info (wdata);
         ftp_list_files (wdata, 1);
         gtk_clist_thaw (GTK_CLIST (wdata->listbox));
         return (1);
      }
      else {
         tempstr = GFTP_GET_LAST_RESPONSE (wdata->hdata->ftpdata);
         if (tempstr && *tempstr == '4') disconnect (wdata);
         return (0);
      }
   }
   return (0);
}
/*****************************************************************************/
int chdiredit (GtkWidget *widget, struct ftp_window_data *wdata) {
   char *tempstr;
   int success;
   
   if (wdata->local == -1 && *gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (hostedit)->entry)) != '\0') {
      toolbar_hostedit (NULL, NULL);
      return (0);
   }
   if (!check_status (_("Chdir"), wdata, 0, 0, wdata->hdata->ftpdata->chdir != NULL)) return (FALSE);
   if ((tempstr = expand_path (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry)))) == NULL) return (FALSE);
   success = 0;
   if (wdata->local == 1) {
      if (chdir (tempstr) == 0) {
         success = 1;
         ftp_log (gftp_logging_misc, NULL, _("Successfully changed local directory to %s\n"), tempstr);
         update_ftp_info (wdata);
         gtk_clist_freeze (GTK_CLIST (wdata->listbox));
         delete_ftp_file_info (wdata);
         add_local_files (wdata);
         gtk_clist_thaw (GTK_CLIST (wdata->listbox));
      }
      else {
         if (wdata->local == 1) {
            ftp_log (gftp_logging_error, NULL, _("Could not change local directory to %s: %s\n"),
            	tempstr, g_strerror (errno));
         }
         else {
            ftp_log (gftp_logging_misc, NULL, _("Could not change remote directory to %s: %s\n"), 
            	tempstr, g_strerror (errno));
         }
      }
   }
   else {
      if (gftp_set_directory (wdata->hdata->ftpdata, tempstr) == 0) {
         if (wdata->local == 2 && !ftp_connect (wdata->hdata, 0)) {
            g_free (tempstr);
            return (FALSE);
         }
         wdata->hdata->getdir = 1;
         success = 1;
         update_ftp_info (wdata);
         gtk_clist_freeze (GTK_CLIST (wdata->listbox));
         delete_ftp_file_info (wdata);
         ftp_list_files (wdata, 1);
         gtk_clist_thaw (GTK_CLIST (wdata->listbox));
      }
   }
   if (success) {
      add_history (wdata->combo, &wdata->history, &wdata->histlen, tempstr);
   }
   g_free (tempstr);
   return (FALSE);
}
/*****************************************************************************/
void toolbar_hostedit (GtkWidget *widget, gpointer data) {
   GtkWidget *tempwid;
   void (*init) (gftp_request *request);
   char *txt;
   int num;

   txt = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (hostedit)->entry));
   if (*txt == '\0') {
      ftp_log (gftp_logging_error, NULL, _("Error: You must type in a host to connect to\n"));
      return;
   }
   
   if (window2.local != -1) disconnect (&window2);
   
   if (*gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (useredit)->entry)) == '\0') change_setting (NULL, 3, NULL);
   gftp_set_hostname (window2.hdata->ftpdata, txt);
   add_history (hostedit, &host_history, &host_len, txt);

   txt = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (portedit)->entry));
   gftp_set_port (window2.hdata->ftpdata, strtol (txt, NULL, 10));
   add_history (portedit, &port_history, &port_len, txt);

   txt = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (useredit)->entry));
   gftp_set_username (window2.hdata->ftpdata, txt);
   add_history (useredit, &user_history, &user_len, txt);
   
   gftp_set_password (window2.hdata->ftpdata, gtk_entry_get_text (GTK_ENTRY (passedit)));

   txt = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (window2.combo)->entry));
   gftp_set_directory (window2.hdata->ftpdata, txt);
   add_history (window2.combo, &window2.history, &window2.histlen, txt);
   
   tempwid = gtk_menu_get_active (GTK_MENU (protocol_menu));
   num = (int) gtk_object_get_user_data (GTK_OBJECT (tempwid));
   init = gftp_protocols[num].init;
   init (window2.hdata->ftpdata);
   ftp_connect (window2.hdata, 1);
}
/*****************************************************************************/
static void disconn (struct ftp_window_data *wdata) {
   if (wdata->local == -1) {
      ftp_log (gftp_logging_misc, NULL, _("Disconnect: Not connected to a remote site\n"));
      return;
   }
   disconnect (wdata);
}
/*****************************************************************************/
static void compare_windows (gpointer data) {
   struct ftp_file_data *curfle, *tempfle;
   GList *templist, *curlist;
   gint num;

   if (!check_status (_("Compare Windows"), &window2, 0, 0, 1)) return;
   
   deselectall (&window1);
   deselectall (&window2);
   
   /* Select the items in Window1 that aren't in Window2 */
   curlist = window1.hdata->files;
   num = 0;
   while (curlist != NULL) {
      curfle = (struct ftp_file_data *) curlist->data;

      templist = window2.hdata->files;
      while (templist != NULL) {
         tempfle = (struct ftp_file_data *) templist->data;
         if (strcmp (tempfle->file, curfle->file) == 0 && tempfle->size == curfle->size) {
            break;
         }
         templist = templist->next;
      }
      if (templist == NULL) gtk_clist_select_row (GTK_CLIST (window1.listbox), num, 0);
      num++;
      curlist = curlist->next;
   }

   /* Select the items in Window2 that aren't in Window1 */
   curlist = window2.hdata->files;
   num = 0;
   while (curlist != NULL) {
      curfle = (struct ftp_file_data *) curlist->data;
      
      templist = window1.hdata->files;
      while (templist != NULL) {
         tempfle = (struct ftp_file_data *) templist->data;
         if (strcmp (tempfle->file, curfle->file) == 0 && tempfle->size == curfle->size) {
            break;
         }
         templist = templist->next;
      }
      if (templist == NULL) gtk_clist_select_row (GTK_CLIST (window2.listbox), num, 0);
      num++;
      curlist = curlist->next;
   }
}
/*****************************************************************************/
void sortrows (GtkCList *clist, gint column, gpointer data) {
   GList *templist, *curlist, *files, *dirs, *dotdot;
   struct ftp_file_data *tempfle, *curfle;
   struct ftp_window_data *wdata;
   GtkWidget *sort_wid;
   int sortdir, pos;

   wdata = (struct ftp_window_data *) data;
   if (!wdata->hdata->files) return;
   if (!check_status (_("Sort"), wdata, 0, 0, 1)) return;
   
   gtk_label_set (GTK_LABEL (wdata->hoststxt), _("Sorting..."));
   fix_display ();
   if (column == 0 || (column == wdata->sortcol && wdata->sorted)) {
      wdata->sortasds = !wdata->sortasds;
      column = wdata->sortcol;
      sort_wid = gtk_clist_get_column_widget (clist, 0);
      gtk_widget_destroy (sort_wid);
      if (wdata->sortasds) sort_wid = toolbar_pixmap (wdata->listbox, down_xpm);
      else sort_wid = toolbar_pixmap (wdata->listbox, up_xpm);
      gtk_clist_set_column_widget (clist, 0, sort_wid);
   }
   else wdata->sortcol = column;
   sortdir = wdata->sortasds;
   dotdot = NULL;
   dirs = files = NULL;
   while (wdata->hdata->files != NULL) {
      curlist = wdata->hdata->files;
      curfle = (struct ftp_file_data *) curlist->data;
      wdata->hdata->files = g_list_remove_link (wdata->hdata->files, curlist);
      if (strcmp (curfle->file, "..") == 0) {
         dotdot = curlist;
         continue;
      }
      templist = sort_dirs_first && curfle->isdir ? dirs : files;
      pos = 0;
      tempfle = NULL;
      if (column == 1) {
         while (templist != NULL && (tempfle = (struct ftp_file_data *) templist->data) &&
         	(sortdir ? strcmp (tempfle->file, curfle->file) <= 0 :
         	strcmp (tempfle->file, curfle->file) >= 0)) {
            pos++;
            templist = templist->next;
         }
      }
      else if (column == 2) {
         while (templist != NULL && (tempfle = (struct ftp_file_data *) templist->data) && 
         	(sortdir ? tempfle->size <= curfle->size : 
         	tempfle->size >= curfle->size)) {
            pos++;
            templist = templist->next;
         }
      }
      else if (column == 3) {
         while (templist != NULL && (tempfle = (struct ftp_file_data *) templist->data) && 
         	(sortdir ? strcmp (tempfle->user, curfle->user) <= 0 :
         	strcmp (tempfle->user, curfle->user) >= 0)) {
            pos++;
            templist = templist->next;
         }
      }
      else if (column == 4) {
         while (templist != NULL && (tempfle = (struct ftp_file_data *) templist->data) && 
         	(sortdir ? strcmp (tempfle->group, curfle->group) <= 0 :
         	strcmp (tempfle->group, curfle->group) >= 0)) {
            pos++;
            templist = templist->next;
         }
      }
      else if (column == 5) { 
         while (templist != NULL && (tempfle = (struct ftp_file_data *) templist->data) && 
         	(sortdir ? tempfle->datetime <= curfle->datetime : 
         	tempfle->datetime >= curfle->datetime)) {
            pos++;
            templist = templist->next;
         }
      }
      else if (column == 6) {
         while (templist != NULL && (tempfle = (struct ftp_file_data *) templist->data) && 
         	(sortdir ? strcmp (tempfle->attribs, curfle->attribs) <= 0 :
         	strcmp (tempfle->attribs, curfle->attribs) >= 0)) {
            pos++;
            templist = templist->next;
         }
      }
      if (sort_dirs_first && curfle->isdir) {
         dirs = g_list_insert (dirs, curfle, pos);
      }
      else {
         files = g_list_insert (files, curfle, pos);
      }
   }

   if (dirs == NULL) wdata->hdata->files = files;
   else {
      wdata->hdata->files = dirs;
      templist = dirs;
      while (templist->next != NULL) templist = templist->next;
      templist->next = files;
      if (files) files->prev = templist;
   }
   if (dotdot != NULL) {
      dotdot->next = wdata->hdata->files;
      if (wdata->hdata->files) wdata->hdata->files->prev = dotdot;
      wdata->hdata->files = dotdot;
   }
   wdata->sorted = 1;
      
   gtk_clist_freeze (clist);
   wdata->numselected = 0;
   gtk_clist_clear (clist);
   templist = wdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      add_file_listbox (wdata, tempfle);
      templist = templist->next;
   }
   gtk_clist_thaw (clist);
   update_ftp_info (wdata);
}
/*****************************************************************************/
void delete_ftp_file_info (struct ftp_window_data *wdata) {
   gtk_clist_clear (GTK_CLIST (wdata->listbox));
   free_file_list (wdata->hdata->files);
   wdata->hdata->files = NULL;
}
/*****************************************************************************/
void queue_log (gftp_logging_type level, void *ptr, const char *string, ...) {
   struct ftp_log_queue *newlog;
   va_list argp;
   
   newlog = g_malloc (sizeof (struct ftp_log_queue));
   newlog->type = level;
   va_start (argp, string);
   newlog->msg = g_strdup_vprintf (string, argp);
   pthread_mutex_lock (&log_mutex);
   file_transfer_logs = g_list_append (file_transfer_logs, newlog);
   pthread_mutex_unlock (&log_mutex);
}
/*****************************************************************************/
void ftp_log (gftp_logging_type level, void *ptr, const char *string, ...) {
   GdkColor fore;
   char *tempstr;
   va_list argp;
   guint pos;
   int upd;

   upd = GTK_TEXT (logwdw)->vadj->upper - GTK_TEXT (logwdw)->vadj->page_size == GTK_TEXT (logwdw)->vadj->value;
   memset (&fore, 0, sizeof (fore));
   if (level == gftp_logging_send) fore.green = 0x8600;
   else if (level == gftp_logging_recv) fore.blue = 0xffff;
   else fore.red = 0xffff;
   gtk_text_freeze (GTK_TEXT (logwdw));

   va_start (argp, string);
   if (strncmp (string, "PASS", 4) == 0) {
      tempstr = g_malloc (12);
      strcpy (tempstr, "PASS xxxx\r\n");
   }
   else {
      tempstr = g_strdup_vprintf (string, argp);
   }
   gtk_text_insert (GTK_TEXT (logwdw), NULL, &fore, NULL, tempstr, -1);
   g_free (tempstr);
   
   pos = gtk_text_get_length (GTK_TEXT(logwdw));
   gtk_text_set_point (GTK_TEXT (logwdw), pos);
   gtk_text_thaw (GTK_TEXT (logwdw));
   if (upd) {
      gtk_adjustment_set_value (GTK_TEXT (logwdw)->vadj, 
      		GTK_TEXT (logwdw)->vadj->upper - GTK_TEXT (logwdw)->vadj->page_size);
   }
   fix_display ();
}
/*****************************************************************************/
void update_ftp_info (struct ftp_window_data *wdata) {
   char *tempstr, *dir, empty[] = "";
   
   dir = GFTP_GET_DIRECTORY (wdata->hdata->ftpdata);
   if (wdata->local == 1) {
      tempstr = g_strconcat (_("Local ["), strcmp (wdata->filespec, "*") == 0 ? _("All Files") : wdata->filespec, "]", NULL);
      gtk_label_set (GTK_LABEL (wdata->hoststxt), tempstr);
      if (dir != NULL) gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry), dir);
      else gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry), "");
      g_free (tempstr);
   }
   else if (wdata->local == 0 || wdata->local == 2) {
      tempstr = g_strconcat (GFTP_GET_HOSTNAME (wdata->hdata->ftpdata), wdata->cached ? _(" (Cached) [") : " [", strcmp(wdata->filespec, "*") == 0 ? _("All Files") : wdata->filespec, "]", NULL);
      gtk_label_set (GTK_LABEL(wdata->hoststxt), tempstr);
      if (dir != NULL) gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry), dir);
      else gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry), "");
      g_free (tempstr);
      
      if ((tempstr = GFTP_GET_HOSTNAME (wdata->hdata->ftpdata)) == NULL) {
         tempstr = empty;
      }
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (hostedit)->entry), tempstr);
      
      if ((tempstr = GFTP_GET_USERNAME (wdata->hdata->ftpdata)) == NULL) {
         tempstr = empty;
      }
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (useredit)->entry), tempstr);
      
      if ((tempstr = GFTP_GET_PASSWORD (wdata->hdata->ftpdata)) == NULL) {
         tempstr = empty;
      }
      gtk_entry_set_text (GTK_ENTRY (passedit), tempstr);

      tempstr = g_strdup_printf ("%d", GFTP_GET_PORT (wdata->hdata->ftpdata));
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (portedit)->entry), tempstr);
      g_free (tempstr);
   }
   else {
      gtk_label_set (GTK_LABEL (wdata->hoststxt), _("Not connected"));
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (wdata->combo)->entry), "");
   }
   fix_display ();
}
/*****************************************************************************/
void refresh (struct ftp_window_data *wdata) {
   if (!check_status (_("Refresh"), wdata, 0, 0, 1)) return;
   gtk_clist_freeze (GTK_CLIST (wdata->listbox));
   delete_ftp_file_info (wdata);
   if (wdata->local == 1) add_local_files (wdata);
   else ftp_list_files (wdata, 0);
   gtk_clist_thaw (GTK_CLIST (wdata->listbox));
   update_ftp_info (wdata);
}
/*****************************************************************************/
void fix_display (void) {
   g_main_iteration (FALSE);
}
/*****************************************************************************/
