/*****************************************************************************/
/*  transfer_gui.c - All of the ui routines for file transfers               */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"

static void update_file_status (struct ftp_transfer_data *tdata);
static void check_done_process (void);
static void on_next_transfer (struct ftp_transfer_data *tdata);
static void show_transfer (struct ftp_transfer_data *tdata);
static void transfer_done (GList *node);
static void create_transfer (struct ftp_transfer_data *tdata);
static void get_trans_password (GtkWidget *widget, struct dialog_data *data);
static void cancel_get_trans_password (GtkWidget *widget, struct dialog_data *data);
static void do_upload (GtkWidget *widget, struct dialog_data *data);
static void free_edit_data (GtkWidget *widget, struct dialog_data *data);

gint update_downloads (gpointer data) {
   struct ftp_transfer_data *tdata;
   struct ftp_log_queue *templog;
   char *tempstr, *temp1str;
   GList *templist, *next;
   
   /* These are logs that are generated from the other threads */
   if (file_transfer_logs != NULL) {
      pthread_mutex_lock (&log_mutex);
      templist = file_transfer_logs;
      while (templist != NULL) {
         templog = (struct ftp_log_queue *) templist->data;
         ftp_log (templog->type, NULL, templog->msg);
         g_free (templog->msg);
         templist = templist->next;
      }
      g_list_free (file_transfer_logs);
      file_transfer_logs = NULL;
      pthread_mutex_unlock (&log_mutex);
   }

   if (window2.hdata->gotbytes != 0) {
      if (window2.hdata->gotbytes == -1) {
         update_ftp_info (&window2);
         window2.hdata->gotbytes = 0;
      }
      else {
         tempstr = insert_commas (window2.hdata->gotbytes);
         temp1str = g_strdup_printf (_("Retrieving file names...%s bytes"), tempstr);
         gtk_label_set (GTK_LABEL (window2.hoststxt), temp1str);
         g_free (temp1str);
         g_free (tempstr);
      }
   }
   
   if (viewedit_process_done) check_done_process ();

   if (file_transfers != NULL) {
      templist = file_transfers;
      while (templist != NULL) {
         tdata = (struct ftp_transfer_data *) templist->data;
         pthread_mutex_lock (&tdata->mutex);
         
         if (tdata->next_file) on_next_transfer (tdata);
         if (tdata->show) show_transfer (tdata);
         else if (tdata->done) {
            next = templist->next;
            transfer_done (templist);
            templist = next;
            continue;
         }
         
         if (tdata->curfle != NULL) {
            if (!tdata->started && start_file_transfers && (transfer_in_progress == 0 || !do_one_transfer_at_a_time)) {
               create_transfer (tdata);
            }
            if (tdata->started) update_file_status (tdata);
         }
         pthread_mutex_unlock (&tdata->mutex);
         templist = templist->next;
      }
   }
   gtk_timeout_add (1000, update_downloads, data);
   return (0);
}
/*****************************************************************************/
gint dl_click (GtkWidget *widget, GdkEventButton *event, gpointer data) {
   GtkItemFactory *dl_factory;
   
   dl_factory = (GtkItemFactory *) data;
   if (event->button == 3) {
      gtk_item_factory_popup (dl_factory, (guint) event->x_root, (guint) event->y_root, 1, 0);
   }
   return (FALSE);
}
/*****************************************************************************/
static void update_file_status (struct ftp_transfer_data *tdata) {
   char tempstr[MAXSTR], *gotstr, *ofstr;
   int hours, mins, secs, pcent, st;
   struct ftp_file_data *tempfle;
   struct timeval tv;
   long remaining;
   float kbs;

   gettimeofday (&tv, NULL);

   if (tdata->trans_bytes == 0 || tdata->total_bytes == 0) {
      hours = mins = secs = 0;
   }
   else {
      if ((remaining = tv.tv_sec - tdata->inittime.tv_sec) == 0) remaining = 1;
      kbs = (float) tdata->trans_bytes / 1024.0 / (float) remaining;
      remaining = (long) ((float) tdata->total_bytes - tdata->trans_bytes) / 1024.0 / kbs;
      hours = remaining / 3600;
      remaining -= hours * 3600;
      mins = remaining / 60;
      remaining -= mins * 60;
      secs = remaining;
   }
   if (hours < 0 || mins < 0 || secs < 0) return;
   
   pcent = (int) ((float) tdata->trans_bytes / (float) tdata->total_bytes * 100);
   if (pcent < 0 || pcent > 100) pcent = 0;
   
   g_snprintf (tempstr, sizeof (tempstr), _("%d%% complete, %02d:%02d:%02d est. time remaining. (File %d of %d)"), 
   	pcent, hours, mins, secs, 
   	tdata->current_file_number, tdata->hdata->totalfiles);
   gtk_ctree_node_set_text (GTK_CTREE (dlwdw), tdata->node, 1, tempstr);

   tempfle = (struct ftp_file_data *) tdata->curfle->data;
   if (tdata->stalled) return;
   gotstr = insert_commas (tdata->curtrans);
   ofstr = insert_commas (tdata->curmax);
   st = 1;
   if (tv.tv_sec - tdata->lasttime.tv_sec <= 5) {
      remaining = (long) ((float) tempfle->remote_size - tdata->curtrans) / 1024.0 / tdata->kbs;
      hours = remaining / 3600;
      remaining -= hours * 3600;
      mins = remaining / 60;
      remaining -= mins * 60;
      secs = remaining;

      if (!(hours < 0 || mins < 0 || secs < 0)) {
         g_snprintf (tempstr, sizeof (tempstr), _("Recv %s of %s at %.2fKB/s, %02d:%02d:%02d est. time remaining"), 
         	gotstr == NULL ? "0" : gotstr, ofstr, tdata->kbs, hours, mins, secs);
         gtk_ctree_node_set_text (GTK_CTREE (dlwdw), tempfle->node, 1, tempstr);
         st = 0;
      }
   }
   if (st) {
      tdata->stalled = 1;
      g_snprintf (tempstr, sizeof (tempstr), _("Recv %s of %s, transfer stalled, unknown time remaining"), 
      	gotstr == NULL ? "0" : gotstr, ofstr);
      gtk_ctree_node_set_text (GTK_CTREE (dlwdw), tempfle->node, 1, tempstr);
   }
   g_free (gotstr);
   g_free (ofstr);
}
/*****************************************************************************/
static void check_done_process (void) {
   struct viewedit_data *ve_proc;
   GList *curdata, *deldata;
   struct stat st;
   int ret, i;
   char *str;
   pid_t pid;

   viewedit_process_done = 0;
   while ((pid = waitpid (-1, &ret, WNOHANG)) > 0) {
      curdata = viewedit_processes;
      while (curdata != NULL) {
         ve_proc = (struct viewedit_data *) curdata->data;
         if (ve_proc->pid == pid) {
            for (i=0; ve_proc->argv[i] != NULL; i++) g_free (ve_proc->argv[i]);
            if (ret != 0) {
               ftp_log (gftp_logging_error, NULL, _("Error: Child %d returned %d\n"), pid, ret);
            }
            else {
               ftp_log (gftp_logging_misc, NULL, _("Child %d returned successfully\n"), pid);
            }
            if (!ve_proc->view) {
               /* We was editing the file. Upload it */
               if (stat (ve_proc->filename, &st) == -1) {
                  ftp_log (gftp_logging_error, NULL, _("Error: Cannot get information about file %s: %s\n"), ve_proc->filename, g_strerror (errno));
               }
               else if (st.st_mtime == ve_proc->st.st_mtime) {
                  ftp_log (gftp_logging_misc, NULL, _("File %s was not changed\n"), ve_proc->remote_filename);               }
               else {
                  memcpy (&ve_proc->st, &st, sizeof (ve_proc->st));
                  str = g_strdup_printf (_("File %s has changed.\nWhat would you like to do?"), ve_proc->remote_filename);
                  MakeYesNoDialog (_("Edit File"), str, 1, 2, _("Upload"), do_upload, curdata, _("  Cancel  "), free_edit_data, curdata);
                  g_free (str);
                  curdata = g_list_next (curdata);
                  continue;
               }
            }
            else if (ve_proc->rm) unlink (ve_proc->filename);
         
            deldata = curdata;
            curdata = curdata->next;
            if (ve_proc->filename) g_free (ve_proc->filename);
            if (ve_proc->remote_filename) g_free (ve_proc->remote_filename);
            g_free (ve_proc);
            viewedit_processes = g_list_remove_link (viewedit_processes, deldata);
            continue;
         }
         curdata = curdata->next;
      }
   }
}
/*****************************************************************************/
static void on_next_transfer (struct ftp_transfer_data *tdata) {
   struct ftp_window_data *tempwdata;
   struct ftp_file_data *tempfle;
   
   tdata->next_file = 0;
   while (tdata->updfle != tdata->curfle) {
      tempfle = (struct ftp_file_data *) tdata->updfle->data;
      if (tempfle->done_view) {
         view_file (tempfle->file, 1, 1, 1, tempfle->remote_file);
      }
      else if (tempfle->done_edit) {
         view_file (tempfle->file, 0, 1, 1, tempfle->remote_file);
      }
      else if (tempfle->done_rm) {
         unlink (tempfle->file);
      }
      gtk_ctree_node_set_text (GTK_CTREE (dlwdw), tempfle->node, 1, _("Finished"));
      tdata->updfle = tdata->updfle->next;
   }

   tempwdata = tdata->direction ? &window1 : &window2;
   if (refresh_files && tdata->curfle && tdata->curfle->next && 
   	(tdata->direction || compare_hdata_structs (tdata->hdata->ftpdata, tempwdata->hdata->ftpdata, 1))) {
      refresh (tempwdata);
   }
}
/*****************************************************************************/
static void show_transfer (struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   char *pos, *text[2];
   GList *templist;
   
   text[0] = GFTP_GET_HOSTNAME (tdata->hdata->ftpdata);
   text[1] = _("Waiting...");
   tdata->node = gtk_ctree_insert_node (GTK_CTREE (dlwdw), NULL, NULL, text, 5,
   	dir_pixmap, dir_mask, open_dir_pixmap, open_dir_mask, FALSE, FALSE);
   gtk_ctree_node_set_row_data (GTK_CTREE (dlwdw), tdata->node, tdata);
   tdata->show = 0;
   tdata->ready = 1;
   tdata->curfle = tdata->updfle = tdata->hdata->files;

   tdata->total_bytes = 0;
   templist = tdata->hdata->files;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if ((pos = strrchr (tempfle->file, '/')) == NULL) {
         pos = tempfle->file;
      }
      else pos++;
      text[0] = pos;
      text[1] = _("Waiting...");
      
      tempfle->node = gtk_ctree_insert_node (GTK_CTREE (dlwdw), tdata->node, NULL,
      	text, 5, NULL, NULL, NULL, NULL, FALSE, FALSE);
      gtk_ctree_node_set_row_data (GTK_CTREE (dlwdw), tempfle->node, tdata);
      if (tempfle->remote_size == 0) tempfle->remote_size = tempfle->size;
      tdata->total_bytes += tempfle->remote_size;
      
      templist = templist->next;
   }
}
/*****************************************************************************/
static void transfer_done (GList *node) {
   struct ftp_window_data *tempwdata, *ckwdata;
   struct ftp_transfer_data *tdata;

   tdata = (struct ftp_transfer_data *) node->data;
   tempwdata = tdata->direction ? &window1 : &window2;
   ckwdata = tdata->direction ? &window2 : &window1;
   
   if (tdata->started) {
      if (!window2.hdata->stopable && window2.local == 2 && 
      		GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata) != NULL && compare_hdata_structs (tdata->hdata->ftpdata, window2.hdata->ftpdata, 0)) {
         window2.local = 0;
         GFTP_GET_CONTROL_FD (window2.hdata->ftpdata) = GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata);
         GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata) = NULL;
         window2.hdata->ftpdata->sockfd_unbuffered = tdata->hdata->ftpdata->sockfd_unbuffered;
         tdata->hdata->ftpdata->sockfd_unbuffered = -1;
         if (tdata->direction) {
            window2.cached = 0;
            update_ftp_info (&window2);
         }
         else refresh (&window2);
      }
      else gftp_disconnect (tdata->hdata->ftpdata);
            
      if (tdata->direction || compare_hdata_structs (tdata->hdata->ftpdata, ckwdata->hdata->ftpdata, 1)) {
         refresh (tempwdata);
      }
      transfer_in_progress--;
   }
   
   if (!tdata->show) gtk_ctree_remove_node (GTK_CTREE (dlwdw), tdata->node);
   file_transfers = g_list_remove_link (file_transfers, node);
   free_tdata (tdata);
}
/*****************************************************************************/
static void create_transfer (struct ftp_transfer_data *tdata) {
   if (!tdata->hdata->wait && (tdata->hdata->ftpdata->password == NULL || *tdata->hdata->ftpdata->password == '\0')) {
      tdata->hdata->wait = 1;
      MakeEditDialog (_("Enter Password"), _("Please enter your password for this site"), NULL, 1, 1,
      	_("Connect"), get_trans_password, tdata, _("  Cancel  "), cancel_get_trans_password, tdata);
   }
   else if (!tdata->hdata->wait) {
      if (window2.local == 0 && !window2.hdata->stopable && compare_hdata_structs (tdata->hdata->ftpdata, window2.hdata->ftpdata, 0)) {
         GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata) = GFTP_GET_CONTROL_FD (window2.hdata->ftpdata);
         GFTP_GET_CONTROL_FD (window2.hdata->ftpdata) = NULL;
         tdata->hdata->ftpdata->sockfd_unbuffered = window2.hdata->ftpdata->sockfd_unbuffered;
         window2.hdata->ftpdata->sockfd_unbuffered = -1;
         window2.local = 2;
         window2.cached = 1;
         update_ftp_info (&window2);
      }
      transfer_in_progress++;
      tdata->started = 1;
      tdata->stalled = 1;
      gtk_ctree_node_set_text (GTK_CTREE (dlwdw), tdata->node, 1, _("Connecting..."));
      if (tdata->direction) pthread_create (&tdata->tid, NULL, ftp_get_files, tdata);
      else pthread_create (&tdata->tid, NULL, ftp_put_files, tdata);
   }
}
/*****************************************************************************/
static void get_trans_password (GtkWidget *widget, struct dialog_data *data) {
   struct ftp_transfer_data *tdata;
   
   tdata = (struct ftp_transfer_data *) data->data;
   pthread_mutex_lock (&tdata->mutex);
   gftp_set_password (tdata->hdata->ftpdata, gtk_entry_get_text (GTK_ENTRY (data->edit)));
   data->data = NULL;
   tdata->hdata->wait = 0;
   pthread_mutex_unlock (&tdata->mutex);
}
/*****************************************************************************/
static void cancel_get_trans_password (GtkWidget *widget, struct dialog_data *data) {
   struct ftp_transfer_data *tdata;
   
   tdata = (struct ftp_transfer_data *) data->data;
   if (tdata->hdata->wait == 0) return;
   pthread_mutex_lock (&tdata->mutex);
   if (tdata->started) tdata->cancel = 1;
   else tdata->done = 1;
   tdata->hdata->wait = 0;
   data->data = NULL;
   pthread_mutex_unlock (&tdata->mutex);
   ftp_log (gftp_logging_misc, NULL, _("Stopping the transfer of %s\n"), ((struct ftp_file_data *) tdata->curfle->data)->file);
}
/*****************************************************************************/
static void do_upload (GtkWidget *widget, struct dialog_data *data) {
   struct ftp_transfer_data *tdata;
   struct viewedit_data *ve_proc;
   struct ftp_file_data *tempfle;
   GList *curdata;

   curdata = (GList *) data->data;
   ve_proc = (struct viewedit_data *) curdata->data;
   tdata = transfer_one_file (ve_proc->filename, ve_proc->remote_filename, 0);
   tempfle = (struct ftp_file_data *) tdata->hdata->files->data;
   tempfle->size = ve_proc->st.st_size;
   tempfle->done_rm = 1;
   add_file_transfer (tdata);
   free_edit_data (NULL, data);
}
/*****************************************************************************/
static void free_edit_data (GtkWidget *widget, struct dialog_data *data) {
   struct viewedit_data *ve_proc;
   GList *curdata;

   curdata = (GList *) data->data;
   ve_proc = (struct viewedit_data *) curdata->data;
   if (ve_proc->filename) g_free (ve_proc->filename);
   if (ve_proc->remote_filename) g_free (ve_proc->remote_filename);
   g_free (ve_proc);
   viewedit_processes = g_list_remove_link (viewedit_processes, curdata);
}
/*****************************************************************************/
void start_transfer (GtkWidget *widget, gpointer data) {
   struct ftp_transfer_data *tdata;
   GtkCTreeNode *node;
   
   if (GTK_CLIST (dlwdw)->selection == NULL) {
      ftp_log (gftp_logging_misc, NULL, _("There are currently no file transfers to start\n"));
      return;
   }
   node = (GtkCTreeNode *) GTK_CLIST (dlwdw)->selection->data;
   tdata = (struct ftp_transfer_data *) gtk_ctree_node_get_row_data (GTK_CTREE (dlwdw), node);

   pthread_mutex_lock (&tdata->mutex);
   if (!tdata->started) create_transfer (tdata);
   pthread_mutex_unlock (&tdata->mutex);
}
/*****************************************************************************/
void stop_transfer (GtkWidget *widget, gpointer data) {
   struct ftp_transfer_data *tdata;
   GtkCTreeNode *node;
   
   if (GTK_CLIST (dlwdw)->selection == NULL) {
      ftp_log (gftp_logging_misc, NULL, _("There are currently no file transfers in progress to stop\n"));
      return;
   }
   node = (GtkCTreeNode *) GTK_CLIST (dlwdw)->selection->data;
   tdata = (struct ftp_transfer_data *) gtk_ctree_node_get_row_data (GTK_CTREE (dlwdw), node);

   pthread_mutex_lock (&tdata->mutex);
   if (tdata->started) tdata->cancel = 1;
   else tdata->done = 1;
   pthread_mutex_unlock (&tdata->mutex);
   ftp_log (gftp_logging_misc, NULL, _("Stopping the transfer of %s\n"), ((struct ftp_file_data *) tdata->curfle->data)->file);
}
/*****************************************************************************/
