/*
 bot-events.c : IRC bot plugin for irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "bot.h"

#define BOTNET_RECONNECT_TIME (60*5)

static PLUGIN_DATA *plugdata;

static gboolean ip_compare(IPADDR *ip1, IPADDR *ip2)
{
    return ip1->family == ip2->family &&
	memcmp(&ip1->addr, &ip2->addr, sizeof(ip1->addr)) == 0;
}

BOTNET_REC *botnet_find(gchar *name)
{
    GList *tmp;

    for (tmp = plugdata->botnets; tmp != NULL; tmp = tmp->next)
    {
	BOTNET_REC *rec = tmp->data;

	if (g_strcasecmp(rec->name, name) == 0)
	    return rec;
    }

    return NULL;
}

BOT_REC *botnet_find_ip(BOTNET_REC *botnet, IPADDR *ip)
{
    GList *tmp, *tmp2;

    for (tmp = botnet->bots; tmp != NULL; tmp = tmp->next)
    {
	BOT_REC *rec = tmp->data;

	for (tmp2 = rec->valid_ips; tmp2 != NULL; tmp2 = tmp2->next)
	{
	    IPADDR *botip = tmp2->data;

	    if (ip_compare(botip, ip)) return rec;
	}
    }

    return NULL;
}

BOT_REC *botnet_find_nick(BOTNET_REC *botnet, gchar *nick)
{
    GList *tmp;

    for (tmp = botnet->bots; tmp != NULL; tmp = tmp->next)
    {
	BOT_REC *rec = tmp->data;

	if (rec->nick != NULL && g_strcasecmp(rec->nick, nick) == 0)
	    return rec;
    }

    return NULL;
}

/* broadcast message to everyone */
void botnet_broadcast(BOTNET_REC *botnet, BOT_REC *bot, gchar *data, gint len)
{
    GList *tmp;

    g_return_if_fail(botnet != NULL);
    g_return_if_fail(data != NULL);

    for (tmp = botnet->bots; tmp != NULL; tmp = tmp->next)
    {
	BOT_REC *rec = tmp->data;

	if (rec != bot && rec->handle != -1)
	    net_transmit(rec->handle, data, len);
    }
}

static gboolean botnet_disconnect_bot(BOT_REC *bot)
{
    if (bot->read_tag != -1)
    {
	gui_input_remove(bot->read_tag);
        bot->read_tag = -1;
    }
    if (bot->handle != -1)
    {
	net_disconnect(bot->handle);
	bot->handle = -1;
    }
    if (bot->buffer != NULL)
    {
	g_string_free(bot->buffer, TRUE);
	bot->buffer = NULL;
    }
    if (bot->output != NULL)
    {
	g_string_free(bot->output, TRUE);
	bot->output = NULL;
    }

    if (bot->host != NULL)
	return TRUE;

    /* just some client bot that connected to us, remove from bots list */
    bot->botnet->bots = g_list_remove(bot->botnet->bots, bot);
    g_free(bot);
    return FALSE;
}

static void sig_bot_read(BOT_REC *bot)
{
    BOTNET_REC *botnet;
    gint ret;

    botnet = bot->botnet;
    while (g_list_find(botnet->bots, bot))
    {
	ret = read_line(TRUE, bot->handle, bot->output, bot->buffer);
	if (ret == 0) break;
	if (ret == -1)
	{
	    /* connection lost */
	    if (botnet_disconnect_bot(bot) && !bot->disconnect && bot->botnet->link == bot)
	    {
		/* wasn't intentional disconnection from host, reconnect */
		bot->botnet->link = NULL;
		botnet_connect(bot->botnet->name);
	    }
	    break;
	}

	signal_emit("botnet event", 2, bot, bot->output->str);
    }
}

static void bot_connect_init(BOT_REC *bot, gint handle)
{
    bot->handle = handle;
    bot->read_tag = -1;
    bot->buffer = g_string_new(NULL);
    bot->output = g_string_new(NULL);
}

static void sig_botnet_listen(BOTNET_REC *botnet)
{
    BOT_REC *bot;
    IPADDR ip;
    gint handle, port;

    /* accept connection */
    handle = net_accept(botnet->listen_handle, &ip, &port);
    if (handle == -1)
	return;

    /* identify the bot who's trying to connect.. */
    bot = botnet_find_ip(botnet, &ip);
    if (bot == NULL || bot->password == NULL)
    {
	/* unknown bot, close connection /
	   bot didn't have password, don't let it connect to us */
	net_disconnect(handle);
	return;
    }

    if (bot->handle != 0)
    {
	/* already one connection from it, create another one */
	gchar *password;

	password = bot->password;

	bot = g_new0(BOT_REC, 1);
	bot->botnet = botnet;
	bot->password = g_strdup(password);
        botnet->bots = g_list_append(botnet->bots, bot);
    }

    /* connected.. */
    bot_connect_init(bot, handle);
    bot->read_tag = gui_input_add(handle, GUI_INPUT_READ, (GUIInputFunction) sig_bot_read, bot);
}

static gboolean botnet_listen(BOTNET_REC *botnet)
{
    IPADDR *addr;
    gint port;

    addr = net_host2ip(botnet->addr);
    port = botnet->port;

    botnet->listen_handle = net_listen(addr, &port);
    if (botnet->listen_handle == -1)
	return FALSE;

    botnet->listen_tag = gui_input_add(botnet->listen_handle, GUI_INPUT_READ,
				       (GUIInputFunction) sig_botnet_listen, botnet);

    return TRUE;
}

static void sig_botnet_connected(gint handle, BOT_REC *bot)
{
    BOTNET_REC *botnet;
    gchar *str;

    botnet = bot->botnet;
    bot->last_connect = time(NULL);

    if (handle == -1)
    {
	/* error, try another bot */
	botnet_connect(botnet->name);
	return;
    }

    /* connected to bot */
    bot_connect_init(bot, handle);
    botnet->link = bot;
    bot->uplink = TRUE;
    bot->read_tag = gui_input_add(handle, GUI_INPUT_READ, (GUIInputFunction) sig_bot_read, bot);

    /* send nick/pass */
    str = g_strdup_printf("PASS %s\nNICK %s\n", bot->password, botnet->nick);
    net_transmit(bot->handle, str, strlen(str));
    g_free(str);
}

gboolean botnet_connect(gchar *data)
{
    BOTNET_REC *botnet;
    BOT_REC *bot, *best;
    GList *tmp;
    time_t now;

    /* find botnet */
    botnet = botnet_find(data);
    if (botnet == NULL) return TRUE;

    if (botnet->listen_handle == -1)
    {
	/* start listening */
        botnet_listen(botnet);
    }

    /* find some bot where we can try to connect to */
    now = time(NULL);
    bot = best = NULL;
    for (tmp = botnet->bots; tmp != NULL; tmp = tmp->next)
    {
	bot = tmp->data;

	if (bot->host == NULL || bot->port <= 0 ||
	    bot->last_connect+BOTNET_RECONNECT_TIME > now ||
	    (bot->port == botnet->port && strcmp(bot->host, botnet->addr) == 0))
            continue; /* can't connect to this bot */

	if (bot->last_connect == 0)
	{
	    /* haven't yet tried to connect to this bot */
	    best = bot;
	    break;
	}

	if (best == NULL || best->last_connect > bot->last_connect)
	    best = bot;
    }

    if (best == NULL)
	return TRUE;

    /* connect to bot */
    net_connect_nonblock(best->host, best->port, NULL, (NET_CALLBACK) sig_botnet_connected, best);
    return TRUE;
}

void botnet_destroy(BOTNET_REC *botnet)
{
    plugdata->botnets = g_list_remove(plugdata->botnets, botnet);

    while (botnet->bots)
    {
	BOT_REC *bot = botnet->bots->data;

	botnet_disconnect_bot(bot);
	if (bot->host == NULL) continue;

	g_list_foreach(bot->valid_ips, (GFunc) g_free, NULL);
	g_list_free(bot->valid_ips);

	if (bot->buffer != NULL) g_string_free(bot->buffer, TRUE);
	if (bot->output != NULL) g_string_free(bot->output, TRUE);
	if (bot->host != NULL) g_free(bot->host);
	if (bot->nick != NULL) g_free(bot->nick);
	if (bot->password != NULL) g_free(bot->password);
	g_free(bot);

	botnet->bots = g_list_remove(botnet->bots, bot);
    }

    if (botnet->listen_tag != -1)
	gui_input_remove(botnet->listen_tag);
    if (botnet->listen_handle != -1)
        net_disconnect(botnet->listen_handle);
    g_free(botnet->name);
    g_free(botnet->addr);
    g_free(botnet->nick);
    g_free(botnet);
}

static void botnet_send_links(BOT_REC *bot, gboolean downlinks)
{
    GList *tmp;
    gchar *str;

    for (tmp = bot->botnet->bots; tmp != NULL; tmp = tmp->next)
    {
	BOT_REC *rec = tmp->data;

	if (rec != bot && rec->connected && (!downlinks || (downlinks && !rec->uplink)))
	{
	    str = g_strdup_printf("BOTLINK %s %s %d\n", rec->nick, rec->host, rec->port);
	    net_transmit(bot->handle, str, strlen(str));
	    g_free(str);
	}
    }
}

static gboolean botnet_event(BOT_REC *bot, gchar *data)
{
    BOTNET_REC *botnet;
    gchar *event, *args, *str, *p;
    gint num;

    botnet = bot->botnet;
    if (!bot->connected && botnet->link == bot)
    {
	/* we're trying to connect to our uplink */
	if (g_strcasecmp(data, "NICKERROR") == 0)
	{
	    /* nick already in use, change it by adding a number
	       at the end of it */
	    p = botnet->nick+strlen(botnet->nick);
	    while (p > botnet->nick && isdigit(p[-1])) p--; *p = '\0';
	    str = *p == '\0' || sscanf(p, "%d", &num) != 1 ?
		g_strdup_printf("%s2", botnet->nick) :
		g_strdup_printf("%s%d", botnet->nick, num);
	    g_free(botnet->nick); botnet->nick = str;

	    /* try again.. */
	    str = g_strdup_printf("NICK %s\n", botnet->nick);
            net_transmit(bot->handle, str, strlen(str));
	    g_free(str);
	}
	else if (g_strcasecmp(data, "CONNECTED") == 0)
	{
	    /* connected */
	    bot->connected = TRUE;

	    /* send our downlinks to host */
            botnet_send_links(bot, TRUE);
	}
	return TRUE;
    }

    if (!bot->connected && bot->password != NULL && g_strncasecmp(data, "PASS ", 5) == 0)
    {
	/* password sent, check that it matches */
	if (strcmp(data+5, bot->password) == 0)
	{
	    /* ok, connected! */
	    g_free(bot->password);
	    bot->password = NULL;
	}
	else
	{
	    /* wrong password, disconnect */
            botnet_disconnect_bot(bot);
	}
	return TRUE;
    }

    if (!bot->connected && g_strncasecmp(data, "NICK ", 5) == 0)
    {
	/* set bot's nick */
	if (bot->password != NULL)
	{
	    /* it hasn't sent password yet, don't bother setting nick.. */
	    return TRUE;
	}

	if (g_strcasecmp(bot->botnet->nick, data+5) == 0 ||
	    botnet_find_nick(bot->botnet, data+5) != NULL)
	{
	    /* nick already exists */
            net_transmit(bot->handle, "NICKERROR\n", 10);
            return TRUE;
	}
	else
	{
	    /* set the nick */
	    bot->nick = g_strdup(data+5);
	    bot->connected = TRUE;
            net_transmit(bot->handle, "CONNECTED\n", 10);

	    /* send info about all the bots that are connected now
	       to this botnet */
	    botnet_send_links(bot, FALSE);
	    return TRUE;
	}
    }

    if (!bot->connected)
    {
	/* nick/pass not sent yet */
	net_transmit(bot->handle, "ERROR\n", 6);
	return TRUE;
    }

    /* get command.. */
    event = g_strconcat("botnet event ", data, NULL);
    args = strchr(event+13, ' ');
    if (args != NULL) *args++ = '\0'; else args = "";
    while (*args == ' ') args++;

    g_strdown(event);
    if (!signal_emit(event, 2, bot, args))
        signal_emit("botnet default event", 2, bot, data);

    g_free(event);

    return TRUE;
}

static gboolean botnet_event_quit(BOT_REC *bot, gchar *data)
{
    /* disconnect bot */
    botnet_disconnect_bot(bot);
    return TRUE;
}

static gboolean botnet_event_bcast(BOT_REC *bot, gchar *data)
{
    gchar *str;

    /* broadcast message to all bots */
    str = g_strdup_printf("BCAST %s\n", data);
    botnet_broadcast(bot->botnet, bot, str, strlen(str));
    g_free(str);
    return TRUE;
}

static void read_ips(BOT_REC *bot, proplist_t prop)
{
    proplist_t pvalue;
    IPADDR *ip;
    gint num, max;

    max = PLGetNumberOfElements(prop);

    for (num = 0; num < max; num++)
    {
	pvalue = PLGetArrayElement(prop, num);
	if (pvalue == NULL) continue; /* hm?? */

	ip = g_new0(IPADDR, 1);
	memcpy(ip, net_host2ip(PLGetString(pvalue)), sizeof(IPADDR));
	bot->valid_ips = g_list_append(bot->valid_ips, ip);
    }
}

static void read_bots(BOTNET_REC *botnet, proplist_t prop)
{
    proplist_t pvalue;
    BOT_REC *bot;
    gint num, max;
    gchar *value;

    /* add the bots */
    max = PLGetNumberOfElements(prop);
    for (num = 0; num < max; num++)
    {
	pvalue = PLGetArrayElement(prop, num);

	value = config_get_str(pvalue, "host", NULL);
	if (value == NULL) continue; /* host required */

	bot = g_new0(BOT_REC, 1);
	bot->botnet = botnet;
	bot->host = g_strdup(value);
	bot->port = config_get_int(pvalue, "port", DEFAULT_BOTNET_PORT);
	bot->password = config_get_str(pvalue, "password", NULL);
	if (bot->password != NULL) g_strdup(bot->password);

	bot_connect_init(bot, -1);
	botnet->bots = g_list_append(botnet->bots, bot);

	value = config_get_prop(pvalue, "valid_ips");
	if (value != NULL)
	    read_ips(bot, value);
    }
}

static void read_config(PLUGIN_DATA *data)
{
    proplist_t prop, bprop, pbotnets, pkey, pvalue;
    BOTNET_REC *botnet;
    gchar *fname;
    gint max, num;

    g_return_if_fail(data != NULL);

    /* Read botnets from ~/.irssi/botnets */
    fname = g_strdup_printf("%s/.irssi/botnets", g_get_home_dir());
    prop = PLGetProplistWithPath(fname);
    g_free(fname);

    if (prop == NULL)
    {
	/* No botnets.. */
	return;
    }

    bprop = config_get_prop(prop, "botnets");
    if (bprop == NULL) return;

    pbotnets = PLGetAllDictionaryKeys(bprop);
    max = pbotnets == NULL ? 0 : PLGetNumberOfElements(pbotnets);
    for (num = 0; num < max; num++)
    {
	/* get the key & value */
	pkey = PLGetArrayElement(pbotnets, num);
	pvalue = PLGetDictionaryEntry(bprop, pkey);
	if (pkey == NULL || pvalue == NULL) continue; /* hm?? */

	/* New botnet */
	botnet = g_new0(BOTNET_REC, 1);
	botnet->listen_handle = -1;
	botnet->listen_tag = -1;
	botnet->name = g_strdup(PLGetString(pkey));
	botnet->addr = g_strdup(config_get_str(pvalue, "listen_addr", "127.0.0.1"));
	botnet->port = config_get_int(pvalue, "listen_port", DEFAULT_BOTNET_PORT);
	botnet->nick = g_strdup(config_get_str(pvalue, "nick", "bot"));

	data->botnets = g_list_append(data->botnets, botnet);

	/* read bots in botnet */
	read_bots(botnet, config_get_prop(pvalue, "bots"));
    }
}

void plugin_bot_botnet_init(PLUGIN_REC *plugin)
{
    PLUGIN_DATA *data = plugin->data;

    plugdata = data;
    read_config(data);
    botnet_connect("irssinet");

    signal_add("botnet event", (SIGNAL_FUNC) botnet_event);
    signal_add("botnet event quit", (SIGNAL_FUNC) botnet_event_quit);
    signal_add("botnet event bcast", (SIGNAL_FUNC) botnet_event_bcast);
}

void plugin_bot_botnet_deinit(PLUGIN_REC *plugin)
{
    PLUGIN_DATA *data = plugin->data;

    while (data->botnets)
	botnet_destroy(data->botnets->data);

    signal_remove("botnet event", (SIGNAL_FUNC) botnet_event);
    signal_remove("botnet event quit", (SIGNAL_FUNC) botnet_event_quit);
    signal_remove("botnet event bcast", (SIGNAL_FUNC) botnet_event_bcast);
}
