/*
 external.c : Plugin for sending commands to irssi via named pipe

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <irssi-plugin.h>
#include "external.h"

gchar *plugin_description(void)
{
    return "Plugin for sending commands to irssi via named pipe";
}

gint plugin_version(void)
{
    return PLUGIN_LAYER_VERSION;
}

static void deinit(PLUGIN_REC *plugin)
{
    PLUGIN_DATA *data;

    data = plugin->data;
    if (data->loaded)
        printtext(NULL, NULL, MSGLEVEL_CLIENTNOTICE, "External plugin unloaded");
    plugin_external_setup_deinit(plugin);

    if (data->pipename != NULL) g_free(data->pipename);
    if (data->buffer != NULL) g_string_free(data->buffer, TRUE);
    if (data->inputtag != -1) gui_input_remove(data->inputtag);
    if (data->handle != -1) close(data->handle);

    g_free(data);
}

void plugin_deinit(PLUGIN_REC *plugin)
{
    deinit(plugin);
}

static void ext_input(PLUGIN_DATA *data)
{
    GString *str;

    g_return_if_fail(data != NULL);

    str = g_string_new(NULL);

    while (read_line(FALSE, data->handle, str, data->buffer) > 0)
        signal_emit("send command", 3, str->str, cur_channel->server, cur_channel);

    if (data->buffer->len == 0)
    {
        /* End of file, close and open the pipe again .. Otherwise select()
           just keeps thinking there's more and more data coming while there's
           nothing and slowing down the system.. why? don't know. but this
           fixes it.. */
        close(data->handle);
        data->handle = open(data->pipename, O_RDONLY | O_NONBLOCK);
        if (data->handle == -1)
            plugin_destroy(data->plugin);
    }

    g_string_free(str, TRUE);
}

gboolean plugin_init(gint gui, PLUGIN_REC *plugin, gchar *args)
{
    PLUGIN_DATA *data;
    struct stat buf;
    gchar *str;

    plugin->data = data = g_new0(PLUGIN_DATA, 1);
    data->plugin = plugin;
    data->handle = -1;
    data->inputtag = -1;
    data->gui = gui;

    plugin_external_setup_init(plugin);
    if (args != NULL && *args != '\0')
    {
        g_free(data->pipename);
        data->pipename = g_strdup(args);
    }

    str = convert_home(data->pipename);
    g_free(data->pipename);
    data->pipename = g_strdup(str);

    data->handle = open(data->pipename, O_RDONLY | O_NONBLOCK);
    if (data->handle == -1)
    {
#ifdef HAVE_MKFIFO
        data->handle = mkfifo(data->pipename, 0660);
#else
        data->handle = mknod(data->pipename, 0660 | S_IFIFO, 0);
#endif
        if (data->handle == -1)
        {
            printtext(NULL, NULL, MSGLEVEL_CLIENTERROR, "External plugin: cannot create fifo %_%s%_", data->pipename);
            deinit(plugin);
            return FALSE;
        }
        printtext(NULL, NULL, MSGLEVEL_CLIENTNOTICE, "External plugin: created fifo %_%s%_ with mode %_0660%_", data->pipename);
        close(data->handle);

        data->handle = open(data->pipename, O_RDONLY | O_NONBLOCK);
        if (data->handle == -1)
        {
            printtext(NULL, NULL, MSGLEVEL_CLIENTERROR, "External plugin: couldn't reopen it!");
            deinit(plugin);
            return FALSE;
        }
    }

    if (fstat(data->handle, &buf) == 0)
    {
        if (!S_ISFIFO(buf.st_mode))
        {
            printtext(NULL, NULL, MSGLEVEL_CLIENTERROR, "External plugin: %_%s%_ isn't FIFO", data->pipename);
            deinit(plugin);
            return FALSE;
        }
        else if (buf.st_mode & 0x002)
        {
            /* world writable, not a good idea */
            printtext(NULL, NULL, MSGLEVEL_CLIENTNOTICE, "External plugin: WARNING: %_%s%_ is world writable!", data->pipename);
        }
        else
            printtext(NULL, NULL, MSGLEVEL_CLIENTNOTICE, "External plugin: reading fifo %_%s%_", data->pipename);
    }

    fcntl(data->handle, F_SETFL, O_NONBLOCK);

    data->buffer = g_string_new(NULL);
    data->inputtag = gui_input_add(data->handle, GUI_INPUT_READ | GUI_INPUT_EXCEPTION,
                                   (GUIInputFunction) ext_input, data);

    if (gui == IRSSI_GUI_GTK || gui == IRSSI_GUI_GNOME)
	plugin_external_gtk_init(plugin);
    data->loaded = TRUE;
    return TRUE;
}
