/*
 dialog-status.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static GtkWidget *dialog = NULL;
static gboolean show_servers, show_channels, show_queries, show_dcc_chats;

static void sig_destroy(GtkWidget **dialog)
{
    g_return_if_fail(dialog != NULL);
    g_return_if_fail(*dialog != NULL);

    gtk_widget_destroy(*dialog);
    *dialog = NULL;
}

static gboolean redraw_status(void)
{
    GtkCTree *ctree;
    GtkCTreeNode *node, *subnode;
    gchar *texts[2];
    GList *tmp;
    gint type;

    if (dialog == NULL) return TRUE;

    ctree = gtk_object_get_data(GTK_OBJECT(dialog), "ctree");

    gtk_clist_freeze(GTK_CLIST(ctree));
    gtk_clist_clear(GTK_CLIST(ctree));
    if (show_servers)
    {
        for (tmp = g_list_first(servers); tmp != NULL; tmp = tmp->next)
        {
            SERVER_REC *server = tmp->data;

            texts[0] = server->tag;
	    texts[1] = server->ircnet == NULL ?
		g_strdup_printf("%s:%d", server->address, server->port) :
		g_strdup_printf("%s:%d (%s)", server->address, server->port, server->ircnet);
            node = gtk_ctree_insert_node(ctree, NULL, NULL, texts, 0,
                                         NULL, NULL, NULL, NULL, FALSE, TRUE);
            gtk_ctree_node_set_row_data(ctree, node, server);
            g_free(texts[1]);
        }
    }

    if (show_channels || show_queries || show_dcc_chats)
    {
        for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
        {
            CHANNEL_REC *rec = tmp->data;

            if (((rec->type == CHANNEL_TYPE_CHANNEL) && show_channels) ||
                (rec->type == CHANNEL_TYPE_QUERY && show_queries) ||
                (rec->type == CHANNEL_TYPE_DCC_CHAT && show_dcc_chats))
            {
                node = gtk_ctree_find_by_row_data(ctree, NULL, rec->server);

                type = rec->type == CHANNEL_TYPE_EMPTY ? CHANNEL_TYPE_CHANNEL : rec->type;
                subnode = gtk_ctree_find_by_row_data(ctree, node, GINT_TO_POINTER(type));

                if (subnode == NULL)
                {
                    switch (type)
                    {
                        case CHANNEL_TYPE_CHANNEL:
                            texts[0] = _("Channels");
                            break;

                        case CHANNEL_TYPE_QUERY:
                            texts[0] = _("Queries");
                            break;

                        case CHANNEL_TYPE_DCC_CHAT:
                            texts[0] = _("DCC chats");
                            break;
                    }

                    texts[1] = "";
                    subnode = gtk_ctree_insert_node(ctree, node, NULL, texts, 0,
                                                    NULL, NULL, NULL, NULL, FALSE, TRUE);
                    gtk_ctree_node_set_row_data(ctree, subnode, GINT_TO_POINTER(type));
                }

                texts[0] = rec->name;
                texts[1] = rec->type == CHANNEL_TYPE_CHANNEL ? rec->topic : "";
                node = gtk_ctree_insert_node(ctree, subnode, NULL, texts, 0,
                                             NULL, NULL, NULL, NULL, FALSE, TRUE);
                gtk_ctree_node_set_row_data(ctree, node, rec);
            }
        }
    }

#if 0
    if (show_dcc_chats)
    {
        for (tmp = g_list_first(dcc_conns); tmp != NULL; tmp = tmp->next)
        {
            DCC_REC *rec = tmp->data;

            if (rec->type == DCC_TYPE_CHAT)
            {
                node = gtk_ctree_find_by_row_data(ctree, NULL, rec->server);
                subnode = gtk_ctree_find_by_row_data(ctree, node, GINT_TO_POINTER(3));

                if (subnode == NULL)
                {
                    texts[0] = _("DCC chats");
                    texts[1] = "";
                    subnode = gtk_ctree_insert_node(ctree, node, NULL, texts, 0,
                                                    NULL, NULL, NULL, NULL, FALSE, TRUE);
                    gtk_ctree_node_set_row_data(ctree, subnode, GINT_TO_POINTER(3));
                }

                texts[0] = rec->nick;
                texts[1] = rec->addr;
                node = gtk_ctree_insert_node(ctree, subnode, NULL, texts, 0,
                                             NULL, NULL, NULL, NULL, FALSE, TRUE);
                gtk_ctree_node_set_row_data(ctree, node, rec);
            }
        }
    }
#endif

    gtk_clist_thaw(GTK_CLIST(ctree));

    return TRUE;
}

static gboolean sig_channel_changed(CHANNEL_REC *channel)
{
    GtkCTree *ctree;
    GtkCTreeNode *node;

    if (channel == NULL || dialog == NULL) return TRUE;

    ctree = gtk_object_get_data(GTK_OBJECT(dialog), "ctree");

    node = gtk_ctree_find_by_row_data(ctree, NULL, channel);
    if (node != NULL) gtk_ctree_select(ctree, node);

    return TRUE;
}

static gint sig_button_pressed(GtkCTree *ctree, GdkEventButton *event)
{
    gpointer data;
    gint row, col;

    g_return_val_if_fail(ctree != NULL, 0);
    g_return_val_if_fail(event != NULL, 0);

    if (gtk_clist_get_selection_info(GTK_CLIST(ctree), event->x, event->y, &row, &col) < 0)
        return 0;

    data = gtk_clist_get_row_data(GTK_CLIST(ctree), row);
    if (data == NULL) return 0;

    if (event->button == 1 && event->type == GDK_2BUTTON_PRESS)
    {
        /* doubleclicked */
        if (g_list_find(channels, data) != NULL)
        {
            /* open channel */
            signal_emit("gui channel open", 1, data);
        }
    }

    return 1;
}

static void sig_clicked(GtkToggleButton *widget, gboolean *flag)
{
    g_return_if_fail(flag != NULL);

    *flag = gtk_toggle_button_get_active(widget);

    redraw_status();
}

#if 0
static void sig_select_row(GtkCTree *ctree, GtkCTreeNode *node, gint column)
{
    gpointer data;

    gtk_object_set_data(GTK_OBJECT(dialog), "selection", node);

    data = gtk_ctree_node_get_row_data(ctree, node);
    if (data == NULL) return;

    if (g_list_find(servers, data) != NULL)
    {
        SERVER_REC *server = data;

        printf("server %s\n", server->tag);
    }
    else if (g_list_find(channels, data) != NULL)
    {
        CHANNEL_REC *channel = data;

        printf("channel %s\n", channel->name);
    }
}

static void sig_unselect_row(GtkCTree *ctree, GtkCTreeNode *node, gint column)
{
    gtk_object_set_data(GTK_OBJECT(dialog), "selection", NULL);
}
#endif

void dialog_status(void)
{
    GtkWidget *table, *scrollbox, *ctree, *cb;

    if (dialog != NULL)
    {
        /* dialog already open */
        gdk_window_raise(dialog->window);
        return;
    }
    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_CLOSE, NULL);
    gtk_widget_set_usize(dialog, 400, 350);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_signal_connect_object(GTK_OBJECT(dialog), "destroy",
                              GTK_SIGNAL_FUNC(sig_destroy), (GtkObject *) &dialog);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    table = gtk_table_new(1, 1, FALSE);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, TRUE, TRUE, 0);

    /* Create server ctree widget */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_table_attach_defaults(GTK_TABLE(table), scrollbox, 0, 2, 0, 1);

    ctree = gtk_ctree_new(2, 0);
    gtk_clist_set_column_auto_resize(GTK_CLIST(ctree), 0, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(ctree), 1, TRUE);

    show_servers = show_channels = show_queries = show_dcc_chats = TRUE;
    cb = gtk_check_button_new_with_label(_("Display servers"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cb), TRUE);
    gtk_signal_connect(GTK_OBJECT(cb), "clicked",
                       GTK_SIGNAL_FUNC(sig_clicked), &show_servers);
    gtk_table_attach(GTK_TABLE(table), cb, 0, 1, 1, 2, GTK_FILL | GTK_EXPAND, 0, 0, 0);
    cb = gtk_check_button_new_with_label(_("Display channels"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cb), TRUE);
    gtk_signal_connect(GTK_OBJECT(cb), "clicked",
                       GTK_SIGNAL_FUNC(sig_clicked), &show_channels);
    gtk_table_attach(GTK_TABLE(table), cb, 1, 2, 1, 2, GTK_FILL | GTK_EXPAND, 0, 0, 0);
    cb = gtk_check_button_new_with_label(_("Display queries"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cb), TRUE);
    gtk_signal_connect(GTK_OBJECT(cb), "clicked",
                       GTK_SIGNAL_FUNC(sig_clicked), &show_queries);
    gtk_table_attach(GTK_TABLE(table), cb, 0, 1, 2, 3, GTK_FILL | GTK_EXPAND, 0, 0, 0);
    cb = gtk_check_button_new_with_label(_("Display DCC chats"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cb), TRUE);
    gtk_signal_connect(GTK_OBJECT(cb), "clicked",
                       GTK_SIGNAL_FUNC(sig_clicked), &show_dcc_chats);
    gtk_table_attach(GTK_TABLE(table), cb, 1, 2, 2, 3, GTK_FILL | GTK_EXPAND, 0, 0, 0);

    gtk_container_add(GTK_CONTAINER(scrollbox), ctree);

    gtk_object_set_data(GTK_OBJECT(dialog), "ctree", ctree);
    gtk_clist_set_selection_mode(GTK_CLIST(ctree), GTK_SELECTION_BROWSE);
    gtk_signal_connect_after(GTK_OBJECT(ctree), "button_press_event",
                             GTK_SIGNAL_FUNC(sig_button_pressed), NULL);
#if 0
    gtk_signal_connect(GTK_OBJECT(ctree), "tree_select_row",
                       GTK_SIGNAL_FUNC(sig_select_row), NULL);
    gtk_signal_connect(GTK_OBJECT(ctree), "tree_unselect_row",
                       GTK_SIGNAL_FUNC(sig_unselect_row), NULL);
#endif
    redraw_status();
    gtk_clist_select_row(GTK_CLIST(ctree), 0, 0);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(sig_destroy), (GtkObject *) &dialog);

    gtk_widget_show_all(dialog);
}

void dialog_status_init(void)
{
    dialog = NULL;
    signal_add("server connected", (SIGNAL_FUNC) redraw_status);
    signal_add("server disconnected", (SIGNAL_FUNC) redraw_status);
    signal_add("channel created", (SIGNAL_FUNC) redraw_status);
    signal_add("channel destroyed", (SIGNAL_FUNC) redraw_status);
    signal_add("channel changed", (SIGNAL_FUNC) sig_channel_changed);
}

void dialog_status_deinit(void)
{
    signal_remove("server connected", (SIGNAL_FUNC) redraw_status);
    signal_remove("server disconnected", (SIGNAL_FUNC) redraw_status);
    signal_remove("channel created", (SIGNAL_FUNC) redraw_status);
    signal_remove("channel destroyed", (SIGNAL_FUNC) redraw_status);
    signal_remove("channel changed", (SIGNAL_FUNC) sig_channel_changed);
}
