/*
 gui-statusbar.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

typedef struct
{
    gint tag;

    gint xpos;
    gint size;
    gboolean right_justify;
    STATUSBAR_FUNC func;
}
STATUSBAR_REC;

static GList *sbars;
static gint sbars_tag;

static void gui_statusbar_redraw_all(void)
{
    GList *tmp;
    gint xpos, rxpos;

    screen_refresh_freeze();
    set_bg((1<<4)+15);
    move(LINES-2, 0); clrtoeol();
    set_bg(0);
    screen_refresh();

    xpos = 1;
    for (tmp = g_list_first(sbars); tmp != NULL; tmp = tmp->next)
    {
        STATUSBAR_REC *rec = tmp->data;

        if (!rec->right_justify)
        {
            if (xpos+rec->size < COLS)
            {
                rec->xpos = xpos;
                rec->func(xpos, LINES-2, rec->size);
                if (rec->size > 0) xpos += rec->size+1;
            }
        }
    }

    rxpos = COLS-1;
    for (tmp = g_list_first(sbars); tmp != NULL; tmp = tmp->next)
    {
        STATUSBAR_REC *rec = tmp->data;

        if (rec->right_justify)
        {
            if (rxpos-rec->size > xpos)
            {
                rec->xpos = rxpos-rec->size;
                rec->func(rec->xpos, LINES-2, rec->size);
                if (rec->size > 0) rxpos -= rec->size+1;
            }
        }
    }
    screen_refresh_thaw();
}

void gui_statusbar_redraw(gint tag)
{
    GList *tmp;

    if (tag == -1)
    {
        gui_statusbar_redraw_all();
        return;
    }

    for (tmp = g_list_first(sbars); tmp != NULL; tmp = tmp->next)
    {
        STATUSBAR_REC *rec = tmp->data;

        if (rec->tag == tag)
        {
            rec->func(rec->xpos, LINES-2, rec->size);
            break;
        }
    }
}

/* allocate area in statusbar, returns tag or -1 if failed */
gint gui_statusbar_allocate(gint size, gboolean right_justify, STATUSBAR_FUNC func)
{
    STATUSBAR_REC *rec;

    g_return_val_if_fail(func != NULL, -1);

    rec = g_new0(STATUSBAR_REC, 1);
    sbars = g_list_append(sbars, rec);

    rec->tag = ++sbars_tag;
    rec->xpos = -1;
    rec->size = size;
    rec->right_justify = right_justify;
    rec->func = func;

    gui_statusbar_redraw_all();
    return rec->tag;
}

void gui_statusbar_resize(gint tag, gint size)
{
    GList *tmp;

    for (tmp = g_list_first(sbars); tmp != NULL; tmp = tmp->next)
    {
        STATUSBAR_REC *rec = tmp->data;

        if (rec->tag == tag)
        {
            rec->size = size;
            gui_statusbar_redraw_all();
            break;
        }
    }
}

void gui_statusbar_remove(gint tag)
{
    GList *tmp;

    for (tmp = sbars; tmp != NULL; tmp = tmp->next)
    {
        STATUSBAR_REC *rec = tmp->data;

        if (rec->tag == tag)
        {
            g_free(rec);
            sbars = g_list_remove(sbars, rec);
            if (!quitting) gui_statusbar_redraw_all();
            break;
        }
    }
}

void gui_statusbar_init(void)
{
    sbars = NULL;
    sbars_tag = 0;

    set_bg((1<<4)+15);
    move(LINES-2, 0); clrtoeol();
    set_bg(0);
}

void gui_statusbar_deinit(void)
{
    STATUSBAR_REC *rec;

    while (sbars != NULL)
    {
        rec = sbars->data;
        gui_statusbar_remove(rec->tag);
    }
}
