/*
 ctcp.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

/* Send CTCP reply with flood protection */
void ctcp_send_reply(SERVER_REC *server, gchar *data)
{
    GList *tmp, *next;
    gint len, tag;

    g_return_if_fail(server != NULL);
    g_return_if_fail(data != NULL);

    len = 0;
    for (tmp = g_list_first(server->ctcpqueue); tmp != NULL; tmp = tmp->next)
    {
	next = tmp->next;
	if (server_idle_find(server, GPOINTER_TO_INT(tmp->data)))
	    len++;
	else
	{
	    /* Doesn't exist anymore, remove.. */
	    server->ctcpqueue = g_list_remove(server->ctcpqueue, tmp->data);
	}
    }

    if (len < setup_get_int("max_ctcp_queue"))
    {
        /* Add to first in idle queue */
	tag = server_idle_add_first(server, data, NULL, 0, NULL);
	server->ctcpqueue = g_list_append(server->ctcpqueue, GINT_TO_POINTER(tag));
    }
}

/* CTCP ping */
static gboolean ctcp_ping(gchar *data, SERVER_REC *server, gchar *nick)
{
    gchar *str;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(server != NULL, FALSE);
    g_return_val_if_fail(nick != NULL, FALSE);

    str = g_strdup_printf("NOTICE %s :\001PING %s\001", nick, data);
    ctcp_send_reply(server, str);
    g_free(str);

    return TRUE;
}

/* CTCP version */
static gboolean ctcp_version(gchar *data, SERVER_REC *server, gchar *nick)
{
    GString *str;
    gchar *sysname, *release;
    struct utsname un;

    g_return_val_if_fail(server != NULL, FALSE);
    g_return_val_if_fail(nick != NULL, FALSE);

    if (uname(&un) == 0)
    {
	sysname = un.sysname;
	release = un.release;
    }
    else
    {
	sysname = "??";
	release = "??";
    }

    str = g_string_new(NULL);
    g_string_sprintf(str, "NOTICE %s :\001VERSION ", nick);
    g_string_sprintfa(str, setup_get_str("ctcp_version_reply"), VERSION, sysname, release);
    g_string_append_c(str, '\001');
    ctcp_send_reply(server, str->str);
    g_string_free(str, TRUE);

    return TRUE;
}

static gboolean ctcp_msg(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr, gchar *target)
{
    gchar *args, *str;

    str = g_strconcat("ctcp msg ", data, NULL);
    args = strchr(str+9, ' ');
    if (args != NULL) *args++ = '\0'; else args = "";

    g_strdown(str+9);
    if (!signal_emit(str, 5, args, server, nick, addr, target))
        signal_emit("default ctcp msg", 5, data, server, nick, addr, target);
    g_free(str);

    return TRUE;
}

static gboolean ctcp_reply(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr, gchar *target)
{
    gchar *args, *str;

    str = g_strconcat("ctcp reply ", data, NULL);
    args = strchr(str+11, ' ');
    if (args != NULL) *args++ = '\0'; else args = "";

    g_strdown(str+11);
    if (!signal_emit(str, 5, args, server, nick, addr, target))
        signal_emit("default ctcp reply", 5, data, server, nick, addr, target);
    g_free(str);

    return TRUE;
}

static gboolean event_privmsg(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    gchar *params, *target, *msg, *ptr;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, &target, &msg);

    /* handle only ctcp messages.. */
    if (*msg == 1)
    {
        /* remove the later \001 */
        ptr = strrchr(++msg, 1);
        if (ptr != NULL) *ptr = '\0';

        signal_emit("ctcp msg", 5, msg, server, nick, addr, target);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_notice(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    gchar *params, *target, *ptr, *msg;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, &target, &msg); /* Channel or nick name */

    /* handle only ctcp replies */
    if (*msg == 1)
    {
        ptr = strrchr(++msg, 1);
        if (ptr != NULL) *ptr = '\0';

        signal_emit("ctcp reply", 5, msg, server, nick, addr, target);
    }

    g_free(params);
    return TRUE;
}

static gboolean ctcp_init_server(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    return TRUE;
}

static gboolean ctcp_deinit_server(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    g_list_free(server->ctcpqueue);
    return TRUE;
}

void ctcp_init(void)
{
    signal_add("server connected", (SIGNAL_FUNC) ctcp_init_server);
    signal_add("server disconnected", (SIGNAL_FUNC) ctcp_deinit_server);
    signal_add("event privmsg", (SIGNAL_FUNC) event_privmsg);
    signal_add("event notice", (SIGNAL_FUNC) event_notice);
    signal_add("ctcp msg", (SIGNAL_FUNC) ctcp_msg);
    signal_add("ctcp reply", (SIGNAL_FUNC) ctcp_reply);
    signal_add("ctcp msg ping", (SIGNAL_FUNC) ctcp_ping);
    signal_add("ctcp msg version", (SIGNAL_FUNC) ctcp_version);
}

void ctcp_deinit(void)
{
    signal_remove("server connected", (SIGNAL_FUNC) ctcp_init_server);
    signal_remove("server disconnected", (SIGNAL_FUNC) ctcp_deinit_server);
    signal_remove("event privmsg", (SIGNAL_FUNC) event_privmsg);
    signal_remove("event notice", (SIGNAL_FUNC) event_notice);
    signal_remove("ctcp msg", (SIGNAL_FUNC) ctcp_msg);
    signal_remove("ctcp reply", (SIGNAL_FUNC) ctcp_reply);
    signal_remove("ctcp msg ping", (SIGNAL_FUNC) ctcp_ping);
    signal_remove("ctcp msg version", (SIGNAL_FUNC) ctcp_version);
}
