/*

 flood.c : Flood protection (see also ctcp.c)

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

typedef struct
{
    gint type;
    gint num;
    gchar *nick;
    SERVER_REC *server;
}
FLOOD_REC;

static gint flood_tag;
static gint flood_max_msgs;

static gboolean flood_rec_deinit(gchar *key, FLOOD_REC *rec)
{
    g_return_val_if_fail(key != NULL, FALSE);
    g_return_val_if_fail(rec != NULL, FALSE);

    g_free(rec->nick);
    g_free(rec);
    return TRUE;
}

/* timeout function: flood protection */
static gint flood_timeout(void)
{
    GList *tmp;

    for (tmp = g_list_first(servers); tmp != NULL; tmp = tmp->next)
    {
	SERVER_REC *rec = tmp->data;

	/* remove everyone from hash table */
	g_hash_table_foreach_remove(rec->floodlist, (GHRFunc) flood_rec_deinit, NULL);
    }
    return 1;
}

/* Initialize flood protection */
static gboolean flood_init_server(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    server->floodlist = g_hash_table_new((GHashFunc) g_istr_hash, (GCompareFunc) g_istr_equal);
    return TRUE;
}

/* Deinitialize flood protection */
static gboolean flood_deinit_server(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    if (server->floodlist != NULL)
    {
        g_hash_table_freeze(server->floodlist);
        g_hash_table_foreach(server->floodlist, (GHFunc) flood_rec_deinit, NULL);
        g_hash_table_thaw(server->floodlist);
        g_hash_table_destroy(server->floodlist);
    }
    return TRUE;
}

static gboolean flood(SERVER_REC *server, gchar *nick, gchar *host, gchar *typestr)
{
    gint type;

    type = level2bits(typestr);

    if ((type & MSGLEVEL_MSGS || type & MSGLEVEL_CTCPS) &&
        !ignore_check(server, nick, host, MSGLEVEL_NEVER))
    {
        autoignore_add(server, type, nick);
    }
    return TRUE;
}

/* All messages should go through here.. */
static void flood_newmsg(SERVER_REC *server, gint type, gchar *nick, gchar *host, gchar *target)
{
    FLOOD_REC *rec;

    g_return_if_fail(server != NULL);
    g_return_if_fail(nick != NULL);

    rec = g_hash_table_lookup(server->floodlist, nick);
    if (rec != NULL)
    {
        if (++rec->num > flood_max_msgs)
        {
            /* flooding!!! */
            signal_emit("flood", 5, server, nick, host, bits2level(rec->type), target);
        }
        return;
    }

    rec = g_new(FLOOD_REC, 1);
    rec->type = type;
    rec->num = 1;
    rec->server = server;
    rec->nick = g_strdup(nick);

    g_hash_table_insert(server->floodlist, rec->nick, rec);
}

static gboolean flood_privmsg(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    gint publiclevel;
    gchar *params, *target, *text;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(server != NULL, FALSE);

    if (nick == NULL)
    {
        /* don't try to ignore server messages.. */
        return TRUE;
    }

    params = event_get_params(data, 2, &target, &text);
    publiclevel = ischannel(*target) ? MSGLEVEL_PUBLIC : MSGLEVEL_MSGS;

    if (*text == 1)
    {
        /* CTCP */
        if (!ignore_check(server, nick, addr, MSGLEVEL_CTCPS))
            flood_newmsg(server, MSGLEVEL_CTCPS | publiclevel, nick, addr, target);
    }
    else
    {
        if (addr != NULL && !ignore_check(server, nick, addr, publiclevel))
            flood_newmsg(server, publiclevel, nick, addr, target);
    }

    g_free(params);
    return TRUE;
}

static gboolean flood_notice(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    gchar *params, *target, *text;

    g_return_val_if_fail(text != NULL, FALSE);
    g_return_val_if_fail(server != NULL, FALSE);

    if (nick == NULL)
    {
        /* don't try to ignore server messages.. */
        return TRUE;
    }

    params = event_get_params(data, 2, &target, &text);
    if (addr != NULL && !ignore_check(server, nick, addr, MSGLEVEL_NOTICES))
        flood_newmsg(server, MSGLEVEL_NOTICES | ischannel(*target) ? MSGLEVEL_PUBLIC : MSGLEVEL_MSGS, nick, addr, target);

    g_free(params);
    return TRUE;
}

static gboolean sig_config_read(void)
{
    if (flood_tag != -1) gui_timeout_remove(flood_tag);
    flood_tag = gui_timeout_add(setup_get_int("flood_timecheck"), (GUITimeoutFunction) flood_timeout, NULL);

    flood_max_msgs = setup_get_int("flood_max_msgs");
    return TRUE;
}

void flood_init(void)
{
    flood_tag = -1;
    flood_max_msgs = 0;

    signal_add("startup settings read", (SIGNAL_FUNC) sig_config_read);
    signal_add("setup changed", (SIGNAL_FUNC) sig_config_read);
    signal_add("server connected", (SIGNAL_FUNC) flood_init_server);
    signal_add("server disconnected", (SIGNAL_FUNC) flood_deinit_server);
    signal_add("event privmsg", (SIGNAL_FUNC) flood_privmsg);
    signal_add("event notice", (SIGNAL_FUNC) flood_notice);
    signal_add("flood", (SIGNAL_FUNC) flood);
}

void flood_deinit(void)
{
    if (flood_tag != -1) gui_timeout_remove(flood_tag);
    signal_remove("startup settings read", (SIGNAL_FUNC) sig_config_read);
    signal_remove("setup changed", (SIGNAL_FUNC) sig_config_read);
    signal_remove("server connected", (SIGNAL_FUNC) flood_init_server);
    signal_remove("server disconnected", (SIGNAL_FUNC) flood_deinit_server);
    signal_remove("event privmsg", (SIGNAL_FUNC) flood_privmsg);
    signal_remove("event notice", (SIGNAL_FUNC) flood_notice);
    signal_remove("flood", (SIGNAL_FUNC) flood);
}
