/*
 plugin-commands.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

void plugin_bind(PLUGIN_REC *plugin, gchar *signal, SIGNAL_FUNC func, gboolean last)
{
    GSList *emitlist;

    g_return_if_fail(plugin != NULL);
    g_return_if_fail(signal != NULL);
    g_return_if_fail(func != NULL);

    signal = g_strdup(signal);
    emitlist = g_hash_table_lookup(plugin->signals, signal);
    if (emitlist != NULL) g_hash_table_remove(plugin->signals, signal);
    emitlist = g_slist_prepend(emitlist, func);
    g_hash_table_insert(plugin->signals, signal, emitlist);

    if (last)
	signal_add_last(signal, func);
    else
	signal_add(signal, func);
}

void plugin_unbind(PLUGIN_REC *plugin, gchar *signal, SIGNAL_FUNC func)
{
    GSList *emitlist;
    gchar *origkey;

    g_return_if_fail(plugin != NULL);
    g_return_if_fail(signal != NULL);
    g_return_if_fail(func != NULL);

    signal_remove(signal, func);

    signal = g_strdup(signal);
    if (!g_hash_table_lookup_extended(plugin->signals, signal, (gpointer *) &origkey, (gpointer *) &emitlist))
        return;

    g_hash_table_remove(plugin->signals, origkey);
    g_free(origkey);

    emitlist = g_slist_remove(emitlist, func);
    g_hash_table_insert(plugin->signals, signal, emitlist);
}

static gboolean plugin_created(PLUGIN_REC *plugin)
{
    g_return_val_if_fail(plugin != NULL, FALSE);

    plugin->signals = g_hash_table_new((GHashFunc) g_str_hash, (GCompareFunc) g_str_equal);
    return TRUE;
}

static void destroy_all_binds(gchar *signal, GSList *emitlist, PLUGIN_REC *plugin)
{
    GSList *tmp;

    for (tmp = emitlist; tmp != NULL; tmp = tmp->next)
        signal_remove(signal, tmp->data);
    g_slist_free(emitlist);

    g_free(signal);
}

static gboolean plugin_destroyed(PLUGIN_REC *plugin)
{
    g_return_val_if_fail(plugin != NULL, FALSE);

    g_hash_table_freeze(plugin->signals);
    g_hash_table_foreach(plugin->signals, (GHFunc) destroy_all_binds, plugin);
    g_hash_table_thaw(plugin->signals);

    g_hash_table_destroy(plugin->signals);
    return TRUE;
}

void plugin_commands_init(void)
{
    signal_add("plugin created", (SIGNAL_FUNC) plugin_created);
    signal_add("plugin destroyed", (SIGNAL_FUNC) plugin_destroyed);
}

void plugin_commands_deinit(void)
{
    signal_remove("plugin created", (SIGNAL_FUNC) plugin_created);
    signal_remove("plugin destroyed", (SIGNAL_FUNC) plugin_destroyed);
}
