/*
 setup-lists.c : Irssi settings

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

extern gchar *default_config;

/* lists */
GList *aliases, *ignores, *completions, *notifies, *hilights, *replaces, *popups;

typedef LIST_REC *(LIST_FUNC) (GList **, gchar *, gchar *);

/* add new item to list */
static LIST_REC *popup_list_add(GList **list, gchar *key, gchar *value)
{
    LIST_REC *rec;

    g_return_val_if_fail(list != NULL, NULL);
    g_return_val_if_fail(key != NULL, NULL);
    g_return_val_if_fail(value != NULL, NULL);

    rec = g_new(LIST_REC, 1);
    rec->key = g_strdup(key);
    rec->value = g_strdup(value);

    *list = g_list_append(*list, rec);
    return rec;
}

static gboolean read_list(GList **list, gchar *section, gchar *key, gchar *value, LIST_FUNC addfunc)
{
    proplist_t sprop, iprop;
    gchar *rkey, *rvalue;
    gint num, max;

    g_return_val_if_fail(list != NULL, FALSE);
    g_return_val_if_fail(section != NULL, FALSE);

    *list = NULL;

    sprop = config_get_prop(cprop, section);
    if (sprop == NULL) return FALSE;

    max = PLGetNumberOfElements(sprop);
    for (num = 0; num < max; num++)
    {
	/* get the key & value */
	iprop = PLGetArrayElement(sprop, num);

	rkey = config_get_str(iprop, key, NULL);
	rvalue = config_get_str(iprop, value, NULL);
	if (rkey == NULL || rvalue == NULL) continue; /* hm?? */

	addfunc(list, rkey, rvalue);
    }

    return TRUE;
}

static LIST_REC *setup_ignore_add(GList **list, gchar *key, gchar *value)
{
    return ignore_add(key, value);
}

static LIST_REC *setup_notifylist_add(GList **list, gchar *key, gchar *value)
{
    return notifylist_add(key, value);
}

static void setup_read_internal_popups(void)
{
    proplist_t oldprop;
    gchar *fname, *home;
    FILE *f;

    oldprop = cprop;

    /* write internal config to some temp file */
    home = convert_home("~/.irssi");
    fname = tempnam(home, "int"); g_free(home);
    if (fname == NULL) return;

    f = fopen(fname, "w");
    if (f == NULL)
    {
	g_free(fname);
	return;
    }
    fwrite(default_config, strlen(default_config), 1, f);
    fclose(f);

    /* read popups from temp file.. */
    cprop = PLGetProplistWithPath(fname);
    if (cprop != NULL)
    {
	read_list(&popups, "popups", "label", "command", popup_list_add);
	PLRelease(cprop);
    }
    cprop = oldprop;

    g_free(fname);
}

void setup_lists_init(void)
{
    read_list(&aliases, "aliases", "alias", "command", list_add);
    read_list(&ignores, "ignores", "mask", "level", setup_ignore_add);
    read_list(&completions, "completions", "short", "long", list_add);
    read_list(&notifies, "notifies", "mask", "ircnets", setup_notifylist_add);
    read_list(&hilights, "hilights", "text", "channels", list_add);
    read_list(&replaces, "replaces", "text", "replace", list_add);
    if (!read_list(&popups, "popups", "label", "command", popup_list_add))
    {
	/* Popup menus not found from config file, use the internal.. */
        setup_read_internal_popups();
    }
}

void setup_lists_deinit(void)
{
    while (aliases != NULL)
        list_remove(&aliases, aliases->data);

    while (ignores != NULL)
        ignore_remove(((LIST_REC *) ignores->data)->key, "");

    while (completions != NULL)
        list_remove(&completions, completions->data);

    while (notifies != NULL)
        notifylist_remove(notifies->data);

    while (hilights != NULL)
        list_remove(&hilights, hilights->data);

    while (replaces != NULL)
        list_remove(&replaces, replaces->data);

    while (popups != NULL)
        list_remove(&popups, popups->data);
}
