/*
 ui-events.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gchar *ret_texts[] =
{
    "Invalid parameter",
    "Not enough parameters given",
    "Not connected to IRC server yet",
    "Not joined to any channels yet",
    "Error: getsockname() failed",
    "Error: listen() failed",
    "Multiple matches found, be more specific",
    "Nick not found",
    "Not joined to such channel",
    "Server not found",
    "Channel not fully synchronized yet, try again after a while",
    "Doing this is not a good idea. Add -YES if yoou really mean it",
};

static gboolean event_privmsg(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    CHANNEL_REC *chanrec;
    gchar *params, *target, *msg;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2 | PARAM_FLAG_GETREST, &target, &msg);
    if (nick == NULL) nick = server->real_address;

    if (*msg == 1)
    {
        /* ctcp message, handled in ui-ctcp.c */
    }
    else if (ignore_check(server, nick, addr, ischannel(*target) ? MSGLEVEL_PUBLIC : MSGLEVEL_MSGS) ||
	     (ischannel(*target) && ignore_check(server, target, NULL, MSGLEVEL_PUBLIC)))
    {
	/* ignored */
    }
    else if (ischannel(*target))
    {
        /* message to some channel */
        gboolean toyou;

	chanrec = channel_find(server, target);
        toyou = ui_completion_msgtoyou(server, msg);

        if (chanrec != NULL && CHANNEL_PARENT(chanrec)->active == chanrec)
        {
            /* message to active channel in window */
	    printformat(server, target, MSGLEVEL_PUBLIC | (toyou ? MSGLEVEL_NOHILIGHT : 0),
                        toyou ? IRCTXT_PUBMSG_ME : IRCTXT_PUBMSG, nick, msg);
        }
        else
        {
            /* message to not existing/active channel */
            printformat(server, target, MSGLEVEL_PUBLIC | (toyou ? MSGLEVEL_NOHILIGHT : 0),
                        toyou ? IRCTXT_PUBMSG_ME_CHANNEL : IRCTXT_PUBMSG_CHANNEL,
                        nick, target, msg);
        }
    }
    else
    {
        /* private message */
        if (setup_get_bool("toggle_autocreate_query") && channel_find(server, nick) == NULL)
	    chanrec = channel_create(server, nick, CHANNEL_TYPE_QUERY);
	else
	    chanrec = channel_find(server, nick);
	if (setup_get_bool("toggle_privmsg_beeps"))
	    ui_beep();

	printformat(server, nick, MSGLEVEL_MSGS,
		    chanrec == NULL ? IRCTXT_MSG_PRIVATE : IRCTXT_MSG_PRIVATE_QUERY, nick, addr == NULL ? "" : addr, msg);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_notice(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    gchar *params, *target, *msg;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2 | PARAM_FLAG_GETREST, &target, &msg);
    if (nick == NULL) nick = server->real_address;

    if (*msg == 1)
    {
        /* ctcp reply */
    }
    else if (addr == NULL)
    {
        /* notice from server */
        if (nick == NULL || !ignore_check(server, nick, "", MSGLEVEL_SNOTES))
	    printformat(server, target, MSGLEVEL_SNOTES, IRCTXT_NOTICE_SERVER, nick == NULL ? "" : nick, msg);
    }
    else if (ischannel(*target))
    {
        /* notice in some channel */
	if (!ignore_check(server, nick, addr, MSGLEVEL_NOTICES) &&
	    !ignore_check(server, target, NULL, MSGLEVEL_NOTICES))
	    printformat(server, target, MSGLEVEL_NOTICES, IRCTXT_NOTICE_PUBLIC, nick, target, msg);
    }
    else
    {
        /* private notice */
        if (!ignore_check(server, nick, addr, MSGLEVEL_NOTICES))
	    printformat(server, nick, MSGLEVEL_NOTICES, IRCTXT_NOTICE_PRIVATE, nick, addr, msg);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_join(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    gchar *params, *channel, *tmp;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 1, &channel);
    tmp = strchr(channel, 7); /* ^G does something weird.. */
    if (tmp != NULL) *tmp = '\0';

    if (!ignore_check(server, nick, addr, MSGLEVEL_JOINS) &&
	!ignore_check(server, channel, NULL, MSGLEVEL_JOINS))
	printformat(server, channel, MSGLEVEL_JOINS, IRCTXT_JOIN, nick, addr, channel);
    g_free(params);
    return TRUE;
}

static gboolean event_part(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    gchar *params, *channel, *reason;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2 | PARAM_FLAG_GETREST, &channel, &reason);

    if (!ignore_check(server, nick, addr, MSGLEVEL_PARTS) &&
	!ignore_check(server, channel, NULL, MSGLEVEL_PARTS))
	printformat(server, channel, MSGLEVEL_PARTS, IRCTXT_PART, nick, addr, channel, reason);
    g_free(params);
    return TRUE;
}

static gboolean event_quit(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);

    if (ignore_check(server, nick, addr, MSGLEVEL_QUITS))
	return TRUE;

    if (*data == ':') data++; /* quit message */
    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *channel = tmp->data;

	if (channel->server == server && nicklist_find(channel, nick) &&
	    !ignore_check(server, channel->name, NULL, MSGLEVEL_QUITS))
	    printformat(server, channel->name, MSGLEVEL_QUITS, IRCTXT_QUIT, nick, addr, data);
    }

    return TRUE;
}

static gboolean event_kick(gchar *data, SERVER_REC *server, gchar *kicker, gchar *addr)
{
    gchar *params, *channel, *nick, *reason;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 3 | PARAM_FLAG_GETREST, &channel, &nick, &reason);
    if (!ignore_check(server, kicker, addr, MSGLEVEL_KICKS) &&
	!ignore_check(server, channel, NULL, MSGLEVEL_KICKS))
    {
	printformat(server, channel, MSGLEVEL_KICKS,
		    IRCTXT_KICK, nick, channel, kicker, reason);
    }
    g_free(params);
    return TRUE;
}

static gboolean event_nick(gchar *data, SERVER_REC *server, gchar *sender, gchar *addr)
{
    gboolean ownnick, msgprint;
    gchar *params, *newnick;
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);

    if (ignore_check(server, sender, addr, MSGLEVEL_NICKS))
	return TRUE;

    params = event_get_params(data, 1, &newnick);

    msgprint = FALSE;
    ownnick = g_strcasecmp(sender, server->nick) == 0;
    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *chanrec = tmp->data;
        gboolean found;

        if (chanrec->server != server) continue;

        found = FALSE;
        if (chanrec->type == CHANNEL_TYPE_CHANNEL)
        {
            /* channel */
            if (nicklist_find(chanrec, sender))
                found = TRUE;
        }
        else if (chanrec->type == CHANNEL_TYPE_QUERY)
        {
            /* query */
            if (g_strcasecmp(chanrec->name, sender) == 0)
                found = TRUE;
        }

        if (found)
        {
            /* print nick change to window.. */
	    if (!ignore_check(server, chanrec->name, NULL, MSGLEVEL_NICKS))
	    {
		if (!ownnick)
		    printformat(server, chanrec->name, MSGLEVEL_NICKS, IRCTXT_NICK_CHANGED, sender, newnick);
		else
		    printformat(server, chanrec->name, MSGLEVEL_NICKS, IRCTXT_YOUR_NICK_CHANGED, newnick);
	    }
            msgprint = TRUE;
        }
    }

    if (!msgprint && ownnick)
        printformat(server, NULL, MSGLEVEL_NICKS, IRCTXT_YOUR_NICK_CHANGED, newnick);
    g_free(params);

    return TRUE;
}

static gboolean event_mode(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    gchar *params, *channel, *mode;

    g_return_val_if_fail(data != NULL, FALSE);
    if (nick == NULL) nick = server->real_address;

    params = event_get_params(data, 2 | PARAM_FLAG_GETREST, &channel, &mode);
    if (!ischannel(*channel))
    {
        /* user mode change */
	if ((addr == NULL && nick != NULL && !ignore_check(server, nick, NULL, MSGLEVEL_SMODES)) ||
	    (addr != NULL && !ignore_check(server, nick, addr, MSGLEVEL_MODES)))
	    printformat(server, NULL, MSGLEVEL_MODES, IRCTXT_USERMODE_CHANGE, mode, channel);
    }
    else
    {
        /* channel mode change */
	if (addr == NULL)
	{
	    /* changed by server */
	    if ((nick == NULL || !ignore_check(server, nick, "", MSGLEVEL_SMODES)) &&
		!ignore_check(server, channel, NULL, MSGLEVEL_SMODES))
	    {
		printformat(server, channel, MSGLEVEL_SMODES,
			    IRCTXT_SERVER_CHANMODE_CHANGE, channel, mode, nick);
	    }
	}
	else
	{
	    /* changed by normal user */
	    if (!ignore_check(server, nick, addr, MSGLEVEL_MODES) &&
		!ignore_check(server, channel, NULL, MSGLEVEL_MODES))
	    {
		printformat(server, channel, MSGLEVEL_MODES,
			    IRCTXT_CHANMODE_CHANGE, channel, mode, nick);
	    }
	}
    }
    g_free(params);
    return TRUE;
}

static gboolean event_pong(gchar *data, SERVER_REC *server, gchar *nick)
{
    gchar *params, *host, *reply;

    g_return_val_if_fail(data != NULL, FALSE);
    if (nick == NULL) nick = server->real_address;

    params = event_get_params(data, 2 | PARAM_FLAG_GETREST, &host, &reply);
    if (nick == NULL || !ignore_check(server, nick, "", MSGLEVEL_PONGS))
	printformat(server, NULL, MSGLEVEL_PONGS, IRCTXT_PONG, host, reply);
    g_free(params);
    return TRUE;
}

static gboolean event_invite(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    gchar *params, *channel;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &channel);
    if (*channel != '\0' && !ignore_check(server, nick, addr, MSGLEVEL_INVITES) &&
	!ignore_check(server, channel, NULL, MSGLEVEL_INVITES))
        printformat(server, NULL, MSGLEVEL_INVITES, IRCTXT_INVITE, nick, channel);
    g_free(params);
    return TRUE;
}

static gboolean event_topic(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    gchar *params, *channel, *topic;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2 | PARAM_FLAG_GETREST, &channel, &topic);

    if (!ignore_check(server, nick, addr, MSGLEVEL_TOPICS) &&
	!ignore_check(server, channel, NULL, MSGLEVEL_TOPICS))
	printformat(server, channel, MSGLEVEL_TOPICS,
		    *topic != '\0' ? IRCTXT_NEW_TOPIC : IRCTXT_TOPIC_UNSET,
		    nick, channel, topic);
    g_free(params);
    return TRUE;
}

static gboolean event_error(gchar *data, SERVER_REC *server)
{
    g_return_val_if_fail(data != NULL, FALSE);

    if (*data == ':') data++;
    printformat(server, NULL, MSGLEVEL_CRAP, IRCTXT_ERROR, data);
    return TRUE;
}

static gboolean event_wallops(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    g_return_val_if_fail(data != NULL, FALSE);

    if (*data == ':') data++;
    if (!ignore_check(server, nick == NULL ? "" : nick, addr, MSGLEVEL_WALLOPS))
    {
	if (g_strncasecmp(data, "\001ACTION", 7) != 0)
	    printformat(server, NULL, MSGLEVEL_WALLOPS, IRCTXT_WALLOPS, nick, data);
	else
	{
	    /* Action in WALLOP */
	    gint len;

	    data = g_strdup(data);
	    len = strlen(data);
            if (data[len-1] == 1) data[len-1] = '\0';
	    printformat(server, NULL, MSGLEVEL_WALLOPS, IRCTXT_ACTION_WALLOPS, nick, data);
	    g_free(data);
	}
    }
    return TRUE;
}

static gboolean channel_sync(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    printformat(channel->server, channel->name, MSGLEVEL_CLIENTNOTICE|MSGLEVEL_NO_ACT, IRCTXT_CHANNEL_SYNCED,
                channel->name, (glong) (time(NULL)-channel->createtime));
    return TRUE;
}

static gboolean notifylist_joined(SERVER_REC *server, gchar *nick, gchar *username, gchar *host, gchar *realname)
{
    g_return_val_if_fail(nick != NULL, FALSE);

    printformat(server, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_IRCJOIN,
		nick, username, host, realname,
		server->ircnet == NULL ? "IRC" : server->ircnet);
    return TRUE;
}

static gboolean notifylist_left(SERVER_REC *server, gchar *nick)
{
    g_return_val_if_fail(nick != NULL, FALSE);

    printformat(server, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_IRCPART,
		nick, server->ircnet == NULL ? "IRC" : server->ircnet);
    return TRUE;
}

static gboolean event_connected(SERVER_REC *server)
{
    gchar *str;

    g_return_val_if_fail(server != NULL, FALSE);

    if (g_strcasecmp(server->nick, setup_get_str("default_nick")) == 0)
        return TRUE;

    /* someone has our nick, find out who. */
    str = g_strdup_printf("WHOIS %s", setup_get_str("default_nick"));
    irc_send_cmd(server, str);
    g_free(str);

    server_redirect_event(server, setup_get_str("default_nick"), 1,
			  "event 318", "event empty", 1,
			  "event 401", "event empty", 1,
			  "event 311", "nickfind event whois", 1,
			  "event 301", "event empty", 1,
			  "event 312", "event empty", 1,
			  "event 313", "event empty", 1,
			  "event 317", "event empty", 1,
			  "event 319", "event empty", 1, NULL);

    return TRUE;
}

static gboolean event_nickfind_whois(gchar *data, SERVER_REC *server)
{
    gchar *params, *nick, *user, *host, *realname;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 6, NULL, &nick, &user, &host, NULL, &realname);
    printformat(server, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_YOUR_NICK_OWNED, nick, user, host, realname);
    g_free(params);
    return TRUE;
}

static gboolean event_autoignore_new(SERVER_REC *server, AUTOIGNORE_REC *ignore)
{
    g_return_val_if_fail(ignore != NULL, FALSE);

    printformat(server, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_AUTOIGNORE,
                ignore->nick, (ignore->timeleft+59)/60);
    return TRUE;
}

static gboolean event_autoignore_remove(SERVER_REC *server, AUTOIGNORE_REC *ignore)
{
    g_return_val_if_fail(ignore != NULL, FALSE);

    printformat(server, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_AUTOUNIGNORE, ignore->nick);
    return TRUE;
}

static gboolean event_ban_type_changed(gchar *bantype)
{
    GString *str;

    g_return_val_if_fail(bantype != NULL, FALSE);

    if (strcmp(bantype, "UD") == 0)
        printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_BANTYPE, "Normal");
    else if (strcmp(bantype, "HD") == 0 || strcmp(bantype, "H") == 0)
        printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_BANTYPE, "Host");
    else if (strcmp(bantype, "D") == 0)
        printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_BANTYPE, "Domain");
    else
    {
        str = g_string_new("Custom:");
        if (*bantype == 'N')
        {
            g_string_append(str, " Nick");
            bantype++;
        }
        if (*bantype == 'U')
        {
            g_string_append(str, " User");
            bantype++;
        }
        if (*bantype == 'H')
        {
            g_string_append(str, " Host");
            bantype++;
        }
        if (*bantype == 'D')
            g_string_append(str, " Domain");

        printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_BANTYPE, str->str);
        g_string_free(str, TRUE);
    }
    return TRUE;
}

static gboolean event_plugin_load_error(gchar *plugin, gchar *error)
{
    printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_PLUGIN_LOAD_ERROR, plugin, error);
    return TRUE;
}

static gboolean event_plugin_invalid(gchar *plugin)
{
    printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_PLUGIN_INVALID, plugin);
    return TRUE;
}

static gboolean event_plugin_invalid_version(gchar *plugin)
{
    printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_PLUGIN_INVALID_VERSION, plugin);
    return TRUE;
}

static gboolean event_plugin_already_loaded(gchar *plugin)
{
    printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_PLUGIN_ALREADY_LOADED, plugin);
    return TRUE;
}

static gboolean event_plugin_not_loaded(gchar *plugin)
{
    printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_PLUGIN_NOT_LOADED, plugin);
    return TRUE;
}

static gboolean event_cmderror(gpointer error)
{
    printtext(NULL, NULL, MSGLEVEL_CLIENTERROR, ret_texts[GPOINTER_TO_INT(error)]);
    return TRUE;
}

static gboolean event_received(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    g_return_val_if_fail(data != NULL, FALSE);

    if (!isdigit((gint) *data))
        printtext(server, NULL, MSGLEVEL_CRAP, "%s", data);
    else
    {
        /* numeric event. */
        gchar *params, *cmd, *args, *ptr;

        params = event_get_params(data, 3 | PARAM_FLAG_GETREST, &cmd, NULL, &args);
        ptr = strstr(args, " :");
        if (ptr != NULL) *(ptr+1) = ' ';
        printtext(server, NULL, MSGLEVEL_CRAP, "%s", args);
        g_free(params);
    }

    return TRUE;
}

void ui_events_init(void)
{
    signal_add("event privmsg", (SIGNAL_FUNC) event_privmsg);
    signal_add("event notice", (SIGNAL_FUNC) event_notice);
    signal_add("event join", (SIGNAL_FUNC) event_join);
    signal_add("event part", (SIGNAL_FUNC) event_part);
    signal_add("event quit", (SIGNAL_FUNC) event_quit);
    signal_add("event kick", (SIGNAL_FUNC) event_kick);
    signal_add("event nick", (SIGNAL_FUNC) event_nick);
    signal_add("event mode", (SIGNAL_FUNC) event_mode);
    signal_add("event pong", (SIGNAL_FUNC) event_pong);
    signal_add("event invite", (SIGNAL_FUNC) event_invite);
    signal_add("event topic", (SIGNAL_FUNC) event_topic);
    signal_add("event error", (SIGNAL_FUNC) event_error);
    signal_add("event wallops", (SIGNAL_FUNC) event_wallops);

    signal_add("default event", (SIGNAL_FUNC) event_received);

    signal_add("channel sync", (SIGNAL_FUNC) channel_sync);
    signal_add("notifylist joined", (SIGNAL_FUNC) notifylist_joined);
    signal_add("notifylist left", (SIGNAL_FUNC) notifylist_left);
    signal_add("event connected", (SIGNAL_FUNC) event_connected);
    signal_add("nickfind event whois", (SIGNAL_FUNC) event_nickfind_whois);
    signal_add("autoignore new", (SIGNAL_FUNC) event_autoignore_new);
    signal_add("autoignore remove", (SIGNAL_FUNC) event_autoignore_remove);
    signal_add("ban type changed", (SIGNAL_FUNC) event_ban_type_changed);
    signal_add("plugin error invalid", (SIGNAL_FUNC) event_plugin_invalid);
    signal_add("plugin error version", (SIGNAL_FUNC) event_plugin_invalid_version);
    signal_add("plugin error load", (SIGNAL_FUNC) event_plugin_load_error);
    signal_add("plugin error already loaded", (SIGNAL_FUNC) event_plugin_already_loaded);
    signal_add("plugin error not loaded", (SIGNAL_FUNC) event_plugin_not_loaded);
    signal_add("error command", (SIGNAL_FUNC) event_cmderror);
}

void ui_events_deinit(void)
{
    signal_remove("event privmsg", (SIGNAL_FUNC) event_privmsg);
    signal_remove("event notice", (SIGNAL_FUNC) event_notice);
    signal_remove("event join", (SIGNAL_FUNC) event_join);
    signal_remove("event part", (SIGNAL_FUNC) event_part);
    signal_remove("event quit", (SIGNAL_FUNC) event_quit);
    signal_remove("event kick", (SIGNAL_FUNC) event_kick);
    signal_remove("event nick", (SIGNAL_FUNC) event_nick);
    signal_remove("event mode", (SIGNAL_FUNC) event_mode);
    signal_remove("event pong", (SIGNAL_FUNC) event_pong);
    signal_remove("event invite", (SIGNAL_FUNC) event_invite);
    signal_remove("event topic", (SIGNAL_FUNC) event_topic);
    signal_remove("event error", (SIGNAL_FUNC) event_error);
    signal_remove("event wallops", (SIGNAL_FUNC) event_wallops);

    signal_remove("default event", (SIGNAL_FUNC) event_received);

    signal_remove("channel sync", (SIGNAL_FUNC) channel_sync);
    signal_remove("notifylist joined", (SIGNAL_FUNC) notifylist_joined);
    signal_remove("notifylist left", (SIGNAL_FUNC) notifylist_left);
    signal_remove("event connected", (SIGNAL_FUNC) event_connected);
    signal_remove("nickfind event whois", (SIGNAL_FUNC) event_nickfind_whois);
    signal_remove("autoignore new", (SIGNAL_FUNC) event_autoignore_new);
    signal_remove("autoignore remove", (SIGNAL_FUNC) event_autoignore_remove);
    signal_remove("ban type changed", (SIGNAL_FUNC) event_ban_type_changed);
    signal_remove("plugin error invalid", (SIGNAL_FUNC) event_plugin_invalid);
    signal_remove("plugin error version", (SIGNAL_FUNC) event_plugin_invalid_version);
    signal_remove("plugin error load", (SIGNAL_FUNC) event_plugin_load_error);
    signal_remove("plugin error already loaded", (SIGNAL_FUNC) event_plugin_already_loaded);
    signal_remove("plugin error not loaded", (SIGNAL_FUNC) event_plugin_not_loaded);
    signal_remove("error command", (SIGNAL_FUNC) event_cmderror);
}
