/*
 ui-server.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gboolean ui_server_looking(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    printformat(server, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_LOOKING_UP, server->address);
    return TRUE;
}

static gboolean ui_server_connecting(SERVER_REC *server, IPADDR *ip)
{
    g_return_val_if_fail(server != NULL, FALSE);
    g_return_val_if_fail(ip != NULL, FALSE);

    printformat(server, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_CONNECTING,
                server->address, net_ip2host(ip), server->port);
    return TRUE;
}

static gboolean ui_server_connected(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    printformat(server, NULL, MSGLEVEL_CLIENTNOTICE,
                IRCTXT_CONNECTION_ESTABLISHED, server->address);
    return TRUE;
}

static gboolean ui_connect_failed(SERVER_REC *server, gchar *msg)
{
    g_return_val_if_fail(server != NULL, FALSE);

    if (msg == NULL)
    {
	/* no message so this wasn't unexpected fail - send
	   connection_lost message instead */
	printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE,
		    IRCTXT_CONNECTION_LOST, server->address);
    }
    else
    {
	printformat(NULL, NULL, MSGLEVEL_CLIENTERROR,
		    IRCTXT_CANT_CONNECT, server->address, server->port, msg);
    }
    return TRUE;
}

static gboolean ui_server_disconnected(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE,
                IRCTXT_CONNECTION_LOST, server->address);
    return TRUE;
}

static gboolean ui_server_lag_disconnected(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE,
                IRCTXT_LAG_DISCONNECTED, server->address, time(NULL)-server->lag_sent);
    return TRUE;
}

static gboolean event_motd(gchar *data, SERVER_REC *server, gchar *nick, gchar *addr)
{
    /* numeric event. */
    gchar *params, *args, *ptr;

    if (setup_get_bool("toggle_skip_motd"))
	return TRUE;

    params = event_get_params(data, 2 | PARAM_FLAG_GETREST, NULL, &args);
    ptr = strstr(args, " :");
    if (ptr != NULL) *(ptr+1) = ' ';
    printtext(server, NULL, MSGLEVEL_CRAP, "%s", args);
    g_free(params);
    return TRUE;
}

void ui_server_init(void)
{
    signal_add("server looking", (SIGNAL_FUNC) ui_server_looking);
    signal_add("server connecting", (SIGNAL_FUNC) ui_server_connecting);
    signal_add("server connected", (SIGNAL_FUNC) ui_server_connected);
    signal_add("server connect failed", (SIGNAL_FUNC) ui_connect_failed);
    signal_add("server disconnected", (SIGNAL_FUNC) ui_server_disconnected);
    signal_add("server lag disconnect", (SIGNAL_FUNC) ui_server_lag_disconnected);
    signal_add("event 375", (SIGNAL_FUNC) event_motd);
    signal_add("event 376", (SIGNAL_FUNC) event_motd);
    signal_add("event 372", (SIGNAL_FUNC) event_motd);
}

void ui_server_deinit(void)
{
    signal_remove("server looking", (SIGNAL_FUNC) ui_server_looking);
    signal_remove("server connecting", (SIGNAL_FUNC) ui_server_connecting);
    signal_remove("server connected", (SIGNAL_FUNC) ui_server_connected);
    signal_remove("server connect failed", (SIGNAL_FUNC) ui_connect_failed);
    signal_remove("server disconnected", (SIGNAL_FUNC) ui_server_disconnected);
    signal_remove("server lag disconnect", (SIGNAL_FUNC) ui_server_lag_disconnected);
    signal_remove("event 375", (SIGNAL_FUNC) event_motd);
    signal_remove("event 376", (SIGNAL_FUNC) event_motd);
    signal_remove("event 372", (SIGNAL_FUNC) event_motd);
}
