#include "links.h"

#ifdef LEAK_DEBUG
long mem_amount = 0;
long last_mem_amount = -1;
#ifdef MAX_DEBUG_SIZE
long debug_sizes[MAX_DEBUG_SIZE];
long last_debug_sizes[MAX_DEBUG_SIZE];
#endif
#ifdef MAX_LIST_SIZE
struct md memory_list[MAX_LIST_SIZE];
struct md last_memory_list[MAX_LIST_SIZE];
#endif
#endif

static inline void force_dump()
{
	fprintf(stderr, "\n\033[1mForcing core dump\033[0m\n");
	raise(SIGSEGV);
}

void check_memory_leaks()
{
#ifdef LEAK_DEBUG
	if (mem_amount) {
		fprintf(stderr, "\n\033[1mMemory leak by %ld bytes\033[0m\n", mem_amount);
#ifdef MAX_DEBUG_SIZE
		{
			int i;
			int r = 0;
			fprintf(stderr, "Differences: ");
			for (i = 0; i < MAX_DEBUG_SIZE; i++) if (debug_sizes[i])
				fprintf(stderr, "%s%ld x %d", r ? ", " : "", debug_sizes[i], i), r = 1;
			fprintf(stderr, "\n");
		}
#ifdef MAX_LIST_SIZE
		{
			int i;
			int r = 0;
			fprintf(stderr, "List: ");
			for (i = 0; i < MAX_LIST_SIZE; i++) if (memory_list[i].p) {
				fprintf(stderr, "%s%p:%d @ %s:%d", r ? ", " : "", memory_list[i].p, (int)memory_list[i].size, memory_list[i].file, memory_list[i].line), r = 1;
				if (memory_list[i].comment) fprintf(stderr, ":\"%s\"", memory_list[i].comment);
			}
			fprintf(stderr, "\n");
		}
#endif
#endif
		force_dump();
	}
#endif
}

void err(int b, unsigned char *m, va_list l)
{
	if (b) fprintf(stderr, "%c", (char)7);
	vfprintf(stderr, m, l);
	fprintf(stderr, "\n");
	sleep(1);
}

void error(unsigned char *m, ...)
{
	va_list l;
	va_start(l, m);
	err(1, m, l);
}

int errline;
unsigned char *errfile;

unsigned char errbuf[4096];

void int_error(unsigned char *m, ...)
{
	va_list l;
	va_start(l, m);
	sprintf(errbuf, "\033[1mINTERNAL ERROR\033[0m at %s:%d: ", errfile, errline);
	strcat(errbuf, m);
	err(1, errbuf, l);
	force_dump();
}

void debug_msg(unsigned char *m, ...)
{
	va_list l;
	va_start(l, m);
	sprintf(errbuf, "DEBUG MESSAGE at %s:%d: ", errfile, errline);
	strcat(errbuf, m);
	err(0, errbuf, l);
}

#ifdef LEAK_DEBUG

void *debug_mem_alloc(unsigned char *file, int line, size_t size)
{
	void *p;
	if (!size) return DUMMY;
#ifdef LEAK_DEBUG
	mem_amount += size;
#ifdef MAX_DEBUG_SIZE
	if (size < MAX_DEBUG_SIZE) debug_sizes[size]++;
#endif
	size += L_D_S;
#endif
	if (!(p = xmalloc(size))) {
		error("ERROR: out of memory (malloc returned NULL)\n");
		return NULL;
	}
#ifdef LEAK_DEBUG
	*(size_t *)p = size - L_D_S;
	p = (char *)p + L_D_S;
#ifdef MAX_LIST_SIZE
	{
		long i;
		for (i = 0; i < MAX_LIST_SIZE; i++) if (!memory_list[i].p) {
			memory_list[i].p = p;
			memory_list[i].size = size;
			memory_list[i].file = file;
			memory_list[i].line = line;
			memory_list[i].comment = NULL;
			goto b;
		}
		debug("warning: memory list overflow");
		b:;
	}
#endif
#endif
	return p;
}

void debug_mem_free(unsigned char *file, int line, void *p)
{
	if (p == DUMMY) return;
	if (!p) {
		errfile = file, errline = line, int_error("mem_free(NULL)");
		return;
	}
#ifdef LEAK_DEBUG
#ifdef MAX_LIST_SIZE
	{
		long i;
		for (i = 0; i < MAX_LIST_SIZE; i++) if (memory_list[i].p == p) {
			memory_list[i].p = NULL;
			goto b;
		}
		errfile = file, errline = line, int_error("free: address %p not from malloc", p);
		b:;
	}
#endif
	p = (char *)p - L_D_S;
	mem_amount -= *(size_t *)p;
#ifdef MAX_DEBUG_SIZE
	if (*(int *)p < MAX_DEBUG_SIZE) debug_sizes[*(int *)p]--;
#endif
#endif
	xfree(p);
}

void *debug_mem_realloc(unsigned char *file, int line, void *p, size_t size)
{
#ifdef LEAK_DEBUG
#ifdef MAX_LIST_SIZE
	void *q = p;
#endif
#endif
	if (p == DUMMY) return debug_mem_alloc(file, line, size);
	if (!p) {
		errfile = file, errline = line, int_error("mem_realloc(NULL, %d)", size);
		return NULL;
	}
	if (!size) {
		debug_mem_free(file, line, p);
		return DUMMY;
	}
#ifdef LEAK_DEBUG
	p = (char *)p - L_D_S;
	mem_amount += size - *(size_t *)p;
#ifdef MAX_DEBUG_SIZE
	if (size < MAX_DEBUG_SIZE) debug_sizes[size]++;
	if (*(int *)p < MAX_DEBUG_SIZE) debug_sizes[*(int *)p]--;
#endif
	*(size_t *)p = size;
	size += L_D_S;
#endif
	if (!(p = xrealloc(p, size))) {
		error("ERROR: out of memory (realloc returned NULL)\n");
		return NULL;
	}
#ifdef LEAK_DEBUG
	p = (char *)p + L_D_S;
#ifdef MAX_LIST_SIZE
	{
		long i;
		for (i = 0; i < MAX_LIST_SIZE; i++) if (memory_list[i].p == q) {
			memory_list[i].p = p;
			memory_list[i].size = size;
			goto b;
		}
		errfile = file, errline = line, int_error("realloc: address %p not from malloc", p);
		b:;
	}
#endif
#endif
	return p;
}

void set_mem_comment(void *p, unsigned char *c, int l)
{
#ifdef MAX_LIST_SIZE
	long i;
	for (i = 0; i < MAX_LIST_SIZE; i++) if (memory_list[i].p == p) {
		if ((memory_list[i].comment = malloc(l + 1)))
			memcpy(memory_list[i].comment, c, l),
			memory_list[i].comment[l] = 0;
		goto b;
	}
	internal("set_mem_comment: not allocated");
	b:
#endif
}

#endif

#ifdef SPECIAL_MALLOC

struct malloc_chunk {
	struct malloc_block *ptr;
	int free;
};

struct malloc_block {
	int next:16;
#ifdef DEBUG
	int size:15;
	int free:1;
#else
	int size:16;
#endif
};

struct malloc_chunk *chunks = DUMMY;
int n_chunks = 0;
struct malloc_chunk *last_chunk = NULL;
int last_f_chunk;

#ifdef DEBUG

void sp_check()
{
	int i;
	for (i = 0; i < n_chunks; i++) {
		struct malloc_block *pr = chunks[i].ptr;
		while (pr->next) {
			pr = (struct malloc_block *)((char *)pr + pr->next);
			if (pr->size & (sizeof(struct malloc_block)-1)) internal("unaligned block; chunk %d, block %p", i, pr);
			if (pr->next & (sizeof(struct malloc_block)-1)) internal("unaligned block; chunk %d, block %p", i, pr);
			if (!pr->free) internal("block on freelist is not free; chunk %d, block %p", i, pr);
			if (pr < chunks[i].ptr || (void *)pr > (void *)((char *)chunks[i].ptr + CHUNK_SIZE)) internal("out of chunk %d, pointer %p @ 1", i, pr);
		}
		pr = chunks[i].ptr;
		while (pr->next) {
			pr = (struct malloc_block *)((char *)pr + pr->size);
			if (pr->size & (sizeof(struct malloc_block)-1)) internal("unaligned block; chunk %d, block %p", i, pr);
			if (pr->free && ((pr->next & sizeof(struct malloc_block)-1))) internal("unaligned block; chunk %d, block %p", i, pr);
			if (pr < chunks[i].ptr || (void *)pr > (void *)((char *)chunks[i].ptr + CHUNK_SIZE)) internal("out of chunk %d, pointer %p @ 1", i, pr);
		}
	}
}

#else

inline void sp_check() {}

#endif

void new_chunk()
{
	struct malloc_block *p;
	if (!(p = malloc(CHUNK_SIZE))) return;
	if (!(n_chunks & ALLOC_GR)) {
		struct malloc_chunk *ch;
		if (n_chunks) ch = realloc(chunks, (n_chunks + ALLOC_GR) * sizeof(struct malloc_chunk));
		else ch = malloc(ALLOC_GR * sizeof(struct malloc_chunk));
		if (!ch) {
			free(p);
			return;
		}
		chunks = ch;
	}
	last_chunk = &chunks[n_chunks++];
	last_chunk->ptr = p;
	last_chunk->free = CHUNK_SIZE - sizeof(struct malloc_block);
	p->next = sizeof(struct malloc_block);
	p->size = sizeof(struct malloc_block);;
#ifdef DEBUG
	p->free = 0;
#endif
	p++;
	p->next = 0;
	p->size = CHUNK_SIZE - sizeof(struct malloc_block);
#ifdef DEBUG
	p->free = 1;
#endif
	sp_check();
}

void *alloc_in_chunk(size_t size)
{
	struct malloc_block *pr, *b, *n;
	if (!last_chunk) return NULL;
	size = (size + 2 * sizeof(struct malloc_block) - 1) & ~(sizeof(struct malloc_block) - 1);
	pr = last_chunk->ptr;
	while (pr->next) {
		b = (struct malloc_block *)((char *)pr + pr->next);
#ifdef DEBUG
		if (!b->free) {
			internal("block on freelist is not free");
			pr = b;
			continue;
		}
#endif
		if (b->size >= size) goto found_it;
		pr = b;
	}
	return NULL;
	found_it:
	if (b->size <= size + sizeof(struct malloc_block)) {
#ifdef DEBUG
		b->free = 0;
#endif
		if (!b->next) pr->next = 0;
		else pr->next = (char *)b + b->next - (char *)pr;
		last_chunk->free -= b->size;
		sp_check();
		return b + 1;
	}
	n = (struct malloc_block *)((char *)b + size);
	if (n->next = b->next) n->next -= size;
	n->size = b->size - size;
#ifdef DEBUG
	n->free = 1;
#endif
	b->size = size;
#ifdef DEBUG
	b->free = 0;
#endif
	pr->next += size;
	last_chunk->free -= size;
	sp_check();
	return b + 1;
}

void *sp_malloc(size_t size)
{
	void *p;
	struct malloc_chunk *c, *cc;
	int f;
	if (size >= SMALL_MALLOC) {
		struct malloc_block *m = malloc(size + sizeof(struct malloc_block));
		if (m) {
			m->size = 0;
			return m + 1;
		}
		return NULL;
	}
	if (last_chunk) if ((p = alloc_in_chunk(size))) return p;
	f = 0;
	for (c = chunks; c < chunks + n_chunks; c++) if (c->free >= MIN_CHUNK_FREE && c->free > f) {
		cc = c;
		f = c->free;
	}
	if (f > 0) {
		last_chunk = cc;
		if ((p = alloc_in_chunk(size))) return p;
	}
	new_chunk();
	return alloc_in_chunk(size);
}

void sp_free(void *p)
{
	struct malloc_block *pr, *b, *f;
	int c;
	f = (struct malloc_block *)p - 1;
	if (!f->size) {
		free(f);
		return;
	}
	if (last_f_chunk < n_chunks && p >= (void *)chunks[last_f_chunk].ptr && p < (void *)((char *)chunks[last_f_chunk].ptr + CHUNK_SIZE)) {
		f:
		chunks[last_f_chunk].free += f->size;
		pr = chunks[last_f_chunk].ptr;
		while (pr->next) {
			b = (struct malloc_block *)((char *)pr + pr->next);
#ifdef DEBUG
			if (!b->free) {
				internal("block on freelist is not free");
				pr = b;
				continue;
			}
#endif
			if ((char *)b + b->size == (char *)f) {
				b->size += f->size;
				sp_check();
				return;
			}
			if (b < f) {
				pr = b;
				continue;
			}
			pr->next = (char *)f - (char *)pr;
			if ((char *)f + f->size != (char *)b) {
				f->next = (char *)b - (char *)f;
#ifdef DEBUG
				f->free = 1;
#endif
				sp_check();
				return;
			}
			if ((f->next = b->next)) f->next += f->size;
			f->size += b->size;
#ifdef DEBUG
			f->free = 1;
#endif
			sp_check();
			return;
		}
		b->next = (char *)f - (char *)b;
		f->next = 0;
#ifdef DEBUG
		f->free = 1;
#endif
	}
	for (last_f_chunk = 0; last_f_chunk < n_chunks; last_f_chunk++)
		if (p >= (void *)chunks[last_f_chunk].ptr && p < (void *)((char *)chunks[last_f_chunk].ptr + CHUNK_SIZE)) goto f;
	internal("sp_free: address %p not from sp_malloc", p);
}

void *sp_realloc(void *p, size_t psize)
{
	struct malloc_block *pr, *b, *f, *n;
	void *np;
	int c;
	int size = (psize + 2 * sizeof(struct malloc_block) - 1) & ~(sizeof(struct malloc_block) - 1);
	f = (struct malloc_block *)p - 1;
	if (!f->size) {
		b = realloc(f, size);
		if (b) return b + 1;
		return NULL;
	}
	if (f->size > size) goto shrink;
	if (last_f_chunk < n_chunks && p >= (void *)chunks[last_f_chunk].ptr && p < (void *)((char *)chunks[last_f_chunk].ptr + CHUNK_SIZE)) {
		f:
		pr = chunks[last_f_chunk].ptr;
		while (pr->next) {
			b = (struct malloc_block *)((char *)pr + pr->next);
#ifdef DEBUG
			if (!b->free) {
				internal("block on freelist is not free");
				break;
			}
#endif
			if ((char *)f + f->size != (char *)b)
				if (b > f) break;
				else {
					pr = b;
					continue;
				}
			if ((char *)f + size > (char *)b + b->size) break;
			if ((char *)f + size >= (char *)b + b->size - sizeof(struct malloc_block)) {
				if (!b->next) pr->next = 0;
				else pr->next = (char *)b + b->next - (char *)pr;
				last_chunk->free -= b->size;
				f->size += b->size;
				sp_check();
				return f + 1;
			}
			n = (struct malloc_block *)((char *)f + size);
			pr->next += (char *)n - (char *)b;
			if ((n->next = b->next)) n->next -= size - f->size;
			n->size = b->size - size + f->size;
#ifdef DEBUG
			n->free = 1;
#endif
			last_chunk->free -= size - f->size;
			f->size = size;
			sp_check();
			return f + 1;
		}
		if ((np = sp_malloc(psize))) {
			memcpy(np, p, f->size - sizeof(struct malloc_block));
			sp_free(p);
		}
		sp_check();
		return np;
	}
	for (last_f_chunk = 0; last_f_chunk < n_chunks; last_f_chunk++)
		if (p >= (void *)chunks[last_f_chunk].ptr && p < (void *)((char *)chunks[last_f_chunk].ptr + CHUNK_SIZE)) goto f;
	internal("sp_free: address %p not from sp_malloc", p);
	return p;

	shrink:
	if (size >= f->size - sizeof(struct malloc_block)) return p;
	n = (struct malloc_block *)((char *)f + size);
	n->size = f->size - size;
	f->size = size;
	sp_check();
	sp_free(n + 1);
	return p;
}

#endif
