/*
 * nwpurge.c,v1.00
 * 
 * Utility for purging deleted files from NetWare volumes
 *
 * Copyright (c) 1998 Petr Vandrovec
 *
 * For license see COPYING in main package directory
 *
 */

#include "config.h"

#include <ncp/nwcalls.h>
#include <unistd.h>

#include <libintl.h>
#define _(X) gettext(X)

void usage(void) {
	printf(_(
"usage: nwpurge [options] [directory]\n"
"\n"
"-h         Print this help text\n"
"-a         Purge files in subdirectories\n"
"-l         Do not purge files\n"
"-s         Silent mode\n"
"\n"
"directory  Directory to purge\n"
"\n"
));
}

int subdirs = 0;
int show = 1;
int purge = 1;
int files = 0;
int failures = 0;

void processpurge(struct ncp_conn* conn, int volume, u_int32_t directory_id) {
	struct ncp_deleted_file info;
	char tmp[1024];

	if (show) {
		if (!ncp_ns_get_full_name(conn, NW_NS_DOS, NW_NS_DOS,
			1, volume, directory_id, NULL, 0,
			tmp, sizeof(tmp))) {
			printf("%s\n", tmp);
		}
	}

	info.seq = -1;
	while (!ncp_ns_scan_salvageable_file(conn, NW_NS_DOS, 
			1, volume, directory_id, NULL, 0, 
			&info, tmp, sizeof(tmp))) {
		if (show) {
			printf("%8s%s\n", "", tmp);
		}
		files++;
		if (purge) {
			NWCCODE err;
			if ((err = ncp_ns_purge_file(conn, &info)) != 0) {
				failures++;
				if (!show) {
					printf("%8s%s\n", "", tmp);
				}
				printf(_("%8s-- failed (%s)\n"), "", strnwerror(err));
			}
		}
	}
	if (show) printf("\n");
	if (subdirs) {
		struct nw_info_struct dir;
		struct ncp_search_seq seq;

		dir.volNumber = volume;
		dir.dirEntNum = directory_id;
		if (!ncp_initialize_search(conn, &dir, NW_NS_DOS, &seq)) {
			while (!ncp_search_for_file_or_subdir2(conn,
						SA_SUBDIR_ALL, 
						/* RIM_ATTRIBUTES | */ RIM_DIRECTORY,
						&seq,
						&dir)) {
				processpurge(conn, dir.volNumber, dir.DosDirNum);
			}
		}

	}
}

int main(int argc, char* argv[]) {
	struct NWCCRootEntry root;
	const char* path;
	struct ncp_conn* conn;
	int err;
	int c;

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
	
	while ((c = getopt(argc, argv, "ahls")) != -1) {
		switch (c) {
			case '?':
			case ':':break;
			case 'a':subdirs = 1;
				 break;
			case 'h':usage(); exit(2);
			case 'l':purge = 0;
				 break;
			case 's':show = 0;
				 break;
			default: fprintf(stderr, _("Unexpected option `-%c'\n"), c);
				 break;
		}
	}
	if (optind < argc) {
		path = argv[optind++];
	} else {
		path = ".";
	}
	err = ncp_open_mount(path, &conn);
	if (err) {
		com_err("nwpurge", err, _("in ncp_open_mount"));
		exit(1);
	}
	err = NWCCGetConnInfo(conn, NWCC_INFO_ROOT_ENTRY, sizeof(root), &root);
	if (err) {
		com_err("nwpurge", err, _("when retrieving root entry"));
		goto finished;
	}
	processpurge(conn, root.volume, root.dirEnt);

	files = files-failures;
	if (!files) {
		printf(_("No files were purged from server.\n"));
	} else if (files == 1) {
		printf(_("1 file was purged from server.\n"));
	} else {
		printf(_("%d files were purged from server.\n"), files);
	}
	if (failures) {
		if (failures == 1) {
			printf(_("1 file was not purged due to error.\n"));
		} else {
			printf(_("%d files were not purged due to errors.\n"), failures);
		}
	}

finished:;
	ncp_close(conn);
	return 0;
}

