#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>

#include <config.h>
#include <support.h>
#include <xcio.h>
#include <sysmsg.h>

#include <grp.h>
#include <pwd.h>

#include "option.h"
#include "env.h"
#include "log.h"
#include "console.h"

struct envlist_s {
    struct envlist_s *next;
    struct env_s *envs;
    void (*setup)();
    const char *domain;
    u_int8_t f_setup;
};

static struct envlist_s *elHead;

int
RegisterEnvs(struct env_s *envs, const char *domain, void (*setup)())
{
    struct envlist_s *elp;

    elp = TALLOC(struct envlist_s);
    elp->domain = domain;
    elp->envs = envs;
    elp->setup = setup;
    elp->f_setup = FALSE;
/*
    if (type&ENV_MASK == ENV_SET) esp->d.set = d;
    else esp->d.entry = d;
*/
    if (!elHead) {
	elHead = elp;
    } else {
	struct envlist_s *elp1;

	elp1 = elHead;
	while (elp1->next) elp1 = elp1->next;
	elp1->next = elp;
    }
    elp->next = NULL;
    return(0);
}

static bool_t
EnvRunMode(int argc, char **argv, char *outs)
{
    static const char *runname[]={
	"passive",
	"active",
	"kick",
	"direct",
	"getty"
    };
    int i;

    if (!argc) {
	strcpy(outs, runname[pppOpt.mode]);
	return FALSE;
    }
    for (i = 0; i < RUN_MAX; i ++) {
	if (!strcasecmp(runname[i], argv[1])) {
	    pppOpt.mode = i;
	    return TRUE;
	}
    }
    return FALSE;
}

static bool_t
EnvDateTime(int argc, char **argv, char *outs)
{
    time_t now;
    struct tm *tm;

    time(&now);
    tm = localtime(&now);
    strftime(outs, 10,
	     (*argv[0] | ' ') == 'd' ? "%y%m%d": "%H%M%S", tm);
    return FALSE;
}

bool_t
EnvLine(int argc, char **argv, char *outs)
{
    char **line=pppOpt.line;
    int a;

    if (!argc) {
	if (line) while (*line) {
	    strcat(outs, *line);
	    line ++;
	    if (*line) strcat(outs, " ");
	}
	return FALSE;
    }
    if (SetDevice(&argv[1])) {
	Logf(LOG_ERROR, "%s: unknown device type\n", argv[1]);
	return FALSE;
    }
    if (line) {
	char *l0;

	while (*line) {
	    l0 = *line;
	    line ++;
	    Free(l0);
	}
	Free(pppOpt.line);
    }
    pppOpt.line = (char **)Malloc(argc * sizeof(char *));
    for (a = 1; a < argc; a ++)
	pppOpt.line[a - 1] = Strdup(argv[a]);
    pppOpt.line[a - 1] = NULL;
    return TRUE;
}

void
EnvSetup(const char *v)
{
    extern bool_t EnvAuthProto(), EnvFrame(), EnvPhase();
    static char *version;
    static struct env_s envlist[]={
	{"VERSION", {&version}, ENV_PRIVATE|ENV_RDONLY|ENV_STRING, 
		0, 0, 0444},
	{"DATE", {EnvDateTime}, ENV_RDONLY|ENV_SET, 0, 0, 0444},
	{"TIME", {EnvDateTime}, ENV_RDONLY|ENV_SET, 0, 0, 0444},
	{"FRAME", {EnvFrame}, ENV_SET, 0, 0, 0444},
	/*	{"PHASE", {EnvPhase}, ENV_SET|ENV_RDONLY},*/
	{"NAME", {&pppOpt.name}, ENV_STRING, 0, 0, 0666},
	{"MODE", {EnvRunMode}, ENV_SET, 0, 0, 0666},
	{"LINE", {EnvLine}, ENV_SET, 0, 0, 0666},
	{"CHAT", {&pppOpt.chat}, ENV_STRING, 0, 0, 0666},
	{NULL}
    };

    version = Strdup(v);
    RegisterEnvs(envlist, NULL, NULL);
}

static int
EnvAccess(struct env_s *esp, int mode) {
	uid_t acc_uid, env_uid;
	gid_t acc_gid, env_gid;
	u_int16_t env_mode;

	if (mode != (mode & 00007))
		return (-1);
	if (!esp)
		return (-1);
	if (mode == 0)
		return (0);

	env_uid = esp->user;
	env_gid = esp->group;
	acc_uid = geteuid();
	acc_gid = getegid();
	if (acc_uid == 0)
		return (0);
/*
ConsoleOutf("user %d:%d   group %d:%d   mode %o:%o\n",
	env_uid, acc_uid, env_gid, acc_gid, esp->mode, mode);
*/

	if (env_uid == acc_uid) {
		env_mode = (esp->mode >> 6) & 00007;
	} else if (env_gid == acc_gid) {
		env_mode = (esp->mode >> 3) & 00007;
	} else {
		env_mode = esp->mode & 00007;
	}

	if (env_mode & mode) {
		return (0);
	}
	return (1);
}

static char *
AEnvToStr(struct env_s *esp, env_t typemask)
{
    static char strenv[256];

    if (typemask & ENV_HIDE) {
	if ((esp->type & ENV_SECRET) && !ISLOG(LOG_SECRET))
	    return(NULL);
	if ((esp->type & ENV_PRIVATE) && !ISLOG(LOG_PRIVATE))
	    return(NULL);
    }
    if (typemask & ENV_RDONLY & esp->type) return(NULL);
    if (EnvAccess(esp, R_OK)) {
    	sprintf(strenv, "<hide>");
    	return (strenv);
    }
    strenv[0] = '\0';
    switch (esp->type&ENV_MASK) {
    case ENV_INT8:
	sprintf(strenv, "%d", *(int8_t *)esp->d.entry);
	break;
    case ENV_INT16:
	sprintf(strenv, "%d", *(int16_t *)esp->d.entry);
	break;
    case ENV_INT32:
	sprintf(strenv, "%d", *(int32_t *)esp->d.entry);
	break;
    case ENV_INT8X:
	sprintf(strenv, "%#0x", *(int8_t *)esp->d.entry);
	break;
    case ENV_INT16X:
	sprintf(strenv, "%#0x", *(int16_t *)esp->d.entry);
	break;
    case ENV_INT32X:
	sprintf(strenv, "%#0x", *(int32_t *)esp->d.entry);
	break;
    case ENV_BOOL:
	sprintf(strenv, (*(bool_t *)esp->d.entry == TRUE) ? "yes": "no");
	break;
    case ENV_STRING:
	if (*(char **)esp->d.entry)
	    sprintf(strenv, "%s", *(char **)esp->d.entry);
	break;
    case ENV_SET:
	esp->d.set(0, &esp->name, strenv);
	break;
    }
    return(strenv[0] ? strenv: NULL);
}

static char *
EnvModeStr(u_int16_t mode)
{
    static char modebuf[11];

    modebuf[0] = '-';
    modebuf[1] = (mode & 0400) ? 'r' : '-';
    modebuf[2] = (mode & 0200) ? 'w' : '-';
    modebuf[3] = (mode & 0100) ? 'x' : '-';
    modebuf[4] = (mode & 0040) ? 'r' : '-';
    modebuf[5] = (mode & 0020) ? 'w' : '-';
    modebuf[6] = (mode & 0010) ? 'x' : '-';
    modebuf[7] = (mode & 0004) ? 'r' : '-';
    modebuf[8] = (mode & 0002) ? 'w' : '-';
    modebuf[9] = (mode & 0001) ? 'x' : '-';
    modebuf[10] = 0;

    return modebuf;
}

static char *
EnvUserStr(uid_t uid)
{
    static char userbuf[9];
    struct passwd *pwe;

    pwe = getpwuid(uid);
    if (pwe) {
    	sprintf(userbuf, "%-8.8s", pwe->pw_name);
    } else {
    	sprintf(userbuf, "%5d   ",  uid);
    }

    return userbuf;
}

static char *
EnvGroupStr(gid_t gid)
{
    static char groupbuf[9];
    struct group *gre;

    gre = getgrgid(gid);
    if (gre) {
    	sprintf(groupbuf, "%-8.8s", gre->gr_name);
    } else {
    	sprintf(groupbuf, "%5d   ",  gid);
    }

    return groupbuf;
}

static void
AEnvPrint(FILE *fp, const char *domain, struct env_s *esp, env_t typemask)
{
    char *str;

    str = AEnvToStr(esp, typemask);
    if (str) {
	if (domain) {
	    if (fp) fprintf(fp, "set %s.%s %s\n", domain, esp->name, str);
	    else ConsoleOutf("%s %s %s %s.%s=%s\n",
	    			EnvModeStr(esp->mode),
	    			EnvUserStr(esp->user),
	    			EnvGroupStr(esp->group),
	    			domain, esp->name, str);
	} else {
	    if (fp) fprintf(fp, "%s set %s %s\n", esp->name, str);
	    else ConsoleOutf("%s %s %s %s=%s\n",
	    			EnvModeStr(esp->mode),
	    			EnvUserStr(esp->user),
	    			EnvGroupStr(esp->group),
	    			esp->name, str);
	}
    }
}

static struct env_s *
NameToEnv(char *fullname, struct envlist_s **epp)
{
    struct envlist_s *elp;
    struct env_s *esp;
    char *name, *domain;
    int i;

    elp = elHead;
    domain = Strdup(fullname);
    if ((name = strchr(domain, '.')) != NULL) {
	*name = '\0';
	name ++;
	elp = elp->next;
	while (elp) {
	    if (!strcasecmp(elp->domain, domain)) break;
	    elp = elp->next;
	}
    } else name = domain;
    if (epp) *epp = elp;
    if (elp) {
	esp = elp->envs;

	for (i = 0; esp[i].name; i ++) {
	    if (!strcasecmp(esp[i].name, name)) {
		Free(domain);
		return(&esp[i]);
	    }
	}
    }
    Free(domain);
    return(NULL);
}

char *
EnvToStr(char *fullname, env_t typemask)
{
    struct env_s *esp;

    if ((esp = NameToEnv(fullname, NULL)) != NULL)
	return(AEnvToStr(esp, typemask));
    return(NULL);
}

static void
EnvsPrint(FILE *fp, struct envlist_s *elp, env_t typemask)
{
    int i;
    struct env_s *esp=elp->envs;

    for (i = 0; esp[i].name; i ++)
	AEnvPrint(fp, elp->domain, &esp[i], typemask);
}

void
AllEnvPrint(FILE *fp, env_t typemask)
{
    struct envlist_s *elp;
    elp = elHead;

    while (elp) {
	if (elp->f_setup == TRUE) EnvsPrint(fp, elp, typemask);
	elp = elp->next;
    }
}

int
EnvPrint(int argc, char **argv)
{
    struct envlist_s *elp;
    struct env_s *esp;

    elp = elHead;
    if (argc > 1 && strcmp(argv[1],"all")) {
	int a;

	if (!strcmp(argv[1], "?")) {
	    elp = elHead->next;
	    a = 1;
	    while (elp) {
		if (!(a % 4)) ConsoleOutf("\n");
		ConsoleOutf("%-20s", elp->domain);
		elp = elp->next;
	    }
	    ConsoleOutf("\n");
	    return(0);
	}
	if (!strcmp(argv[1], ".")) {
	    EnvsPrint(NULL, elHead, 0);
	}
	for (a = 1; a < argc; a ++) {
	    if ((esp = NameToEnv(argv[a], &elp)) == NULL) {
		elp = elHead->next;
		while (elp) {
		    if (!strcasecmp(elp->domain, argv[a])) {
			EnvsPrint(NULL, elp, 0);
			break;
		    }
		    elp = elp->next;
		}
	    } else AEnvPrint(NULL, elp->domain, esp, 0);
	}
	return(0);
    }
    AllEnvPrint(NULL, ENV_HIDE);
    return(0);
}


static int
Env2Buf(const char *domain, struct env_s *esp, env_t typemask,
	char *buf, int buflen);

int
EnvSet(int argc, char *argv[])
{
    struct envlist_s *elp;
    struct env_s *esp=NULL;
    int data;
    bool_t update = TRUE;

    if (argc < 3) return(EnvPrint(argc, argv));
    esp = NameToEnv(argv[1], &elp);
    if (!elp || !esp) {
	ConsoleMsg(MS_E_ENV_UNKNOWN, argv[1]);
	return(-1);
    }
    if (elp->setup) elp->setup(TRUE);
    elp->f_setup = TRUE;
    if (EnvAccess(esp, W_OK)) {
    	ConsoleMsg(MS_E_ENV_PERM, argv[1]);
    	return(-1);
    }
    if (esp->type & ENV_RDONLY) {
	ConsoleMsg(MS_E_ENV_RDONLY, argv[1]);
	return(-1);
    }
    if ((esp->type & ENV_SECURE) && geteuid()) {
	ConsoleMsg(MS_E_ENV_PERM, argv[1]);
	return(-1);
    }
    switch (esp->type&ENV_MASK) {
    case ENV_INT8:
	sscanf(argv[2], "%d", &data);
	*(int8_t *)esp->d.entry = data;
	break;
    case ENV_INT16:
	sscanf(argv[2], "%d", &data);
	*(int16_t *)esp->d.entry = data;
	break;
    case ENV_INT32:
	sscanf(argv[2], "%d", &data);
	*(int32_t *)esp->d.entry = data;
	break;
    case ENV_INT8X:
	sscanf(argv[2], "%x", &data);
	*(int8_t *)esp->d.entry = data;
	break;
    case ENV_INT16X:
	sscanf(argv[2], "%x", &data);
	*(int16_t *)esp->d.entry = data;
	break;
    case ENV_INT32X:
	sscanf(argv[2], "%x", &data);
	*(int32_t *)esp->d.entry = data;
	break;
    case ENV_BOOL:
	if (!strcasecmp(argv[2], "yes"))
	    *(bool_t *)esp->d.entry = TRUE;
	else
	    *(bool_t *)esp->d.entry = FALSE;
	break;
    case ENV_STRING:
	if (*(char **)esp->d.entry) Free(*(char **)esp->d.entry);
	*(char **)esp->d.entry = Strdup(argv[2]);
	break;
    case ENV_SET:
	update = esp->d.set(argc - 1, &argv[1], NULL);
	break;
    }
    esp->user = geteuid();
    esp->group = getegid();

    if ( update == TRUE ) {
	char buf[XCBUFSIZ];
	int len;

	len = Env2Buf(elp->domain, esp, ENV_HIDE, buf, XCBUFSIZ);
	if (len > 0)
	    EnvUpdate(FALSE, buf, len);
    }
    return(0);
}

int
EnvUnset(int argc, char *argv[])
{
    struct envlist_s *elp;
    struct env_s *esp=NULL;

    if (argc < 2) return(-1);
    esp = NameToEnv(argv[1], &elp);
    if (EnvAccess(esp, W_OK)) {
    	ConsoleMsg(MS_E_ENV_PERM, argv[1]);
    	return(-1);
    }
    if (esp->type & ENV_RDONLY) return(-1);
    switch (esp->type&ENV_MASK) {
    case ENV_INT8:
	*(int8_t *)esp->d.entry = 0;
	break;
    case ENV_INT16:
	*(int16_t *)esp->d.entry = 0;
	break;
    case ENV_INT32:
	*(int32_t *)esp->d.entry = 0;
	break;
    case ENV_INT8X:
	*(int8_t *)esp->d.entry = 0;
	break;
    case ENV_INT16X:
	*(int16_t *)esp->d.entry = 0;
	break;
    case ENV_INT32X:
	*(int32_t *)esp->d.entry = 0;
	break;
    case ENV_BOOL:
	*(bool_t *)esp->d.entry = FALSE;
	break;
    case ENV_STRING:
	if (*(char **)esp->d.entry) Free(*(char **)esp->d.entry);
	*(char **)esp->d.entry = NULL;
	break;
    case ENV_SET:
	break;
    }

    esp->user = geteuid();
    esp->group = getegid();
    return(0);
}

/*
 * New Scheme
 */

static char *
Env2Str(struct env_s *esp, env_t typemask)
{
    static char strenv[XCBUFSIZ];

    if (typemask & ENV_HIDE) {
	if ((esp->type & ENV_SECRET) && !ISLOG(LOG_SECRET))
	    return(NULL);
	if ((esp->type & ENV_PRIVATE) && !ISLOG(LOG_PRIVATE))
	    return(NULL);
    }
    if (typemask & ENV_RDONLY & esp->type) return(NULL);
    if (EnvAccess(esp, R_OK)) {
    	sprintf(strenv, "<hide>");
    	return (strenv);
    }
    strenv[0] = '\0';
    switch (esp->type&ENV_MASK) {
    case ENV_INT8:
	sprintf(strenv, "%d", *(int8_t *)esp->d.entry);
	break;
    case ENV_INT16:
	sprintf(strenv, "%d", *(int16_t *)esp->d.entry);
	break;
    case ENV_INT32:
	sprintf(strenv, "%d", *(int32_t *)esp->d.entry);
	break;
    case ENV_INT8X:
	sprintf(strenv, "%#0x", *(int8_t *)esp->d.entry);
	break;
    case ENV_INT16X:
	sprintf(strenv, "%#0x", *(int16_t *)esp->d.entry);
	break;
    case ENV_INT32X:
	sprintf(strenv, "%#0x", *(int32_t *)esp->d.entry);
	break;
    case ENV_BOOL:
	sprintf(strenv, (*(bool_t *)esp->d.entry == TRUE) ? "yes": "no");
	break;
    case ENV_STRING:
	if (*(char **)esp->d.entry)
	    sprintf(strenv, "%s", *(char **)esp->d.entry);
	break;
    case ENV_SET:
	esp->d.set(0, &esp->name, strenv);
	break;
    }
    return(strenv[0] ? strenv: NULL);
}

static int
Env2Buf(const char *domain, struct env_s *esp, env_t typemask,
	char *buf, int buflen)
{
    char *str;
    int len = 0;
    struct passwd *pw;
    struct group *gr;

    strncpy(buf + len, esp->name, buflen - len);
    len = strlen(esp->name) + 1;
    if (len >= buflen) return(0);

    str = Env2Str(esp, typemask);
    if(str) {
	strncpy(buf + len, str, buflen - len);
	len += strlen(str) + 1;
    } else {
	buf[len] = '\0';
	len ++;
    }
    if (len >= buflen) return(0);

    if (domain && *domain) {
	strncpy(buf + len, domain, buflen - len);
	len += strlen(domain) + 1;
    } else {
	buf[len] = '\0';
	len ++;
    }
    if (len >= buflen) return(0);

    pw = getpwuid(esp->user);
    if (pw) {
	strncpy(buf + len, pw->pw_name, buflen - len);
	len += strlen(pw->pw_name) + 1;
    } else {
	sprintf(buf + len, "%d", esp->user); /* shold be snprintf() */
	len += strlen(buf + len) + 1;
    }
    if (len >= buflen) return(0);

    gr = getgrgid(esp->group);
    if (gr) {
	strncpy(buf + len, gr->gr_name, buflen - len);
	len += strlen(gr->gr_name) + 1;
    } else {
	sprintf(buf + len, "%d", esp->group); /* shold be snprintf() */
	len += strlen(buf + len) + 1;
    }
    if (len >= buflen) return(0);

    sprintf(buf + len, "%05o", esp->mode); /* should be snprintf() */
    len += strlen(buf + len) + 1;
    if (len >= buflen) return(0);

    return(len);
}

static int
EnvOut(struct console_s *cp, const char *domain,
       struct env_s *esp, env_t typemask)
{
    if (cp->fd >= 0) {
	struct xcio_s xc;

	xc.type = XCIO_ENV_SET;
	xc.len = Env2Buf(domain, esp, typemask, xc.buf, XCBUFSIZ);
	if (xc.len == 0) return(-1);
	xc.xid = cp->xid;

	return(XcioWrite(cp->fd, &xc));
    }
/*
    if (domain)
	fprintf(cp->cs_fp, "set %s.%s %s", domain, esp->name, str);
    else
	fprintf(cp->cs_fp, "set %s %s", esp->name, str);
*/
    return(0);
}

static void
EnvsOut(struct console_s *cp, struct envlist_s *elp, env_t typemask)
{
    int i;
    struct env_s *esp=elp->envs;

    for (i = 0; esp[i].name; i ++)
	EnvOut(cp, elp->domain, &esp[i], typemask);
}

void
AllEnvsOut(struct console_s *cp, env_t typemask)
{
    struct envlist_s *elp;
    elp = elHead;

    while (elp) {
	if (elp->f_setup == TRUE) EnvsOut(cp, elp, typemask);
	elp = elp->next;
    }
}

int
EnvsGet(struct console_s *cp, int argc, char **argv)
{
    struct envlist_s *elp;
    struct env_s *esp;
    struct xcio_s xc;

    elp = elHead;
    if (argc > 0 && strcmp(argv[0], "all")) {
	int a;

	if (!strcmp(argv[0], ".")) {
	    EnvsOut(cp, elHead, 0);
	}
	for (a = 0; a < argc; a ++) {
	    if ((esp = NameToEnv(argv[a], &elp)) == NULL) {
		elp = elHead->next;
		while (elp) {
		    if (!strcasecmp(elp->domain, argv[a])) {
			EnvsOut(cp, elp, 0);
			break;
		    }
			    elp = elp->next;
		}
	    } else EnvOut(cp, elp->domain, esp, 0);
	}
    } else AllEnvsOut(cp, ENV_HIDE);
    xc.type = XCIO_LAST;
    xc.len = 0;
    xc.xid = cp->xid;
    return(XcioWrite(cp->fd, &xc));
}

char *
ExtractEnvs(char *src, env_t typemask)
{
    static char extbuf[BUFSIZ];
    char *p, *dst, *cp, *org, *env;

    org = cp = Strdup(src);
    dst = extbuf;
    while (*cp && ((unsigned)(dst - extbuf) < sizeof(extbuf))) {
	if (*cp == '$' && *(cp + 1) == '(') {
	    cp += 2;
	    if ((p = strchr(cp, ')')) == NULL) {
		Logf(LOG_ERROR, "%s: missing ')'\n", src);
		strcpy(extbuf, src);
		dst = NULL;
		break;
	    }
	    *p = '\0';
	    if ((env = EnvToStr(cp, typemask)) != NULL) {
		strcpy(dst, env);
		dst += strlen(env);
	    }
	    cp = p + 1;
	} else *dst ++ = *cp ++;
    }
    if (dst) *dst = '\0';
    Free(org);
    return(extbuf);
}

void
ListupEnv(struct console_s *cp)
{
    struct envlist_s *elp;
    struct env_s *esp;
    struct xcio_s xc;
    u_char *p;
    int i, n=0;

    elp = elHead;
    xc.xid = cp->xid;
    xc.type = XCIO_LISTUP;
    while (elp) {
	esp = elp->envs;
	xc.buf[0] = (u_char) n ++;
	for (i = 0; esp[i].name; i ++) {
	    xc.buf[1] = (u_char) i;
	    /* xc.buf[2] = help; */
	    p = &xc.buf[3];
	    if (elp->domain && *elp->domain)
		p += SprintF(p, "%s.", elp->domain);
	    p += SprintF(p, "%s", esp[i].name);
	    *p = '\0';
	    xc.len = p - xc.buf + 1;
	    XcioWrite(cp->fd, &xc);
	}
	elp = elp->next;
    }
}

static int
EnvParseMode(char *s)
{
    int m;

    if (s == 0 || *s == 0) 
	return (-1);

    for (m = 0; *s >= '0' && *s <= '7'; s++) {
    	m = m * 8 + *s - '0';
    	if (m > 07777) return(-1);
    }
    if (*s) return (-1);

    return (m);
}

int
EnvVmod(int argc, char *argv[])
{
    int mode;
    uid_t uid;
    gid_t gid;
    struct envlist_s *elp;
    struct env_s *esp=NULL;

    if (argc <3) return (-1);

    mode = EnvParseMode(argv[1]);
    if (mode <0) {
    	return (-1);
    }

    esp = NameToEnv(argv[2], &elp);
    if (!elp || !esp) {
	ConsoleMsg(MS_E_ENV_UNKNOWN, argv[2]);
	return(-1);
    }
    if ((esp->type & ENV_RDONLY) && (mode & 0222)) {
    	ConsoleMsg(MS_E_ENV_RDONLY, argv[2]);
    	return(-1);
    }
    uid = geteuid();
    gid = getegid();
    if (uid != 0 && esp->user != uid && EnvAccess(esp, W_OK)) {
	ConsoleMsg(MS_E_ENV_PERM, argv[2]);
    	return(-1);
    }

    esp->user  = uid;
    esp->group = gid;
    esp->mode  = mode;

    return (0);
}
