/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include "vector_commands.h"
#include "common_gtk.h"
#include "modify_file_set.h"
#include "filepicker.h"
#include "command.h"
#include "linebuffer.h"
#include "optimize_usage.h"
#include "contractions.h"
#include "locks.h"
#include "mainwindow.h"
#include "globals.h"

/* static gint sel_row_path = -1; */
static long int lastsize = 0;
static char *dir_base = NULL;

static void update_totalsize_label() {
	char *sizeinfolabel;
	char tmp_buf[SIZE_BUF_SIZE];

	if (sectorsize < 0)
		sectorsize = 150;
	if (totalsize < 0)
		totalsize = 0;

	sizeinfolabel = string_append(_("Sectors: "), NULL);
	if (g_snprintf(tmp_buf, SIZE_BUF_SIZE, "%6.0f", (double)sectorsize) == -1)
		g_warning("%s %i: sector size too large? Sector-result is probably wrong",
				__FILE__, __LINE__);
	sizeinfolabel = string_append(sizeinfolabel, tmp_buf);
	sizeinfolabel = string_append(sizeinfolabel, _("\nSize: "));
	if (g_snprintf(tmp_buf, SIZE_BUF_SIZE, "%5.1f", (double)totalsize/(1024*1024)) == -1)
		g_warning("%s %i: file size too large? Total size-result is probably wrong",
				__FILE__, __LINE__);
	sizeinfolabel = string_append(sizeinfolabel, tmp_buf);
	sizeinfolabel = string_append(sizeinfolabel, " MB");

	gtk_label_set(GTK_LABEL(total_size), sizeinfolabel);
	free(sizeinfolabel);
/*
TODO: figure out some smart way to show this:
	if (opt_cdsize >= sectorsize)
		g_message("Ok!");
	else
		g_message("Won't fit!");
*/
}

void execute_recalc_size(cmd_v *command) {
	int done = 0;
	FILE *output;
	char in_buf[BUF_S];
	char *start;

/*	print_cmd(command); */

	if (NULL == (output = popen_r_stderr(command))) {
		g_warning("%s %i: error executing", __FILE__, __LINE__);
		return;
	}

	while (!(done)) {
		if (fgets(in_buf, BUF_S, output) != NULL) {
			if (feof(output)) {
				g_warning("%s %i: EOF reached reading mkisofs size output",
						__FILE__, __LINE__);
				pclose(output);
				return;
			} else if (ferror(output)) {
				g_warning("%s %i: error reading mkisofs size output!",
						__FILE__, __LINE__);
				pclose(output);
				return;
			} else if ((start = strstr(in_buf,
						"Total extents scheduled to be written"))) {
				int add = 40;
				if (strstr(start, "(inc HFS)"))
					add = 50;
				lastsize = atol(start + add);
				sectorsize += lastsize;
				totalsize = 2048 * sectorsize;
				done = 1;
			} else if ((strstr(in_buf, "Usage: mkisofs "))
					|| (strstr(in_buf, "Usage: mkhybrid "))) {
				g_warning("%s %i: %s didn't understand the arguments, maybe "
						"you are using too old mkisofs/mkhybrid version?!",
						__FILE__, __LINE__,
						command->elements[0]);
				pclose(output);
				return;
			} else if (strstr(in_buf, "Invalid node -")) {
				g_warning("%s %i: %s", __FILE__, __LINE__, in_buf);
				pclose(output);
				return;
			} else if (strstr(in_buf, "volume size must be >= 800K")) {
				alert_user_of_error(_(" HFS volume size must be >= 800K "
							"- file/dir size too small the size \n"
							" estimate might be a little bit wrong, "
							"add more files before burning "));
				pclose(output);
				return;
			}
		} else {
			g_warning("%s %i: There were problems calculating the size; "
					"perhaps your mkisofs/mkhybrid is too old for gcombust?",
					__FILE__, __LINE__);
			lastsize = 0;
			done = 1;
		}
	}

	pclose(output);
}

/* TODO: the size for each entry should be recalculated to reflect changes
 * 	 on follow symlink flags etc. (at least if auto bin packing is
 *	 implemented someday)
 */
void recalc_size(GtkWidget *widget, gpointer data) {
	cmd_v *mkisofs_size_cmd;
	config_cdr_data *cdr_info = (config_cdr_data *) data;

	if ((mkisofs_size_cmd = make_mkisofs_command(TRUE, NULL, cdr_info)) == NULL)
		return;

	sectorsize = 150; /* for 74m CDs, see comment elsewhere */

	if (!(is_running())) {
		execute_recalc_size(mkisofs_size_cmd);
		not_running();
	}

	destroy_cmd(mkisofs_size_cmd);

	update_totalsize_label();
	make_estimate_inaccurate();
	gtk_widget_set_sensitive(total_size, 1);
}

/* Checks if a path is in the file-list
 * returns TRUE if it's already there, FALSE if it's not */
static int path_is_in_iso(char *path) {
	int i;
	file_data *info;

	for (i = 0; i < GTK_CLIST(clist)->rows; i ++) {
		info = gtk_clist_get_row_data(GTK_CLIST(clist), i);
		g_assert(NULL != info);
		if (strlen(path) == strlen(info->path) && (!(strncmp(path, info->path, strlen(path)))))
			return TRUE;
	}
	return FALSE;
}

static void add_path_to_iso_files(gpointer data, gpointer user_data) {
	cmd_v *exec_string;
	char tmp_buf[SIZE_BUF_SIZE];
	char *text[2];
	int newrow;
	file_data *info;
	char *path;

	if (NULL == user_data)
		path = g_strdup(data);
	else {
		char *tmp;
		gtk_clist_get_text(user_data, GPOINTER_TO_INT(data), 0, &tmp);
		path = string_append(dir_base, NULL);
		path = string_append(path, tmp);
	}

	info = malloc(sizeof(file_data));

	if (path_is_in_iso(path)) {
		alert_user_of_error(_("You tried to add a duplicate file/dir to the file-list!"));
		free(path);
		return;
	}

	if ((exec_string = make_mkisofs_command(TRUE, path, mainptr->cdr_option_info)) == NULL) {
		free(path);
		return;
	}
	if (!(is_running())) {
		execute_recalc_size(exec_string);
		not_running();
	}
	destroy_cmd(exec_string);

	if (g_snprintf(tmp_buf, SIZE_BUF_SIZE, "%.1f",
			(double)lastsize*2048/(1024*1024)) == -1)
		g_warning("file size too large? Size-result is probably wrong");

	if (path[strlen(path)-1] == '/') {
		path[strlen(path)-1] = 0;
		text[0] = my_basename(path);
		path[strlen(path)] = '/';
	} else
		text[0] = my_basename(path);
	text[1] = tmp_buf;

	newrow = gtk_clist_append(GTK_CLIST(clist), text);
	info->size = lastsize;
	info->path = g_strdup(path);
	gtk_clist_set_row_data(GTK_CLIST(clist), newrow, info); 
	free(path);
}

/* Add file to the (iso) file-list
 * if widget != 0, add the filepath contained in the file_selection data to
 *	the isofile clist
 * if widget == 0, add the path data to the isofile clist
 * TODO: go trough the clist and check if adding a duplicate entry */
void selected_file(GtkWidget *widget, gpointer data) {
	if (widget != 0) {
		char *tmp;
		GList *list;

		dir_base = g_strdup(gtk_file_selection_get_filename(data));
		if (NULL == (tmp = strrchr(dir_base, '/'))) {
			g_warning("%s %i: no / in path, this is a bug", __FILE__, __LINE__);
			return;
		}
		tmp++;
		*tmp = '\0';

		list = (GTK_CLIST(GTK_FILE_SELECTION(data)->file_list))->selection;
		if (NULL != list) {
			g_list_foreach(list, add_path_to_iso_files,
					(GTK_CLIST(GTK_FILE_SELECTION(data)->file_list)));
		} else {
			tmp--;
			*tmp = '\0';
			if (NULL == (tmp = strrchr(dir_base, '/'))) {
				/* FIXME: adding /asdasd causes this to occur*/
				g_warning("%s %i: no / in path, this is probably a bug "
						"(did you maybe add a non existant root dir?)",
						__FILE__, __LINE__);
				return;
			}
			tmp++;
			*tmp = '\0';
		}

		list = (GTK_CLIST(GTK_FILE_SELECTION(data)->dir_list))->selection;
		if (NULL != list)
			g_list_foreach(list, add_path_to_iso_files,
					(GTK_CLIST(GTK_FILE_SELECTION(data)->dir_list)));

		free(dir_base);
		dir_base = NULL;
		gtk_clist_unselect_all(GTK_CLIST(GTK_FILE_SELECTION(data)->file_list));
		gtk_clist_unselect_all(GTK_CLIST(GTK_FILE_SELECTION(data)->dir_list));
	} else {
		add_path_to_iso_files(data, NULL);
	}

	update_totalsize_label();
	make_estimate_inaccurate();

	resize_clist_col_ratio(NULL, NULL, clist);

	mark_size_inaccurate();
}

void del_selected_rows(GtkWidget *widget, gpointer data) {
	file_data *info;
	int i;

	for (i = MAX_ROWS-1; i >= 0; i--) {
		if (1 == selected_files[i]) {
			info = gtk_clist_get_row_data(GTK_CLIST(clist), i);
			if (info == NULL) {
				g_warning("modify_file_set.c::del_selected_rows: "
						"trying to access non-existant clist row!");
				break;
			}
			totalsize -= info->size*2048;
			sectorsize -= info->size;
			update_totalsize_label();
			make_estimate_inaccurate();
			mark_size_inaccurate();
			gtk_clist_remove(GTK_CLIST(clist), i);
			free(info->path);
			free(info);
		}
	}
}

/*
void selected_dir(GtkWidget *widget, gpointer data) {
	gchar *text[1];

	text[0] = g_strdup(gtk_file_selection_get_filename(data));

	gtk_clist_append(GTK_CLIST(exclude_dir), text);
	totalrows_path++;
	mark_size_inaccurate();
}

void remove_dir(GtkWidget *widget, gpointer data) {
	if (sel_row_path != -1) {
		gtk_clist_remove(GTK_CLIST(exclude_dir), sel_row_path);
		totalrows_path--;
		mark_size_inaccurate();
	}
}
*/
