/* tracktype for files dropped onto the track manager plus additional 
 * handling requirements for this tracktype (e.g. suffix2tracktype database) */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

void swab(const void *from,void *to,size_t n);

#include "tracks.h"
#include "stdfiletrack.h"
#include "varman.h"
#include "main.h"
#include "preferences.h"
#include "filetypes.h"
#include "piping.h"
#include "helpings.h"
#include "calc.h"

/* uncomment if you want to debug stdfiletrack.c */
/* #define DEBUG */

#ifdef DEBUG
# include "colors.h"
# include "rterm.h"
#endif

/* contains program output after detectsize calls */
char outbuf[16384];

/* this function is run in background and implements the byteswapping used
 * to convert little endian streams to big endian streams and back */

/* cstdin and cstdout are the pipes used to communicate with the main process */
void stdfiletrack_byteswap(int cstdin,int cstdout,int cstderr,gpointer data)
{
   char buf[BUFSIZE];
   char conv[BUFSIZE];
   int readcount;
   int writecount;

#ifdef DEBUG
   dprintf(cstderr,"stdfiletrack: started byteswap\n");
#endif
   do 
     {
	readcount=read(cstdin,&buf,BUFSIZE);
	if (readcount>0) 
	  {
	     /* swap high and lowbyte */
	     swab(&buf,&conv,readcount);
	     writecount=write(cstdout,&conv,readcount);
	  }
	;
     }
   while ((readcount>0)&&(writecount==readcount));
   close(cstdin);close(cstdout);
#ifdef DEBUG
   dprintf(cstderr,"stdfiletrack: byteswap closed IO channels...\n");
#endif   
   close(cstderr);
   
   _exit(0);
}
;

int stdfiletrack_openpipe(void*trackinfo)
{	
	int outp=-1;
	stdfiletrack_info *i;
	char *call;
	int *errpp;
	
	i=(stdfiletrack_info*)((tracks_trackinfo*)trackinfo)->data;

	if (i->filedes==-1)
	  {
#ifdef DEBUG
		  printf ("stdfiletrack.c: reading file \"%s\"\n",(char*)&i->filename);
#endif		  
		  i->filedes=open((char*)&i->filename,O_RDONLY);
		  if (i->filedes==-1)
		    {			    
			    perror ("error opening source file");
		    }
		  else
		    {
			    if (strlen(filetypes_getfiltername(i->suffix))>0)
			      {
				      /* make local copy of piping call */
				      call=varman_replacevars_copy(global_defs,
								   filetypes_getfiltername(i->suffix));
				      /* this is for "filters" that can't take their input from stdin */
				      varman_replacestring(call,"$file",(char*)&i->filename);
				 				 	
#ifdef DEBUG
				      printf ("calling converter: %s\n",call);
#endif
				      i->convpid=piping_create(call,
							       &i->filedes,
							       &outp,
							       NULL);
				      i->filedes=outp;
				      free(call);
			      }
			    else
				i->convpid=-1;
			    /* do a little->big endian conversion if requested */
			    if (!strcmp(filetypes_getfilter(i->suffix),
					"little Endian"))
			      {
#ifdef DEBUG
				      errpp=&i->errp;i->errp=-1;
#else				      
				      errpp=NULL;
#endif
				 	
				      outp=-1; /* create yet another outp descriptor */
				      i->filterpid=piping_create_function(stdfiletrack_byteswap,
									  NULL,
									  &i->filedes,
									  &outp,
									  errpp);
#ifdef DEBUG
				      i->debugfilter=rterm_connectpipe(&Red,i->errp);
#endif
				      i->filedes=outp;
			      }
			    else
				i->filterpid=-1;
		    }
		  ;
	  }
	;
#ifdef DEBUG
	printf ("stdfiletrack_openpipe: returning track pipe %i\n",
		i->filedes);
#endif
	return i->filedes;
}
;

void stdfiletrack_closepipe(void*trackinfo)
{
	stdfiletrack_info *i;
	int status;
	
	i=(stdfiletrack_info*)((tracks_trackinfo*)trackinfo)->data;
	if (i->filedes!=-1)
	  {
#ifdef DEBUG
	     printf ("stdfiletrack_closepipe: closing filedescriptor\n");
#endif
	     close(i->filedes);
	     i->filedes=-1;
	     
	     if (i->filterpid!=-1)
	       {
#ifdef DEBUG		  
		  rterm_print ("waiting for filter ...\n");
#endif
		  waitpid(i->filterpid,&status,0);
	       }		  
	     ;
	     if (i->convpid!=-1)
	       {
		  /* send SIGTERM to converter */
		  kill(i->convpid,SIGTERM);
#ifdef DEBUG
		  rterm_print ("waiting for converter ...\n");
#endif
		  waitpid(i->convpid,&status,0);
	       }
	     ;
#ifdef DEBUG	       
	     /* disactivate debugging pipe if in debugging mode */
	     rterm_disconnectpipe(i->debugfilter);
	     close(i->errp);
#endif
	  }
	;
	
}
;

int stdfiletrack_tracksize(void*trackinfo)
{
	stdfiletrack_info *i;
	
	i=(stdfiletrack_info*)((tracks_trackinfo*)trackinfo)->data;
	
	return i->tracksize;	
}
;

int stdfiletrack_dataavail(void*trackinfo)
{
	stdfiletrack_info *i;

	i=(stdfiletrack_info*)((tracks_trackinfo*)trackinfo)->data;

	return (i->filedes!=-1);
}
;	

int  stdfiletrack_valid(void*trackinfo)  /* isotrack is always valid */
{
	return 1;
}
;

void stdfiletrack_destroy(void*trackinfo)
{
	stdfiletrack_info *i;
	
	i=(stdfiletrack_info*)((tracks_trackinfo*)trackinfo)->data;
	
	if (stdfiletrack_dataavail(trackinfo))
	    close (i->filedes);
	free(i);
}
;

tracks_trackinfo *stdfiletrack_create(char *filename)
{
	tracks_trackinfo *dt;
	stdfiletrack_info *info;
	char *tracktype;
   	tracks_precache cachemode;
	
	int fd;
	int pos=0;
	
	char description[256];
	
	info=(stdfiletrack_info*)malloc(sizeof(stdfiletrack_info));
	strcpy((char*)&info->filename,filename);
	info->filedes=-1;
	info->suffix=strrchr((char*)&info->filename,"."[0]); /* get suffix */
	if (info->suffix==NULL) /* if suffix not present set it to "" */
	    info->suffix="";
	tracktype=filetypes_gettracktype(info->suffix);
	if (strlen(tracktype)==0)
	    tracktype="data"; /* default to a data track if no tracktype is given. */
	/* it is *intended* to overwrite the pointer to the string here as the
	 * string returned by filetypes_gettracktype() is not a copy but the real one */
	
	strcpy((char*)&description,tracktype);
	strcat((char*)&description," file ");
	strcat((char*)&description,filename);
	
	if (strstr(filetypes_getsizealg(info->suffix),
		   "$filesize")!=NULL)
	  {
		  pos=0;
		  fd=open((char*)&info->filename,O_RDONLY);
		  if (fd!=-1)
		    {
			    pos=lseek(fd,0,SEEK_END);
			    close(fd);
		    }
		  ;
	  }
	;	
	
	/* is client output required ? */
	if (strstr(filetypes_getsizealg(info->suffix),
		   "getpos")!=NULL)
	  {		  
		  piping_create_getoutput(varman_replacestring_copy(filetypes_getdetectsize(info->suffix),
								    "$file",
								    (char*)&info->filename),
					  (char*)&outbuf,16384,PIPING_WATCHALL
					  );
	  }
	;
	
	info->tracksize=calc_function(filetypes_getsizealg(info->suffix),
				      1,
				      "$filesize",
				      (float)pos);		      
	
	
   	/* precache track either "atadd" or not at all */
   	if (filetypes_getprecache(info->suffix))
     		cachemode=atadd;
        else
     		cachemode=none;
   
	dt=tracks_create((char*)&description,
			 tracktype,cachemode,
			 stdfiletrack_openpipe,
			 stdfiletrack_closepipe,
			 stdfiletrack_tracksize,
			 stdfiletrack_dataavail,
			 stdfiletrack_valid,
			 stdfiletrack_destroy,
			 info);
	
	return dt;
}
;

void stdfiletrack_getpos_handler(char *arg1,char *arg2,char *output)
{
	char *xpos,*ypos;
	int xposv,yposv;

#ifdef DEBUG
	printf ("stdfiletrack_getpos_handler: arg2 \"%s\"\n",arg2);
#endif
	ypos=arg2;xpos=strchr(arg2,","[0]);
	if (xpos==NULL) 
	  {
		  printf ("stdfiletrack_getpos_handler: Illegal argument.\n");
		  xpos="0";
	  }
	else 
	    xpos++;
	sscanf(ypos,"%d",&yposv);
	sscanf(xpos,"%d",&xposv);
#ifdef DEBUG
	printf ("stdfiletrack_getpos_handler: get number (%d,%d)\n",
		yposv,xposv);
#endif
	
	sprintf(output,"%f",
		helpings_getvalueinstring(helpings_getlineno((char*)&outbuf,
							     yposv),
					  xposv));
#ifdef DEBUG
	printf ("stdfiletrack_getpos_handler: result \"%s\"\n",
		output);
#endif
	
}
;

void stdfiletrack_init()
{
	calc_expression_register_prepend(calc_expression_create("getpos",
								stdfiletrack_getpos_handler));
	
}
;
