#
#  $Id: DLP.py,v 1.14 1999/12/16 10:00:32 rob Exp $
#
#  Copyright 1999 Rob Tillotson <robt@debian.org>
#  All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee or royalty is
#  hereby granted, provided that the above copyright notice appear in
#  all copies and that both the copyright notice and this permission
#  notice appear in supporting documentation or portions thereof,
#  including modifications, that you you make.
#
#  THE AUTHOR ROB TILLOTSON DISCLAIMS ALL WARRANTIES WITH REGARD TO
#  THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
#  AND FITNESS.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
#  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
#  RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
#  CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
#  CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE!
#
"""Databases on an attached Palm Computing(R) platform device.
"""

__version__ = '$Id: DLP.py,v 1.14 1999/12/16 10:00:32 rob Exp $'

__copyright__ = 'Copyright 1999 Rob Tillotson <robt@debian.org>'

import time

import Pyrite.Store
import Pyrite
from Pyrite import Database, PrefBlock, _

from Pyrite import pisock

class DBWrapper:
    def __init__(self, socket, db):
	self.socket = socket
	self.db = db

    def getRecords(self):
	return pisock.dlp_ReadOpenDBInfo(self.socket, self.db)

    def getResource(self, i):
	return pisock.dlp_ReadResourceByIndex(self.socket, self.db, i)

    def getRecord(self, i):
	ret = pisock.dlp_ReadRecordByIndex(self.socket, self.db, i)
	if ret is None: return None
	raw, id, attr, cat = ret
	return (raw, i, id, attr, cat)

    def setResource(self, type, id, raw):
	return pisock.dlp_WriteResource(self.socket, self.db, type, id, raw)

    def setRecord(self, attr, id, cat, raw):
	return pisock.dlp_WriteRecord(self.socket, self.db, attr, id, cat, raw)

    def getResourceByID(self, type, id):
	ret = pisock.dlp_ReadResourceByType(self.socket, self.db, type, id)
	if ret is None: return None
	raw, index = ret
	return (raw, type, id)

    def getRecordByID(self, id):
	ret = pisock.dlp_ReadRecordById(self.socket, self.db, id)
	if ret is None: return None
	raw, index, attr, cat = ret
	return (raw, index, id, attr, cat)

    def deleteResource(self, type, id):
	return pisock.dlp_DeleteResource(self.socket, self.db, 0, type, id)
    
    def deleteRecord(self, id):
	return pisock.dlp_DeleteRecord(self.socket, self.db, 0, id)

    def deleteResources(self):
	return pisock.dlp_DeleteResource(self.socket, self.db, 1, None, 0)

    def deleteRecords(self):
	return pisock.dlp_DeleteRecord(self.socket, self.db, 1, 0)

    def getAppBlock(self):
	return pisock.dlp_ReadAppBlock(self.socket, self.db, 0)

    def setAppBlock(self, raw):
	return pisock.dlp_WriteAppBlock(self.socket, self.db, raw)

    def getSortBlock(self):
	return pisock.dlp_ReadSortBlock(self.socket, self.db, 0)

    def setSortBlock(self, raw):
	return pisock.dlp_WriteSortBlock(self.socket, self.db, raw)

    def moveCategory(self, frm, to):
	return pisock.dlp_MoveCategory(self.socket, self.db, frm, to)

    def deleteCategory(self, cat):
	return pisock.dlp_DeleteCategory(self.socket, self.db, cat)

    # getPref and setPref do not belong here, so they are not here.
    def resetFlags(self):
	return pisock.dlp_ResetSyncFlags(self.socket, self.db)

    def purge(self):
	return pisock.dlp_CleanUpDatabase(self.socket, self.db)

    def resetNext(self):
	return pisock.dlp_ResetDBIndex(self.socket, self.db)

    def getNextRecord(self, cat):
	ret = pisock.dlp_ReadNextRecInCategory(self.socket,
					       self.db,
					       cat)
	if ret is None: return None
	buf, id, index, attr = ret
	return (buf, index, id, attr, cat)

    def getNextModRecord(self, cat):
	ret = pisock.dlp_ReadNextModifiedRecInCategory(self.socket,
						       self.db,
						       cat)
	if ret is None: return None
	buf, id, index, attr = ret
	return (buf, index, id, attr, cat)

    def getRecordIDs(self, sort):
	l = []
	st = 0
	while 1:
	    ret = pisock.dlp_ReadRecordIDList(self.socket, self.db, sort, st, 16383)
	    l = l + ret
	    if not ret or len(ret) < 16383: #less than max
		return l
	    else:
		st = st + len(ret)

    def close(self):
	r = pisock.dlp_CloseDB(self.socket, self.db)
	self.db = None
	return r
    
    def __del__(self):
	if self.db is not None:
	    pisock.dlp_CloseDB(self.socket, self.db)
    
	
class DLPStore(Pyrite.Store.BaseStore):
    properties = ('read','write','create','delete','list','install')
    db_properties = ('id-unique', 'id-replace', 'id-auto-assign')
    
    def __init__(self, port='/dev/pilot', cardno=0, preconnect_hook=None,
		 postconnect_hook = None):
	Pyrite.Store.BaseStore.__init__(self)

	self.port = port
	self.cardno = cardno
	self.socket = None
	self.preconnect_hook = preconnect_hook
	self.postconnect_hook = postconnect_hook
	self.user_info = None
	self.reset_on_close = 0
	
    def open(self, name, mode='rws', dbclass=Database, properties=(), **kw):
	if not self.test_db_properties(properties):
	    raise RuntimeError, _("cannot provide a db with properties %s") % properties
	m = 0
	if 'r' in mode: m = m | pisock.dlpOpenRead
	if 'w' in mode: m = m | pisock.dlpOpenWrite
	if 'x' in mode: m = m | pisock.dlpOpenExclusive
	if 's' in mode: m = m | pisock.dlpOpenSecret
	self.try_connect()
	i = pisock.dlp_FindDBInfo(self.socket, self.cardno, 0, name, None, None)
	db = pisock.dlp_OpenDB(self.socket, self.cardno, m, name)
	return dbclass(DBWrapper(self.socket, db), i, mode, self.db_properties)

    def create(self, name, creator, type, flags=0, version=1,
	       dbclass=Database, properties=(), **kw):
	if not self.test_db_properties(properties):
	    raise RuntimeError, _("cannot provide a db with properties %s") % properties
	self.try_connect()
	db = pisock.dlp_CreateDB(self.socket, creator, type, self.cardno, flags, version,
				 name)
	i = pisock.dlp_FindDBInfo(self.socket, self.cardno, 0, name, None, None)
	return dbclass(DBWrapper(self.socket, db), i, 'rws', self.db_properties)

    def delete(self, name):
	self.try_connect()
	return pisock.dlp_DeleteDB(self.socket, self.cardno, name)

    def info(self, name):
	self.try_connect()
	return pisock.dlp_FindDBInfo(self.socket, self.cardno, 0, name, None, None)

    def list(self):
	return map(lambda x: x['name'], self.listinfo())
    
    def listinfo(self, name=None, creator=None, type=None):
	l = []
	i = 0
	self.try_connect()
	while 1:
	    try:
		info = pisock.dlp_ReadDBList(self.socket, self.cardno, 0x80, i)
	    except:
		break
	    i = info['index']+1
	    if name is not None and info['name'] != name: continue
	    if creator is not None and info['creator'] != creator: continue
	    if type is not None and info['type'] != type: continue
	    l.append(info)
	return l

    # FIXME: pick one or the other, this or the implementation below,
    # and get rid of the other one.
    def getpref(self, creator, id, saved=1):
	self.try_connect()
	raw, creator, id, version, saved = self.dlp.getPref(creator, id, saved)
	return pisock.dlp_ReadAppPreference(self.socket, creator, id, saved)

    def setpref(self, raw, creator, id, version=0, saved=1):
	self.try_connect()
	return pisock.dlp_WriteAppPreference(self.socket, creator, id, saved,
					     version, raw)
    
    # --------------
    # DLP API
    # --------------
    # Note: this is a bit sparse right now, the rest will be added as
    # needed.
    def remote_version(self):
	self.try_connect()
	return pisock.pi_version(self.socket)

    def get_time(self):
	self.try_connect()
	return pisock.dlp_GetSysDateTime(self.socket)
    
    def set_time(self, t):
	self.try_connect()
	return pisock.dlp_SetSysDateTime(self.socket,t)

    def card_info(self, cardno=0):
	self.try_connect()
	return pisock.dlp_ReadStorageInfo(self.socket, cardno)

    def system_info(self):
	self.try_connect()
	return pisock.dlp_ReadSysInfo(self.socket)

    def get_feature(self, creator, number):
	self.try_connect()
	return pisock.dlp_ReadFeature(self.socket, creator, number)

    def sync_log(self, s):
	self.try_connect()
	return pisock.dlp_AddSyncLogEntry(self.socket, s)
    
    # Preferences
    def get_preference(self, creator, id, backup=1, prefclass=PrefBlock):
	self.try_connect()
	p = pisock.dlp_ReadAppPreference(self.socket, creator, id, backup)
	return prefclass(p[0], creator, id, p[1], backup)

    def set_preference(self, pref):
	self.try_connect()
	return pisock.dlp_WriteAppPreference(self.socket, pref.creator,
					     pref.id, pref.backup,
					     pref.version, pref.pack())

    # magic :)  don't ask me how to use this stuff.
    #
    # call an application on the palmtop.  ret_length is the maximum length
    # of return value you want back.
    #def call_application(self, creator, type, action, data, ret_length):
	#self.try_connect()
	#return self.dlp.callApp(creator, type, action, data, length, maxlength)

    #def call_rpc(self):
	#self.try_connect()
	#raise RuntimeError, "unimplemented"
    
    # What don't we have yet:
    #    dlp.dirty() - convenience to reset LastSyncPC to 0
    #    dlp.abort() - abort immediately without notifying pilot
    
    # --------------
    # Connection API
    # --------------
    def try_connect(self):
	if self.socket is None:
	    self.connect()
	    
    def connect(self):
	if callable(self.preconnect_hook): self.preconnect_hook(self)

	sd = pisock.pi_socket(pisock.PI_AF_SLP, pisock.PI_SOCK_STREAM, pisock.PI_PF_PADP)
	if not sd: raise RuntimeError, "failed to make socket"
	if (pisock.pi_bind(sd, (pisock.PI_AF_SLP, self.port))) == -1:
	    raise RuntimeError, "failed to bind socket"
	if (pisock.pi_listen(sd,1)) == -1:
	    raise RuntimeError, "failed to listen on socket"
	ret = pisock.pi_accept(sd)
	if ret == -1:
	    raise RuntimeError, "failed to accept connection"
	self.socket = ret[0]
	# set the message on the display
	pisock.dlp_OpenConduit(self.socket)
	self.user_info = pisock.dlp_ReadUserInfo(self.socket)
	if callable(self.postconnect_hook): self.postconnect_hook(self)

    def disconnect(self, successful=1, log=_("Sync completed.")):
	if self.socket is not None:
	    # dlp_CloseDB_All will throw an exception if there are no DBs
	    # for it to close, so we catch it and ignore it.
	    try:
		pisock.dlp_CloseDB_All(self.socket)
	    except pisock.error:
		pass
	    if self.reset_on_close:
		self.reset_on_close = 0
		pisock.dlp_ResetSystem(self.socket)
	    self.user_info['lastSyncPC'] = 0x00010000
	    self.user_info['lastSyncDate'] = time.time()
	    if successful:
		self.user_info['successfulSyncDate'] = self.user_info['lastSyncDate']
	    pisock.dlp_WriteUserInfo(self.socket, self.user_info)
	    pisock.dlp_AddSyncLogEntry(self.socket, log)
	    pisock.pi_close(self.socket)
	    self.socket = None
	    self.user_info = None

    def connected(self):
	return self.socket is not None

    def install(self, store, name):
	"""Install a database from another store to a connected handheld.

	Note that this is slightly different than the generic 'copy' method,
	because the installation process does a few extra things based on
	the contents of the database.  Theoretically you can use 'copy' to
	install databases on the handheld, but you will not get the proper
	behavior for certain databases.
	"""
	try:
	    self.delete(name)
	except:
	    pass

	idb = store.open(name, 'rs')
	
	flags = 0
	if idb.info['flagReset']: flags = flags | 0x0020
	if idb.info['flagResource']: flags = flags | 0x0001
	if idb.info['flagNewer']: flags = flags | 0x0010
	if idb.info['flagExcludeFromSync']: flags = flags | 0x0080
	if idb.info['flagAppInfoDirty']: flags = flags | 0x0004
	if idb.info['flagReadOnly']: flags = flags | 0x0002
	if idb.info['flagBackup']: flags = flags | 0x0008
	if idb.info['flagOpen']: flags = flags | 0x8000

	if name == 'Graffiti Shortcuts ':
	    flags = flags | 0x0010
	    self.reset_on_close = 1

	odb = self.create(name, idb.info['creator'], idb.info['type'],
			  flags, idb.info['version'], info=idb.info)

	try:
	    try:
		a = idb.get_appblock()
		# pad memodb appblock to fix bug here.  that belongs
		# in the appmodule, i think.
		odb.set_appblock(a)
	    except:
		pass

	    if idb.info['creator'] == 'ptch' or idb.info['flagReset']:
		self.reset_on_close = 1

	    if idb.info['flagResource']:
		for i in range(0, len(idb)):
		    r = idb[i]
		    if r.type == 'boot':
			self.reset_on_close = 1
		    l = len(r)
		    if not l:
			continue #skip empties
		    if l > 65535:
			raise RuntimeError, _("resource #%s (%s/0x%x) > 64k bytes") % (i, r.type, r.id)
		    odb.append(r)
	    else:
		for i in range(0, len(idb)):
		    r = idb[i]
		    if r.deleted or r.archived:
			continue
		    if len(r) > 65535:
			raise RuntimeError, _("record #%s (%s/0x%x) > 64k bytes") % (i, r.id)
		    odb.append(r)

	    idb.close()
	    odb.close()
	except RuntimeError, a:
	    odb.close()
	    self.delete(name)
	    raise RuntimeError, a
	except:
	    odb.close()
	    self.delete(name)
	    raise RuntimeError, _("unknown installation error")
	    
    
class Store(Pyrite.Store.Store):
    name = 'DLP'
    version = Pyrite.version
    author = Pyrite.author
    description = _("Contents of an attached device.")
    properties = ['read','write','create','delete','list','install']
    
    store_class = DLPStore
    
