/* Copyright (c) 1997-1999 Miller Puckette.
* For information on usage and redistribution, and for a DISCLAIMER OF ALL
* WARRANTIES, see the file, "LICENSE.txt," in this distribution.  */

/* This file defines the "scalar" object, which is not a Text obgect, just a
"gobj".  Scalars have templates which describe their structures, which
can contain numbers, sublists, and arrays.
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>  	/* for read/write to files */
#include "m_pd.h"
#include "g_canvas.h"

t_class *scalar_class;

void word_restore(t_word *wp, t_datatype *datatypes, int nitems,
    int argc, t_atom *argv, t_gpointer *gp)
{
    int i;
    for (i = 0; i < nitems; i++, datatypes++, wp++)
    {
    	int type = datatypes->dt_type;
    	if (type == DT_FLOAT)
    	{
    	    float f;
    	    if (argc)
    	    {
    	    	f =  atom_getfloat(argv);
    	    	argv++, argc--;
    	    }
    	    else f = 0;
    	    wp->w_float = f; 
    	}
    	else if (type == DT_SYMBOL)
    	{
    	    t_symbol *s;
    	    if (argc)
    	    {
    	    	s =  atom_getsymbol(argv);
    	    	argv++, argc--;
    	    }
    	    else s = &s_;
    	    wp->w_symbol = s;
    	}
    	else if (type == DT_ARRAY)
    	{
    	    wp->w_array = array_new(datatypes->dt_arraytemplate, gp);
    	}
    	else if (type == DT_LIST)
    	{
    	    	/* LATER test this and get it to work */
    	    wp->w_list = canvas_new(0, 0, 0);
    	}
    }
    if (argc) post("warning: word_restore: extra arguments");
}

t_scalar *scalar_new(t_glist *owner,
    t_symbol *templatesym, t_int argc, t_atom *argv)
{
    t_canvas *c;
    t_scalar *x;
    int nitems;
    t_datatype *datatypes;
    t_gpointer gp;
    gpointer_init(&gp);
    c = (t_canvas *)pd_findbyclass(templatesym, canvas_class);
    if (!c)
    {
    	error("scalar: couldn't find template %s", templatesym->s_name);
    	return (0);
    }
    canvas_setusedastemplate(c);
    datatypes = canvas_getdatatypes(c, &nitems);
    x = (t_scalar *)getbytes(sizeof(t_scalar) +
    	(nitems - 1) * sizeof(*x->x_vec));
    x->x_gobj.g_pd = scalar_class;
    x->x_template = templatesym;
    gpointer_setglist(&gp, owner, x);
    word_restore(x->x_vec, datatypes, nitems, argc, argv, &gp);
    freebytes(datatypes, nitems * sizeof(*datatypes));
    return (x);
}

void glist_scalar(t_glist *owner,
    t_symbol *classname, t_int argc, t_atom *argv)
{
    t_symbol *templatesym = atom_getsymbolarg(0, argc, argv);
    t_scalar *x;
    if (argc) argc--, argv++;
    x = scalar_new(owner, templatesym, argc, argv);
    if (x) glist_add(owner, &x->x_gobj);
}

/* -------------------- widget behavior for scalar ------------ */
static void scalar_getbasexy(t_scalar *x, float *basex, float *basey)
{
    t_canvas *canvas = (t_canvas *)pd_findbyclass(x->x_template, canvas_class);
    *basex = canvas_getfloat(canvas, gensym("x"), x->x_vec, 0);
    *basey = canvas_getfloat(canvas, gensym("y"), x->x_vec, 0);
}

static void scalar_getrect(t_gobj *z, t_glist *owner,
    int *xp1, int *yp1, int *xp2, int *yp2)
{
    t_scalar *x = (t_scalar *)z;
    int hit = 0;
    t_glist *glist = (t_glist *)pd_findbyclass(x->x_template, canvas_class);
    int x1 = 0, x2 = 0, y1 = 0, y2 = 0;
    t_gobj *y;
    float basex, basey;
    scalar_getbasexy(x, &basex, &basey);
    if (!glist) bug("scalar_getrect");
    for (y = glist->gl_list; y; y = y->g_next)
    {
	t_parentwidgetbehavior *wb = pd_getparentwidget(&y->g_pd);
    	int nx1, ny1, nx2, ny2;
	if (!wb) continue;
	(*wb->w_parentgetrectfn)(y, owner,
	    x->x_vec, (t_canvas *)glist, basex, basey,
	    &nx1, &ny1, &nx2, &ny2);
	if (hit)
	{
	    if (nx1 < x1) x1 = nx1;
	    if (ny1 < y1) y1 = ny1;
	    if (nx2 > x2) x2 = nx2;
	    if (ny2 > y2) y2 = ny2;
	}
	else x1 = nx1, y1 = ny1, x2 = nx2, y2 = ny2, hit = 1;
    }
    if (!hit) x1 = y1 = x2 = y2 = 0;
	
    *xp1 = x1;
    *yp1 = y1;
    *xp2 = x2;
    *yp2 = y2; 
}

static void scalar_displace(t_gobj *z, t_glist *glist, int dx, int dy)
{
    /* later */
}

static void scalar_select(t_gobj *z, t_glist *owner, int state)
{
    t_scalar *x = (t_scalar *)z;
    post("scalar_select %d", state);
    /* later */
}

static void scalar_activate(t_gobj *z, t_glist *owner, int state)
{
    post("scalar_activate %d", state);
    /* later */
}

static void scalar_delete(t_gobj *z, t_glist *glist)
{
    /* nothing to do */
}

static void scalar_vis(t_gobj *z, t_glist *owner, int vis)
{
    t_scalar *x = (t_scalar *)z;
    t_glist *glist = (t_glist *)pd_findbyclass(x->x_template, canvas_class);
    t_gobj *y;
    float basex, basey;
    scalar_getbasexy(x, &basex, &basey);
    /* post("basex %f, basey %f", basex, basey); */
    if (!glist) bug ("scalar_vis");

    for (y = glist->gl_list; y; y = y->g_next)
    {
	t_parentwidgetbehavior *wb = pd_getparentwidget(&y->g_pd);
	if (!wb) continue;
	(*wb->w_parentvisfn)(y, owner,
	    x->x_vec, (t_canvas *)glist, basex, basey, vis);
    }
}

static void scalar_save(t_gobj *z, t_binbuf *b)
{
    /* ? */
}

static void scalar_free(t_scalar *x)
{
    t_canvas *c;
    int nitems, i;
    t_datatype *datatypes, *dt;
    t_symbol *templatesym = x->x_template;
    c = (t_canvas *)pd_findbyclass(templatesym, canvas_class);
    if (!c)
    {
    	error("scalar: couldn't find template %s", templatesym->s_name);
    	return;
    }
    datatypes = canvas_getdatatypes(c, &nitems);
    for (dt = datatypes, i = 0; i < nitems; i++, dt++)
    {
    	if (dt->dt_type == DT_ARRAY)
    	    array_free(x->x_vec[i].w_array);
    	else if (dt->dt_type == DT_LIST)
    	    canvas_free(x->x_vec[i].w_list);
    }
    freebytes(x, sizeof(t_scalar) + (nitems - 1) *  sizeof(t_atom));
    freebytes(datatypes, nitems * sizeof(*datatypes));
}

static t_widgetbehavior scalar_widgetbehavior =
{
    scalar_getrect,
    scalar_displace,
    scalar_select,
    scalar_activate,
    scalar_delete,
    scalar_vis,
    scalar_save
};

/* ----------------- setup function ------------------- */

void g_scalar_setup(void)
{
    scalar_class = class_new(gensym("scalar"), 0, (t_method)scalar_free, 0,
    	CLASS_GOBJ, 0);
    class_setwidget(scalar_class, &scalar_widgetbehavior);
}
