#include "snd.h"


/* -------------------------------- DATA STRUCTURES -------------------------------- 
 *
 * s_type used for run-time type checks (generic functions)
 * 
 * axis_info: axis data
 * file_info: header data describing sound data in file
 * fft_info: data relating to one fft display
 * chan_info: state info for one channel
 * snd_info: state info for one sound
 * snd_state: overall state of program
 */

static lisp_grf *free_lisp_info(chan_info *cp)
{
  lisp_grf *lg;
  if (cp)
    {
      lg = cp->lisp_info;
      if (lg)
	{
	  if (lg->axis) free_axis_info(lg->axis);
	  if (lg->data) FREE(lg->data);
	  FREE(lg);
	}
    }
  return(NULL);
}

chan_info *make_chan_info(chan_info *cip, int chan, snd_info *sound, snd_state *state)
{
  chan_info *cp; /* may be re-use */
  if ((!cip) || (((snd_any *)cip)->s_type != CHAN_INFO))
    {
      cp = (chan_info *)CALLOC(1,sizeof(chan_info)); 
      cp->s_type = CHAN_INFO;
      cp->cgx = (chan_context *)CALLOC(1,sizeof(chan_context));
      (cp->cgx)->ax = (axis_context *)CALLOC(1,sizeof(axis_context));
      cp->mixes = NULL;
      cp->last_sonogram = NULL;
    }
  else cp = cip;
  cp->tcgx = NULL;
  cp->chan = chan;
  cp->sound = sound;
  cp->sound_ctr = -1;
  cp->edit_ctr = -1;
  cp->sound_size = 0;
  cp->edit_size = 0;
  cp->cursor_on = 0;
  cp->cursor_visible = 0;
  cp->cursor = 0;
  cp->squelch_update = 0;
  cp->waving = 1; /* the default state (button is set when we start) */
  cp->ffting = 0;
  cp->printing = 0;
  cp->state = state;
  cp->amp_env = NULL;
  cp->original_env = NULL;
  cp->sonogram_data = NULL;
  cp->lisp_info = NULL;
  cp->stats = NULL;
  cp->drawing = 1;
  cp->hookable = 1;
  cp->gzy = 1.0;
  cp->gsy = 1.0;
  cp->selection_transform_size = 0;
  if (cp->last_sonogram) {FREE(cp->last_sonogram); cp->last_sonogram = NULL;}
  return(cp);
}

static chan_info *free_chan_info(chan_info *cp)
{
  /* this does not free the associated widgets -- they are merely unmanaged */
  snd_state *ss;
  chan_info_cleanup(cp);
  ss = cp->state;
  cp->squelch_update = 1;
  cp->tcgx = NULL;
  cp->axis = free_axis_info(cp->axis);
  if (cp->fft) cp->fft = free_fft_info(cp->fft);
  if (cp->fft_data) {FREE(cp->fft_data); cp->fft_data = NULL;}
  /* this may leave ->wp window unfreed? -- see snd-fft.c free_fft_state */
  cp->ffting = 0;
  cp->printing = 0;
  cp->waving = 1;
  if (cp->samples) {FREE(cp->samples); cp->samples = NULL;}
  if (cp->edits) free_edit_list(cp);
  if (cp->sounds) free_sound_list(cp);
  if (cp->marks) free_mark_list(cp,-1);
  if (cp->sounds) {FREE(cp->sounds); cp->sounds = NULL;}
  if (cp->mixes) free_mixes(cp);
  cp->sound = NULL;  /* a backpointer */
  cp->state = NULL;
  cp->cursor_on = 0;
  cp->cursor_visible = 0;
  cp->cursor = 0;
  cp->drawing = 1;
  if ((cp->amp_env) || (cp->original_env)) free_chan_env(cp);
  if (cp->sonogram_data) free_sono_info(cp);
  if (cp->last_sonogram) {FREE(cp->last_sonogram); cp->last_sonogram = NULL;}
  if (cp->lisp_info) cp->lisp_info = free_lisp_info(cp);
  if (cp->stats) 
    { 
      FREE(cp->stats); 
      cp->stats = NULL;
      if (show_usage_stats(ss)) update_stats_display(ss,FALSE);
    }
  cp->lisp_graphing = 0;
  cp->selection_transform_size = 0;
  return(cp);  /* pointer is left for possible future re-use */
}

snd_info *make_snd_info(snd_info *sip, snd_state *state, char *filename, file_info *hdr, int snd_slot)
{
  snd_info *sp = NULL;
  int chans,i;
  float secs;
  snd_state *ss = (snd_state *)state;
  /* assume file has been found and header read before reaching us */
  /* if a reused pointer, may need to extend current chans array */
  chans = hdr->chans;
  if ((!sip) || (((snd_any *)sip)->s_type != SND_INFO))
    {
      sp = (snd_info *)CALLOC(1,sizeof(snd_info));
      sp->s_type = SND_INFO;
      sp->chans = (chan_info **)CALLOC(chans,sizeof(chan_info *));
      sp->allocated_chans = chans;
      sp->sgx = (snd_context *)CALLOC(1,sizeof(snd_context));
    }
  else 
    {
      sp = sip;
      if (sp->allocated_chans < chans) 
	{
	  sp->chans = (chan_info **)REALLOC(sp->chans,chans*sizeof(chan_info *));
	  for (i=sp->allocated_chans;i<chans;i++) sp->chans[i] = NULL;
	  sp->allocated_chans = chans;
	}
    }
  sp->index = snd_slot;
  sp->nchans = chans;
  sp->hdr = hdr;
  sp->inuse = 1;
  sp->fullname = copy_string(filename);
  sp->shortname = filename_without_home_directory(sp->fullname); /* a pointer into fullname, not a new string */
  sp->state = ss;
  sp->expand = default_expand(ss);
  sp->last_expand = 450;
  sp->expanding = default_expanding(ss);
  secs = (float)hdr->samples/(float)(hdr->chans*hdr->srate);
  if (secs < 1.0) 
    sp->sx_scroll_max = 100;
  else sp->sx_scroll_max = (int)pow(10,(ceil(log10(secs)) + 2));
  sp->amp = default_amp(ss);
  sp->last_amp = (int)(default_amp(ss) * 50);
  sp->srate = fabs(default_speed(ss));
  sp->last_srate = 450;
  if (default_speed(ss) > 0.0) sp->play_direction = 1; else sp->play_direction = -1;
  sp->contrasting = default_contrasting(ss);
  sp->contrast = default_contrast(ss);
  sp->last_contrast = 0;
  sp->reverbing = default_reverbing(ss);
  sp->filtering = default_filtering(ss);
  sp->searching = 0;
  if (chans > 1)
    sp->combining = channel_style(ss);
  else sp->combining = CHANNELS_SEPARATE;
  sp->loading = 0;
  sp->evaling = 0;
  sp->marking = 0;
  sp->filing = 0;
  sp->read_only = 0;
  sp->minibuffer_on = 0;
  sp->minibuffer_temp = 0;
  sp->local_explen = default_expand_length(ss);
  sp->local_exprmp = default_expand_ramp(ss);
  sp->local_exphop = default_expand_hop(ss);
  sp->local_revfb = default_reverb_feedback(ss);
  sp->local_revlp = default_reverb_lowpass(ss);
  sp->revscl = default_reverb_scale(ss);
  sp->last_revscl = 20;
  sp->revlen = default_reverb_length(ss);
  sp->last_revlen = 0;
  sp->filter_order = default_filter_order(ss);
  sp->filter_dBing = default_filter_dBing(ss);
  sp->filter_changed = 0;
  sp->selected_channel = NO_SELECTION;
  sp->recording = 0;
  sp->replaying = 0;
  sp->playing = 0;
  sp->applying = 0;
  sp->lisp_graphing = 0;
  sp->initial_controls = NULL;
  sp->env_anew = 0;
  sp->contrast_amp = default_contrast_amp(ss);
  sp->fit_data_amps = NULL;
  sp->search_expr = NULL;
  sp->search_tree = NULL;
  sp->lacp = NULL;
#if HAVE_GUILE
  sp->search_proc = SCM_UNDEFINED;
  sp->eval_proc = SCM_UNDEFINED;
#endif
  sp->filter_env = default_env(1.0);
  sp->delete_me = 0;
  sp->mix_chan_amps = NULL;

  if (fit_data_on_open(ss)) 
    {
      sp->fit_data_amps = (float *)CALLOC(sp->nchans,sizeof(float));
      file_maxamps(state,sp->fullname,sp->fit_data_amps,hdr->chans,hdr->format);
      /* can't use snd-clm.c get_maxamp here because the file edit tree is not yet set up */
      /* can't use sound_chans etc here because this might be a raw header file */
    }
  return(sp);
}

snd_info *free_snd_info(snd_info *sp)
{
  int i;
  /* leave most for reuse as in free_chan_info */
  if (sp->sgx)
    {
      if ((sp->sgx)->env_data) remove_amp_env(sp);
      if ((sp->sgx)->apply_in_progress) remove_apply(sp);
    }
  snd_info_cleanup(sp);
  snd_filter_cleanup(sp);
  sp->syncing = 0;
  for (i=0;i<sp->nchans;i++)
    {
      if (sp->chans[i]) sp->chans[i]=free_chan_info(sp->chans[i]);
    }
  sp->state = NULL;
  sp->inuse = 0;
  sp->amp = 1.0;
  sp->srate = 1.0;
  sp->expand = 0.0;
  sp->expanding = 0;
  sp->contrasting = 0;
  sp->reverbing = 0;
  sp->filtering = 0;
  sp->filter_changed = 0;
  sp->play_direction = 1;
  sp->playing_mark = NULL;
  sp->playing = 0;
  sp->searching = 0;
  sp->loading = 0;
  sp->evaling = 0;
  sp->marking = 0;
  sp->filing = 0;
  sp->recording = 0;
  sp->replaying = 0;
  sp->applying = 0;
  sp->combining = CHANNELS_SEPARATE;
  sp->read_only = 0;
  sp->lisp_graphing = 0;
  sp->minibuffer_on = 0;   /* if it's on, should we clear it first ?? */
  sp->minibuffer_temp = 0;
  if (sp->search_expr) {FREE(sp->search_expr); sp->search_expr = NULL;}
  if (sp->search_tree) {free_sop(sp->search_tree); sp->search_tree = NULL;}
  if (sp->eval_expr) {FREE(sp->eval_expr); sp->eval_expr = NULL;}
  if (sp->eval_tree) {free_sop(sp->eval_tree); sp->eval_tree = NULL;}
#if HAVE_GUILE
  sp->search_proc = SCM_UNDEFINED;
  sp->eval_proc = SCM_UNDEFINED;
#endif
  sp->selected_channel = NO_SELECTION;
  sp->shortname = NULL; /* was a pointer into fullname */
  if (sp->fullname) FREE(sp->fullname);
  sp->fullname = NULL;
  if (sp->filter_env) sp->filter_env = free_env(sp->filter_env);
  if (sp->actions) flush_actions(sp);
  if (sp->initial_controls) free_controls(sp);
  sp->env_anew = 0;
  sp->contrast_amp = 1.0;
  sp->delete_me = 0;
  sp->mix_chan_amps = NULL;
  sp->lacp = NULL;
  if (sp->fit_data_amps)
    {
      FREE(sp->fit_data_amps);
      sp->fit_data_amps = NULL;
    }
  if (sp->hdr) sp->hdr = free_file_info(sp->hdr);
  if (sp->edited_region) clear_region_backpointer(sp);
  return(sp);  /* pointer is left for possible future re-use */
}

snd_info *completely_free_snd_info(snd_info *sp)
{
  int i;
  chan_info *cp;
  free_snd_info(sp);
  if (sp->sgx) FREE(sp->sgx);
  for (i=0;i<sp->allocated_chans;i++) 
    {
      cp = sp->chans[i];
      if (cp)
	{
	  if (cp->cgx) 
	    {
	      if ((cp->cgx)->ax) FREE((cp->cgx)->ax);
	      FREE(cp->cgx);
	    }
	  FREE(cp);
	}
    }
  FREE(sp->chans);
  FREE(sp);
  return(NULL);
}

int map_over_chans (snd_state *ss, int (*func)(chan_info *,void *), void *userptr)
{
  /* argument to func is chan_info pointer+void pointer of user spec, return non-zero = abort map, skips inactive sounds */
  int i,j,val;
  snd_info *sp;
  chan_info *cp;
  val = 0;
  if (ss)
    {
      for (i=0;i<ss->max_sounds;i++)
	{
	  if ((sp=((snd_info *)(ss->sounds[i]))))
	    {
	      if (sp->inuse)
		{
		  for (j=0;j<(sp->nchans);j++)
		    {
		      if ((cp=((chan_info *)(sp->chans[j]))))
			{
			  val = (*func)(cp,userptr);
			  if (val) return(val);}}}}}} /* I wish I were a'Lispin... */
  return(val);
}

int map_over_sound_chans (snd_info *sp, int (*func)(chan_info *,void *), void *userptr)
{
  /* argument to func is chan_info pointer+void pointer of user spec, return non-zero = abort map, skips inactive sounds */
  int j,val;
  chan_info *cp;
  val = 0;
  for (j=0;j<(sp->nchans);j++)
    {
      if ((cp=sp->chans[j]))
	{
	  val = (*func)(cp,userptr);
	  if (val) return(val);
	}
    }
  return(val);
}

int map_over_sounds (snd_state *ss, int (*func)(snd_info *,void *), void *userptr)
{
  /* argument to func is snd_info pointer, return non-zero = abort map, skips inactive sounds */
  int i,val;
  snd_info *sp;
  val = 0;
  if (ss)
    {
      for (i=0;i<ss->max_sounds;i++)
	{
	  if ((sp=((snd_info *)(ss->sounds[i]))))
	    {
	      if (sp->inuse)
		{
		  val = (*func)(sp,userptr);
		  if (val) return(val);}}}}
  return(val);
}

int map_over_separate_chans(snd_state *ss, int (*func)(chan_info *,void *), void *userptr)
{
  int i,val;
  snd_info *sp;
  val = 0;
  if (ss)
    {
      for (i=0;i<ss->max_sounds;i++)
	{
	  if ((sp=((snd_info *)(ss->sounds[i]))))
	    {
	      if (sp->inuse)
		{
		  if (sp->combining != CHANNELS_SEPARATE)
		    val = (*func)(sp->chans[0],userptr);
		  else val = map_over_sound_chans(sp,func,userptr);
		  if (val) return(val);
		}
	    }
	}
    }
  return(val);
}

snd_state *main_STATE(void *w)
{
  switch (((snd_any *)w)->s_type)
    {
    case SND_STATE: return((snd_state *)w); break;
    case SND_INFO: return(((snd_info *)w)->state); break;
    case CHAN_INFO: return(((chan_info *)w)->state); break;
    case FFT_INFO: if (((fft_info *)w)->chan) return(main_STATE((chan_info *)(((fft_info *)w)->chan))); break;
    case AXIS_INFO: return(((axis_info *)w)->ss); break;
#if DEBUGGING
    default: 
      snd_error("%s[%d] %s: s_type of %d is %d??",__FILE__,__LINE__,__FUNCTION__,(int)w,((snd_any *)w)->s_type);
      abort();
      break;
#endif
    }
  return(NULL);
}

char *snd_NAME(void *w)
{
  switch (((snd_any *)w)->s_type)
    {
    case SND_INFO: return(((snd_info *)w)->fullname); break;
    case CHAN_INFO: return(((snd_info *)(((chan_info *)w)->sound))->fullname); break;
    case FFT_INFO: if (((fft_info *)w)->chan) return(snd_NAME((chan_info *)(((fft_info *)w)->chan))); break;
    case AXIS_INFO: if (((axis_info *)w)->cp) return(snd_NAME((chan_info *)(((axis_info *)w)->cp))); break;
    case SND_STATE: return(snd_NAME(any_selected_sound((snd_state *)w))); break;
#if DEBUGGING
    default: 
      snd_error("%s[%d] %s: s_type of %d is %d??",__FILE__,__LINE__,__FUNCTION__,(int)w,((snd_any *)w)->s_type);
      abort();
      break;
#endif
    }
  return(NULL);
}

static file_info *snd_HEADER(void *w)
{
  switch (((snd_any *)w)->s_type)
    {
    case FILE_INFO: return((file_info *)w); break;
    case SND_INFO: return(((snd_info *)w)->hdr); break;
    case CHAN_INFO: return( ((snd_info *) (((chan_info *)w)->sound))->hdr); break;
    case FFT_INFO: if (((fft_info *)w)->chan) return(snd_HEADER((chan_info *)(((fft_info *)w)->chan))); break;
    case AXIS_INFO: if (((axis_info *)w)->cp) return(snd_HEADER((chan_info *)(((axis_info *)w)->cp))); break;
    case SND_STATE: return(snd_HEADER(any_selected_sound((snd_state *)w))); break;
#if DEBUGGING
    default: 
      snd_error("%s[%d] %s: s_type of %d is %d??",__FILE__,__LINE__,__FUNCTION__,(int)w,((snd_any *)w)->s_type);
      abort();
      break;
#endif
    }
  return(NULL);
}

int snd_SRATE (void *ptr)
{
  return(((file_info *)(snd_HEADER(ptr)))->srate);
}

int graph_low_SAMPLE (void *ptr)
{
  switch (((snd_any *)ptr)->s_type)
    {
    case CHAN_INFO: return(graph_low_SAMPLE(((chan_info *)ptr)->axis)); break;
    case AXIS_INFO: return(((axis_info *)ptr)->losamp); break;
#if DEBUGGING
    default: 
      snd_error("%s[%d] %s: s_type of %d is %d??",__FILE__,__LINE__,__FUNCTION__,(int)ptr,((snd_any *)ptr)->s_type);
      abort();
      break;
#endif
    }
  return(0);
}

int graph_high_SAMPLE (void *ptr)
{
  switch (((snd_any *)ptr)->s_type)
    {
    case CHAN_INFO: return(graph_high_SAMPLE(((chan_info *)ptr)->axis)); break;
    case AXIS_INFO: return(((axis_info *)ptr)->hisamp); break;
#if DEBUGGING
    default: 
      snd_error("%s[%d] %s: s_type of %d is %d??",__FILE__,__LINE__,__FUNCTION__,(int)ptr,((snd_any *)ptr)->s_type);
      abort();
      break;
#endif
    }
  return(0);
}

int snd_ok (snd_info *sp) {return((sp) && (sp->inuse));}

int active_channels (snd_state *ss,int count_virtual_channels)
{
  int chans,i;
  snd_info *sp;
  chans = 0;
  for (i=0;i<ss->max_sounds;i++)
    {
      if (snd_ok(sp = (ss->sounds[i])))
	{
	  if ((count_virtual_channels) || (sp->combining == CHANNELS_SEPARATE))
	    chans += sp->nchans;
	  else chans++;
	}
    }
  return(chans);
}

int find_free_sound_slot (snd_state *state, int desired_chans)
{
  int i,j;
  snd_info *sp;
  /* first try to find an unused slot that can accomodate desired_chans (to reduce Widget creation) */
  for (i=0;i<state->max_sounds;i++)
    {
      sp = state->sounds[i];
      if ((sp) && (sp->inuse == 0) && (sp->allocated_chans == desired_chans)) return(i);
    }
  for (i=0;i<state->max_sounds;i++)
    {
      sp = state->sounds[i];
      if ((sp) && (sp->inuse == 0) && (sp->allocated_chans > desired_chans)) return(i);
    }
  for (i=0;i<state->max_sounds;i++)
    {
      sp = state->sounds[i];
      if (sp == NULL) return(i);
      if (sp->inuse == 0) return(i);
    }
  /* need to REALLOC sounds to make space */
  j = state->max_sounds;
  state->max_sounds += 4;
  state->sounds = (snd_info **)REALLOC(state->sounds,state->max_sounds*sizeof(snd_info *));
  for (i=j;i<state->max_sounds;i++) state->sounds[i] = NULL;
  return(j);
}

static snd_info *any_active_sound(snd_state *ss)
{
  snd_info *sp;
  int i;
  for (i=0;i<ss->max_sounds;i++)
    {
      if ((sp=(ss->sounds[i])) && (sp->inuse)) return(sp);
    }
  return(NULL);
}

snd_info *selected_sound(snd_state *ss)
{
  if (ss->selected_sound != NO_SELECTION) return(ss->sounds[ss->selected_sound]);
  return(NULL);
}

snd_info *any_selected_sound (snd_state *ss)
{
  snd_info *sp;
  sp = selected_sound(ss);
  if (!sp) sp = any_active_sound(ss);
  return(sp);
}

chan_info *any_selected_channel(snd_info *sp)
{
  if (sp->selected_channel != NO_SELECTION) return(sp->chans[sp->selected_channel]);
  return(sp->chans[0]);
}

chan_info *selected_channel (void *ptr)
{
  snd_state *ss;
  snd_info *sp;
  ss = main_STATE(ptr);
  if (ss->selected_sound != NO_SELECTION)
    {
      sp = ss->sounds[ss->selected_sound];
      if ((sp->inuse) && (sp->selected_channel != NO_SELECTION))
	return(sp->chans[sp->selected_channel]);
    }
  return(NULL);
}

static void select_sound (snd_state *ss, snd_info *sp)
{
  snd_info *osp = NULL;
  if (ss->selected_sound != sp->index)
    {
      if (!ss->using_schemes)
	{
	  if (ss->selected_sound != NO_SELECTION) osp = ss->sounds[ss->selected_sound];
	  if ((osp) && (sp != osp) && (osp->inuse)) 
	    {
	      highlight_color(ss,w_snd_name(osp));
#if (XmVERSION > 1)
	      if (sound_style(ss) == SOUNDS_IN_NOTEBOOK) 
		XmChangeColor((osp->sgx)->tab,(ss->sgx)->graph_color);
#endif
	    }
	  if (sp->selected_channel != NO_SELECTION) 
	    {
	      white_color(ss,w_snd_name(sp));
#if (XmVERSION > 1)
	      if (sound_style(ss) == SOUNDS_IN_NOTEBOOK) 
		XmChangeColor((sp->sgx)->tab,(ss->sgx)->selected_graph_color);
#endif
	    }
	}
      ss->selected_sound = sp->index;
      highlight_selected_sound(ss);
      reflect_undo_or_redo_in_menu(any_selected_channel(sp));
      new_active_channel_alert(ss);
    }
}

void select_channel(snd_info *sp, int chan)
{
  snd_state *ss = sp->state;
  chan_info *cp,*ncp;
  cp = selected_channel(ss);
  if (cp != sp->chans[chan])
    {
      sp->selected_channel = chan;
      select_sound(ss,sp);
      if (cp) 
	{
	  recolor_graph(cp,FALSE);
	  (cp->cgx)->selected = 0;
	  if (sp != cp->sound) (cp->sound)->selected_channel = NO_SELECTION;
	  update_graph(cp,NULL);
	}
      ncp = sp->chans[chan];
      reflect_undo_or_redo_in_menu(ncp);
      recolor_graph(ncp,TRUE);
      (ncp->cgx)->selected = 1;
      if ((ss->sgx)->data_color != (ss->sgx)->selected_data_color) 
	update_graph(ncp,NULL);
      else draw_graph_border(ncp);
      goto_graph(ncp);
    }
}

chan_info *current_channel(void *ptr)
{
  snd_info *sp = NULL;
  snd_state *ss;
  if (!ptr) return(NULL); /* ptr can be null when Snd has only the menu bar */
  switch (((snd_any *)ptr)->s_type)
    {
    case CHAN_INFO: 
      return((chan_info *)ptr); 
      break;
    case SND_INFO: 
      sp = (snd_info *)ptr;
      if (sp->selected_channel != NO_SELECTION)
	return(sp->chans[sp->selected_channel]);
      else return(sp->chans[0]);
      break;
    case SND_STATE:
      ss = (snd_state *)ptr;
      if (ss->selected_sound != NO_SELECTION)
	sp = ss->sounds[ss->selected_sound];
      else sp = any_active_sound(ss);
      if (sp) return(current_channel(sp));
      break;
    }
  return(NULL);
}

int syncd_chans(snd_state *ss, int sync)
{
  int chans,i;
  snd_info *sp;
  chans = 0;
  for (i=0;i<ss->max_sounds;i++)
    {
      sp = ss->sounds[i];
      if ((sp) && (sp->inuse) && (sp->syncing == sync)) chans += sp->nchans;
    }
  return(chans);
}

void free_sync_info (sync_info *si)
{
  if (si)
    {
      if (si->begs) FREE(si->begs);
      si->begs = NULL;
      if (si->cps) FREE(si->cps);
      si->cps = NULL;
      FREE(si);
    }
}

sync_info *snd_sync(snd_state *ss, int sync)
{
  int i,j,k,chans;
  snd_info *sp;
  sync_info *si;
  chans = syncd_chans(ss,sync);
  if (chans > 0)
    {
      si = (sync_info *)CALLOC(1,sizeof(sync_info));
      si->begs = (int *)CALLOC(chans,sizeof(int));
      si->cps = (chan_info **)CALLOC(chans,sizeof(chan_info *));
      si->chans = chans;
      j=0;
      for (i=0;i<ss->max_sounds;i++)
	{
	  sp = ss->sounds[i];
	  if ((sp) && (sp->inuse) && (sp->syncing == sync))
	    {
	      for (k=0;k<sp->nchans;k++,j++)
		{
		  si->cps[j] = sp->chans[k];
		}
	    }
	}
      return(si);
    }
  return(NULL);
}

sync_info *make_simple_sync (chan_info *cp, int beg)
{
  sync_info *si;
  si = (sync_info *)CALLOC(1,sizeof(sync_info));
  si->chans = 1;
  si->cps = (chan_info **)CALLOC(1,sizeof(chan_info *));
  si->cps[0] = cp;
  si->begs = (int *)CALLOC(1,sizeof(int));
  si->begs[0] = beg;
  return(si);
}

snd_info *find_sound(snd_state *ss, char *name)
{
  snd_info *sp;
  char *sname;
  int i;
  for (i=0;i<ss->max_sounds;i++)
    {
      sp = ss->sounds[i];
      if ((sp) && (sp->inuse))
	{
	  if ((strcmp(name,sp->shortname) == 0) || (strcmp(name,sp->fullname) == 0)) return(sp);
	}
    }
  sname = filename_without_home_directory(name);
  for (i=0;i<ss->max_sounds;i++)
    {
      sp = ss->sounds[i];
      if ((sp) && (sp->inuse) && (strcmp(sname,sp->shortname) == 0)) return(sp);
    }
  return(NULL);
}

#define TIME_STR_SIZE 64
static char timestr[TIME_STR_SIZE];
#define INFO_BUFFER_SIZE 1024

void display_info(snd_info *sp)
{
  char *buffer = NULL;
  file_info *hdr;
  char *comment,*cstr = NULL;
  if (sp)
    {
      hdr = sp->hdr;
      if (hdr)
	{
	  buffer = (char *)CALLOC(INFO_BUFFER_SIZE,sizeof(char));
	  cstr = sound_comment(sp->fullname);
	  comment = cstr;
	  while ((comment) && (*comment) && 
		 (((*comment) == '\n') || ((*comment) == '\t') || 
		  ((*comment) == ' ') || ((*comment) == '\xd')))
	    comment++;
#if (!defined(HAVE_CONFIG_H)) || defined(HAVE_STRFTIME)
	  strftime(timestr,TIME_STR_SIZE,STRFTIME_FORMAT,localtime(&(sp->write_date)));
#endif
	  sprintf(buffer,STR_display_info,
		  hdr->srate,
		  hdr->chans,
		  (float)(hdr->samples)/(float)(hdr->chans * hdr->srate),(hdr->samples)/(hdr->chans),(hdr->chans == 1) ? "samples" : "frames",
		  sound_type_name(hdr->type),
		  sound_format_name(hdr->format),
		  timestr,
		  (comment) ? comment : "");
	  ssnd_help(sp->state,
		    sp->shortname,
		    buffer,
		    NULL);
	  if (cstr) FREE(cstr);
	  FREE(buffer);
	}
    }
}

