#include "snd.h"

/* a bit of a grab-bag -- nominally file-related */

enum {fdata_form,fdata_sep1,fdata_hlab,fdata_hlist,fdata_dlab,fdata_dlist,
      fdata_slab,fdata_stext,fdata_clab,fdata_ctext,fdata_sep2,fdata_sep3, 
      fdata_comment_label,fdata_comment_text,fdata_out_form,fdata_sep4,
      fdata_loclab,fdata_loctext};
#define FILE_DATA_WIDGETS 18

/* these are needed only in snd-xrec.c */
Widget file_data_header_list(Widget *wids) {return(wids[fdata_hlist]);}
Widget file_data_format_list(Widget *wids) {return(wids[fdata_dlist]);}
Widget file_data_srate_text(Widget *wids) {return(wids[fdata_stext]);}
Widget file_data_chans_text(Widget *wids) {return(wids[fdata_ctext]);}
Widget file_data_comment_text(Widget *wids) {return(wids[fdata_comment_text]);}

static char file_string[256];

static void color_file_selection_box(Widget w, snd_state *ss)
{
  Widget wtmp = NULL,ftmp = NULL;
  if (!(ss->using_schemes)) 	
    {
      map_over_children(w,set_main_color_of_widget,(void *)ss);
      XtVaSetValues(XmFileSelectionBoxGetChild(w,XmDIALOG_DIR_LIST),XmNbackground,(ss->sgx)->white,NULL);
      XtVaSetValues(XmFileSelectionBoxGetChild(w,XmDIALOG_LIST),XmNbackground,(ss->sgx)->white,NULL);
#ifndef LESSTIF_VERSION
      XtVaSetValues(XtNameToWidget(w,"Apply"),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaSetValues(XtNameToWidget(w,"Cancel"),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaSetValues(XtNameToWidget(w,"Help"),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaSetValues(XtNameToWidget(w,"OK"),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      wtmp = XtNameToWidget(w,"Text");
      if (!wtmp) wtmp = XmFileSelectionBoxGetChild(w,XmDIALOG_TEXT);
      ftmp = XtNameToWidget(w,"FilterText");
      if (!ftmp) ftmp = XmFileSelectionBoxGetChild(w,XmDIALOG_FILTER_TEXT);	
#else
      XtVaSetValues(XmFileSelectionBoxGetChild(w,XmDIALOG_APPLY_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaSetValues(XmFileSelectionBoxGetChild(w,XmDIALOG_CANCEL_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaSetValues(XmFileSelectionBoxGetChild(w,XmDIALOG_HELP_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaSetValues(XmFileSelectionBoxGetChild(w,XmDIALOG_OK_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      wtmp = XmFileSelectionBoxGetChild(w,XmDIALOG_TEXT);
      ftmp = XmFileSelectionBoxGetChild(w,XmDIALOG_FILTER_TEXT);	
#endif
      if (wtmp)
	{
	  XtAddCallback(wtmp,XmNfocusCallback,textfield_focus_Callback,ss);
	  XtAddCallback(wtmp,XmNlosingFocusCallback,textfield_unfocus_Callback,ss);
	}
      if (ftmp)
	{
	  XtAddCallback(ftmp,XmNfocusCallback,textfield_focus_Callback,ss);
	  XtAddCallback(ftmp,XmNlosingFocusCallback,textfield_unfocus_Callback,ss);
	}
    }
}

static void File_Open_OK_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_info *sp;
  snd_state *ss = (snd_state *)clientData;
  XmFileSelectionBoxCallbackStruct *cbs = (XmFileSelectionBoxCallbackStruct *) callData;
  char *fileName;
  XtUnmanageChild(w);
  XmStringGetLtoR (cbs->value,XmFONTLIST_DEFAULT_TAG,&fileName);
  /* this can be a directory name if the user clicked 'ok' when he meant 'cancel' */
  if (!(is_directory(fileName)))
    {
      sp = snd_open_file(fileName,ss);
      if (sp) select_channel(sp,0);           /* add_sound_window (snd-xsnd.c) will report reason for error, if any */
    }
  else
    snd_error(STR_is_a_directory,fileName);
}

static void File_Open_Help_Callback (Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
       "File Open",
"If you click the 'Sound Files Only'\n\
button, only those files in the current\n\
directory that look vaguely like sound\n\
files will be displayed.\n\
");
}

static void File_Open_Cancel_Callback (Widget w,XtPointer clientData,XtPointer callData) 
{
  XtUnmanageChild (w);
}

static Widget open_dialog = NULL; 
static XmSearchProc default_search_proc;

static int string_compare(const void *ss1, const void *ss2)
{
  return(strcmp((*((char **)ss1)),(*((char **)ss2)))); /* sweet baby jesus... ain't C grand? */
}

static void just_sounds_help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Sound Files Only",
"If you click the 'Sound Files Only'\n\
button, only those files in the current\n\
directory that look vaguely like sound\n\
files will be displayed.  The decision\n\
is based on the file's extension.\n\
");
}

static char *fullpathname = NULL;
static int file_SB_list_needs_update = 0; /* browser list needs update (new 'filter' or whatever) */
static int new_file_written = 0;          /* sound file list needs update because we wrote a new file */

void alert_new_file(void) {new_file_written = 1;}

static void sound_file_search(Widget FSB_w, XmFileSelectionBoxCallbackStruct *callData)
{
  /* generate list of sound files, set XmNfileListItems, XmNfileListItemCount, XmNlistUpdated
   * the latter if new file list generated -- if no files, XmNfileListItems NULL, Count 0
   * can also set XmNdirSpec to full file spec of dir.  The callbackstruct is:
   *    typedef struct
   *    {
   *      int      reason;         Why called
   *      XEvent   * event;  
   *      XmString value;          current value of XmNdirSpec
   *      int      length;         number of bytes in value
   *      XmString mask;           current value of XmNdirMask
   *      int      mask_length;    number of bytes in mask
   *      XmString dir;            current base directory
   *      int      dir_length;     number of bytes in dir
   *      XmString pattern;        current search pattern
   *      int      pattern_length; number of bytes in pattern
   *    } XmFileSelectionBoxCallbackStruct;
   *
   * proc should stick to XmNfileTypeMask (type unsigned char): 
   *   XmFILE_REGULAR -- regular files
   *   XmFILE_DIRECTORY -- directories
   *   XmFILE_ANY_TYPE 
   *
   * the pattern (file name mask) only matters if the filter button is hit, 
   * it appears to be "*" until the filter is invoked.
   */
  char *pattern,*our_dir,*sp,*sn;
  static char *save_dir = NULL;
  static char *last_dir = NULL;
  static dir *sound_files,*current_files;
  static char *last_pattern = NULL;
  dir *cdp;
  XmFileSelectionBoxCallbackStruct *data = (XmFileSelectionBoxCallbackStruct *)callData;
  XmString *names = NULL;
  int i,filter_callback,need_update;

  XmStringGetLtoR (data->pattern,XmFONTLIST_DEFAULT_TAG,&pattern);
  XmStringGetLtoR (data->dir,XmFONTLIST_DEFAULT_TAG,&our_dir);

  if (!fullpathname) fullpathname = (char *)CALLOC(FILENAME_MAX,sizeof(char));
  filter_callback = (strcmp(pattern,"*") != 0);
  need_update = file_SB_list_needs_update;
  file_SB_list_needs_update = 0;
  if (!filter_callback)
    {
      if ((!last_dir) || (strcmp(our_dir,last_dir) != 0) || (new_file_written))
	{
	  if (current_files) current_files = free_dir(current_files);
	  if (last_dir) {FREE(last_dir); last_dir = NULL;}
	  last_dir = (char *)CALLOC(snd_strlen(our_dir)+1,sizeof(char));
	  strcpy(last_dir,our_dir);
	  strcpy(fullpathname,our_dir);
	  save_dir = (char *)(fullpathname+snd_strlen(our_dir));
	  sound_files = find_sound_files_in_dir(our_dir);
	  need_update = 1;
	}
      if (last_pattern)
	{
	  FREE(last_pattern);
	  last_pattern = NULL;
	}
      cdp = sound_files;
    }
  else 
    {
      if ((!last_pattern) || (strcmp(pattern,last_pattern) != 0) || (new_file_written))
	  {
	    if (last_pattern) {FREE(last_pattern); last_pattern = NULL;}
	    last_pattern = (char *)CALLOC(snd_strlen(pattern)+1,sizeof(char));
	    strcpy(last_pattern,pattern);
	    if (current_files)  current_files = free_dir(current_files);
	    if ((sound_files) && (sound_files->len > 0)) current_files = filter_sound_files(sound_files,pattern);
	    need_update = 1;
	  }
      cdp = current_files;
    }  
  new_file_written = 0;
  if (need_update)
    {
      if ((cdp) && (cdp->len > 0))
	{
	  qsort((void *)(cdp->files),cdp->len,sizeof(char *),string_compare);
	  names = (XmString *)CALLOC(cdp->len,sizeof(XmString));

#ifdef SGI
	  /* this is true only if the SGI "enhanced FSB" is in use, I hope */
	  if (!(XtNameToWidget(FSB_w,"Text"))) save_dir = fullpathname;
	  /* can't use SgDIALOG_FINDER here as suggested by SGI "Integration Guide" because
	   * XmFileSelectionBoxGetChild(FSB_w,SgDIALOG_FINDER)) generates an error if
	   * snd was loaded without -lSgm.
	   */
#endif

	  for (i=0;i<cdp->len;i++) 
	    {
	      for (sp=save_dir,sn=cdp->files[i];((*sp)=(*sn)) != '\0';sp++,sn++);
	      /* save_dir is a pointer into fullpathname after the directory portion */
	      names[i] = XmStringCreate(fullpathname,XmFONTLIST_DEFAULT_TAG);
	    }
	}
      else names=NULL;
      if (cdp) XtVaSetValues(FSB_w,XmNfileListItems,names,XmNfileListItemCount,cdp->len,XmNlistUpdated,TRUE,NULL);
      if (names)
	{
	  for (i=0;i<cdp->len;i++) if (names[i]) XmStringFree(names[i]);
	  FREE(names);
	}
    }
}

static void force_directory_reread(void)
{
  XmString dirmask;
  XtVaGetValues(open_dialog,XmNdirMask,&dirmask,NULL);
  XmFileSelectionDoSearch(open_dialog,dirmask);
  XmStringFree(dirmask);
}

static void just_sounds_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  XmToggleButtonCallbackStruct *cb = (XmToggleButtonCallbackStruct *)callData;
  if (cb->set)
    XtVaSetValues(open_dialog,XmNfileSearchProc,sound_file_search,NULL);
  else XtVaSetValues(open_dialog,XmNfileSearchProc,default_search_proc,NULL);
  file_SB_list_needs_update = 1;
  force_directory_reread();
}

static Widget just_sounds_button = NULL;
static int just_sounds_state = FALSE;

void toggle_just_sounds(int n)
{
  if (just_sounds_button)
    XmToggleButtonSetState(just_sounds_button,n,TRUE);
  just_sounds_state = n;
}


Boolean CreateOpenDialog(Widget w,XtPointer clientData)
{
  /* file selection dialog box with added "Just Sound Files" toggle button */
  Arg args[20];
  int n;
  XmString s1;
  Widget wtmp = NULL;
  snd_state *ss = (snd_state *)clientData;
  n=0;
  if (!open_dialog)
    {
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      if (just_sounds_state)
	{
	  XtSetArg(args[n],XmNfileSearchProc,sound_file_search); n++;
	}
      s1 = XmStringCreate(STR_open_p,XmFONTLIST_DEFAULT_TAG);
      XtSetArg(args[n],XmNselectionLabelString,s1); n++;
      open_dialog = XmCreateFileSelectionDialog(w,STR_File,args,n);
      XmStringFree(s1);
#ifdef LESSTIF_VERSION
      XtManageChild(open_dialog);
#endif
#if OVERRIDE_TOGGLE
      override_form_translation(open_dialog);
#endif

      just_sounds_button = XtVaCreateManagedWidget(STR_Sound_Files_Only,xmToggleButtonWidgetClass,open_dialog,
						   XmNset,just_sounds_state,
						   XmNalignment,XmALIGNMENT_BEGINNING,
						   NULL);
      color_file_selection_box(open_dialog,ss);

#ifndef LESSTIF_VERSION
      wtmp = XtNameToWidget(open_dialog,"Text");
      if (!wtmp) wtmp = XmFileSelectionBoxGetChild(open_dialog,XmDIALOG_TEXT);
#else
      wtmp = XmFileSelectionBoxGetChild(open_dialog,XmDIALOG_TEXT);
#endif
      if (wtmp) add_completer_to_textfield(ss,wtmp,add_completer_func(filename_completer));
      
#ifndef LESSTIF_VERSION
      wtmp = XtNameToWidget(open_dialog,"FilterText");
      if (!wtmp) wtmp = XmFileSelectionBoxGetChild(open_dialog,XmDIALOG_FILTER_TEXT);
#else
      wtmp = XmFileSelectionBoxGetChild(open_dialog,XmDIALOG_FILTER_TEXT);
#endif
      if (wtmp) add_completer_to_textfield(ss,wtmp,add_completer_func(filename_completer));

      if (!(ss->using_schemes)) XtVaSetValues(just_sounds_button,XmNselectColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaGetValues(open_dialog,XmNfileSearchProc,&default_search_proc,NULL);
      XtAddCallback(open_dialog,XmNokCallback,File_Open_OK_Callback,clientData);
      XtAddCallback(open_dialog,XmNcancelCallback,File_Open_Cancel_Callback,clientData);
      XtAddCallback(open_dialog,XmNhelpCallback,File_Open_Help_Callback,clientData);
      XtAddCallback(just_sounds_button,XmNvalueChangedCallback,just_sounds_Callback,NULL);
      XtAddCallback(just_sounds_button,XmNhelpCallback,just_sounds_help_Callback,clientData);
    }
  return(TRUE);
}

void snd_open_file_dialog(Widget w, snd_state *ss)
{
  finish_keyboard_selection();
  if (!open_dialog) CreateOpenDialog(w,ss);
  if (new_file_written) 
    {
      force_directory_reread();
      new_file_written = 0;
    }
  if (!(XtIsManaged(open_dialog))) XtManageChild(open_dialog);
}


/* -------- save as dialog (file and edit menus) -------- */
/* 
 * changed 19-June-97 to simply save the current state under the new name and return
 * to the current state/file (different from emacs) -- this keeps mix console intact
 * across backups and so on, and seems more useful to me than switching to the new file.
 */

typedef struct {
  Widget dialog;
  Widget header_list; 
  Widget data_list;
  Widget text;
  Widget srtext;
  Widget comtext;
  snd_state *state;
  int type;
  int header_choice,data_choice;
} save_as_info;

static save_as_info *save_as_dialog = NULL;

#define FILE_SAVE_AS 1
#define EDIT_SAVE_AS 2

#define NUM_HEADER_TYPES 7
#define NUM_VISIBLE_HEADERS 4
#define NUM_NEXT_FORMATS 8
#define NUM_IRCAM_FORMATS 5
#define NUM_WAVE_FORMATS 7
#define NUM_AIFC_FORMATS 13
#define NUM_AIFF_FORMATS 4
#define NUM_NIST_FORMATS 7

static char *next_data_formats[NUM_NEXT_FORMATS] = {"short","mulaw","signed byte  ","float","long","alaw","24 bit","double"};
static char *ircam_data_formats[NUM_IRCAM_FORMATS] = {"short","mulaw","float        ","long","alaw"};
static char *wave_data_formats[NUM_WAVE_FORMATS] = {"mulaw","alaw","unsigned byte","short","long","float","24 bit"};
static char *aifc_data_formats[NUM_AIFC_FORMATS] = {"short","mulaw","signed byte  ","long","alaw","24 bit",
						    "float","double","unsigned byte","short swapped",
						    "long swapped","24 bit swapped","unsigned short"};
static char *aiff_data_formats[NUM_AIFF_FORMATS] = {"short","long","signed byte","24 bit"};
static char *nist_data_formats[NUM_NIST_FORMATS] = {"16-bit BE","16-bit LE","32-bit BE","32-bit LE","8-bit","24-bit BE","24-bit LE"};

static int next_dfs[NUM_NEXT_FORMATS] = {SNDLIB_16_LINEAR,SNDLIB_8_MULAW,SNDLIB_8_LINEAR,SNDLIB_32_FLOAT,SNDLIB_32_LINEAR,SNDLIB_8_ALAW,
					 SNDLIB_24_LINEAR,SNDLIB_64_DOUBLE};
static int ircam_dfs[NUM_IRCAM_FORMATS] = {SNDLIB_16_LINEAR,SNDLIB_8_MULAW,SNDLIB_32_FLOAT,SNDLIB_32_LINEAR,SNDLIB_8_ALAW};
static int wave_dfs[NUM_WAVE_FORMATS] = {SNDLIB_8_MULAW,SNDLIB_8_ALAW,SNDLIB_8_UNSIGNED,SNDLIB_16_LINEAR_LITTLE_ENDIAN,
					 SNDLIB_32_LINEAR_LITTLE_ENDIAN,SNDLIB_32_FLOAT_LITTLE_ENDIAN,SNDLIB_24_LINEAR_LITTLE_ENDIAN};
static int aifc_dfs[NUM_AIFC_FORMATS] = {SNDLIB_16_LINEAR,SNDLIB_8_MULAW,SNDLIB_8_LINEAR,SNDLIB_32_LINEAR,SNDLIB_8_ALAW,SNDLIB_24_LINEAR,
					 SNDLIB_32_FLOAT,SNDLIB_64_DOUBLE,SNDLIB_8_UNSIGNED,SNDLIB_16_LINEAR_LITTLE_ENDIAN,
					 SNDLIB_32_LINEAR_LITTLE_ENDIAN,SNDLIB_24_LINEAR_LITTLE_ENDIAN,SNDLIB_16_UNSIGNED};
static int aiff_dfs[NUM_AIFF_FORMATS] = {SNDLIB_16_LINEAR,SNDLIB_32_LINEAR,SNDLIB_8_LINEAR,SNDLIB_24_LINEAR};
static int nist_dfs[NUM_NIST_FORMATS] = {SNDLIB_16_LINEAR,SNDLIB_16_LINEAR_LITTLE_ENDIAN,SNDLIB_32_LINEAR,SNDLIB_32_LINEAR_LITTLE_ENDIAN,
					 SNDLIB_8_LINEAR,SNDLIB_24_LINEAR,SNDLIB_24_LINEAR_LITTLE_ENDIAN};

#define NUM_DATA_FORMATS 16
static char *data_formats[NUM_DATA_FORMATS] = {
  "16 bit big-endian int",
  "8 bit mulaw",
  "8 bit signed int",
  "32 bit big-endian float",
  "32 bit big-endian int",
  "8 bit alaw",
  "8 bit unsigned int",
  "24 bit big-endian int",
  "64 bit big-endian double",
  "16 bit little-endian int",
  "32 bit little-endian int",
  "32 bit little-endian float",
  "64 bit little-endian double",
  "16 bit big-endian unsigned int",
  "16 bit little-endian unsigned int",
  "24 bit little-endian int"};

/* must parallel sndlib.h definitions */

static int get_header_type_from_position(int position)
{
  switch (position)
    {
    case NEXT_POSITION: return(NeXT_sound_file); break;
    case AIFC_POSITION: return(AIFF_sound_file); break;
    case RIFF_POSITION: return(RIFF_sound_file); break;
    case IRCAM_POSITION: return(IRCAM_sound_file); break;
    case RAW_POSITION: return(raw_sound_file); break;
    case AIFF_POSITION: return(old_style_AIFF_sound_file); break;
    case NIST_POSITION: return(NIST_sound_file); break;
    }
  return(raw_sound_file);
}

static int get_data_format_from_header_and_position(int header, int position)
{
  switch (header)
    {
    case NeXT_sound_file: return(next_dfs[position-1]); break;
    case AIFF_sound_file: return(aifc_dfs[position-1]); break;
    case RIFF_sound_file: return(wave_dfs[position-1]); break;
    case IRCAM_sound_file: return(ircam_dfs[position-1]); break;
    case raw_sound_file: return(position); break;
    case old_style_AIFF_sound_file: return(aiff_dfs[position-1]); break;
    case NIST_sound_file: return(nist_dfs[position-1]); break;
    }
  return(position);
}

typedef struct {
  snd_info *sp;
  char *fullname;
  int edits;
} same_name_info;

static int check_for_same_name(snd_info *sp1, void *ur_info)
{
  int i;
  chan_info *cp;
  same_name_info *info = (same_name_info *)ur_info;
  if ((sp1) && (strcmp(sp1->fullname,info->fullname) == 0))
    {
      
      info->sp = sp1;
      for (i=0;i<sp1->nchans;i++) 
	{
	  cp = sp1->chans[i];
	  if (info->edits < cp->edit_ctr) info->edits = cp->edit_ctr;
	}
      return(1); /* stop immediately and deal with this one */
    }
  return(0);
}

void read_file_data_choices(Widget stext, Widget ctext, Widget hlist, Widget dlist, int *srate, int *chans, int *type, int *format)
{
  char *str;
  int n;
#if HAVE_SELECTED_POSITION_COUNT
  unsigned int *ns = NULL;
#else
  int *ns = NULL;
#endif
  if (stext) {str = XmTextGetString(stext); if (str) {sscanf(str,"%d",&n); (*srate) = n; FREE(str);}}
  if (ctext) {str = XmTextGetString(ctext); if (str) {sscanf(str,"%d",&n); (*chans) = n; FREE(str);}}
#if HAVE_SELECTED_POSITION_COUNT
  if (hlist)
    {
      XtVaGetValues(hlist,XmNselectedPositionCount,&n,NULL);
      if (n>0)
	{
	  XtVaGetValues(hlist,XmNselectedPositions,&ns,NULL);
	  (*type) = get_header_type_from_position(ns[0]);
	}
    }
  if (dlist)
    {
      XtVaGetValues(dlist,XmNselectedPositionCount,&n,NULL);
      if (n>0)
	{
	  XtVaGetValues(dlist,XmNselectedPositions,&ns,NULL);
	  (*format) = get_data_format_from_header_and_position((*type),ns[0]);
	}
    }
#else
  if (hlist)
    {
      XmListGetSelectedPos(hlist,&ns,&n);
      if (n>0)
	(*type) = get_header_type_from_position(ns[0]);
    }
  if (dlist)
    {
      XmListGetSelectedPos(dlist,&ns,&n);
      if (n>0)
	(*format) = get_data_format_from_header_and_position((*type),ns[0]);
    }
#endif
  if (ns) FREE(ns);
}

static void save_as_ok_callback(Widget w,XtPointer clientData,XtPointer callData)
{ /* called from both save as options */
  save_as_info *fd = (save_as_info *)clientData;
  XmAnyCallbackStruct *cb = (XmAnyCallbackStruct *)callData;
  Widget text;
  char *str = NULL,*fullname,*comment,*ofile;
  same_name_info *collision = NULL;
  snd_info *sp;
  int result;
  snd_state *ss;
  int type,format,srate,true_type,err;
  ss = fd->state;
  if (cb->event == ((ss->sgx)->text_activate_event)) return; /* <cr> in one of text fields */
  str = XmTextGetString(fd->srtext);
  sscanf(str,"%d",&srate);
  if (str) FREE(str);
  comment = XmTextGetString(fd->comtext);
  type = get_header_type_from_position(fd->header_choice);
  format = get_data_format_from_header_and_position(type,fd->data_choice);
  text = fd->text;
  str = XmTextGetString(text);
  sp = any_selected_sound(ss);
  if ((str) && (*str))
    {
      if (sp) clear_minibuffer(sp);
      alert_new_file();
      /* now check in-core files -- need to close any of same name -- if edited what to do? */
      /* also it's possible the new file name is the same as the current file name(!) */
      fullname = copy_string(mus_complete_filename(str));
      if (!(snd_overwrite_ok(ss,fullname))) {FREE(fullname); FREE(str); return;}
      if (strcmp(fullname,sp->fullname) == 0)
	{
	  /* normally save-as saves the current edit tree, merely saving the current state
	   * in a separate, presumably inactive file; here we're being asked to overwrite
	   * the current file in save-as; we can't have it both ways -- we'll save the edits 
	   * in a temp file, then rename/copy the temp, and call update 
	   */
	  if (sp->read_only)
	    {
	      sprintf(file_string,STR_cant_save_as,fullname,sp->shortname);
	      report_in_minibuffer(sp,file_string);
	      FREE(fullname);
	      FREE(str);
	      return;
	    }
	  /* it's possible also that the same-named file is open in several windows -- for now we'll ignore that */
	  /* also what if a sound is write-protected in one window, and not in another? */
	  ofile = snd_tempnam(ss); 
	  if (fd->type == FILE_SAVE_AS)
	    result = save_edits_2(sp,ofile,type,format,srate,comment);
	  else result = save_selection(ss,ofile,type,format,srate,comment);
	  if (result != SND_NO_ERROR)
	    {
	      sprintf(file_string,STR_save_as_temp,ofile,strerror(errno),snd_error_name(result));
	      report_in_minibuffer(sp,file_string);
	    }
	  else err = snd_copy_file(ss,ofile,sp->fullname);
	  snd_update(ss,sp);
	  FREE(ofile);
	  FREE(str);
	  FREE(fullname);
	}
      else
	{
	  collision = (same_name_info *)CALLOC(1,sizeof(same_name_info));
	  collision->fullname = fullname;
	  collision->edits = 0;
	  collision->sp = NULL;
	  map_over_sounds(ss,check_for_same_name,(void *)collision);
	  if (collision->sp)
	    {
	      /* if no edits, we'll just close, overwrite, reopen */
	      /* if edits, we need to ask luser what to do */
	      /* we don't need to check for overwrites at this point */
	      if (collision->edits > 0)
		{
		  sprintf(file_string,STR_has_unsaved_edits,str,str);
		  if (!(snd_yes_or_no_p(ss,file_string))) {FREE(fullname); FREE(collision); return;}
		}
	      snd_close_file(collision->sp,ss);
	    }
	  if (type == old_style_AIFF_sound_file)
	    {
	      mus_set_aifc_header(0);
	      true_type = AIFF_sound_file;
	    }
	  else true_type = type;
	  if (fd->type == FILE_SAVE_AS)
	    result = save_edits_2(sp,str,true_type,format,srate,comment);
	  else result = save_selection(ss,str,true_type,format,srate,comment);
	  if (result != SND_NO_ERROR)
	    sprintf(file_string,"%s: %s (%s)",str,strerror(errno),snd_error_name(result));
	  else sprintf(file_string,STR_saved_as_p,(fd->type == FILE_SAVE_AS) ? sp->shortname : "selection",str);
	  report_in_minibuffer(sp,file_string);
	  if (type == old_style_AIFF_sound_file) mus_set_aifc_header(1);
	  if (collision->sp) snd_open_file(fullname,ss);
	  FREE(str);
	  FREE(fullname);
	  FREE(collision);
	}
    }
  else if (sp) report_in_minibuffer(sp,STR_not_saved_no_name_given);
  XtUnmanageChild(fd->dialog);
} 

static void save_as_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Save As",
"You can save the current state of a file or region\n\
under a different file name using the Save\n\
As option.  The output header type, data format,\n\
and sampling rate can also be set.  The data formats\n\
are big-endian where relevant except for 'wave'\n\
output.  If a file by the chosen name already exists\n\
it is silently overwritten, unless that file is\n\
already open in Snd and has edits.  In that case,\n\
you'll be asked what to do.  If you want to be warned\n\
whenever a file is about to be overwritten by this\n\
option, set the resource overwriteCheck to 1.\n\
If you give the current file name to Save As,\n\
any current edits will be saved and the current\n\
version in Snd will be updated (that is, in this\n\
case, the current edit tree is not preserved).\n\
");
}

static void save_as_cancel_callback(Widget w,XtPointer clientData,XtPointer callData)
{ 
  save_as_info *fd = (save_as_info *)clientData;
  XtUnmanageChild(fd->dialog);
} 

void load_header_and_data_lists(Widget hlist, Widget dlist, int type, int format, int *hpos, int *dpos)
{
  int dformats = 0,header_pos = 0,data_pos = 0,i;
  char **fl = NULL;
  XmString *strs;
  switch (type)
    {
    case AIFF_sound_file: 
      dformats = NUM_AIFC_FORMATS; 
      fl = aifc_data_formats; 
      header_pos = AIFC_POSITION; 
      data_pos = 1;
      for (i=0;i<NUM_AIFC_FORMATS;i++) if (format == aifc_dfs[i]) {data_pos = i+1; break;}
      break;
    case RIFF_sound_file: 
      dformats = NUM_WAVE_FORMATS; 
      fl = wave_data_formats; 
      header_pos = RIFF_POSITION;
      data_pos = 4;
      for (i=0;i<NUM_WAVE_FORMATS;i++) if (format == wave_dfs[i]) {data_pos = i+1; break;}
      break;
    case IRCAM_sound_file: 
      dformats =NUM_IRCAM_FORMATS; 
      fl = ircam_data_formats; 
      header_pos = IRCAM_POSITION; 
      data_pos = 1;
      for (i=0;i<NUM_IRCAM_FORMATS;i++) if (format == ircam_dfs[i]) {data_pos = i+1; break;}
      break;
    case NeXT_sound_file:
      dformats = NUM_NEXT_FORMATS; 
      fl = next_data_formats; 
      header_pos = NEXT_POSITION; 
      data_pos = 1;
      for (i=0;i<NUM_NEXT_FORMATS;i++) if (format == next_dfs[i]) {data_pos = i+1; break;}
      break;
    case NIST_sound_file:
      dformats = NUM_NIST_FORMATS; 
      fl = nist_data_formats; 
      header_pos = NIST_POSITION; 
      data_pos = 1;
      for (i=0;i<NUM_NIST_FORMATS;i++) if (format == nist_dfs[i]) {data_pos = i+1; break;}
      break;
    case raw_sound_file:
      dformats = NUM_DATA_FORMATS; 
      fl = data_formats; 
      header_pos = RAW_POSITION; 
      data_pos = format;
      break;
    case old_style_AIFF_sound_file: 
      dformats = NUM_AIFF_FORMATS; 
      fl = aiff_data_formats; 
      header_pos = AIFF_POSITION; 
      data_pos = 1;
      for (i=0;i<NUM_AIFF_FORMATS;i++) if (format == aiff_dfs[i]) {data_pos = i+1; break;}
      break;
    }
  XmListSelectPos(hlist,header_pos,FALSE);
  strs = (XmString *)CALLOC(dformats,sizeof(XmString)); 
  for (i=0;i<dformats;i++) strs[i] = XmStringCreate(fl[i],XmFONTLIST_DEFAULT_TAG);
  XtVaSetValues(dlist,XmNitems,strs,XmNitemCount,dformats,NULL);
  for (i=0;i<dformats;i++) XmStringFree(strs[i]);
  FREE(strs);
  XmListSelectPos(dlist,data_pos,FALSE);
  (*hpos) = header_pos;
  (*dpos) = data_pos;
}

static void Save_as_Header_type_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* set fdw->header_choice, if needed reload data_formats window */
  int pos;
  file_info *hdr;
  XmListCallbackStruct *cbs = (XmListCallbackStruct *)callData;
  save_as_info *fdw = (save_as_info *)clientData;
  pos = cbs->item_position;
  if (fdw->header_choice != pos)
    {
      hdr = (file_info *)CALLOC(1,sizeof(file_info));
      hdr->comment = NULL;
      switch (pos)
	{
	case NEXT_POSITION: hdr->type = NeXT_sound_file; hdr->format = SNDLIB_16_LINEAR; break;
	case NIST_POSITION: hdr->type = NIST_sound_file; hdr->format = SNDLIB_16_LINEAR; break;
	case AIFC_POSITION: hdr->type = AIFF_sound_file; hdr->format = SNDLIB_16_LINEAR; break;
	case RIFF_POSITION: hdr->type = RIFF_sound_file; hdr->format = SNDLIB_16_LINEAR_LITTLE_ENDIAN; break;
	case IRCAM_POSITION: hdr->type = IRCAM_sound_file; hdr->format = SNDLIB_16_LINEAR; break;
	case RAW_POSITION: hdr->type = raw_sound_file; hdr->format = SNDLIB_16_LINEAR; break;
	case AIFF_POSITION: hdr->type = old_style_AIFF_sound_file; hdr->format = SNDLIB_16_LINEAR; break;
	}
      load_header_and_data_lists(fdw->header_list,fdw->data_list,hdr->type,hdr->format,&(fdw->header_choice),&(fdw->data_choice));
      FREE(hdr);
    }
}

static void Save_as_Data_format_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  XmListCallbackStruct *cbs = (XmListCallbackStruct *)callData;
  save_as_info *fdw = (save_as_info *)clientData;
  fdw->data_choice = cbs->item_position;
}

static void file_header_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "File Header Type",
"This list shows the output header types that\n\
Snd is willing to write.\n\
");
}

static void file_data_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "File Data Format",
"This list shows the data format choices\n\
available with the currently selected header\n\
choice.  'Short' means 16-bit two's complement\n\
integer.  'Mulaw' and 'Alaw' are common 8-bit\n\
compression schemes. 'Long' is 32-bit integer.\n\
'Float' is 32-bit float.  In each case, the\n\
decision as to byte order ('endianess') depends\n\
on the header type.\n\
");
}

static void file_data_location_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "File Data Location",
"If you know the data location (in bytes)\n\
you can set it in this field.\n\
");
}

static void file_srate_help_callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,
	   "File Srate",
"This field sets the nominal file sampling\n\
rate.\n\
");
} 

static void file_chans_help_callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,
	   "File Srate",
"This field sets the number of channels\n\
in the output file.\n\
");
} 

static void file_comment_label_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  snd_help(ss,
	   "Output File Comment",
"This optional field provides any comments\n\
that you want saved in the output file's\n\
header.\n\
");	   
}

static char *srate_completer(char *text)
{
  set_completion_matches(1);
  while ((text) && (*text == ' ')) text++;
  if (strcmp(text,"4410") == 0) return(copy_string("44100"));
  if (strcmp(text,"441") == 0) return(copy_string("44100"));
  if (strcmp(text,"44") == 0) return(copy_string("44100"));
  if (strcmp(text,"2205") == 0) return(copy_string("22050"));
  if (strcmp(text,"220") == 0) return(copy_string("22050"));
  if (strcmp(text,"22") == 0) return(copy_string("22050"));
  if (strcmp(text,"2") == 0) return(copy_string("22050"));
  if (strcmp(text,"4800") == 0) return(copy_string("48000"));
  if (strcmp(text,"480") == 0) return(copy_string("48000"));
  if (strcmp(text,"48") == 0) return(copy_string("48000"));
  if (strcmp(text,"800") == 0) return(copy_string("8000"));
  if (strcmp(text,"80") == 0) return(copy_string("8000"));
  if (strcmp(text,"8") == 0) return(copy_string("8000"));
  set_completion_matches(0);
  return(copy_string(text));
}

Widget *sndCreateFileDataForm(snd_state *ss, Widget parent, char *name, Arg *in_args, int in_n, int with_chan, int header_type, int data_format, int with_loc)
{
  Widget *arr;
  Arg args[32];
  int n;
  XmString *strs;
  int dformats = 0;
  char **formats = NULL;
  int *dfs = NULL;
  int i,header_pos= 0,data_pos = 0;
  switch (header_type)
    {
    case NeXT_sound_file: dformats = NUM_NEXT_FORMATS; formats = next_data_formats; dfs = next_dfs; header_pos = NEXT_POSITION; break;
    case NIST_sound_file: dformats = NUM_NIST_FORMATS; formats = nist_data_formats; dfs = nist_dfs; header_pos = NIST_POSITION; break;
    case AIFF_sound_file: dformats = NUM_AIFC_FORMATS; formats = aifc_data_formats; header_pos = AIFC_POSITION; dfs = aifc_dfs; break;
    case RIFF_sound_file: dformats = NUM_WAVE_FORMATS; formats = wave_data_formats; header_pos = RIFF_POSITION; dfs = wave_dfs; break;
    case IRCAM_sound_file: dformats = NUM_IRCAM_FORMATS; formats = ircam_data_formats; header_pos = IRCAM_POSITION; dfs = ircam_dfs; break;
    case raw_sound_file: dformats = NUM_DATA_FORMATS; formats = data_formats; header_pos = RAW_POSITION; dfs = NULL; data_pos = data_format; break;
    case old_style_AIFF_sound_file: dformats = NUM_AIFF_FORMATS; formats = aiff_data_formats; header_pos = AIFF_POSITION; dfs = aiff_dfs; break;
    }
  if (header_type != raw_sound_file)
    {
      data_pos = 1;
      for (i=0;i<dformats;i++) if (data_format == dfs[i]) {data_pos = i+1; break;}
    }

  arr = (Widget *)CALLOC(FILE_DATA_WIDGETS,sizeof(Widget));
  arr[fdata_out_form] = sndCreateFormWidget(name,parent,in_args,in_n);

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  arr[fdata_form] = sndCreateFormWidget(name,arr[fdata_out_form],args,n);

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNorientation,XmVERTICAL); n++;
  XtSetArg(args[n],XmNwidth,5); n++;
  XtSetArg(args[n],XmNseparatorType,XmNO_LINE); n++;
  arr[fdata_sep1] = XtCreateManagedWidget("sep1",xmSeparatorWidgetClass,arr[fdata_form],args,n);

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,arr[fdata_sep1]); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  arr[fdata_hlab] = XtCreateManagedWidget(STR_header,xmLabelWidgetClass,arr[fdata_form],args,n);
  XtAddCallback(arr[fdata_hlab],XmNhelpCallback,file_header_help_callback,ss);

  n=0;
#ifdef LESSTIF_VERSION
  if (!(ss->using_schemes)) n = background_white_color(args,n,ss);
#endif
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,arr[fdata_hlab]); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,arr[fdata_sep1]); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNlistMarginWidth,1); n++;
  /* what is selected depends on current type */
  strs = (XmString *)CALLOC(NUM_HEADER_TYPES,sizeof(XmString)); 
  strs[NEXT_POSITION-1] = XmStringCreate("sun  ",XmFONTLIST_DEFAULT_TAG);
  strs[AIFC_POSITION-1] = XmStringCreate("aifc ",XmFONTLIST_DEFAULT_TAG);
  strs[RIFF_POSITION-1] = XmStringCreate("wave ",XmFONTLIST_DEFAULT_TAG);
  strs[RAW_POSITION-1] = XmStringCreate("raw  ",XmFONTLIST_DEFAULT_TAG);
  strs[IRCAM_POSITION-1] = XmStringCreate("ircam",XmFONTLIST_DEFAULT_TAG);
  strs[AIFF_POSITION-1] = XmStringCreate("aiff ",XmFONTLIST_DEFAULT_TAG);
  strs[NIST_POSITION-1] = XmStringCreate("nist ",XmFONTLIST_DEFAULT_TAG);
  XtSetArg(args[n],XmNitems,strs); n++;
  XtSetArg(args[n],XmNitemCount,NUM_HEADER_TYPES); n++;
  XtSetArg(args[n],XmNvisibleItemCount,NUM_VISIBLE_HEADERS); n++;
  arr[fdata_hlist] = XmCreateScrolledList(arr[fdata_form],"header type",args,n);
  XtAddCallback(arr[fdata_hlist],XmNhelpCallback,file_header_help_callback,ss);
  XtManageChild(arr[fdata_hlist]);
  for (i=0;i<NUM_HEADER_TYPES;i++) XmStringFree(strs[i]);
  FREE(strs);

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,arr[fdata_hlist]); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNorientation,XmVERTICAL); n++;
  XtSetArg(args[n],XmNwidth,15); n++;
  XtSetArg(args[n],XmNseparatorType,XmNO_LINE); n++;
  arr[fdata_sep2] = XtCreateManagedWidget("sep2",xmSeparatorWidgetClass,arr[fdata_form],args,n);

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,arr[fdata_sep2]); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  arr[fdata_dlab] = XtCreateManagedWidget(STR_data,xmLabelWidgetClass,arr[fdata_form],args,n);
  XtAddCallback(arr[fdata_dlab],XmNhelpCallback,file_data_help_callback,ss);

  n=0;
#ifdef LESSTIF_VERSION
  if (!(ss->using_schemes)) n = background_white_color(args,n,ss);
#endif
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,arr[fdata_dlab]); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,arr[fdata_sep2]); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  arr[fdata_dlist] = XmCreateScrolledList(arr[fdata_form],"data format",args,n);
  /* what is displayed and selected depends on current type */
  XtAddCallback(arr[fdata_dlist],XmNhelpCallback,file_data_help_callback,ss);

  XmListSelectPos(arr[fdata_hlist],header_pos,FALSE);
  strs = (XmString *)CALLOC(dformats,sizeof(XmString)); 
  for (i=0;i<dformats;i++) strs[i] = XmStringCreate(formats[i],XmFONTLIST_DEFAULT_TAG);
  XtVaSetValues(arr[fdata_dlist],XmNitems,strs,XmNitemCount,dformats,NULL);
  for (i=0;i<dformats;i++) XmStringFree(strs[i]);
  FREE(strs);
  XmListSelectPos(arr[fdata_dlist],data_pos,FALSE);
  XtManageChild(arr[fdata_dlist]);

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
#ifndef LESSTIF_VERSION
  XtSetArg(args[n],XmNleftWidget,arr[fdata_dlist]); n++;
#else
  XtSetArg(args[n],XmNleftWidget,XtParent(arr[fdata_dlist])); n++;
#endif
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNorientation,XmVERTICAL); n++;
  XtSetArg(args[n],XmNwidth,15); n++;
  XtSetArg(args[n],XmNseparatorType,XmNO_LINE); n++;
  arr[fdata_sep3] = XtCreateManagedWidget("sep3",xmSeparatorWidgetClass,arr[fdata_form],args,n);

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,arr[fdata_sep3]); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  arr[fdata_slab] = XtCreateManagedWidget(STR_srate_p,xmLabelWidgetClass,arr[fdata_form],args,n);
  XtAddCallback(arr[fdata_slab],XmNhelpCallback,file_srate_help_callback,ss);

  n=0;
  XtSetArg(args[n],XmNcolumns,6); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,arr[fdata_slab]); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,arr[fdata_sep3]); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
  arr[fdata_stext] = sndCreateTextFieldWidget(ss,"text",arr[fdata_form],args,n,NOT_ACTIVATABLE,add_completer_func(srate_completer));
  XtAddCallback(arr[fdata_stext],XmNhelpCallback,file_srate_help_callback,ss);

  if (with_chan)
    {
      n=0;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,arr[fdata_stext]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,arr[fdata_sep3]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      arr[fdata_clab] = XtCreateManagedWidget(STR_chans_p,xmLabelWidgetClass,arr[fdata_form],args,n);
      XtAddCallback(arr[fdata_clab],XmNhelpCallback,file_chans_help_callback,ss);

      n=0;
      XtSetArg(args[n],XmNcolumns,6); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,arr[fdata_clab]); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,arr[fdata_sep3]); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
      arr[fdata_ctext] = sndCreateTextFieldWidget(ss,"text",arr[fdata_form],args,n,NOT_ACTIVATABLE,NO_COMPLETER);
      XtAddCallback(arr[fdata_ctext],XmNhelpCallback,file_chans_help_callback,ss);

      if (with_loc)
	{
	  n=0;
	  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
	  XtSetArg(args[n],XmNtopWidget,arr[fdata_ctext]); n++;
	  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
	  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
	  XtSetArg(args[n],XmNleftWidget,arr[fdata_sep3]); n++;
	  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
	  arr[fdata_loclab] = XtCreateManagedWidget(STR_location_p,xmLabelWidgetClass,arr[fdata_form],args,n);
	  XtAddCallback(arr[fdata_loclab],XmNhelpCallback,file_data_location_help_callback,ss);

	  n=0;
	  XtSetArg(args[n],XmNcolumns,6); n++;
	  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
	  XtSetArg(args[n],XmNtopWidget,arr[fdata_loclab]); n++;
	  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
	  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
	  XtSetArg(args[n],XmNleftWidget,arr[fdata_sep3]); n++;
	  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
	  XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;	
	  arr[fdata_loctext] = sndCreateTextFieldWidget(ss,"text",arr[fdata_form],args,n,NOT_ACTIVATABLE,NO_COMPLETER);
	  XtAddCallback(arr[fdata_loctext],XmNhelpCallback,file_data_location_help_callback,ss);
	}
    }

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,arr[fdata_form]); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
  XtSetArg(args[n],XmNheight,5); n++;
  XtSetArg(args[n],XmNseparatorType,XmNO_LINE); n++;
  arr[fdata_sep4] = XtCreateManagedWidget("sep4",xmSeparatorWidgetClass,arr[fdata_out_form],args,n);

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,arr[fdata_sep4]); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  arr[fdata_comment_label] = XtCreateManagedWidget(STR_comment_p,xmLabelWidgetClass,arr[fdata_out_form],args,n);
  XtAddCallback(arr[fdata_comment_label],XmNhelpCallback,file_comment_label_help_callback,ss);

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_OPPOSITE_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,arr[fdata_comment_label]); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,arr[fdata_comment_label]); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrows,2); n++;
#if (HAVE_OSS || HAVE_ALSA)
  XtSetArg(args[n],XmNcolumns,40); n++;
  /* this pushes the right boundary over a ways -- otherwise the button holder box takes up all available space */
#endif
  arr[fdata_comment_text] = sndCreateTextWidget(ss,"comment",arr[fdata_out_form],args,n);
  XtAddCallback(arr[fdata_comment_text],XmNhelpCallback,file_comment_label_help_callback,ss);

  return(arr);
}

static void make_save_as_dialog(Widget w,save_as_info *fdw, char *name, file_info *hdr)
{
  Arg args[32];
  int n;
  Widget *wids;
  snd_state *ss;
  XmString xmstr1,xmstr2,s1;
  char *tmpstr = NULL;
  ss=fdw->state;

  n=0;
  if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
  s1 = XmStringCreate(STR_save_as_p,XmFONTLIST_DEFAULT_TAG);
  XtSetArg(args[n],XmNselectionLabelString,s1); n++;
  xmstr1=XmStringCreate(STR_Save,XmFONTLIST_DEFAULT_TAG);
  XtSetArg(args[n],XmNokLabelString,xmstr1); n++;
  sprintf(file_string,STR_saving,name);
  xmstr2 = XmStringCreate(file_string,XmFONTLIST_DEFAULT_TAG);
  XtSetArg(args[n],XmNdialogTitle,xmstr2); n++;
#if RESIZE_DIALOG
  XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
  XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
  XtSetArg(args[n],XmNautoUnmanage,FALSE); n++;
  XtSetArg(args[n],XmNchildPlacement,XmPLACE_ABOVE_SELECTION); n++;
  XtSetArg(args[n],XmNallowOverlap,FALSE); n++;
  fdw->dialog = XmCreateFileSelectionDialog(w,"save-as",args,n);
#if OVERRIDE_TOGGLE
  override_form_translation(fdw->dialog);
#endif

  XtManageChild(fdw->dialog);
  XmStringFree(s1);
  XmStringFree(xmstr1);
  XmStringFree(xmstr2);
#ifndef LESSTIF_VERSION
  fdw->text = XtNameToWidget(fdw->dialog,"Text");
  if (!(fdw->text)) fdw->text = XmFileSelectionBoxGetChild(fdw->dialog,XmDIALOG_TEXT);
#else
  fdw->text = XmFileSelectionBoxGetChild(fdw->dialog,XmDIALOG_TEXT);
#endif

  XtAddCallback(fdw->dialog,XmNhelpCallback,save_as_help_callback,ss);
  XtAddCallback(fdw->dialog,XmNcancelCallback,save_as_cancel_callback,fdw);
  XtAddCallback(fdw->dialog,XmNokCallback,save_as_ok_callback,fdw);

  n=0;
#ifdef LESSTIF_VERSION
  XtSetArg(args[n],XmNheight,100); n++;
#endif
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  wids = sndCreateFileDataForm(ss,fdw->dialog,"data-form",args,n,FALSE,NeXT_sound_file,SNDLIB_16_LINEAR,FALSE);
  fdw->header_list = wids[fdata_hlist];
  fdw->comtext = wids[fdata_comment_text];
  XtAddCallback(fdw->header_list,XmNbrowseSelectionCallback,Save_as_Header_type_Callback,fdw);
  fdw->data_list = wids[fdata_dlist];
  XtAddCallback(fdw->data_list,XmNbrowseSelectionCallback,Save_as_Data_format_Callback,fdw);
  fdw->srtext = wids[fdata_stext];
  sprintf(file_string,"%d",(hdr) ? hdr->srate : region_srate(0));
  XmTextSetString(fdw->srtext,file_string);
  if ((hdr) && (fdw->comtext)) 
    {
      XmTextSetString(fdw->comtext,tmpstr = sound_comment(hdr->name));
      if (tmpstr) FREE(tmpstr);
    }
  load_header_and_data_lists(fdw->header_list,fdw->data_list,
			     (hdr) ? hdr->type : NeXT_sound_file,(hdr) ? hdr->format : SNDLIB_16_LINEAR,
			     &(fdw->header_choice),&(fdw->data_choice));
  color_file_selection_box(fdw->dialog,ss);
  if (!(ss->using_schemes))	
    {
      XtVaSetValues(fdw->data_list,XmNbackground,(ss->sgx)->white,NULL);
      XtVaSetValues(fdw->header_list,XmNbackground,(ss->sgx)->white,NULL);
    }
}

void snd_save_as_dialog(Widget w, snd_state *ss)
{
  /* save old as new, close old, open new */
  snd_info *sp;
  XmString xmstr2;
  file_info *hdr;
  finish_keyboard_selection();
  sp = any_selected_sound(ss);
  if (!save_as_dialog)
    {
      save_as_dialog = (save_as_info *)CALLOC(1,sizeof(save_as_info));
      save_as_dialog->state = ss;
      make_save_as_dialog(w,save_as_dialog,sp->shortname,sp->hdr);
    }
  else
    {
      sprintf(file_string,STR_saving,sp->shortname);
      xmstr2 = XmStringCreate(file_string,XmFONTLIST_DEFAULT_TAG);
      XtVaSetValues(save_as_dialog->dialog,XmNtitleString,xmstr2,NULL);
      XmStringFree(xmstr2);
      hdr = sp->hdr;
      load_header_and_data_lists(save_as_dialog->header_list,save_as_dialog->data_list,
				 hdr->type,hdr->format,
				 &(save_as_dialog->header_choice),&(save_as_dialog->data_choice));
    }
  save_as_dialog->type = FILE_SAVE_AS;
  if (!XtIsManaged(save_as_dialog->dialog)) XtManageChild(save_as_dialog->dialog);
}


/* -------- files browser -------- */
/*
 * the region and file browsers share much widgetry -- they are supposed to look the same
 */

typedef struct {
  Widget rw,nm,pl,sv;
  int pos;
  snd_state *ss;
} regrow;

typedef struct {
  Widget ww;
  Widget list;
  Widget plw;
  Widget svw;
  Widget dbline;
  Widget bydate;
  Widget bysize;
  Widget byname;
  Widget byentry;
  Widget panes,toppane;
} ww_info;

typedef struct {
  Widget dialog;
  Widget curlst;
  Widget prevlst;
  Widget curww; /* work area of scrolled window */
  Widget prevww;
  int selected_file;
  snd_state *state;
} fb_info;

static fb_info *fbd = NULL;

static int prevfile_size = 0;
static int curfile_size = 0;
static char **curnames = NULL;
static char **prevnames = NULL;
static char **prevfullnames = NULL;
static int *a_big_star = NULL;
static int *prevtimes = NULL;
static int curfile_end = 0;
static int prevfile_end = -1;
static int max_curfile_end = 0;
static int max_prevfile_end = -1;
static int prevtime = 0;

static void make_curfiles_list (snd_state *ss);
static void make_prevfiles_list (snd_state *ss);  /* these are needed as local forward declarations */

static regrow **cur_name_row = NULL;
static regrow **prev_name_row = NULL;

#define WITHOUT_PANED_WINDOW 0
#define WITH_PANED_WINDOW 1
#define DONT_PAD_TITLE 0
#define PAD_TITLE_ON_RIGHT 1
#define PAD_TITLE_ON_LEFT 2
#define WITHOUT_SORT_BUTTON 0
#define WITH_SORT_BUTTON 1

static ww_info *make_title_row(snd_state *ss, Widget formw, char *first_str, char *second_str, char *main_str, int pad, int with_sort, int with_pane)
{
  int n;
  Arg args[32];
  Widget plw,svw,rlw,sep1;
  Widget smenu,scascade,sbar;
  ww_info *wwi;

  wwi = (ww_info *)CALLOC(1,sizeof(ww_info));
  
  n=0;
  if (!(ss->using_schemes)) n = background_highlight_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNalignment,XmALIGNMENT_CENTER); n++;	
  rlw = XtCreateManagedWidget(main_str,xmLabelWidgetClass,formw,args,n);
      
  n=0;
  if (!(ss->using_schemes)) n = background_white_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,rlw); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
  XtSetArg(args[n],XmNseparatorType,XmDOUBLE_LINE); n++;
  sep1 = XtCreateManagedWidget("sep",xmSeparatorWidgetClass,formw,args,n);
  wwi->dbline = sep1;

  if (with_pane == WITH_PANED_WINDOW)
    {
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sep1); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNallowResize,TRUE); n++;
      wwi->panes = sndCreatePanedWindowWidget("panes",formw,args,n);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNpaneMinimum,40); n++;
      wwi->toppane = sndCreateFormWidget("toppane",wwi->panes,args,n);
      formw = wwi->toppane;
    }
  else 
    {
      wwi->panes = formw;
      wwi->toppane = formw;
    }

  n=0;
  if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
  if (pad == PAD_TITLE_ON_LEFT)
    {
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNleftPosition,5); n++;
    }
  else
    {
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
    }
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  if (with_pane == WITH_PANED_WINDOW)
    {
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
    }
  else
    {
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sep1); n++;
    }
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNfontList,button_FONT(ss)); n++;
  svw = XtCreateManagedWidget(first_str,xmLabelWidgetClass,formw,args,n);
  make_button_label(svw,first_str);
  wwi->svw = svw;

  n=0;
  if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,svw); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  if (with_pane == WITH_PANED_WINDOW)
    {
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_NONE); n++;
    }
  else
    {
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sep1); n++;
    }
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNfontList,button_FONT(ss)); n++;
  plw = XtCreateManagedWidget(second_str,xmLabelWidgetClass,formw,args,n);
  make_button_label(plw,second_str);
  wwi->plw = plw;

  if (with_sort == WITH_SORT_BUTTON)
    {
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sep1); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNshadowThickness,0); n++;
      XtSetArg(args[n],XmNhighlightThickness,0); n++;
      XtSetArg(args[n],XmNmarginHeight,0); n++;
      sbar = XmCreateMenuBar(formw,"menuBar",args,n);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      smenu = XmCreatePulldownMenu(sbar,STR_sort,args,n);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNsubMenuId,smenu); n++;
      XtSetArg(args[n],XmNfontList,button_FONT(ss)); n++;
      XtSetArg(args[n],XmNshadowThickness,0); n++;
      XtSetArg(args[n],XmNhighlightThickness,0); n++;
      XtSetArg(args[n],XmNmarginHeight,1); n++;
      scascade = XtCreateManagedWidget(STR_sort,xmCascadeButtonWidgetClass,sbar,args,n);
      
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNfontList,button_FONT(ss)); n++;
      wwi->byname = XtCreateManagedWidget(STR_name,xmPushButtonWidgetClass,smenu,args,n);
      wwi->bydate = XtCreateManagedWidget(STR_date,xmPushButtonWidgetClass,smenu,args,n);
      wwi->bysize = XtCreateManagedWidget(STR_size,xmPushButtonWidgetClass,smenu,args,n);
      wwi->byentry = XtCreateManagedWidget(STR_entry,xmPushButtonWidgetClass,smenu,args,n);

      XtManageChild(sbar);
    }
  
  n=0;
  if (pad == PAD_TITLE_ON_LEFT) 
    {
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNleftPosition,5); n++;
    }
  else
    {
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
    }
  if (pad == PAD_TITLE_ON_RIGHT)
    {
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNrightPosition,95); n++;
    }
  else
    {
      if (pad == PAD_TITLE_ON_LEFT)
	{
	  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
	}
      else
	{
	  XtSetArg(args[n],XmNrightAttachment,XmATTACH_POSITION); n++;
	  XtSetArg(args[n],XmNrightPosition,70); n++;
	}
    }
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,svw); n++;
  if (pad == DONT_PAD_TITLE)
    {
      if (with_pane == WITH_PANED_WINDOW)
	{
	  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
	}
      else
	{
	  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_POSITION); n++;
	  XtSetArg(args[n],XmNbottomPosition,40); n++;
	}
    }
  else
    {
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
    }
  XtSetArg(args[n],XmNscrollingPolicy,XmAUTOMATIC); n++;
  XtSetArg(args[n],XmNscrollBarDisplayPolicy,XmSTATIC); n++;
  wwi->list = XmCreateScrolledWindow(formw,"reglist",args,n);

  n=0;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  wwi->ww = sndCreateFormWidget("ww",wwi->list,args,n);
  XtVaSetValues(wwi->list,XmNworkWindow,wwi->ww,NULL);
  
  return(wwi);
}

static regrow *make_regrow(snd_state *ss, Widget ww, Widget last_row, 
			   XtCallbackProc first_callback, XtCallbackProc second_callback, XtCallbackProc third_callback)
{
  int n;
  Arg args[32];
  regrow *r;
  XmString s1;
  s1 = XmStringCreate("",XmFONTLIST_DEFAULT_TAG);
  r = (regrow *)CALLOC(1,sizeof(regrow));

  n=0;
  if (!(ss->using_schemes)) n = background_highlight_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,(last_row) ? XmATTACH_WIDGET : XmATTACH_FORM); n++;
  if (last_row) {XtSetArg(args[n],XmNtopWidget,last_row); n++;}
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
#if defined(LINUX) || defined(SCO5) || defined(UW2) || defined(SUN)
  XtSetArg(args[n],XmNheight,17); n++; 
#else
  XtSetArg(args[n],XmNheight,20); n++; 
#endif
  r->rw = XtCreateWidget("rw",xmFormWidgetClass,ww,args,n);

  n=0;
  if (!(ss->using_schemes)) n = background_highlight_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  if (!(ss->using_schemes)) {XtSetArg(args[n],XmNselectColor,(ss->sgx)->pushed_button_color); n++;}
  XtSetArg(args[n],XmNlabelString,s1); n++;
  XtSetArg(args[n],XmNvalueChangedCallback,make_callback_list(first_callback,(XtPointer)r)); n++;
  if (ss->toggle_size > 0) {XtSetArg(args[n],XmNindicatorSize,ss->toggle_size); n++;}
  r->sv = sndCreateToggleButtonWidget("sv",r->rw,args,n);

  n=0;
  if (!(ss->using_schemes)) n = background_highlight_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,r->sv); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  if (!(ss->using_schemes)) {XtSetArg(args[n],XmNselectColor,(ss->sgx)->pushed_button_color); n++;}
  XtSetArg(args[n],XmNlabelString,s1); n++;
  XtSetArg(args[n],XmNvalueChangedCallback,make_callback_list(second_callback,(XtPointer)r)); n++;
  if (ss->toggle_size > 0) {XtSetArg(args[n],XmNindicatorSize,ss->toggle_size); n++;}
  r->pl = sndCreateToggleButtonWidget("pl",r->rw,args,n);

  n=0;
  if (!(ss->using_schemes)) n = background_highlight_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,r->pl); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNshadowThickness,0); n++;
  XtSetArg(args[n],XmNhighlightThickness,0); n++;
  XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;
  XtSetArg(args[n],XmNfillOnArm,FALSE); n++;
  XtSetArg(args[n],XmNfontList,bold_button_FONT(ss)); n++;
  XtSetArg(args[n],XmNrecomputeSize,FALSE); n++;
  XtSetArg(args[n],XmNwidth,300); n++;
  XtSetArg(args[n],XmNactivateCallback,make_callback_list(third_callback,(XtPointer)r)); n++;
  r->nm = XtCreateManagedWidget("nm",xmPushButtonWidgetClass,r->rw,args,n);
  XmStringFree(s1);
  return(r);
}

static void make_row_name(regrow *r, char *name, int big_star)
{
  if (big_star)
    {
      sprintf(file_string,"%s*",name);
      make_bold_button_label(r->nm,file_string);
    }
  else make_bold_button_label(r->nm,name);
}

static void fill_in_row(regrow *r, char *name, int big_star)
{
  XtUnmanageChild(r->rw);
  make_row_name(r,name,big_star);
  XmToggleButtonSetState(r->sv,FALSE,FALSE);
  XmToggleButtonSetState(r->pl,FALSE,FALSE);
  XtManageChild(r->rw);
}

static int find_curfile_regrow(char *shortname)
{
  int i;
  for (i=0;i<curfile_end;i++)
    {
      if (strcmp(curnames[i],shortname) == 0) return(i);
    }
  return(-1);
}

static int find_prevfile_regrow(char *shortname)
{
  int i;
  for (i=0;i<=prevfile_end;i++)
    {
      if (strcmp(prevnames[i],shortname) == 0) return(i);
    }
  return(-1);
}

void save_prevlist(FILE *fd)
{
  int i;
  for (i=0;i<=prevfile_end;i++)
    fprintf(fd,"(%s \"%s\")\n",S_preload_file,prevfullnames[i]);
}

static void file_uncurlist(char *filename)
{
  int i,j;
  i = find_curfile_regrow(filename);
  if (i != -1)
    {
      if (curnames[i]) FREE(curnames[i]);
      curnames[i] = NULL;
      for (j=i;j<curfile_end-1;j++)
	{
	  curnames[j] = curnames[j+1];
	  a_big_star[j] = a_big_star[j+1];
	}
      curnames[curfile_end-1] = NULL;
      curfile_end--;
    }
}

static void file_unprevlist(char *filename)
{
  int i,j;
  i = find_prevfile_regrow(filename);
  if (i != -1)
    {
      FREE(prevnames[i]);
      prevnames[i] = NULL;
      FREE(prevfullnames[i]);
      prevfullnames[i] = NULL;
      for (j=i;j<prevfile_end;j++) 
	{
	  prevnames[j] = prevnames[j+1];
	  prevfullnames[j] = prevfullnames[j+1]; 
	  prevtimes[j] = prevtimes[j+1];
	}
      prevnames[prevfile_end] = NULL; 
      prevfile_end--;
    }
}

static void clear_prevlist(void)
{
  int i;
  for (i=0;i<=prevfile_end;i++)
    {
      if (prevnames[i]) 
	{
	  FREE(prevnames[i]); prevnames[i]=NULL;
	  FREE(prevfullnames[i]); prevfullnames[i]=NULL;
	}
    }
  prevfile_end = -1;
  prevtime = 0;
}

static void update_prevlist(void)
{
  /* here we need the file's full name */
  int i,j,fd;
  for (i=0;i<=prevfile_end;i++)
    {
      if (prevnames[i]) 
	{
	  fd = open(prevfullnames[i],O_RDONLY,0);
	  if (fd == -1) 
	    {
	      FREE(prevnames[i]); prevnames[i]=NULL;
	      FREE(prevfullnames[i]); prevfullnames[i]=NULL;
	    }
	  else close(fd);
	}
    }
  for (i=0,j=0;i<=prevfile_end;i++)
    {
      if (prevnames[i])
	{
	  if (i != j) 
	    {
	      prevnames[j] = prevnames[i]; 
	      prevfullnames[j] = prevfullnames[i];
	      prevtimes[j] = prevtimes[i];
	    }
	  j++;
	}
    }
  prevfile_end = j-1;
}

static void file_curlist(char *filename)
{
  int i,new_size;
  if (curfile_end == curfile_size)
    {
      new_size = curfile_size+32;
      if (curfile_size == 0)
	{
	  curnames = (char **)CALLOC(new_size,sizeof(char *));
	  cur_name_row = (regrow **)CALLOC(new_size,sizeof(regrow *));
	  a_big_star = (int *)CALLOC(new_size,sizeof(int *));
	}
      else
	{
	  curnames = (char **)REALLOC(curnames,new_size * sizeof(char *));
	  cur_name_row = (regrow **)REALLOC(cur_name_row,new_size * sizeof(regrow *));
	  a_big_star = (int *)REALLOC(a_big_star,new_size * sizeof(int *));
	  for (i=curfile_size;i<new_size;i++) {curnames[i] = NULL; a_big_star[i] = 0; cur_name_row[i] = NULL;}
	}
      curfile_size = new_size;
    }
  curnames[curfile_end] = copy_string(filename);
  a_big_star[curfile_end] = 0;
  curfile_end++;
  if (max_curfile_end < curfile_end) max_curfile_end = curfile_end;
}

static void file_prevlist(char *filename, char *fullname)
{
  int k,i,new_size;
  prevfile_end++;
  if (prevfile_end == prevfile_size)
    {
      new_size = prevfile_size+32;
      if (prevfile_size == 0)
	{
	  prevnames = (char **)CALLOC(new_size,sizeof(char *));
	  prevfullnames = (char **)CALLOC(new_size,sizeof(char *));
	  prevtimes = (int *)CALLOC(new_size,sizeof(char *));
	  prev_name_row = (regrow **)CALLOC(new_size,sizeof(regrow *));
	}
      else
	{
	  prevnames = (char **)REALLOC(prevnames,new_size * sizeof(char *));
	  prevfullnames = (char **)REALLOC(prevfullnames,new_size * sizeof(char *));
	  prevtimes = (int *)REALLOC(prevtimes,new_size * sizeof(int *));
	  prev_name_row = (regrow **)REALLOC(prev_name_row, new_size * sizeof(regrow *));
	  for (i=prevfile_size;i<new_size;i++) {prevnames[i] = NULL; prevfullnames[i] = NULL; prevtimes[i] = 0; prev_name_row[i] = NULL;}
	}
      prevfile_size = new_size;
    }
  for (k=prevfile_end;k>0;k--) 
    {
      prevnames[k]=prevnames[k-1]; 
      prevfullnames[k]=prevfullnames[k-1];
      prevtimes[k] = prevtimes[k-1];
    }
  prevnames[0] = copy_string(filename);
  prevfullnames[0] = copy_string(fullname);
  prevtimes[0] = prevtime++;
  if (max_prevfile_end < prevfile_end) max_prevfile_end = prevfile_end;
}

static void add_files_to_prevlist(snd_state *ss, char **shortnames, char **longnames, int len)
{
  int i;
  for (i=0;i<len;i++) file_prevlist(shortnames[i],longnames[i]);
  if ((fbd) && (XtIsManaged(fbd->dialog))) make_prevfiles_list(ss);
}

void update_prevfiles(snd_state *ss)
{
  if ((fbd) && (XtIsManaged(fbd->dialog))) make_prevfiles_list(ss);
}

void add_directory_to_prevlist(snd_state *ss, char *dirname)
{
  dir *sound_files = NULL;
  char *fullpathname = NULL;
  char **fullnames;
  int i,end;
  sound_files = find_sound_files_in_dir(dirname);
  if ((sound_files) && (sound_files->len > 0))
    {
      fullpathname = (char *)CALLOC(FILENAME_MAX,sizeof(char));
      strcpy(fullpathname,dirname);
      if (dirname[strlen(dirname)-1] != '/') strcat(fullpathname,"/");
      end = strlen(fullpathname);
      fullnames = (char **)CALLOC(sound_files->len,sizeof(char *));
      for (i=0;i<sound_files->len;i++) 
	{
	  fullnames[i] = copy_string(strcat(fullpathname,sound_files->files[i]));
	  fullpathname[end] = '\0';
	}
      add_files_to_prevlist(ss,sound_files->files,fullnames,sound_files->len);
      for (i=0;i<sound_files->len;i++) {FREE(fullnames[i]); fullnames[i]=NULL;}
      FREE(fullnames);
      free_dir(sound_files);
      FREE(fullpathname);
    }
  if ((fbd) && (XtIsManaged(fbd->dialog))) make_prevfiles_list(ss);
}

void remember_me(snd_state *ss, char *shortname,char *fullname)
{
  file_prevlist(shortname,fullname);
  file_uncurlist(shortname);
  if ((fbd) && (XtIsManaged(fbd->dialog)))
    {
      make_curfiles_list(ss);
      make_prevfiles_list(ss);
    }
}

void greet_me(snd_state *ss, char *shortname)
{
  file_curlist(shortname);
  file_unprevlist(shortname);
  if ((fbd) && (XtIsManaged(fbd->dialog)))
    {
      make_curfiles_list(ss);
      make_prevfiles_list(ss);
    }
}

void make_a_big_star_outa_me(snd_state *ss, char *shortname, int big_star)
{
  int i;
  regrow *r;
  i = find_curfile_regrow(shortname);
  if ((i != -1) && (a_big_star[i] != big_star))
    {
      if ((fbd) && (XtIsManaged(fbd->dialog)))
	{
	  r = cur_name_row[i];
	  make_row_name(r,curnames[i],big_star);
	}
      a_big_star[i] = big_star;
    }
}

static void View_Files_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
       "File Browser",
"This dialog provides two lists, one of\n\
the currently active files in Snd, and\n\
the other of previously active files.\n\
The save button saves current edits, the\n\
play button plays the file, and the\n\
unlist button removes a file from the\n\
previous files list.  If a file is deleted\n\
while Snd is running, it will not notice\n\
its deletion automatically.  To update the\n\
previous files list to account for such\n\
actions, click on the 'update' button.\n\
To reopen a previous file, simply select\n\
that file from the previous files list.\n\
To remove all files from the previous files\n\
list, click the 'clear' button.  To select\n\
one of the current files in Snd, opening\n\
its associated windows, select it in the\n\
current files list.\n\
\n\
To preload all the sound files in a directory\n\
into the previous files list, use either the\n\
command (" S_preload_directory " dir), as in\n\
\n\
  M-x (" S_preload_directory " \"/usr/people/bil/hdr)\"\n\
\n\
or give the directory name to the -p flag\n\
when starting Snd:\n\
\n\
  snd -p . oboe.snd\n\
\n\
To preload a specific file,\n\
\n\
  (" S_preload_file " <name>)\n\
\n\
The 'sort' label on the right activates\n\
a menu of sorting choices; 'name' sorts the\n\
previous files list alphabetically, 'date'\n\
sorts by date written, 'size' sorts by the\n\
number of samples in the sound, and 'entry'\n\
sorts by the order the sound appears in the\n\
absence of explicit sorting.  The variable\n\
" S_previous_files_sort " (default 0:\n\
unsorted) refers to this menu.\n\
");	   
}

static void View_Files_Dismiss_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  fb_info *fd = (fb_info *)clientData;
  XtUnmanageChild(fd->dialog);
}

static void View_Files_Clear_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* clear previous files list and associated widget list */
  clear_prevlist();
  make_prevfiles_list((snd_state *)clientData);
}

static void View_Files_Update_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* run through previous files list looking for any that have been deleted behind our back */
  update_prevlist();
  make_prevfiles_list((snd_state *)clientData);
}

static void View_CurFiles_Save_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  regrow *r = (regrow *)clientData;
  snd_info *sp;
  sp = find_sound(r->ss,curnames[r->pos]);
  if (sp)
    {
      finish_keyboard_selection();
      save_edits(sp,NULL);
    }
  XmToggleButtonSetState(r->sv,FALSE,FALSE);
}

void set_file_browser_play_button(char *name, int state)
{
  int i,list;
  regrow *r;
  list = 0;
  if ((fbd) && (XtIsManaged(fbd->dialog)))
    {
      i=find_curfile_regrow(name); 
      if (i != -1) list = 1; else i=find_prevfile_regrow(name);
      if (i != -1)
	{
	  if (list) r=cur_name_row[i]; else r=prev_name_row[i];
	  XmToggleButtonSetState(r->pl,state,FALSE);
	}
    }
}

static void View_CurFiles_Play_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  regrow *r = (regrow *)clientData;
  XmToggleButtonCallbackStruct *cb = (XmToggleButtonCallbackStruct *)callData;
  snd_info *sp;
  sp = find_sound(r->ss,curnames[r->pos]);
  if (sp)
    {
      if (sp->playing) stop_playing_sound(sp);
      if (cb->set)
	{
	  start_playing(sp,0);
	  set_play_button(sp,1);
	}
      else set_play_button(sp,0);
    }
}

static void curfile_unhighlight(snd_state *ss)
{
  regrow *r;
  if ((fbd) && (XtIsManaged(fbd->dialog)))
    {
      if ((!(ss->using_schemes)) && (fbd->selected_file != -1))
	{
	  r = cur_name_row[fbd->selected_file];
	  XtVaSetValues(r->rw,XmNbackground,(ss->sgx)->highlight_color,NULL);
	  XtVaSetValues(r->nm,XmNbackground,(ss->sgx)->highlight_color,NULL);
	  fbd->selected_file = -1;
	}
    }
}

static void curfile_highlight(snd_state *ss, int i)
{
  regrow *r;
  if ((fbd) && (XtIsManaged(fbd->dialog)))
    {
      if (!(ss->using_schemes)) 
	{
	  if (fbd->selected_file != -1) curfile_unhighlight(ss);
	  r = cur_name_row[i];
	  XtVaSetValues(r->rw,XmNbackground,(ss->sgx)->zoom_color,NULL);
	  XtVaSetValues(r->nm,XmNbackground,(ss->sgx)->zoom_color,NULL);
	  fbd->selected_file = i;
	}
    }
}

static void View_CurFiles_Select_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  regrow *r = (regrow *)clientData;
  snd_info *sp,*osp;
  snd_state *ss;
  ss = r->ss;
  curfile_highlight(ss,r->pos);
  sp = find_sound(ss,curnames[r->pos]);
  osp = any_selected_sound(ss);
  if (sp != osp)
    {
      select_channel(sp,0);
      normalize_sound(ss,sp,osp,sp->chans[0]);
      /* goto_graph(sp->chans[0]); */
    }
}

static void View_PrevFiles_Unlist_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  regrow *r = (regrow *)clientData;
  file_unprevlist(prevnames[r->pos]);
  make_prevfiles_list(r->ss);
}

static void View_PrevFiles_Play_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* open and play -- close at end or when button off toggled */
  regrow *r = (regrow *)clientData;
  XmToggleButtonCallbackStruct *cb = (XmToggleButtonCallbackStruct *)callData;
  static snd_info *play_sp;
  int play;
  play = cb->set;
  if (play)
    {
      if (play_sp)
	{
	  if (play_sp->playing) {XmToggleButtonSetState(w,FALSE,FALSE); return;} /* can't play two of these at once */
	  if (strcmp(play_sp->shortname,prevnames[r->pos]) != 0)
	    {
	      completely_free_snd_info(play_sp);
	      play_sp = NULL;
	    }
	}
      if (!play_sp) play_sp = make_sound_readable(r->ss,prevfullnames[r->pos],FALSE);
      if (play_sp)
	{
	  play_sp->shortname = prevnames[r->pos];
	  play_sp->fullname = NULL;
	  start_playing(play_sp,0);
	}
      else
	{ /* can't find or setup file */
	  XmToggleButtonSetState(w,FALSE,FALSE);
	  play = 0;
	}
    }
  else
    { /* play toggled off */
      if ((play_sp) && (play_sp->playing)) stop_playing_sound(play_sp);
    }
}

static void View_PrevFiles_Select_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* open and set as selected */
  regrow *r = (regrow *)clientData;
  snd_info *sp;
  sp = snd_open_file(prevfullnames[r->pos],r->ss);
  if (sp) select_channel(sp,0); 
}

void highlight_selected_sound(snd_state *ss)
{
  snd_info *sp;
  int i;
  sp = selected_sound(ss);
  if (sp)
    {
      i = find_curfile_regrow(sp->shortname);
      if (i != -1) curfile_highlight(ss,i); else curfile_unhighlight(ss);
    }
  else curfile_unhighlight(ss);
}

static void make_curfiles_list (snd_state *ss)
{
  int i;
  Widget last_row = NULL;
  regrow *r;
  for (i=0;i<curfile_end;i++)
    {
      r = cur_name_row[i];
      if (r == NULL)
	{
	  r = make_regrow(ss,fbd->curww,last_row,View_CurFiles_Save_Callback,View_CurFiles_Play_Callback,View_CurFiles_Select_Callback);
	  cur_name_row[i] = r;
	  r->pos = i;
	  r->ss = ss;
	}
      fill_in_row(r,curnames[i],a_big_star[i]);
      last_row = r->rw;
    }
  for (i=curfile_end;i<max_curfile_end;i++)
    {
      if ((r = cur_name_row[i]))
	{
	  if (XtIsManaged(r->rw)) XtUnmanageChild(r->rw);
	}
    }
  max_curfile_end = curfile_end;
  highlight_selected_sound(ss);
  XtManageChild(fbd->curlst);
}

/* sort prevfiles list by name (aphabetical), or some number (date written, size, entry order, srate? type?) */
enum {ALPHABET,VALS_GREATER,VALS_LESS};

static void heapsort(unsigned int n, int choice, int *vals, char **a1, char **a2, int *times)
{
  /* sort a1 or vals, parallel sort in a2, borrowed from Numerical Recipes (zero-based) */
  unsigned int i,j,ir,k,pt;
  char *s1,*s2;
  int s3,val;
  if (n<2) return;
  k=(n>>1)+1;
  ir=n;
  for (;;)
    {
      if (k>1)
	{
	  k--;
	  s1 = a1[k-1];
	  s2 = a2[k-1];
	  s3 = vals[k-1];
	  pt = times[k-1];
	}
      else
	{
	  s1 = a1[ir-1];
	  s2 = a2[ir-1];
	  s3 = vals[ir-1];
	  pt = times[ir-1];
	  a1[ir-1] = a1[0];
	  a2[ir-1] = a2[0];
	  vals[ir-1] = vals[0];
	  times[ir-1] = times[0];
	  ir--;
	  if (ir == 1)
	    {
	      a1[0] = s1;
	      a2[0] = s2;
	      vals[0] = s3;
	      times[0] = pt;
	      break;
	    }
	}
      i=k;
      j=k+1;
      while (j<=ir)
	{
	  if (j<ir) 
	    {
	      switch (choice)
		{
		case ALPHABET: if (strcmp(a1[j-1],a1[j]) < 0) j++; break;
		case VALS_GREATER: if (vals[j-1] < vals[j]) j++; break;
		case VALS_LESS: if (vals[j-1] > vals[j]) j++; break;
		}
	    }
	  val = 0;
	  switch (choice)
	    {
	    case ALPHABET: val = (strcmp(s1,a1[j-1]) < 0); break;
	    case VALS_GREATER: val = (s3 < vals[j-1]); break;
	    case VALS_LESS: val = (s3 > vals[j-1]); break;
	    }
	  if (val)
	    {
	      a1[i-1] = a1[j-1];
	      a2[i-1] = a2[j-1];
	      vals[i-1] = vals[j-1];
	      times[i-1] = times[j-1];
	      i=j;
	      j<<=1;
	    }
	  else j=ir+1;
	}
      a1[i-1]=s1;
      a2[i-1]=s2;
      vals[i-1] = s3;
      times[i-1] = pt;
    }
}

static void sort_prevfiles_by_name(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  set_previous_files_sort(ss,1);
  make_prevfiles_list(ss);
}

static void sort_prevfiles_by_date(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  set_previous_files_sort(ss,2);
  make_prevfiles_list(ss);
}

static void sort_prevfiles_by_size(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  set_previous_files_sort(ss,3);
  make_prevfiles_list(ss);
}

static void sort_prevfiles_by_entry_order(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  set_previous_files_sort(ss,4);
  make_prevfiles_list(ss);
}

static void make_prevfiles_list (snd_state *ss)
{
  int i;
  Widget last_row = NULL;
  regrow *r;
  int *vals;
  if (prevfile_end >= 0)
    {
      vals = (int *)CALLOC(prevfile_end+1,sizeof(int));
      switch (previous_files_sort(ss))
	{
	case 0: 
	  break;
	case 1: 
	  heapsort(prevfile_end+1,ALPHABET,vals,prevnames,prevfullnames,prevtimes); 
	  break;
	case 2:
	  for (i=0;i<=prevfile_end;i++)
	    vals[i] = file_write_date(prevfullnames[i]);
	  heapsort(prevfile_end+1,VALS_LESS,vals,prevnames,prevfullnames,prevtimes);
	  break;
	case 3:
	  for (i=0;i<=prevfile_end;i++)
	    vals[i] = sound_samples(prevfullnames[i]);
	  heapsort(prevfile_end+1,VALS_GREATER,vals,prevnames,prevfullnames,prevtimes);
	  break;
	case 4:
	  for (i=0;i<=prevfile_end;i++) vals[i] = prevtimes[i];
	  heapsort(prevfile_end+1,VALS_LESS,vals,prevnames,prevfullnames,prevtimes);
	  break;
	}
      FREE(vals);
      for (i=0;i<=prevfile_end;i++)
	{
	  if (!((r = prev_name_row[i])))
	    {
	      r = make_regrow(ss,fbd->prevww,last_row,View_PrevFiles_Unlist_Callback,View_PrevFiles_Play_Callback,View_PrevFiles_Select_Callback);
	      prev_name_row[i] = r;
	      r->pos = i;
	      r->ss = ss;
	    }
	  fill_in_row(r,prevnames[i],0);
	  last_row = r->rw;
	}
    }
  for (i=prevfile_end+1;i<=max_prevfile_end;i++)
    {
      if ((r = prev_name_row[i]))
	{
	  if (XtIsManaged(r->rw)) XtUnmanageChild(r->rw);
	}
    }
  max_prevfile_end = prevfile_end;
  if (!(XtIsManaged(fbd->prevlst))) XtManageChild(fbd->prevlst);
}


/* play open unlist for prevfile, play save select for curfile, preload process for prevfile (snd-clm) */

void View_Files_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  /* fire up a dialog window with a list of currently open files, 
   * currently selected file also selected in list --
   * if user selects one (browse mode), so does Snd (via normalize_sound etc)
   * use snd_info label as is (short-form with '*' etc)
   * secondary list of previously edited files (if still in existence) --
   * click here re-opens the file.  (The overall form is similar to the regions browser).
   * The previous files list requires that we keep such a list as we go along, on the
   * off-chance this browser will be fired up.  (Such files may be subsequently moved or deleted).
   */
  snd_state *ss = (snd_state *)clientData;
  int n;
  Arg args[20];
  ww_info *wwl;
  regrow *r;
  XmString xdismiss,xhelp,xclear,titlestr;
  Widget mainform,curform,prevform,updateB,sep;
  if (!fbd)
    {
      fbd = (fb_info *)CALLOC(1,sizeof(fb_info));
      fbd->state = ss;
      fbd->selected_file = -1;
      xdismiss = XmStringCreate(STR_Dismiss,XmFONTLIST_DEFAULT_TAG);
      xhelp = XmStringCreate(STR_Help,XmFONTLIST_DEFAULT_TAG);
      xclear = XmStringCreate(STR_Clear,XmFONTLIST_DEFAULT_TAG);
      titlestr = XmStringCreate(STR_Files,XmFONTLIST_DEFAULT_TAG);
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNcancelLabelString,xclear); n++;
      XtSetArg(args[n],XmNhelpLabelString,xhelp); n++;
      XtSetArg(args[n],XmNokLabelString,xdismiss); n++;
      XtSetArg(args[n],XmNautoUnmanage,FALSE); n++;
      XtSetArg(args[n],XmNdialogTitle,titlestr); n++;
#if RESIZE_DIALOG
      XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
      XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
      XtSetArg(args[n],XmNtransient,FALSE); n++;
      fbd->dialog = XmCreateTemplateDialog(main_SHELL(ss),STR_File_Browser,args,n);
      add_dialog(ss,fbd->dialog);
#if OVERRIDE_TOGGLE
      override_form_translation(fbd->dialog);
#endif

      XtAddCallback(fbd->dialog,XmNcancelCallback,View_Files_Clear_Callback,ss);
      XtAddCallback(fbd->dialog,XmNhelpCallback,View_Files_Help_Callback,ss);
      XtAddCallback(fbd->dialog,XmNokCallback,View_Files_Dismiss_Callback,fbd);
      XmStringFree(xhelp);
      XmStringFree(xdismiss);
      XmStringFree(xclear);
      XmStringFree(titlestr);

      if (!(ss->using_schemes))
	{
	  XtVaSetValues(XmMessageBoxGetChild(fbd->dialog,XmDIALOG_OK_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(fbd->dialog,XmDIALOG_CANCEL_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(fbd->dialog,XmDIALOG_HELP_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	}

      n=0;
      if (!(ss->using_schemes)) 
	{
	  n = background_basic_color(args,n,ss);
	  XtSetArg(args[n],XmNarmColor,(ss->sgx)->pushed_button_color); n++;
	}
      updateB = XtCreateManagedWidget(STR_Update,xmPushButtonWidgetClass,fbd->dialog,args,n);
      XtAddCallback(updateB,XmNactivateCallback,View_Files_Update_Callback,ss);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,XmMessageBoxGetChild(fbd->dialog,XmDIALOG_SEPARATOR)); n++;
      mainform = sndCreateFormWidget("formd",fbd->dialog,args,n);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNrightPosition,49); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      curform = sndCreateFormWidget("curform",mainform,args,n);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,curform); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNorientation,XmVERTICAL); n++;
      XtSetArg(args[n],XmNseparatorType,XmSHADOW_ETCHED_IN); n++;
      sep = XtCreateManagedWidget("sep",xmSeparatorWidgetClass,mainform,args,n);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,sep); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      prevform = sndCreateFormWidget("prevform",mainform,args,n);


      /* current files section: save play current files | files */
      wwl = make_title_row(ss,curform,STR_save,STR_play,STR_current_files,
			   PAD_TITLE_ON_RIGHT,WITHOUT_SORT_BUTTON,WITHOUT_PANED_WINDOW);
      fbd->curww = wwl->ww;
      fbd->curlst = wwl->list;
      if (!(ss->using_schemes)) map_over_children(fbd->curlst,set_main_color_of_widget,(void *)clientData);
      FREE(wwl); 
      wwl=NULL;
      if (curfile_size == 0) /* apparently we need at least one row to get Motif to allocate the outer widgets correctly */
	{                    /* not curfile_end here since it is tracking currently active files before this dialog is created */
	  curfile_size = 4;
	  curnames = (char **)CALLOC(curfile_size,sizeof(char *));
	  cur_name_row = (regrow **)CALLOC(curfile_size,sizeof(regrow *));
	  a_big_star = (int *)CALLOC(curfile_size,sizeof(int *));
	  r = make_regrow(ss,fbd->curww,NULL,View_CurFiles_Save_Callback,View_CurFiles_Play_Callback,View_CurFiles_Select_Callback);
	  cur_name_row[0] = r;
	  r->pos = 0;
	  r->ss = ss;
	}

      /* previous files section: unlist play previous files | files */
      wwl = make_title_row(ss,prevform,STR_unlist,STR_play,STR_previous_files,
			   PAD_TITLE_ON_LEFT,WITH_SORT_BUTTON,WITHOUT_PANED_WINDOW);
      XtAddCallback(wwl->byname,XmNactivateCallback,sort_prevfiles_by_name,ss);
      XtAddCallback(wwl->bydate,XmNactivateCallback,sort_prevfiles_by_date,ss);
      XtAddCallback(wwl->bysize,XmNactivateCallback,sort_prevfiles_by_size,ss);
      XtAddCallback(wwl->byentry,XmNactivateCallback,sort_prevfiles_by_entry_order,ss);

      fbd->prevww = wwl->ww;
      fbd->prevlst = wwl->list;
      if (!(ss->using_schemes)) map_over_children(fbd->prevlst,set_main_color_of_widget,(void *)clientData);
      FREE(wwl); 
      wwl=NULL;
      if (prevfile_size == 0)
	{
	  prevfile_size = 4;
	  prevnames = (char **)CALLOC(prevfile_size,sizeof(char *));
	  prevfullnames = (char **)CALLOC(prevfile_size,sizeof(char *));
	  prevtimes = (int *)CALLOC(prevfile_size,sizeof(char *));
	  prev_name_row = (regrow **)CALLOC(prevfile_size,sizeof(regrow *));
	  r = make_regrow(ss,fbd->prevww,NULL,View_PrevFiles_Unlist_Callback,View_PrevFiles_Play_Callback,View_PrevFiles_Select_Callback);
	  prev_name_row[0] = r;
	  r->pos = 0;
	  r->ss = ss;
	}
    }
  else raise_dialog(fbd->dialog);
  make_curfiles_list(ss);
  make_prevfiles_list(ss);
  if (!XtIsManaged(fbd->dialog)) XtManageChild(fbd->dialog);
  highlight_selected_sound(ss);
}


/* -------- edit find -------- */

typedef struct {
  int size; 
  int window; 
  float beta; 
  Widget dialog;
  Widget list; 
  Widget scale; 
  Widget cancelB;
  snd_state *state;
  int type;
} dialog_info;

static void edit_find_cancel_callback(Widget w,XtPointer clientData,XtPointer callData)
{ /* "Done" */
  dialog_info *fd = (dialog_info *)clientData;
  snd_state *ss;
  ss = fd->state;
  if (ss->checking_explicitly)
    ss->stopped_explicitly = 1;
  else XtUnmanageChild(fd->dialog);
} 

static void edit_find_help_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Global Find",
"This search travels through all the current channels\n\
in parallel until a match is found.  The expression\n\
uses a C-like syntax.  y>.1, for example, searches\n\
until a sample is found greater than .1.\n\
");
} 

static void edit_find_ok_callback(Widget w,XtPointer clientData,XtPointer callData)
{ /* "Find" is the label here */
  dialog_info *fd = (dialog_info *)clientData;
  Widget text;
  char *str;
  XmString s1;
  snd_state *ss;
#if HAVE_GUILE
  SCM proc;
#endif
  text = fd->list;
  ss = fd->state;
  str = XmTextGetString(text);
  if ((str) && (*str))
    {
      if (ss->search_tree) {free_sop(ss->search_tree); ss->search_tree = NULL;}
      if (ss->search_expr) FREE(ss->search_expr);
      ss->search_expr = str;
#if HAVE_GUILE
      ss->search_proc = SCM_UNDEFINED;
      proc = parse_proc(str);
      if (gh_procedure_p(proc))
	ss->search_proc = proc;
      else
#endif
      ss->search_tree = sop_parse(str,SEARCH_TREE);
#if HAVE_GUILE
      if ((!ss->search_tree) && (!(gh_procedure_p(ss->search_proc))))
#else
      if (!ss->search_tree)
#endif
	{
	  sprintf(file_string,"%s: %s",str,sop_parse_error_name());
	  make_button_label(fd->scale,file_string);
	}
      else 
	{
	  sprintf(file_string,STR_find_s,str);
	  make_button_label(fd->scale,file_string);
	}
      XmTextSetString(text,NULL);
    }
#if HAVE_GUILE
  if ((ss->search_tree) || (gh_procedure_p(ss->search_proc)))
#else
  if (ss->search_tree) 
#endif
    {
      s1 = XmStringCreate(STR_Stop,XmFONTLIST_DEFAULT_TAG);
      XtVaSetValues(fd->cancelB,XmNlabelString,s1,NULL);
      XmStringFree(s1);
      str = global_search(ss);
      s1 = XmStringCreate(STR_Dismiss,XmFONTLIST_DEFAULT_TAG);
      XtVaSetValues(fd->cancelB,XmNlabelString,s1,NULL);
      XmStringFree(s1);
      if ((str) && (*str)) make_button_label(fd->scale,str);
    }
} 

static void make_edit_find_dialog(Widget w,dialog_info *fdw)
{
  Arg args[20];
  int n;
  Widget df,dl,rc,ds;
  XmString xmstr1,xmstr2,titlestr;
  snd_state *ss;
  ss = fdw->state;

  n=0;
  if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
  xmstr1=XmStringCreate(STR_Find,XmFONTLIST_DEFAULT_TAG);
  xmstr2=XmStringCreate(STR_Dismiss,XmFONTLIST_DEFAULT_TAG);
  titlestr=XmStringCreate(STR_Find,XmFONTLIST_DEFAULT_TAG);
  XtSetArg(args[n],XmNokLabelString,xmstr1); n++;
  XtSetArg(args[n],XmNcancelLabelString,xmstr2); n++;
  XtSetArg(args[n],XmNautoUnmanage,FALSE); n++;
  XtSetArg(args[n],XmNdialogTitle,titlestr); n++;
#if RESIZE_DIALOG
  XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
  XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
  XtSetArg(args[n],XmNtransient,FALSE); n++;
  fdw->dialog = XmCreateMessageDialog(w,STR_find,args,n);
  add_dialog(fdw->state,fdw->dialog);
#if OVERRIDE_TOGGLE
  override_form_translation(fdw->dialog);
#endif

  XmStringFree(xmstr1);
  XmStringFree(xmstr2);
  XmStringFree(titlestr);

  XtUnmanageChild(XmMessageBoxGetChild(fdw->dialog,XmDIALOG_SYMBOL_LABEL));
  XtUnmanageChild(XmMessageBoxGetChild(fdw->dialog,XmDIALOG_MESSAGE_LABEL));

  XtAddCallback(fdw->dialog,XmNhelpCallback,edit_find_help_callback,fdw->state);
  XtAddCallback(fdw->dialog,XmNcancelCallback,edit_find_cancel_callback,fdw);
  XtAddCallback(fdw->dialog,XmNokCallback,edit_find_ok_callback,fdw);

  rc = sndCreateFormWidget("row",fdw->dialog,NULL,0);

  n=0;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  dl = XtCreateManagedWidget(STR_find_p,xmLabelWidgetClass,rc,args,n);

  n=0;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,dl); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  df = sndCreateTextFieldWidget(fdw->state,"text",rc,args,n,ACTIVATABLE,NO_COMPLETER);
  fdw->list = df;

  n=0;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,df); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  ds = XtCreateManagedWidget(STR_global_search,xmLabelWidgetClass,rc,args,n);
  fdw->scale = ds;
}

void Edit_Find_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  static dialog_info *fdw;
  snd_state *ss = (snd_state *)clientData;
  finish_keyboard_selection();
  if (!fdw) 
    {
      fdw = (dialog_info *)CALLOC(1,sizeof(dialog_info));
      fdw->state = ss;
      make_edit_find_dialog(w,fdw);
#if MANAGE_DIALOG
      XtManageChild(fdw->dialog);
#endif
      if (!(ss->using_schemes)) 
	{
	  map_over_children(fdw->dialog,set_main_color_of_widget,(void *)clientData);
	  XtVaSetValues(XmMessageBoxGetChild(fdw->dialog,XmDIALOG_OK_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(fdw->dialog,XmDIALOG_CANCEL_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(fdw->dialog,XmDIALOG_HELP_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	}
      fdw->cancelB = XmMessageBoxGetChild(fdw->dialog,XmDIALOG_CANCEL_BUTTON);
    }
  else raise_dialog(fdw->dialog);
  if (!XtIsManaged(fdw->dialog)) XtManageChild(fdw->dialog);
}


/* -------- region browser -------- */

static regrow **region_rows = NULL;
static snd_info *reg_sp = NULL;
static dialog_info *reg_fd = NULL;
static int current_region = -1;
static Widget selectw,reg_srtxt,reg_lentxt,reg_chntxt,reg_maxtxt;

static void region_update_graph(chan_info *cp)
{
  reg_sp->nchans = region_chans(current_region);
  update_graph(cp,NULL);
  reg_sp->nchans = 1;
}

static void make_region_element(region_state *rs, int i)
{
  regrow *r;
  r = region_rows[i];
  make_bold_button_label(r->nm,rs->name[i]);
  XmToggleButtonSetState(r->sv,rs->save[i],FALSE);
  XmToggleButtonSetState(r->pl,FALSE,FALSE);
  XtManageChild(r->rw);
}

static void unhighlight_region(snd_state *ss)
{
  regrow *oldr;
  if (current_region != -1)
    {
      oldr = region_rows[current_region];
      if (!(ss->using_schemes)) 
	{
	  XtVaSetValues(oldr->rw,XmNbackground,(ss->sgx)->highlight_color,NULL);
	  XtVaSetValues(oldr->nm,XmNbackground,(ss->sgx)->highlight_color,NULL);
	}
    }
}

static void unmake_region_labels (void)
{
  make_bold_button_label(reg_srtxt,STR_srate_p);
  make_bold_button_label(reg_chntxt,STR_chans_p);
  make_bold_button_label(reg_lentxt,STR_length_p);
  make_bold_button_label(reg_maxtxt,STR_maxamp_p);
}

static void highlight_region(snd_state *ss)
{
  regrow *oldr;
  if (current_region != -1)
    {
      oldr = region_rows[current_region];
      if (!(ss->using_schemes)) 
	{
	  XtVaSetValues(oldr->rw,XmNbackground,(ss->sgx)->zoom_color,NULL);
	  XtVaSetValues(oldr->nm,XmNbackground,(ss->sgx)->zoom_color,NULL);
	}
    }
}

static void make_region_labels(file_info *hdr)
{
  sprintf(file_string,STR_srate,hdr->srate);
  make_bold_button_label(reg_srtxt,file_string);
  sprintf(file_string,STR_chans,hdr->chans);
  make_bold_button_label(reg_chntxt,file_string);
  sprintf(file_string,STR_length,(float)(hdr->samples)/(float)(hdr->chans * hdr->srate));
  make_bold_button_label(reg_lentxt,file_string);
  sprintf(file_string,STR_maxamp,region_maxamp(current_region));
  make_bold_button_label(reg_maxtxt,file_string);
}

void update_region_browser(int grf_too)
{
  int i,len;
  region_state *rs;
  snd_state *ss;
  chan_info *cp;
  rs = region_report();
  ss = reg_fd->state;
  len = rs->len;
  for (i=0;i<len;i++) make_region_element(rs,i);
  for (i=len;i<max_regions(ss);i++) XtUnmanageChild(region_rows[i]->rw);
  free_region_state(rs);
  if (len == 0) return;
  XtManageChild(reg_fd->list);
  if (grf_too)
    {
      unhighlight_region(ss);
      current_region = 0;
      highlight_region(ss);
      goto_window(region_rows[0]->nm);
      cp = reg_sp->chans[0];
      if (cp) 
	{
	  cp->chan = 0;
	  XtSetSensitive(channel_f(cp),FALSE);
	  XtSetSensitive(channel_w(cp),(region_chans(0) > 1));
	  if (region_ok(0)) 
	    {
	      reg_sp->hdr = fixup_region_data(cp,0,0);
	      make_region_labels(reg_sp->hdr);
	      region_update_graph(cp);
	    }
	  else unmake_region_labels();
	}
    }
}

static void region_ok_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  dialog_info *fd = (dialog_info *)clientData;
  XtUnmanageChild(fd->dialog);
}

int region_browser_is_active(void)
{
  return((reg_fd) && (XtIsRealized(reg_fd->dialog)));
}

static void region_resize_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  region_update_graph((chan_info *)clientData);
}

static int region_dialog_active(void);

void delete_region_and_update_browser(snd_state *ss, int n)
{
  int act;
  act = delete_region(n);
  if (act == INVALID_REGION) return;
  if (region_dialog_active())
    {
      if (act != NO_REGIONS)
	{
	  current_region = 0;
	  highlight_region(ss);
	  goto_window(region_rows[0]->nm);
	}
      else 
	{
	  unhighlight_region(ss);
	  current_region = -1;
	}
      update_region_browser(1);
    }
}

static void region_delete_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  if (current_region != -1)
    delete_region_and_update_browser(ss,current_region);
}

static void region_help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  dialog_info *fd = (dialog_info *)clientData;
#if HAVE_XmHTML
  snd_help(fd->state,STR_Region_Browser,"#regionbrowser");
#else
  snd_help(fd->state,STR_Region_Browser,get_region_browser_help());
#endif
}

void select_region_and_update_browser(snd_state *ss, int n)
{
  deactivate_selection();  /* just in case there's a region being highlighted */
  if (region_dialog_active())
    {
      unhighlight_region(ss);
      select_region(n);
      current_region = 0;
      highlight_region(ss);
      goto_window(region_rows[0]->nm);
      XtSetSensitive(selectw,FALSE);
      update_region_browser(0);
    }
}

static void region_select_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  if (current_region != -1)
    select_region_and_update_browser(ss,current_region);
}

static void region_up_arrow_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  chan_info *cp;
  cp = reg_sp->chans[0];
  if (cp->chan > 0)
    {
      cp->chan--;
      XtSetSensitive(channel_f(cp),(cp->chan > 0));
      XtSetSensitive(channel_w(cp),TRUE);
      fixup_region_data(cp,cp->chan,current_region);
      region_update_graph(cp);
    }
}

static void region_down_arrow_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  chan_info *cp;
  cp = reg_sp->chans[0];
  if ((cp->chan+1) < region_chans(current_region))
    {
      cp->chan++;
      XtSetSensitive(channel_f(cp),TRUE);
      XtSetSensitive(channel_w(cp),(region_chans(current_region) > (cp->chan+1)));
      fixup_region_data(cp,cp->chan,current_region);
      region_update_graph(cp);
    }
}

static void region_focus_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss;
  chan_info *cp;
  regrow *r = (regrow *)clientData;
  ss = r->ss;
  unhighlight_region(ss);
  current_region=r->pos;
  cp = reg_sp->chans[0];
  cp->chan  = 0;
  highlight_region(ss);
  XtSetSensitive(channel_f(cp),FALSE);
  XtSetSensitive(channel_w(cp),(region_chans(current_region) > 1));
  XtSetSensitive(selectw,(current_region != 0));
  reg_sp->hdr = fixup_region_data(cp,0,current_region);
  make_region_labels(reg_sp->hdr);
  region_update_graph(cp);
}


void reflect_play_region_stop(region_info *r)
{
  regrow *rg;
  rg = (regrow *)(r->rg);
  if (rg) XmToggleButtonSetState(rg->pl,FALSE,FALSE);
}

static void region_play_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  regrow *r = (regrow *)clientData;
  if (XmToggleButtonGetState(r->pl))
    play_region(r->ss,r->pos,(void *)r);
  else stop_playing_region(r->pos);
}

static void region_save_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  regrow *r = (regrow *)clientData;
  XmToggleButtonCallbackStruct *cb = (XmToggleButtonCallbackStruct *)callData;
  protect_region(r->pos,cb->set);
}

void set_region_protect(int reg, int protect)
{
  regrow *r;
  protect_region(reg,protect);
  if (region_rows)
    {
      r = region_rows[reg];
      if ((r) && (r->sv)) XmToggleButtonSetState(r->sv,protect,FALSE);
    }
}

static void region_print_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  if (current_region != -1)
    region_print(eps_file(ss),"region",reg_sp->chans[0]);
}

static void region_edit_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  if (current_region != -1) 
    region_edit((snd_state *)clientData,current_region);
}

static Widget region_ww;

static void make_region_dialog(Widget w, dialog_info *fd)
{
  int n,i;
  Arg args[32];
  Widget formw,last_row,ww,infosep,prtb,editb;
  XmString xok,xdelete,xhelp,titlestr;
  regrow *r;
  snd_state *ss;
  chan_info *cp;
  file_info *hdr;
  ww_info *wwl;

  ss = fd->state;
  reg_fd = fd;

  xok = XmStringCreate(STR_Dismiss,XmFONTLIST_DEFAULT_TAG);
  xhelp = XmStringCreate(STR_Help,XmFONTLIST_DEFAULT_TAG);
  xdelete = XmStringCreate(STR_Delete,XmFONTLIST_DEFAULT_TAG);
  titlestr = XmStringCreate(STR_Regions,XmFONTLIST_DEFAULT_TAG);

  n=0;
  if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
  XtSetArg(args[n],XmNcancelLabelString,xdelete); n++;
  XtSetArg(args[n],XmNhelpLabelString,xhelp); n++;
  XtSetArg(args[n],XmNokLabelString,xok); n++;
  XtSetArg(args[n],XmNautoUnmanage,FALSE); n++;
  XtSetArg(args[n],XmNdialogTitle,titlestr); n++;
#if RESIZE_DIALOG
  XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
  XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
  XtSetArg(args[n],XmNtransient,FALSE); n++;
  fd->dialog = XmCreateTemplateDialog(w,STR_Regions,args,n);
  add_dialog(ss,fd->dialog);
#if OVERRIDE_TOGGLE
  override_form_translation(fd->dialog);
#endif

  XtAddCallback(fd->dialog,XmNokCallback,region_ok_Callback,fd);
  XtAddCallback(fd->dialog,XmNcancelCallback,region_delete_Callback,ss);
  XtAddCallback(fd->dialog,XmNhelpCallback,region_help_Callback,fd);
  XmStringFree(xhelp);
  XmStringFree(xok);
  XmStringFree(xdelete);
  XmStringFree(titlestr);

  if (!(ss->using_schemes))
    {
      XtVaSetValues(XmMessageBoxGetChild(fd->dialog,XmDIALOG_OK_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaSetValues(XmMessageBoxGetChild(fd->dialog,XmDIALOG_CANCEL_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaSetValues(XmMessageBoxGetChild(fd->dialog,XmDIALOG_HELP_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
    }

  n=0;
  if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
  selectw = XtCreateManagedWidget(STR_Select,xmPushButtonWidgetClass,fd->dialog,args,n);
  XtAddCallback(selectw,XmNactivateCallback,region_select_Callback,ss);
  XtSetSensitive(selectw,FALSE);

  n=0;
  if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNbottomWidget,XmMessageBoxGetChild(fd->dialog,XmDIALOG_SEPARATOR)); n++;
  formw = sndCreateFormWidget("formw",fd->dialog,args,n);

  wwl = make_title_row(ss,formw,STR_save,STR_play,STR_regions,DONT_PAD_TITLE,WITHOUT_SORT_BUTTON,WITH_PANED_WINDOW);
  ww = wwl->ww;
  region_ww = ww;
  fd->list = wwl->list;
  if (!(ss->using_schemes)) map_over_children(fd->list,set_main_color_of_widget,(void *)ss);
  last_row = NULL;
  
  region_rows = (regrow **)CALLOC(max_regions(ss),sizeof(regrow *));
  for (i=0;i<max_regions(ss);i++)
    {
      r = make_regrow(ss,ww,last_row,region_save_Callback,region_play_Callback,region_focus_Callback);
      region_rows[i] = r;
      r->pos = i;
      r->ss = ss;
      last_row = r->rw;
    }

  update_region_browser(0);

  n=0;
  if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,fd->list); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNorientation,XmVERTICAL); n++;
  XtSetArg(args[n],XmNseparatorType,XmSHADOW_ETCHED_IN); n++;
  infosep = XtCreateManagedWidget("infosep",xmSeparatorWidgetClass,wwl->toppane,args,n);

  n=0;
  if (!(ss->using_schemes)) n = background_highlight_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,infosep); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNfontList,bold_button_FONT(ss)); n++;
  XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;
  reg_srtxt = XtCreateManagedWidget(STR_srate_p,xmLabelWidgetClass,wwl->toppane,args,n);

  n=0;
  if (!(ss->using_schemes)) n = background_highlight_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,infosep); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,reg_srtxt); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNfontList,bold_button_FONT(ss)); n++;
  XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;
  reg_chntxt = XtCreateManagedWidget(STR_chans_p,xmLabelWidgetClass,wwl->toppane,args,n);

  n=0;
  if (!(ss->using_schemes)) n = background_highlight_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,infosep); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,reg_chntxt); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNfontList,bold_button_FONT(ss)); n++;
  XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;
  reg_lentxt = XtCreateManagedWidget(STR_length_p,xmLabelWidgetClass,wwl->toppane,args,n);

  n=0;
  if (!(ss->using_schemes)) n = background_highlight_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,infosep); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,reg_lentxt); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNfontList,bold_button_FONT(ss)); n++;
  XtSetArg(args[n],XmNalignment,XmALIGNMENT_BEGINNING); n++;
  reg_maxtxt = XtCreateManagedWidget(STR_maxamp_p,xmLabelWidgetClass,wwl->toppane,args,n);

  n=0;
  if (!(ss->using_schemes)) 
    {
      n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNarmColor,(ss->sgx)->pushed_button_color); n++;
    }
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,infosep); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,reg_maxtxt); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNfontList,bold_button_FONT(ss)); n++;
  XtSetArg(args[n],XmNalignment,XmALIGNMENT_CENTER); n++;
  editb = XtCreateManagedWidget(STR_edit,xmPushButtonWidgetClass,wwl->toppane,args,n);
  XtAddCallback(editb,XmNactivateCallback,region_edit_Callback,(XtPointer)ss);

  n=0;
  if (!(ss->using_schemes)) 
    {
      n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNarmColor,(ss->sgx)->pushed_button_color); n++;
    }
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,infosep); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,editb); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNfontList,bold_button_FONT(ss)); n++;
  XtSetArg(args[n],XmNalignment,XmALIGNMENT_CENTER); n++;
  prtb = XtCreateManagedWidget(STR_print,xmPushButtonWidgetClass,wwl->toppane,args,n);
  XtAddCallback(prtb,XmNactivateCallback,region_print_Callback,(XtPointer)ss);

  n=0;
  if (!(ss->using_schemes)) n = background_white_color(args,n,ss);
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNpaneMinimum,150); n++;
  fd->scale = sndCreateFormWidget("grf",wwl->panes,args,n);

  XtManageChild(fd->dialog);

  if (!reg_sp) 
    { /* just a place holder, I think -- see make_region_readable in snd-clip.c */
      reg_sp = (snd_info *)CALLOC(1,sizeof(snd_info));
      reg_sp->s_type = SND_INFO;
      reg_sp->nchans = 1;
      reg_sp->allocated_chans = 1;
      reg_sp->chans = (chan_info **)CALLOC(1,sizeof(chan_info *));
      reg_sp->sx_scroll_max = 100;
      reg_sp->hdr = (file_info *)CALLOC(1,sizeof(file_info));
      hdr = reg_sp->hdr;
      hdr->s_type = FILE_INFO;
      hdr->samples = region_len(0);
      hdr->srate = region_srate(0);
      hdr->comment = NULL;
      hdr->chans = 1;
      current_region = 0;
      add_channel_window(reg_sp,0,ss,0,0,fd->scale,WITH_ARROWS);
      cp = reg_sp->chans[0];
      cp->edit_size = 1;
      cp->edit_ctr = 0;
      cp->edits = (ed_list **)CALLOC(cp->edit_size,sizeof(ed_list *));
      cp->samples = (int *)CALLOC(cp->edit_size,sizeof(int));
      cp->sound_size = 1;
      cp->sound_ctr = 0;
      cp->sounds = (snd_data **)CALLOC(cp->sound_size,sizeof(snd_data *));
      cp->samples[0] = region_len(0);
    }
  else 
    {
      add_channel_window(reg_sp,0,ss,0,0,fd->scale,WITH_ARROWS);
      cp = reg_sp->chans[0];
    }

  cp->hookable = 0;
  if (!(ss->using_schemes)) 
    {
      XtVaSetValues(region_rows[0]->nm,XmNbackground,(ss->sgx)->white,NULL);
      map_over_children(wwl->panes,color_sashes,(void *)ss);
    }
  XtVaSetValues(wwl->toppane,XmNpaneMinimum,1,NULL);
  XtVaSetValues(fd->scale,XmNpaneMinimum,1,NULL);

  XtAddCallback(channel_graph(cp),XmNresizeCallback,region_resize_Callback,(XtPointer)cp);
  XtAddCallback(channel_graph(cp),XmNexposeCallback,region_resize_Callback,(XtPointer)cp);

  /* channel_f is up arrow, channel_w is down arrow */
  XtAddCallback(channel_f(cp),XmNactivateCallback,region_up_arrow_Callback,(XtPointer)ss);
  XtAddCallback(channel_w(cp),XmNactivateCallback,region_down_arrow_Callback,(XtPointer)ss);
  XtSetSensitive(channel_f(cp),FALSE);
  if (region_chans(0) > 1) XtSetSensitive(channel_w(cp),TRUE);
  cp->chan = 0;
  reg_sp->hdr = fixup_region_data(cp,0,0);
  make_region_labels(reg_sp->hdr);
  highlight_region(ss);
  region_update_graph(cp);
  FREE(wwl); 
  wwl=NULL;
}

static dialog_info *region_fd = NULL;

void View_Region_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  /* put up scrollable dialog describing/playing/editing the region list */
  snd_state *ss = (snd_state *)clientData;
  if (region_fd == NULL)
    {
      region_fd = (dialog_info *)CALLOC(1,sizeof(dialog_info));
      region_fd->size = 0;
      region_fd->state = ss;
      make_region_dialog(w,region_fd);
    }
  else raise_dialog(region_fd->dialog);
  if (!XtIsManaged(region_fd->dialog)) {current_region = 0; XtManageChild(region_fd->dialog);}
}

static int region_dialog_active(void) {return(region_fd != NULL);}

int region_dialog_is_active(void)
{
  return((region_fd) && (region_fd->dialog) && (XtIsManaged(region_fd->dialog)));
}


void allocate_region_rows(snd_state *ss, int n)
{
  Widget last_row;
  regrow *r;
  int i;
  if ((region_fd) && (n > max_regions(ss)))
    {
      r = region_rows[max_regions(ss) - 1];
      last_row = r->rw;
      region_rows = (regrow **)REALLOC(region_rows,n * sizeof(regrow *));
      for (i=max_regions(ss);i<n;i++)
	{
	  r = make_regrow(ss,region_ww,last_row,region_save_Callback,region_play_Callback,region_focus_Callback);
	  region_rows[i] = r;
	  r->pos = i;
	  r->ss = ss;
	  last_row = r->rw;
	}
    }
}



/* -------------------------------- Raw Data Dialog -------------------------------- */
/*
 * used also by New File menu option
 */

static Widget raw_data_dialog = NULL;
static Widget raw_srate_text,raw_chans_text,raw_location_text;
static int raw_data_location = 0;

static int new_ctr = 0;

static void cancel_new_file(snd_state *ss)
{
  if (ss->pending_new)
    {
      remove(ss->pending_new);
      ss->pending_new = NULL;
      new_ctr--;
    }
}


static snd_info *finish_new_file(snd_state *ss,char *newname,int header_type, int data_format, int srate, int chans)
{
  snd_info *sp;
  int chan,size;
  unsigned char* buf;
  if (mus_header_writable(header_type,data_format))
    {
      snd_write_header(ss,newname,header_type,srate,chans,28,chans*2,data_format,NULL,0);
      chan = snd_reopen_write(ss,newname);
      lseek(chan,mus_header_data_location(),SEEK_SET);
      size = raw_chans(ss) * mus_samples2bytes(data_format,2); /* why 2 samples? */
      buf = (unsigned char *)CALLOC(size,sizeof(unsigned char));
      write(chan,buf,size);
      close(chan);
      FREE(buf);
      sp = snd_open_file(newname,ss);
      ss->pending_new = NULL;
      return(sp);
    }
  else 
    {
      snd_error(STR_cant_write_type_with_format,
		(header_type == AIFF_sound_file) ? "an" : "a",
		sound_type_name(header_type),
		data_formats[data_format-1]);
      cancel_new_file(ss);
      return(NULL);
    }
}

static void raw_data_ok_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  state_context *sgx;
  snd_info *sp = NULL;
  char *str;
  int tmp;
  XmAnyCallbackStruct *cb = (XmAnyCallbackStruct *)callData;
  sgx = ss->sgx;
  if (cb->event != sgx->text_activate_event)
    {
      str = XmTextGetString(raw_srate_text);
      if ((str) && (*str)) {sscanf(str,"%d",&tmp); FREE(str);}
      set_raw_srate(ss,tmp);
      str = XmTextGetString(raw_chans_text);
      if ((str) && (*str)) {sscanf(str,"%d",&tmp); FREE(str);}
      set_raw_chans(ss,tmp);
      str = XmTextGetString(raw_location_text);
      if ((str) && (*str)) {sscanf(str,"%d",&raw_data_location); FREE(str);}
      mus_set_raw_header_defaults(raw_srate(ss),raw_chans(ss),raw_format(ss));
      if (ss->pending_new)
	sp = finish_new_file(ss,ss->pending_new,raw_type(ss),raw_format(ss),raw_srate(ss),raw_chans(ss));
      else 
	{
	  if (ss->pending_open)
	    {
	      override_sound_header(ss->pending_open, raw_srate(ss), raw_chans(ss), raw_format(ss), raw_sound_file, raw_data_location, 
				    mus_bytes2samples(raw_format(ss),mus_true_file_length() - raw_data_location));
	      sp = snd_open_file(ss->pending_open,ss);
	    }
	}
      if (sp) select_channel(sp,0);
      reflect_raw_open_in_menu();
      XtUnmanageChild(raw_data_dialog);
    }
}

static void raw_data_help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
       STR_Raw_Data,
"To display and edit sound data, Snd needs\n\
to know how the data's sampling rate, number\n\
of channels, and numerical format.  This dialog\n\
gives you a chance to set those fields.\n\
To make the current settings the default\n\
for any future headerless files, click the\n\
'Default' button.\n\
");
}

static void raw_data_cancel_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  ss->pending_open = NULL;
  if (ss->pending_new) cancel_new_file(ss);
      reflect_raw_open_in_menu();
  XtUnmanageChild(raw_data_dialog);
}

static void raw_data_default_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  set_use_raw_defaults(ss,1);
  raw_data_ok_Callback(w,clientData,callData);
}

static void raw_data_browse_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  XmListCallbackStruct *cbs = (XmListCallbackStruct *)callData;
  set_raw_format(ss,cbs->item_position);
}


static char dfs_str[8];
static char dfc_str[4];

static void make_raw_data_dialog(char *filename, snd_state *ss)
{
  XmString *formats;
  XmString xstr1,xstr2,xstr3,xstr4,titlestr;
  int i,n;
  Arg args[20];
  Widget lst,defw,dls,dfs,dfc,rform,dlab,dloc,dloclab,chnlab;
  n=0;
  xstr1 = XmStringCreate(STR_Cancel,XmFONTLIST_DEFAULT_TAG); /* needed by template dialog */
  xstr2 = XmStringCreate(STR_Help,XmFONTLIST_DEFAULT_TAG);
  xstr3 = XmStringCreate(STR_Ok,XmFONTLIST_DEFAULT_TAG);
  titlestr = XmStringCreate(STR_No_Header_on_File,XmFONTLIST_DEFAULT_TAG);
  sprintf(file_string,STR_No_header_found_for,filename_without_home_directory(filename));
  xstr4 = XmStringCreate(file_string,XmFONTLIST_DEFAULT_TAG);
  if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
  XtSetArg(args[n],XmNcancelLabelString,xstr1); n++;
  XtSetArg(args[n],XmNhelpLabelString,xstr2); n++;
  XtSetArg(args[n],XmNokLabelString,xstr3); n++;
  XtSetArg(args[n],XmNmessageString,xstr4); n++;
  XtSetArg(args[n],XmNdialogTitle,titlestr); n++;
  XtSetArg(args[n],XmNautoUnmanage,FALSE); n++;
#if RESIZE_DIALOG
  XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
  XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
  /* not transient -- we want this window to remain visible if possible */
  raw_data_dialog = XmCreateTemplateDialog(main_SHELL(ss),STR_raw_data,args,n);
#if OVERRIDE_TOGGLE
  override_form_translation(raw_data_dialog);
#endif

  XtAddCallback(raw_data_dialog,XmNcancelCallback,raw_data_cancel_Callback,ss);
  XtAddCallback(raw_data_dialog,XmNhelpCallback,raw_data_help_Callback,ss);
  XtAddCallback(raw_data_dialog,XmNokCallback,raw_data_ok_Callback,ss);
  XmStringFree(xstr1);
  XmStringFree(xstr2);
  XmStringFree(xstr3);
  XmStringFree(xstr4);
  XmStringFree(titlestr);

  n=0;
  if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
  defw = XtCreateManagedWidget(STR_Default,xmPushButtonWidgetClass,raw_data_dialog,args,n);
  XtAddCallback(defw,XmNactivateCallback,raw_data_default_Callback,ss);

  rform = sndCreateFormWidget("sretc",raw_data_dialog,NULL,0);

  n=0;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNmarginTop,6); n++;
  dls = XtCreateManagedWidget(STR_srate_p,xmLabelWidgetClass,rform,args,n);

  n=0;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,dls); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNcolumns,6); n++;
  XtSetArg(args[n],XmNresizeWidth,FALSE); n++;
  dfs = sndCreateTextFieldWidget(ss,"text",rform,args,n,NOT_ACTIVATABLE,add_completer_func(srate_completer));
  if (raw_srate(ss) < 100000) sprintf(dfs_str," %d",raw_srate(ss)); else sprintf(dfs_str,"%d",raw_srate(ss));
  XmTextSetString(dfs,dfs_str);

  n=0;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,dfs); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNalignment,XmALIGNMENT_END); n++;	
  XtSetArg(args[n],XmNmarginTop,6); n++;
  chnlab = XtCreateManagedWidget(STR_chans_p,xmLabelWidgetClass,rform,args,n);

  n=0;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,chnlab); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNcolumns,3); n++;
  XtSetArg(args[n],XmNresizeWidth,FALSE); n++;
  dfc = sndCreateTextFieldWidget(ss,"text",rform,args,n,NOT_ACTIVATABLE,NO_COMPLETER);
  if (raw_chans(ss) < 10) sprintf(dfc_str,"  %d",raw_chans(ss)); else sprintf(dfc_str," %d",raw_chans(ss));
  XmTextSetString(dfc,dfc_str);
  
  n=0;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,dfs); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNalignment,XmALIGNMENT_END); n++;
  XtSetArg(args[n],XmNmarginTop,6); n++;
  dloclab = XtCreateManagedWidget(STR_data_location_p,xmLabelWidgetClass,rform,args,n);
  
  n=0;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,dloclab); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,dfs); n++;
  XtSetArg(args[n],XmNcolumns,8); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  dloc = sndCreateTextFieldWidget(ss,"text",rform,args,n,NOT_ACTIVATABLE,NO_COMPLETER);
  XmTextSetString(dloc,"0");

  n=0;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,dloc); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  dlab = XtCreateManagedWidget(STR_data_format_p,xmLabelWidgetClass,rform,args,n);

  n=0;
#ifdef LESSTIF_VERSION
  if (!(ss->using_schemes)) n = background_white_color(args,n,ss);
#endif
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNtopWidget,dlab); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  formats = (XmString *)CALLOC(NUM_DATA_FORMATS,sizeof(XmString));
  for (i=0;i<NUM_DATA_FORMATS;i++) formats[i] = XmStringCreate(data_formats[i],XmFONTLIST_DEFAULT_TAG);
  lst = XmCreateScrolledList(rform,"raw-data-format-list",args,n);
  XtVaSetValues(lst,XmNitems,formats,XmNitemCount,NUM_DATA_FORMATS,XmNvisibleItemCount,6,NULL);
  XtManageChild(lst); 
  XmListSelectPos(lst,raw_format(ss),FALSE);
  for (i=0;i<NUM_DATA_FORMATS;i++) XmStringFree(formats[i]);
  FREE(formats);
  XtAddCallback(lst,XmNbrowseSelectionCallback,raw_data_browse_Callback,ss);

#if MANAGE_DIALOG
  XtManageChild(raw_data_dialog);
#endif

  if (!(ss->using_schemes)) 
    {
      map_over_children(rform,set_main_color_of_widget,(void *)ss);
      XtVaSetValues(lst,XmNbackground,(ss->sgx)->white,NULL);
      XtVaSetValues(XmMessageBoxGetChild(raw_data_dialog,XmDIALOG_OK_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaSetValues(XmMessageBoxGetChild(raw_data_dialog,XmDIALOG_CANCEL_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaSetValues(XmMessageBoxGetChild(raw_data_dialog,XmDIALOG_HELP_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
      XtVaSetValues(defw,XmNselectColor,(ss->sgx)->pushed_button_color,NULL);
    }

  raw_srate_text = dfs;
  raw_chans_text = dfc;
  raw_location_text = dloc;
}

file_info *get_file_info(char *filename, snd_state *ss)
{
  /* put up dialog for srate, chans, data format */
  XmString xstr;
  if ((ss->pending_open) || (use_raw_defaults(ss)))
    {
      /* choices already made, so just send back a header that reflects those choices */
      return(make_file_info_1(filename,ss));
    }
  if (!(ss->pending_new))
    {
      set_snd_IO_error(SND_PENDING_OPEN);
      ss->pending_open = copy_string(filename);
    }
  if (!raw_data_dialog) 
    make_raw_data_dialog(filename,ss);
  else
    {
      /* set filename in label */
      sprintf(file_string,STR_No_header_found_for,filename_without_home_directory(filename));
      xstr = XmStringCreate(file_string,XmFONTLIST_DEFAULT_TAG);
      XtVaSetValues(raw_data_dialog,XmNmessageString,xstr,NULL);
      XmStringFree(xstr);
      raise_dialog(raw_data_dialog);
    }
  /* we don't want to lock out the main program (or the dialog help window), but
   * it would be very confusing to have two file openings in progress.
   */
  if (!XtIsManaged(raw_data_dialog)) XtManageChild(raw_data_dialog);
  reflect_raw_pending_in_menu();
  return(NULL);
}

static int swap_int (int n)
{
  int o;
  unsigned char *inp,*outp; 
  inp=(unsigned char *)&n; 
  outp=(unsigned char *)&o;
  outp[0]=inp[3]; outp[1]=inp[2]; outp[2]=inp[1]; outp[3]=inp[0];
  return(o);
}

static short swap_short (short n)
{
  short o;
  unsigned char *inp,*outp; 
  inp=(unsigned char *)&n; 
  outp=(unsigned char *)&o;
  outp[0]=inp[1]; outp[1]=inp[0]; 
  return(o);
}

file_info *get_reasonable_file_info(char *filename, snd_state *ss, file_info *hdr)
{
  XmString xstr;
  char *reason_str,*tmp_str;
  int ns,better_srate = 0,better_chans = 0;
  reason_str = (char *)CALLOC(1024,sizeof(char));
  tmp_str = (char *)CALLOC(64,sizeof(char));
  /* try to provide some notion of what might be the intended header (currently limited to byte-order mistakes) */
  sprintf(reason_str,"srate: %d",hdr->srate);
  ns = swap_int(hdr->srate);
  if ((ns<4000) || (ns>100000)) ns = swap_short(hdr->srate);
  if ((ns>4000) && (ns<100000))
    {
      better_srate = ns;
      sprintf(tmp_str," (swapped: %d)",ns);
      strcat(reason_str,tmp_str);
    }
  sprintf(tmp_str,"\nchans: %d",hdr->chans);
  strcat(reason_str,tmp_str);
  ns = swap_int(hdr->chans);
  if ((ns<0) || (ns>8)) ns=swap_short(hdr->chans);
  if ((ns>0) && (ns <= 8))
    {
      better_chans = ns;
      sprintf(tmp_str," (swapped: %d)",ns);
      strcat(reason_str,tmp_str);
    }
  sprintf(tmp_str,"\nlength: %.3f (%d samples, %d bytes total)",
	  (float)(hdr->samples)/(float)(hdr->chans * hdr->srate),
	  hdr->samples,
	  mus_true_file_length());
  strcat(reason_str,tmp_str);
  ns = swap_int(hdr->samples);
  if (ns < mus_true_file_length())
    {
      sprintf(tmp_str,"\n  (swapped: %d",ns);
      strcat(reason_str,tmp_str);
      if ((better_chans) && (better_srate))
	{
	  sprintf(tmp_str,", swapped length: %.3f / sample-size-in-bytes)",(float)ns/(float)(better_chans * better_srate));
	  strcat(reason_str,tmp_str);
	}
      else strcat(reason_str,")");
    }
  sprintf(tmp_str,"\ndata location: %d",hdr->data_location);
  strcat(reason_str,tmp_str);
  ns = swap_int(hdr->data_location);
  if ((ns>0) && (ns<=1024)) 
    {
      sprintf(tmp_str," (swapped: %d)",ns);
      strcat(reason_str,tmp_str);
    }
  sprintf(tmp_str,"\ntype: %s",sound_type_name(hdr->type));
  strcat(reason_str,tmp_str);
  sprintf(tmp_str,"\nformat: %s\n",sound_format_name(hdr->format));
  strcat(reason_str,tmp_str);
  hdr->type = raw_sound_file;
  snd_help(ss,"Current header values",reason_str);
  sprintf(file_string,STR_Bogus_header_found_for,filename_without_home_directory(filename));
  xstr = XmStringCreate(file_string,XmFONTLIST_DEFAULT_TAG);
  if (!raw_data_dialog) make_raw_data_dialog(filename,ss);
  XtVaSetValues(raw_data_dialog,XmNmessageString,xstr,NULL);
  XmStringFree(xstr);
  raise_dialog(raw_data_dialog);
  if (!XtIsManaged(raw_data_dialog)) XtManageChild(raw_data_dialog);
  set_snd_IO_error(SND_PENDING_OPEN);
  ss->pending_open = copy_string(filename);
  reflect_raw_pending_in_menu();
  FREE(tmp_str);
  FREE(reason_str);
  return(NULL);
}

/* new file needs raw data dialog to get file description */

static char new_file_name[SNDLIB_MAX_FILE_NAME];

snd_info *snd_new_file(snd_state *ss, char *new_file_name, int header_type, int data_format, int srate, int chans)
{
  file_info *hdr;
  finish_keyboard_selection();
  snd_create(ss,new_file_name);          /* needs to exist for get_file_info */
  if (header_type == unsupported_sound_file)
    {
      ss->pending_new = new_file_name;
      hdr = get_file_info(new_file_name,ss); /* post the header dialog */
      if (hdr) return(finish_new_file(ss,new_file_name,raw_type(ss),raw_format(ss),raw_srate(ss),raw_chans(ss)));
    }
  else return(finish_new_file(ss,new_file_name,header_type,data_format,srate,chans));
  return(NULL);
}

snd_info *snd_new_file_checked(snd_state *ss, char *newname, int header_type, int data_format, int srate, int chans)
{
  if (!newname) sprintf(new_file_name,"new-%d.snd",new_ctr++); else strcpy(new_file_name,newname);
  if (snd_overwrite_ok(ss,new_file_name))
    return(snd_new_file(ss,new_file_name,header_type,data_format,srate,chans));
  return(NULL);
}




/* -------- color browser -------- */

typedef struct {
  Widget dialog;
  Widget list; 
  Widget scale; 
  Widget invert;
  Widget cutoff;
  snd_state *state;
} color_chooser_info;

static color_chooser_info *ccd = NULL;

static void Invert_Color_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_state *ss;
  color_chooser_info *cd = (color_chooser_info *)clientData;
  XmToggleButtonCallbackStruct *cb = (XmToggleButtonCallbackStruct *)callData;
  ss = cd->state;
  in_set_color_inverted(ss,cb->set);
  map_over_chans(ss,update_graph,NULL);
}

void set_color_inverted(snd_state *ss, int val)
{
  in_set_color_inverted(ss,val);
  if (ccd) XmToggleButtonSetState(ccd->invert,val,FALSE);
  if (!(ss->graph_hook_active)) map_over_chans(ss,update_graph,NULL);
}

static void Scale_Color_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_state *ss;
  float val;
  int scale_val;
  XmScaleCallbackStruct *cbs = (XmScaleCallbackStruct *)callData;
  color_chooser_info *cd = (color_chooser_info *)clientData;
  ss = cd->state;
  scale_val = cbs->value;
  if (scale_val <= 50) 
    val = (float)(scale_val+1)/51.0;
  else val = 1.0 + (float)(scale_val-50)*20.0;
  in_set_color_scale(ss,val);
  map_over_chans(ss,update_graph,NULL);
}

static void reflect_color_scale(float val)
{
  if (val<=1.0) 
    XmScaleSetValue(ccd->scale,(int)(val*51.0 - 1));
  else XmScaleSetValue(ccd->scale,(int)((val-1.0)/20.0 + 50.0));
}

void set_color_scale(snd_state *ss, float val)
{
  in_set_color_scale(ss,val);
  if (ccd) reflect_color_scale(color_scale(ss));
  if (!(ss->graph_hook_active)) map_over_chans(ss,update_graph,NULL);
}

static void List_Color_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_state *ss;
  XmListCallbackStruct *cbs = (XmListCallbackStruct *)callData;
  color_chooser_info *cd = (color_chooser_info *)clientData;
  ss = cd->state;
  in_set_color_map(ss,(cbs->item_position - 1));
  map_over_chans(ss,update_graph,NULL);
}

void set_color_map(snd_state *ss, int val)
{
  in_set_color_map(ss,val);
  if (ccd) XmListSelectPos(ccd->list,val+1,FALSE);
  if (!(ss->graph_hook_active)) map_over_chans(ss,update_graph,NULL);
}

static void Cutoff_Color_Callback(Widget w,XtPointer clientData,XtPointer callData) /* cutoff point */
{
  /* cutoff point for color chooser */
  snd_state *ss;
  XmScaleCallbackStruct *cbs = (XmScaleCallbackStruct *)callData;
  color_chooser_info *cd = (color_chooser_info *)clientData;
  ss = cd->state;
  in_set_color_cutoff(ss,(float)(cbs->value)/1000.0);
  map_over_chans(ss,update_graph,NULL);
}

void set_color_cutoff(snd_state *ss, float val)
{
  in_set_color_cutoff(ss,val);
  if (ccd) XmScaleSetValue(ccd->cutoff,(int)(val*1000.0));
  if (!(ss->graph_hook_active)) map_over_chans(ss,update_graph,NULL);
}


static void Dismiss_Color_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  color_chooser_info *cd = (color_chooser_info *)clientData;
  XtUnmanageChild(cd->dialog);
}

static void Help_Color_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_state *ss;
  color_chooser_info *cd = (color_chooser_info *)clientData;
  ss = cd->state;
  snd_help(ss,
       "View Color",
"This dialog sets the colormap and associated\n\
variables used during sonogram, spectrogram,\n\
and perhaps wavogram display. The cutoff scale refers\n\
to the minimum data value to be displayed.\n\
");	   
}

#define NUM_COLORMAPS 16
static char *colormaps[] = {"gray","hsv","hot","cool","bone","copper","pink","jet","prism",
			    "autumn","winter","spring","summer","colorcube","flag","lines"};
/* I tried a scrolled window with each colormap name in an appropriate color, but it looked kinda dumb */

void View_Color_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  Arg args[32];
  int n,i;
  XmString xhelp,xdismiss,xcutoff,xinvert,titlestr;
  XmString *cmaps;
  Widget mainform,list_label,light_label,sep,sep1;
  snd_state *ss = (snd_state *)clientData;
  if (!ccd)
    {
      /* create color chooser dialog window */
      ccd = (color_chooser_info *)CALLOC(1,sizeof(color_chooser_info));
      ccd->state = ss;

      xdismiss = XmStringCreate(STR_Dismiss,XmFONTLIST_DEFAULT_TAG); /* needed by template dialog */
      xhelp = XmStringCreate(STR_Help,XmFONTLIST_DEFAULT_TAG);
      titlestr = XmStringCreate(STR_Color_Editor,XmFONTLIST_DEFAULT_TAG);
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNcancelLabelString,xdismiss); n++;
      XtSetArg(args[n],XmNhelpLabelString,xhelp); n++;
      XtSetArg(args[n],XmNautoUnmanage,FALSE); n++;
      XtSetArg(args[n],XmNdialogTitle,titlestr); n++;
#if RESIZE_DIALOG
      XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
      XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
      XtSetArg(args[n],XmNtransient,FALSE); n++;
      ccd->dialog = XmCreateTemplateDialog(main_SHELL(ss),STR_Color,args,n);
      add_dialog(ss,ccd->dialog);
#if OVERRIDE_TOGGLE
      override_form_translation(ccd->dialog);
#endif

      XtAddCallback(ccd->dialog,XmNcancelCallback,Dismiss_Color_Callback,ccd);
      XtAddCallback(ccd->dialog,XmNhelpCallback,Help_Color_Callback,ccd);
      XmStringFree(xhelp);
      XmStringFree(xdismiss);
      XmStringFree(titlestr);

      if (!(ss->using_schemes))
	{
	  XtVaSetValues(XmMessageBoxGetChild(ccd->dialog,XmDIALOG_CANCEL_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(ccd->dialog,XmDIALOG_HELP_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	}

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,XmMessageBoxGetChild(ccd->dialog,XmDIALOG_SEPARATOR)); n++;
      mainform = sndCreateFormWidget("formd",ccd->dialog,args,n);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      list_label = XtCreateManagedWidget(STR_colormap,xmLabelWidgetClass,mainform,args,n);
      
      n=0;
      cmaps = (XmString *)CALLOC(NUM_COLORMAPS,sizeof(XmString));
      for (i=0;i<NUM_COLORMAPS;i++) cmaps[i] = XmStringCreate(colormaps[i],XmFONTLIST_DEFAULT_TAG);
#ifdef LESSTIF_VERSION
      if (!(ss->using_schemes)) n = background_white_color(args,n,ss);
#else
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
#endif
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,list_label); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNlistMarginWidth,3); n++;
      ccd->list = XmCreateScrolledList(mainform,"colormap-list",args,n);
      if (!(ss->using_schemes)) XtVaSetValues(ccd->list,XmNbackground,(ss->sgx)->white,NULL);
      XtVaSetValues(ccd->list,XmNitems,cmaps,XmNitemCount,NUM_COLORMAPS,XmNvisibleItemCount,6,NULL);
      XtAddCallback(ccd->list,XmNbrowseSelectionCallback,List_Color_Callback,ccd);
      for (i=0;i<NUM_COLORMAPS;i++) XmStringFree(cmaps[i]);
      FREE(cmaps);
      XtManageChild(ccd->list);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
#ifdef LESSTIF_VERSION
      XtSetArg(args[n],XmNrightWidget,XtParent(ccd->list)); n++;
#else
      XtSetArg(args[n],XmNrightWidget,ccd->list); n++;
#endif
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
#ifdef LESSTIF_VERSION
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
#else
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,XmMessageBoxGetChild(ccd->dialog,XmDIALOG_SEPARATOR)); n++;
#endif
      XtSetArg(args[n],XmNseparatorType,XmNO_LINE); n++;
      XtSetArg(args[n],XmNorientation,XmVERTICAL); n++;
      XtSetArg(args[n],XmNwidth,10); n++;
      sep = XtCreateManagedWidget("sep",xmSeparatorWidgetClass,mainform,args,n);

      /* this horizontal separator exists solely to keep the "light" label from clobbering the "dark" label! */
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNseparatorType,XmNO_LINE); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNwidth,250); n++;
      XtSetArg(args[n],XmNheight,10); n++;
      sep1 = XtCreateManagedWidget("sep1",xmSeparatorWidgetClass,mainform,args,n);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,sep); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,sep1); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNshowValue,TRUE); n++;
      XtSetArg(args[n],XmNvalue,50); n++;
      ccd->scale = XtCreateManagedWidget("ccdscl",xmScaleWidgetClass,mainform,args,n);
      XtAddCallback(ccd->scale,XmNvalueChangedCallback,Scale_Color_Callback,ccd);
      XtAddCallback(ccd->scale,XmNdragCallback,Scale_Color_Callback,ccd);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,ccd->scale); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      light_label = XtCreateManagedWidget(STR_light,xmLabelWidgetClass,mainform,args,n);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,sep); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,ccd->scale); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtCreateManagedWidget(STR_dark,xmLabelWidgetClass,mainform,args,n);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNrightWidget,sep); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,light_label); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNshowValue,TRUE); n++;
      XtSetArg(args[n],XmNmaximum,250); n++;
      XtSetArg(args[n],XmNdecimalPoints,3); n++;
      xcutoff = XmStringCreate(STR_cutoff,XmFONTLIST_DEFAULT_TAG);
      XtSetArg(args[n],XmNtitleString,xcutoff); n++;
      XtSetArg(args[n],XmNvalue,(int)(color_cutoff(ss) * 1000)); n++;
      ccd->cutoff = XtCreateManagedWidget("cutoff",xmScaleWidgetClass,mainform,args,n);
      XtAddCallback(ccd->cutoff,XmNvalueChangedCallback,Cutoff_Color_Callback,ccd);
      XtAddCallback(ccd->cutoff,XmNdragCallback,Cutoff_Color_Callback,ccd);
      XmStringFree(xcutoff);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      if (!(ss->using_schemes)) {XtSetArg(args[n],XmNselectColor,(ss->sgx)->pushed_button_color); n++;}
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,ccd->cutoff); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNset,color_inverted(ss)); n++;
      xinvert = XmStringCreate(STR_invert,XmFONTLIST_DEFAULT_TAG);
      XtSetArg(args[n],XmNlabelString,xinvert); n++;
      ccd->invert = sndCreateToggleButtonWidget(STR_invert,mainform,args,n);
      XtAddCallback(ccd->invert,XmNvalueChangedCallback,Invert_Color_Callback,ccd);
      XmStringFree(xinvert);
      if (color_scale(ss) != 1.0) reflect_color_scale(color_scale(ss));
    }
  else raise_dialog(ccd->dialog);
  if (!XtIsManaged(ccd->dialog)) XtManageChild(ccd->dialog);
}

int color_dialog_is_active(void)
{
  return((ccd) && (ccd->dialog) && (XtIsManaged(ccd->dialog)));
}


/* -------- orientation browser -------- */

typedef struct {
  Widget dialog;
  Widget ax,ay,az,sx,sy,sz,hop,cut; 
  snd_state *state;
} orientation_info;

static orientation_info *oid = NULL;

static void AX_Orientation_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss;
  XmScaleCallbackStruct *cbs = (XmScaleCallbackStruct *)callData;
  orientation_info *od = (orientation_info *)clientData;
  ss = od->state;
  in_set_spectro_x_angle(ss,(float)(cbs->value));
  map_over_chans(ss,update_graph,NULL);
}

void set_spectro_x_angle(snd_state *ss, float val)
{
  in_set_spectro_x_angle(ss,val);
  if (oid) XmScaleSetValue(oid->ax,(int)val);
  if (!(ss->graph_hook_active)) map_over_chans(ss,update_graph,NULL);
}

static void AX_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"x angle slider","This slider causes the graph to rotate\naround the x axis.\n");
}

static void AY_Orientation_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss;
  XmScaleCallbackStruct *cbs = (XmScaleCallbackStruct *)callData;
  orientation_info *od = (orientation_info *)clientData;
  ss = od->state;
  in_set_spectro_y_angle(ss,(float)(cbs->value));
  map_over_chans(ss,update_graph,NULL);
}

void set_spectro_y_angle(snd_state *ss, float val)
{
  in_set_spectro_y_angle(ss,val);
  if (oid) XmScaleSetValue(oid->ay,(int)val);
  if (!(ss->graph_hook_active)) map_over_chans(ss,update_graph,NULL);
}

static void AY_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"y angle slider","This slider causes the graph to rotate\naround the y axis.\n");
}

static void AZ_Orientation_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss;
  XmScaleCallbackStruct *cbs = (XmScaleCallbackStruct *)callData;
  orientation_info *od = (orientation_info *)clientData;
  ss = od->state;
  in_set_spectro_z_angle(ss,(float)(cbs->value));
  map_over_chans(ss,update_graph,NULL);
}

void set_spectro_z_angle(snd_state *ss, float val)
{
  in_set_spectro_z_angle(ss,val);
  if (oid) XmScaleSetValue(oid->az,(int)val);
  if (!(ss->graph_hook_active)) map_over_chans(ss,update_graph,NULL);
}

static void AZ_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"z angle slider","This slider causes the graph to rotate\naround the z axis.\n");
}

static void SX_Orientation_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss;
  XmScaleCallbackStruct *cbs = (XmScaleCallbackStruct *)callData;
  orientation_info *od = (orientation_info *)clientData;
  ss = od->state;
  in_set_spectro_x_scale(ss,(float)(cbs->value)*0.02);
  map_over_chans(ss,update_graph,NULL);
}

void set_spectro_x_scale(snd_state *ss, float val)
{
  in_set_spectro_x_scale(ss,val);
  if (oid) XmScaleSetValue(oid->sx,(int)(val*50));
  if (!(ss->graph_hook_active)) map_over_chans(ss,update_graph,NULL);
}

static void SX_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"x scale slider","This slider causes the graph to expand or\ncontract along the x axis.\n");
}

static void SY_Orientation_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss;
  XmScaleCallbackStruct *cbs = (XmScaleCallbackStruct *)callData;
  orientation_info *od = (orientation_info *)clientData;
  ss = od->state;
  in_set_spectro_y_scale(ss,(float)(cbs->value)*0.02);
  map_over_chans(ss,update_graph,NULL);
}

void set_spectro_y_scale(snd_state *ss, float val)
{
  in_set_spectro_y_scale(ss,val);
  if (oid) XmScaleSetValue(oid->sy,(int)(val*50));
  if (!(ss->graph_hook_active)) map_over_chans(ss,update_graph,NULL);
}

static void SY_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"y scale slider","This slider causes the graph to expand or\ncontract along the y axis.\n");
}

static void SZ_Orientation_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss;
  XmScaleCallbackStruct *cbs = (XmScaleCallbackStruct *)callData;
  orientation_info *od = (orientation_info *)clientData;
  ss = od->state;
  in_set_spectro_z_scale(ss,(float)(cbs->value)*0.01);
  map_over_chans(ss,update_graph,NULL);
}

void set_spectro_z_scale(snd_state *ss, float val)
{
  in_set_spectro_z_scale(ss,val);
  if (oid) XmScaleSetValue(oid->sz,(int)(val*100));
  if (!(ss->graph_hook_active)) map_over_chans(ss,update_graph,NULL);
}

static void SZ_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"z scale slider","This slider causes the graph to expand or\ncontract along the z axis.\n");
}

static void Hop_Orientation_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss;
  XmScaleCallbackStruct *cbs = (XmScaleCallbackStruct *)callData;
  orientation_info *od = (orientation_info *)clientData;
  ss = od->state;
  if (cbs->value <= 0) 
    in_set_spectro_hop(ss,1);
  else in_set_spectro_hop(ss,cbs->value);
  map_over_chans(ss,update_graph,NULL);
}

void set_spectro_hop(snd_state *ss, int val)
{
  if (val>0)
    {
      in_set_spectro_hop(ss,val);
      if (oid) XmScaleSetValue(oid->hop,val);
      if (!(ss->graph_hook_active)) map_over_chans(ss,update_graph,NULL);
    }
}

static void Hop_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"hop slider","This slider changes the hop size.\n");
}

static void Cut_Orientation_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* y axis limit */
  snd_state *ss;
  XmScaleCallbackStruct *cbs = (XmScaleCallbackStruct *)callData;
  orientation_info *od = (orientation_info *)clientData;
  ss = od->state;
  set_spectro_cutoff_and_redisplay(ss,(float)(cbs->value)*0.01); /* calls in_set... */
} 

void set_spectro_cutoff(snd_state *ss, float val)
{
  in_set_spectro_cutoff(ss,val);
  if (oid) XmScaleSetValue(oid->cut,(int)(val*100));
  if (!(ss->graph_hook_active)) map_over_chans(ss,update_graph,NULL);
}

static void Cut_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,"% of spectrum slider","This slider determines how much of\nthe spectrum is displayed\n");
}

static void Help_Orientation_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
       "View Orientation",
"This dialog sets the rotation and scaling\n\
variables used during sonogram, spectrogram,\n\
and wavogram display.\n\
");	   
}

static void Dismiss_Orientation_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  orientation_info *od = (orientation_info *)clientData;
  XtUnmanageChild(od->dialog);
}

static int fixup_angle(float ang)
{
  int na;
  na = (int)ang;
  na = na%360;
  if (na < 0) na+=360;
  return(na);
}

void reflect_spectro(snd_state *ss)
{
  /* set color/orientaton widget values */
  if (ccd) 
    {
      XmToggleButtonSetState(ccd->invert,color_inverted(ss),FALSE);
      XtVaSetValues(ccd->cutoff,XmNvalue,(int)((color_cutoff(ss))*1000),NULL);
      reflect_color_scale(color_scale(ss));
    }
  if (oid) 
    {
      XtVaSetValues(oid->ax,XmNvalue,fixup_angle(spectro_x_angle(ss)),NULL);
      XtVaSetValues(oid->ay,XmNvalue,fixup_angle(spectro_y_angle(ss)),NULL);
      XtVaSetValues(oid->az,XmNvalue,fixup_angle(spectro_z_angle(ss)),NULL);
      XtVaSetValues(oid->sx,XmNvalue,(int)(spectro_x_scale(ss) * 50),NULL);
      XtVaSetValues(oid->sy,XmNvalue,(int)(spectro_y_scale(ss) * 50),NULL);
      XtVaSetValues(oid->sz,XmNvalue,(int)(spectro_z_scale(ss) * 100),NULL);
      XtVaSetValues(oid->hop,XmNvalue,(spectro_hop(ss) > 100) ? 100 : (spectro_hop(ss)),NULL);
      XtVaSetValues(oid->cut,XmNvalue,(int)(spectro_cutoff(ss) * 100),NULL);
    }
}

static void Reset_Orientation_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss;
  orientation_info *od = (orientation_info *)clientData;
  /* put everything back the way it was at the start */
  ss = od->state;
  reset_spectro(ss);
  reflect_spectro(ss);
  map_over_chans(ss,update_graph,NULL);
}

void View_Orientation_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_state *ss = (snd_state *)clientData;
  Widget mainform,rightbox,leftbox;
  XmString xdismiss,xhelp,xstr,xreset,titlestr;
  int n;
  Arg args[20];
  if (!oid)
    {
      /* create orientation window */
      oid = (orientation_info *)CALLOC(1,sizeof(orientation_info));
      oid->state = ss;

      xdismiss = XmStringCreate(STR_Dismiss,XmFONTLIST_DEFAULT_TAG); /* needed by template dialog */
      xhelp = XmStringCreate(STR_Help,XmFONTLIST_DEFAULT_TAG);
      xreset = XmStringCreate(STR_Reset,XmFONTLIST_DEFAULT_TAG);
      titlestr = XmStringCreate(STR_Spectrogram_Orientation,XmFONTLIST_DEFAULT_TAG);
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNcancelLabelString,xdismiss); n++;
      XtSetArg(args[n],XmNhelpLabelString,xhelp); n++;
      XtSetArg(args[n],XmNokLabelString,xreset); n++;
      XtSetArg(args[n],XmNautoUnmanage,FALSE); n++;
      XtSetArg(args[n],XmNdialogTitle,titlestr); n++;
#if RESIZE_DIALOG
      XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
      XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
      XtSetArg(args[n],XmNtransient,FALSE); n++;
      oid->dialog = XmCreateTemplateDialog(main_SHELL(ss),STR_Orientation,args,n);
      add_dialog(ss,oid->dialog);
#if OVERRIDE_TOGGLE
      override_form_translation(oid->dialog);
#endif


      XtAddCallback(oid->dialog,XmNcancelCallback,Dismiss_Orientation_Callback,oid);
      XtAddCallback(oid->dialog,XmNhelpCallback,Help_Orientation_Callback,ss);
      XtAddCallback(oid->dialog,XmNokCallback,Reset_Orientation_Callback,oid);
      XmStringFree(xhelp);
      XmStringFree(xdismiss);
      XmStringFree(titlestr);
      XmStringFree(xreset);

      if (!(ss->using_schemes))
	{
	  XtVaSetValues(XmMessageBoxGetChild(oid->dialog,XmDIALOG_OK_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(oid->dialog,XmDIALOG_CANCEL_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(oid->dialog,XmDIALOG_HELP_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	}

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNbottomWidget,XmMessageBoxGetChild(oid->dialog,XmDIALOG_SEPARATOR)); n++;
      mainform = sndCreateFormWidget("formd",oid->dialog,args,n);
      
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_POSITION); n++;
      XtSetArg(args[n],XmNrightPosition,50); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNorientation,XmVERTICAL); n++;
      leftbox = sndCreateRowColumnWidget("leftb",mainform,args,n);
      
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,leftbox); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNorientation,XmVERTICAL); n++;
      rightbox = sndCreateRowColumnWidget("rightb",mainform,args,n);
      
      /* left box */
      n=0;
      xstr = XmStringCreate(STR_x_angle,XmFONTLIST_DEFAULT_TAG);
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNshowValue,TRUE); n++;
      XtSetArg(args[n],XmNvalue,fixup_angle(spectro_x_angle(ss))); n++;
      XtSetArg(args[n],XmNmaximum,360); n++;
      XtSetArg(args[n],XmNtitleString,xstr); n++;
      oid->ax = XtCreateManagedWidget("ax",xmScaleWidgetClass,leftbox,args,n);
      XtAddCallback(oid->ax,XmNvalueChangedCallback,AX_Orientation_Callback,oid);
      XtAddCallback(oid->ax,XmNdragCallback,AX_Orientation_Callback,oid);
      XtAddCallback(oid->ax,XmNhelpCallback,AX_Help_Callback,ss);
      XmStringFree(xstr);

      n=0;
      xstr = XmStringCreate(STR_y_angle,XmFONTLIST_DEFAULT_TAG);
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNshowValue,TRUE); n++;
      XtSetArg(args[n],XmNvalue,fixup_angle(spectro_y_angle(ss))); n++;
      XtSetArg(args[n],XmNmaximum,360); n++;
      XtSetArg(args[n],XmNtitleString,xstr); n++;
      oid->ay = XtCreateManagedWidget("ay",xmScaleWidgetClass,leftbox,args,n);
      XtAddCallback(oid->ay,XmNvalueChangedCallback,AY_Orientation_Callback,oid);
      XtAddCallback(oid->ay,XmNdragCallback,AY_Orientation_Callback,oid);
      XtAddCallback(oid->ay,XmNhelpCallback,AY_Help_Callback,ss);
      XmStringFree(xstr);

      n=0;
      xstr = XmStringCreate(STR_z_angle,XmFONTLIST_DEFAULT_TAG);
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNshowValue,TRUE); n++;
      XtSetArg(args[n],XmNvalue,fixup_angle(spectro_z_angle(ss))); n++;
      XtSetArg(args[n],XmNmaximum,360); n++;
      XtSetArg(args[n],XmNtitleString,xstr); n++;
      oid->az = XtCreateManagedWidget("az",xmScaleWidgetClass,leftbox,args,n);
      XtAddCallback(oid->az,XmNvalueChangedCallback,AZ_Orientation_Callback,oid);
      XtAddCallback(oid->az,XmNdragCallback,AZ_Orientation_Callback,oid);
      XtAddCallback(oid->az,XmNhelpCallback,AZ_Help_Callback,ss);
      XmStringFree(xstr);

      n=0;
      xstr = XmStringCreate(STR_hop,XmFONTLIST_DEFAULT_TAG);
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNshowValue,TRUE); n++;
      XtSetArg(args[n],XmNvalue,(spectro_hop(ss) > 20) ? 20 : (spectro_hop(ss))); n++;
      XtSetArg(args[n],XmNmaximum,20); n++;
      XtSetArg(args[n],XmNtitleString,xstr); n++;
      oid->hop = XtCreateManagedWidget("hop",xmScaleWidgetClass,leftbox,args,n);
      XtAddCallback(oid->hop,XmNvalueChangedCallback,Hop_Orientation_Callback,oid);
      XtAddCallback(oid->hop,XmNdragCallback,Hop_Orientation_Callback,oid);
      XtAddCallback(oid->hop,XmNhelpCallback,Hop_Help_Callback,ss);
      XmStringFree(xstr);

      /* right box */
      n=0;
      xstr = XmStringCreate(STR_x_scale,XmFONTLIST_DEFAULT_TAG);
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNshowValue,TRUE); n++;
      XtSetArg(args[n],XmNvalue,(int)(spectro_x_scale(ss) * 50)); n++;
      XtSetArg(args[n],XmNtitleString,xstr); n++;
      oid->sx = XtCreateManagedWidget("xs",xmScaleWidgetClass,rightbox,args,n);
      XtAddCallback(oid->sx,XmNvalueChangedCallback,SX_Orientation_Callback,oid);
      XtAddCallback(oid->sx,XmNdragCallback,SX_Orientation_Callback,oid);
      XtAddCallback(oid->sx,XmNhelpCallback,SX_Help_Callback,ss);
      XmStringFree(xstr);

      n=0;
      xstr = XmStringCreate(STR_y_scale,XmFONTLIST_DEFAULT_TAG);
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNshowValue,TRUE); n++;
      XtSetArg(args[n],XmNvalue,(int)(spectro_y_scale(ss) * 50)); n++;
      XtSetArg(args[n],XmNtitleString,xstr); n++;
      oid->sy = XtCreateManagedWidget("ys",xmScaleWidgetClass,rightbox,args,n);
      XtAddCallback(oid->sy,XmNvalueChangedCallback,SY_Orientation_Callback,oid);
      XtAddCallback(oid->sy,XmNdragCallback,SY_Orientation_Callback,oid);
      XtAddCallback(oid->sy,XmNhelpCallback,SY_Help_Callback,ss);
      XmStringFree(xstr);

      n=0;
      xstr = XmStringCreate(STR_z_scale,XmFONTLIST_DEFAULT_TAG);
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNshowValue,TRUE); n++;
      XtSetArg(args[n],XmNvalue,(int)(spectro_z_scale(ss) * 100)); n++;
      XtSetArg(args[n],XmNtitleString,xstr); n++;
      oid->sz = XtCreateManagedWidget("zs",xmScaleWidgetClass,rightbox,args,n);
      XtAddCallback(oid->sz,XmNvalueChangedCallback,SZ_Orientation_Callback,oid);
      XtAddCallback(oid->sz,XmNdragCallback,SZ_Orientation_Callback,oid);
      XtAddCallback(oid->sz,XmNhelpCallback,SZ_Help_Callback,ss);
      XmStringFree(xstr);

      n=0;
      xstr = XmStringCreate(STR_percent_of_spectrum,XmFONTLIST_DEFAULT_TAG);
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      XtSetArg(args[n],XmNshowValue,TRUE); n++;
      XtSetArg(args[n],XmNvalue,(int)(spectro_cutoff(ss) * 100)); n++;
      XtSetArg(args[n],XmNtitleString,xstr); n++;
      oid->cut = XtCreateManagedWidget("cut",xmScaleWidgetClass,rightbox,args,n);
      XtAddCallback(oid->cut,XmNvalueChangedCallback,Cut_Orientation_Callback,oid);
      XtAddCallback(oid->cut,XmNdragCallback,Cut_Orientation_Callback,oid);
      XtAddCallback(oid->cut,XmNhelpCallback,Cut_Help_Callback,ss);
      XmStringFree(xstr);
    }
  else raise_dialog(oid->dialog);
  if (!XtIsManaged(oid->dialog)) XtManageChild(oid->dialog);
}

int orientation_dialog_is_active(void)
{
  return((oid) && (oid->dialog) && (XtIsManaged(oid->dialog)));
}

void start_color_dialog(snd_state *ss, int width, int height)
{
  View_Color_Callback(NULL,(XtPointer)ss,NULL);
  if (width != 0) XtVaSetValues(ccd->dialog,XmNwidth,(Dimension)width,XmNheight,(Dimension)height,NULL);
}

void start_orientation_dialog(snd_state *ss, int width, int height)
{
  View_Orientation_Callback(NULL,(XtPointer)ss,NULL);
  if (width != 0) XtVaSetValues(oid->dialog,XmNwidth,(Dimension)width,XmNheight,(Dimension)height,NULL);
}

void start_file_dialog(snd_state *ss, int width, int height)
{
  View_Files_Callback(NULL,(XtPointer)ss,NULL);
  if (width > 0) XtVaSetValues(fbd->dialog,XmNwidth,(Dimension)width,XmNheight,(Dimension)height,NULL);
}

int file_dialog_is_active(void)
{
  return((fbd) && (fbd->dialog) && (XtIsManaged(fbd->dialog)));
}


static Widget file_mix_dialog = NULL;
static Widget file_mix_name = NULL;

static void file_mix_help_callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_help((snd_state *)clientData,
	   "File Mix",
"The file you specify to the mix file prompt\n\
will be mixed into the current active sound at\n\
the current cursor location of the active channel.\n\
The equivalent keyboard command is C-x C-q.\n\
");
}

static void file_mix_cancel_callback(Widget w,XtPointer clientData,XtPointer callData)
{
  XtUnmanageChild(w);
}

static void file_mix_ok_callback(Widget w,XtPointer clientData,XtPointer callData)
{
  char *str = NULL;
  XtUnmanageChild(w);
  mix_complete_file(any_selected_sound((snd_state *)clientData),str=XmTextGetString(file_mix_name),"File: mix");
  if (str) FREE(str);
}

void File_Mix_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  Arg args[20];
  int n;
  XmString s1;
  Widget wtmp;
  snd_state *ss = (snd_state *)clientData;
  if (!file_mix_dialog)
    {
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
#if RESIZE_DIALOG
      XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
      XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
      s1 = XmStringCreate(STR_mix_in_p,XmFONTLIST_DEFAULT_TAG);
      XtSetArg(args[n],XmNselectionLabelString,s1); n++;
      file_mix_dialog = XmCreateFileSelectionDialog(w,STR_mix_file_p,args,n);
#if OVERRIDE_TOGGLE
      override_form_translation(file_mix_dialog);
#endif
      XmStringFree(s1);
      XtAddCallback(file_mix_dialog,XmNhelpCallback,file_mix_help_callback,ss);
      XtAddCallback(file_mix_dialog,XmNcancelCallback,file_mix_cancel_callback,ss);
      XtAddCallback(file_mix_dialog,XmNokCallback,file_mix_ok_callback,ss);
      XtManageChild(file_mix_dialog);
      color_file_selection_box(file_mix_dialog,ss);
#ifndef LESSTIF_VERSION
      file_mix_name = XtNameToWidget(file_mix_dialog,"Text");
      if (!file_mix_name) file_mix_name = XmFileSelectionBoxGetChild(file_mix_dialog,XmDIALOG_TEXT);
#else
      file_mix_name = XmFileSelectionBoxGetChild(file_mix_dialog,XmDIALOG_TEXT);
#endif
      if (file_mix_name) add_completer_to_textfield(ss,file_mix_name,add_completer_func(filename_completer));

#ifndef LESSTIF_VERSION
      wtmp = XtNameToWidget(file_mix_dialog,"FilterText");
      if (!wtmp) wtmp = XmFileSelectionBoxGetChild(file_mix_dialog,XmDIALOG_FILTER_TEXT);
#else
      wtmp = XmFileSelectionBoxGetChild(file_mix_dialog,XmDIALOG_FILTER_TEXT);
#endif
      if (wtmp) add_completer_to_textfield(ss,wtmp,add_completer_func(filename_completer));

    }
  if (!XtIsManaged(file_mix_dialog)) XtManageChild(file_mix_dialog);
}

void snd_edit_save_as_dialog(Widget w, snd_state *ss)
{
  XmString xmstr2;
  finish_keyboard_selection();
  if (!save_as_dialog)
    {
      save_as_dialog = (save_as_info *)CALLOC(1,sizeof(save_as_info));
      save_as_dialog->state = ss;
      make_save_as_dialog(w,save_as_dialog,STR_current_selection,NULL);
    }
  else
    {
      sprintf(file_string,STR_saving,STR_current_selection);
      xmstr2 = XmStringCreate(file_string,XmFONTLIST_DEFAULT_TAG);
      XtVaSetValues(save_as_dialog->dialog,XmNmessageString,xmstr2,NULL);
      XmStringFree(xmstr2);
      load_header_and_data_lists(save_as_dialog->header_list,save_as_dialog->data_list,
				 0,1,
				 &(save_as_dialog->header_choice),&(save_as_dialog->data_choice));
    }
  save_as_dialog->type = EDIT_SAVE_AS;
  if (!XtIsManaged(save_as_dialog->dialog)) XtManageChild(save_as_dialog->dialog);
}


/* ---------------- EDIT_HEADER ---------------- */

static Widget edit_header_dialog = NULL;
static char edit_string[128];
static Widget *edit_header_data;

static void edit_header_help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
       STR_Edit_Header,
"This dialog edits the header of a sound file.\n\
No change is made to the actual sound data; the\n\
new header is blindly written, any unsaved edits\n\
are ignored. If you specify 'raw' as the type,\n\
any existing header is removed.  This dialog is\n\
aimed at adding or removing an entire header, \n\
or editing the header comments; anything else\n\
is obviously dangerous.\n\
");
}

static void edit_header_cancel_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  XtUnmanageChild(edit_header_dialog);
}

#define RIPPLE_SIZE 65536
/* needs to be big enough to accomodate any newly added header or header comments */

static void edit_header_ok_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  unsigned char *ripple0,*ripple1,*zerobuf;
  int fd,err,chans,srate,loc,comlen,type,format,bytes0,bytes1,curloc,readloc,writeloc,curbytes,totalbytes;
  char *comment,*str;
  snd_state *ss;
  file_info *hdr;
  snd_info *sp = (snd_info *)clientData;
  XmAnyCallbackStruct *cb = (XmAnyCallbackStruct *)callData;
  ss = sp->state;
  if (cb->event == ((ss->sgx)->text_activate_event)) return; /* <cr> in one of text fields */
  if (!(sp->read_only))
    {
#ifndef _MSC_VER
      err = access(sp->fullname,W_OK);
#else
      err = 0;
#endif
      if (err == 0)
	{
	  fd = open(sp->fullname,O_RDWR,0);
	  if (fd != -1)
	    {
	      hdr = sp->hdr;
	      ripple0 = (unsigned char *)CALLOC(RIPPLE_SIZE,sizeof(unsigned char));
	      ripple1 = (unsigned char *)CALLOC(RIPPLE_SIZE,sizeof(unsigned char));
	      lseek(fd,hdr->data_location,SEEK_SET);
	      bytes0 = read(fd,ripple0,RIPPLE_SIZE);
	      if (bytes0 == RIPPLE_SIZE) bytes1 = read(fd,ripple1,RIPPLE_SIZE); else bytes1 = -1;
	      srate = hdr->srate;
	      chans = hdr->chans;
	      type = hdr->type;
	      format = hdr->format;
	      read_file_data_choices(edit_header_data[fdata_stext], edit_header_data[fdata_ctext], 
				     edit_header_data[fdata_hlist], edit_header_data[fdata_dlist], 
				     &srate, &chans, &type, &format);
	      str = XmTextGetString(edit_header_data[fdata_loctext]); 
	      if (str) {sscanf(str,"%d",&loc); FREE(str);} else loc = hdr->data_location;
	      comment = XmTextGetString(edit_header_data[fdata_comment_text]);
	      comlen = snd_strlen(comment);
	      curloc = 0;
	      curbytes = 0;
	      totalbytes = hdr->samples * mus_format2bytes(hdr->format);
	      lseek(fd,0,SEEK_SET);
	      if (type != raw_sound_file)
		{
		  mus_write_header_with_fd(fd,type,srate,chans,loc,hdr->samples,format,comment,comlen);
		  curloc = mus_header_data_location();
		  if ((loc != curloc) && (loc != hdr->data_location)) /* user changed it */
		    {
		      /* pad if possible ? */
		      if (loc > curloc)
			{
			  zerobuf = (unsigned char *)CALLOC(loc-curloc,sizeof(unsigned char));
			  write(fd,zerobuf,loc-curloc);
			  FREE(zerobuf);
			  curloc = loc;
			}
		    }
		}
	      readloc = RIPPLE_SIZE * 2;
	      writeloc = curloc;
	      if (writeloc > readloc) snd_error("%s[%d] %s: writeloc > readloc!",__FILE__,__LINE__,__FUNCTION__);
	      while (bytes0 > 0)
		{
		  write(fd,ripple0,bytes0);
		  curbytes += bytes0;
		  writeloc += RIPPLE_SIZE;
		  if (bytes1 > 0)
		    {
		      lseek(fd,readloc,SEEK_SET);
		      readloc += RIPPLE_SIZE;
		      bytes0 = read(fd,ripple0,RIPPLE_SIZE);
		      lseek(fd,writeloc,SEEK_SET);
		      writeloc += RIPPLE_SIZE;
		      write(fd,ripple1,bytes1);
		      curbytes += bytes1;
		      if (bytes0 > 0)
			{
			  lseek(fd,readloc,SEEK_SET);
			  readloc += RIPPLE_SIZE;
			  bytes1 = read(fd,ripple1,RIPPLE_SIZE);
			}
		    }
		  if (curbytes > totalbytes) break; /* ?? this should not happen */
		}
	      close(fd);
	      clear_minibuffer(sp);
	      snd_file_bomb_icon(sp,TRUE);
	      FREE(ripple0);
	      FREE(ripple1);
	      FREE(comment);
	      FREE(str);
	      if (auto_update(ss)) map_over_sounds(ss,snd_not_current,NULL);
	    }
	  else 
	    snd_error(STR_cant_open,sp->shortname);
	}
      else 
	snd_error(STR_cant_write,sp->shortname);
    }
  else
    snd_error(STR_is_write_protected,sp->shortname);
  XtUnmanageChild(edit_header_dialog);
}

static int current_type_pos,current_format_pos;

static void edit_header_type_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  int pos;
  file_info *hdr;
  XmListCallbackStruct *cbs = (XmListCallbackStruct *)callData;
  pos = cbs->item_position;
  if (current_type_pos != pos)
    {
      hdr = (file_info *)CALLOC(1,sizeof(file_info));
      hdr->comment = NULL;
      switch (pos)
	{
	case NEXT_POSITION: hdr->type = NeXT_sound_file; hdr->format = SNDLIB_16_LINEAR; break;
	case NIST_POSITION: hdr->type = NIST_sound_file; hdr->format = SNDLIB_16_LINEAR; break;
	case AIFC_POSITION: hdr->type = AIFF_sound_file; hdr->format = SNDLIB_16_LINEAR; break;
	case RIFF_POSITION: hdr->type = RIFF_sound_file; hdr->format = SNDLIB_16_LINEAR_LITTLE_ENDIAN; break;
	case IRCAM_POSITION: hdr->type = IRCAM_sound_file; hdr->format = SNDLIB_16_LINEAR; break;
	case RAW_POSITION: hdr->type = raw_sound_file; hdr->format = SNDLIB_16_LINEAR; break;
	case AIFF_POSITION: hdr->type = old_style_AIFF_sound_file; hdr->format = SNDLIB_16_LINEAR; break;
	}
      load_header_and_data_lists(edit_header_data[fdata_hlist], edit_header_data[fdata_dlist],hdr->type,hdr->format,&current_type_pos,&current_format_pos);
      FREE(hdr);
    }
}

static void edit_header_format_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  XmListCallbackStruct *cbs = (XmListCallbackStruct *)callData;
  current_format_pos = cbs->item_position;
}

void edit_header(snd_info *sp)
{
  /* like display-info, but writable.
   * need fields for srate, channels, type, format, data location, comment
   * if any are changed, need save button, cancel button, dismiss (leave unsaved but pending), reflect (change Snd display, not file)
   * this means the Snd-effective header is separate from the in-file header even across saves??
   */
  XmString xstr1,xstr2,xstr3,xstr4,titlestr;
  int n;
  Arg args[20];
  snd_state *ss;
  file_info *hdr;
  char *tmpstr = NULL;
  if (!sp) return;
  ss = sp->state;
  hdr = sp->hdr;

  if (!edit_header_dialog)
    {
      n=0;
      xstr1 = XmStringCreate(STR_Cancel,XmFONTLIST_DEFAULT_TAG); /* needed by template dialog */
      xstr2 = XmStringCreate(STR_Help,XmFONTLIST_DEFAULT_TAG);
      xstr3 = XmStringCreate(STR_Save,XmFONTLIST_DEFAULT_TAG);
      titlestr = XmStringCreate(STR_Edit_Header,XmFONTLIST_DEFAULT_TAG);
      sprintf(edit_string,STR_Edit_header_of,sp->shortname);
      xstr4 = XmStringCreate(edit_string,XmFONTLIST_DEFAULT_TAG);
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNcancelLabelString,xstr1); n++;
      XtSetArg(args[n],XmNhelpLabelString,xstr2); n++;
      XtSetArg(args[n],XmNokLabelString,xstr3); n++;
      XtSetArg(args[n],XmNmessageString,xstr4); n++;
      XtSetArg(args[n],XmNdialogTitle,titlestr); n++;
      XtSetArg(args[n],XmNautoUnmanage,FALSE); n++;
#if RESIZE_DIALOG
      XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
      XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
      XtSetArg(args[n],XmNtransient,FALSE); n++;
      edit_header_dialog = XmCreateTemplateDialog(main_SHELL(ss),STR_Edit_Header,args,n);
      add_dialog(ss,edit_header_dialog);
#if OVERRIDE_TOGGLE
      override_form_translation(edit_header_dialog);
#endif

      XtAddCallback(edit_header_dialog,XmNcancelCallback,edit_header_cancel_Callback,ss);
      XtAddCallback(edit_header_dialog,XmNhelpCallback,edit_header_help_Callback,ss);
      XtAddCallback(edit_header_dialog,XmNokCallback,edit_header_ok_Callback,sp);
      XmStringFree(xstr1);
      XmStringFree(xstr2);
      XmStringFree(xstr3);
      XmStringFree(xstr4);
      XmStringFree(titlestr);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      edit_header_data = sndCreateFileDataForm(ss,edit_header_dialog,STR_Edit_Header,args,n,TRUE,hdr->type,hdr->format,TRUE);
      load_header_and_data_lists(edit_header_data[fdata_hlist],edit_header_data[fdata_dlist],hdr->type,hdr->format,&current_type_pos,&current_format_pos);
      XtVaSetValues(edit_header_data[fdata_comment_text],XmNvalue,(hdr->comment) ? (hdr->comment) : "",NULL);
      sprintf(edit_string,"%d",hdr->data_location);
      XtVaSetValues(edit_header_data[fdata_loctext],XmNvalue,edit_string,NULL);
      sprintf(edit_string,"%d",hdr->srate);
      XtVaSetValues(edit_header_data[fdata_stext],XmNvalue,edit_string,NULL);
      sprintf(edit_string,"%d",hdr->chans);
      XtVaSetValues(edit_header_data[fdata_ctext],XmNvalue,edit_string,NULL);
      XmTextSetString(edit_header_data[fdata_comment_text],tmpstr = sound_comment(sp->fullname));
      if (tmpstr) FREE(tmpstr);
      XtAddCallback(edit_header_data[fdata_hlist],XmNbrowseSelectionCallback,edit_header_type_Callback,NULL);
      XtAddCallback(edit_header_data[fdata_dlist],XmNbrowseSelectionCallback,edit_header_format_Callback,NULL);

#if MANAGE_DIALOG
      XtManageChild(edit_header_dialog);
#endif
      if (!(ss->using_schemes)) 
	{
	  XtVaSetValues(XmMessageBoxGetChild(edit_header_dialog,XmDIALOG_OK_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(edit_header_dialog,XmDIALOG_CANCEL_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(edit_header_dialog,XmDIALOG_HELP_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  map_over_children(edit_header_dialog,set_main_color_of_widget,ss);
	  XtVaSetValues(edit_header_data[fdata_hlist],XmNbackground,(ss->sgx)->white,NULL);
	  XtVaSetValues(edit_header_data[fdata_dlist],XmNbackground,(ss->sgx)->white,NULL);
	}
    }
  else raise_dialog(edit_header_dialog);
  if (!(XtIsManaged(edit_header_dialog))) XtManageChild(edit_header_dialog);
}



/* -------- STATS WINDOW -------- */

#include <malloc.h>

static char *kmg (int num)
{
  /* return number 0..1024, then in terms of K, M, G */
  char *str;
  str = (char *)CALLOC(16,sizeof(char));
  if (num > 1024)
    {
      if (num > (1024*1024))
	{
	  if (num > (1024*1024*1024))
	    sprintf(str,"%.3fG",(float)num/(float)(1024*1024*1024));
	  else sprintf(str,"%.2fM",(float)num/(float)(1024*1024));
	}
      else sprintf(str,"%.1fK",(float)num/1024.0);
    }
  else sprintf(str,"%d",num);
  return(str);
}

static Widget stats_window = NULL;

static void stats_help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
       STR_Disk_and_Memory_Usage,
"This window gives an approximate notion of how\n\
much memory (RAM) and disk space each channel is\n\
taking up.  As a channel is edited, the relevant\n\
data is saved either in arrays or temporary files.\n\
The number of bytes in these arrays, and the number\n\
of such arrays are the first two numbers; then\n\
comes the space in bytes the channel takes up in\n\
the main (presumably permanent) file; the next\n\
three numbers give the number of bytes in the\n\
temporary files, the number of such files, and\n\
the number of these files that are currently\n\
being held open.  The related variable is\n\
" S_show_usage_stats ".  The 'Update' button forces\n\
the stats to be regathered, in case the display\n\
somehow gets out of sync with the actual data.\n\
\n\
");
}

static void stats_dismiss_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  set_show_usage_stats((snd_state *)clientData,FALSE);
}

static void stats_update_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  update_all_usage_stats(ss);
  check_stats_window(ss,TRUE);
}

static Widget stats_form;

static char *update_chan_stats(chan_info *cp)
{
  char *desc;
  char *vals[6];
  int i;
  desc = (char *)CALLOC(256,sizeof(char));
  vals[0] = kmg(cp->stats[ARRAY_USAGE]);
  vals[1] = kmg(cp->stats[ARRAYS_ACTIVE]);
  vals[2] = kmg(cp->stats[FILE_USAGE]);
  vals[3] = kmg(cp->stats[TEMP_USAGE]);
  vals[4] = kmg(cp->stats[TEMPS_ACTIVE]);
  vals[5] = kmg(cp->stats[TEMPS_OPEN]);
  if (cp->stats[TEMPS_ACTIVE] != cp->stats[TEMPS_OPEN])
    sprintf(desc,"%s %d: %s (%s), %s, %s (%s, %s)\n",(cp->sound)->shortname,cp->chan + 1,
	    vals[0],vals[1],vals[2],vals[3],vals[4],vals[5]);
  else
    sprintf(desc,"%s %d: %s (%s), %s, %s (%s)\n",(cp->sound)->shortname,cp->chan + 1,
	    vals[0],vals[1],vals[2],vals[3],vals[4]);
  for (i=0;i<6;i++) FREE(vals[i]);
  return(desc);
}

#ifdef DEBUG_MEMORY
  char *mem_stats(snd_state *ss,int ub);
#endif

void update_stats(snd_state *ss)
{
  int i,j,pos,regs,used_bytes=0;
  int vals[2];
  snd_info *sp;
  chan_info *cp;
  char *str,*r0 = NULL,*r1 = NULL;
  XmTextSetString(stats_form,"file chn: mem(#bufs), main, temp(#files)\n\n");
  for (i=0;i<ss->max_sounds;i++)
    {
      if ((sp=((snd_info *)(ss->sounds[i]))))
	{
	  if (sp->inuse)
	    {
	      for (j=0;j<(sp->nchans);j++)
		{
		  if ((cp=((chan_info *)(sp->chans[j]))))
		    {
		      if (cp->stats)
			{
			  pos = XmTextGetLastPosition(stats_form);
			  str = update_chan_stats(cp);
			  XmTextInsert(stats_form,pos,str);
			  FREE(str);
			}}}}}}
  regs = snd_regions();
  if (regs > 0)
    {
      region_stats(vals);
      str = (char *)CALLOC(256,sizeof(char));
      sprintf(str,"\nregions (%d): %s + %s\n",regs,r0=kmg(vals[0]),r1=kmg(vals[1]));
      pos = XmTextGetLastPosition(stats_form);
      XmTextInsert(stats_form,pos,str);
      if (r0) FREE(r0);
      if (r1) FREE(r1);
      FREE(str);
    }
#if HAVE_MALLINFO
  {
    struct mallinfo mall;
    char *m0=NULL,*m1=NULL,*m2=NULL,*m3=NULL;
    mall = mallinfo();
    str = (char *)CALLOC(256,sizeof(char));
    sprintf(str,"\nmalloc: %s + %s (in use: %s, freed: %s)\n",
	    m0=kmg(mall.arena),m1=kmg(mall.hblkhd),
	    m2=kmg(used_bytes=mall.uordblks),m3=kmg(mall.fordblks));
    pos = XmTextGetLastPosition(stats_form);
    XmTextInsert(stats_form,pos,str);
    if (m0) FREE(m0);
    if (m1) FREE(m1);
    if (m2) FREE(m2);
    if (m3) FREE(m3);
    FREE(str);
  }
#endif
#ifdef DEBUG_MEMORY
  str = mem_stats(ss,used_bytes);
  pos = XmTextGetLastPosition(stats_form);
  XmTextInsert(stats_form,pos,str);
  free(str);
#endif
}

void update_stats_display(snd_state *ss, int all)
{
  /* dismiss update help -- update forces recalc of all stats */
  XmString xstr1,xstr2,xstr3,titlestr;
  int n;
  Arg args[20];

  if (!stats_window)
    {
      n=0;
      xstr1 = XmStringCreate(STR_Dismiss,XmFONTLIST_DEFAULT_TAG);
      xstr2 = XmStringCreate(STR_Help,XmFONTLIST_DEFAULT_TAG);
      xstr3 = XmStringCreate(STR_Update,XmFONTLIST_DEFAULT_TAG);
      titlestr = XmStringCreate(STR_Disk_and_Memory_Usage,XmFONTLIST_DEFAULT_TAG);
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNcancelLabelString,xstr3); n++;
      XtSetArg(args[n],XmNhelpLabelString,xstr2); n++;
      XtSetArg(args[n],XmNokLabelString,xstr1); n++;
      XtSetArg(args[n],XmNdialogTitle,titlestr); n++;
      XtSetArg(args[n],XmNautoUnmanage,FALSE); n++;
#if RESIZE_DIALOG
      XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
      XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
      XtSetArg(args[n],XmNtransient,FALSE); n++;
      stats_window = XmCreateTemplateDialog(main_SHELL(ss),STR_Disk_and_Memory_Usage,args,n);
      add_dialog(ss,stats_window);
#if OVERRIDE_TOGGLE
      override_form_translation(stats_window);
#endif

      XtAddCallback(stats_window,XmNcancelCallback,stats_update_Callback,ss);
      XtAddCallback(stats_window,XmNhelpCallback,stats_help_Callback,ss);
      XtAddCallback(stats_window,XmNokCallback,stats_dismiss_Callback,ss);
      XmStringFree(xstr1);
      XmStringFree(xstr2);
      XmStringFree(xstr3);
      XmStringFree(titlestr);

      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNheight,60); n++;
      XtSetArg(args[n],XmNwidth,300); n++;
      stats_form = sndCreateTextWidget(ss,"stats",stats_window,args,n);

#if MANAGE_DIALOG
      XtManageChild(stats_window);
#endif
      if (!(ss->using_schemes)) 
	{
	  XtVaSetValues(XmMessageBoxGetChild(stats_window,XmDIALOG_OK_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(stats_window,XmDIALOG_CANCEL_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(stats_window,XmDIALOG_HELP_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  map_over_children(stats_window,set_main_color_of_widget,ss);
	  XtVaSetValues(stats_form,XmNbackground,(ss->sgx)->white,NULL);
	}
    }
  else raise_dialog(stats_window);
  if (all) update_all_usage_stats(ss);
  update_stats(ss);
  if (!(XtIsManaged(stats_window))) XtManageChild(stats_window);
}

void check_stats_window(snd_state *ss, int val)
{
  /* if val==0, close active display if any, if val==1, open and spin through all current chans setting/gathering */
  if (val == 0)
    {
      if ((stats_window) && (XtIsManaged(stats_window)))
	XtUnmanageChild(stats_window);
    }
  else
    {
      update_stats_display(ss,TRUE);
    }
}

