#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "snack.h"

#if defined Linux || defined WIN || defined _LITTLE_ENDIAN
#  define LE
#endif

extern struct jkFileFormat *Snack_GetFileFormats();
static struct jkFileFormat *snackFileFormats;

/* extern struct jkFileFormat *snackFileFormats; */

typedef struct {
   short min;
   short max;
} Minmax;

#define BE_OK(x) {int res = (x); if (res != TCL_OK) return res;}

/* maximal nb of channels allowed */
#define MAXCHAN 4
/* number of samples to read in one step */
#define NBSAMP 4096  

/* Global struct should be included inside Sound object when possible */
static Minmax *shap = NULL;
static char *centiname = NULL;
static jkFileFormat *ff;
static int available = 0;
static char *buf = NULL;
#ifdef CONTROL
static int nbech = 0;
static double sum = 0;
static double squ = 0;
#endif

/* --------------------------------------------------------------------- */

static char *StringDup( char *s)
{
   int l = strlen(s);
   char *t = Tcl_Alloc(l+1);
   strncpy( t, s, l);
   t[l] = '\0';
   return t;
}

/* --------------------------------------------------------------------- */

static void read_close(Sound *s);

static int read_init(Sound *s, long pos) {
   snackFileFormats = Snack_GetFileFormats();
   switch (s->storeType) {
   case SOUND_IN_FILE:
      for (ff = snackFileFormats; ff != NULL; ff = ff->next) {
	 if (strcmp(s->fileType, ff->formatName) == 0) {
	    break;
	 }
      }
      if (ff == NULL) return TCL_ERROR;
      BE_OK( SnackOpenFile(ff->openProc, s, s->interp,
			   &s->rwchan, "r"));
      s->buffersize = NBSAMP;
      if ((s->tmpbuf = (short *) ckalloc(s->buffersize * s->sampsize *
					 s->nchannels)) == NULL) {
	 Tcl_AppendResult(s->interp, "Couldn't allocate buffer!", NULL);
	 return TCL_ERROR;
      }
      available = 0;
#ifdef CONTROL
      sum = 0; squ = 0; nbech = 0;
#endif
      if (SnackSeekFile(ff->seekProc, s, s->interp, s->rwchan, 
		s->headSize+pos*s->sampsize*s->nchannels) == TCL_OK) {
	 return TCL_OK;
      }
      read_close(s);
   default:
      ;
   }
   return TCL_ERROR;
}

static int read_next(Sound *s, short smp[MAXCHAN]) {
   short x = 0;
   int needed, read, c;

   if (available <= 0) {
      needed = s->buffersize * Snack_GetNumChannels(s)
	 * Snack_GetBytesPerSample(s);
      if (ff->readProc == NULL) {
	 read = Tcl_Read(s->rwchan, (char *) s->tmpbuf, needed);
      } else {
	 read = (ff->readProc)(s, s->interp, s->rwchan, NULL,
			       (char *) s->tmpbuf, needed);
      }
      available = read / (Snack_GetNumChannels(s) * Snack_GetBytesPerSample(s));
      buf = (char *)s->tmpbuf;
   }

   /* on eof: zero  signal */
   if (available <= 0) {
      for (c=0; c<Snack_GetNumChannels(s); c++)
	 smp[c] = 0;
      return 0;
   }

   for (c=0; c<Snack_GetNumChannels(s); c++) {
      switch (Snack_GetSampleFormat(s)) {
      case LIN16:
	 x = *((short *)buf);
	 if (s->swap)
	    x = Snack_SwapShort(x);
	 break;
      case ALAW:
	 x = Snack_Alaw2Lin(*((unsigned char *)buf));
	 break;
      case MULAW:
	 x = Snack_Mulaw2Lin(*((unsigned char *)buf));
	 break;
      case LIN8:
	 x = (*((char *)buf)) << 8;
	 break;
      case LIN8OFFSET:
	 x = ((*((unsigned char *)buf)) - 128)<< 8;
	 break;
      }
      smp[c] = x;
#ifdef CONTROL
      sum += x; squ += x*x; nbech ++;
#endif
      buf += Snack_GetBytesPerSample(s);
   }
   available--;
   return 1;
}

static void read_close(Sound *s) {

#ifdef CONTROL
   if (nbech > 0) fprintf(stderr, "(shape/centi) swap %d - %f mean / %f sigma for %d ech\n", s->swap, sum/nbech, sqrt((squ-sum*sum/nbech)/nbech), nbech);
#endif
   SnackCloseFile(ff->closeProc, s, s->interp, &s->rwchan);
   s->rwchan = NULL;
   ckfree((char *)s->tmpbuf);
}

/* --------------------------------------------------------------------- */

int CentiCmd(Sound *s, Tcl_Interp *interp, int objc,
            Tcl_Obj *CONST objv[])
{
  char *fname = NULL;
  FILE *file = NULL;
  long k0, k1;
  int c, first, cont;
  int nchan;
  double hRatio, pos;
  short v[MAXCHAN];
  Minmax cshap[MAXCHAN];

  if (objc > 3) {
    Tcl_WrongNumArgs(interp, 1, objv, "centi ?filename?");
    return TCL_ERROR;
  }
      
  s->interp = interp;
  nchan = Snack_GetNumChannels(s);

  /* Create centi-second shape in file */
  if (objc == 3) {
     fname = Tcl_GetStringFromObj(objv[2], NULL);
  }
  if (fname && (file=fopen(fname,"rb"))==NULL) {
     if ((file=fopen(fname,"wb+")) != NULL) {
	hRatio = (float) Snack_GetFrequency(s) / 100.0;
	pos = 0;
	BE_OK(read_init(s, (long)pos));
	/* compute min/max for each point */
	cont = 1;
	while (cont) {
	   k0 = pos; pos += hRatio; k1 = pos;
	   first = 1;
	   while (k0<k1) {
	      cont = read_next(s, v);
	      for (c=0; c<nchan; c++) {
		 if (first || (v[c] < cshap[c].min)) cshap[c].min = v[c]; 
		 if (first || (v[c] > cshap[c].max)) cshap[c].max = v[c]; 
	      }
	      if (!cont) break;
	      first = 0;
	      k0++;
	   }
	   if (first & !cont) break;
	   fwrite(cshap, sizeof(Minmax), nchan, file);
	}
	read_close(s);
     } else {
	Tcl_AppendResult(interp, "Couldn't write to file: \"", fname, "\"", NULL);
	return TCL_ERROR;
     }
  }
  if (centiname) Tcl_Free(centiname);
  if (file) {
     fclose(file);
     centiname = StringDup(fname);
     Tcl_SetResult(interp, "1", TCL_STATIC);
  } else {
     centiname = NULL;
     Tcl_SetResult(interp, "0", TCL_STATIC);
  }
  return TCL_OK;
}

/* --------------------------------------------------------------------- */

int ShapeCmd(Sound *s, Tcl_Interp *interp, int objc,
            Tcl_Obj *CONST objv[])
{
   FILE *file = NULL;
   int width, i, j, c, first;
   double begin, len;
   int nchan;
   float Fe;
   double hRatio, pos;
   long k0, k1;
   short v[MAXCHAN];
   Minmax mm, last[MAXCHAN] = {{0,0},{0,0},{0,0},{0,0}};
   Tcl_Obj *resObj = NULL;

  if (objc != 5) {
     Tcl_WrongNumArgs(interp, 1, objv, "shape <width> <begin_in_sec> <duration_in_sec>");
    return TCL_ERROR;
  }
  BE_OK(Tcl_GetIntFromObj(interp, objv[2], &width));
  BE_OK(Tcl_GetDoubleFromObj(interp, objv[3], &begin));
  BE_OK(Tcl_GetDoubleFromObj(interp, objv[4], &len));
      
  s->interp = interp;
  Fe = Snack_GetFrequency(s);
  nchan = Snack_GetNumChannels(s);

  if (width<=0 || begin<0 || len<=0 || nchan<=0) {
     Tcl_SetResult(interp, "Bad configuration for shape", TCL_STATIC);
     return TCL_ERROR;
  }

  if (shap == NULL) {
     shap = (Minmax *) Tcl_Alloc(sizeof(Minmax)*nchan*width);
  } else {
     shap = (Minmax *) Tcl_Realloc((char *)shap,
				      sizeof(Minmax)*nchan*width);
  }
	 
  /* Read centi-second shape or signal samples ? */
  if (100*len/width > 2.0) {
     /* Use centi-second shape from file if possible */
     if (centiname != NULL && (file=fopen(centiname,"rb"))) {
	Fe = 100;
     }
  }

  /* number of samples per point */
  hRatio = Fe * len / width;
  /* round first sample to be a multiple of hRatio */
  pos = floor(begin*Fe/hRatio)*hRatio;
  if (file) {
     fseek( file, sizeof(Minmax) * nchan * (long)pos, SEEK_SET);
  } else {
     BE_OK(read_init(s, (long)pos));
  }

  /* compute min/max for each point */
  for (i=0; i<width; i++) {
     k0 = pos; pos += hRatio; k1 = pos;
     if (k0<k1) {
	first = 1;
	while (k0<k1) {
	   if (!file) read_next(s, v);
	   for (c=0; c<nchan; c++) {
	      j = c+i*nchan;
	      if (file) {
		 mm.min = mm.max = 0;
		 fread( &mm, sizeof(Minmax), 1, file);
	      } else {
		 mm.min = mm.max = v[c];
	      }
	      if (first || (mm.min < shap[j].min))
		 shap[j].min = mm.min; 
	      if (first || (mm.max > shap[j].max))
		 shap[j].max = mm.max; 
	      if (k0+1==k1) last[c] = mm;
	   }
	   first = 0;
	   k0++;
	}
     } else {
	for (c=0; c<nchan; c++) {
	   shap[c+i*nchan] = last[c];
	}
     }
  }
  if (file) {
     fclose(file);
  } else {
     read_close(s);
  }
#ifdef LE
  if (s->byteOrder == SNACK_BIGENDIAN) {
#else
  if (s->byteOrder == SNACK_LITTLEENDIAN) {
#endif
     swab(shap,shap,nchan*width*sizeof(Minmax));
  }
  resObj = Tcl_NewStringObj((char *)shap, nchan*width*sizeof(Minmax));
  Tcl_SetObjResult( s->interp, resObj);
  return TCL_OK;
}

/* --------------------------------------------------------------------- */

int GetCmd(Sound *s, Tcl_Interp *interp, int objc,
            Tcl_Obj *CONST objv[])
{
   Tcl_Obj *resObj = NULL;
   long pos, ech;
   short v[MAXCHAN];

   if (objc != 4) {
      Tcl_WrongNumArgs(interp, 1, objv, "get <sample_start> <sample_count>");
      return TCL_ERROR;
   }
   BE_OK(Tcl_GetLongFromObj(interp, objv[2], &pos));
   BE_OK(Tcl_GetLongFromObj(interp, objv[3], &ech));
      
   s->interp = interp;
   
   read_init(s, pos);
   resObj = Tcl_NewStringObj("",-1);
   /* Should we IncrRefCount for this object ? I think not, but... */
   while ((ech--) > 0) {
      read_next(s, v);
#ifdef LE
      if (s->byteOrder == SNACK_BIGENDIAN) {
#else
      if (s->byteOrder == SNACK_LITTLEENDIAN) {
#endif
	 swab(&v,&v,sizeof(short) * Snack_GetNumChannels(s));
      }
      Tcl_AppendToObj( resObj, (char *) &v, 
		       sizeof(short) * Snack_GetNumChannels(s));
   }
   read_close(s);
   Tcl_SetObjResult( s->interp, resObj);
   return TCL_OK;
}

/* --------------------------------------------------------------------- */

int OrderCmd(Sound *s, Tcl_Interp *interp, int objc,
            Tcl_Obj *CONST objv[])
{
   int length;
   char *str;

   if (objc != 3) {
      Tcl_WrongNumArgs(interp, 1, objv, "order bigEndian|littleEndian");
      return TCL_ERROR;
   }

   str = Tcl_GetStringFromObj(objv[2], &length);
   if (strncasecmp(str, "littleEndian", length) == 0) {
      s->byteOrder = SNACK_LITTLEENDIAN;
   } else if (strncasecmp(str, "bigEndian", length) == 0) {
      s->byteOrder = SNACK_BIGENDIAN;
   } else {
      Tcl_AppendResult(interp, "order should be bigEndian or littleEndian", NULL);
      return TCL_ERROR;
   }
   /* fprintf(stderr,"%d\n",s->byteOrder); */
   return TCL_OK;
}

/* --------------------------------------------------------------------- */
