/* ---------------------------------------------------------------------
   basic cddb functions for XfreeCD

   Copyright 1998 by Brian C. Lane
   nexus@tatoosh.com
   http://www.tatoosh.com/nexus

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   =============================[ HISTORY ]=============================
   06/19/98     Adding write extended data to the database. DONE.
                Switching to using g_malloc, g_free, g_realloc -- they're
		safer than libc and can provide usage stats.

   06/16/98     Need to add support for extended data lines. Split lines
                too of course. This data needs to make it into the
		database, and be able to send it to the server when
		the CD is edited.

		Revision # support needs to be added to. Read it and
		increment it when the CD is submitted to the server.
		FIXED.

   06/06/98     I need to support split TTITLE lines, and get the
                track # from the TTITLE line instead of assuming they
		just go in order.
		I found my garbage at the end of the file problem. If
		A long file was written then it was shortened it
		wouldn't erase it first, it would just write over
		the top and leave the old stuff hanging off the end.
		So I added an unlink to delete it first.

   05/27/98     Adding creation of the cddb database directories if
                they don't exist (only creates the ones that don't
		exist). Added to write_cddb(). Only creates what it
		needs to. This way it isn't limited to a static list of
		categories. Works.
		Found a bug. CDDB read/write don't handle multiple
		DTITLEs very well.

   05/16/98     Added expansion of ~/ to $HOME enviornmental variable
                Added a local_cddb path in the cdinfo structure to point
		to the local storage location for writing and reading
		cd information.

   05/11/98     Split off reading code to read_cddb_file so it can be
                used to read a specific temporary file after the data
		is downloaded from the server.

   05/09/98     Started this part of the XfreeCD code. 
                Correct operation confirmed for discid using
		Bryan Adams / Waking up the Neighbors id 0xce118c0f
		Writing database entries works well so far, frames,
		length, and discid are correct. Default track and title
		strings are saved ok too.
		find_discid works.

   ---------------------------------------------------------------------
   This file includes support for saving and loading local database
   information.
   --------------------------------------------------------------------- */
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <dirent.h>
#include <stdlib.h>
#include <errno.h>
#include <gtk/gtk.h>
#include "xfreecd.h"
#include "cd_control.h"
#include "cddb.h"

#undef DEBUG1
#undef DEBUG2
#undef DEBUG3
#undef DEBUG6
#undef DEBUG9

int
cddb_sum(int n)
{
	char	buf[12],
		*p;
	int	ret = 0;

	/* For backward compatibility this algorithm must not change */
	sprintf(buf, "%lu", n);
	for (p = buf; *p != '\0'; p++)
		ret += (*p - '0');

	return (ret);
}

unsigned long cddb_discid( struct CDINFO *cdinfo )
{
  int	i,
        t = 0,
        n = 0,
        tot_trks;

  tot_trks = cdinfo->tochdr.cdth_trk1;

  /* For backward compatibility this algorithm must not change */
  for (i = 0; i < tot_trks; i++)
    n += cddb_sum((cdinfo->track[i].te.cdte_addr.msf.minute * 60) + cdinfo->track[i].te.cdte_addr.msf.second);
  
  t = ((cdinfo->leadout.cdte_addr.msf.minute * 60) + cdinfo->leadout.cdte_addr.msf.second) - ((cdinfo->track[0].te.cdte_addr.msf.minute * 60) + cdinfo->track[0].te.cdte_addr.msf.second);
  
  return ((n % 0xff) << 24 | t << 8 | tot_trks);
}


/* -------------------------------------------------------------------------
   Write the data from cdinfo structure to the correct file in the selected
   category.

   The master database directory is in cdinfo->local_cddb

   How do I check for the same CD with a different ID so I can symlink?

   if over is 0 then check to make sure the file doesn't already exist
   if over is 1 then go ahead and overwrite it.

   return -2 if the file already exists.
   return -3 if the HOME variable cannot be found
   return -4 if we had trouble creating the local cddb directories
   ------------------------------------------------------------------------- */
int write_cddb( struct CDINFO *cdinfo, int over )
{
  char  fname[1024],
        tmp_fname[1024],
        tmpstr[255],
        idstr[9],
        *p;
  int   fd,
        x,
        i;
  struct stat   sbuf;

  /* Copy the path to the local datbase to a temporary variable */
  strncpy( fname, cdinfo->local_cddb, 1023 );

  /* Convert a leading ~ into the user's HOME directory */
  if( fname[0] == '~' )
    {
      /* Copy the reset of the path/filename to tmp_fname */
      strncpy( tmp_fname, &fname[1], 1023 );

      if( ( p = getenv("HOME") ) == NULL )
	{
	  return(-3);
	}
      strncpy( fname, p, 1023 );

      /* Make sure there is a slash inbetween */
      if( (fname[strlen(fname)-1] != '/') && (tmp_fname[0] != '/') )
	{
	  strcat( fname, "/" );
	}

      strncat( fname, tmp_fname, 1023-strlen( p ) );
    }

#ifdef DEBUG1
  g_print("write_cddb( %s )\n", fname );
#endif

  /* 
     See if we have the top level directory. If not, try to create it.
  */
  if( stat( fname, &sbuf ) < 0 )
    {
      /* If the top level directory doesn't exist, create it */
      if( errno == ENOENT )
	{
	  /* Try to make the top level directory, if it fails, quit */
	  if( mkdir( fname, S_IRWXU | S_IRGRP | S_IXGRP | S_IXOTH | S_IROTH ) < 0 )
	    {
	      return(-4);
	    }
	} else {
	  return(-4);
	}
    }


  /* Make sure we have slashes between the path and category directory */
  if( fname[strlen(fname)] != '/' )
    strcat( fname, "/" );
  strcat( fname, cdinfo->category->str );
  if( fname[strlen(fname)] != '/' )
    strcat( fname, "/" );

  /*
    See if the category directory exists. If not, create it.
  */
  if( stat( fname, &sbuf ) < 0 )
    {
      /* If the top level directory doesn't exist, create it */
      if( errno == ENOENT )
	{
	  /* Try to make the category directory, if it fails, quit with an error*/
	  if( mkdir( fname, S_IRWXU | S_IRGRP | S_IXGRP | S_IXOTH | S_IROTH ) < 0 )
	    {
	      return(-4);
	    }
	} else {
	  return(-4);
	}
    }

  sprintf( idstr, "%08lx", cdinfo->discid );
  strcat( fname, idstr );

  /* Should we check for prior existance? */  
  if( !over )
    {
      if( ( fd = open( fname, O_RDONLY ) ) != -1 )
	{
	  close( fd );
	  return(-2);
	}
    }

  /* Make sure it is deleted */
  unlink( fname );

  if( ( fd = open( fname, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ) < 0 )
    {
      return(-1);
    }

  /* Write the cddb entry from the info in the cdinfostructure */
  sprintf( tmpstr, "# xmcd CD database file generated by XfreeCD %s\n", VERSION );

  if( write( fd, tmpstr, strlen( tmpstr) ) < 0 )
    {
      close( fd );
      return(-1);
    }

  sprintf( tmpstr, "#\n# Track frame offsets:\n" );
  if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
    {
      close( fd );
      return(-1);
    }

  /* Write the frame offset for each track */
  for( x = 0; x < cdinfo->tochdr.cdth_trk1; x++ )
    {
      sprintf( tmpstr, "#\t%ld\n", cdinfo->track[x].frame_offset );
      if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
	{
	  close( fd );
	  return(-1);
	}
    }

  strcpy( tmpstr, "#\n" );
  if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
    {
      close( fd );
      return(-1);
    }

  sprintf( tmpstr, "# Disc length: %d seconds\n#\n", cdinfo->cd_length );
  if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
    {
      close( fd );
      return(-1);
    }
 
  sprintf( tmpstr, "# Revision: %d\n", cdinfo->revision );
  if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
    {
      close( fd );
      return(-1);
    }

  sprintf( tmpstr, "# Submitted via: XfreeCD %s\n#\n", VERSION );
  if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
    {
      close( fd );
      return(-1);
    } 

  sprintf( tmpstr, "DISCID=%08lx\n", cddb_discid( cdinfo ) );
  if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
    {
      close( fd );
      return(-1);
    } 

  if( (cdinfo->title == NULL) || (cdinfo->title->len == 0) )
    {
      if( write( fd, "DTITLE=\n", 8 ) < 0 )
	{
	  close( fd );
	  return(-1);
	} 
    } else {
      /* If the title is too long, split it up into 70 byte chunks */
      i = 0;
      p = cdinfo->title->str;
      while( i < cdinfo->title->len )
	{
	  strcpy( tmpstr, "DTITLE=" );
	  strncat( tmpstr, p, 70 );
	  strcat( tmpstr, "\n" );

	  if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
	    {
	      close( fd );
	      return(-1);
	    } 
	  
	  if( cdinfo->title->len > 70 )
	    {
	      p = p + 70;
	      i += 70;
	    } else { 
	      i = cdinfo->title->len;
	    }
	}
    }

  /* Write the titles */
  for( x = 0; x < cdinfo->tochdr.cdth_trk1; x++ )
    {
      if( (cdinfo->name[x] == NULL) || (cdinfo->name[x]->len == 0) )
	{
	  sprintf( tmpstr, "TTITLE%d=\n", x );
	  if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
	    {
	      close( fd );
	      return(-1);
	    } 
	} else {
	  
	  /* If the track name is too long, split it up into 70 byte chunks */
	  i = 0;
	  p = cdinfo->name[x]->str;
	  while( i < cdinfo->name[x]->len )
	    {
	      sprintf( tmpstr, "TTITLE%d=", x );
	      strncat( tmpstr, p, 70 );
	      strcat( tmpstr, "\n" );

	      if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
		{
		  close( fd );
		  return(-1);
		} 

	      if( cdinfo->name[x]->len > 70 )
		{
		  p = p + 70;
		  i += 70;
		} else { 
		  i = cdinfo->name[x]->len;
		}
	    }
	}
    }
  
  /* Check for a null entry first */
  if( (cdinfo->extd == NULL) || (cdinfo->extd->len == 0) )
    {
      if( write( fd, "EXTD=\n", 6 ) < 0 )
	{
	  close( fd );
	  return(-1);
	} 
    } else {
      /* Write the extended disc data to the file. It it is too long, break it
	 up into 70 byte chuncks
      */
      i = 0;
      p = cdinfo->extd->str;
      while( i < cdinfo->extd->len )
	{
	  strcpy( tmpstr, "EXTD=" );
	  strncat( tmpstr, p, 70 );
	  strcat( tmpstr, "\n" );
	  
	  if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
	    {
	      close( fd );
	      return(-1);
	    } 

	  if( cdinfo->extd->len > 70 )
	    {
	      p = p + 70;
	      i += 70;
	    } else { 
	      i = cdinfo->extd->len;
	    }
	}
    }

  /* Write the extended title information to the database */
  for( x = 0; x < cdinfo->tochdr.cdth_trk1; x++ )
    {
      if( (cdinfo->extt[x] == NULL) || (cdinfo->extt[x]->len == 0) )
	{
	  sprintf( tmpstr, "EXTT%d=\n", x );
	  if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
	    {
	      close( fd );
	      return(-1);
	    } 
	} else {
	  /* If the track name is too long, split it up into 70 byte chunks */
	  i = 0;
	  p = cdinfo->extt[x]->str;
	  while( i < cdinfo->extt[x]->len )
	    {
	      sprintf( tmpstr, "EXTT%d=", x );
	      strncat( tmpstr, p, 70 );
	      strcat( tmpstr, "\n" );

	      if( write( fd, tmpstr, strlen( tmpstr ) ) < 0 )
		{
		  close( fd );
		  return(-1);
		} 

	      if( cdinfo->extt[x]->len > 70 )
		{
		  p = p + 70;
		  i += 70;
		} else { 
		  i = cdinfo->extt[x]->len;
		}
	    }
	}
    }

  /* XfreeCD doesn't use this at all */
  if( write( fd, "PLAYORDER=\n", 11 ) < 0 )
    {
      close( fd );
      return(-1);
    } 
    
  close( fd );

  return(0);
}


/* -----------------------------------------------------------------------
   Search all sub-directories below CDDB_PATH for the file to read
   Fills in the category with the name of the trailing directory that
   it is found in.


   Return 0 on file found
   Return -1 on an error
   Return -2 on no file found
   ----------------------------------------------------------------------- */
int find_discid( char *fname,                /* Path to local database */
		 unsigned long id,           /* discid to search for   */
		 char *path,                 /* Return full path       */
		 GString **category )        /* Return category found  */
{
  DIR   *dp;
  struct dirent *dirp;
  char  idstr[9],
        tmp_fname[1024],
        *p;
  int   fp;

  /* Convert a leading ~ into the user's HOME directory */
  if( fname[0] == '~' )
    {
      /* Copy the reset of the path/filename to tmp_fname */
      strncpy( tmp_fname, &fname[1], 1023 );

      if( ( p = (char *) getenv("HOME") ) == NULL )
	{
	  return(-2);
	}
      strncpy( fname, p, 1023 );

      /* Make sure there is a slash inbetween the two */
      if( (fname[strlen(fname)-1] != '/') && (tmp_fname[0] != '/') )
	{
	  strcat( fname, "/" );
	}

      strncat( fname, tmp_fname, 1023-strlen( p ) );
    }

#ifdef DEBUG1
  g_print("find_discid( %s )\n", fname );
#endif


  sprintf( idstr, "%08lx", id );

  if( ( dp = opendir( fname ) ) == NULL )
    return(-1);

  while( ( dirp = readdir( dp ) ) != NULL )
    {
      if( (strcmp(dirp->d_name,"." )==0) || (strcmp(dirp->d_name, "..")==0) )
	continue;

      strcpy( path, fname );
      if( path[strlen(path)] != '/' )
	strcat( path, "/" );
      strcat( path, dirp->d_name );
      if( path[strlen(path)] != '/' )
	strcat( path, "/" );
      strcat( path, idstr );

      /* Copy the directory name as the category name */
      if( *category == NULL )
	*category = g_string_new( dirp->d_name );
      else
	*category = g_string_assign( *category, dirp->d_name );

#ifdef DEBUG1
      g_print("Checking %s\n", path );
#endif

      /* Does this file exist? */
      if( ( fp = open( path, O_RDONLY ) ) != -1 )
	{
	  close( fp );
	  closedir( dp );
	  return(0);
	}
    } 

  closedir( dp );

  return(-2);
}


/* -----------------------------------------------------------------------
   Read the CDDB info from a filename into a cdinfo structure
   Allocate all needed string storage using gtk's g_string functions

   return -2  = failed to find HOME enviornmental variable
   ----------------------------------------------------------------------- */
int read_cddb_file( char *fname, struct CDINFO *tmpinfo )
{
  int   gotid,
        i,
        x,
        f;
  FILE  *fp;
  char  line[255],
        discid[9],
        *p;
  
  if( ( fp = fopen( fname, "r" ) ) == NULL )
    {
      return(-1);
    }

  /* Read the cddb file, placing data into tmpinfo */
  if( fgets( line, 255, fp ) == NULL )
    {
      fclose( fp );
      return(-1);
    }

  /* Check the file to make sure its a cddb file */
  if( strncmp( line, "# xmcd", 6 ) != 0 )
    {
      fclose( fp );
      return(-1);
    }

#ifdef DEBUG1
  g_print("%s", line );
#endif

  /* Find the track offsets, abort serarch at the end of comments */
  while( ( line[0] == '#' ) && ( strncmp( line, "# Track frame offsets:",22 ) != 0 )  )
    {
      if( fgets( line, 255, fp ) == NULL )
	{
	  fclose( fp );
	  return(-1);
	}
    }

  /* Skip all the reset of the comments up to Revision: */
  while( line[0] == '#' && (strncmp( line, "# Revision:", 11)!=0) )
    {
      if( fgets( line, 255, fp ) == NULL )
	{
	  fclose( fp );
	  return(-1);
	}
    }

  /* If we got the Revision line, get the revision # */
  if( strncmp( line, "# Revision:", 11)==0 )
    {
      sscanf( line, "# Revision: %d", &tmpinfo->revision );

      /* Now read the rest of the comments */
      while( line[0] == '#' )
	{
	  if( fgets( line, 255, fp ) == NULL )
	    {
	      fclose( fp );
	      return(-1);
	    }
	}
    }

  /* Read discid lines */
  /*
     How should this be handled? check for our id, and discard all others
     that may be present?

     DISCID= lines are comma seperated and can be multiple lines
  */
  gotid = 0;
  sprintf( discid, "%08lx", tmpinfo->discid );
  while( strncmp( line, "DISCID=", 7 ) == 0 )
    {
      if( strstr( line, discid ) != NULL )
	gotid = 1;
      if( fgets( line, 255, fp ) == NULL )
	{
	  fclose( fp );
	  return(-1);
	}
    }

  /* Process multiple DTITLE lines and concatanate them */
  i = 0;
  f = 0;
  while( strncmp( line, "DTITLE", 6 ) == 0 )
  {
    p = strtok( line, "=\n" );
    p = strtok( NULL, "=\n" );

    /* Add the title to the tmpinfo.title string */
    if( f == 0 )
      {
	tmpinfo->title = g_string_new( p );
	f = 1;
      } else {
	tmpinfo->title = g_string_append( tmpinfo->title, p );
      }

    /* Keep reading DTITLE no matter what */
    if( fgets( line, 255, fp ) == NULL )
      {
	fclose( fp );
	return(-1);
      }
  }

#ifdef DEBUG1
  g_print("title read = %s\n", tmpinfo->title->str );
#endif

  /*
     Copy the titles from the TTITLE strings

     This has to:
       Get the track # from the TTITLEx
       strcat split title lines up to the limit of storage (255)
  */
  x = -1;
  f = 0;
  while( strncmp( line, "TTITLE", 6 ) == 0 )
  {
    /* Get the track # */
    p = strtok( &line[6], "=\n" );

    /* Is it a new track? */
    if( atoi(p) != x )
      {
	/* Yes, reset the length counter and track name */
	i = 0;
	f = 0;
	tmpinfo->name[atoi(p)] = NULL;
      }

    /* Get the track number and make sure its not too big. */
    if( ( x = atoi( p ) ) < 99 )
      {
	/* Get the track name */
	p = strtok( NULL, "=\n" );

	/* If its blank, then insert default track name */
	if( p == NULL )
	  {
	    tmpinfo->name[x] = NULL;
	  } else {
	    if( f == 0 )
	      {
		tmpinfo->name[x] = g_string_new( p );
		f = 1;
	      } else {
		tmpinfo->name[x] = g_string_append( tmpinfo->name[x], p );
	      }
	  }
      }

    /* Read the next line */
    if( fgets( line, 255, fp ) == NULL )
      {
	fclose( fp );
	return(-1);
      }
  }


  /* Process multiple EXTD lines and concatanate them, dynamically
     allocating memory at tmpinfo->extd for it
   */
  i = 0;
  f = 0;
  tmpinfo->extd = NULL;
  while( strncmp( line, "EXTD", 4 ) == 0 )
  {
    /* Add to the data until the end is reached */
    p = strtok( line, "=\n" );
    p = strtok( NULL, "=\n" );

    if( p != NULL )
      {
	/* Move the pointer and copy the new string */
	if( f == 0 )
	  {
	    tmpinfo->extd = g_string_new( p );
	    f = 1;
	  } else {
	    tmpinfo->extd = g_string_append( tmpinfo->extd, p );
	  }
      }

    /* Keep reading lines */
    if( fgets( line, 255, fp ) == NULL )
      {
	fclose( fp );
	return(-1);
      }
  }


  /*
     Copy the extended title data from the EXTTx entries

     This has to:
       Get the track # from the EXTTx
       Allocate storage for it and copy it over, and handle multiple lines
       for each track entry.
  */
  x = -1;
  f = 0;
  while( strncmp( line, "EXTT", 4 ) == 0 )
  {
    /* Get the track # */
    p = strtok( &line[4], "=\n" );

    /* Is it a new track? */
    if( atoi(p) != x )
      {
	/* Yes, reset the length counter and track name */
	i = 0;
        f = 0;
	tmpinfo->extt[atoi(p)] = NULL;
      }

    /* Get the track number, make sure it isn't too big */
    if( ( x = atoi( p ) ) < 99 )
      {
	/* Get the extended data from the rest of the line*/
	p = strtok( NULL, "=\n" );

	/* Process multiple EXTT lines and concatanate them, dynamicly
	   allocating memory at tmpinfo->extd for it
	*/
	if( p != NULL )
	  {
	    /* Move the pointer and copy the new string */
	    if( f == 0 )
	      {
		tmpinfo->extt[x] = g_string_new( p );
		f = 1;
	      } else {
		tmpinfo->extt[x] = g_string_append( tmpinfo->extt[x], p );
	      }
	  }
      }

    /* Read the next line */
    if( fgets( line, 255, fp ) == NULL )
      {
	fclose( fp );
	return(-1);
      }
  }

  fclose( fp );

  return(0);
}


/* -----------------------------------------------------------------------
   Read a cddb entry for the current CD

   This needs to search through the sub-directories in CDDB_PATH to find
   the discid of the current CD.

   We then read the file, filling in the track names, etc.
   We should also compare the frame count to make sure it is the
   correct CD.

   Returns -1 if there was an error
   Returns -2 if it cannot find the file
   ----------------------------------------------------------------------- */
int read_cddb( struct CDINFO *cdinfo )
{
  int   x;
  char  fname[255];
  struct CDINFO tmpinfo;

  /* Clean out the structure */
  bzero( &tmpinfo, sizeof( struct CDINFO ) );

  tmpinfo.discid = cdinfo->discid;

  /* Find out where this ID lives and fill in fname with full name */
  if( find_discid( cdinfo->local_cddb, tmpinfo.discid, fname, &tmpinfo.category ) < 0 )
    {
      /* Could not find an entry for this */
      return(-2);
    }

  if( read_cddb_file( fname, &tmpinfo ) == 0 )
    {
      cdinfo->title = tmpinfo.title;
      for( x = 0; x < cdinfo->tochdr.cdth_trk1; x++ )
	{
	  cdinfo->name[x] = tmpinfo.name[x];
	}

      /* Copy the category over too */
      cdinfo->category = tmpinfo.category;

      /* Copy the revision # over if it is valid */
      if( tmpinfo.revision >= 0 )
	cdinfo->revision = tmpinfo.revision;
      else
	cdinfo->revision = 1;

      /* Copy the extd pointer over */
      cdinfo->extd = tmpinfo.extd;

      /* Copy all the extended track pointers */
      for( x = 0; x < 99; x++ )
	cdinfo->extt[x] = tmpinfo.extt[x];
    }

  return(0);
}
