/*
 * GUI menu bar module
 *
 * Almost every callback function internally calls the related action routine
 * defined in actions.[ch].
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <gnome.h>
#include "gdiff.h"
#include "gui.h"
#include "menu.h"
#include "guiwin.h"
#include "dirview.h"
#include "fileview.h"
#include "actions.h"
#include "hide.h"
#include "properties.h"


/* Private function declarations */
static void update_common(const Preference *pref);
static void update_noview(const Preference *pref);
static void update_dirview(GDiffWindow *gdwin, const Preference *pref);
static void menu_dirv_block(GDiffWindow *gdwin);
static void menu_dirv_unblock(GDiffWindow *gdwin);
static void update_fileview(GDiffWindow *gdwin, const Preference *pref);
static void menu_filev_block(GDiffWindow *gdwin);
static void menu_filev_unblock(GDiffWindow *gdwin);

static void file_open_cb(GtkWidget *widget, gpointer data);
static void file_close_cb(GtkWidget *widget, gpointer data);
static void mode_change_cb(GtkWidget *widget, gpointer data);
static void line_num_show_cb(GtkWidget *widget, gpointer data);
static void go_dirview_cb(GtkWidget *widget, gpointer data);
static void move_first_cb(GtkWidget *widget, gpointer data);
static void move_last_cb(GtkWidget *widget, gpointer data);
static void move_next_cb(GtkWidget *widget, gpointer data);
static void move_prev_cb(GtkWidget *widget, gpointer data);
static void move_rel_next_cb(GtkWidget *widget, gpointer data);
static void move_rel_prev_cb(GtkWidget *widget, gpointer data);
static void text_wrap_cb(GtkWidget *widget, gpointer data);
static void set_preference_cb(GtkWidget *widget, gpointer data);
static void show_path_cb(GtkWidget *widget, gpointer data);
static void row_hide_func_cb(GtkWidget *widget, gpointer data);
static void row_hide_stat_cb(GtkWidget *widget, gpointer data);
static void show_tabs_cb(GtkWidget *widget, gpointer data);
static void reload_cb(GtkWidget *widget, gpointer data);
static void quit_cb(GtkWidget *widget, gpointer data);
static void about_cb(GtkWidget *widget, gpointer data);
static void about_destroy_cb(GtkWidget *widget, gpointer data);


/* Private variables */
/* The menu definitions */
static GnomeUIInfo file_menu[] = {
	GNOMEUIINFO_MENU_OPEN_ITEM(file_open_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_CLOSE_ITEM(file_close_cb, NULL),
	GNOMEUIINFO_MENU_EXIT_ITEM(quit_cb, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo settings_menu[] = {
	GNOMEUIINFO_MENU_PREFERENCES_ITEM(set_preference_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
#define MENUITEM_SHOWTABS		2
	{GNOME_APP_UI_TOGGLEITEM, N_("Show Ta_bs"),
	 N_("Show notebook tabs"),
	 show_tabs_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
	GNOMEUIINFO_END
};

static GnomeUIInfo dirview_menu[] = {
#define MENUITEM_SHOWPATH		0
	{GNOME_APP_UI_TOGGLEITEM, N_("_Show path"),
	 N_("Show absolute path on directory view"),
	 show_path_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
	GNOMEUIINFO_SEPARATOR,
#define MENUITEM_HIDEFUNC_BASE	2/* base index */
	{GNOME_APP_UI_TOGGLEITEM, N_("_Hide emacs backup"),
	 N_("Hide emacs backup files on directory view"),
	 row_hide_func_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
	{GNOME_APP_UI_TOGGLEITEM, N_("H_ide object files"),
	 N_("Hide object files on directory view"),
	 row_hide_func_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
	GNOMEUIINFO_SEPARATOR,
#define MENUITEM_HIDE_STAT_FIRST	5
#define MENUITEM_HIDE_STAT_BINARY	5
	{GNOME_APP_UI_TOGGLEITEM, N_("Hide _Binary"),
	 N_("Hide binary files"),
	 row_hide_stat_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_HIDE_STAT_ONLY		6
	{GNOME_APP_UI_TOGGLEITEM, N_("Hide _Only"),
	 N_("Hide only files"),
	 row_hide_stat_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_HIDE_STAT_DIFF		7
	{GNOME_APP_UI_TOGGLEITEM, N_("Hide _Different"),
	 N_("Hide different files"),
	 row_hide_stat_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_HIDE_STAT_LAST		8
	GNOMEUIINFO_END
};

static GnomeUIInfo pane_radio_menu[] = {
#define MENUITEM_ONE_PANE		0
	GNOMEUIINFO_RADIOITEM(N_("_One pane"), N_("One pane view"),
						  mode_change_cb, NULL),
#define MENUITEM_TWO_PANES		1
	GNOMEUIINFO_RADIOITEM(N_("_Two panes"), N_("Two panes view"),
						  NULL, NULL),
	GNOMEUIINFO_END
};
static GnomeUIInfo pane_menu[] = {
	GNOMEUIINFO_RADIOLIST(pane_radio_menu),
	GNOMEUIINFO_END
};

static GnomeUIInfo line_num_radio_menu[] = {
#define MENUITEM_LINE_SHOW		0
	GNOMEUIINFO_RADIOITEM(N_("_Show"), N_("Show line numbers"),
						  line_num_show_cb, NULL),
#define MENUITEM_LINE_HIDE		1
	GNOMEUIINFO_RADIOITEM(N_("_Hide"), N_("Hide line numbers"),
						  NULL, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo line_num_menu[] = {
	GNOMEUIINFO_RADIOLIST(line_num_radio_menu),
	GNOMEUIINFO_END
};

static GnomeUIInfo fileview_menu[] = {
#define MENUITEM_TEXT_WRAP		0
	{GNOME_APP_UI_TOGGLEITEM, N_("_Text wrap"), N_("Toggle text wrap"),
	 text_wrap_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
	GNOMEUIINFO_SUBTREE(N_("P_ane"), pane_menu),
	GNOMEUIINFO_SUBTREE(N_("_Line number"), line_num_menu),
	GNOMEUIINFO_END
};

static GnomeUIInfo dview_action_menu[] = {
	{GNOME_APP_UI_ITEM, N_("_Reload"),
	 N_("Reload the current directories"),
	 reload_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 GDK_r, GDK_CONTROL_MASK, NULL},
	GNOMEUIINFO_END
};

static GnomeUIInfo fview_action_menu[] = {
	{GNOME_APP_UI_ITEM, N_("_Go directory view"),
	 N_("Go directory view"),
	 go_dirview_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 GDK_h, GDK_CONTROL_MASK, NULL},
#define MENUITEM_NEXT_DIFF		1
	{GNOME_APP_UI_ITEM, N_("_Next difference"),
	 N_("Go to next difference"),
	 move_next_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 GDK_n, GDK_CONTROL_MASK, NULL},
#define MENUITEM_PREV_DIFF		2
	{GNOME_APP_UI_ITEM, N_("_Prev difference"),
	 N_("Go to previous difference"),
	 move_prev_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 GDK_p, GDK_CONTROL_MASK, NULL},
#define MENUITEM_REL_NEXT_DIFF		3
	{GNOME_APP_UI_ITEM, N_("Relative N_ext difference"),
	 N_("Go to relative next difference"),
	 move_rel_next_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 GDK_n, GDK_SHIFT_MASK, NULL},
#define MENUITEM_REL_PREV_DIFF		4
	{GNOME_APP_UI_ITEM, N_("Relative Pre_v difference"),
	 N_("Go to relative previous difference"),
	 move_rel_prev_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 GDK_p, GDK_SHIFT_MASK, NULL},
#define MENUITEM_FIRST_DIFF			5
	{GNOME_APP_UI_ITEM, N_("_First difference"),
	 N_("Go to the first difference"),
	 move_first_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 GDK_t, GDK_CONTROL_MASK, NULL},
#define MENUITEM_LAST_DIFF			6
	{GNOME_APP_UI_ITEM, N_("_Last difference"),
	 N_("Go to the last difference"),
	 move_last_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 GDK_v, GDK_CONTROL_MASK, NULL},
	{GNOME_APP_UI_ITEM, N_("_Reload"),
	 N_("Reload the current files"),
	 reload_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 GDK_r, GDK_CONTROL_MASK, NULL},
	GNOMEUIINFO_END
};

static GnomeUIInfo help_menu[] = {
	/*GNOMEUIINFO_HELP(APPNAME),*/
	GNOMEUIINFO_MENU_ABOUT_ITEM(about_cb, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo main_menu[] = {
	GNOMEUIINFO_MENU_FILE_TREE(file_menu),
	GNOMEUIINFO_MENU_SETTINGS_TREE(settings_menu),
#define MENUITEM_DIR_VIEW		2
	GNOMEUIINFO_SUBTREE(N_("D_irectory view"), dirview_menu),
#define MENUITEM_FILE_VIEW		3
	GNOMEUIINFO_SUBTREE(N_("Fi_le view"), fileview_menu),
#define MENUITEM_DVIEW_ACTION	4
	GNOMEUIINFO_SUBTREE(N_("_Action"), dview_action_menu),
#define MENUITEM_FVIEW_ACTION	5
	GNOMEUIINFO_SUBTREE(N_("_Action"), fview_action_menu),
	GNOMEUIINFO_MENU_HELP_TREE(help_menu),
	GNOMEUIINFO_END
};

/* each is related to menu item */
const struct {
	RowHideFunc rh_func;
} rh_func_array[] = {
	{ hide_emacs_backup },
	{ hide_obj_file },
	{ NULL },
};

/**
 * menu_create:
 * Create menu-bar's menu.
 **/
void
menu_create(GDiffWindow *gdwin)
{
	int i;
	
	gnome_app_create_menus_with_data(gdwin->app, main_menu, gdwin);

	/* Set index to menu item, used in the callback function */
	for (i = 0; rh_func_array[i].rh_func; i++) {
		gtk_object_set_user_data(
			GTK_OBJECT(dirview_menu[MENUITEM_HIDEFUNC_BASE+i].widget),
			GINT_TO_POINTER(i));
	}
	for (i = MENUITEM_HIDE_STAT_FIRST; i < MENUITEM_HIDE_STAT_LAST; i++) {
		gtk_object_set_user_data(GTK_OBJECT(dirview_menu[i].widget),
								 GINT_TO_POINTER(i));
	}
	
	menu_update(gdwin, &g_pref, MENUSTAT_NO_VIEW);
}

/**
 * menu_install_hints_for_toolbar:
 **/
void
menu_install_hints_for_toolbar(GDiffWindow *gdwin)
{
	gnome_app_install_menu_hints(gdwin->app, main_menu);
}

/**
 * menu_update:
 * During update, do block signal handlers.
 **/
void
menu_update(GDiffWindow *gdwin, const Preference *pref, MenuCurStat cur_stat)
{
	update_common(pref);

	switch (cur_stat) {
	case MENUSTAT_NO_VIEW:
		update_noview(pref);
		break;
	case MENUSTAT_DIR_VIEW:
		update_dirview(gdwin, pref);
		break;
	case MENUSTAT_FILE_VIEW:
		update_fileview(gdwin, pref);
		break;
	default:
		g_assert_not_reached();
		break;
	}
}


/* ---The followings are private functions--- */
/**
 * update_common:
 **/
static void
update_common(const Preference *pref)
{
	/* window's preferences */
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(settings_menu[MENUITEM_SHOWTABS].widget),
		pref->winpref.show_tabs);
}

/**
 * update_noview:
 * menu update routine for no-view case.
 **/
static void
update_noview(const Preference *pref)
{
	/* top-level menus */
	if (!GTK_WIDGET_VISIBLE(main_menu[MENUITEM_DIR_VIEW].widget))
		gtk_widget_show(main_menu[MENUITEM_DIR_VIEW].widget);
	if (!GTK_WIDGET_VISIBLE(main_menu[MENUITEM_FILE_VIEW].widget))
		gtk_widget_show(main_menu[MENUITEM_FILE_VIEW].widget);
	if (GTK_WIDGET_VISIBLE(main_menu[MENUITEM_DVIEW_ACTION].widget))
		gtk_widget_hide(main_menu[MENUITEM_DVIEW_ACTION].widget);
	if (GTK_WIDGET_VISIBLE(main_menu[MENUITEM_FVIEW_ACTION].widget))
		gtk_widget_hide(main_menu[MENUITEM_FVIEW_ACTION].widget);
}

/**
 * update_dirview:
 * menu update routine for directory view case.
 **/
static void
update_dirview(GDiffWindow *gdwin, const Preference *pref)
{
	int i;
	
	menu_dirv_block(gdwin);
	
	/* top-level menus */
	if (!GTK_WIDGET_VISIBLE(main_menu[MENUITEM_DIR_VIEW].widget))
		gtk_widget_show(main_menu[MENUITEM_DIR_VIEW].widget);
	if (!GTK_WIDGET_VISIBLE(main_menu[MENUITEM_FILE_VIEW].widget))
		gtk_widget_show(main_menu[MENUITEM_FILE_VIEW].widget);/* show it */
	if (!GTK_WIDGET_VISIBLE(main_menu[MENUITEM_DVIEW_ACTION].widget))
		gtk_widget_show(main_menu[MENUITEM_DVIEW_ACTION].widget);
	if (GTK_WIDGET_VISIBLE(main_menu[MENUITEM_FVIEW_ACTION].widget))
		gtk_widget_hide(main_menu[MENUITEM_FVIEW_ACTION].widget);

	/* Various states */
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(dirview_menu[MENUITEM_SHOWPATH].widget),
		pref->dvpref.show_path);

	/* XXX:I keep only pointers to functions, so menu update becomes kludge */
	for (i = 0; rh_func_array[i].rh_func; i++) {
		GSList *list;
		gboolean b_enabled = FALSE;
		for (list = pref->dvpref.row_hide_func_list; list; list = list->next) {
			RowHideFunc rh_func = list->data;
			if (rh_func == rh_func_array[i].rh_func) {
				b_enabled = TRUE;
				break;
			}
		}
		gtk_check_menu_item_set_active(
			GTK_CHECK_MENU_ITEM(dirview_menu[MENUITEM_HIDEFUNC_BASE+i].widget),
			b_enabled);
	}

	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(dirview_menu[MENUITEM_HIDE_STAT_BINARY].widget),
		(pref->dvpref.row_hide_stat_mask & BINARY_FILES));
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(dirview_menu[MENUITEM_HIDE_STAT_ONLY].widget),
		(pref->dvpref.row_hide_stat_mask & ONLY_FILE1_EXISTS & ONLY_FILE2_EXISTS));
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(dirview_menu[MENUITEM_HIDE_STAT_DIFF].widget),
		(pref->dvpref.row_hide_stat_mask & DIFFERENT_FILES));
	
	menu_dirv_unblock(gdwin);
}

/* block signal handers during update menus */
static void
menu_dirv_block(GDiffWindow *gdwin)
{
	int i;
	
	gtk_signal_handler_block_by_func(
		GTK_OBJECT(dirview_menu[MENUITEM_SHOWPATH].widget),
		GTK_SIGNAL_FUNC(show_path_cb), gdwin);

	for (i = 0; rh_func_array[i].rh_func; i++) {
		gtk_signal_handler_block_by_func(
			GTK_OBJECT(dirview_menu[MENUITEM_HIDEFUNC_BASE+i].widget),
			GTK_SIGNAL_FUNC(row_hide_func_cb), gdwin);
	}

	for (i = MENUITEM_HIDE_STAT_FIRST; i < MENUITEM_HIDE_STAT_LAST; i++) {
		gtk_signal_handler_block_by_func(
			GTK_OBJECT(dirview_menu[i].widget),
			GTK_SIGNAL_FUNC(row_hide_stat_cb), gdwin);
	}
}	

static void
menu_dirv_unblock(GDiffWindow *gdwin)
{
	int i;
	
	gtk_signal_handler_unblock_by_func(
		GTK_OBJECT(dirview_menu[MENUITEM_SHOWPATH].widget),
		GTK_SIGNAL_FUNC(show_path_cb), gdwin);

	for (i = 0; rh_func_array[i].rh_func; i++) {
		gtk_signal_handler_unblock_by_func(
			GTK_OBJECT(dirview_menu[MENUITEM_HIDEFUNC_BASE+i].widget),
			GTK_SIGNAL_FUNC(row_hide_func_cb), gdwin);
	}

	for (i = MENUITEM_HIDE_STAT_FIRST; i < MENUITEM_HIDE_STAT_LAST; i++) {
		gtk_signal_handler_unblock_by_func(
			GTK_OBJECT(dirview_menu[i].widget),
			GTK_SIGNAL_FUNC(row_hide_stat_cb), gdwin);
	}
}


/**
 * update_fileview:
 * menu update routine for file view case.
 **/
static void
update_fileview(GDiffWindow *gdwin, const Preference *pref)
{
	menu_filev_block(gdwin);
	
	/* top-level menus */
	if (GTK_WIDGET_VISIBLE(main_menu[MENUITEM_DIR_VIEW].widget))
		gtk_widget_hide(main_menu[MENUITEM_DIR_VIEW].widget);
	if (!GTK_WIDGET_VISIBLE(main_menu[MENUITEM_FILE_VIEW].widget))
		gtk_widget_show(main_menu[MENUITEM_FILE_VIEW].widget);
	if (GTK_WIDGET_VISIBLE(main_menu[MENUITEM_DVIEW_ACTION].widget))
		gtk_widget_hide(main_menu[MENUITEM_DVIEW_ACTION].widget);
	if (!GTK_WIDGET_VISIBLE(main_menu[MENUITEM_FVIEW_ACTION].widget))
		gtk_widget_show(main_menu[MENUITEM_FVIEW_ACTION].widget);

	/* various states */
	if (pref->fvpref.pane_mode == ONE_PANE) {
		gtk_check_menu_item_set_active(
			GTK_CHECK_MENU_ITEM(pane_radio_menu[MENUITEM_ONE_PANE].widget),
			TRUE);
	} else {
		gtk_check_menu_item_set_active(
			GTK_CHECK_MENU_ITEM(pane_radio_menu[MENUITEM_TWO_PANES].widget),
			TRUE);
	}
	if (pref->fvpref.show_line_num == TRUE) {
		gtk_check_menu_item_set_active(
			GTK_CHECK_MENU_ITEM(line_num_radio_menu[MENUITEM_LINE_SHOW].widget),
			TRUE);
	} else {
		gtk_check_menu_item_set_active(
			GTK_CHECK_MENU_ITEM(line_num_radio_menu[MENUITEM_LINE_HIDE].widget),
			TRUE);
	}
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(fileview_menu[MENUITEM_TEXT_WRAP].widget),
		pref->fvpref.line_wrap);

	/* If line-wrap is on, I have to disable some actions */
	gtk_widget_set_sensitive(fview_action_menu[MENUITEM_NEXT_DIFF].widget,
							 !pref->fvpref.line_wrap);
	gtk_widget_set_sensitive(fview_action_menu[MENUITEM_PREV_DIFF].widget,
							 !pref->fvpref.line_wrap);
	gtk_widget_set_sensitive(fview_action_menu[MENUITEM_REL_NEXT_DIFF].widget,
							 !pref->fvpref.line_wrap);
	gtk_widget_set_sensitive(fview_action_menu[MENUITEM_REL_PREV_DIFF].widget,
							 !pref->fvpref.line_wrap);
	gtk_widget_set_sensitive(fview_action_menu[MENUITEM_FIRST_DIFF].widget,
							 !pref->fvpref.line_wrap);
	gtk_widget_set_sensitive(fview_action_menu[MENUITEM_LAST_DIFF].widget,
							 !pref->fvpref.line_wrap);

	menu_filev_unblock(gdwin);
}

/* block signal handers during update menus */
static void
menu_filev_block(GDiffWindow *gdwin)
{
	gtk_signal_handler_block_by_func(
		GTK_OBJECT(pane_radio_menu[MENUITEM_ONE_PANE].widget),
		GTK_SIGNAL_FUNC(mode_change_cb), gdwin);
	gtk_signal_handler_block_by_func(
		GTK_OBJECT(line_num_radio_menu[MENUITEM_LINE_SHOW].widget),
		GTK_SIGNAL_FUNC(line_num_show_cb), gdwin);
	gtk_signal_handler_block_by_func(
		GTK_OBJECT(fileview_menu[MENUITEM_TEXT_WRAP].widget),
		GTK_SIGNAL_FUNC(text_wrap_cb), gdwin);
}	

static void
menu_filev_unblock(GDiffWindow *gdwin)
{
	gtk_signal_handler_unblock_by_func(
		GTK_OBJECT(pane_radio_menu[MENUITEM_ONE_PANE].widget),
		GTK_SIGNAL_FUNC(mode_change_cb), gdwin);
	gtk_signal_handler_unblock_by_func(
		GTK_OBJECT(line_num_radio_menu[MENUITEM_LINE_SHOW].widget),
		GTK_SIGNAL_FUNC(line_num_show_cb), gdwin);
	gtk_signal_handler_unblock_by_func(
		GTK_OBJECT(fileview_menu[MENUITEM_TEXT_WRAP].widget),
		GTK_SIGNAL_FUNC(text_wrap_cb), gdwin);
}	
	


/**
 * file_open_cb:
 **/
static void
file_open_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;

	files_open(gdwin);
}

/**
 * file_close_cb:
 **/
static void
file_close_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean isdir;
	void *p;
	GDiffDirViews *gdviews;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir) {
		gdviews = p;
		gdiff_dirviews_delete(gdviews);
	} else {
		GDiffFileViews *gfviews = p;
		gdviews = gfviews->gdirviews;
		if (gdviews->isdir)
			gtk_widget_hide(GTK_WIDGET(gfviews->base));			
		else /* Comparing two files */
			gdiff_dirviews_delete(gdviews);
	}

	/* When every view is closed, update menu */
	if (gdwin->gdirv_list == NULL)
		menu_update(gdwin, &g_pref, MENUSTAT_NO_VIEW);
}

/**
 * mode_change_cb:
 * Change the mode(onepane or twopane).
 * Update a global variable, g_pref(it's redundant, as it's updated in act_fv_mode_change()).
 **/
static void
mode_change_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	PaneMode pmode;
	gboolean isdir;
	void *p;

	pmode = GTK_CHECK_MENU_ITEM(widget)->active ? ONE_PANE : TWO_PANE;
	g_pref.fvpref.pane_mode = pmode;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == FALSE) {
		GDiffFileViews *gfviews = p;
		act_fv_mode_change(gfviews, pmode);
	}
}

/**
 * line_num_show_cb:
 * Toggle show(hide) the line numbers for the current view.
 * Update a global variable, "g_pref"(it's redundant, as it's updated in act_fv_linenum_show()).
 **/
static void
line_num_show_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean b_show_line_num;
	gboolean isdir;
	void *p;

	b_show_line_num = GTK_CHECK_MENU_ITEM(widget)->active ? TRUE : FALSE;
	g_pref.fvpref.show_line_num = b_show_line_num;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == FALSE) {
		GDiffFileViews *gfviews = p;
		act_fv_linenum_show(gfviews, b_show_line_num);
	}
}

/**
 * go_dirview_cb:
 * If comparing two directories, and the current view is a file view,
 * look for the appropriate directory view, and focus it.
 **/
static void
go_dirview_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean isdir;
	void *p;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == FALSE) {
		GDiffFileViews *gfviews = p;
		act_fv_go_dirview(gfviews);
	}
}

/**
 * move_first_cb:
 **/
static void
move_first_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean isdir;
	void *p;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == FALSE) {
		GDiffFileViews *gfviews = p;
		act_fv_move_diff(gfviews, MOVED_FIRST);
	}
}

/**
 * move_last_cb:
 **/
static void
move_last_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean isdir;
	void *p;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == FALSE) {
		GDiffFileViews *gfviews = p;
		act_fv_move_diff(gfviews, MOVED_LAST);
	}
}

/**
 * move_next_cb:
 **/
static void
move_next_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean isdir;
	void *p;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == FALSE) {
		GDiffFileViews *gfviews = p;
		act_fv_move_diff(gfviews, MOVED_NEXT);
	}
}

/**
 * move_prev_cb:
 **/
static void
move_prev_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean isdir;
	void *p;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == FALSE) {
		GDiffFileViews *gfviews = p;
		act_fv_move_diff(gfviews, MOVED_PREV);
	}
}

/**
 * move_rel_next_cb:
 * Move relative next difference.
 **/
static void
move_rel_next_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean isdir;
	void *p;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == FALSE) {
		GDiffFileViews *gfviews = p;
		act_fv_move_diff(gfviews, MOVED_REL_NEXT);
	}
}

/**
 * move_rel_prev_cb:
 * Move relative previous difference.
 **/
static void
move_rel_prev_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean isdir;
	void *p;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == FALSE) {
		GDiffFileViews *gfviews = p;
		act_fv_move_diff(gfviews, MOVED_REL_PREV);
	}
}

/**
 * text_wrap_cb:
 * Update a global variable, "g_pref"(it's redundant, as it's updated in act_fv_toggle_textwrap()).
 **/
static void
text_wrap_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean b_line_wrap;
	gboolean isdir;
	void *p;

	b_line_wrap = GTK_CHECK_MENU_ITEM(widget)->active ? TRUE : FALSE;
	g_pref.fvpref.line_wrap = b_line_wrap;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == FALSE) {
		GDiffFileViews *gfviews = p;
		act_fv_toggle_textwrap(gfviews, b_line_wrap);
	}
}

/**
 * set_preference_cb:
 **/
static void
set_preference_cb(GtkWidget *widget, gpointer data)
{
	properties_show();
}

/**
 * show_path_cb:
 * Update a global variable, "g_pref"(it's redundant, as it's updated in act_dv_show_path()).
 **/
static void
show_path_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean b_show_path;
	gboolean isdir;
	void *p;

	b_show_path = GTK_CHECK_MENU_ITEM(widget)->active ? TRUE : FALSE;
	g_pref.dvpref.show_path = b_show_path;
	
	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == TRUE) {
		GDiffDirViews *gdviews = p;
		act_dv_show_path(gdviews, b_show_path);
	}
}

/**
 * row_hide_func_cb:
 * Use the index kept in menu item widget to know what function is chosen.
 **/
static void
row_hide_func_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean isdir;
	void *p;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == TRUE) {
		GDiffDirViews *gdviews = p;
		gboolean b_hide;
		int i;

		i = GPOINTER_TO_INT(gtk_object_get_user_data(GTK_OBJECT(widget)));
		b_hide = (gboolean)GTK_CHECK_MENU_ITEM(widget)->active;
		if (b_hide == TRUE)
			act_dv_add_rowhide(gdviews, rh_func_array[i].rh_func);
		else
			act_dv_remove_rowhide(gdviews, rh_func_array[i].rh_func);
	}
}

/**
 * row_hide_stat_cb:
 * Use the index kept in menu item widget to know what function is chosen.
 **/
static void
row_hide_stat_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean isdir;
	void *p;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == TRUE) {
		GDiffDirViews *gdviews = p;
		gboolean b_hide;
		FilesSpecialStatus hide_mask = 0;
		int i;

		i = GPOINTER_TO_INT(gtk_object_get_user_data(GTK_OBJECT(widget)));
		if (i == MENUITEM_HIDE_STAT_BINARY)
			hide_mask = BINARY_FILES;
		else if (i == MENUITEM_HIDE_STAT_ONLY)
			hide_mask = (ONLY_FILE1_EXISTS | ONLY_FILE2_EXISTS);
		else if (i == MENUITEM_HIDE_STAT_DIFF)
			hide_mask = DIFFERENT_FILES;
		else {
			g_assert_not_reached();
		}
		b_hide = (gboolean)GTK_CHECK_MENU_ITEM(widget)->active;
		act_dv_rowhide_stat(gdviews, hide_mask, b_hide);
	}
}

/**
 * show_tabs_cb:
 **/
static void
show_tabs_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean isdir;
	void *p;
	GtkNotebook *notebook;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir == TRUE) {
		GDiffDirViews *gdviews = p;
		notebook = gdviews->gdwin->notebook;
	} else {
		GDiffFileViews *gfviews = p;
		notebook = gfviews->gdwin->notebook;
	}
	
	g_pref.winpref.show_tabs = GTK_CHECK_MENU_ITEM(widget)->active ? TRUE : FALSE;
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), g_pref.winpref.show_tabs);
}

/**
 * reload_cb:
 **/
static void
reload_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean isdir;
	void *p;
	GDiffDirViews *gdviews;

	p = gdiff_current_views(gdwin, &isdir);
	if (p == NULL) {
		return;
	}
	if (isdir) {
		gdviews = p;
		act_dv_reload(gdviews);
	} else {
		GDiffFileViews *gfviews = p;
		gdviews = gfviews->gdirviews;
		if (gdviews->isdir)
			act_fv_reload(gfviews);
		else /* Comparing two files */
			act_dv_reload(gdviews);			
	}
}


/**
 * quit_cb:
 **/
static void
quit_cb(GtkWidget *widget, gpointer data)
{
	gtkdiff_exit(NULL, NULL);
}

/**
 * about_cb:
 **/
static void
about_cb(GtkWidget *widget, gpointer data)
{
	static GtkWidget *about = NULL;
	static const gchar *authors[] = {
		"INOUE Seiichiro <inoue@ainet.or.jp>",
		NULL
	};

	if (about) {
		gdk_window_raise(GTK_WIDGET(about)->window);
	} else {
		about = gnome_about_new(APPNAME, VERSION,
								_("Copyright 1999, under the GNU General Public License."),
								authors,
								/* another comments */
								_("GTK+(GNOME) diff front-end."),
								NULL);
		/* follow ghex technique */
		gtk_signal_connect(GTK_OBJECT(about), "destroy",
						   GTK_SIGNAL_FUNC(about_destroy_cb), &about);
		gtk_window_set_position(GTK_WINDOW(about), GTK_WIN_POS_CENTER);
		gtk_widget_show(about);
	}
	return;
}

/**
 * about_destroy_cb:
 **/
static void
about_destroy_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget **p_about = data;
	*p_about = NULL;
}
