/*
 * Copyright (c) 1997, 1998, 1999  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <syslog.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#ifndef HAVE_STRCHR
#define strchr index
#define strrchr rindex
#endif /* HAVE_STRCHR */

/*
 * The maximum length of path name.
 */
#ifndef PATH_MAX
#ifdef MAXPATHLEN
#define PATH_MAX        MAXPATHLEN
#else /* not MAXPATHLEN */
#define PATH_MAX        1024
#endif /* not MAXPATHLEN */
#endif /* not PATH_MAX */

#include <eb/eb.h>
#include <eb/appendix.h>

#include "ndtpd.h"
#include "confutil.h"
#include "filename.h"
#include "hostname.h"
#include "readconf.h"
#include "permission.h"

#ifdef USE_FAKELOG
#include "fakelog.h"
#endif


/*
 * Unexported functions.
 */
static int conf_begin             NDTPD_P((const char *, const char *, int));
static int conf_end               NDTPD_P((const char *, const char *, int));
static int conf_server_name       NDTPD_P((const char *, const char *, int));
static int conf_ndtp_port         NDTPD_P((const char *, const char *, int));
static int conf_user              NDTPD_P((const char *, const char *, int));
static int conf_group             NDTPD_P((const char *, const char *, int));
static int conf_max_clients       NDTPD_P((const char *, const char *, int));
static int conf_hosts             NDTPD_P((const char *, const char *, int));
static int conf_timeout           NDTPD_P((const char *, const char *, int));
static int conf_ident_hosts       NDTPD_P((const char *, const char *, int));
static int conf_ident_timeout     NDTPD_P((const char *, const char *, int));
static int conf_work_path         NDTPD_P((const char *, const char *, int));
static int conf_max_hits          NDTPD_P((const char *, const char *, int));
static int conf_max_text_size     NDTPD_P((const char *, const char *, int));
static int conf_syslog_facility   NDTPD_P((const char *, const char *, int));
static int conf_begin_book        NDTPD_P((const char *, const char *, int));
static int conf_end_book          NDTPD_P((const char *, const char *, int));
static int conf_book_name         NDTPD_P((const char *, const char *, int));
static int conf_book_title        NDTPD_P((const char *, const char *, int));
static int conf_book_path         NDTPD_P((const char *, const char *, int));
static int conf_appendix_path     NDTPD_P((const char *, const char *, int));
static int conf_book_max_clients  NDTPD_P((const char *, const char *, int));
static int conf_book_hosts        NDTPD_P((const char *, const char *, int));
static int conf_book_alias_eiwa   NDTPD_P((const char *, const char *, int));
static int conf_book_alias_waei   NDTPD_P((const char *, const char *, int));
static int conf_book_alias_kojien NDTPD_P((const char *, const char *, int));

/*
 * Syntax of the configuration file.
 */
Configuration configuration_table[] = {
    {"", "begin",          conf_begin,          READCONF_ZERO_OR_ONCE, 0},
    {"", "end",            conf_end,            READCONF_ZERO_OR_ONCE, 0},
    /* single directives. */
    {"", "server-name",    conf_server_name,    READCONF_ZERO_OR_ONCE, 0},
    {"", "ndtp-port",      conf_ndtp_port,      READCONF_ZERO_OR_ONCE, 0},
    {"", "user",           conf_user,           READCONF_ZERO_OR_ONCE, 0},
    {"", "group",          conf_group,          READCONF_ZERO_OR_ONCE, 0},
    {"", "max-clients",    conf_max_clients,    READCONF_ZERO_OR_ONCE, 0},
    {"", "hosts",          conf_hosts,          READCONF_ZERO_OR_MORE, 0},
    {"", "timeout",        conf_timeout,        READCONF_ZERO_OR_ONCE, 0},
    {"", "ident-hosts",    conf_ident_hosts,    READCONF_ZERO_OR_MORE, 0},
    {"", "ident-timeout",  conf_ident_timeout,  READCONF_ZERO_OR_ONCE, 0},
    {"", "work-path",      conf_work_path,      READCONF_ZERO_OR_ONCE, 0},
    {"", "max-hits",       conf_max_hits,       READCONF_ZERO_OR_ONCE, 0},
    {"", "max-text-size",  conf_max_text_size,  READCONF_ZERO_OR_ONCE, 0},
    {"", "syslog-facility",conf_syslog_facility,READCONF_ZERO_OR_ONCE, 0},
    /* book group directive. */
    {"",     "begin book",   conf_begin_book,       READCONF_ZERO_OR_MORE, 0},
    {"book", "name",         conf_book_name,        READCONF_ONCE,         0},
    {"book", "title",        conf_book_title,       READCONF_ONCE,         0},
    {"book", "path",         conf_book_path,        READCONF_ONCE,         0},
    {"book", "appendix-path",conf_appendix_path,    READCONF_ZERO_OR_ONCE, 0},
    {"book", "max-clients",  conf_book_max_clients, READCONF_ZERO_OR_ONCE, 0},
    {"book", "hosts",        conf_book_hosts,       READCONF_ZERO_OR_MORE, 0},
    {"book", "alias-eiwa",   conf_book_alias_eiwa,  READCONF_ZERO_OR_ONCE, 0},
    {"book", "alias-waei",   conf_book_alias_waei,  READCONF_ZERO_OR_ONCE, 0},
    {"book", "alias-kojien", conf_book_alias_kojien,READCONF_ZERO_OR_ONCE, 0},
    {"",     "end book",     conf_end_book,         READCONF_ZERO_OR_MORE, 0},
    {"",     NULL,           NULL,                  READCONF_ONCE,         0}
};


/*
 * The hook is called before reading a configuration file.
 */
static int
conf_begin(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    char server_address[MAXLEN_HOSTNAME + 1];

    /*
     * Reset lists.
     */
    clear_book_registry();
    clear_permission(&permissions);
    clear_permission(&identifications);

    /*
     * Initialize global variables.
     */
    server_name[0] = '\0';
    user_id = getuid();
    group_id = getgid();
    max_clients = DEFAULT_MAX_CLIENTS;
    idle_timeout = DEFAULT_IDLE_TIMEOUT;
    identification_timeout = DEFAULT_IDENT_TIMEOUT;
    max_hits = DEFAULT_MAX_HITS;
    max_text_size = DEFAULT_MAX_TEXT_SIZE;
    listening_port = -1;
    alias_eiwa[0] = '\0';
    alias_waei[0] = '\0';
    alias_kojien[0] = '\0';

    /*
     * The length of the filename "<work-path>/<basename>"
     * must not exceed PATH_MAX.
     */
    if (PATH_MAX < strlen(DEFAULT_WORK_PATH) + 1 + MAXLEN_WORK_PATH_BASENAME) {
	syslog(LOG_ERR, "internal error, too long DEFAULT_WORK_PATH");
	return -1;
    }
    strcpy(work_path, DEFAULT_WORK_PATH);

    /*
     * Get a host name of the server.
     */
    if (identify_my_host(server_name, server_address) < 0)
	syslog(LOG_ERR, "warning, cannot get a server hostname");

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s: debug: beginning of the configuration", filename);

    return 0;
}


/*
 * The hook is called just after reading a configuration file.
 */
static int
conf_end(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    size_t worklen;

    /*
     * If the configuration file has no `hosts' directive, and if
     * the mode is SERVER_MODE_CHECK, then output a warning message.
     */
    if (server_mode == SERVER_MODE_CHECK
	&& count_permission(&permissions) == 0) {
	syslog(LOG_ERR, "warning: nobody can connect to the server \
because the configuration file has no `hosts' directive");
    }

    /*
     * If the configuration file has no `book' group directive, and if
     * the mode is SERVER_MODE_CHECK, then output a warning message.
     */
    if (server_mode == SERVER_MODE_CHECK && count_book_registry() == 0)
	syslog(LOG_ERR, "warning: no book definition");

    /*
     * Check for the server port number.
     */
    if (listening_port < 0
	&& parse_tcp_port(DEFAULT_PORT, &listening_port) < 0) {
	syslog(LOG_ERR, "unknown service name: %s", DEFAULT_PORT);
	return -1;
    }

    /*
     * Set the pid filename.
     */
    sprintf(pid_filename, "%s/%s", work_path, PID_BASENAME);
    if (canonicalize_filename(pid_filename) < 0)
	return -1;

    /*
     * Set the connection lock filename.
     */
    sprintf(connection_lock_filename, "%s/%s", work_path, NDTPD_LOCK_BASENAME);

    /*
     * debugging information.
     */
    syslog(LOG_DEBUG, "%s: debug: end of the configuration", filename);

    return 0;
}

    
/*
 * `server-name' directive.
 */
static int
conf_server_name(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Check for the length of the server name.
     */
    if (MAXLEN_HOSTNAME < strlen(arg)) {
	syslog(LOG_ERR, "%s:%d: too long hostname", filename, lineno);
	return -1;
    }

    /*
     * Set the server name.
     */
    strcpy(server_name, arg);

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set server-name: %s",
	filename, lineno, server_name);

    return 0;
}


/*
 * `ndtp-port' directive.
 */
static int
conf_ndtp_port(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Parse and set the port number.
     */
    if (parse_tcp_port(arg, &listening_port) < 0) {
	syslog(LOG_ERR, "%s:%d: unknown service: %s", filename, lineno, arg);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set ndtp-port: %d", filename,
	lineno, listening_port);

    return 0;
}


/*
 * `user' directive.
 */
static int
conf_user(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Parse and set the user ID.
     */
    if (parse_user(arg, &user_id) < 0) {
	syslog(LOG_ERR, "%s:%d: unknown user: %s", filename, lineno, arg);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set user: %d", filename, lineno,
	(int)user_id);

    return 0;
}


/*
 * `group' directive.
 */
static int
conf_group(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Parse and set the group ID.
     */
    if (parse_group(arg, &group_id) < 0) {
	syslog(LOG_ERR, "%s:%d: unknown group: %s", filename, lineno, arg);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set group: %d", filename, lineno,
	(int)group_id);

    return 0;
}


/*
 * `max-clients' directive.
 */
static int
conf_max_clients(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Parse and set the max-clients.
     */
    if (parse_integer(arg, &max_clients) < 0) {
	syslog(LOG_ERR, "%s:%d: not an integer: %s", filename, lineno, arg);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set max-clients: %d", filename,
	lineno, max_clients);

    return 0;
}


/*
 * `hosts' directive.
 */
static int
conf_hosts(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Add `arg' to the host permission list.
     */
    if (add_permission(&permissions, arg) < 0) {
	syslog(LOG_ERR, "%s:%d: memory exhausted", filename, lineno);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: add hosts: %s", filename, lineno, arg);

    return 0;
}


/*
 * `timeout' directive.
 */
static int
conf_timeout(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Parse and set the timeout.
     */
    if (parse_integer(arg, &idle_timeout) < 0) {
	syslog(LOG_ERR, "%s:%d: not an integer: %s", filename, lineno, arg);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set timeout: %d", filename,
	lineno, idle_timeout);

    return 0;
}


/*
 * `ident-hosts' directive.
 */
static int
conf_ident_hosts(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Add `arg' to the ideintification list.
     */
    if (add_permission(&identifications, arg) < 0) {
	syslog(LOG_ERR, "%s:%d: memory exhausted", filename, lineno);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: add ident: %s", filename, lineno, arg);

    return 0;
}


/*
 * `ident-timeout' directive.
 */
static int
conf_ident_timeout(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Parse and set the ident timeout.
     */
    if (parse_integer(arg, &identification_timeout) < 0) {
	syslog(LOG_ERR, "%s:%d: not an integer: %s", filename, lineno, arg);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set ident-timeout: %d", filename,
	lineno, identification_timeout);

    return 0;
}


/*
 * `work-path' directive.
 */
static int
conf_work_path(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Check for the length of the path.
     * The length of `<work-path>/<basename>' must not exceed PATH_MAX.
     */
    if (PATH_MAX < strlen(arg) + 1 + MAXLEN_WORK_PATH_BASENAME) {
	syslog(LOG_ERR, "%s:%d: too long path", filename, lineno);
	return -1;
    }

    /*
     * The filename must be an absolute path.
     */
    if (*arg != '/') {
	syslog(LOG_ERR, "%s:%d: not an absolute path: %s", filename, lineno,
	    arg);
	return -1;
    }

    /*
     * Set the working directory path.
     */
    strcpy(work_path, arg);

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set work-path: %s", filename, lineno,
	work_path);

    return 0;
}


/*
 * `max-hits' directive.
 */
static int
conf_max_hits(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Parse and set the max hits.
     */
    if (parse_integer(arg, &max_hits) < 0) {
	syslog(LOG_ERR, "%s:%d: not an integer: %s", filename, lineno, arg);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set max-hits: %d", filename,
	lineno, max_hits);

    return 0;
}


/*
 * `max-text-size' directive.
 */
static int
conf_max_text_size(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    int size;

    /*
     * Parse and set the max text size.
     */
    if (parse_integer(arg, &size) < 0) {
	syslog(LOG_ERR, "%s:%d: not an integer: %s", filename, lineno, arg);
	return -1;
    }
    max_text_size = size;

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set max-text-size: %d", filename,
	lineno, max_text_size);

    return 0;
}


/*
 * `syslog-facility' directive.
 */
static int
conf_syslog_facility(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Parse and set a syslog facility.
     */
    if (parse_syslog_facility(arg, &syslog_facility) < 0) {
	syslog(LOG_ERR, "%s:%d: unknown syslog facility: %s", filename,
	    lineno, arg);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set syslog-facility: %s", filename,
	lineno, arg);

    return 0;
}


/*
 * `begin book' sequence.
 */
static int
conf_begin_book(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Create a new book node.
     */
    if (add_book() < 0) {
	syslog(LOG_ERR, "%s:%d: memory exhausted", filename, lineno);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: beginning of the book definition",
	filename, lineno);

    return 0;
}


/*
 * `end book' sequence.
 */
static int
conf_end_book(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    char tmpbuf[MAXLEN_BOOK_NAME + 1 + EB_MAXLEN_BASENAME + 1];

    /*
     * Convert aliases to "book-name/subbook-name" style.
     */
    if (alias_eiwa[0] != '\0' && strchr(alias_eiwa, '/') == NULL) {
	sprintf(tmpbuf, "%s/%s", current_book->name, alias_eiwa);
	strcpy(alias_eiwa, tmpbuf);
    }
    if (alias_waei[0] != '\0' && strchr(alias_waei, '/') == NULL) {
	sprintf(tmpbuf, "%s/%s", current_book->name, alias_waei);
	strcpy(alias_waei, tmpbuf);
    }
    if (alias_kojien[0] != '\0' && strchr(alias_kojien, '/') == NULL) {
	sprintf(tmpbuf, "%s/%s", current_book->name, alias_kojien);
	strcpy(alias_kojien, tmpbuf);
    }

    /*
     * If the group directive has no `hosts' sub-directive, and if
     * the mode is SERVER_MODE_CHECK, then output a warning message.
     */
    if (server_mode == SERVER_MODE_CHECK
	&& count_permission(&current_book->permissions) == 0) {
	syslog(LOG_ERR, "%s:%d: warning: nobody can access the book \
because the book definition has no `hosts' sub-directive",
	    filename, lineno);
    }

    /*
     * Set `max-clients' for the book to 0 when test mode.
     */
    if (server_mode == SERVER_MODE_TEST) {
	current_book->max_clients = 0;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: end of the book definition",
	filename, lineno);

    return 0;
}


/*
 * `name' directive (in book structure).
 */
static int
conf_book_name(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    const char *p;

    /*
     * Check for the length of the book name.
     */
    if (MAXLEN_BOOK_NAME < strlen(arg)) {
	syslog(LOG_ERR, "%s:%d: too long book name", filename, lineno);
	return -1;
    }

    /*
     * The tagname must consist of upper letters, digit, '_', and '-'.
     */
    for (p = arg; *p != '\0'; p++) {
	if (!isupper(*p) && !isdigit(*p) && *p != '_' && *p != '-') {
	    syslog(LOG_ERR,
		"%s:%d: the book name contains invalid character: %s",
		filename, lineno, arg);
	    return -1;
	}
    }

    /*
     * Check for uniqness of book names.
     */
    if (find_book(arg) != NULL) {
	syslog(LOG_ERR, "%s:%d: the book name redefined: %s", filename,
	    lineno, arg);
	return -1;
    }

    /*
     * Set the book name.
     */
    strcpy(current_book->name, arg);

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set book-name: %s", filename,
	lineno, current_book->name);

    return 0;
}


/*
 * `title' directive (in book structure).
 */
static int
conf_book_title(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Check for the length of the title.
     */
    if (MAXLEN_BOOK_TITLE < strlen(arg)) {
	syslog(LOG_ERR, "%s:%d: too long book title", filename, lineno);
	return -1;
    }

    /*
     * Set the title.
     */
    strcpy(current_book->title, arg);

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set book-title: %s", filename,
	lineno, current_book->title);

    return 0;
}    


/*
 * `path' directive (in book structure).
 */
static int
conf_book_path(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    size_t len;

    /*
     * Check for the length of the path.
     */
    len = strlen(arg);
    if (PATH_MAX < len) {
	syslog(LOG_ERR, "%s:%d: too long path", filename, lineno);
	return -1;
    }

    /*
     * The path must be an absolute path.
     */
    if (*arg != '/') {
	syslog(LOG_ERR, "%s:%d: not an absolute path: %s", filename, lineno,
	    arg);
	return -1;
    }

    /*
     * Set the book path.
     */
    current_book->path = (char *)malloc(strlen(arg) + 1);
    if (current_book->path == NULL) {
	syslog(LOG_ERR, "%s:%d: memory exhausted", filename, lineno);
	return -1;
    }
    strcpy(current_book->path, arg);

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set book-path: %s", filename,
	lineno, arg);

    return 0;
}


/*
 * `appendix-path' directive.
 */
static int
conf_appendix_path(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    size_t len;

    /*
     * Check for the length of the path.
     */
    len = strlen(arg);
    if (PATH_MAX < len) {
	syslog(LOG_ERR, "%s:%d: too long appendix path", filename);
	return -1;
    }

    /*
     * The path must be an absolute path.
     */
    if (*arg != '/') {
	syslog(LOG_ERR, "%s:%d: not an absolute path: %s", filename, lineno,
	    arg);
	return -1;
    }

    /*
     * Set the book path.
     */
    current_book->appendix_path = (char *)malloc(strlen(arg) + 1);
    if (current_book->appendix_path == NULL) {
	syslog(LOG_ERR, "%s:%d: memory exhausted", filename, lineno);
	return -1;
    }
    strcpy(current_book->appendix_path, arg);

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set appendix-path: %s", filename,
	lineno, arg);

    return 0;
}


/*
 * `max-clients' directive (in book structure).
 */
static int
conf_book_max_clients(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * An argument represents a remote hostname or address.
     */
    if (parse_integer(arg, &current_book->max_clients) < 0) {
	syslog(LOG_ERR, "%s:%d: not an integer: %s", filename, lineno, arg);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set book-max-clients: %d",
	filename, lineno, current_book->max_clients);

    return 0;
}


/*
 * `hosts' directive (in book structure).
 */
static int
conf_book_hosts(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    /*
     * Add `arg' to the permission list.
     */
    if (add_permission(&current_book->permissions, arg) < 0) {
	syslog(LOG_ERR, "%s:%d: memory exhausted", filename, lineno);
	return -1;
    }

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: add book-hosts: %s", filename, lineno,
	arg);

    return 0;
}


/*
 * `alias-eiwa' directive (in book structure).
 */
static int
conf_book_alias_eiwa(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    const char *p;

    /*
     * Check for the length of the subbook name.
     */
    if (EB_MAXLEN_BASENAME < strlen(arg)) {
	syslog(LOG_ERR, "%s:%d: too long aliased name", filename, lineno);
	return -1;
    }

    /*
     * The subbook name must consist of upper letters, digit, '_' and '-'.
     */
    for (p = arg; *p != '\0'; p++) {
	if (!isupper(*p) && !isdigit(*p) && *p != '_' && *p != '-') {
	    syslog(LOG_ERR, "%s:%d: the name contains invalid character: %s",
		filename, lineno, arg);
	    return -1;
	}
    }

    /*
     * Check for redefinition.
     */
    if (alias_eiwa[0] != '\0') {
	syslog(LOG_ERR, "%s:%d: redifined: alias-eiwa", filename, lineno);
	return -1;
    }

    /*
     * Set the alias `eiwa'.
     */
    strcpy(alias_eiwa, arg);

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set alias-eiwa: %s", filename,
	lineno, alias_eiwa);

    return 0;
}


/*
 * `alias-waei' directive (in book structure).
 */
static int
conf_book_alias_waei(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    const char *p;

    /*
     * Check for the length of the subbook name.
     */
    if (EB_MAXLEN_BASENAME < strlen(arg)) {
	syslog(LOG_ERR, "%s:%d: too long aliased name", filename, lineno);
	return -1;
    }

    /*
     * The subbook name must consist of upper letters, digit, '_' and '-'.
     */
    for (p = arg; *p != '\0'; p++) {
	if (!isupper(*p) && !isdigit(*p) && *p != '_' && *p != '-') {
	    syslog(LOG_ERR, "%s:%d: the name contains invalid character: %s",
		filename, lineno, arg);
	    return -1;
	}
    }

    /*
     * Set the alias `waei'.
     */
    if (alias_waei[0] != '\0') {
	syslog(LOG_ERR, "%s:%d: redefined: alias-waei", filename, lineno);
	return -1;
    }

    /*
     * Set the alias `waei'.
     */
    strcpy(alias_waei, arg);

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set alias-waei: %s", filename,
	lineno, alias_waei);

    return 0;
}


/*
 * `alias-kojien' directive (in book structure).
 */
static int
conf_book_alias_kojien(arg, filename, lineno)
    const char *arg;
    const char *filename;
    int lineno;
{
    const char *p;

    /*
     * Check for the length of the subbook name.
     */
    if (EB_MAXLEN_BASENAME < strlen(arg)) {
	syslog(LOG_ERR, "%s:%d: too long aliased name", filename, lineno);
	return -1;
    }

    /*
     * The subbook name must consist of upper letters, digit, '_' and '-'.
     */
    for (p = arg; *p != '\0'; p++) {
	if (!isupper(*p) && !isdigit(*p) && *p != '_' && *p != '-') {
	    syslog(LOG_ERR, "%s:%d: the name contains invalid character: %s",
		filename, lineno, arg);
	    return -1;
	}
    }

    if (alias_kojien[0] != '\0') {
	syslog(LOG_ERR, "%s:%d: redifined: alias-kojien", filename, lineno);
	return -1;
    }

    /*
     * Set the alias `kojien'.
     */
    strcpy(alias_kojien, arg);

    /*
     * Debugging information.
     */
    syslog(LOG_DEBUG, "%s:%d: debug: set alias-kojien: %s", filename,
	lineno, alias_kojien);

    return 0;
}


