/*
 * Copyright (c) 1997, 1998, 1999  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <errno.h>
#include <syslog.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#if defined(HAVE_SYS_WAIT_H) || defined(HAVE_UNION_WAIT)
#include <sys/wait.h>
#endif

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#ifndef HAVE_STRCHR
#define strchr index
#define strrchr rindex
#endif /* HAVE_STRCHR */

#ifndef HAVE_GETCWD
#define getcwd(d,n) getwd(d)
#endif

#ifndef PATH_MAX
#ifdef MAXPATHLEN
#define PATH_MAX        MAXPATHLEN
#else /* not MAXPATHLEN */
#define PATH_MAX        1024
#endif /* not MAXPATHLEN */
#endif /* not PATH_MAX */

#ifdef HAVE_UNION_WAIT
#ifndef WTERMSIG
#define WTERMSIG(x)     ((x).w_termsig)
#endif
#ifndef WCOREDUMP
#define WCOREDUMP(x)    ((x).w_coredump)
#endif
#ifndef WEXITSTATUS
#define WEXITSTATUS(x)  ((x).w_retcode)
#endif
#ifndef WIFSIGNALED
#define WIFSIGNALED(x)  (WTERMSIG(x) != 0)
#endif
#ifndef WIFEXITED
#define WIFEXITED(x)    (WTERMSIG(x) == 0)
#endif
#else /* not HAVE_UNION_WAIT */
#ifndef WTERMSIG
#define WTERMSIG(x) ((x) & 0x7f)
#endif
#ifndef WCOREDUMP
#define WCOREDUMP(x) ((x) & 0x80)
#endif
#ifndef WEXITSTATUS
#define WEXITSTATUS(x) (((x) >> 8) & 0xff)
#endif
#ifndef WIFSIGNALED
#define WIFSIGNALED(x) (WTERMSIG (x) != 0)
#endif
#ifndef WIFEXITED
#define WIFEXITED(x) (WTERMSIG (x) == 0)
#endif
#endif  /* not HAVE_UNION_WAIT */


#include "confutil.h"
#include "daemon.h"
#include "filename.h"
#include "getopt.h"
#include "hostname.h"
#include "ident.h"
#include "linebuf.h"
#include "logpid.h"
#include "openmax.h"
#include "permission.h"
#include "privilege.h"
#include "serverport.h"
#include "signame.h"
#include "ticket.h"
#include "wildcard.h"

#include "ndtpd.h"

/*
 * Unexported functions.
 */
static void standalone_main NDTPD_P((void));
static void inetd_main NDTPD_P((void));
static void test_main NDTPD_P((void));
static void output_version NDTPD_P((void));
static void output_help NDTPD_P((void));
static void output_try_help NDTPD_P((void));
static void set_signal_handler NDTPD_P((int, RETSIGTYPE (*)(int)));
static void block_signal NDTPD_P((int));
static void unblock_signal NDTPD_P((int));
static RETSIGTYPE abort_parent NDTPD_P((int));
static RETSIGTYPE abort_child NDTPD_P((int));
static RETSIGTYPE terminate_parent NDTPD_P((int));
static RETSIGTYPE terminate_child NDTPD_P((int));
static RETSIGTYPE restart_parent NDTPD_P((int));
static RETSIGTYPE wait_child NDTPD_P((int));

/*
 * Program name.
 */
#define PROGRAM_NAME	"ndtpd"

/*
 * Command line options.
 */
static const char *short_options = "c:hitv";
static struct option long_options[] = {
    {"configuration-file", required_argument, NULL, 'c'},
    {"help",               no_argument,       NULL, 'h'},
    {"inetd",              no_argument,       NULL, 'i'},
    {"test",               no_argument,       NULL, 't'},
    {"version",            no_argument,       NULL, 'v'},
    {NULL, 0, NULL, 0}
};

/*
 * The flag is set when the server is started.
 */
static int once_started;

/*
 * The flag is set if SIGHUP is received.
 */
static int restart_trigger;

/*
 * Process group ID of the server.
 */
pid_t process_group_id;

/*
 * Signal mask.
 */
#if !defined(HAVE_SIGPROCMASK) && defined(HAVE_SIGSETMASK)
static int signal_mask = 0;
#endif


int
main(argc, argv)
    int argc;
    char *argv[];
{
    int ch;

    /*
     * Set program name and version.
     */
    invoked_name = argv[0];
    program_name = PROGRAM_NAME;
    program_version = VERSION;

    /*
     * Set umask.
     */
    umask(SERVER_UMASK);

    /*
     * Open syslog.
     * If LOG_DAEMON is not defined, it assumes old style syslog
     * It doesn't have facility.
     */
    syslog_facility = DEFAULT_SYSLOG_FACILITY;
#ifdef LOG_DAEMON
    openlog(program_name, LOG_NDELAY | LOG_PID, syslog_facility);
#else
    openlog(program_name, LOG_NDELAY | LOG_PID);
#endif
    syslog(LOG_DEBUG, "debug: open syslog");

    /*
     * Initialize variables.
     */
    once_started = 0;
    listening_file = -1;
    server_mode = SERVER_MODE_STANDALONE;

    if (initialize_line_buffer(&line_buffer, MAXLEN_NDTP_LINE) < 0) {
	syslog(LOG_ERR, "memory exhausted");
	goto die;
    }

    if (PATH_MAX < strlen(DEFAULT_CONFIG_FILENAME)) {
	syslog(LOG_ERR, "internal error, too long DEFAULT_CONFIG_FILENAME");
	goto die;
    }
    strcpy(configuration_filename, DEFAULT_CONFIG_FILENAME);

    initialize_permission(&permissions);
    initialize_permission(&identifications);
    initialize_book_registry();
    initialize_hookset();
    initialize_ticket_stock(&connection_ticket_stock);

    /*
     * Parse command line options.
     */
    for (;;) {
	ch = getopt_long(argc, argv, short_options, long_options, NULL);
	if (ch == EOF)
	    break;
	switch (ch) {
	case 'c':
	    /*
	     * Option `-c file'.  Specify the configuration filename.
	     */
	    if (PATH_MAX < strlen(optarg)) {
		fprintf(stderr, "%s: too long configuration filename\n",
		    invoked_name);
		goto die;
	    }
	    strcpy(configuration_filename, optarg);
	    if (canonicalize_filename(configuration_filename) < 0)
		goto die;
	    break;

	case 'h':
	    /*
	     * Option `-h'.  Help.
	     */
	    output_help();
	    exit(0);

	case 'i':
	    /*
	     * Option `-i'.  Inetd mode.
	     */
	    server_mode = SERVER_MODE_INETD;
	    break;

	case 't':
	    /*
	     * Option `-t'.  Test mode.
	     */
	    server_mode = SERVER_MODE_TEST;
	    break;

	case 'v':
	    /*
	     * Option `-v'.  Show version and exit immediately.
	     */
	    output_version();
	    exit(0);

	default:
	    output_try_help();
	    exit(1);
	}
    }

    /*
     * Check for the number of rest arguments.
     */
    if (0 < argc - optind) {
	fprintf(stderr, "%s: too many arguments\n", invoked_name);
	output_try_help();
	exit(1);
    }

    /*
     * Main routine for each server mode.
     */
    switch (server_mode) {
    case SERVER_MODE_STANDALONE:
	for (;;)
	    standalone_main();
	break;
    case SERVER_MODE_INETD:
	inetd_main();
	break;
    case SERVER_MODE_TEST:
	test_main();
	break;
    }

    return 0;

    /*
     * A fatal error occurrs...
     */
  die:
    if (server_mode == SERVER_MODE_STANDALONE)
	syslog(LOG_CRIT, "the server exits");
    else if (server_mode == SERVER_MODE_INETD)
	syslog(LOG_INFO, "the server exits");

    exit(1);
}


/*
 * Main routine for standalone mode.
 */
static void
standalone_main()
{
    struct sockaddr_in addr;
    int len;
    int fd;
    int pid;

    /*
     * Block signals.
     */
#if !defined(HAVE_SIGPROCMASK) && defined(HAVE_SIGSETMASK)
    signal_mask = 0;
#endif
    block_signal(SIGHUP);
    block_signal(SIGINT);
    block_signal(SIGTERM);

    /*
     * Reset restart trigger.
     */
    restart_trigger = 0;
    
    /*
     * Kill child processes.
     */
    if (once_started) {
#ifdef HAVE_KILLPG
	killpg(process_group_id, SIGINT);
#else
	kill(-process_group_id, SIGINT);
#endif
	syslog(LOG_DEBUG, "debug: kill all child processes");
    }

    /*
     * Close unwanted files.
     */
    if (!once_started) {
	closelog();
	for (fd = get_open_max() - 1; 0 <= fd; fd--)
	    close(fd);
#ifdef LOG_DAEMON
	openlog(program_name, LOG_NDELAY | LOG_PID, syslog_facility);
#else
	openlog(program_name, LOG_NDELAY | LOG_PID);
#endif
	syslog(LOG_DEBUG, "debug: reopen syslog");
    }

    /*
     * Read a configuration file.
     */
    if (read_configuration(configuration_filename, configuration_table) < 0) {
	syslog(LOG_ERR, "configuration failure");
	goto die;
    }

    /*
     * Reopen syslog.
     */
    closelog();
#ifdef LOG_DAEMON
    openlog(program_name, LOG_NDELAY | LOG_PID, syslog_facility);
#else
    openlog(program_name, LOG_NDELAY | LOG_PID);
#endif

    /*
     * Run as standalone daemon.
     */
    if (!once_started) {
	if (daemonize() < 0)
	    goto die;
#ifdef GETPGRP_VOID
	process_group_id = getpgrp();
#else
	process_group_id = getpgrp(getpid());
#endif
    }

    /*
     * Change the current working directory.
     */
    if (chdir(work_path) < 0) {
	syslog(LOG_ERR, "cannot change the directory, %m: %s", work_path);
	goto die;
    }

    /*
     * Shutdown old connections if the server port is changed.
     */
    if (once_started && listening_port != old_listening_port) {
	if (shutdown(listening_file, 2) < 0) {
	    syslog(LOG_ERR, "shutdown() failed, %m");
	    listening_file = -1;
	    goto die;
	}
	listening_file = -1;
	syslog(LOG_DEBUG, "debug: shutdown the old socket: %d/tcp",
	    listening_port);
    }

    /*
     * Bind and listen to the server port. 
     */
    if (listening_file < 0) {
	listening_file = set_server_port(listening_port, LISTENING_BACKLOG);
	if (listening_file < 0)
	    goto die;
	old_listening_port = listening_port;
    }

    /*
     * Set UID and GID.
     */
    if (set_privilege(user_id, group_id) < 0)
	goto die;

    /*
     * Activate all books.
     */
    if (activate_book_registry() == 0)
	syslog(LOG_ERR, "no book is available");
	
    /*
     * Unlink an old PID file if exists.
     */
    if (once_started)
	remove_pid_file(old_pid_filename);
    strcpy(old_pid_filename, pid_filename);

    /*
     * Bind a ticket stock.
     */
    if (once_started)
	clear_ticket_stock(&connection_ticket_stock);
    if (max_clients != 0 && bind_ticket_stock(&connection_ticket_stock,
	connection_lock_filename, max_clients) < 0)
	goto die;

    /*
     * Log pid to the file.
     */
    if (log_pid_file(pid_filename) < 0)
	goto die;

    /*
     * Set signal handlers.
     */
    set_signal_handler(SIGBUS, abort_parent);
    set_signal_handler(SIGSEGV, abort_parent);
    set_signal_handler(SIGINT, SIG_IGN);
    set_signal_handler(SIGQUIT, SIG_IGN);
    set_signal_handler(SIGTERM, terminate_parent);
    set_signal_handler(SIGCHLD, wait_child);
    set_signal_handler(SIGHUP, restart_parent);

    /*
     * Set `once_started' flag.
     */
    if (once_started)
	syslog(LOG_NOTICE, "server restarted");
    else
	syslog(LOG_NOTICE, "server started");
    once_started = 1;

    /*
     * Main loop for the parent process.
     */
    for (;;) {
	/*
	 * Unblock fatal signals.
	 */
	unblock_signal(SIGHUP);
	unblock_signal(SIGINT);
	unblock_signal(SIGTERM);
	unblock_signal(SIGCHLD);

	/*
	 * Wait a connection from a client.
	 */ 
	if (wait_server_port(listening_file) < 0) {
	    if (errno != EINTR)
		goto die;
	    else if (restart_trigger)
		return;
	    else
		continue;
	}

	/*
	 * Block fatal signals.
	 */
	block_signal(SIGHUP);
	block_signal(SIGINT);
	block_signal(SIGTERM);
	block_signal(SIGCHLD);
	
	/*
	 * Accept a connection.
	 */
	len = sizeof(struct sockaddr_in);
	accepted_file = accept(listening_file, (struct sockaddr *)&addr, &len);
	if (accepted_file < 0) {
	    syslog(LOG_ERR, "accept() failed, %m");
	    continue;
	}
	syslog(LOG_DEBUG, "debug: accept the connection");

	/*
	 * Fork.  The child process talk to the client.
	 */
	pid = fork();
	if (pid < 0) {
	    syslog(LOG_ERR, "fork() failed, %m");
	    continue;
	} else if (pid == 0) {
	    syslog(LOG_DEBUG, "debug: forked");
	    break;
	}

	/*
	 * Close an accepted socket (at the parent process).
	 */
	if (close(accepted_file) < 0) {
	    syslog(LOG_ERR, "close() failed, %m");
	    goto die;
	}
    }

    /*
     * Set signal handlers.
     */
    set_signal_handler(SIGBUS, abort_child);
    set_signal_handler(SIGSEGV, abort_child);
    set_signal_handler(SIGINT, terminate_child);
    set_signal_handler(SIGTERM, terminate_child);
    set_signal_handler(SIGCHLD, SIG_IGN);
    set_signal_handler(SIGHUP, terminate_child);
    set_signal_handler(SIGPIPE, SIG_IGN);

    /*
     * Unblock fatal signals.
     */
    unblock_signal(SIGHUP);
    unblock_signal(SIGINT);
    unblock_signal(SIGTERM);
    unblock_signal(SIGCHLD);

    /*
     * Close the file to listen to the server port.
     */
    close(listening_file);

    /*
     * Get the client hostname and address.
     */
    if (identify_remote_host(accepted_file, client_hostname, client_address)
	< 0) {
	syslog(LOG_ERR, "connection denied: host=%s(%s)", client_hostname,
	    client_address);
	shutdown(accepted_file, 2);
	exit(1);
    }

    /*
     * Identify a remote user by Identification protocol, if needed.
     */
    if (test_permission(&identifications, client_hostname, client_address, 
	match_wildcard))
	identify_user(accepted_file, client_user, identification_timeout);
    else
	strcpy(client_user, UNKNOWN_USER);

    /*
     * Check for the access permission to the client.
     */
    if (!test_permission(&permissions, client_hostname, client_address,
	match_wildcard)) {
	syslog(LOG_ERR, "connection denied: user=%s, host=%s(%s)",
	    client_user, client_hostname, client_address);
	shutdown(accepted_file, 2);
	exit(0);
    }

    /*
     * Get a ticket for the client.
     */
    if (max_clients != 0) {
	if (get_ticket(&connection_ticket_stock) < 0) {
	    syslog(LOG_INFO, "full of clients");
	    shutdown(accepted_file, 2);
	    exit(0);
	}
    }

    syslog(LOG_INFO, "connected: user=%s, host=%s(%s)", client_user,
	client_hostname, client_address);

    /*
     * Set the signal handler, SIGALRM.
     */
    set_signal_handler(SIGALRM, terminate_child);

    /*
     * Set book permissions to the current client.
     */
    check_book_permissions();

    /*
     * Communicate with the client.
     */
    ndtp_main();
    shutdown(accepted_file, 2);

    /*
     * Clear lists and buffers.
     */
    clear_book_registry();
    clear_permission(&permissions);
    clear_permission(&identifications);
    clear_line_buffer(&line_buffer);
    clear_ticket_stock(&connection_ticket_stock);

    syslog(LOG_INFO, "the child server process exits");
    exit(0);

    /*
     * A fatal error occurrs on a parent...
     */
  die:
    if (0 <= listening_file)
	shutdown(listening_file, 2);

    if (once_started)
	remove_pid_file(old_pid_filename);

    if (once_started) {
	set_signal_handler(SIGINT, SIG_IGN);
#ifdef HAVE_KILLPG
	killpg(process_group_id, SIGINT);
#else
	kill(-process_group_id, SIGINT);
#endif
    }

    clear_book_registry();
    clear_permission(&permissions);
    clear_permission(&identifications);
    clear_line_buffer(&line_buffer);
    clear_ticket_stock(&connection_ticket_stock);

    syslog(LOG_CRIT, "the server exits");
    exit(1);
}


/*
 * Main routine for inetd mode.
 */
static void
inetd_main()
{
    accepted_file = 0;

    /*
     * Set signal handlers.
     */
    set_signal_handler(SIGBUS, abort_parent);
    set_signal_handler(SIGSEGV, abort_parent);
    set_signal_handler(SIGINT, SIG_IGN);
    set_signal_handler(SIGQUIT, SIG_IGN);
    set_signal_handler(SIGTERM, terminate_parent);
    set_signal_handler(SIGHUP, terminate_parent);
    set_signal_handler(SIGPIPE, SIG_IGN);
    set_signal_handler(SIGALRM, terminate_parent);

    /*
     * Read a configuration file.
     */
    if (read_configuration(configuration_filename, configuration_table) < 0) {
	syslog(LOG_ERR, "configuration failure");
	goto die;
    }

    /*
     * Reopen syslog.
     */
    closelog();
#ifdef LOG_DAEMON
    openlog(program_name, LOG_NDELAY | LOG_PID, syslog_facility);
#else
    openlog(program_name, LOG_NDELAY | LOG_PID);
#endif

    /*
     * Change the current working directory.
     */
    if (chdir(work_path) < 0) {
	syslog(LOG_ERR, "cannot change the directory, %m: %s", work_path);
	goto die;
    }

    /*
     * Set UID and GID.
     */
    if (set_privilege(user_id, group_id) < 0) {
	syslog(LOG_ERR,
	    "cannot set owner and group of the process");
	goto die;
    }

    /*
     * Get the client hostname and address.
     */
    if (identify_remote_host(accepted_file, client_hostname, client_address)
	< 0) {
	syslog(LOG_ERR, "connection denied: host=%s(%s)", client_hostname,
	    client_address);
	goto die;
    }

    /*
     * Identify a remote user by Identification protocol, if needed.
     */
    if (test_permission(&identifications, client_hostname, client_address, 
	match_wildcard))
	identify_user(accepted_file, client_user, identification_timeout);
    else
	strcpy(client_user, UNKNOWN_USER);

    /*
     * Check for the access permission to the client.
     */
    if (!test_permission(&permissions, client_hostname, client_address,
	match_wildcard)) {
	syslog(LOG_ERR, "connection denied: user=%s, host=%s(%s)",
	    client_user, client_hostname, client_address);
	shutdown(accepted_file, 2);
	exit(0);
    }

    /*
     * Get a ticket for the client.
     */
    if (max_clients != 0) {
	bind_ticket_stock(&connection_ticket_stock, connection_lock_filename,
	    max_clients);
	if (get_ticket(&connection_ticket_stock) < 0) {
	    syslog(LOG_INFO, "full of clients");
	    goto die;
	}
    }

    syslog(LOG_INFO, "connected: user=%s, host=%s(%s)", client_user,
	client_hostname, client_address);

    /*
     * Activate all books.
     */
    if (activate_book_registry() == 0)
	syslog(LOG_ERR, "no book is available");
	
    /*
     * Check book permissions to the current client.
     */
    check_book_permissions();

    /*
     * Communicate with the client.
     */
    ndtp_main();

    /*
     * Clear lists and buffers.
     */
    clear_book_registry();
    clear_permission(&permissions);
    clear_permission(&identifications);
    clear_line_buffer(&line_buffer);
    clear_ticket_stock(&connection_ticket_stock);

    syslog(LOG_INFO, "the server exits");
    exit(0);

    /*
     * A fatal error occurrs...
     */
  die:
    fflush(stderr);
    shutdown(accepted_file, 2);

    clear_book_registry();
    clear_permission(&permissions);
    clear_permission(&identifications);
    clear_line_buffer(&line_buffer);
    clear_ticket_stock(&connection_ticket_stock);

    syslog(LOG_INFO, "the server exits");
    exit(1);
}


/*
 * Main routine for inetd mode.
 */
static void
test_main()
{
    /*
     * Read a configuration file.
     */
    if (read_configuration(configuration_filename, configuration_table) < 0) {
	fprintf(stderr, "configuration failure\n");
	syslog(LOG_ERR, "configuration failure");
	exit(1);
    }

    /*
     * Activate all books.
     */
    if (activate_book_registry() == 0)
	syslog(LOG_ERR, "no book is available");
	
    /*
     * Set dummy hostname and address.
     */
    strcpy(client_address, "test-mode");
    if (isatty(0))
	strncpy(client_hostname, ttyname(0), MAXLEN_HOSTNAME);
    else
	strncpy(client_hostname, "stdin", MAXLEN_HOSTNAME);
    *(client_hostname + MAXLEN_HOSTNAME) = '\0';

    idle_timeout = 0;
    strcpy(client_user, UNKNOWN_USER);

    /*
     * Reopen syslog.
     */
    closelog();
#ifdef LOG_DAEMON
    openlog(program_name, LOG_NDELAY | LOG_PID, syslog_facility);
#else
    openlog(program_name, LOG_NDELAY | LOG_PID);
#endif

    syslog(LOG_INFO, "connected: user=%s, host=%s(%s)", client_user,
	client_hostname, client_address);

    /*
     * Set book permissions.
     */
    set_all_book_permissions();

    /*
     * Communicate with the client.
     */
    printf("# %s version %s\n", program_name, program_version);
    printf("# test mode (type `Q' to quit)\n");
    fflush(stdout);
    ndtp_main();

    /*
     * Clear lists and buffers.
     */
    clear_book_registry();
    clear_permission(&permissions);
    clear_permission(&identifications);
    clear_line_buffer(&line_buffer);
    clear_ticket_stock(&connection_ticket_stock);

    exit(0);
}


/*
 * Output version number to standard out.
 */
static void
output_version()
{
    printf("%s (NDTPD) version %s\n", program_name, program_version);
    printf("Copyright (c) 1997, 1998, 1999  Motoyuki Kasahara\n\n");
    printf("This is free software; you can redistribute it and/or modify\n");
    printf("it under the terms of the GNU General Public License as published by\n");
    printf("the Free Software Foundation; either version 2, or (at your option)\n");
    printf("any later version.\n\n");
    printf("This program is distributed in the hope that it will be useful,\n");
    printf("but WITHOUT ANY WARRANTY; without even the implied warranty\n");
    printf("of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
    printf("GNU General Public License for more details.\n");
    fflush(stdout);
}


/*
 * Output usage to standard out.
 */
static void
output_help()
{
    printf("Usage: %s [option...]\n", program_name);
    printf("Options:\n");
    printf("  -c FILE  --configuration-file FILE\n");
    printf("                             specify a configuration file\n");
    printf("                             (default: %s)\n",
	DEFAULT_CONFIG_FILENAME);
    printf("  -h  --help                 display this help, then exit\n");
    printf("  -i  --inetd                inetd mode\n");
    printf("  -t  --test                 test mode\n");
    printf("  -v  --version              display version number, then exit\n");
    printf("\nDefault value used in a configuration file:\n");
    printf("  work-path                  (default: %s)\n", DEFAULT_WORK_PATH);
    printf("\nReport bugs to %s.\n", MAILING_ADDRESS);
    fflush(stdout);
}


/*
 * Output ``try ...'' message to standard error.
 */
static void
output_try_help()
{
    fprintf(stderr, "try `%s --help' for more information\n", invoked_name);
    fflush(stderr);
}


/*
 * Set a signal handler.
 * `signal' on some SystemV based systems is not reliable.
 * We use `sigaction' if available.
 */
static void
set_signal_handler(sig, func)
    int sig;
    RETSIGTYPE (*func) NDTPD_P((int));
{
#ifdef HAVE_SIGPROCMASK
    struct sigaction act;

    act.sa_handler = func;
    sigemptyset(&act.sa_mask);
    act.sa_flags = SA_NOCLDSTOP;

    if (sig == SIGALRM) {
#ifdef SA_INTERRUPT
	act.sa_flags |= SA_INTERRUPT;
#endif
    } else {
#ifdef SA_RESTART
	act.sa_flags |= SA_RESTART;
#endif
    }
    sigaction(sig, &act, NULL);

#else /* not HAVE_SIGPROCMASK */
    signal(sig, func);
#endif /* not HAVE_SIGPROCMASK */
}


/*
 * Block signals of `sig' type (SIGHUP, SIGINT, ...).
 *
 * `sigprocmask' is used, if available.
 * Otherwise, `sigsetmask' is used if the system has it.
 * If neither `sigprocmask' nor `sigsetmask' is available, we give up
 * blocking a signal.
 */
static void
block_signal(sig)
    int sig;
{
#ifdef HAVE_SIGPROCMASK
    sigset_t sigset;
#endif

#ifdef HAVE_SIGPROCMASK
    sigemptyset(&sigset);
    sigaddset(&sigset, sig);
    sigprocmask(SIG_BLOCK, &sigset, NULL);
#else /* not HAVE_SIGPROCMASK */
#ifdef HAVE_SIGSETMASK
    signal_mask |= sigmask(sig);
    sigsetmask(signal_mask);
#endif /* HAVE_SIGSETMASK */
#endif /* not HAVE_SIGPROCMASK */
}


/*
 * Block signals of `sig' type (SIGHUP, SIGINT, ...).
 *
 * `sigprocmask' is used, if available.
 * Otherwise, `sigsetmask' is used if the system has it.
 * If neither `sigprocmask' nor `sigsetmask' is available, we give up
 * blocking a signal.
 */
static void
unblock_signal(sig)
    int sig;
{
#ifdef HAVE_SIGPROCMASK
    sigset_t sigset;
#endif

#ifdef HAVE_SIGPROCMASK
    sigemptyset(&sigset);
    sigaddset(&sigset, sig);
    sigprocmask (SIG_UNBLOCK, &sigset, NULL);
#else /* not HAVE_SIGPROCMASK */
#ifdef HAVE_SIGSETMASK
    signal_mask = (signal_mask | sigmask(sig)) ^ sigmask(sig);
    sigsetmask(signal_mask);
#endif /* HAVE_SIGSETMASK */
#endif /* not HAVE_SIGPROCMASK */
}


/*
 * Signal handler for SIGBUS and SIGSEGV. (for parent)
 */
static RETSIGTYPE
abort_parent(sig)
    int sig;
{
    alarm(0);
    if (server_mode == SERVER_MODE_STANDALONE) {
	syslog(LOG_CRIT, "the server process aborts, receives SIG%s",
	    signal_name(sig));
    } else {
	syslog(LOG_ERR, "the server process aborts, receives SIG%s",
	    signal_name(sig));
    }
    abort();

#ifndef RETSIGTYPE_VOID
    return 0;
#endif
}


/*
 * Signal handler for SIGBUS and SIGSEGV. (for child)
 */
static RETSIGTYPE
abort_child(sig)
    int sig;
{
    alarm(0);
    syslog(LOG_ERR, "the child server process aborts, receives SIG%s",
	signal_name(sig));
    abort();

#ifndef RETSIGTYPE_VOID
    return 0;
#endif
}


/*
 * Signal handler for SIGTERM, SIGINT SIGQUIT, and SIGHUP. (for parent)
 */
static RETSIGTYPE
terminate_parent(sig)
    int sig;
{
    alarm(0);
    if (sig == SIGALRM) {
	syslog(LOG_INFO, "the server process exits, timeout");
    } else if (server_mode == SERVER_MODE_STANDALONE) {
	syslog(LOG_CRIT, "the server process exits, receives SIG%s",
	    signal_name(sig));
    } else {
	syslog(LOG_ERR, "the server process exits, receives SIG%s",
	    signal_name(sig));
    }

    /*
     * Kill child processes.
     */
    set_signal_handler(SIGINT, SIG_IGN);
    if (server_mode == SERVER_MODE_STANDALONE) {
#ifdef HAVE_KILLPG
	killpg(process_group_id, SIGINT);
#else
	kill(-process_group_id, SIGINT);
#endif
    }

    /*
     * Shutdown a server port.
     */
    shutdown(listening_file, 2);

    /*
     * Remove a PID file.
     */
    if (server_mode == SERVER_MODE_STANDALONE)
	remove_pid_file(old_pid_filename);

    /*
     * Clear lists and buffers.
     */
    clear_book_registry();
    clear_permission(&permissions);
    clear_permission(&identifications);
    clear_line_buffer(&line_buffer);
    clear_ticket_stock(&connection_ticket_stock);

    exit(1);

#ifndef RETSIGTYPE_VOID
    return 0;
#endif
}


/*
 * Signal handler for SIGHUP, SIGINT, SIGQUIT and SIGTERM. (for child)
 */
static RETSIGTYPE
terminate_child(sig)
    int sig;
{
    alarm(0);
    syslog(LOG_ERR, "the child server process exits, receives SIG%s",
	signal_name(sig));
    shutdown(accepted_file, 2);

    /*
     * Clear lists and buffers.
     */
    clear_book_registry();
    clear_permission(&permissions);
    clear_permission(&identifications);
    clear_line_buffer(&line_buffer);
    clear_ticket_stock(&connection_ticket_stock);

    exit(0);

#ifndef RETSIGTYPE_VOID
    return 0;
#endif
}


/*
 * Signal handler for SIGCHLD.
 */
static RETSIGTYPE
wait_child(sig)
    int sig;
{
#ifdef HAVE_UNION_WAIT
    union wait status;
#else
    int status;
#endif

    syslog(LOG_INFO, "receives SIGCHLD");

#ifdef HAVE_WAITPID
    while (0 < waitpid(-1, &status, WNOHANG))
	;
#else /* not HAVE WAITPID */
    while (0 < wait3(&status, WNOHANG, NULL))
	;
#endif /* not HAVE WAITPID */

    /*
     * Set a signal handler again.
     */
    set_signal_handler(SIGCHLD, wait_child);

#ifndef RETSIGTYPE_VOID
    return 0;
#endif
}


/*
 * Signal handler for SIGHUP (for parent).
 */
static RETSIGTYPE
restart_parent(sig)
    int sig;
{
    alarm(0);
    syslog(LOG_ERR, "the server restart, receives SIGHUP");
    restart_trigger++;

#ifndef RETSIGTYPE_VOID
    return 0;
#endif
}
