/* To find Easter day and the moveable feasts, according to BCP.
   The Orthodox Church still uses the Julian calendar, so they are about
   2 weeks different for everything.  They and the Catholic Church 
   probably keep many more saints days. */

/* See README.easter.c for licence notice and bug warnings. */

/* 1999/10/26  hacked by cpbs@debian.org to generate a calendar(1)
               file directly
*/

#include <stdio.h>
#include <time.h>

void syntax(char *program);
time_t easter(int year);
int calc_offset(int year);
void easter_sunday(struct tm *date);
void display_date(char *day, struct tm *date);
void feasts(struct tm *date);

struct tab {int month; int day;};

struct tab full_moons[] = {
  /* month, day,    golden number */
	{0,  0},
	{4, 14}, 	/*  1 */
	{4,  3}, 	/*  2 */
	{3, 23}, 	/*  3 */
	{4, 11}, 	/*  4 */
	{3, 31}, 	/*  5 */
	{4, 18}, 	/*  6 */
	{4,  8},	/*  7 */
	{3, 28},	/*  8 */
	{4, 16},	/*  9 */
	{4,  5},	/* 10 */
	{3, 25},	/* 11 */
	{4, 13},	/* 12 */
	{4,  2},	/* 13 */
	{3, 22},	/* 14 */
	{4, 10},	/* 15 */
	{3, 30},	/* 16 */
	{4, 17},	/* 17 */
	{4,  7},	/* 18 */
	{3, 27}		/* 19 */
	};

char *month[] = {
	"January",
	"February",
	"March",
	"April",
	"May",
	"June",
	"July",
	"August",
	"September",
	"October",
	"November",
	"December"
	};

int main(int argc, char **argv) {
	int year;
	struct tm *dateptr;
	time_t t;

	/* for which year? */
	switch (argc) {
	    case 1: 
		t = time(0);
		dateptr = localtime(&t);
		year = 1900 + dateptr->tm_year;
		break;
	    case 2:
		if (!sscanf(argv[1], "%d", &year)) {
			syntax(argv[0]);
			return(1);
		}
		if (year < 29) {
			fprintf(stderr,"There can be no date for Easter before the resurrection of Jesus!\n");
                        /* and the formula wasn't settled until the 4th century */
			return(3);
		}
		if (year < 1753) {
			/* I think it was 1753 in Britain */
			fprintf(stderr,"Warning: dates before 1753 were not according to the Gregorian calendar.\n");
			return(3);
		}
		break;
	    default:
		syntax(argv[0]);
		return(1);
		break;
	}

	t = easter(year);
	dateptr = localtime(&t);
	printf("/* Christian calendar for year %d (excluding fixed events)\n  ", year);
	easter_sunday(dateptr);
	printf("*/\n\n", year);
	feasts(dateptr);
	return(0);
}

/* Display required syntax */
void syntax(char *program) {
	fprintf(stderr, "%s: syntax error\n%s [year]\n", program, program);
}

/* Calculate Easter Day */
time_t easter(int year) {
	int golden_number;
	char sunday_letter;
	int i, j;
	struct tm date;
	time_t second;

	date.tm_sec = 0;
	date.tm_min = 0;
	date.tm_hour = 0;
	date.tm_isdst = -1;

	golden_number = (year + 1) % 19;
	if (!golden_number) {
		golden_number = 19;
	}
	
	j = calc_offset(year);
	i = (year + year/4 + j) % 7;
	sunday_letter = (!i ? 'A' : ('H' - i));

	date.tm_wday = -1;
	date.tm_yday = -1;
	date.tm_mday = 19;
	date.tm_mon = 2;
	date.tm_year = (year - 1900);
	second = mktime(&date);	/* 19th March */

	if (date.tm_wday == -1) {
		fprintf(stderr,"Year %d is outside the Unix date range\n", year);
		exit(2);
	}

	i = date.tm_yday;	/* 3 days before the first possible Paschal
				   full moon */

	date.tm_wday = -1;
	date.tm_yday = -1;
	date.tm_mday = full_moons[golden_number].day;
	date.tm_mon = (full_moons[golden_number].month - 1);
	second = mktime(&date);	/* Paschal full moon */

	if (date.tm_wday == -1) {
		fprintf(stderr,"Year %d is outside the Unix date range\n", year);
		exit(2);
	}

	for (j = i; j < date.tm_yday; j += 7);

	j += (sunday_letter - 'A');

	j = (j <= i) ? j + 7 : j;

	date.tm_wday = -1;
	date.tm_yday = -1;
	date.tm_mday = j;
	date.tm_mon = 0;
	
	second = mktime(&date);	/* Easter Day */
	if (date.tm_wday != 0) {
		fprintf(stderr, "Calculation failed!\n");
	}
	return (second);
}

/* The offset for the golden number depends on the century */
int calc_offset(int year) {
	int i = 0;

	i = (year / 100);
	i -= (year / 400);
	i = (i + 1) % 7;
	if (i) {
		i = 7 - i;
	}
	return (i);
}

/* Display the date of Easter Day */
void easter_sunday(struct tm *date) {
	display_date("Easter Day", date);
}

void display_date(char *day, struct tm *date) {
	printf("%3.3s %.2d\t%s\n", month[date->tm_mon], date->tm_mday, /* 1900+date->tm_year, */ day);
}

/* Display the moveable feasts */
void feasts(struct tm *date) {
	struct tm d;
	time_t t;
	int j;

	d = *date;
	printf("/* Moveable feasts: */\n");

	d.tm_mday -= (9 * 7);		/* 9 weeks before Easter */
	t = mktime(&d);
	display_date("Septuagesima", &d);

	d.tm_mday += 7;			/* 8 weeks before Easter */
	t = mktime(&d);
	display_date("Sexagesima", &d);

	d.tm_mday += 7;			/* 7 weeks before Easter */
	t = mktime(&d);
	display_date("Quinquagesima", &d);

	d = *date;
	d.tm_mday -= 47;		/* Shrove Tuesday */
	t = mktime(&d);
	display_date("Shrove Tuesday", &d);

	d = *date;
	d.tm_mday -= 46;		/* Ash Wednesday */
	t = mktime(&d);
	display_date("Ash Wednesday", &d);

	d = *date;
	d.tm_mday -= (6 * 7);		/* 6 weeks before Easter */
	t = mktime(&d);
	display_date("Quadragesima (1st Sunday of Lent)", &d);

	/* Ember days -
	   Wednesday, Friday and Saturday after 1st Sunday of Lent */
	d.tm_mday += 3;			/* Wednesday */
	t = mktime(&d);
	display_date("Ember day", &d);
	d.tm_mday += 2;			/* Friday */
	t = mktime(&d);
	display_date("Ember day", &d);
	d.tm_mday++;			/* Saturday */
	t = mktime(&d);
	display_date("Ember day", &d);

	d = *date;
	d.tm_mday -= 7;			/* 1 week before Easter */
	t = mktime(&d);
	display_date("Palm Sunday", &d);
	d.tm_mday += 4;			/* Maundy Thursday */
	t = mktime(&d);
	display_date("Maundy Thursday", &d);
	d.tm_mday++;			/* Good Friday */
	t = mktime(&d);
	display_date("Good Friday", &d);

	easter_sunday(date);

	d = *date;
	d.tm_mday++;			/* Easter Monday */
	t = mktime(&d);
	display_date("Easter Monday", &d);

	d = *date;
	d.tm_mday += (5 * 7);			/* Rogation Sunday */
	t = mktime(&d);
	display_date("Rogation Sunday", &d);

	/* Monday to Wednesday before Ascension are Rogation days */
	d = *date;
	d.tm_mday += 36;		/* Monday */
	t = mktime(&d);
	display_date("Rogation Day", &d);
	d.tm_mday++;			/* Tuesday */
	t = mktime(&d);
	display_date("Rogation Day", &d);
	d.tm_mday++;			/* Wednesday */
	t = mktime(&d);
	display_date("Rogation Day", &d);
	d.tm_mday++;			/* Ascension Day */
	t = mktime(&d);
	display_date("Ascension Day", &d);

	d = *date;
	d.tm_mday += (7 * 7);		/* Pentecost */
	t = mktime(&d);
	display_date("Pentecost", &d);

	/* Ember days -
	   Wednesday, Friday and Saturday after Pentecost */
	d.tm_mday += 2;			/* Wednesday */
	t = mktime(&d);
	display_date("Ember day", &d);
	d.tm_mday += 2;			/* Friday */
	t = mktime(&d);
	display_date("Ember day", &d);
	d.tm_mday++;			/* Saturday */
	t = mktime(&d);
	display_date("Ember day", &d);

	d = *date;
	d.tm_mday += (8 * 7);		/* Trinity Sunday */
	t = mktime(&d);
	display_date("Trinity Sunday", &d);
	d.tm_mday += 4;			/* Corpus Christi */
	t = mktime(&d);
	display_date("Corpus Christi", &d);

	/* Ember days -
	   Wednesday, Friday and Saturday after 14th September */
	d.tm_mday = 14;
	d.tm_mon = 8;			/* 14th September */
	t = mktime(&d);
	d.tm_mday += 3 - d.tm_wday + (d.tm_wday > 2 ? 7 : 0);
					/* Wednesday */
	t = mktime(&d);
	display_date("Ember day", &d);
	d.tm_mday += 2;			/* Friday */
	t = mktime(&d);
	display_date("Ember day", &d);
	d.tm_mday++;			/* Saturday */
	t = mktime(&d);
	display_date("Ember day", &d);

	d.tm_mday = 30;
	d.tm_mon = 10;			/* St Andrew's day */
	t = mktime(&d);
	j = ((d.tm_wday > 3) ?
		d.tm_mday + 7 - d.tm_wday :
		d.tm_mday - d.tm_wday);	/* nearest Sunday is Advent Sunday */
	d.tm_mday = j;
	t = mktime(&d);
	display_date("Advent Sunday (First Sunday of Advent)", &d);

	/* Ember days -
	   Wednesday, Friday and Saturday after 13th December */
	d.tm_mday = 13;
	d.tm_mon = 11;			/* 13th December */
	t = mktime(&d);
	d.tm_mday += 3 - d.tm_wday + (d.tm_wday > 2 ? 7 : 0);
					/* Wednesday */
	t = mktime(&d);
	display_date("Ember day", &d);
	d.tm_mday += 2;			/* Friday */
	t = mktime(&d);
	display_date("Ember day", &d);
	d.tm_mday++;			/* Saturday */
	t = mktime(&d);
	display_date("Ember day", &d);

}
