/******************************************************************************\
 gnofin/record.c   $Revision: 1.13 $
 Copyright (C) 1999 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <glib.h>
#include "gnofin.h"
#include "account.h"
#include "record.h"

static gchar * record_type_map[] =
{
  "ATM",
  "CC" ,
  "CHK",
  "DEP",
  "EFT",
  "XFR"
};


FinRecordInfo *
fin_record_info_new (const gchar * string)
{
  FinRecordInfo * info;

  fin_trace("");

  g_return_val_if_fail(string != NULL, NULL);
  
  info = g_new(FinRecordInfo, 1);
  info->string = g_strdup(string);
  info->ref_count = 0;

  return info;
}

void
fin_record_info_ref (FinRecordInfo * info)
{
  fin_trace("");

  g_return_if_fail(info);
  
  info->ref_count++;
}

void
fin_record_info_unref (FinRecordInfo * info)
{
  fin_trace("");

  /* DONT DELETE.. keep this around for future records

  if (info)
  {
    if (--info->ref_count == 0)
    {
      g_list_remove(list,info);
      g_free(info->string);
      g_free(info);
    }
  }
  */
}

gint
fin_record_info_sort_fcn (const FinRecordInfo * info1,
			  const FinRecordInfo * info2)
{
  fin_trace("");

  g_return_val_if_fail(info1 != NULL, 0);
  g_return_val_if_fail(info2 != NULL, 0);

  return strcmp(info1->string, info2->string);
}


/* external interface */

FinRecord *
fin_record_new (GList * info_cache)
{
  FinRecord * record;

  fin_trace("");

  record = g_new0(FinRecord, 1);
  record->ref_count = 1;

  /* allow NULL info cache */
  record->info = info_cache;

  return record;
}

void
fin_record_ref (FinRecord * record)
{
  fin_trace("");

  g_return_if_fail(record);

  ++record->ref_count;
}

void
fin_record_unref (FinRecord * record)
{
  fin_trace("");

  g_return_if_fail(record);

  if (--record->ref_count == 0)
  {
    if (record->info)
      fin_record_info_unref((FinRecordInfo *) record->info->data);

    if (record->type == FIN_RECORD_TYPE_XFR && record->have_linked_XFR)
    {
      /* just free the transfer_link structure.. do not try to resolve
         the breakage on the other end.. that will get handled elsewhere */
      g_free(record->type_d.transfer_link);
    }

    g_free(record);
  }
}

void
fin_record_set_info_string (FinRecord   * record,
			    const gchar * string)
{
  FinRecordInfo * info = NULL;
  GList * it, * first;

  fin_trace("");

  g_return_if_fail(record);
  g_return_if_fail(string);

  /* search backward, remember the first element in the list */
  first = record->info;
  for (it=record->info; it; it=it->prev)
  {
    if (it->prev)
      first = it->prev;
    info = LIST_GET(FinRecordInfo, it);
    g_assert(info != NULL);
    if (strcmp(info->string, string) == 0)
      break;
  }

  /* search forward, if no match yet */
  if (it == NULL)
  {
    for (it=record->info; it; it=it->next)
    {
      info = LIST_GET(FinRecordInfo, it);
      g_assert(info != NULL);
      if (strcmp(info->string, string) == 0)
	break;
    }
  }

  /* make a new record info entry, if still no match
   * append to info_cache */
  if (it == NULL)
  {
    info = fin_record_info_new(string);
    it = g_list_insert_sorted(first, info, (GCompareFunc) fin_record_info_sort_fcn);
    
    /* now it points to the top of the list, so we need to find the new entry */
    g_assert(it->prev == NULL);
    while (it)
    {
      info = LIST_GET(FinRecordInfo,it);
      if (strcmp(info->string,string) == 0)
	break;
      it = it->next;
    }
    g_assert(it != NULL);
  }
  else
    fin_record_info_ref(LIST_GET(FinRecordInfo, it));

  /* set info pointer in record */
  record->info = it;
  record->have_info_string = 1;
}

const gchar *
fin_record_get_info_string (const FinRecord * record)
{
  fin_trace("");

  g_return_val_if_fail(record != NULL, NULL);

  if (record->have_info_string)
    return LIST_GET(FinRecordInfo, record->info)->string;
  else
    return NULL;
}

GList *
fin_record_stringize_types (gint multi_accounts)
{
  GList * text = NULL;

  fin_trace("");

  /* if multi_accounts, then we include XFR...
     otherwise it should not be included */
  {
    int i;

    /* build list backwards... prepend is faster than append */
    for (i=FIN_RECORD_TYPE_NUM - (multi_accounts == 0) - 1; i>=0; --i)
      text = g_list_prepend(text, g_strdup(record_type_map[i]));
  }

  return text;
}

FinRecordType
fin_record_parse_type (const gchar * string)
{
  int i;

  for (i=0; i<FIN_RECORD_TYPE_NUM; ++i)
  {
    if (strcmp(string,record_type_map[i]) == 0)
      return (FinRecordType) i;
  }

  return FIN_RECORD_TYPE_NUM;
}


GList *
fin_record_info_stringize (GList * info)
{
  GList * it, * strings = NULL;

  fin_trace("");

  it = g_list_last(info);

  for (; it; it=it->prev)
    strings = g_list_prepend(strings, LIST_GET(FinRecordInfo,it)->string);

  return strings;
}

const gchar *
fin_record_stringize_type (FinRecordType type)
{
  fin_trace("");

  return (type < FIN_RECORD_TYPE_NUM) ? record_type_map[type] : NULL;
}

gchar **
fin_stringized_record_new (const FinRecord      * record,
			   const FinRecordField * types,
			   gint		          num_fields)
{
  gchar ** text;
  int k;

  /* NOTE: this is indexed by FinRecordField */
  static const int field_lens[] =
  {
    FIN_RECORD_FIELD_MAXLEN_DATE,
    FIN_RECORD_FIELD_MAXLEN_TYPE,
    FIN_RECORD_FIELD_MAXLEN_TYPE_DATA,
    FIN_RECORD_FIELD_MAXLEN_TYPE_FULL,
    FIN_RECORD_FIELD_MAXLEN_INFO,
    FIN_RECORD_FIELD_MAXLEN_STATUS,
    FIN_RECORD_FIELD_MAXLEN_AMOUNT,
    FIN_RECORD_FIELD_MAXLEN_BALANCE,
  };

  fin_trace("");

  g_return_val_if_fail(record, NULL);
  g_return_val_if_fail(types, NULL);
  g_return_val_if_fail(num_fields > 0, NULL);

  /* allocate memory */
  {
    gchar * buf;
    int len = 0;

    /* allocate array of pointers */
    text = g_new(gchar *, num_fields);

    /* compute total memory required */
    for (k=0; k<num_fields; ++k)
      len += field_lens[types[k]];

    /* allocate contiguous memory chunk */
    buf = g_new0(gchar, len);

    /* assign pointers */
    len = 0;
    for (k=0; k<num_fields; ++k)
    {
      text[k] = buf + len;
      len += field_lens[types[k]];
    }
  }

  for (k=0; k<num_fields; ++k)
  {
    int field_len = field_lens[types[k]];

    switch (types[k])
    {
    case FIN_RECORD_FIELD_DATE:
      fin_date_stringize(text[k], field_len, (GDate *) &record->date);
      break;

    case FIN_RECORD_FIELD_TYPE:
      strncpy(text[k], fin_record_stringize_type(record->type), field_len);
      break;

    case FIN_RECORD_FIELD_TYPE_DATA:
      switch (record->type)
      {
      case FIN_RECORD_TYPE_CHK:
	snprintf(text[k], field_len, "%d", record->type_d.check_no);
	break;
      case FIN_RECORD_TYPE_XFR:
	g_assert(record->have_linked_XFR);
	g_assert(record->type_d.transfer_link);
	g_assert(record->type_d.transfer_link->account);
	g_assert(record->type_d.transfer_link->account->name);
	strncpy(text[k], record->type_d.transfer_link->account->name, field_len);
	break;
      default:
      }
      break;

    case FIN_RECORD_FIELD_TYPE_FULL:
      {
        const gchar * type_name = fin_record_stringize_type(record->type);

        switch (record->type)
	{
	case FIN_RECORD_TYPE_CHK:
	  snprintf(text[k], field_len, "%s : %d", type_name, record->type_d.check_no);
	  break;
	case FIN_RECORD_TYPE_XFR:
	  g_assert(record->have_linked_XFR);
	  g_assert(record->type_d.transfer_link);
	  g_assert(record->type_d.transfer_link->account);
	  g_assert(record->type_d.transfer_link->account->name);
	  snprintf(text[k], field_len, "%s : %s", type_name, 
	           record->type_d.transfer_link->account->name);
	  break;
	default:
          strncpy(text[k], type_name, field_len);
	}
      }
      break;

    case FIN_RECORD_FIELD_INFO:
      if (record->have_info_string)
	strncpy(text[k], fin_record_get_info_string(record), field_len);
      break;

    case FIN_RECORD_FIELD_STATUS:
      if (record->cleared)
	text[k][0] = 'c';
      break;

    case FIN_RECORD_FIELD_AMOUNT:
      if (record->have_amount)
	fin_money_stringize(text[k], field_len, record->amount);
      break;

    case FIN_RECORD_FIELD_BALANCE:
      if (record->have_amount)
	fin_money_stringize(text[k], field_len, record->balance);
      break;

    default:
      g_assert_not_reached();
    }
  }
  return text;
}

void
fin_stringized_record_free (gchar ** text)
{
  fin_trace("");

  g_return_if_fail(text);
  g_return_if_fail(text[0]);

  g_free(text[0]);
  g_free(text);
}

int
fin_record_parse_fields (FinRecord            * record,
			 const FinRecordField * types,
			 gint	                num_fields,
			 gchar               ** text)
{
  int k = 0;

  fin_trace("");

  g_return_val_if_fail(record, -1);
  g_return_val_if_fail(text, -1);

  for (k=0; k<num_fields; ++k)
  {
    switch (types[k])
    {
    case FIN_RECORD_FIELD_DATE:
      {
	g_date_clear(&record->date, 1);
	g_date_set_parse(&record->date, text[k]);

	if (!g_date_valid(&record->date))
	{
	  fin_trace("error parsing date field!");
	  g_date_clear(&record->date, 1);
	  return k+1;
	}

	fin_trace("record->date = %d/%d/%d", 
		  g_date_day(&record->date),
		  g_date_month(&record->date),
		  g_date_year(&record->date));
      }
      break;
    case FIN_RECORD_FIELD_TYPE:
      {
        record->type = fin_record_parse_type(text[k]);
	fin_trace("record->type = %d", record->type);

	if (record->type == FIN_RECORD_TYPE_NUM)
	{
	  fin_trace("error parsing type field!");
	  return k+1;
	}
      }
      break;
    case FIN_RECORD_FIELD_TYPE_DATA:
      {
        switch (record->type)
	{
	case FIN_RECORD_TYPE_CHK:
	  if (sscanf(text[k], "%d", &record->type_d.check_no) == -1)
	  {
	    fin_trace("error parsing type_data field!");
	    return k+1;
	  }
	  break;
	case FIN_RECORD_TYPE_XFR:
	  /* just mark this record as unlinked */
	  record->have_linked_XFR = 0;
	  break;
	default:
	  fin_trace("ignoring type_data field");
	}
      }
      break;
    case FIN_RECORD_FIELD_INFO:
      {
	fin_record_set_info_string(record, text[k]);
	fin_trace("record->info->string = %s", 
		  LIST_GET(FinRecordInfo, record->info)->string);
      }
      break;
    case FIN_RECORD_FIELD_STATUS:
      {
        record->cleared = (text[k][0] == 'c');
	fin_trace("record->cleared = %s", record->cleared ? "true" : "false" );
      }
      break;
    case FIN_RECORD_FIELD_AMOUNT:
      {
	/* allow empty string */
	if (text[k][0] == '\0')
	  record->amount = 0;
	else if (!fin_money_parse(text[k], &record->amount))
	{
	  fin_trace("error parsing amount!");
	  return k+1;
	}
	fin_trace("record->amount = %ld", record->amount);
	record->have_amount = 1;
      }
      break;
    case FIN_RECORD_FIELD_TYPE_FULL:
    case FIN_RECORD_FIELD_BALANCE:
      fin_trace("invalid field !!\n");
      return k+1;
    default:
      g_assert_not_reached();
    }
  }
  return 0;
}

void
fin_record_info_dump (FinRecordInfo * info,
		      FILE	    * stream)
{
  fprintf(stream,"    record_info at %p\n", info);
  fprintf(stream,"      ref_count\t%d\n", info->ref_count);
  fprintf(stream,"      string   \t\"%s\"\n", info->string);
}

void
fin_record_dump (FinRecord * record,
		 FILE      * stream)
{
  fprintf(stream,"    record at %p\n", record);
  fprintf(stream,"      ref_count      \t%d\n", record->ref_count);
  fprintf(stream,"      date.day       \t%d\n", record->date.day);
  fprintf(stream,"      date.month     \t%d\n", record->date.month);
  fprintf(stream,"      date.year      \t%d\n", record->date.year);
  fprintf(stream,"      type           \t%d\n", record->type);
  fprintf(stream,"      type_d.raw[0]  \t%x\n", record->type_d.raw[0]);
  fprintf(stream,"      type_d.raw[1]  \t%x\n", record->type_d.raw[1]);

  switch (record->type)
  {
  case FIN_RECORD_TYPE_CHK:
    fprintf(stream,"      type_d.check_no\t%d\n", record->type_d.check_no);
    break;
  case FIN_RECORD_TYPE_XFR:
    if (record->have_linked_XFR)
      fprintf(stream,"      type_d.XFR_link\t%p\t(%s)\n", record->type_d.transfer_link, (record->type_d.transfer_link && record->type_d.transfer_link->account ? record->type_d.transfer_link->account->name : "(null)") );
    else
      fprintf(stream,"      type_d.XFR_link\t<not linked>\n");
    break;
  default:
  }
  fprintf(stream,"      info      \t%p\t(%p) \"%s\"\n", record->info, (record->info ? record->info->data : NULL), (record->info ? (record->info->data ? LIST_GET(FinRecordInfo, record->info)->string : NULL) : NULL));
  fprintf(stream,"      cleared   \t%s\n", record->cleared ? "true" : "false" );
  fprintf(stream,"      amount    \t%ld\n", record->amount);
  fprintf(stream,"      balance   \t%ld\n", record->balance);
  fprintf(stream,"      have_info_string\t%s\n", record->have_info_string ? "true" : "false");
  fprintf(stream,"      have_amount     \t%s\n", record->have_amount ? "true" : "false");
}

gint
fin_record_sort_fcn (const FinRecord * a,
		     const FinRecord * b)
{
  gint c;

  fin_trace("");

  c = g_date_compare((GDate *)&a->date, (GDate *)&b->date);

  if (c == 0)
  {
    /* dates match, check amount */

    if (a->amount * b->amount < 0)
      c = (a->amount < 0) ? +1 : -1;
    else
    {
      /* both same sign, sort by info string */
      g_assert(a->have_info_string);
      g_assert(b->have_info_string);
      c = strcmp(fin_record_get_info_string(a), fin_record_get_info_string(b));

      if (c == 0)
      {
        /* both have same info string, sort by check number if appropriate */
	if (a->type == FIN_RECORD_TYPE_CHK && b->type == FIN_RECORD_TYPE_CHK)
	  c = (a->type_d.check_no < b->type_d.check_no ? -1 : +1);
        }
    }
  }

  return c;
}

FinRecord *
fin_record_dup (FinRecord * record)
{
  FinRecord * dup;

  g_return_val_if_fail(record, NULL);
  
  dup = fin_record_new(g_list_first(record->info));

  memcpy(dup, record, sizeof(*record));

  dup->ref_count = 1;
  LIST_GET(FinRecordInfo, dup->info)->ref_count++;
  
  return dup;
}

void
fin_record_swap (FinRecord * a,
		 FinRecord * b)
{
  FinRecord temp;

  memcpy(&temp, a, sizeof(*a));

  /* make sure record's maintain individual reference count */
  temp.ref_count = b->ref_count;
  b->ref_count = a->ref_count;

  memcpy(a, b, sizeof(*a));
  memcpy(b, &temp, sizeof(*a));
}

void
fin_record_sync_link (FinRecord * dest, const FinRecord * src)
{
  fin_trace("");

  g_return_if_fail(dest->type != FIN_RECORD_TYPE_XFR);
  g_return_if_fail(dest->type != src->type);

  fin_record_info_ref(LIST_GET(FinRecordInfo, src->info));
  fin_record_info_unref((FinRecordInfo *) dest->info->data);  /* ok if data is NULL */

  dest->date = src->date;
  dest->info = src->info;
  dest->amount = -1 * src->amount;
  dest->have_amount = src->have_amount;
}
