/* GnomeCard - a graphical contact manager.
 *
 * canvas.c: This file is part of GnomeCard.
 * 
 * Copyright (C) 1999 The Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <gnome.h>

#include "card.h"
#include "dialog.h"
#include "deladdrlist.h"
#include "phonelist.h"
#include "gnomecard.h"
#include "canvas.h"
#include "dnd.h"
#include "my.h"

#define CARDNAME_FONTSET "-adobe-helvetica-medium-r-normal-*-18-*-*-*-p-*-*-*,-cronyx-helvetica-medium-r-normal-*-20-*-*-*-p-*-koi8-r,-*-*-medium-r-normal-*-18-*-*-*-*-*-ksc5601.1987-0,*"
#define TITLE_FONTSET "-adobe-helvetica-medium-r-*-*-14-*-*-*-p-*-*-*,-cronyx-helvetica-medium-r-normal-*-17-*-*-*-p-*-koi8-r,-*-*-medium-r-no-*-*-14-*-*-*-*-*-ksc5601.1987-0,*"
#define COMMENT_FONTSET "-adobe-helvetica-medium-r-*-*-10-*-*-*-p-*-*-*,-cronyx-helvetica-medium-r-normal-*-11-*-*-*-p-*-koi8-r,-*-*-medium-r-*-*-10-*-*-*-*-*-ksc5601.1987-0,*"
#define CANVAS_FONTSET "-adobe-helvetica-medium-r-*-*-12-*-*-*-p-*-*-*,-cronyx-helvetica-medium-r-normal-*-14-*-*-*-p-*-koi8-r,-*-*-medium-r-*-*-12-*-*-*-*-*-ksc5601.1987-0,*"
#define ADDR_FONTSET "-adobe-helvetica-medium-r-*-*-10-*-*-*-p-*-*-*,-cronyx-helvetica-medium-r-normal-*-11-*-*-*-p-*-koi8-r,-*-*-medium-r-*-*-10-*-*-*-*-*-ksc5601.1987-0,*"
#define CANVAS_WIDTH 225
#define CANVAS_HEIGHT 350
#define LIST_SPACING 15.0
#define ADDR_SPACING 11.0

#define HEADER_BOX_COLOR  "khaki"
#define SUBHEADER_BOX_COLOR ""
#define LABEL_BOX_COLOR "dark khaki"

#define HEADER_TEXT_COLOR "black"
#define LABEL_TEXT_COLOR   "black"
#define BODY_TEXT_COLOR    "black"

typedef struct {
    GnomeCanvasItem *street1;
    GnomeCanvasItem *street2;
    GnomeCanvasItem *state;
    GnomeCanvasItem *country;
} CanvasAddressItem;

GtkWidget *gnomecard_canvas;
static GnomeCanvasItem *cardname, *fullname, *workphone, *homephone;
static GnomeCanvasItem *cardname_box, *phone_box, *deladdr_box, *categories_box, *memo_box;
static GnomeCanvasItem *email, *url, *org, *title, *categories, *memos;
static CanvasAddressItem homeaddr, workaddr;
static GnomeCanvasGroup *root;

static void gnomecard_canvas_text_item_set(GnomeCanvasItem *p, gchar *text);

GtkWidget *
gnomecard_canvas_new(void)
{
	GtkWidget *canvas;
	gdouble ypos, x1, x2, y1, y2;
	GtkStyle *style;
	gint i;
        GdkFont *cardname_font, *title_font, *canvas_font, *addr_font, *comment_font; 

	canvas = gnomecard_canvas = gnome_canvas_new();
	configure_source_card_dnd (canvas);
	gtk_widget_pop_visual();
	gtk_widget_pop_colormap();

	style = gtk_style_copy(gtk_widget_get_style(canvas));
	for (i=0; i<5; i++)
	    style->bg[i].red = style->bg[i].green = style->bg[i].blue = 0xffff;

	gtk_widget_set_style(GTK_WIDGET(canvas), style);

	gtk_widget_set_usize(GTK_WIDGET(canvas), 
			     CANVAS_WIDTH, CANVAS_HEIGHT);

	root = GNOME_CANVAS_GROUP (gnome_canvas_root(GNOME_CANVAS(canvas)));

        cardname_font = gdk_fontset_load (CARDNAME_FONTSET);
	title_font = gdk_fontset_load (TITLE_FONTSET);
	canvas_font = gdk_fontset_load (CANVAS_FONTSET);
	addr_font = gdk_fontset_load (ADDR_FONTSET);
	comment_font = gdk_fontset_load (COMMENT_FONTSET);

	/* label at top of canvas, contains current cardname */
	cardname_box = gnome_canvas_item_new (root, gnome_canvas_rect_get_type (),
			       "x1", 5.0,
			       "y1", 5.0,
			       "x2", (double) (CANVAS_WIDTH - 6.0),
			       "y2", (double) 35,
			       "fill_color", HEADER_BOX_COLOR,
			       "outline_color", HEADER_BOX_COLOR,
			       "width_pixels", 0,
			       NULL);

	cardname = gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", "",
				      "x", 6.0, /*CANVAS_WIDTH / 2.0,*/
				      "y", 20.0,
				      "fontset", CARDNAME_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", HEADER_TEXT_COLOR,
				      NULL);

	/* Full name for card */
	ypos = 45.0;
	fullname = gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", "",
				      "x", 5.0,
				      "y", ypos,
				      "fontset", CANVAS_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", BODY_TEXT_COLOR,
				      NULL);

	/* Organization and title */
	ypos += LIST_SPACING;
	title = gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", "",
				      "x", 5.0,
				      "y", ypos,
				      "fontset", CANVAS_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", BODY_TEXT_COLOR,
				      NULL);

	ypos += LIST_SPACING;
	org = gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", "",
				      "x", 5.0,
				      "y", ypos,
				      "fontset", CANVAS_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", BODY_TEXT_COLOR,
				      NULL);

	/* web site and email */
	ypos += LIST_SPACING;
	email = gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", "",
				      "x", 5.0,
				      "y", ypos,
				      "fontset", CANVAS_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", BODY_TEXT_COLOR,
				      NULL);
	ypos += LIST_SPACING;
	url = gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", "",
				      "x", 5.0,
				      "y", ypos,
				      "fontset", CANVAS_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", BODY_TEXT_COLOR,
				      NULL);

	/* phone #'s */
	ypos += LIST_SPACING;
	phone_box = gnome_canvas_item_new (root, gnome_canvas_rect_get_type (),
			       "x1", 5.0,
			       "y1", ypos,
			       "x2", (double) (CANVAS_WIDTH - 6.0),
			       "y2", (double) ypos+20,
			       "fill_color", HEADER_BOX_COLOR,
			       "outline_color", HEADER_BOX_COLOR,
			       "width_pixels", 0,
			       NULL);

	gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", _("Phone Numbers"),
				      "x", 10.0,
				      "y", ypos+10,
				      "fontset",  TITLE_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", HEADER_TEXT_COLOR,
				      NULL);

	ypos += 30;
	gnome_canvas_item_new (root, gnome_canvas_rect_get_type (),
			       "x1", 5.0,
			       "y1", ypos-6.0,
			       "x2", (double) 55.0,
			       "y2", (double) ypos+6,
			       "fill_color", LABEL_BOX_COLOR,
			       "outline_color", LABEL_BOX_COLOR,
			       "width_pixels", 0,
			       NULL);
	gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", _("Home:"),
				      "x", 50.0,
				      "y", ypos,
				      "fontset",  COMMENT_FONTSET,
				      "anchor", GTK_ANCHOR_EAST,
				      "fill_color", LABEL_TEXT_COLOR,
				      NULL);

	homephone = gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", "",
				      "x", 60.0,
				      "y", ypos,
				      "fontset", CANVAS_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", BODY_TEXT_COLOR,
				      NULL);


	ypos += 15;
	gnome_canvas_item_new (root, gnome_canvas_rect_get_type (),
			       "x1", 5.0,
			       "y1", ypos-6.0,
			       "x2", (double) 55.0,
			       "y2", (double) ypos+6,
			       "fill_color", LABEL_BOX_COLOR,
			       "outline_color", LABEL_BOX_COLOR,
			       "width_pixels", 0,
			       NULL);
	gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", _("Work:"),
				      "x", 50.0,
				      "y", ypos,
				      "fontset",  COMMENT_FONTSET,
				      "anchor", GTK_ANCHOR_EAST,
				      "fill_color", LABEL_TEXT_COLOR,
				      NULL);

	workphone = gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", "",
				      "x", 60.0,
				      "y", ypos,
				      "fontset", CANVAS_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", BODY_TEXT_COLOR,
				      NULL);

	/* addresses */
	ypos += LIST_SPACING;
	deladdr_box = gnome_canvas_item_new (root, gnome_canvas_rect_get_type (),
			       "x1", 5.0,
			       "y1", ypos,
			       "x2", (double) (CANVAS_WIDTH - 6.0),
			       "y2", (double) ypos+20,
			       "fill_color", HEADER_BOX_COLOR,
			       "outline_color", HEADER_BOX_COLOR,
			       "width_pixels", 0,
			       NULL);

	gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", _("Addresses"),
				      "x", 10.0,
				      "y", ypos+10,
				      "fontset",  TITLE_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", HEADER_TEXT_COLOR,
				      NULL);

	/* home address */
	ypos += 30;
	gnome_canvas_item_new (root, gnome_canvas_rect_get_type (),
			       "x1", 10.0,
			       "y1", ypos,
			       "x2", (double) 150.0,
			       "y2", (double) ypos+10,
			       "fill_color", LABEL_BOX_COLOR,
			       "outline_color", LABEL_BOX_COLOR,
			       "width_pixels", 0,
			       NULL);

	gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", _("Home"),
				      "x", 15.0,
				      "y", ypos+5,
				      "fontset",  COMMENT_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", LABEL_TEXT_COLOR,
				      NULL);
	
	ypos += 20;
	homeaddr.street1 = gnome_canvas_item_new (root, 
						  gnome_canvas_text_get_type(),
						  "text", "",
						  "x", 10.0,
						  "y", ypos,
						  "fontset", ADDR_FONTSET,
						  "anchor", GTK_ANCHOR_WEST,
						  "fill_color",BODY_TEXT_COLOR,
						  NULL);

	ypos += ADDR_SPACING;
	homeaddr.street2 = gnome_canvas_item_new (root, 
						  gnome_canvas_text_get_type(),
						  "text", "",
						  "x", 10.0,
						  "y", ypos,
						  "fontset", ADDR_FONTSET,
						  "anchor", GTK_ANCHOR_WEST,
						  "fill_color",BODY_TEXT_COLOR,
						  NULL);

	ypos += ADDR_SPACING;
	homeaddr.state = gnome_canvas_item_new (root, 
						  gnome_canvas_text_get_type(),
						  "text", "",
						  "x", 10.0,
						  "y", ypos,
						  "fontset", ADDR_FONTSET,
						  "anchor", GTK_ANCHOR_WEST,
						  "fill_color",BODY_TEXT_COLOR,
						  NULL);

	ypos += ADDR_SPACING;
	homeaddr.country = gnome_canvas_item_new (root, 
						  gnome_canvas_text_get_type(),
						  "text", "",
						  "x", 10.0,
						  "y", ypos,
						  "fontset", ADDR_FONTSET,
						  "anchor", GTK_ANCHOR_WEST,
						  "fill_color",BODY_TEXT_COLOR,
						  NULL);



	/* work address */
	ypos += 10;
	gnome_canvas_item_new (root, gnome_canvas_rect_get_type (),
			       "x1", 10.0,
			       "y1", ypos,
			       "x2", (double) 150.0,
			       "y2", (double) ypos+10,
			       "fill_color", LABEL_BOX_COLOR,
			       "outline_color", LABEL_BOX_COLOR,
			       "width_pixels", 0,
			       NULL);

	gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", _("Work"),
				      "x", 15.0,
				      "y", ypos+5,
				      "fontset", COMMENT_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", LABEL_TEXT_COLOR,
				      NULL);
	
	ypos += 20;
	workaddr.street1 = gnome_canvas_item_new (root, 
						  gnome_canvas_text_get_type(),
						  "text", "",
						  "x", 10.0,
						  "y", ypos,
						  "fontset", ADDR_FONTSET,
						  "anchor", GTK_ANCHOR_WEST,
						  "fill_color",BODY_TEXT_COLOR,
						  NULL);

	ypos += ADDR_SPACING;
	workaddr.street2 = gnome_canvas_item_new (root, 
						  gnome_canvas_text_get_type(),
						  "text", "",
						  "x", 10.0,
						  "y", ypos,
						  "fontset", ADDR_FONTSET,
						  "anchor", GTK_ANCHOR_WEST,
						  "fill_color",BODY_TEXT_COLOR,
						  NULL);

	ypos += ADDR_SPACING;
	workaddr.state = gnome_canvas_item_new (root, 
						  gnome_canvas_text_get_type(),
						  "text", "",
						  "x", 10.0,
						  "y", ypos,
						  "fontset", ADDR_FONTSET,
						  "anchor", GTK_ANCHOR_WEST,
						  "fill_color",BODY_TEXT_COLOR,
						  NULL);

	ypos += ADDR_SPACING;
	workaddr.country = gnome_canvas_item_new (root, 
						  gnome_canvas_text_get_type(),
						  "text", "",
						  "x", 10.0,
						  "y", ypos,
						  "fontset", ADDR_FONTSET,
						  "anchor", GTK_ANCHOR_WEST,
						  "fill_color",BODY_TEXT_COLOR,
						  NULL);


	/* categories */
	ypos += LIST_SPACING;
	categories_box = gnome_canvas_item_new (root, gnome_canvas_rect_get_type (),
			       "x1", 5.0,
			       "y1", ypos,
			       "x2", (double) (CANVAS_WIDTH - 6.0),
			       "y2", (double) ypos+20,
			       "fill_color", HEADER_BOX_COLOR,
			       "outline_color", HEADER_BOX_COLOR,
			       "width_pixels", 0,
			       NULL);

	gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", _("Categories"),
				      "x", 10.0,
				      "y", ypos+10,
				      "fontset",  TITLE_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", HEADER_TEXT_COLOR,
				      NULL);


	ypos += 30;
	categories = gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", "",
				      "x", 5.0,
				      "y", ypos,
				      "fontset", CANVAS_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", BODY_TEXT_COLOR,
				      NULL);

	/* memo(s) */
	ypos += LIST_SPACING;
	memo_box = gnome_canvas_item_new (root, gnome_canvas_rect_get_type (),
			       "x1", 5.0,
			       "y1", ypos,
			       "x2", (double) (CANVAS_WIDTH - 6.0),
			       "y2", (double) ypos+20,
			       "fill_color", HEADER_BOX_COLOR,
			       "outline_color", HEADER_BOX_COLOR,
			       "width_pixels", 0,
			       NULL);

	gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", _("Memo(s)"),
				      "x", 10.0,
				      "y", ypos+10,
				      "fontset",  TITLE_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", HEADER_TEXT_COLOR,
				      NULL);


	ypos += 30;
	memos = gnome_canvas_item_new (root, gnome_canvas_text_get_type (),
				      "text", "",
				      "x", 5.0,
				      "y", ypos,
				      "fontset", CANVAS_FONTSET,
				      "anchor", GTK_ANCHOR_WEST,
				      "fill_color", BODY_TEXT_COLOR,
				      NULL);

	/* set scrolling region */
	gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(root), &x1,&y1,&x2,&y2);
	gnome_canvas_set_scroll_region(GNOME_CANVAS(canvas), x1, y1, x2, y2);

	/* all done */
	return canvas;
}


static void
gnomecard_canvas_text_item_set(GnomeCanvasItem *p, gchar *text)
{
	gnome_canvas_item_set (p, "text", text, NULL);
}

void gnomecard_clear_canvas(void)
{
    gnomecard_canvas_text_item_set(cardname, "");
		gnomecard_canvas_text_item_set(fullname, "");
		
    /* title and organization */
		gnomecard_canvas_text_item_set(title, "");
    gnomecard_canvas_text_item_set(org, "");
		
    /* email and web site */
    gnomecard_canvas_text_item_set(email, "");
		
		gnomecard_canvas_text_item_set(url, "");
		
		gnomecard_canvas_text_item_set(homephone, "");
		gnomecard_canvas_text_item_set(workphone, "");
		
    /* address */
		gnomecard_canvas_text_item_set(homeaddr.street1, "");
		gnomecard_canvas_text_item_set(homeaddr.street2, "");
		gnomecard_canvas_text_item_set(homeaddr.state, "");
		gnomecard_canvas_text_item_set(homeaddr.country, "");
		gnomecard_canvas_text_item_set(workaddr.street1, "");
		gnomecard_canvas_text_item_set(workaddr.street2, "");
		gnomecard_canvas_text_item_set(workaddr.state, "");
		gnomecard_canvas_text_item_set(workaddr.country, "");
		
		/* categories and memos */
		gnomecard_canvas_text_item_set(categories, "");
		gnomecard_canvas_text_item_set(memos, "");
}

void
gnomecard_update_canvas(Card *crd) 
{
	GList *l;
	gchar *name;
	gdouble x1, x2, y1, y2;
	
	g_assert(crd != NULL);
	
	/* card name */
	{
		GdkFont *font;
		double width;
		char *str;
		
		str = (crd->fname.str)? crd->fname.str : _("No fname for this card.");
		
		font = gdk_fontset_load(CARDNAME_FONTSET);
		if (!font) {
			font = gdk_font_load ("fixed");
			g_assert (font != NULL);
		}

		width = gdk_string_width(font, str);
		gdk_font_unref(font);
		
		width = MAX(CANVAS_WIDTH - 6.0, width + 7.0);
		gnomecard_canvas_text_item_set(cardname, str);
		gnome_canvas_item_set(cardname_box, "x2", width, NULL);
		gnome_canvas_item_set(phone_box, "x2", width, NULL);
		gnome_canvas_item_set(deladdr_box, "x2", width, NULL);
		gnome_canvas_item_set(categories_box, "x2", width, NULL);
		gnome_canvas_item_set(memo_box, "x2", width, NULL);
	}
	
	name = gnomecard_join_name(crd->name.prefix,
														 crd->name.given, 
														 crd->name.additional,
														 crd->name.family, 
														 crd->name.suffix);
	/* full name */
	gnomecard_canvas_text_item_set(fullname, name);
	
	/* title and organization */
	if (crd->title.str)
		gnomecard_canvas_text_item_set(title, crd->title.str);
	else
		gnomecard_canvas_text_item_set(title, "");
	
	gnomecard_canvas_text_item_set(org, crd->org.name);
	
	/* email and web site */
	if (crd->email.l)
	  gnomecard_canvas_text_item_set
	  	(email, ((CardEMail *) crd->email.l->data)->data);

	if (crd->url.str)
	  gnomecard_canvas_text_item_set(url, crd->url.str);
	else
	  gnomecard_canvas_text_item_set(url, "");
	
	/* phone numbers */
	if (crd->phone.l) {
		GList *l;
		
		l = phonelist_find_by_type (crd->phone.l, PHONE_HOME);
		if (l) {
			CardPhone *phone;
			
			phone = (CardPhone *)l->data;
			gnomecard_canvas_text_item_set(homephone, phone->data);
		} else
		  gnomecard_canvas_text_item_set(homephone, "");
		
		l = phonelist_find_by_type (crd->phone.l, PHONE_WORK);
		if (l) {
	    CardPhone *phone;
	    
	    phone = (CardPhone *)l->data;
	    gnomecard_canvas_text_item_set(workphone, phone->data);
		} else {
	    gnomecard_canvas_text_item_set(workphone, "");
		}
	} else {
		gnomecard_canvas_text_item_set(homephone, "");
		gnomecard_canvas_text_item_set(workphone, "");
	}	
	
	/* address */
	if (crd->deladdr.l) {
		l = deladdrlist_find_by_type(crd->deladdr.l, ADDR_HOME);
		if (l) {
			gchar tmp[200];
			gchar *fmt1 = "%s, %s   %s";
			gchar *fmt2 = "%s%s  %s";
			gchar *fmt;

			CardDelAddr *p = (CardDelAddr *) l->data;
			
			if (p->data[CITY] && *p->data[CITY] && 
			    p->data[REGION] && *p->data[REGION])
				fmt = fmt1;
			else
				fmt = fmt2;
			
			gnomecard_canvas_text_item_set(homeaddr.street1, 
						       MY_STR(p->data[EXT]));
			gnomecard_canvas_text_item_set(homeaddr.street2,  
						       MY_STR(p->data[STREET]));
			
			snprintf(tmp, sizeof(tmp), fmt, 
				 MY_STR(p->data[CITY]),
				 MY_STR(p->data[REGION]),
				 MY_STR(p->data[CODE]));
			
			gnomecard_canvas_text_item_set(homeaddr.state, tmp);
			gnomecard_canvas_text_item_set(homeaddr.country, 
						       MY_STR(p->data[COUNTRY]));
		} else {
			gnomecard_canvas_text_item_set(homeaddr.street1, "");
			gnomecard_canvas_text_item_set(homeaddr.street2, "");
			gnomecard_canvas_text_item_set(homeaddr.state, "");
			gnomecard_canvas_text_item_set(homeaddr.country, "");
		}
		
		l = deladdrlist_find_by_type(crd->deladdr.l, ADDR_WORK);
		if (l) {
			gchar tmp[200];	
			gchar *fmt1 = "%s, %s   %s";
			gchar *fmt2 = "%s%s  %s";
			gchar *fmt;

			CardDelAddr *p = (CardDelAddr *) l->data;
			
			if (p->data[CITY] && *p->data[CITY] && 
			    p->data[REGION] && *p->data[REGION])
			  fmt = fmt1;
			else
			  fmt = fmt2;
			
			gnomecard_canvas_text_item_set(workaddr.street1, 
						       MY_STR(p->data[EXT]));
			gnomecard_canvas_text_item_set(workaddr.street2,
						       MY_STR(p->data[STREET]));

			snprintf(tmp, sizeof(tmp), fmt, 
				 MY_STR(p->data[CITY]),
				 MY_STR(p->data[REGION]),
				 MY_STR(p->data[CODE]));
			
			gnomecard_canvas_text_item_set(workaddr.state, tmp);
			gnomecard_canvas_text_item_set(homeaddr.country, 
						       MY_STR(p->data[COUNTRY]));
		} else {
			gnomecard_canvas_text_item_set(workaddr.street1, "");
			gnomecard_canvas_text_item_set(workaddr.street2, "");
			gnomecard_canvas_text_item_set(workaddr.state, "");
			gnomecard_canvas_text_item_set(workaddr.country, "");
		}
	} else {
		gnomecard_canvas_text_item_set(homeaddr.street1, "");
		gnomecard_canvas_text_item_set(homeaddr.street2, "");
		gnomecard_canvas_text_item_set(homeaddr.state, "");
		gnomecard_canvas_text_item_set(homeaddr.country, "");
		gnomecard_canvas_text_item_set(workaddr.street1, "");
		gnomecard_canvas_text_item_set(workaddr.street2, "");
		gnomecard_canvas_text_item_set(workaddr.state, "");
		gnomecard_canvas_text_item_set(workaddr.country, "");
	}
	
	/* categories */
	if (crd->categories.str)
		gnomecard_canvas_text_item_set(categories, crd->categories.str);
	else
		gnomecard_canvas_text_item_set(categories, "");
	
	/* memos */
	if (crd->comment.str)
		gnomecard_canvas_text_item_set(memos, crd->comment.str);
	else
		gnomecard_canvas_text_item_set(memos, "");
	
	/* set scrolling region */
	gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(root), &x1,&y1,&x2,&y2);
	gnome_canvas_set_scroll_region(GNOME_CANVAS(gnomecard_canvas),
																 x1, y1, x2, y2);
}
