/* lib/cmd -- irqtune support library (command execution) */

#pragma member irqtune.h

/* cmdopen -- open command pipe */
/* RETURNS: stream pointer */
struct cmdblk *
cmdopen(const char *cmdbf)
{
	int ok;
	int err;
	int unit;
	int idx;
	int argc;
	struct cmdblk *cmd;
	char *argv[100];

	do {
		/* allocate a new command block */
		ok = 0;
		for (CMDFOR(cmd)) {
			if (! (cmd->cmd_opt & CMDBUSY)) {
				ok = 1;
				break;
			}
		}
		if (! ok) {
			sysfault("cmdopen: unable to locate cmdblk\n");
			break;
		}
		memset(cmd,0,sizeof(struct cmdblk));
		cmd->cmd_opt |= CMDBUSY;

		/* enable signal handler */
		cmdsigon();

		/* copy command to buffer */
		strcpy(cmd->cmd_bf,cmdbf);

		/* create a pipe */
		err = pipe(cmd->cmd_units);
		if (err < 0) {
			sysfault("cmdopen: unable to create pipe -- %s\n",
				strerror(errno));
			cmd->cmd_units[0] = -1;
			cmd->cmd_units[1] = -1;
			break;
		}

		/* fork a child */
		cmd->cmd_pid = fork();
		if (cmd->cmd_pid < 0)
			sysfault("cmdopen: unable to fork -- %s\n",strerror(errno));

		/* handle parent */
		if (cmd->cmd_pid > 0) {
			ok = 1;
			close(cmd->cmd_units[1]);
			break;
		}

		/* close parent's unit */
		close(cmd->cmd_units[0]);

		/* open our stuff on the write units */
		for (unit = 1;  unit <= 2;  ++unit) {
			switch (unit) {
			case 1:
				fclose(stdout);
				break;
			case 2:
				fclose(stderr);
				break;
			}

			/* force it onto the correct unit */
			err = dup2(cmd->cmd_units[1],unit);
			if (err < 0)
				exit(errno);

			/* preserve it across exec */
			err = fcntl(F_SETFD,unit,0);
			if (err < 0)
				exit(errno);
		}

		/* close pipe units now that they've been dup'ed */
		for (idx = 0;  idx <= 1;  ++idx)
			close(cmd->cmd_units[idx]);

		/* split the arguments */
		argc = xstrargv(argv,Cntof(argv),cmd->cmd_bf);

		/* execute the program */
		err = execvp(argv[0],argv);
		if (err < 0)
			exit(errno);
	} while (0);

	/* kill off stuff if error */
	if (! ok) {
		cmdclose(cmd);
		cmd = NULL;
	}

	return cmd;
}

/* cmdclose -- close command pipe */
/* RETURNS: final status */
int
cmdclose(struct cmdblk *cmd)
{
	int status;
	char linebf[4096];

	status = -1;
	do {
		if (cmd == NULL)
			break;

		/* autoreap the child cleanly */
		while (1) {
			if (cmdread(cmd,linebf,sizeof(linebf)) < 0)
				break;
		}

		/* get status */
		status = cmd->cmd_status;

		/* close open pipe */
		if (cmd->cmd_units[0] >= 0) {
			close(cmd->cmd_units[0]);
			cmd->cmd_units[0] = -1;
		}

		/* release storage */
		cmd->cmd_opt &= ~CMDBUSY;

		/* release signal handler */
		cmdsigoff();
	} while (0);

	return status;
}

/* cmdcoredump -- decide if program terminated due to core dump */
/* RETURNS: 0=normal, 1=limited, 2=dumped */
int
cmdcoredump(int status)
{
	int err;

	err = 0;
	do {
		/* all core dumps are done via the signal mechanism */
		if (! WIFSIGNALED(status))
			break;

		switch (WTERMSIG(status)) {
		case SIGQUIT:
		case SIGILL:
		case SIGTRAP:
		case SIGIOT:
		case SIGFPE:
		case SIGSEGV:
			err = 1;
			break;
		}
		if (! err)
			break;

		/* FIXME -- WCOREDUMP only defined if __USE_BSD? */
		if (WCOREDUMP(status))
			err = 2;
	} while (0);

	return err;
}

/* cmdread -- read in command line */
/* RETURNS: length (-1=EOF or error) */
int
cmdread(struct cmdblk *cmd,char *bf,int bflen)
{
	int err;
	int totlen;
	int unit;
	char *bp;
	char *bfe;
	fd_set selmsk;

	bp = bf;
	bfe = bp + bflen - 1;

	totlen = 0;
	unit = cmd->cmd_units[0];
	for (;  bp < bfe;  ++bp, ++totlen) {
		/* wait for data */
		FD_ZERO(&selmsk);
		FD_SET(unit,&selmsk);
		err = select(unit + 1,&selmsk,NULL,NULL,NULL);
		if (err <= 0)
			break;

		/* get a character */
		err = read(unit,bp,1);
		if (err < 0) {
			sysfault("cmdread: error -- %s\n",strerror(errno));
			totlen = -1;
			break;
		}

		/* handle an EOF */
		if (err == 0) {
			if (totlen <= 0)
				totlen = -1;
			break;
		}

		/* stop when we've got a line */
		if (*bp == '\n')
			break;
	}

	/* close off buffer */
	if (totlen < 0)
		bp = bf;
	*bp = 0;

	return totlen;
}

/* cmdsigon -- enable signal handler */
void
cmdsigon(void)
{
	int err;
	sigset_t sigmsk;
	struct sigaction newact;

	/* attach signal handler for SIGCHLD */
	if (cmdsigcnt++ == 0) {
		/* prepare */
		sigemptyset(&sigmsk);
		sigaddset(&sigmsk,SIGCHLD);
		newact.sa_handler = cmdsighdr;
		newact.sa_mask = sigmsk;
		newact.sa_flags = SA_RESTART;

		/* block the signals before setting signal handler */
		err = sigprocmask(SIG_BLOCK,&sigmsk,&cmdsigmsksv);
		if (err < 0) {
			sysfault("cmdsigon: unable to sigprocmask -- %s\n",
				strerror(errno));
		}

		/* set handler */
		err = sigaction(SIGCHLD,&newact,&cmdsigactsv);
		if (err < 0) {
			sysfault("cmdsigon: unable to sigaction -- %s\n",
				strerror(errno));
		}

		/* unblock the signals after setting signal handler */
		err = sigprocmask(SIG_UNBLOCK,&sigmsk,NULL);
		if (err < 0) {
			sysfault("cmdsigon: unable to sigprocmask -- %s\n",
				strerror(errno));
		}
	}
}

/* cmdsigoff -- disable signal handler */
void
cmdsigoff(void)
{
	int err;

	/* detach signal handler for SIGCHLD */
	if (--cmdsigcnt == 0) {
		/* restore original handler */
		err = sigaction(SIGCHLD,&cmdsigactsv,NULL);
		if (err < 0) {
			sysfault("cmdsigoff: unable to sigaction -- %s\n",
				strerror(errno));
		}

		/* restore original block mask */
		err = sigprocmask(SIG_SETMASK,&cmdsigmsksv,NULL);
		if (err < 0) {
			sysfault("cmdsigoff: unable to sigaction -- %s\n",
				strerror(errno));
		}
	}
}

/* cmdsighdr -- command signal handler */
void
cmdsighdr(int signo)
{
	struct cmdblk *cmd;
	int status;
	int pid;

	switch (signo) {
	case SIGCHLD:
		pid = wait(&status);
		for (CMDFOR(cmd)) {
			if (! (cmd->cmd_opt & CMDBUSY))
				continue;

			/* reap the child */
			if (cmd->cmd_pid == pid) {
				cmd->cmd_status = status;
				cmd->cmd_pid = 0;
				break;
			}
		}
		break;
	}
}

/* vsystem -- execute program */
/* RETURNS: error code */
int
vsystem(int verbose,const char *tag0,const char *tag1,const char *tag2,const char *cmdbf0)
/* tag0 -- primary command (e.g. loading, unloading, etc.) */
/* tag1 -- primary tag */
/* tag2 -- secondary tag */
/* cmdbf0 -- actual command buffer */
{
	char *cp;
	int code;
	int status;
	struct cmdblk *cmd;
	char cmdbf[4096];
	char linebf[4096];

	/* copy command to buffer */
	strcpy(cmdbf,cmdbf0);

	/* show blah_blah_blah only if full errors or verbosity enabled */
	if (mstopt & OPTVERBOSE)
		verbose = 1;

	/* show stuff beforehand */
	if (mstopt & OPTERR)
		zprtx("irqtune: trying command -- %s\n",cmdbf0);
	else {
		zprtx("irqtune: %s",tag0);
		if (tag1 != NULL)
			zprtx(" %s",tag1);
		if (tag2 != NULL)
			zprtx(" (%s)",tag2);
		zprtx("\n");
	}

	/* get command status */
	cmd = cmdopen(cmdbf);
	while (verbose) {
		if (cmdread(cmd,linebf,sizeof(linebf)) < 0)
			break;
		zprtx("%s\n",linebf);
	}
	status = cmdclose(cmd);

	/* check the status */
	code = 1;
	do {
		/* handle exec error */
		if (status < 0) {
			zprtx("irqtune: exec failed of '%s' -- %s\n",
				cmdbf0,strerror(errno));
			break;
		}

		/* handle semi-normal program termination */
		if (WIFEXITED(status)) {
			code = WEXITSTATUS(status);
			code = BOOL(code);
		}
		if (! code)
			break;

		/* show a short, sweet, to the point error message */
		cp = strtok(cmdbf," \t");
		cp = filetail(cp);
		zprtx("irqtune: %s failed on '%s'",cp,tag1);

		/* show the really nasty core dumps */
		code = cmdcoredump(status);
		switch (code) {
		case 1:
			zprtx(" (crashed -- core dump not taken)");
			break;
		case 2:
			zprtx(" (crashed -- core dumped)");
			break;
		}

		/* show the absolute raw status */
		if (mstopt & (OPTERR | OPTVERBOSE))
			zprtx(" status=%8.8X",cmd->cmd_status);

		code = 1;
	} while (0);
	if (code)
		zprtx("\n");

	return code;
}

