/*
    libparted
    Copyright (C) 1998-2000  Andrew Clausen  <clausen@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "fat.h"
#include "traverse.h"
#include "count.h"
#include "fatio.h"
#include "calc.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <stdarg.h>
#include <string.h>

#ifdef USE_NEWT
#include <newt.h>
#endif

#define MIN(a, b) (a<b) ? a : b;
#define MAX(a, b) (a>b) ? a : b;
#define BUFFER_SIZE 64

/*
    Recursively duplicates a directory tree, modified for the new file
    locations.
*/
static int
fat_construct_directory (FatOpContext* ctx, FatTraverseInfo* trav_info)
{
	FatTraverseInfo*	sub_dir_info;
	FatDirEntry*		dir_entry;

	while ( (dir_entry = fat_traverse_next_dir_entry (trav_info)) ) {
		if (!fat_dir_entry_has_first_cluster (dir_entry, ctx->old_fs))
			continue;

		fat_traverse_mark_dirty (trav_info);
		fat_dir_entry_set_first_cluster (dir_entry, ctx->new_fs,
			ctx->remap [fat_dir_entry_get_first_cluster
					(dir_entry, ctx->old_fs)]);

		if (fat_dir_entry_is_directory (dir_entry)
				&& dir_entry->name [0] != '.') {
			sub_dir_info
				= fat_traverse_directory (trav_info, dir_entry);
			if (!sub_dir_info)
				return 0;
			if (!fat_construct_directory (ctx, sub_dir_info))
				return 0;
		}
	}
	fat_traverse_complete (trav_info);
	return 1;
}

static int
duplicate_legacy_root_dir (FatOpContext* ctx)
{
	FatSpecific*		old_fs_info = FAT_SPECIFIC (ctx->old_fs);
	FatSpecific*		new_fs_info = FAT_SPECIFIC (ctx->new_fs);

	PED_ASSERT (old_fs_info->root_dir_sector_count
			== new_fs_info->root_dir_sector_count, return 0);

	if (!ped_geometry_read (ctx->old_fs->geom, old_fs_info->buffer,
				old_fs_info->root_dir_offset,
				old_fs_info->root_dir_sector_count))
		return 0;

	if (!ped_geometry_write (ctx->new_fs->geom, old_fs_info->buffer,
				 new_fs_info->root_dir_offset,
				 new_fs_info->root_dir_sector_count))
		return 0;

	return 1;
}

/*
    Constructs the new directory tree for legacy (FAT16) filesystems.
*/
static int
fat_construct_legacy_root (FatOpContext* ctx)
{
	FatTraverseInfo*	trav_info;

	if (!duplicate_legacy_root_dir (ctx))
		return 0;
	trav_info = fat_traverse_begin (ctx->new_fs, FAT_ROOT, "\\");
	return fat_construct_directory (ctx, trav_info);
}

/*
    Constructs the new directory tree for new (FAT32) filesystems.
*/
static int
fat_construct_root (FatOpContext* ctx) {
	FatSpecific*		new_fs_info = FAT_SPECIFIC (ctx->new_fs);
	FatTraverseInfo*	trav_info;

	trav_info = fat_traverse_begin (ctx->new_fs, new_fs_info->root_cluster,
					"\\");
	fat_construct_directory (ctx, trav_info);
	return 1;
}

/* Converts the root directory between FAT16 and FAT32.  NOTE: this code
 * can also do no conversion.  I'm leaving fat_construct_directory(), because
 * it's really pretty :-)  It also leaves a higher chance of deleted file
 * recovery, because it doesn't remove redundant entries.  (We do this here,
 * because brain-damaged FAT16 has an arbitary limit on root directory entries,
 * so we save room)
 */
static int
fat_convert_directory (FatOpContext* ctx, FatTraverseInfo* old_trav,
		       FatTraverseInfo* new_trav)
{
	FatTraverseInfo*	sub_old_dir_trav;
	FatTraverseInfo*	sub_new_dir_trav;
	FatDirEntry*		new_dir_entry;
	FatDirEntry*		old_dir_entry;

	while ( (old_dir_entry = fat_traverse_next_dir_entry (old_trav)) ) {
		new_dir_entry = fat_traverse_next_dir_entry (new_trav);

		if (!fat_dir_entry_is_active (old_dir_entry))
			continue;

		if (!new_dir_entry) {
			return ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_IGNORE_CANCEL,
				_("There's not enough room in the root "
				  "directory, for all of the files.  Either "
				  "cancel, or ignore to lose the files."))
					== PED_EXCEPTION_IGNORE;
		}

		*new_dir_entry = *old_dir_entry;
		fat_traverse_mark_dirty (new_trav);

		if (!fat_dir_entry_has_first_cluster (old_dir_entry,
						      ctx->old_fs))
			continue;

		fat_dir_entry_set_first_cluster (new_dir_entry, ctx->new_fs,
			ctx->remap [fat_dir_entry_get_first_cluster
					(old_dir_entry, ctx->old_fs)]);

		if (fat_dir_entry_is_directory (old_dir_entry)
				&& old_dir_entry->name [0] != '.') {
			sub_old_dir_trav
			    = fat_traverse_directory (old_trav, old_dir_entry);
			sub_new_dir_trav
			    = fat_traverse_directory (new_trav, new_dir_entry);
			if (!sub_old_dir_trav || !sub_new_dir_trav)
				return 0;

			if (!fat_convert_directory (ctx, sub_old_dir_trav,
						    sub_new_dir_trav))
				return 0;
		}
	}

	fat_traverse_complete (old_trav);
	fat_traverse_complete (new_trav);
	return 1;
}

static void
clear_cluster (PedFileSystem* fs, FatCluster cluster)
{
	FatSpecific*		fs_info = FAT_SPECIFIC (fs);

	memset (fs_info->buffer, 0, fs_info->cluster_size);
	fat_write_cluster (fs, fs_info->buffer, cluster);
}

/* This MUST be called BEFORE the fat_construct_new_fat(), because cluster
 * allocation depend on the old FAT.  The reason is, old clusters may
 * still be needed during the resize, (particularly clusters in the directory
 * tree) even if they will be discarded later.
 */
static int
alloc_root_dir (FatOpContext* ctx)
{
	FatSpecific*		new_fs_info = FAT_SPECIFIC (ctx->new_fs);
	int			i;
	FatCluster		cluster;

	PED_ASSERT (new_fs_info->fat_type == FAT_TYPE_FAT32, return 0);

	for (i=0; i<16; i++) {
		cluster = fat_table_alloc_check_cluster (new_fs_info->fat,
							 ctx->new_fs);
		if (!cluster)
			return 0;
		ctx->new_root_dir [i] = cluster;
		clear_cluster (ctx->new_fs, cluster);
	}
	ctx->new_root_dir [i] = 0;
	new_fs_info->root_cluster = ctx->new_root_dir [0];
	return 1;
}

static int
fat_clear_root_dir (PedFileSystem* fs)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);
	int		i;

	PED_ASSERT (fs_info->fat_type == FAT_TYPE_FAT16, return 0);
	PED_ASSERT (fs_info->root_dir_sector_count, return 0);

	memset (fs_info->buffer, 0, 512);

	for (i = 0; i < fs_info->root_dir_sector_count; i++) {
		if (!ped_geometry_write (fs->geom, fs_info->buffer,
					 fs_info->root_dir_offset + i, 1)) {
			if (ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_IGNORE_CANCEL,
				_("Error writing to the root directory."))
					== PED_EXCEPTION_CANCEL)
				return 0;
		}
	}
	return 1;
}

static int
fat_construct_converted_tree (FatOpContext* ctx)
{
	FatSpecific*		old_fs_info = FAT_SPECIFIC (ctx->old_fs);
	FatSpecific*		new_fs_info = FAT_SPECIFIC (ctx->new_fs);
	FatTraverseInfo*	old_trav_info;
	FatTraverseInfo*	new_trav_info;

	if (new_fs_info->fat_type == FAT_TYPE_FAT32) {
		new_trav_info = fat_traverse_begin (ctx->new_fs,
					    new_fs_info->root_cluster, "\\");
		old_trav_info = fat_traverse_begin (ctx->old_fs, FAT_ROOT,
						    "\\");
	} else {
		fat_clear_root_dir (ctx->new_fs);
		new_trav_info = fat_traverse_begin (ctx->new_fs, FAT_ROOT,
						    "\\");
		old_trav_info = fat_traverse_begin (ctx->old_fs,
					    old_fs_info->root_cluster, "\\");
	}
	if (!new_trav_info || !old_trav_info)
		return 0;
	if (!fat_convert_directory (ctx, old_trav_info, new_trav_info))
		return 0;
	return 1;
}

/*
    Constructs the new directory tree to match the new file locations.
*/
static int
fat_construct_dir_tree (FatOpContext* ctx)
{
	FatSpecific*		new_fs_info = FAT_SPECIFIC (ctx->new_fs);
	FatSpecific*		old_fs_info = FAT_SPECIFIC (ctx->old_fs);

	if (new_fs_info->fat_type == old_fs_info->fat_type) {
		switch (old_fs_info->fat_type) {
			case FAT_TYPE_FAT16:
			return fat_construct_legacy_root (ctx);

			case FAT_TYPE_FAT32:
			return fat_construct_root (ctx);
		}
	} else {
		return fat_construct_converted_tree (ctx);
	}

	return 0;
}

/*
    constructs the new fat for the resized filesystem.
*/
static int
fat_construct_new_fat (FatOpContext* ctx) {
	FatSpecific*	old_fs_info = FAT_SPECIFIC (ctx->old_fs);
	FatSpecific*	new_fs_info = FAT_SPECIFIC (ctx->new_fs);
	FatCluster	old_cluster;
	FatCluster	new_cluster;
	FatCluster	old_next_cluster;
	int		i;

	fat_table_clear (new_fs_info->fat);
	new_fs_info->fat->cluster_count = new_fs_info->cluster_count;

	for (old_cluster=2; old_cluster < old_fs_info->cluster_count + 2;
	     old_cluster++) {
		if (old_fs_info->fat_flag_map [old_cluster] == FAT_FLAG_FREE)
			continue;
			
		old_next_cluster = fat_table_get (old_fs_info->fat, old_cluster);
		new_cluster = ctx->remap [old_cluster];
		if (fat_table_is_eof (old_fs_info->fat, old_next_cluster))
			fat_table_set_eof (new_fs_info->fat, new_cluster);
		else
			fat_table_set (new_fs_info->fat, new_cluster,
				       ctx->remap [old_next_cluster]);
	}

#ifdef VERBOSE
	for (old_cluster=2; old_cluster < old_fs_info->cluster_count+2;
	     old_cluster++) {
		if (fat_table_is_available (old_fs_info->fat, old_cluster))
			continue;

		printf ("%d->%d\t(next: %d->%d)\n",
			old_cluster,
			ctx->remap [old_cluster],
			fat_table_get (old_fs_info->fat, old_cluster),
			fat_table_get (new_fs_info->fat,
				       ctx->remap [old_cluster]));
	}
#endif /* VERBOSE */

	if (old_fs_info->fat_type == FAT_TYPE_FAT32
	    && new_fs_info->fat_type == FAT_TYPE_FAT32) {
		new_fs_info->root_cluster
			= ctx->remap [old_fs_info->root_cluster];
	}

	if (old_fs_info->fat_type == FAT_TYPE_FAT16
	    && new_fs_info->fat_type == FAT_TYPE_FAT32) {
		for (i=0; ctx->new_root_dir[i+1]; i++) {
			fat_table_set (new_fs_info->fat,
				       ctx->new_root_dir[i],
				       ctx->new_root_dir[i+1]);
		}
		fat_table_set_eof (new_fs_info->fat, ctx->new_root_dir[i]);
	}

	return 1;
}

/*  Creates the PedFileSystem struct for the new resized file system, and
    sticks it in a FatOpContext.  At the end of the process, the original
    (ctx->old_fs) is destroyed, and replaced with the new one (ctx->new_fs).
 */
static FatOpContext*
create_resize_context (PedFileSystem* fs, PedGeometry* new_geom)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);
	FatSpecific*	new_fs_info;
	PedFileSystem*	new_fs;
	int		new_cluster_size;
	FatCluster	new_cluster_count;
	PedSector	new_fat_sectors;
	FatType		new_fat_type;
	PedSector	root_dir_sector_count;
	FatOpContext*	context;

	/* hypothetical number of root dir sectors, if we end up using
	 * FAT16
	 */
	if (fs_info->root_dir_sector_count)
		root_dir_sector_count = fs_info->root_dir_sector_count;
	else
		root_dir_sector_count = FAT_ROOT_DIR_ENTRY_COUNT
						* sizeof (FatDirEntry) / 512;

	if (!fat_calc_resize_sizes (new_geom,
				    fs_info->cluster_sectors,
				    fs_info->cluster_size,
				    root_dir_sector_count,
				    &new_cluster_count,
				    &new_fat_sectors,
				    &new_fat_type)) {

		if (!fat_calc_sizes (new_geom, fs_info->cluster_sectors,
				     FAT_TYPE_FAT16,
				     root_dir_sector_count,
				     &new_cluster_size,
				     &new_cluster_count,
				     &new_fat_sectors)) {
			ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_CANCEL,
				_("Partition too small for a FAT file "
				  "system."));
		} else {
			ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
				PED_EXCEPTION_CANCEL,
				_("To resize this filesystem to this "
				  "size, the cluster size must be changed "
				  "from %d bytes to %d bytes.  This "
				  "functionality is not yet implemented.  "
				  "Note that there are many weird different "
				  "sizes that can be achieved with the same "
				  "cluster size - if you keep experimenting "
				  "with different partition sizes, you might "
				  "be lucky!!!"),
				fs_info->cluster_size, new_cluster_size);
		}
		return 0;
	}

	if (fs_info->fat_type == FAT_TYPE_FAT16
			&& new_fat_type == FAT_TYPE_FAT32) {
		/* FAT32 */
		if (ped_exception_throw (PED_EXCEPTION_WARNING,
			PED_EXCEPTION_OK_CANCEL,
			_("The filesystem is going to be too big for FAT16, "
			"(with a cluster size of %dk, anyway) so FAT32 will be "
			"used.  This is not compatible with MS-DOS, early "
			"versions of MS-Windows 95 and Windows NT.  If you use "
			"these operating systems, then select cancel, and "
			"create a smaller partition.  If you only use Linux, "
			"BSD, MS Windows 98 and/or MS Windows 95 B, then "
			"select OK."),
			fs_info->cluster_sectors / 2)
				== PED_EXCEPTION_CANCEL)
			return 0;
	}

	if (!fat_check_resize_geometry (fs, new_geom, new_cluster_count))
		goto error;

	new_fs = fat_file_system_alloc (new_geom);
	if (!new_fs)
		goto error;

	new_fs_info = FAT_SPECIFIC (new_fs);
	if (!new_fs_info)
		goto error_free_new_fs;

/* preserve boot code, etc. */
	memcpy (&new_fs_info->boot_sector, &fs_info->boot_sector,
		sizeof (FatBootSector));
	memcpy (&new_fs_info->info_sector, &fs_info->info_sector,
		sizeof (FatInfoSector));

	new_fs_info->logical_sector_size = fs_info->logical_sector_size;
	new_fs_info->sector_count = new_geom->length;

	new_fs_info->cluster_size = fs_info->cluster_size;
	new_fs_info->cluster_sectors = fs_info->cluster_sectors;
	new_fs_info->cluster_count = new_cluster_count;
	new_fs_info->dir_entries_per_cluster = fs_info->dir_entries_per_cluster;

	new_fs_info->fat_type = new_fat_type; 
	new_fs_info->fat_table_count = 2;
	new_fs_info->fat_sectors = new_fat_sectors;

	if (new_fs_info->fat_type == FAT_TYPE_FAT32) {
		new_fs_info->info_sector_offset	= 1;
		new_fs_info->boot_sector_backup_offset = 6;

		new_fs_info->root_dir_offset = 0;
		new_fs_info->root_dir_entry_count = 0;
		new_fs_info->root_dir_sector_count = 0;

		/* we add calc_align_sectors to push the cluster_offset
		   forward, to keep the clusters aligned between the new
		   and old file systems
		 */
		new_fs_info->fat_offset
			= fat_min_reserved_sector_count (FAT_TYPE_FAT32)
			  + fat_calc_align_sectors (new_fs, fs);

		new_fs_info->cluster_offset
			= new_fs_info->fat_offset
			  + 2 * new_fs_info->fat_sectors
			  - 2 * new_fs_info->cluster_sectors;
	} else {
		new_fs_info->root_dir_sector_count = root_dir_sector_count;
		new_fs_info->root_dir_entry_count
			= root_dir_sector_count * 512 / sizeof (FatDirEntry);

		new_fs_info->fat_offset
			= fat_min_reserved_sector_count (FAT_TYPE_FAT16)
			  + fat_calc_align_sectors (new_fs, fs);

		new_fs_info->root_dir_offset = new_fs_info->fat_offset
					       + 2 * new_fs_info->fat_sectors;

		new_fs_info->cluster_offset = new_fs_info->root_dir_offset
					  + new_fs_info->root_dir_sector_count
					  - 2 * new_fs_info->cluster_sectors;
	}
	
	new_fs_info->total_dir_clusters = fs_info->total_dir_clusters;

	context = fat_op_context_new (new_fs, fs);
	if (!context)
		goto error_free_new_fs_info;

	if (!fat_op_context_create_initial_fat (context))
		goto error_free_context;

	if (!fat_file_system_alloc_buffers (new_fs))
		goto error_free_fat;

	return context;

error_free_fat:
	fat_table_destroy (new_fs_info->fat);
error_free_context:
	ped_free (context);
error_free_new_fs_info:
	ped_free (new_fs_info);
error_free_new_fs:
	ped_free (new_fs);
error:
	return NULL;
}

static int
resize_context_assimilate (FatOpContext* ctx)
{
	FatSpecific*	old_fs_info = FAT_SPECIFIC (ctx->old_fs);

	fat_table_destroy (old_fs_info->fat);
	ped_free (old_fs_info);
	ped_geometry_destroy (ctx->old_fs->geom);

	ctx->old_fs->type_specific = ctx->new_fs->type_specific;
	ctx->old_fs->geom = ctx->new_fs->geom;

	ped_free (ctx->new_fs);

	return 1;
}

int
fat_resize (PedFileSystem* fs, PedGeometry* geom)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);
	FatSpecific*	new_fs_info;
	FatOpContext*	ctx;
	PedFileSystem*	new_fs;

	ctx = create_resize_context (fs, geom);
	if (!ctx)
		return 0;
	new_fs = ctx->new_fs;
	new_fs_info = FAT_SPECIFIC (new_fs);

	if (!fat_duplicate_clusters (ctx))
		return 0;
	if (fs_info->fat_type == FAT_TYPE_FAT16
			&& new_fs_info->fat_type == FAT_TYPE_FAT32) {
		if (!alloc_root_dir (ctx))
			return 0;
	}
	if (!fat_construct_new_fat (ctx))
		return 0;
	if (!fat_construct_dir_tree (ctx))
		return 0;
	if (!fat_table_write_all (new_fs_info->fat, new_fs))
		return 0;

	if (!fat_boot_sector_generate (&new_fs_info->boot_sector, new_fs))
		return 0;
	if (!fat_boot_sector_write (&new_fs_info->boot_sector, new_fs))
		return 0;
	if (new_fs_info->fat_type == FAT_TYPE_FAT32) {
		if (!fat_info_sector_generate (&new_fs_info->info_sector,
					       new_fs))
			return 0;
		if (!fat_info_sector_write (&new_fs_info->info_sector, new_fs))
			return 0;
		/* I think this is just some boot code...copy it anyway :-) */
/*
		if (!ped_geometry_read (fs->geom, fs_info->buffer, 2, 1))
			return 0;
		if (!ped_geometry_write (new_fs->geom, fs_info->buffer, 2, 1))
			return 0;
*/
	}

	if (!resize_context_assimilate (ctx))
		return 0;

#ifdef VERBOSE
	fat_print (fs);
#endif

	return 1;
}

