/*
 * Copyright (c) 2000 Dave Jones <dave@powertweak.com>
 *
 * Can be freely distributed and used under the terms of the GNU GPL.
 */

/* Globals */
FILE *configfilehandle;
size_t configfilesize;
char *configfileptr;


int read_configfile(void)
{
	const char configfilename[]="/etc/powertweak.config";
	const char runconftxt[]="Run powertweak-config first.\n\n";
	struct stat fileinfostruct;

	/* Get some info on the config file, so we can determine the file size. */
	if (stat(configfilename, &fileinfostruct) != 0) {
		if (open_log_file(&logfile)==TRUE) {
			fprintf(logfile, "Can't open %s.\n%s", configfilename, runconftxt);
			close_log_file(&logfile);
		}
		printf("Can't open %s.\n%s", configfilename, runconftxt);
		return (FALSE);
	}
	configfilesize = fileinfostruct.st_size;

	if (configfilesize == 0) {
		printf("Error: config file is empty.\n%s", runconftxt);
		return (FALSE);
	}

	/* Allocate memory for config file. */
	configfileptr = malloc(configfilesize);
	if (configfileptr== NULL) {
		if (open_log_file(&logfile)==TRUE) {
			fprintf(logfile, "Couldn't malloc %d bytes for the config file.\n", configfilesize);
			close_log_file(&logfile);
		}
		printf("Couldn't malloc %d bytes for the config file.\n", configfilesize);
		return (FALSE);
	}


	/* We have the memory allocated, so read in the config file. */
	if ((configfilehandle = fopen (configfilename, "rb")) == NULL) {
		fprintf (stderr, "Error opening input file : %s\t%s\n",
		configfilename, sys_errlist[errno]);
		free (configfileptr);
		return (FALSE);
	}

	if (fread (configfileptr, configfilesize, 1, configfilehandle) != 1) {
		fprintf (stderr, "Error reading from input file : %s\t%s\n", configfilename, sys_errlist[errno]);
		if (fclose (configfilehandle) != 0) {
			fprintf (stderr, "Error closing input file : %s\t%s\n", configfilename, sys_errlist[errno]);
		}
		free (configfileptr);
		return (FALSE);
    }

    if (fclose (configfilehandle) != 0) {
		fprintf (stderr, "Error closing input file : %s\t%s\n", configfilename, sys_errlist[errno]);
    }

    return (TRUE);
}



/*
 * This routine greps the config file for specific tags, and returns the
 * value that is attached to it.
 */

#define OPTION_UNSET -1
#define OPTION_OFF 0
#define OPTION_ON  1

int find_PCI_tag(unsigned long device_id, const char* tagname, unsigned char *result)
{
	unsigned int i;
	char *fileptr;

	fileptr=configfileptr;

	for (i=0;i<configfilesize;i++) {

next:
		switch (*fileptr) {
			case '#' :	/* Skip comments to EOL. */
					while (*fileptr!='\n' && i<configfilesize) {
						fileptr++;
						i++;
					}
					goto next;

			case '\n':	fileptr++;
					i++;
					goto next;
		}

		/* Bomb out if we hit the end of the file. */
		if (i>=configfilesize)
			return (FALSE);

		/* Check it has the device number. */
		if (strtoul(fileptr, (&fileptr)+8,16) == device_id) {

			/* Skip the _ after the device ID number */
			if (*(fileptr+8)!='_') {
				printf("Corrupt config file\n");
				return (FALSE);
			}

			/* Check for the tweak name*/
			if (strncmp (fileptr+9, tagname, strlen(tagname))==0) {
				fileptr+=(9+(strlen(tagname)));
				i+=(9+strlen(tagname));
				if (*fileptr=='=') {
					fileptr++;
					i++;
					if (isdigit(*fileptr)!=0) {
						*result=atoi(fileptr);
						return (TRUE);
					} else {
						return (OPTION_UNSET);
					}
				} else {
				/* No =value after tagname! Assume value=1 */
					*result=1;
					return (TRUE);
				}
			} else {
				/*right chip, wrong tweak name. */
				while (*fileptr!='\n' && i<configfilesize) {
					fileptr++;
					i++;
				}
				goto next;
			}
		} else {
			fileptr++;
		}
    }
    return (FALSE);
}


int find_PROC_tag(const char* tagname, char **result)
{
	unsigned int i;
	char *fileptr;

	fileptr=configfileptr;

	for (i=0;i<configfilesize;i++) {

next:
		switch (*fileptr) {
			case '#' :	/* Skip comments to EOL. */
					while (*fileptr!='\n' && i<configfilesize) {
						fileptr++;
						i++;
					}
					goto next;

			case '\n':	fileptr++;
					i++;
					goto next;
		}

		/* Bomb out if we hit the end of the file. */
		if (i>=configfilesize)
			return (FALSE);

		/* Check it has the correct prefix. */
		if (strncmp (fileptr, "PROC_", 5)==0) {
			fileptr+=5;

			/* Check for the tweak name*/
			if (strncmp (fileptr, tagname, strlen(tagname))==0) {
				/* Yes, its the right tag.*/
				fileptr+=((strlen(tagname)));
				i+=(strlen(tagname));
				if (*fileptr=='=') {
					fileptr++;
					*result=fileptr;
					while (*fileptr!='\n' && i<configfilesize) {
						fileptr++;
						i++;
					}
					*fileptr='\0';	/* Null terminate the tag*/
					return (TRUE);
				} else {
					return (OPTION_UNSET);
				}
	
			} else {
				/* wrong tweak name, skip to EOL.*/
				while ((*fileptr!='\n' && *fileptr!=0 ) && i<configfilesize) {
					fileptr++;
					i++;
				}
			}
		}
		fileptr++;
    }
    return (FALSE);
}
