/* alarm.c -- set the alarm power on time of a Toshiba laptop.
 *
 * Copyright (c) 1998-99  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: alarm.c,v $
 * Revision 1.2  2000/02/10 23:49:17  jab
 * fixed reporting of the year the alarm dates are set to
 *
 * Revision 1.1  1999/12/04 13:40:07  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

static char const rcsid[]="$Id: alarm.c,v 1.2 2000/02/10 23:49:17 jab Exp jab $";


#include<stdio.h>
#include<stdlib.h>
#include<unistd.h>
#include<time.h>
#ifdef __GLIBC__
#include<sys/perm.h>
#endif

#include"sci.h"
#include"parsetime.h"


#define VERSION_STRING "\
Alarm version 1.1\n\
Copyright (c) 1998-2000 Jonathan A. Buzzard <jonathan@buzzard.org.uk>\n"

#define USAGE_STRING "\
Usage: alarm [-d] [-l] [-h] [-v] time\n\
Set the alarm power on time or date of a Toshiba laptop\n\n\
  -d,--disable       disable the alarm power on facility\n\
  -l,--localtime     hardware clock is set to localtime\n\
  -h,--help          display this help message\n\
  -v,--version       display version\n\
Report bugs to jonathan@buzzard.org.uk\n"



enum {
	ALARM_TIME  = 0x0001,
	ALARM_DATE  = 0x0002,
	ALARM_EVERY = 0x0004,
	ALARM_YEAR  = 0x0008
};


/*
 * check the alarm features of the laptop
 */
int AlarmSupport(void)
{
	SMMRegisters reg;
	int support;

	support = 0x0000;
	SciOpenInterface();

	reg.ebx = SCI_ALARM_TIME;
	if (SciGet(&reg)==SCI_SUCCESS) {
		support |= ALARM_TIME;
	}

#define SCI_ALARM_EVERYDAY 0x0001

	reg.ebx = SCI_ALARM_DATE;
	if (SciGet(&reg)==SCI_SUCCESS) {
		reg.edx &= 0xffff;
		support |= ALARM_DATE;
		if ((reg.edx & SCI_ALARM_EVERYDAY)==0x0001)
			support |= ALARM_EVERY;
		if (SCI_YEAR(reg.edx)!=1990)
			support |= ALARM_YEAR;
	}

	SciCloseInterface();
	
	return support;
}


/*
 * Is time one before time two?
 */
int Behind(struct tm *one, struct tm *two)
{	
	if ((one->tm_mon<=two->tm_mon) && (one->tm_mday<=two->tm_mday)
		&& (one->tm_hour<=two->tm_hour) && (one->tm_sec<two->tm_sec)) {

		return 1;
	}

	return 0;
}



int main(int argc, char *argv[])
{
	SMMRegisters reg;
	time_t timer;
	int i,version,local,disable,support;
	char buffer[64];
	struct tm *alarm;
	struct tm current;


	/* do some quick checks on the laptop */

	if (SciSupportCheck(&version)==1) {
		fprintf(stderr, "alarm: this computer is not supported\n");
		return 1;
	}

	/* check this model supports alarm power on */

	support = AlarmSupport();
	if (support==0) {
		fprintf(stderr, "alarm: power on not supported on "
			"this model.\n");
		return 1;
	}

	/* process command line arguments */

	local = 0;
	disable = 0;
	for (i=1;i<argc;i++) {
		if ((!strcmp(argv[i], "-h")) || (!strcmp(argv[i], "--help"))) {
			printf(USAGE_STRING);
			exit(0);
		}
		if ((!strcmp(argv[i], "-v")) || (!strcmp(argv[i], "--version"))) {
			printf(VERSION_STRING);
			exit(0);
		}
		if ((!strcmp(argv[i], "-l")) || (!strcmp(argv[i], "--localtime"))) {
			local = 1;
		}		
		if ((!strcmp(argv[i], "-d")) || (!strcmp(argv[i], "--disable"))) {
			disable = 1;
		}		
	}

	/* disable the alarm power on option if requested and exit */

	if (disable==1) {
		SciOpenInterface();
		reg.ebx = SCI_ALARM_TIME;
		reg.ecx = SCI_ALARM_DISABLED;
		SciSet(&reg);
		SciCloseInterface();
		printf("alarm: alarm power on disabled\n");
		return 0;
	}

	/* parse the time string under */

	if (local==1)
		argv++;
	optind = 1;
	if (argc>=optind+1) {
		timer = parsetime(i-optind, argv+optind);
	} else {
		timer = 0;
	}

	/* exit if the time was garbled or is an unrecognised option */

	if ((timer==0) && (argc>optind)) {
		fprintf(stderr, "alarm: garbled time or unrecognised option.\n");
		exit(1);
	}

	/* convert the calander time into UTC or localtime as necessary */

	if (local==1)
		alarm = localtime(&timer);
	else
		alarm = gmtime(&timer);

	/* set the alarm time */

	if (((support & ALARM_TIME)!=0) && (timer!=0)) {
		SciOpenInterface();
		reg.ebx = SCI_ALARM_TIME;
		reg.ecx = SCI_ALARM_ENABLED | SCI_TIME(alarm->tm_hour,
			alarm->tm_min);
		SciSet(&reg);
		SciCloseInterface();
	}

	/* set the alarm date if supported */

	if (((support & ALARM_DATE)!=0) && (timer!=0)) {
		SciOpenInterface();
		reg.ebx = SCI_ALARM_DATE;
		if ((support & ALARM_YEAR)!=0)
			reg.ecx = SCI_FULLDATE((alarm->tm_year-90),
				(1+alarm->tm_mon), alarm->tm_mday);
		else
			reg.ecx = SCI_DATE((1+alarm->tm_mon), alarm->tm_mday);

		SciSet(&reg);
		SciCloseInterface();
	}

	/* save the current time and date */

	time(&timer);
	if (local==1)
		alarm = localtime(&timer);
	else
		alarm = gmtime(&timer);

	current.tm_year = alarm->tm_year;
	current.tm_mon = alarm->tm_mon;
	current.tm_mday = alarm->tm_mday;
	current.tm_hour = alarm->tm_hour;
	current.tm_min = alarm->tm_min;

	/* display the current alarm power on setting */

	alarm->tm_sec = 0;
	SciOpenInterface();
	reg.ebx = SCI_ALARM_TIME;
	if (SciGet(&reg)==SCI_SUCCESS) {
		if (SCI_TIME_ON(reg.ecx)==SCI_ALARM_ENABLED) {
			alarm->tm_min = SCI_MINUTE(reg.ecx);
			alarm->tm_hour = SCI_HOUR(reg.ecx);
		} else {
			printf("alarm power on disabled\n");
			SciCloseInterface();
			return 0;
		}
	}

	reg.ebx = SCI_ALARM_DATE;
	if (SciGet(&reg)==SCI_SUCCESS) {
		alarm->tm_mday = SCI_DAY(reg.ecx);
		alarm->tm_mon = (SCI_MONTH(reg.ecx))-1;
		alarm->tm_year = SCI_YEAR(reg.ecx);
		if ((reg.ecx & SCI_ALARM_EVERYDAY)==0x0001)
			strftime(buffer, 64, "%R everyday", alarm);
		if (alarm->tm_year==1990) {
			if (Behind(alarm, &current)==1)
				alarm->tm_year = 1+current.tm_year;
			else
				alarm->tm_year = current.tm_year;
			mktime(alarm);
			strftime(buffer, 64, "%a %b %e %R %Y", alarm);
		} else {
			alarm->tm_year -= 1900;
			mktime(alarm);
			strftime(buffer, 64, "%a %b %e %R %Y", alarm);
		}
	} else {
		strftime(buffer, 64, "%R", alarm);
	}

	SciCloseInterface();

	printf("alarm power on set to %s\n", buffer);

	return 0;
}
