#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <sys/stat.h>

#include "arena.h"
#include "arena_release.h"
#include "auth_arena.h"
#include "bridge.h"
#include "bridge_i.h"
#ifdef BOOKMARKS
#  include "bookmark.h"
#endif
#ifdef ARENA_DEBUG
#  include "debug.h"
# else
#  include "error.h"
#endif
#include "defs.h"
#include "dialog_arena.h"
#include "display.h"
#include "types.h"
#include "history.h"
#include "html.h"
#include "image.h"
#include "mailcap.h"
#include "mailto.h"
#include "main.h"
#ifdef PNG
#  include "png_arena.h"
#endif
#include "scrollbar.h"
#include "status.h"
#include "style.h"
#include "util.h"
#include "x11.h"

/* libwww includes */
#include "WWWLib.h"
#include "WWWMIME.h"
#include "WWWCache.h"
#include "WWWApp.h"
#include "WWWTrans.h"
#include "WWWInit.h"
#ifdef ARENA_DIRECT_WAIS
#  include "WWWWAIS.h"
#endif


HTAtom* text_atom;
HTAtom* html_atom;
HTAtom* html3_atom;
HTAtom* html_level3_atom;
HTAtom* gif_atom;
HTAtom* jpeg_atom;
HTAtom* png_atom;
HTAtom* png_exp_atom;
HTAtom* xpm_atom;
HTAtom* xbm_atom;
HTAtom* unknown_atom;	/* QingLong.13-02-97 */
HTAtom* css_atom;	/* QingLong.13-02-97 */
HTAtom* css1_atom;	/* QingLong.10-02-97 */

Bool AbortFlag = False;

static int ArenaXsocket = -1;

static HTList* Arena_Requests = NULL;
static HTList* Arena_RequestLeaks = NULL;
static HTList* Arena_PendingAnchors = NULL;
static HTList* Arena_Documents = NULL;
static HTList* Arena_DocumentLeaks = NULL;

static HTErrorMessage HTErrors[HTERR_ELEMENTS] = {
    { 100, "Continue", 					"information" },
    { 101, "Switching Protocols",			"information" },
    { 200, "OK", 					"success" },
    { 201, "Created", 					"success" },
    { 202, "Accepted", 					"success" },
    { 203, "Non-authoritative Information",		"success" },
    { 204, "No Content",				"success" },
    { 205, "Reset Content",				"success" },
    { 206, "Partial Content",				"success" },
    { 207, "Partial Update OK",				"success" },
    { 300, "Multiple Choices",				"redirection" },
    { 301, "Moved Permanently",				"redirection" },
    { 302, "Found",  			                "redirection" },
    { 303, "See Other",					"redirection" },
    { 304, "Not Modified",       			"redirection" },
    { 305, "Use Proxy",					"redirection" },
    { 306, "Proxy Redirect",				"redirection" },
    { 307, "Temporary Redirect",			"redirection" },
    { 400, "Bad Request", 				"client_error" },
    { 401, "Unauthorized",				"client_error" },
    { 402, "Payment Required", 				"client_error" },
    { 403, "Forbidden", 				"client_error" },
    { 404, "Not Found",		       			"client_error" },
    { 405, "Method Not Allowed",	 		"client_error" },
    { 406, "Not Acceptable",		 		"client_error" },
    { 407, "Proxy Authentication Required", 		"client_error" },
    { 408, "Request Timeout",		 		"client_error" },
    { 409, "Conflict",			 		"client_error" },
    { 410, "Gone",			 		"client_error" },
    { 411, "Length Required",		 		"client_error" },
    { 412, "Precondition Failed",	 		"client_error" },
    { 413, "Request Entity Too Large",	 		"client_error" },
    { 414, "Request-URI Too Large",	 		"client_error" },
    { 415, "Unsupported Media Type",	 		"client_error" },
    { 416, "Range Not Satisfiable",	 		"client_error" },
    { 417, "Expectation Failed",	 		"client_error" },
    { 418, "Reauthentication Required",	 		"client_error" },
    { 419, "Proxy Reauthentication Reuired", 		"client_error" },
    { 500, "Internal Server Error",			"server_error" },
    { 501, "Not Implemented", 				"server_error" },
    { 502, "Bad Gateway", 				"server_error" },
    { 503, "Service Unavailable",			"server_error" },
    { 504, "Gateway Timeout", 				"server_error" },
    { 505, "HTTP Version not supported",		"server_error" },
    { 506, "Partial update Not Implemented",		"server_error" },

    /* Cache Warnings */ 
    { 10,  "Response is Stale",				"cache" },
    { 11,  "Revalidation Failed",			"cache" },
    { 12,  "Disconnected Opeartion",			"cache" },
    { 13,  "Heuristic Expiration",			"cache" },
    { 14,  "Transformation Applied",			"cache" },
    { 99,  "Cache warning", 				"cache" },

    /* Non-HTTP Error codes and warnings */
    { 0,   "Can't locate remote host", 			"internal" },
    { 0,   "No host name found", 			"internal" },
    { 0,   "No file name found or file not accessible", "internal" },
    { 0,   "FTP server replies", 			"internal" },
    { 0,   "FTP server doesn't reply", 			"internal" },
    { 0,   "FTP login failure", 			"internal" },
    { 0,   "Server timed out", 				"internal" },
    { 0,   "Gopher-server replies", 			"internal" },
    { 0,   "Data transfer interrupted", 		"internal" },
    { 0,   "Connection establishment interrupted", 	"internal" },
    { 0,   "CSO-server replies", 			"internal" },
    { 0,   "This is probably a HTTP server 0.9 or less","internal" },
    { 0,   "Bad, Incomplete, or Unknown Response",	"internal" },
    { 0,   "Unknown access authentication scheme",	"internal" },
    { 0,   "News-server replies",			"internal" },
    { 0,   "Trying `ftp://' instead of `file://'",	"internal" },
    { 0,   "Too many redirections",			"internal" },
    { 0,   "Method not suited for automatic redirection","internal" },
    { 0,   "Premature End Of File",			"internal" },
    { 0,   "Response from WAIS Server too Large - Extra lines ignored","internal"},
    { 0,   "WAIS-server doesn't return any data", 	"internal" },
    { 0,   "Can't connect to WAIS-server",		"internal" },
    { 0,   "System replies",				"internal" },
    { 0,   "Wrong or unknown access scheme",		"internal" },
    { 0,   "Access scheme not allowed in this context",	"internal" },
    { 0,   "When you are connected, you can log in",	"internal" },
    { 0,   "This version has expired and will be automatically reloaded", "internal" },
    { 0,   "Loading new rules must be explicitly acknowledged", "internal" },
    { 0,   "Automatic proxy redirection must be explicitly acknowledged", "internal" }
};

/* ------------------------------------------------------------------------- */
/*			libwww progres notifications and dialogs             */
/* ------------------------------------------------------------------------- */

/*
 * Proggress indicator callback function.
 */
BOOL AHTProgress(HTRequest* request, HTAlertOpcode op,
		 int msgnum, CONST char* dfault, void* input,
		 HTAlertPar* reply)
{
 switch (op)
   {
    case HT_PROG_DNS:
      Announce(_("Looking up host %s"), input ? (char*)input : "");
      break;

    case HT_PROG_CONNECT:
      Announce(_("Contacting host %s"), input ? (char*)input : "");
      break;

    case HT_PROG_ACCEPT:
      Announce(_("Waiting for connection..."));
      break;

    case HT_PROG_READ:
      {
       long cl = HTAnchor_length(HTRequest_anchor(request));
       if (cl > 0)
         {
          long b_read = HTRequest_bodyRead(request);
          double pro = (double) b_read/cl*100;
          char buf[10];
          HTNumToStr((unsigned long) cl, buf, 10);
	  Announce("%s (%d%% of %s)", _("Read"), (int) pro, buf);
         }
        else
         Announce(_("Reading..."));
      }
      break;

    case HT_PROG_WRITE:
      if (HTRequest_isPostWeb(request))
        {
         HTParentAnchor *anchor = HTRequest_anchor(HTRequest_source(request));
         long cl = HTAnchor_length(anchor);
         if (cl > 0)
           {
            long b_write = HTRequest_bytesWritten(request);
            double pro = (double) b_write/cl*100;
            char buf[10];
            HTNumToStr( (unsigned long)cl, buf, 10);
            Announce("%s (%d%% of %s)", _("Written"), (int) pro, buf);
           }
          else
           Announce(_("Writing..."));
        }
      break;

    case HT_PROG_DONE:
      {
       char* url = HTAnchor_address((HTAnchor*)HTRequest_parent(request));

       Announce(url ? url : _("Finished"));
       Free(url);
      }
      break;

    case HT_PROG_WAIT:
      Announce(_("Waiting for free socket..."));
      break;

    case HT_PROG_GC:
      Announce(_("Collecting cache garbage..."));
      break;

    default:
      Announce(_("UNKNOWN PROGRESS STATE"));
      break;
   }
 return YES;
}


/* Default function that creates an error message using HTAlert() to
 * put out the contents of the error_stack messages.
 */
static BOOL ArenaShowMessage (HTRequest* request, HTAlertOpcode op,
			       int msgnum, const char * dfault, void * input,
			       HTAlertPar* reply)
{
    HTList *cur = (HTList *) input;
    HTError *pres;
    HTErrorShow showmask = HT_ERR_SHOW_FIRST; /* HTError_show(); */
    HTChunk *msg = NULL;
    int code;
    if (!request || !cur) return NO;
    while ((pres = (HTError *) HTList_nextObject(cur))) {
	int index = HTError_index(pres);
	if (HTError_doShow(pres)) {
	    if (!msg) {
		HTSeverity severity = HTError_severity(pres);
		msg = HTChunk_new(128);
		if (severity == ERR_WARN)
		    HTChunk_puts(msg, "Warning: ");
		else if (severity == ERR_NON_FATAL)
		    HTChunk_puts(msg, "Non Fatal Error: ");
		else if (severity == ERR_FATAL)
		    HTChunk_puts(msg, "Fatal Error: ");
		else if (severity == ERR_INFO)
		    HTChunk_puts(msg, "Information: ");
		else {
		    if (WWWTRACE)
			HTTrace("HTError..... Unknown Classification of Error (%d)...\n", severity);
		    HTChunk_delete(msg);
		    return NO;
		}

		/* Error number */
		if ((code = HTErrors[index].code) > 0) {
		    char buf[10];
		    sprintf(buf, "%d ", code);
		    HTChunk_puts(msg, buf);
		}
	    } else
		HTChunk_puts(msg, "\nReason: ");
	    HTChunk_puts(msg, HTErrors[index].msg);	    /* Error message */

	    if (showmask & HT_ERR_SHOW_PARS) {		 /* Error parameters */
		int length;
		int cnt;		
		char *pars = (char *) HTError_parameter(pres, &length);
		if (length && pars) {
		    HTChunk_puts(msg, " (");
		    for (cnt=0; cnt<length; cnt++) {
			char ch = *(pars+cnt);
			if (ch < 0x20 || ch >= 0x7F)
			    HTChunk_putc(msg, '#');
			else
			    HTChunk_putc(msg, ch);
		    }
		    HTChunk_puts(msg, ") ");
		}
	    }

	    if (showmask & HT_ERR_SHOW_LOCATION) {	   /* Error Location */
		HTChunk_puts(msg, "This occured in ");
		HTChunk_puts(msg, HTError_location(pres));
		HTChunk_putc(msg, '\n');
	    }

	    /*
	    ** Make sure that we don't get this error more than once even
	    ** if we are keeping the error stack from one request to another
	    */
	    HTError_setIgnore(pres);
	    
	    /* If we only are show the most recent entry then break here */
	    if (showmask & HT_ERR_SHOW_FIRST)
		break;
	}
    }
    if (msg) {
	HTChunk_putc(msg, '\n');
	Announce("%s\n", HTChunk_data(msg));
	HTChunk_delete(msg);
    }
    return YES;
}

/*
 * Check if the doc is image.
 */
Bool DocImage(Doc* d)
{
 if (d)
   if (d->anchor)
     {
      HTAtom* a = HTAnchor_format(d->anchor);


      if (a)
	{
	 return (
		 (a == gif_atom) ||
#ifdef JPEG
		 (a == jpeg_atom) ||
#endif
#ifdef PNG
		 (a == png_atom) ||
		 (a == png_exp_atom) ||
#endif
		 (a == xpm_atom) ||
		 (a == xbm_atom)
		);
	}
     }

 return False;
}


/*
 * Find out whether the document is raw text.
 */
Bool DocRawText(Doc* d)
{
 if (d)
   if (d->anchor)
     {
      HTAtom* a = HTAnchor_format(d->anchor);


      if ((d->show_raw) || (a == text_atom))
	return True;
       else
	return False;
     }

 return False;
}


/*
 * Is the document of HTML type?
 */
Bool DocHTML(Doc* d)
{
 HTAtom* a;

 if (!d || !d->anchor) return True;	/* this being the default type */

 a = HTAnchor_format(d->anchor);

 return((a == html3_atom) ||
	(a == html_level3_atom) ||
	(a == html_atom));
}


/*
 * Is the document a CSS style
 */
Bool DocCSS(Doc* d)
{
 HTAtom* a;

 if (!d || !d->anchor) return True;	/* this being the default type */

 a = HTAnchor_format(d->anchor);

 return((a == css_atom) ||
	(a == css1_atom));
}


Bool ArenaBridgeInit(void)
{
 return True;
}


Bool ArenaBridgeQuit(void)
{
 Bool TotalSuccess = True, theSuccess;

 /*
  * Free all documents.
  */
 theSuccess = Arena_deleteDocuments();
 TotalSuccess = (TotalSuccess && theSuccess);

 /*
  * Delete the list of leaked documents.
  */
 theSuccess = Arena_deleteDocumentLeaks();
 TotalSuccess = (TotalSuccess && theSuccess);

 theSuccess = HTList_delete(Arena_PendingAnchors);
 TotalSuccess = (TotalSuccess && theSuccess);

#if 0
 theSuccess = HTList_delete(Arena_Requests);
 TotalSuccess = (TotalSuccess && theSuccess);
#endif

 return TotalSuccess;
}


/*
 * A small simple wrapper. Adds only non-null objects.
 */
Bool Arena_HTList_addObject(HTList* theList, void* theObject)
{
 if (theList)
   if (theObject)
     return HTList_appendObject(theList, theObject);

 return NO;
}


/*
 * Find the object index on the objects list.
 */
int IndexObject_in_the_list(HTList* theList, void* theObject)
{
#ifdef ARENA_DEBUG
 char Iam[] = "IndexObject_in_the_list";
#endif


 if (theList)
   {
    if (theObject)
      {
       return HTList_indexOf(theList, theObject);
      }
     else
      {
#ifdef ARENA_DEBUG
       if (BRIDGE_TRACE) Arena_TracePrint(Iam, " NO object?\n");
#endif
       return -1;
      }
   }
  else
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! NO objects list.\n");
#endif
    return -1;
   }
}


/* Adds the object to the list pointed by list pointer.
 * Creates new list if necessary.
 */
Bool AddObject_to_the_list(HTList** theList_p, void* theObject)
{
#ifdef ARENA_DEBUG
 char Iam[] = "AddObject_to_the_list";
#endif


 if (theList_p == NULL)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE) Arena_TracePrint(Iam, " ERROR! No list pointer.\n");
#endif
    return False;
   }

 if (theObject == NULL)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE) Arena_TracePrint(Iam, " NO object?\n");
#endif
    return True;
   }

 if ((*theList_p) == NULL) (*theList_p) = HTList_new();

 if (IndexObject_in_the_list((*theList_p), (void*)theObject) >= 0)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam,
		       " the object "POINTER_FORMAT
		       " already on the list "POINTER_FORMAT".\n",
		       theObject, (*theList_p));
#endif
    return False;
   }

 return Arena_HTList_addObject((*theList_p), (void*)theObject);
}


/* Adds the object to the list pointed by list pointer.
 * Creates new list if necessary.
 */
Bool RemoveObject_from_the_list(HTList** theList_p, void* theObject)
{
#ifdef ARENA_DEBUG
 char Iam[] = "RemoveObject_to_the_list";
#endif


 if (theList_p == NULL)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE) Arena_TracePrint(Iam, " ERROR! No list pointer.\n");
#endif
    return False;
   }

 if ((*theList_p) == NULL)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE) Arena_TracePrint(Iam, " The list empty?\n");
#endif
    return False;
   }

 if (theObject == NULL)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE) Arena_TracePrint(Iam, " NO object?\n");
#endif
    return True;
   }

 if (IndexObject_in_the_list((*theList_p), (void*)theObject) >= 0)
   {
    return HTList_removeObject((*theList_p), (void*)theObject);
   }
  else
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam,
		       " the object "POINTER_FORMAT
		       " is NOT on the list "POINTER_FORMAT".\n",
		       theObject, (*theList_p));
#endif
    return False;
   }
}


/* Free all the objects on the list (with the aid for provided function)
 * and delete the list.
 */
Bool Arena_RemoveObjectList(HTList** theList_p,
			    Bool (*Free_the_Object_F)(void* obj, Bool force))
{
 Bool TotalSuccess = True;
#ifdef ARENA_DEBUG
 char Iam[] = "Arena_RemoveObjectList";
#endif


 if (theList_p == NULL)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE) Arena_TracePrint(Iam, " ERROR! No list pointer.\n");
#endif
    return False;
   }

 if ((*theList_p) == NULL)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE) Arena_TracePrint(Iam, " The list already empty?\n");
#endif
    return True;
   }

 if (Free_the_Object_F == NULL)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! Provided NO object freeing function.\n");
#endif
    return False;
   }

 {
  int i = HTList_count(*theList_p);

  if (i > 0)
    {
     Bool LocalSuccess;
     void* object_i = NULL;
     HTList* l = (*theList_p);

     HTList_nextObject(l);
     while (i--)
       {
	object_i = HTList_objectOf(l);
	HTList_nextObject(l);
	if (object_i)
	  {
	   LocalSuccess = (*Free_the_Object_F)(object_i, True);
#ifdef ARENA_DEBUG
	   if (BRIDGE_TRACE && VERBOSE_TRACE)
	     Arena_TracePrint(Iam,
			      " %ssuccessfull object "POINTER_FORMAT" free.\n",
			      (LocalSuccess ? "" : "un"), object_i);
#endif

	   if (LocalSuccess)
	     {
	      LocalSuccess = HTList_removeObject((*theList_p), object_i);
#ifdef ARENA_DEBUG
	      if (BRIDGE_TRACE && VERBOSE_TRACE)
		Arena_TracePrint(Iam,
				 " %ssuccessfull object "POINTER_FORMAT
				 " deletion from the list "POINTER_FORMAT".\n",
				 (LocalSuccess ? "" : "un"), object_i,
				 (*theList_p));
#endif
	     }

	   TotalSuccess = TotalSuccess && LocalSuccess;
	  }
       }
    }
#ifdef ARENA_DEBUG
   else
    {
     if (BRIDGE_TRACE && VERBOSE_TRACE)
       Arena_TracePrint(Iam,
			" NO objects on the list "POINTER_FORMAT"?\n",
			(*theList_p));
    }
#endif
 }

 if (TotalSuccess)
   {
    if (HTList_delete(*theList_p))
      {
       (*theList_p) = NULL;
       return True;
      }
   }

 return False;
}


Doc* DocNew(void)
{
 Doc* doc;
#ifdef ARENA_DEBUG
 char Iam[] = "DocNew";
#endif


 if ((doc = (Doc*)Arena_MAlloc(sizeof(Doc), False)))
   {
    doc->request           = NULL;
    doc->anchor            = NULL;
    doc->href              = NULL;
    doc->url               = NULL;
    doc->tag               = NULL;
    doc->base              = NULL;
    doc->title             = NULL;
    doc->link_style        = NULL;
    doc->head_style        = NULL;
    doc->content_buffer    = NULL;
    doc->image             = NULL;
    doc->style             = NULL;
    doc->loaded_length     = 0;
    doc->state             = DOC_NOTREGISTERED;
    doc->pending_reload    = False;
    doc->nofree            = False;
    doc->main_doc          = True;
    doc->user_style        = False;
    doc->host_documents    = NULL;
    doc->inline_documents  = NULL;
    doc->bad_flags         = NULL;
    doc->Fragment          = NULL;
    doc->FragmentCounter   = 0;
    doc->FragmentPtr       = NULL;
    doc->already_displayed = False;
    doc->show_raw          = False;
    doc->parsed_length     = 0;
    doc->paint_stream      = NULL;
    doc->paint_length      = 0;
    doc->width             = 0;
    doc->height            = 0;
    doc->edited_field      = NULL;
    doc->field_editor      = NULL;
    doc->style_stack       = NULL;
    doc->current_flat      = NULL;
    doc->current_sheet     = NULL;

    if (AddObject_to_the_list((&Arena_Documents), doc))
      {
#ifdef ARENA_DEBUG
       if (BRIDGE_TRACE)
	 Arena_TracePrint(Iam, " Created doc: "POINTER_FORMAT".\n", doc);
#endif
       return doc;
      }
     else
      {
#ifdef ARENA_DEBUG
       if (BRIDGE_TRACE)
	 Arena_TracePrint(Iam,
			  " Failed to add new doc "POINTER_FORMAT
			  " to the list "POINTER_FORMAT".\n");
#endif
       Free(doc);
      }
   }

 return NULL;
}


/*
 * Deletes bad_flags list of the document.
 */
Bool delete_Document_bad_flags(Doc* thedocument)
{
 BadFlag* theBadFlags;
#ifdef ARENA_DEBUG
 char Iam[] = "delete_Document_bad_flags";
#endif


 if (thedocument == NULL)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " NO document?\n");
#endif
    return True;
   }

 if (thedocument->bad_flags == NULL)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " NO bad flags list in the document?\n");
#endif
    return True;
   }

 {
  HTList* iHT = thedocument->bad_flags;

  while (iHT != NULL)
    {
     theBadFlags = (BadFlag*)(iHT->object);

     if (theBadFlags != NULL)
       {
	Free(theBadFlags->error_text);
	Free(theBadFlags);
       }

     iHT = iHT->next;
    }
 }

 if (HTList_delete(thedocument->bad_flags))
   {
    thedocument->bad_flags = NULL;
    return True;
   }
  else
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam,
		       " failed to delete document "POINTER_FORMAT
		       " bad flags list.\n",
		       thedocument);
#endif
    return False;
   }
}


/*
 * Routines to free document structures.
 */


/*
 * Free and delete the document with it's structure, return success.
 */
static Bool DeleteDocument(void* void_doc, Bool force)
{
 Doc* theDocument = (Doc*)void_doc;
 Bool TotalSuccess = True, localSuccess = True, theSuccess = True;
#ifdef ARENA_DEBUG
 char Iam[] = "DeleteDocument";
#endif


/* Normally, we don't want to free a document if its pending.
 * In other cases, the stat flag should be manipulated first.
 */

#ifdef ARENA_DEBUG	/* QingLong.14-02-97 */
 if (BRIDGE_TRACE)
   {
    if (theDocument)
      {
       if (theDocument->nofree)
	 {
	  Arena_TracePrint(Iam,
			   " refusing to free doc"
			   " as it has `nofree' flag set.\n"
			   "\t(\"%s\")\n",
			   theDocument->url);
	  localSuccess = False;
	 }
        else
	 {
	  if (theDocument->state == DOC_PENDING)
	    {
	     Arena_TracePrint(Iam,
			      " refusing to free pending doc."
			      "\n\t(\"%s\")\n",
			      theDocument->url);
	     localSuccess = False;
	    }
	   else
	    {
	     if (theDocument->url != NULL)	/* QingLong.12-12-96 */
	       Arena_TracePrint(Iam,
				" freeing doc "POINTER_FORMAT" (\"%s\").\n",
				theDocument, theDocument->url);
	      else
	       Arena_TracePrint(Iam,
				" freeing doc "POINTER_FORMAT
				" (doc->url == NULL).\n",
				theDocument);
	    }
	 }
      }
     else
      {
       Arena_TracePrint(Iam, " The doc is already empty (NULL).\n");
       return True;
      }
    }
  else
   {
    if ((theDocument == NULL) ||
	(theDocument->nofree) ||
	(theDocument->state == DOC_PENDING))
      localSuccess = False;
   }
# else
 if ((theDocument == NULL) ||
     (theDocument->nofree) ||
     (theDocument->state == DOC_PENDING))
   localSuccess = False;
#endif

 if (!(localSuccess || force)) return False;

 TotalSuccess = TotalSuccess && localSuccess;
 localSuccess = theSuccess = True;

 if (theDocument->main_doc)
   {
   /* For main doc we have to go through it's inlines,
    * remove ourselves from the list of hosts.
    */
    if (theDocument->inline_documents)
      {
       HTList* l = theDocument->inline_documents;
       Doc* d;

       HTList_nextObject(l);
       while ((d = HTList_objectOf(l)))
	 {
	  HTList_nextObject(l);
	  HTList_removeObjectAll(d->host_documents, theDocument);
	  HTList_removeObjectAll(theDocument->inline_documents, d);
	 }

       localSuccess = (HTList_count(theDocument->inline_documents) == 0);

       if (localSuccess || force)
	 {
	  theSuccess = HTList_delete(theDocument->inline_documents);
	  if (theSuccess || force) theDocument->inline_documents = NULL;
	  localSuccess = localSuccess && theSuccess;
	 }
      }
   }
  else
   {
   /* For inline documents we have to go through the list of host docs,
    * and proceed freeing the doc only if the hosts list becomes empty.
    */
    if (theDocument->host_documents)
      {
       HTList* l = theDocument->host_documents;
       Doc* d;

       HTList_nextObject(l);
       while ((d = HTList_objectOf(l)))
	 {
	  HTList_nextObject(l);
	  HTList_removeObjectAll(d->inline_documents, theDocument);
	  HTList_removeObjectAll(theDocument->host_documents, d);
	 }

       localSuccess = (HTList_count(theDocument->host_documents) == 0);
       if (localSuccess || force)
	 {
	  theSuccess = HTList_delete(theDocument->host_documents);
	  if (theSuccess || force) theDocument->host_documents = NULL;
	  localSuccess = localSuccess && theSuccess;
	 }
      }
   }
 /* End ``if (theDocument->main)'' */

 if (!(localSuccess || force)) return False;

 TotalSuccess = TotalSuccess && localSuccess;
 localSuccess = theSuccess = True;
 
 if (theDocument->anchor)
   {
    /* QingLong.06-03-98
     * Cut the anchor from the document structure.
     * BUT do NOT ``HTAnchor_delete(theDocument->anchor)'' ---
     * leave the anchor stuff for the libwww to take care of.
     */
    HTAnchor_setDocument(theDocument->anchor, NULL);
   }
 /* End ``if (theDocument->anchor)'' */


 /* Do not delete the document request (HTRequest_delete(doc->request)) ---
  * this appears to be the libwww prerogative.
  */
 /* problem: if we free this while the doc is still being fetched...
  *          check state...
  */
 if (theDocument->request)
   {
    AddRequest_to_RequestLeaks_List((void *)theDocument->request);
    HTRequest_setContext(theDocument->request, NULL);
   }


 /* glk 09.09.97
  * Free the document Named Fragment Anchor stuff!
  * I used to be doing _delete(), _free(), then =NULL
  * HTList_free() was releasing memory it should not have!
  * We were dying on ptr=malloc(size) with a SEGFault. (totally unrelated)!
  * USE HTList_free() with extreme caution!
  */
 theDocument->FragmentCounter = 0;
 Free(theDocument->Fragment);
 if (theDocument->FragmentPtr)
   {
    localSuccess = HTList_delete(theDocument->FragmentPtr);
    if (localSuccess || force)
      theDocument->FragmentPtr = NULL;
     else
      return False;

    TotalSuccess = TotalSuccess && localSuccess;
    localSuccess = theSuccess = True;
   }


#define ARENA_Document_Member_Free(X) Free((theDocument->X))

 if (!theDocument->nofree)
   {
#if 1	/* QingLong.24-10-96 */
   /*
    * Be aware! This is just a HACK! I don't understand exactly why
    * this is necessary, but without this check libc free() crashes.
    * Probably this is because HTLoadAnchor() (or some other libWWW function)
    * doesn't allocate memory for the contents of document beeing fetched
    * if that document turns out to be empty (of zero length),
    * and in this case `doc->content_buffer' turns out to be
    * statically allocated memory (or it has already been freed),
    * so it's incorrect to try to free it.
    * A document can be treated as empty if it hasn't been fetched
    * successfully (some error has occured during fetching it).
    * _Question_
    *	Can this method of ``memory freeing'' cause memory leakage?
    */
    if ((theDocument->content_buffer) && (theDocument->loaded_length == 0))
      {
#ifdef ARENA_DEBUG
       if (BRIDGE_TRACE)
	 Arena_TracePrint(Iam, " ERROR! Ancient workaround HACK!\n");
#endif
       theDocument->content_buffer = NULL;
      }
     else
      {
       ARENA_Document_Member_Free(content_buffer);
      }
    /* End if */
# else
    ARENA_Document_Member_Free(content_buffer);
#endif
   }
 /* End ``if (!theDocument->nofree)'' */

 ARENA_Document_Member_Free(tag);
 ARENA_Document_Member_Free(link_style);
 ARENA_Document_Member_Free(head_style);
 ARENA_Document_Member_Free(base);

 if (theDocument->style)
   {
    localSuccess = DelStyleSheet(theDocument->style);
    if (localSuccess || force)
      theDocument->style = NULL;
     else
      return False;

    TotalSuccess = TotalSuccess && localSuccess;
    localSuccess = theSuccess = True;
   }

 ARENA_Document_Member_Free(title);
 ARENA_Document_Member_Free(href);
 ARENA_Document_Member_Free(url);

 /*
  * We must remove all pixmaps hidden in the paint stream before freeing it.
  */
 if (theDocument->image)
   {
    localSuccess = DelImage(theDocument->image);
    if (localSuccess || force)
      theDocument->image = NULL;
     else
      return False;

    TotalSuccess = TotalSuccess && localSuccess;
    localSuccess = theSuccess = True;
   }

 /* Memory Leakage !!!? */
 /* Leaving `theDocument->edited_field' as is! */

 ARENA_Document_Member_Free(field_editor);

 ARENA_Document_Member_Free(paint_stream);

 if (theDocument->bad_flags)
   {
    localSuccess = delete_Document_bad_flags(theDocument);
    if (localSuccess || force)
      theDocument->bad_flags = NULL;
     else
      return False;

    TotalSuccess = TotalSuccess && localSuccess;
    localSuccess = theSuccess = True;
   }

 if (TotalSuccess || force)
   {
   /* QingLong.06-02-97.
    * Seems like we must take care of CurrentDoc too...
    */
    if (theDocument == CurrentDoc) CurrentDoc = NULL;
    Free(theDocument);
   }

 return TotalSuccess;

#undef ARENA_Document_Member_Free
}


/*
 * Remove the document from the context documents list.
 */
static Bool RemoveDocument_from_DocumentList(Doc* theDocument)
{
#ifdef ARENA_DEBUG
 char Iam[] = "RemoveDocument_from_Document_List";
#endif

 if (theDocument)
   {
    Bool FreeingSuccess, RemovingSuccess;

    FreeingSuccess = DeleteDocument((void*)theDocument, False);
    RemovingSuccess = HTList_removeObject(Arena_Documents, (void*)theDocument);

    if (FreeingSuccess)
      {
       if (RemovingSuccess)
	 {
	 /* (doc->anchor has already been cut in ``DeleteDocument'') */
	  theDocument = NULL;
	 }
#ifdef ARENA_DEBUG
        else
	 {
	  if (BRIDGE_TRACE)
	    Arena_TracePrint(Iam,
			     " ERROR! Cannot remove document "POINTER_FORMAT
			     " from docs list.\n",
			     theDocument);
	 }
#endif
      }
     else
      {
#ifdef ARENA_DEBUG
       if (BRIDGE_TRACE)
	 Arena_TracePrint(Iam,
			  " Cannot free document (doc = "
			  POINTER_FORMAT")\n",
			  theDocument);
#endif
       AddDocument_to_DocumentLeaks_List(theDocument);
      }

    return (FreeingSuccess && RemovingSuccess);
   }
  else
   {
    return True;
   }
}


/*
 * This is just the same but the name --- for external usage.
 */
Bool DocDel(Doc* theDoc)
{
 return RemoveDocument_from_DocumentList(theDoc);
}


/* Sort documents list, placing main documents prior to inlines.
 * Also applicable to documents leaks list.
 */
static Bool Arena_SortDocumentsList(HTList* theDocList)
{
 if (theDocList)
   {
    int i;
    void* d;
    HTList* l;
    Bool TotalSuccess = True, theSuccess;

    l = theDocList;
    i = HTList_count(l);
    HTList_nextObject(l);
    while (i--)
      {
       d = HTList_objectOf(l);
       HTList_nextObject(l);
       if (d)
	 {
	  if (!(((Doc*)d)->main_doc))
	    {
	     theSuccess = HTList_removeObject(theDocList, d);
	     TotalSuccess = TotalSuccess && theSuccess;
	     theSuccess = HTList_appendObject(theDocList, d);
	     TotalSuccess = TotalSuccess && theSuccess;
	    }
	 }
      }

    return TotalSuccess;
   }
  else
   {
    return False;
   }
}


/*
 * Free all documents.
 */
static Bool Arena_deleteDocuments(void)
{
 /* It's worth sorting the docuemnts list before freeing */
 Arena_SortDocumentsList(Arena_Documents);
 return Arena_RemoveObjectList((&Arena_Documents), DeleteDocument);
}


/* Adds the document (with the appropriate structure members)
 * to the context document memory leaks list.
 */
static Bool AddDocument_to_DocumentLeaks_List(Doc* theDocument)
{
 /* It's worth sorting the docuemnts list before freeing */
 Arena_SortDocumentsList(Arena_DocumentLeaks);
 return AddObject_to_the_list((&Arena_DocumentLeaks), theDocument);
}


/*
 * Remove the document from the context documents leaks list.
 */
static Bool RemoveDocument_from_DocumentLeaks_List(Doc* theDocument)
{
 Bool FreeingSuccess, RemovingSuccess;

  FreeingSuccess = DeleteDocument((void*)theDocument, True);
 RemovingSuccess = HTList_removeObject(Arena_DocumentLeaks,
				       (void*)theDocument);
 return (FreeingSuccess && RemovingSuccess);
}


/*
 * Delete all the leaked documents in the context.
 */
static Bool Arena_deleteDocumentLeaks(void)
{
 return Arena_RemoveObjectList((&Arena_DocumentLeaks), DeleteDocument);
}


/*
 * Add the request to the context requests leaks list.
 */
static Bool AddRequest_to_RequestLeaks_List(HTRequest* theRequest)
{
 return AddObject_to_the_list((&Arena_RequestLeaks), theRequest);
}


/*
 * Kill the request, free it's structure and delete it.
 */
static Bool DeleteRequest(void* void_req, Bool force)
{
 Bool KillingSuccess;
 HTRequest* theRequest = (HTRequest*)void_req;
#ifdef ARENA_DEBUG
 char Iam[] = "DeleteRequest";
#endif

 KillingSuccess = HTRequest_kill(theRequest);

#ifdef ARENA_DEBUG
 if (BRIDGE_TRACE)
   if (!KillingSuccess)
     {
      Arena_TracePrint(Iam, 
		       " Failed to kill the request "POINTER_FORMAT".\n",
		       theRequest);
     }
#endif

 if (KillingSuccess || force)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam,
		       " deleting the request "POINTER_FORMAT".\n",
		       theRequest);
#endif
    HTRequest_delete(theRequest);
   }

 return KillingSuccess;
}


/*
 * Remove the request from the context requests leaks list.
 */
static Bool RemoveRequest_from_RequestLeaks_List(HTRequest* theRequest)
{
 Bool FreeingSuccess, RemovingSuccess;
#ifdef ARENA_DEBUG
 char Iam[] = "RemoveRequest_from_RequestLeaks_List";
#endif


 FreeingSuccess = DeleteRequest((void*)theRequest, True);

#ifdef ARENA_DEBUG
 if (BRIDGE_TRACE)
   if (!FreeingSuccess)
     {
      Arena_TracePrint(Iam, 
		       " Failed to free the request "POINTER_FORMAT
		       " cleanly.\n",
		       theRequest);
     }

 Arena_TracePrint(Iam,
		  " removing the request "POINTER_FORMAT".\n",
		  theRequest);
#endif

 RemovingSuccess = HTList_removeObject(Arena_RequestLeaks, (void*)theRequest);

#ifdef ARENA_DEBUG
 if (BRIDGE_TRACE)
   if (!RemovingSuccess)
     {
      Arena_TracePrint(Iam, 
		       " Failed to remove the request "POINTER_FORMAT
		       " cleanly.\n",
		       theRequest);
     }
#endif

 return (FreeingSuccess && RemovingSuccess);
}


/*
 * Free all leaked requests and delete them.
 */
static Bool Arena_deleteRequestLeaks(void)
{
 return Arena_RemoveObjectList((&Arena_RequestLeaks), DeleteRequest);
}


/* HTCheckPresentation
 * is used to decide if a content type should be registered or not.
 * If the content type is not registered before,
 * return True to have it registered.
 * If the content type has been registered, check the associated quality.
 */
Bool HTCheckPresentation(HTList* conversions, char* content_type,
                         double quality)
{
 HTList* l = conversions;
 HTPresentation* p;
#ifdef ARENA_DEBUG
 char Iam[] = "HTCheckPresentation";
#endif


 while ((p = (HTPresentation *)HTList_nextObject(l)) != NULL)
   {
    if (strcmp(content_type, p->rep->name) == 0)
      {
       if (p->quality < quality)
         return True;
        else
         {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
          if (MAILCAP_TRACE)
	    Arena_TracePrint(Iam,
			     " returning False for \"%s\"\n", content_type);
#endif
          return False;
         }
      }
   }
 return True;
}




/* ------------------------------------------------------------------------- */
/*				REQUEST FUNCTIONS			     */
/* ------------------------------------------------------------------------- */


/*
 * This fuction sets appropriate conversions list for the request
 * of the given Content-Type.
 * Sets output stream (from Net to application) for the request.
 * Returns True if conversion list set is a common one (for "www/unknown").
 */
Bool SetRequestConversionsList(HTRequest* theRequest, HTFormat theContentType)
{
 Bool SetCommonConversions = True;
 Bool override = False;   /* Only damned W3C knows what it's intended for */
 HTFormat theOutputFormat = WWW_PRESENT;


 if (theContentType == NULL ||
     theContentType == unknown_atom)
   {
    SetCommonConversions = True;
   }
  else
   {
    if (theContentType == css_atom ||
	theContentType == css1_atom)
      {
       theOutputFormat = theContentType;
       override = False;

       SetCommonConversions = True;
      }
     else
      {
       SetCommonConversions = True;
      }
    /* End ``if Content-Type is CSS'' */
   }
 /* End ``if Content-Type is smth common'' */


 if (SetCommonConversions)
   {
    HTList *tmp_conversions;

    override = False;

    tmp_conversions = HTRequest_conversion(theRequest);
    HTRequest_setConversion(theRequest, context->conversions, override);

    if (tmp_conversions) HTList_delete(tmp_conversions);
   }
 /* End ``if (SetCommonConversions)'' */

 HTRequest_setOutputStream(theRequest,
			   HTXParse(theRequest, NULL,
				    NULL, theOutputFormat, NULL));
 HTRequest_setOutputFormat(theRequest, theOutputFormat);

 return SetCommonConversions;
}


/* This function creates a new request structure (adapted for the anchor)
 *           and adds it to the global list of active threads.
 */
HTRequest* ReqNew(HTParentAnchor* theAnchor)
{
 HTRequest *newreq = HTRequest_new(); 	     /* Set up a new request */
#ifdef ARENA_DEBUG
 char Iam[] = "ReqNew";
#endif


 HTFormat theContentType = HTAnchor_format(theAnchor);


#ifdef ARENA_USE_BLOCKING_SOCKETS
 HTRequest_setPreemptive(newreq, YES);
# else
 HTRequest_setPreemptive(newreq, NO);
#endif
 HTRequest_setUserProfile(newreq, UserProfile);
 HTRequest_addRqHd(newreq, HT_C_HOST);
 HTRequest_addRqHd(newreq, HT_C_REFERER);
 HTRequest_setGnHd(newreq, HTRequest_gnHd(newreq) & ~HT_G_CONNECTION);

 SetRequestConversionsList(newreq, theContentType);

 if (AddObject_to_the_list((&Arena_Requests), newreq))
   {
    return newreq;
   }
  else
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam,
		       " Failed to incorporate new request! Giving it up.\n");
#endif
    HTRequest_delete(newreq);
    return NULL;
   }
}


/* This function deletes a request structure
 * and takes it out of the list of active threads.
 */
Bool RemoveRequest_from_RequestList(HTRequest* theRequest)
{
 Bool FreeingSuccess, RemovingSuccess;

  FreeingSuccess = DeleteRequest((void*)theRequest, True);
 RemovingSuccess = HTList_removeObject(Arena_Requests, theRequest);
 return (FreeingSuccess && RemovingSuccess);
}


/*
 * This function deletes the whole list of active threads.
 */
Bool Arena_deleteRequests(void)
{
 return Arena_RemoveObjectList((&Arena_Requests), DeleteRequest);
}


int EventHandler(SOCKET s, void* param, HTEventType theType)
{
#ifdef ARENA_DEBUG
 char Iam[] = "EventHandler";
#endif


 if ((theType != HTEvent_TIMEOUT) && (theType != HTEvent_READ))
   {
#ifdef ARENA_DEBUG	/* QingLong.26-03-97 */
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam, " Unexpected HTEventType!\n");
#endif
    return HT_INTERNAL;
   }


 if (s == ArenaXsocket)
   {
    if (theType == HTEvent_TIMEOUT) return HT_OK;

    GuiEvents(0, NULL, 0);
    GuiEvents(0, NULL, 0);
    GuiEvents(0, NULL, 0);
    return HT_OK;
   }
  else
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      {
       HTRequest* req = ((HTEvent*)param)->request;
       HTParentAnchor* theanchor = HTRequest_anchor(req);

       if (req && theanchor)
	 {
	  char *theaddress = NULL;

	  theaddress = HTAnchor_address((HTAnchor*)theanchor);
	  Arena_TracePrint(Iam, " \"%s\"\n", theaddress);
	  Free(theaddress);   /* As `theaddress' has been malloc'd in
			      `HTAnchor_address()', it must be freed */
	 }
      }
#endif

    if (AbortFlag || (theType == HTEvent_TIMEOUT))
      {	/* we'll have to abort this session */
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
       if (BRIDGE_TRACE)
	 Arena_TracePrint(Iam, " aborting req\n");
#endif
       AbortFlag = False;
       HideBusy();
       DisplayUrl(CurrentDoc);

       HTNet_killAll();  /* interrupt everything !! */
       return HT_OK;
      }
     else
      {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
       if (BRIDGE_TRACE && VERBOSE_TRACE)
	 Arena_TracePrint(Iam, " EVENT_OK\n");
#endif
       return HT_OK;
      }
   }
}


/*
 * Arena own anchor request manager. Very simple. Just to avoid looping.
 */


/*
 * This function finds out if the anchor is already pending.
 */
Bool Arena_anchor_IS_pending(HTParentAnchor* theanchor)
{
#ifdef ARENA_DEBUG
 char Iam[] = "Arena_anchor_IS_pending";
#endif


 if (theanchor == NULL)
   {
#  ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam, " the anchor is NULL!\n");
#  endif
    return False;
   }

 if (IndexObject_in_the_list(Arena_PendingAnchors, (void*)theanchor) >= 0)
   {
#  ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " the anchor "POINTER_FORMAT" is pending.\n",
		       theanchor);
#  endif
    return True;
   }
  else
   {
#  ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " the anchor "POINTER_FORMAT" isn't pending.\n",
		       theanchor);
#  endif
    return False;
   }
}

/*
 * This function adds the anchor to the list of pending anchors.
 * Returns True if the anchor is added to the list and
 *        False if the anchor isn't added (e.g. anchor already is pending)
 */
Bool Arena_anchor_pending_start(HTParentAnchor* theanchor)
{
#ifdef ARENA_DEBUG
 char Iam[] = "Arena_anchor_pending_start";
#endif


 if (theanchor == NULL)
   {
#  ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam, " the anchor is NULL!\n");
#  endif
    return False;
   }

 if (Arena_anchor_IS_pending(theanchor))
   {
#  ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tthe anchor "POINTER_FORMAT" already pending.\n");
#  endif
    return False;
   }
  else
   {
#  ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tthe anchor "POINTER_FORMAT" isn't pending."
		       " Start.\n",
		       theanchor);
#  endif
    AddObject_to_the_list((&Arena_PendingAnchors), (void*)theanchor);
    return True;
   }
 /* End ``if (Arena_anchor_IS_pending)'' */
}


/*
 * This function removes the anchor from the list of pending anchors.
 * Returns True if the anchor has been removed to the list and
 *        False if the anchor hasn't been removed (e.g. anchor isn't pending)
 */
Bool Arena_anchor_pending_stop(HTParentAnchor* theanchor)
{
#ifdef ARENA_DEBUG
 char Iam[] = "Arena_anchor_pending_stop";
#endif


 if (theanchor == NULL)
   {
#  ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam, " the anchor is NULL!\n");
#  endif
    return False;
   }

 if (Arena_anchor_IS_pending(theanchor))
   {
#  ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tthe anchor "POINTER_FORMAT" is pending. Stop.\n",
		       theanchor);
#  endif
    RemoveObject_from_the_list((&Arena_PendingAnchors), (void*)theanchor);
    return True;
   }
  else
   {
#  ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tthe anchor "POINTER_FORMAT" isn't pending.\n",
		       theanchor);
#  endif
    return False;
   }
 /* End ``if (Arena_anchor_IS_pending)'' */
}


/*
 * Register the inline doc in the host doc inline docs list and vice versa.
 */
static Bool ArenaRegisterInlineDocument(Doc* host_doc, Doc* inline_doc)
{
#ifdef ARENA_DEBUG
 char Iam[] = "ArenaRegisterInlineDocument";
#endif


 if (inline_doc && host_doc)
   {
    if (inline_doc->state == DOC_NOTREGISTERED)
      {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
       if (BRIDGE_TRACE)
	 {
	  Arena_TracePrint(Iam,
			   " registering\n"
			   "\t    subdoc \"%s\"\n"
			   "\tin hostdoc \"%s\"\n"
			   "\t subdoc = "POINTER_FORMAT",\n"
			 "\thostdoc inlines list = "POINTER_FORMAT".\n",
			   inline_doc->url, host_doc->url,
			   inline_doc, host_doc->inline_documents);
	 }
#endif

       /* first, put the inline in the host_doc's list of inlines */
       if (host_doc->inline_documents == NULL)
	 {
	  host_doc->inline_documents = HTList_new();
#ifdef ARENA_DEBUG
	  if (BRIDGE_TRACE && VERBOSE_TRACE)
	    Arena_TracePrint(Iam,
			     "\n\tCreated inlines list "POINTER_FORMAT
			     " for host doc "POINTER_FORMAT".\n",
			     host_doc->inline_documents, host_doc);
#endif
	 }

       HTList_appendObject(host_doc->inline_documents, (void*)inline_doc);
    
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
       if (BRIDGE_TRACE)
	 {
	  Arena_TracePrint(Iam,
			   " registering\n"
			   "\t   hostdoc \"%s\"\n"
			   "\t in subdoc \"%s\"\n"
			   "\thostdoc = "POINTER_FORMAT",\n"
			   "\t subdoc hosts list = "POINTER_FORMAT".\n",
			   host_doc->url, inline_doc->url,
			   host_doc, inline_doc->host_documents);
	 }
#endif

       /* then, put the host_doc in the inline's list of host_docs */
       if (inline_doc->host_documents == NULL)
	 {
	  inline_doc->host_documents = HTList_new();
#ifdef ARENA_DEBUG
	  if (BRIDGE_TRACE && VERBOSE_TRACE)
	    Arena_TracePrint(Iam,
			     "\n\tCreated hosts list "POINTER_FORMAT
			     " for inline doc "POINTER_FORMAT".\n",
			     inline_doc->host_documents, inline_doc);
#endif
	 }

       HTList_appendObject(inline_doc->host_documents, (void*)host_doc);

       return True;
      }
     else
      {
#ifdef ARENA_DEBUG
       if (BRIDGE_TRACE)
	 Arena_TracePrint(Iam,
			  " The doc "POINTER_FORMAT" already registered"
			  " (state %d).\n",
			  inline_doc, inline_doc->state);
#endif
       return False;
      }
   }
  else
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE) Arena_TracePrint(Iam, " ERROR! NULL docs.\n");
#endif
    return False;
   }
}

/* Expand href taking in account basehref of the current doc.
 * The result string is allocated inside! You must free it after all!
 */
char* ExpandHref(char* href)
{
 char* exp_href;
#ifdef ARENA_DEBUG
 char Iam[] = "ExpandHref";
#endif


 if (href == NULL)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE) Arena_TracePrint(Iam, " ERROR! The href is NULL.\n");
#endif
    return NULL;
   }

 if(CurrentDoc && CurrentDoc->base)
   {
    exp_href = HTParse(href, CurrentDoc->base, PARSE_ALL);
   }
  else
   if (CurrentDoc && CurrentDoc->url)
     {
      exp_href = HTParse(href, CurrentDoc->url, PARSE_ALL);
     }
    else
     {
#if 1  /* janet 2/8/95 */
      char* relname = HTFindRelatedName(); 	       /*   added */

      exp_href = HTParse(href, relname, PARSE_ALL); /* changed */
      Free(relname);				       /*   added */
# else
      /* janet 2/8/95: HTFindRel.. is not freed when used as such: */
     exp_href = HTParse(href, (char *)HTFindRelatedName(), PARSE_ALL);
#endif
     }
   /* End if */
 /* End if */

#ifdef ARENA_DEBUG	/* QingLong.11-12-96 */
 if (BRIDGE_TRACE)
   Arena_TracePrint(Iam,
		    "\n\texpanding \"%s\"\n\t     into \"%s\"\n",
		    href, exp_href);
 /* End if */
#endif

 return exp_href; 
}


/* Find an anchor for the href after it's expansion.
 * Takes the href of the given hreflen.
 */
HTAnchor* anchorExpandnHref(char *href, size_t hreflen) /* wm 18.Jan.95 */
{
 char* exp_href;
 char* hrefdup;
 HTAnchor* anchor;
 HTChildAnchor*   child_anchor;
 HTParentAnchor* parent_anchor;
#ifdef ARENA_DEBUG
 char Iam[] = "anchorExpandnHref";
#endif


 if (href == NULL || hreflen <= 0)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam,
		       " ERROR! Weird arguments: "POINTER_FORMAT", %d.\n",
		       href, (int)hreflen);
#endif
    return NULL;
   }

 hrefdup = strndup(href, hreflen);
 exp_href = ExpandHref(hrefdup);

           /* We do have to free anchor->address !!! */ /*  ? */
 anchor = HTAnchor_findAddress(exp_href);

 Free(exp_href);

 child_anchor = (HTChildAnchor*)anchor;
 parent_anchor = HTAnchor_parent(anchor);


#ifdef ARENA_DEBUG	/* QingLong.13-12-96 */
 if (BRIDGE_TRACE && VERBOSE_TRACE)
   {
    if (anchor)
      {
       Arena_TracePrint(Iam,
			" \"%s\" ---\n"
			"\t--- the document structure (anchor) follows:\n",
			hrefdup);
       debug_PrintAnchor(Iam, anchor);
      }
     else
       Arena_TracePrint(Iam,
			" \"%s\" ---\n"
			"\t--- no document structure for the url!\n",
			hrefdup);
    /* End ``if (anchor)'' */
   }
 /* End if */
#endif

 Free(hrefdup);

 return anchor;
}


/*
 * Does the main set of proper request termination actions.
 */
int Internal_TerminateHandler(Doc* doc, void* param, int status,
			      HTFormat theDocumentFormat);


/*
 * pseudoLoadAnchor() "loads" the document from the memory,
 * i.e. just
 *          checks if it is already present in the history,
 *          registers (if necessary) in the history,
 *          initiates document parsing and displaying process.
 *
 * You should provide it a pointer to a buffer containing the document
 * and the length of the buffer (`content_buffer' and `loaded_length'),
 * and href/hreflen pair for the document.
 * Any other `Doc' structure parameters will be set by pseudoLoadAnchor.
 * It will allocate new Doc structure,
 * new content_buffer and copy the buffer to it's own container,
 * create HTParentAnchor structure and link it with the document.
 * `reload', `block', and `for_real' parameters would be used for children...
 */
Doc* pseudoLoadAnchor(char* thehref,
		      int   thehrefLength,
		      char* theBaseURL,
		      int   theBaseURLLength,
		      char* theDocContentsBuffer,
		      int   theDocContentsBufferLength,
		      HTFormat theDocumentFormat,
		      Bool main_doc, Bool nofree, Bool record,
		      Bool reload, Bool block, Bool forreal,
		      Doc* host_doc)
{
 Doc* doc = NULL;
 History* h;
 HTList* l;
 HTAnchor* theAnchor;
 HTParentAnchor* theParentAnchor;	/* auxiliary variable! */
#ifdef ARENA_DEBUG
 char Iam[] = "pseudoLoadAnchor";
#endif


 if (theDocContentsBufferLength <= 0)
   {
#ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The contents length <= 0!\n");
#endif
    return NULL;
   }

 if (theDocContentsBuffer == NULL)
   {
#ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The contents buffer is NULL!\n");
#endif
    return NULL;
   }

 if (thehrefLength <= 0)
   {
#ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The href length <=0!\n");
#endif
    return NULL;
   }

 if (thehref == NULL)
   {
#ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The href is NULL!\n");
#endif
    return NULL;
   }

 {
  char* theAbsolutehref;
  char* thehrefcopy = strndup(thehref, thehrefLength);

  if (theBaseURLLength > 0 && theBaseURL)
    {
     char* theBaseURLcopy = strndup(theBaseURL, theBaseURLLength);

     theAbsolutehref = HTParse(thehrefcopy, theBaseURLcopy, PARSE_ALL);
     Free(theBaseURLcopy);
     Free(thehrefcopy);
    }
   else
    theAbsolutehref = thehrefcopy;

  theAnchor = HTAnchor_findAddress(theAbsolutehref);
  theParentAnchor = HTAnchor_parent(theAnchor);
  Free(theAbsolutehref);
 }

 /* Check if the anchor has been found */
 if (theAnchor == NULL)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam,
		       " ERROR! Failed to find anchor for the href:\n\t%.*s\n",
		       thehrefLength, thehref);
#endif
    return NULL;
   }


#ifdef ARENA_DEBUG
  if (theDocumentFormat)
    {
     if (BRIDGE_TRACE && VERBOSE_TRACE)
       Arena_TracePrint(Iam,
			" the doc "POINTER_FORMAT" format is \"%s\".\n",
			doc, theDocumentFormat->name);
    }
   else
    {
     if (BRIDGE_TRACE)
       Arena_TracePrint(Iam,
			" the doc "POINTER_FORMAT" format is NULL!\n", doc);
    }
#endif


 if (Arena_anchor_IS_pending(theParentAnchor))
   {
#  ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " anchor to load ("POINTER_FORMAT") is pending.\n",
		       theParentAnchor);
#  endif
    return HTAnchor_document(theParentAnchor);
   }
  else
   Arena_anchor_pending_start(theParentAnchor);


#ifdef ARENA_DEBUG	/* QingLong.08-12-96 */
 {
  char *theaddress;

  theaddress = HTAnchor_address(theAnchor);
  Announce("%s %s", _("Fetching"), theaddress);
  Free(theaddress);
 }
#endif

 if (main_doc) ShowBusy();

#ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
 if (BRIDGE_TRACE && VERBOSE_TRACE)
   {
    HTFormat theFormat = HTAnchor_format(theParentAnchor);
    char* theContentType = HTAtom_name(theFormat);


    if (theContentType)
      Arena_TracePrint(Iam,
		       " Requested anchor ("POINTER_FORMAT")"
		       " Content-Type: \"%s\"\n",
		       theParentAnchor, theContentType);
     else
      Arena_TracePrint(Iam,
		       " Requested anchor ("POINTER_FORMAT")"
		       " Content-Type isn't set.\n",
		       theParentAnchor);
   }
#endif


 if (reload)
   {
   /*
    * New (libwww-5.0) API. QingLong.27-11-96.
    */
    Doc* thedocument = NULL;

#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      {
       Arena_TracePrint(Iam,
			" Anchor "POINTER_FORMAT" is requested to _RELOAD_.\n",
			theAnchor);
      }
#endif

   /* QingLong.06-03-97.
    * If the doc being reloaded is CurrentDoc we must NOT free it!
    */
    thedocument = HTAnchor_document(theParentAnchor);
    if (thedocument != NULL)
      {
       if (thedocument != CurrentDoc)
	 {
	  RemoveDocument_from_DocumentList(thedocument);
	 }

       HTAnchor_setDocument(theParentAnchor, NULL);
      }
   }
  else
   {
    l = context->history;

    while ((h = (History*)(HTList_nextObject(l))) != NULL)
      {
       theParentAnchor = HTAnchor_parent(h->anchor);

       if ((void*)theParentAnchor == (void*)theAnchor)
	 {
	  doc = (Doc*)HTAnchor_document(theParentAnchor);
	  if (doc)
	    {
#ifdef ARENA_DEBUG	/* QingLong.13-12-96 */
	     if (BRIDGE_TRACE && VERBOSE_TRACE)
	       Arena_TracePrint(Iam,
				" doc found: "POINTER_FORMAT
				" at state 0x%x, returning\n",
				doc, doc->state);
#endif
	     return doc;
	    }
	   else
	    {
#ifdef ARENA_DEBUG	/* QingLong.16-12-96 */
	     {
	      char* theaddress;

	      theaddress = HTAnchor_address(h->anchor);

	      if (BRIDGE_TRACE && VERBOSE_TRACE)
		Arena_TracePrint(Iam,
				 "\n\tthe doc \"%s\" in the history is NULL,"
				 "\n\tanchor = "
				 POINTER_FORMAT
				 "\n\th->anchor->parent->document = "
				 POINTER_FORMAT
				 "\n\tanchor->parent->document = "
				 POINTER_FORMAT"\n"
				 "\tcreating new template doc for request.",
				 theaddress, theAnchor, doc,
				HTAnchor_document(HTAnchor_parent(theAnchor)));
	      Free(theaddress);
	     }
#endif
	    }
	 }
      }
     /* End ``while'' */
   }
 /* End ``if (reload)'' */


#ifdef ARENA_DEBUG
 /*
  * At this point the document must be NULL. Let's ensure.
  */
 if (doc)
   {
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam,
		       " ERROR! doc = "POINTER_FORMAT" (not NULL).\n", doc);

    return doc;
   }
  else
   {
    Doc* thedocument = HTAnchor_document(HTAnchor_parent(theAnchor));

    if (thedocument)
      {
       if (BRIDGE_TRACE)
	 Arena_TracePrint(Iam,
			  " ERROR!"
			  " thedocument = "POINTER_FORMAT" (not NULL).\n",
			  thedocument);
	return thedocument;
      }
   }
#endif

 /*
  * Now create new document structure.
  */
 doc = DocNew();
 doc->state    = DOC_NOTREGISTERED;
 doc->nofree   = nofree;
 doc->main_doc = main_doc;
 doc->anchor   = HTAnchor_parent(theAnchor);
 doc->url      = HTAnchor_address((HTAnchor*)HTAnchor_parent(theAnchor));
 doc->Fragment = NULL;

#ifdef ARENA_DEBUG	/* QingLong.16-12-96 */
 if (BRIDGE_TRACE && VERBOSE_TRACE) debug_PrintDoc(Iam, doc);
#endif

 if (main_doc && record)
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (BRIDGE_TRACE)
      {
       Arena_TracePrint(Iam,
			" recording maindoc "POINTER_FORMAT":\n"
			"\t\"%s\" (anchor "POINTER_FORMAT")\n",
			doc, doc->url, theAnchor);
      }
#endif
    HistoryRecord(theAnchor);
   }

 /*
  * register the document
  */ 
 if (!main_doc && host_doc)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam,
		       " register inline "POINTER_FORMAT" vs."
		       " host "POINTER_FORMAT".\n",
		       doc, host_doc);
#endif
    ArenaRegisterInlineDocument(host_doc, doc);
   }
 /* End ``if (!main_doc && host_doc)'' */

 doc->state = DOC_PENDING;

 /*
  * Now perform document "loading".
  */
 doc->content_buffer = (char*)Arena_CAlloc(theDocContentsBufferLength,
					   sizeof(char),
					   False);
 if (doc->content_buffer)
   {
    memcpy(doc->content_buffer,
	   theDocContentsBuffer, theDocContentsBufferLength);
    doc->loaded_length = theDocContentsBufferLength;
    doc->state = DOC_LOADED;
   }
#ifdef ARENA_DEBUG
  else
   {
    char* theaddress;

    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam, " Failed to allocate new content buffer.\n");

    theaddress = HTAnchor_address(theAnchor);
    Announce("%s \"%s\".\n", _("Failed to load"), theaddress);
    Free(theaddress);
    RemoveDocument_from_DocumentList(doc);
    return NULL;
   }
#endif

#ifdef ARENA_DEBUG
 if (BRIDGE_TRACE && VERBOSE_TRACE) debug_PrintDoc(Iam, doc);
#endif

 /*
  * Now "finish" the "loading" and display the "loaded" document.
  */
 Internal_TerminateHandler(doc, NULL, HT_LOADED, theDocumentFormat);

 return doc;
}


/*
 * libLoadAnchor creates a document structure (if necessary) and
 *               starts the process of filling it. 
 *
 *               this function is the arena frontend to HTLoad
 *
 * href
 * hreflen
 * who        currently unused
 * main_doc   is this a main document or an inline referenced from a host_doc
 * host_doc   if inline, this points to host document
 */
Doc* libLoadAnchor(HTAnchor* anchor, Doc* host_doc,
		   Bool main_doc, Bool nofree, Bool record,
		   Bool reload, Bool block, Bool for_real)
{
 Doc* doc = NULL;
#ifdef ARENA_ALWAYS_FREE_PAST_DOCS	/* QingLong.01-08-97 */
 Bool must_free_doc_after_all = False;
#endif
 int status;
 HTRequest* request;
 HTReload theReloadMode = HT_CACHE_OK;
 Bool forreal;

 History* h;
 HTList* l;

#ifdef ARENA_DEBUG
 char Iam[] = "libLoadAnchor";
#endif


 HTParentAnchor* theParentAnchor;	/* auxiliary variable! */


 if (anchor == NULL)
   {
#ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The anchor to load is NULL!\n");
#endif
    return NULL;
   }

 theParentAnchor = HTAnchor_parent(anchor);

 if (Arena_anchor_IS_pending(theParentAnchor))
   {
#  ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " anchor to load ("POINTER_FORMAT") is pending.\n",
		       theParentAnchor);
#  endif

    return HTAnchor_document(theParentAnchor);
   }


#ifdef ARENA_DEBUG	/* QingLong.24-12-96 */
 {
  char *theaddress;

  theaddress = HTAnchor_address(anchor);
  Announce("%s %s", _("Fetching"), theaddress);
  Free(theaddress);
 }


 if (BRIDGE_TRACE && VERBOSE_TRACE)
   {
    HTFormat theFormat = HTAnchor_format(theParentAnchor);
    char* theContentType = HTAtom_name(theFormat);


    if (theContentType)
      Arena_TracePrint(Iam,
		       " Requested anchor ("POINTER_FORMAT")"
		       " Content-Type: \"%s\"\n",
		       theParentAnchor, theContentType);
     else
      Arena_TracePrint(Iam,
		       " Requested anchor ("POINTER_FORMAT")"
		       " Content-Type isn't set.\n",
		       theParentAnchor);
   }
#endif


 if (main_doc) ShowBusy();

 forreal = for_real;


 if (reload)
   {
   /*
    * New (libwww-5.0) API. QingLong.27-11-96.
    */
    Doc* thedocument = NULL;

#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      {
       Arena_TracePrint(Iam,
			" Anchor "POINTER_FORMAT" is requested to _RELOAD_.\n",
			anchor);
      }
#endif

   /* QingLong.05-03-97.
    * New (libwww-5.0) anchor loading mechanism relies on the absence
    * of the anchor's contents (associated document) --- if the document
    * is present (anchor->document != NULL) NO REAL LOADING is performed
    * and the document is considered to be already in place...
    */
   /* QingLong.06-03-97.
    * If the doc being reloaded is CurrentDoc we must NOT free it!
    */
    thedocument = HTAnchor_document(theParentAnchor);
    if (thedocument != NULL)
      {
       HTRequest* therequest = thedocument->request;

       if (thedocument != CurrentDoc)
	 {
	  RemoveDocument_from_DocumentList(thedocument);
	 }

       HTAnchor_setDocument(theParentAnchor, NULL);
       HTRequest_setContext(therequest, NULL);
      }

    /*
     * I doubt what mode is correct:
     *       HT_CACHE_OK             = Use any version available
     *       HT_CACHE_FLUSH_MEM      = Reload from file cache or network
     *       HT_CACHE_VALIDATE       = Validate cache entry
     *       HT_CACHE_END_VALIDATE   = End to end validation
     *       HT_CACHE_RANGE_VALIDATE = ?
     *       HT_CACHE_FLUSH          = Force full reload
     *    Or maybe:
     *       (HT_CACHE_VALIDATE | HT_CACHE_FLUSH_MEM)
     *    ???
     */
    theReloadMode = HT_CACHE_FLUSH;   /* see HTReq.h */
   }
  else
   {
   /* New (libwww-5.0) API. QingLong.27-11-96 */
#if 1
    theReloadMode = HT_CACHE_OK;
# else
    /* Validate cache entry */
    theReloadMode = HT_CACHE_VALIDATE;   /* see HTReq.h */
#endif
    /*
     * I doubt what mode is correct:
     *       HT_CACHE_OK             = Use any version available
     *       HT_CACHE_FLUSH_MEM      = Reload from file cache or network
     *       HT_CACHE_VALIDATE       = Validate cache entry
     *       HT_CACHE_END_VALIDATE   = End to end validation
     *       HT_CACHE_RANGE_VALIDATE = ?
     *       HT_CACHE_FLUSH          = Force full reload
     *    Or maybe:
     *       (HT_CACHE_VALIDATE | HT_CACHE_FLUSH_MEM)
     *    ???
     */

    l = context->history;

    while ((h = (History*)(HTList_nextObject(l))) != NULL)
      {
       theParentAnchor = HTAnchor_parent(h->anchor);

       if ((void*)theParentAnchor == (void*)anchor)
	 {
	  doc = (Doc*)HTAnchor_document(theParentAnchor);
#ifdef ARENA_DEBUG	/* QingLong.01-08-97 */
	  if (doc)
	    {
	     if (BRIDGE_TRACE && VERBOSE_TRACE)
	       Arena_TracePrint(Iam,
				" doc found: "POINTER_FORMAT" (in history)"
				" at state 0x%x, returning.\n",
				doc, doc->state);
	     return doc;
	    }
	   else
	    {
	     char* theaddress;

	     theaddress = HTAnchor_address(h->anchor);

	     if (BRIDGE_TRACE && VERBOSE_TRACE)
	       Arena_TracePrint(Iam,
				"\n\tthe doc \"%s\" in the history is NULL,"
				"\n\tanchor = "
				POINTER_FORMAT
				"\n\th->anchor->parent->document = "
				POINTER_FORMAT
				"\n\tanchor->parent->document = "
				POINTER_FORMAT"\n"
				"\tcreating new template doc for request.",
				theaddress, anchor, doc,
				HTAnchor_document(HTAnchor_parent(anchor)));
	     Free(theaddress);
	    }
# else
	  if (doc) return doc;
#endif
	 }
      }
     /* End ``while'' */

    /* Now if `doc' is still NULL,
     * we should check the anchor context doc and it's state.
     */
    if (doc == NULL)
      {
       Doc* thedocument;

       theParentAnchor = HTAnchor_parent(anchor);
       thedocument = HTAnchor_document(theParentAnchor);

#ifdef ARENA_DEBUG
       if (thedocument)
	 {
	  if (thedocument->state == DOC_LOADED)
	    {
	     if (BRIDGE_TRACE && VERBOSE_TRACE)
	       Arena_TracePrint(Iam,
				" doc found: "POINTER_FORMAT
				" at state \"DOC_LOADED\", returning.\n",
				doc);
	     return thedocument;
	    }
	   else
	    {
	     if (BRIDGE_TRACE && VERBOSE_TRACE)
	       Arena_TracePrint(Iam,
				" doc found: "POINTER_FORMAT
				" but state is 0x%x. WEIRD?\n",
				doc, doc->state);
	    }
	 }
        else
	 {
	  if (BRIDGE_TRACE && VERBOSE_TRACE)
	    Arena_TracePrint(Iam,
			     " anchor "POINTER_FORMAT" doc is NULL.\n",
			     anchor);
	 }
# else
       if (thedocument) 
	 if (thedocument->state == DOC_LOADED)
	   return thedocument;

#endif
      }
   }
 /* End ``if (reload)'' */


 /* Set up doc before calling HTLoadAnchor.
  * If a file is in the cache, the terminate handler will be called.
  * If the doc is not set up correctly
  * the terminate handler can not find it and will not process the doc.
  * Herman ten Brugge <herman@htbrug.hobby.nl> 10-Mar-96.
  */
 /*
  * New (libwww-5.*) API. QingLong.09-12-96
  */
 {
  Doc* thedocument;

  theParentAnchor = HTAnchor_parent(anchor);
  thedocument = HTAnchor_document(theParentAnchor);

  if (thedocument)	/* QingLong.15-12-96 */
    {
#ifdef ARENA_ALWAYS_FREE_PAST_DOCS	/* QingLong.01-08-97 */
     must_free_doc_after_all = False;
#endif
     doc = (Doc*)thedocument;
    }
   else
    {
     if (doc == NULL)
       {
	doc = DocNew();
#ifdef ARENA_ALWAYS_FREE_PAST_DOCS	/* QingLong.01-08-97 */
	must_free_doc_after_all = True;
#endif
	doc->state = DOC_NOTREGISTERED;
	doc->nofree = nofree;
	doc->main_doc = main_doc;
	doc->anchor = theParentAnchor;
	doc->url = HTAnchor_address((HTAnchor*)HTAnchor_parent(anchor));
	doc->tag = HTAnchor_view(anchor);
	doc->Fragment = HTAnchor_view(anchor);
	/*
	 * Where should this be done? Is this place correct? QingLong.06-03-97.
	 */
	doc->pending_reload = reload;
       }
     /* End ``if (doc == NULL)'' */
    }
  /* End ``if (thedocument)'' */
 }


 /*
  * Now create new request and set the necessary parameters.
  */
 request = ReqNew(theParentAnchor);
 HTRequest_setReloadMode(request, theReloadMode);
 HTRequest_setContext(request, (void*)doc);
 doc->request = request;

 if (host_doc && host_doc->anchor)
   HTRequest_setParent(request, HTAnchor_parent((HTAnchor*)host_doc->anchor));


 /*
  * Record the doc in the history if it's `main' and recording is necessary.
  */
 if (main_doc && record)
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (BRIDGE_TRACE)
      {
       Arena_TracePrint(Iam,
			" recording maindoc "POINTER_FORMAT":\n"
			"\t\"%s\" (anchor "POINTER_FORMAT")\n",
			doc, doc->url, anchor);
      }
#endif
    HistoryRecord(anchor);
   }


 /*
  * register the document
  */ 
 if (!main_doc && host_doc)
   {
#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam,
		       " register inline "POINTER_FORMAT" vs."
		       " host "POINTER_FORMAT".\n",
		       doc, host_doc);
#endif
    ArenaRegisterInlineDocument(host_doc, doc);
   }
 /* End ``if (!main_doc && host_doc)'' */


 if (forreal)
   {
    Arena_anchor_pending_start(theParentAnchor);

#ifdef PROGRESSIVE_PNG
    if ((HTAnchor_format(theParentAnchor) == png_atom) ||
	(HTAnchor_format(theParentAnchor) == png_exp_atom))
      {
       InitProgressiveImage(doc);
      }
#endif

    doc->state = DOC_PENDING;

#ifdef ARENA_DEBUG
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      {
       Arena_TracePrint(Iam,
			"\n\tgoing to call HTLoadAnchorRecursive"
			"(anchor="POINTER_FORMAT","
			" request="POINTER_FORMAT")\n"
			"\tfor %s doc = "POINTER_FORMAT".\n"
			"\tThe anchor structure follows...\n",
			anchor, request, (main_doc ? "main" : "inline"), doc);
       debug_PrintAnchor(Iam, (HTAnchor*)anchor);
       Arena_TracePrint(Iam, "\n\t...and the document...\n");
       debug_PrintDoc(Iam, doc);
      }

    status = HTLoadAnchorRecursive((HTAnchor*)anchor, request);

    if (BRIDGE_TRACE && VERBOSE_TRACE)	/* QingLong.13-12-96 */
      Arena_TracePrint(Iam,
		       "\n\thas called HTLoadAnchorRecursive("
		       " anchor="POINTER_FORMAT", request="POINTER_FORMAT")\n"
		       "\tgot: status = %i, doc = "POINTER_FORMAT".\n",
		       anchor, request, status, doc);
# else
    status = HTLoadAnchor((HTAnchor*)anchor, request);
#endif

    if (status == NO)
      {
#ifdef ARENA_DEBUG
       Bool deleted_successfully;
#endif

       {
	char* theaddress;
	theaddress = HTAnchor_address(anchor);

	Announce("%s \"%s\".\n", _("Failed to load"), theaddress);
	Free(theaddress);
       }

#ifdef ARENA_DEBUG
       if (BRIDGE_TRACE)
	 Arena_TracePrint(Iam,
			  " Failed to load the doc "POINTER_FORMAT",\n"
			  "\tremoving the request "POINTER_FORMAT".\n",
			  doc, request);

       deleted_successfully = RemoveRequest_from_RequestLeaks_List(request);

       if (BRIDGE_TRACE)
	 {
	  if (!deleted_successfully)
	    Arena_TracePrint(Iam, 
			     " Failed to remove the request"
			     " "POINTER_FORMAT" cleanly.\n",
			     request);
	 }
# else
       RemoveRequest_from_RequestLeaks_List(request);
#endif

       if (doc)
	 {
	  if (doc->anchor)
	    {
	     Arena_anchor_pending_stop(doc->anchor);
	    }

	  if (main_doc && record)
	    {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
	     if (BRIDGE_TRACE)
	       {
		Arena_TracePrint(Iam,
				 " deleting maindoc "POINTER_FORMAT
				 " from history:\n"
				 "\t\"%s\" (anchor "POINTER_FORMAT")\n",
				 doc, doc->url, anchor);
	       }
#endif
	     HistoryDelete(anchor);
	    }
	 }

       HideBusy();

#ifdef ARENA_ALWAYS_FREE_PAST_DOCS	/* QingLong.01-08-97 */
       if (must_free_doc_after_all)
	 {
	  if (doc != NULL)
	    {
#ifdef ARENA_DEBUG
	     if (BRIDGE_TRACE && VERBOSE_TRACE)
	       {
		Arena_TracePrint(Iam,
				 " deleting the doc "POINTER_FORMAT".\n", doc);
		debug_PrintDoc(Iam, doc);
	       }
#endif
	     doc->state = DOC_NOTREGISTERED;
	     if (RemoveDocument_from_DocumentList(doc) doc = NULL;
	    }
	 }
#endif
       return NULL;
      }
    /* End if */
   }
 /* End ``if (forreal)'' */ 

#ifdef ARENA_DEBUG
 if (BRIDGE_TRACE && VERBOSE_TRACE) debug_PrintDoc(Iam, doc);
#endif

 return doc;
}



#if 0
int PostCallback (HTRequest * request, HTStream * target)
{
 if (!request || !request->source_anchor || !target) return HT_ERROR;

 {
  int status;
  HTParentAnchor * source = request->source_anchor;
  char * document = (char *) HTAnchor_document(request->source_anchor);
  int len = HTAnchor_length(source);

  status = (*target->isa->put_block)(target, document, len);
  if (status == HT_OK) return (*target->isa->flush)(target);
  if (status == HT_WOULD_BLOCK)
    {
     if (PROT_TRACE) TTYPrint(TDEST,"POST Anchor. Target WOULD BLOCK\n");
     return HT_WOULD_BLOCK;
    }
  else
    if (status == HT_PAUSE)
      {
       if (PROT_TRACE) TTYPrint(TDEST,"POST Anchor. Target PAUSED\n");
       return HT_PAUSE;
      }
    else
      if (status > 0)
	{
	/* Stream specific return code */
	 if (PROT_TRACE)
	   Announce(_("POST Anchor. Target returns %d\n"), status);
	 return status;
        }
      else
	{
	/* We have a real error */
	 if (PROT_TRACE) Announce(_("POST Anchor. Target ERROR\n"));
	 return status;
        }
 }
}
#endif


Doc* libPostDocument(char *href, size_t hreflen, char *post, size_t postlen)
{
 HTAnchor* anchor = anchorExpandnHref(href, hreflen);
 HTAnchor* src;
 Doc* doc = NULL;
 char *name;
 HTRequest *request;

 HTParentAnchor* theParentAnchor;
 char* theaddress;
 Bool bambarbiq = False;	/* What that ``override'' mean is
				   a great W3C secret... */
#ifdef ARENA_DEBUG
 char Iam[] = "libPostDocument";
#endif

 if (anchor == NULL) return NULL;

 theParentAnchor = HTAnchor_parent(anchor);
 theaddress = HTAnchor_address((HTAnchor*)theParentAnchor);

 {
#ifdef HAVE_SNPRINTF
  int thehreflen = Arena_StrLen(href) + 14;
  name = (char *)Arena_MAlloc(thehreflen + 1, False);
  snprintf(name, thehreflen, "internal:post/%s", href);
# else
  int thehreflen = Arena_StrLen(href);
  name = (char *)Arena_MAlloc(thehreflen + 14 + 1, False);
  sprintf(name, "internal:post/%.*s", thehreflen, href);
#endif
 }

 Announce("%s %s", _("Posting"), theaddress);
 ShowBusy();

 request = HTRequest_new();
#ifdef ARENA_USE_BLOCKING_SOCKETS
 HTRequest_setPreemptive(request, YES);
# else
 HTRequest_setPreemptive(request, NO);
#endif
 HTRequest_addRqHd(request, HT_C_HOST);
 HTRequest_setOutputStream(request,
			   HTXParse(request, NULL, NULL, NULL, NULL));
 {
  HTList* tmp_conversions;

  tmp_conversions = HTRequest_conversion(request);
  /* What that bambarbiq ``override'' mean is a great W3C secret... */
  HTRequest_setConversion(request, context->conversions, bambarbiq);
  Free(tmp_conversions);
 }

 HTRequest_setMethod(request, METHOD_POST);

 src = HTAnchor_findAddress(name);
 HTAnchor_setDocument((HTParentAnchor*)src, post);
 HTAnchor_setFormat((HTParentAnchor*)src,
		    HTAtom_for("application/x-www-form-urlencoded"));
 HTAnchor_setLength((HTParentAnchor*)src, postlen);

 /* New (libwww-5.0) API. QingLong.06-12-96 */
 HTLink_add(src, anchor, NULL, METHOD_POST);	

/* HTUploadAnchor((HTAnchor *) src, request, HTUpload_callback);
 * Herman ten Brugge <herman@htbrug.hobby.nl>
 */

 doc = DocNew();
 doc->request = request;
 doc->state = DOC_PENDING;
 doc->anchor = theParentAnchor;
 HTAnchor_setDocument(theParentAnchor, (void *)doc);
 doc->main_doc = True;
 doc->nofree = True;
 doc->url = HTAnchor_address((HTAnchor*)theParentAnchor);

 HTRequest_setContext(request, (void *)doc);
 HTUploadAnchor((HTAnchor*) src, request, HTUpload_callback);
 HistoryRecord(anchor);

 /* register the document */ 

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 if (BRIDGE_TRACE)
   {
    Arena_TracePrint(Iam, " registering maindoc \"%s\"\n", doc->url);
   }
#endif

 return doc;
}

/* glk 09.09.97
 * Locate "fragment" in our list associated with this document.
 * Then give us back the PixOffset from the top of the document...
 * to use for DeltaHTMLPositon()...  This routine will return 0 offset
 * if unable to locate the "fragment".
 * glk.15.11.97
 * Several bugs are corrected by NEVER allowing the Y to be toooo far down
 * the document!  We will set the offset "up" the document based on length
 * less the height of the display window.
 */
long Y_of_Fragment(Doc* d, const char *fragment)
{
 long myY = 0;
 int fl = Arena_StrLen(fragment);
#ifdef ARENA_DEBUG
 char Iam[] = "Y_of_Fragment";
#endif


 if (d != NULL && d->FragmentCounter > 0 && fragment != NULL && fl > 0)
   {
    HTList* lp = d->FragmentPtr;
    if (lp != NULL)
      {
       FragmentAnchor* fa;
       while ((fa = (FragmentAnchor*)HTList_nextObject(lp)) != NULL)
	 {
	  if (fl == fa->NameLen && strncasecmp(fragment, fa->Name, fl) == 0)
	    {
	     myY = fa->NamePixelsFromTop;
	     break;
	    }
	 }
      }

    /* Don't allow Y beyond the Height of the buffer minus the height of the
     * window! This fixes the scrollbar and "black" background near end of
     * document. I've put this same code inside DeltaHTMLPosition().
     */
    {
     unsigned int win_height = GetDocDispWinHeight();

     if ((myY + win_height) > buf_height)
       {
#ifdef ARENA_DEBUG
	if (BRIDGE_TRACE && VERBOSE_TRACE)
	  Arena_TracePrint(Iam,
			   " myY = %ld --- BEYOND END! Set to %ld\n",
			   myY, buf_height - win_height);
#endif
	myY = buf_height - win_height;
	if (myY < 0) myY = 0;
       }
    }
   }

return myY;
}

HTAnchor* libGetDocument(char *href, size_t hreflen,
			 Doc* host_doc,
			 Bool main_doc, Bool nofree, Bool record,
			 Bool reload, Bool block)
{
 char* thref = NULL;
 char* hrefdup = NULL;
 size_t hrefduplen;

 HTAnchor* anchor;
 Doc* d = NULL;
 Doc* theDoc = NULL;

#ifdef ARENA_DEBUG
 char Iam[] = "libGetDocument";
#endif

 /* Expand our input href into FULL specification...
  * That way, <a NAME=#tag> will get ALL the document specifications added
  */
 thref = strndup(href, hreflen);
 hrefdup = ExpandHref(thref);
 hrefduplen = Arena_StrLen(hrefdup);
 Free(thref);

 /* See if going to a mailto URL...
  * If so, create the form and load it!
  */
 if (hrefduplen > 7)
   if (strncasecmp(hrefdup, "mailto:", 7) == 0)
       MTGetMailtoInputForm(hrefdup);

 /* Figure out all the anchor, doc stuff.
  * Really trying to locate old version of the anchor/doc
  * pointers... so we can reuse them!
  */
 if ((anchor = anchorExpandnHref(hrefdup, hrefduplen)) == NULL) return NULL;
 d = (Doc*)HTAnchor_document(HTAnchor_parent(anchor));
 if (reload) { CurrentDoc->pending_reload = True; }
 Free(hrefdup);

 /* Depending on the state of the Document...
  * We may start loading anew... or alter the current!
  */
 if (d)
   {
    /* Store fragment tag info with document we already have...
     * that way the reparse or redisplay of the document will display
     * the proper fragment location.  Note: We may have multiple anchors
     * using "this" Doc*.  So set both tag and Fragment for the Doc*
     *
     * libLoadAnchor() will take care of setting both ->tag and ->Fragment!
     * We will Free ->Fragment after display... then history->y prevails.
     * Tag is useless except to display the proper URL!  DO NOT test ->tag
     * Its value for this Doc* changes with each HTAnchor* that references
     * this document!
     */
    Free(d->tag);
    Free(d->Fragment);
    d->tag = HTAnchor_view(anchor);
    d->Fragment = HTAnchor_view(anchor);

    switch (d->state)
      {
       case DOC_NOTREGISTERED:
       case DOC_ABORTED:
       case DOC_REJECTED:
	 theDoc = libLoadAnchor(anchor, host_doc,
				main_doc, nofree, record, reload, block,
				True);
	 break;

       case DOC_PENDING:
       case DOC_EXTERNAL:
	 break;

       case DOC_LOADED:
       case DOC_PARSING_LOCK:
       case DOC_PROCESSED:
	 if (reload)
	   {
	    theDoc = libLoadAnchor(anchor, host_doc,
				   main_doc, nofree, record, reload, block,
				   True);
	   }
	 else
	   {
	    Bool Reparse = (CurrentDoc != d);
	    CurrentDoc = d;
	    CurrentDoc->show_raw = False;

	    /* QingLong.23-04-97 */
	    /* At the point, we should verify the history in order to
	     * check if the doc found (which is implied to be main)
	     * has already been recorded in the history.
	     * E.g., if the doc found has been previously loaded as inline,
	     * then it has not been recorded to the history,
	     * so we should check and record it here.
	     */
	    if (!HistoryVerify(anchor))
	      if (HistoryRecord(anchor))
		HistoryVerify(anchor);
#  if 0
	    /* if an external view shows an image,
	     * the content_buffer will be NULL
	     */
	    if (d->content_buffer == NULL) d->loaded_length = 0;
#  endif
	    if (Reparse)
	      NewBuffer(CurrentDoc);
	    else
	      {
	       if (CurrentDoc->Fragment)
		 context->current_history->y = Y_of_Fragment(CurrentDoc,
							 CurrentDoc->Fragment);
	       MoveYDisplay(CurrentDoc, context->current_history->y);
	      }

	    /* NOW, allow history->y to control the display of "this" document!
	     * Unless of course we "open" it again with a different HTAnchor*
	     * and therefore maybe a different Fragment tag!
	     */
	    Free(CurrentDoc->Fragment);
	   }
	 break;

       default:
#ifdef ARENA_DEBUG	/* QingLong.15-12-96 */
	 if (BRIDGE_TRACE && VERBOSE_TRACE)	/* QingLong.15-12-96 */
	   {
	    Arena_TracePrint(Iam, " unexpected doc->state = %i.\n", d->state);
	   }
#endif
	 break;
      }
   }
 else
   {
    theDoc = libLoadAnchor(anchor, host_doc,
			   main_doc, nofree, record, reload, block, True);
   }


 return (theDoc ? anchor : NULL);
}



/*
 * Get inline document of the given type. (e.g. css_atom for style sheets)
 */
Doc* GetInlineDoc(HTFormat theType, char *href, int hreflen, Bool reload)
{
 HTAnchor* a;
 HTParentAnchor* theParentAnchor;
 Doc* doc = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "GetInlineDoc";
#endif


 /* check for null name */
 if ((href == NULL) || (hreflen == 0))
   {
    Warn(_("Missing or bad inline name"));
    return NULL;
   }

 /* first, lets expand the href into an anchor */
 a = anchorExpandnHref(href, hreflen);

#ifdef ARENA_DEBUG	/* QingLong.13-12-96 */
 if (BRIDGE_TRACE && VERBOSE_TRACE)
   {
    if (a)
      debug_PrintAnchor(Iam, a);
     else
      Arena_TracePrint(Iam, " anchor expanded from href is NULL!\n");
    /* End ``if (a == NULL)'' */
   }
#endif

  theParentAnchor = HTAnchor_parent(a);

 /* then set it's Content-Type (if specified) */
 if (theType) HTAnchor_setFormat(theParentAnchor, theType);

  /* get the anchor contents */
 doc = (Doc*)HTAnchor_document(theParentAnchor);

 /* check if document is already in place */
 if ((doc == NULL) || reload)
   {
    /* no, the document does not seem to be in place, lets register it..
     * first, clear any reference to previously loaded document..
     */
    if (doc)
      {
       HTRequest* therequest = doc->request;

       if (RemoveDocument_from_DocumentList(doc))
	 {
	  doc = NULL;
	  HTAnchor_setDocument(theParentAnchor, NULL);
	  HTRequest_setContext(therequest, NULL);
	 }
        else
	 {
#ifdef ARENA_DEBUG	/* QingLong.09-02-97 */
	  Arena_TracePrint(Iam,
			   " ERROR! Can't free document "POINTER_FORMAT".\n",
			   doc);
# else
	  Arena_PrintError(_("!nternal error: cannot free document.\n"));
#endif
#if 0
	  Exit(-1);
#endif
	 }
      }


#ifdef ARENA_DEBUG	/* QingLong.11-12-96 */
    if (BRIDGE_TRACE)
      {
       char *s = strndup(href, hreflen);
       if (reload)
	 Arena_TracePrint(Iam, " registering \"%s\" (DO reload)\n", s);
        else
	 Arena_TracePrint(Iam, " registering \"%s\" (do NOT reload)\n", s);
       /* End ``if (reload)'' */
       Free(s);
      }
    /* End ``if (BRIDGE_TRACE)'' */
#endif

    {
     Doc* thedocument = NULL;

    /* howcome 1/8/95: record should be false, no? */
     thedocument = libLoadAnchor(a, CurrentDoc,
				 False, False, False, reload, False, True);

     if (thedocument)
       {
	theParentAnchor = HTAnchor_parent(a);

	doc = (Doc*)HTAnchor_document(theParentAnchor);

#ifdef ARENA_DEBUG
	if (BRIDGE_TRACE)
	  if (thedocument != doc)
	    Arena_TracePrint(Iam,
			     " WARNING!\n"
			     "\tHas called libLoadAnchor and got doc "
			     POINTER_FORMAT",\n"
			     "\twhile anchor->parent->document is "
			     POINTER_FORMAT".\n",
			     thedocument, doc);
#endif
       }
      else
       {
#ifdef ARENA_DEBUG
	if (BRIDGE_TRACE)
	  Arena_TracePrint(Iam,
			   " WARNING!\n"
			   "\tHas called libLoadAnchor and got NULL doc.\n");
#endif

	doc = NULL;
       }
    }

    /* if image or style is still not there.. */
    if (doc == NULL) return NULL;
   }
 /* End if ``document is already in place'' */


 if (doc)
   {
   /*
    * the inline doc has been registered, it can be loading, loaded, processed.
    */
    switch (doc->state)
      {
      case DOC_NOTREGISTERED:
#ifdef ARENA_DEBUG	/* QingLong.16-04-97 */
	 if (BRIDGE_TRACE)	/* QingLong.16-04-97 */
	   Arena_TracePrint(Iam,
			    " doc "POINTER_FORMAT" not registered.\n", doc);
#endif
	 return NULL;
	 break;

       case DOC_PENDING:
#ifdef ARENA_DEBUG	/* QingLong.11-12-96 */
         if (BRIDGE_TRACE)
           {
            char *s = strndup(href, hreflen);
	    Arena_TracePrint(Iam,
			     " doc pending"
			     " (has been registered, but not loaded), =>\n"
			     "\treturning default \"%s\"\n",
			     s);
            Free(s);
           }
         /* End ``if (BRIDGE_TRACE)'' */
#endif

#ifdef PROGRESSIVE_PNG
	 if ((HTAnchor_format(theParentAnchor) == png_atom) ||
	     (HTAnchor_format(theParentAnchor) == png_exp_atom))
	   {
	    return doc;
	   }
#endif

	 return NULL;
         break;

       case DOC_LOADED:
#ifdef ARENA_DEBUG	/* QingLong.11-12-96 */
         if (BRIDGE_TRACE)
	   {
	    Arena_TracePrint(Iam, " doc "POINTER_FORMAT" loaded.\n", doc);
	    if (VERBOSE_TRACE) debug_PrintDoc(Iam, doc);
	   }
         /* End ``if (BRIDGE_TRACE)'' */
#endif
         return doc;
	 break;

       case DOC_PARSING_LOCK:
#ifdef ARENA_DEBUG
         if (BRIDGE_TRACE)
           {
            char *s = strndup(href, hreflen);
	    Arena_TracePrint(Iam,
			     " doc locked while processing,\n"
			     "\treturning default s \"%s\"\n",
			     s);
	    Free(s);
	   }
         /* End ``if (BRIDGE_TRACE)'' */
#endif
         return NULL;
	 break;

       case DOC_PROCESSED:
#ifdef ARENA_DEBUG	/* QingLong.11-12-96 */
         if (BRIDGE_TRACE)
	   Arena_TracePrint(Iam, " doc processed,\n\treturning image \"%s\"\n",
			    doc->url);
         /* End ``if (BRIDGE_TRACE)'' */
#endif
         return doc;
         break;

      case DOC_ABORTED:
#ifdef ARENA_DEBUG	/* QingLong.16-04-97 */
	 if (BRIDGE_TRACE)	/* QingLong.16-04-97 */
	   Arena_TracePrint(Iam, " doc "POINTER_FORMAT" aborted.\n", doc);
#endif
	 return NULL;
	 break;

      case DOC_REJECTED:
#ifdef ARENA_DEBUG	/* QingLong.16-04-97 */
	 if (BRIDGE_TRACE)	/* QingLong.16-04-97 */
	   Arena_TracePrint(Iam, " doc "POINTER_FORMAT" rejected.\n", doc);
#endif
	 return NULL;
	 break;
 
      case DOC_EXTERNAL:
#ifdef ARENA_DEBUG	/* QingLong.16-04-97 */
	 if (BRIDGE_TRACE && VERBOSE_TRACE)	/* QingLong.16-04-97 */
	   Arena_TracePrint(Iam, " doc "POINTER_FORMAT" is external.\n", doc);
#endif
	 return NULL;
	 break;

      default:
#ifdef ARENA_DEBUG	/* QingLong.15-12-96 */
	 if (BRIDGE_TRACE && VERBOSE_TRACE)	/* QingLong.15-12-96 */
	   {
	    Arena_TracePrint(Iam,
			     " unexpected doc->state = %i.\n", doc->state);
	   }
#endif
	 return NULL;
	 break;
      }
    /* End switch */
   }
 /* End ``if (doc)'' */
 return NULL;
}


/*
 * Get inline document --- external style sheet.
 */
Doc* GetStyleSheet(char *href, int hreflen, Bool reload)
{
 return GetInlineDoc(css_atom, href, hreflen, reload);
}


/*
 * Get inline document of unspecified type, e.g. image.
 */
Doc* GetInline(char *href, int hreflen, Bool reload)
{
 return GetInlineDoc(NULL, href, hreflen, reload);
}


/*
 * Open the doc, if specified. If not, ask user for one beforehand.
 */
void OpenDoc(char* theURL)
{
 static char* OpenString = NULL;
 char* newURL;
#ifdef ARENA_DEBUG
 char Iam[] = "OpenDoc";
#endif


 if (Arena_StrLen(theURL))
   newURL = theURL;
  else
   if ((newURL = ArenaPromptUser(_("URL to open?"), OpenString, True)))
     {
      Free(OpenString);
      OpenString = newURL;
     }
    else
     return;

#ifdef ARENA_DEBUG
 if (STATUS_TRACE)
   Arena_TracePrint(Iam,
		    " %s\n\t\"%s\"\n",
		    (ARENA_Index(newURL, ':') ? "remote" : "local (?)"),
		    newURL);
#endif

#ifdef BOOKMARKS
 if (!BMOpenDocCheckForIndex(newURL)) return;
#endif

 libGetDocument(newURL, Arena_StrLen(newURL), NULL,
		True, False, True, False, False);
 ResetFindStringPos();
}


void PostDoc(char *name, char *post)
{
 if (name && post)  /* attempt to get new document */
   {
    ArenaStatusLineDesactivate();   /* quit Open, SaveAs or Find if active */
    libPostDocument(name, Arena_StrLen(name), post, Arena_StrLen(post));
    ResetFindStringPos();
   }
}


void ReloadDoc(char *name)
{
 if (name)  /* attempt to get new document */
   {
    ArenaStatusLineDesactivate();   /* quit Open, SaveAs or Find if active */

   /* free style sheets */

    if (CurrentDoc->style != context->style)
      {
       FreeStyleSheet(CurrentDoc->style);
       CurrentDoc->style = NULL;
      }
     else
      {
       FreeStyleSheet(CurrentDoc->style);
       CurrentDoc->style = NULL;
       context->style = StyleGetInit();
      }

    Free(CurrentDoc->head_style);
    Free(CurrentDoc->link_style);

    HistoryAllowMarkForDeath(False);

    libGetDocument(name, Arena_StrLen(name), NULL,
		   True, False, False, True, False);
    ResetFindStringPos();
   }
}

/*
 * libEntry is called quite early in the process
 */
void libEntry(void)
{
 extern int Caching;
 extern char *CacheRoot;
 extern char *EmailAddr;

 char *TmpDir = NULL;
 char *stemp = NULL;

#ifdef ARENA_DEBUG
 extern char *LibWWW_traceflags;
 char Iam[] = "libEntry";

 if (LibWWW_traceflags != NULL)
   if (*LibWWW_traceflags != '\0')
     {
     /* turn on debugging in Library */
      library_trace = HTSetTraceMessageMask(LibWWW_traceflags);

      if (library_trace || ANY_TRACE)
	Arena_TracePrint(Iam,
			 " W3C library trace: flags = \"%s\", mask = 0x%x\n",
			 LibWWW_traceflags, library_trace);
      /* End if */
     }
#endif

 /* Init the W3C stuff.  It will create a default user profile,
  * which we will throw away, and then create our own!
  */
 HTLibInit(BANNER, ARENA_VERSION);
 UserProfile = HTLib_userProfile();
 HTUserProfile_delete(UserProfile);

 /* Now let's make our version of the "default" user profile...
  * set it up the way WE want it and then set IT as the default!
  */
 UserProfile = HTUserProfile_new(ArenaUser, NULL);
 HTUserProfile_localize(UserProfile);

 /* The temporary files path will be set as follows:
  *   1) from the environment variable TMPDIR
  *   2) from X defaults
  *   3) from ARENA_DEFAULT_TMP_ROOT in arena.h
  *   4) from ~/.Arena/tmp
  *   5) /tmp
  */
 stemp = getenv("TMPDIR");
 if (!stemp) stemp = DetermineValue("tmp",  ARENA_DEFAULT_TMP_ROOT);
 StrAllocCopy(TmpDir, stemp);
 if (!stemp)
   {
    TmpDir = ALibHomePath();
    StrAllocCat(TmpDir, ".");
    StrAllocCat(TmpDir, HTLib_appName());
    StrAllocCat(TmpDir, "/tmp/");
    stemp = NULL;
   }
 else if (Arena_StrLen(stemp) > 2 && stemp[0] == '~' && stemp[1] == '/')
   {
    TmpDir = ALibHomePath();
    StrAllocCat(TmpDir, ".");
    StrAllocCat(TmpDir, HTLib_appName());
    StrAllocCat(TmpDir, &stemp[1]);
   }

 if (ALibCreatePath(TmpDir, 0755) != YES) StrAllocCopy(TmpDir, "/tmp/");
 HTUserProfile_setTmp(UserProfile, TmpDir);
 Free(TmpDir);

 if (EmailAddr) HTUserProfile_setEmail(UserProfile, EmailAddr);
 HTLib_setUserProfile(UserProfile);
#ifdef ARENA_DEBUG
 if (BRIDGE_TRACE)
   {
    Arena_TracePrint(Iam, " fqdn  = %s\n", HTUserProfile_fqdn(UserProfile));
    Arena_TracePrint(Iam, " email = %s\n", HTUserProfile_email(UserProfile));
    Arena_TracePrint(Iam, " tmp   = %s\n", HTUserProfile_tmp(UserProfile));
   }
#endif


 HTNet_setMaxSocket(ARENA_MAX_SOCKETS);


 /*
  * New (libwww-5.0) API. QingLong.06-12-96.
  */

 /* Register the transport protocol for our listen tool */
 HTTransportInit();

 /* Initialize the protocol modules */
#ifdef ARENA_USE_BLOCKING_SOCKETS
 HTProtocolPreemptiveInit();
# else
 HTProtocolInit();
#endif


#ifdef ARENA_DIRECT_WAIS
#  if 0
 HTProtocol_delete("wais");
#  endif
 HTProtocol_add("wais", "tcp", WAIS_PORT, YES, HTLoadWAIS, NULL);
#endif

 /* Initialize persisitent cache... glk 05.07.97
  * But first, figure out where the user wants it!  Up to now, CacheRoot
  * has been set from the command line arguments via the -cache or -nocache
  * switches.  This directory path may or may not be the same as the temp
  * directory above!
  * Order of setup:
  *   1) Command Line arg will take highest priority.
  *   2) X defaults next.
  *   3) ARENA_DEFAULT_CACHE_ROOT from arena.h
  *   &) NULL is the same as using the -nocache switch.
  *
  * The Caching flag can be False...
  * iff -nocache or -cache "" was entered by user from the command line.
  * CacheRoot can only be non-null if the user specified an accessible path.
  */
#ifdef ARENA_DEBUG
 if (BRIDGE_TRACE)
   {
    Arena_TracePrint(Iam, " START Setup of Cache. Flag = %d\n", Caching);
    Arena_TracePrint(Iam, " CacheRoot = %s\n",
		     CacheRoot ? CacheRoot : "(null) No command line value");
   }
#endif
 if (Caching)
   {
    /* User specified path... FORCE it to already exist!
     * If the path is invalid, TURN OFF caching for this user.
     */
    if (CacheRoot)
      {
       struct stat st;
       int istat = stat(CacheRoot, &st);
       if (istat || (!istat && !S_ISDIR(st.st_mode)))
	 {
	  Caching = False;
	  Free(CacheRoot);
	 }
      }
    /* Pick up the defaults... if we need to...
     * We may have turned off caching above!
     */
    if (!CacheRoot && Caching)
      StrAllocCopy(CacheRoot,
		   DetermineValue("CacheRoot", ARENA_DEFAULT_CACHE_ROOT));

    /* Even from the defaults, we can get NULL...
     * If not, substitute /$HOME/.Arena for ~/ at beginning of spec
     */
    if (!CacheRoot)
      Caching = False;
    else
      {
       if (CacheRoot[0] == '~' && CacheRoot[1] == '/')
	 {
	  stemp = strdup(CacheRoot);
	  StrAllocCopy(CacheRoot, ALibHomePath());
	  StrAllocCat(CacheRoot, ".");
	  StrAllocCat(CacheRoot, HTLib_appName());
	  StrAllocCat(CacheRoot, &stemp[1]);
	  if (*(CacheRoot + Arena_StrLen(CacheRoot) - 1) != '/')
	    StrAllocCat(CacheRoot, "/");
	  Free(stemp);
	 }
       if (Arena_StrLen(CacheRoot) <= 0) Caching = False;
      }
    /* Now try to init the libwww cache system...
     * This can still fail!
     */
    if (Caching) Caching = HTCacheInit(CacheRoot, ARENA_CACHE_MAX_SIZE);
   }

 /* If caching is STILL on, set it! Otherwise, make sure that ALL versions
  * of the caching flags are false, and inform libwww!
  */
 if (Caching)
   HTCacheMode_setExpires(HT_EXPIRES_AUTO);
  else
   {
    Caching = False;
    Free(CacheRoot);
    HTCacheMode_setEnabled(False);
   }

#ifdef ARENA_DEBUG
 if (BRIDGE_TRACE)
   {
    Arena_TracePrint(Iam,
		     "\n"
		     "\tFINAL Setup of Cache Flag = 0x%x,\n"
		     "\tLibWWW Flag = 0x%x,\n"
		     "\tRoot = \"%s\",\n"
		     "\tCacheRoot = \"%s\".\n",
		     Caching,
		     HTCacheMode_enabled(),
		     HTCacheMode_getRoot() ? HTCacheMode_getRoot() : "(null)",
		     CacheRoot ? CacheRoot : "(null)");
   }
#endif
 /* OK, we got the cache setup done...
  * We should probably use the two HTCacheMode_xxx routines used above to check
  * the caching flag and to get the cache path.  Since the LibWWW stuff
  * can change the flag on us... I'm guessing, but libwww will probably be
  * turning its caching flag on/off at will, when loading certain types
  * of documents.   If we allow libwww to do whatever it wants
  * with the Flag and Root, then we can always set it again to the defaults
  * in Caching and CacheRoot.  At this time, they SHOULD BE all the same!
  */



 /* read old environment variables */
 HTProxy_getEnvVar();

 /*
  * Initialize conversion modules.
  */
 HTConverterInit(context->conversions);
 HTFormat_setConversion(context->conversions);

 /* Register the default set of transfer encoders and decoders */
 HTTransferEncoderInit(context->encodings);
 HTContentEncoderInit(context->encodings);
 HTFormat_setTransferCoding(context->encodings);

 /* Register the default set of MIME header parsers */
 HTMIMEInit();

 /*
  * The application specific converters...
  */

 HTConversion_add(context->conversions,
                  "www/mime",
                  "*/*", HTMIMEConvert,    1.0, 0.0, 0.0);

 HTConversion_add(context->conversions,
                  TEXT_DOCUMENT,
                  "www/present", HTXParse, 0.4, 0.0, 0.0);
 HTConversion_add(context->conversions,
                  HTML_DOCUMENT,
                  "www/present", HTXParse, 0.6, 0.0, 0.0);
 HTConversion_add(context->conversions,
                  HTML3_DOCUMENT,
                  "www/present", HTXParse, 0.8, 0.0, 0.0);
 HTConversion_add(context->conversions,
                  HTML_LEVEL3_DOCUMENT,
                  "www/present", HTXParse, 0.9, 0.0, 0.0);
 HTConversion_add(context->conversions,
                  GIF_DOCUMENT, 
                  "www/present", HTXParse, 0.6, 0.0, 0.0);
 HTConversion_add(context->conversions,
                  XPM_DOCUMENT, 
                  "www/present", HTXParse, 0.6, 0.0, 0.0);
 HTConversion_add(context->conversions,
                  XBM_DOCUMENT,
                  "www/present", HTXParse, 0.6, 0.0, 0.0);

#ifdef JPEG
 HTConversion_add(context->conversions,
                  JPEG_DOCUMENT,
                  "www/present", HTXParse, 0.8, 0.0, 0.0);
#endif /* JPEG */

#ifdef PNG
 HTConversion_add(context->conversions,
                  PNG_DOCUMENT,
                  "www/present", HTXParse, 0.8, 0.0, 0.0);
 HTConversion_add(context->conversions,
                  PNG_EXP_DOCUMENT,
                  "www/present", HTXParse, 0.8, 0.0, 0.0);
#endif /* PNG */


 /* Fighting ``smart'' libwww default converters to
  * let style sheets reach Arena renderer...
  */

 HTConversion_add(context->conversions,
		  TEXT_DOCUMENT,
                  CSS_DOCUMENT,  HTThroughLine, 0.6, 0.0, 0.0);

 HTConversion_add(context->conversions,
		  TEXT_DOCUMENT,
                  CSS1_DOCUMENT, HTThroughLine, 0.6, 0.0, 0.0);

 HTConversion_add(context->conversions,
		  "*/*",
                  CSS_DOCUMENT,  HTThroughLine, 0.3, 0.0, 0.0);

 HTConversion_add(context->conversions,
		  "*/*",
                  CSS1_DOCUMENT, HTThroughLine, 0.3, 0.0, 0.0);
 /* --- */


 HTConversion_add(context->conversions,
                  "*/*",
                  "www/present", HTXParse, 0.1, 0.0, 0.0);

 HTFormat_setConversion(context->conversions);

 /* ------ */

 HTBind_caseSensitive(False);

 HTBind_addType("txt",  "text/plain", 0.9);
 HTBind_addType("html", "text/html",  0.9);
 HTBind_addType("htm",  "text/html",  0.9);
 HTBind_addType("gif",  "image/gif",  0.9);
 HTBind_addType("png",  "image/png",  0.9);
 HTBind_addType("jpg",  "image/jpeg", 0.9);
 HTBind_addType("jpeg", "image/jpeg", 0.9);
 HTBind_addType("css",  CSS_DOCUMENT, 0.9);	/* QingLong.10-02-91 */

 HTFileInit();

 if (UseMailcap) register_mailcaps();

 HTAlert_add(AHTProgress, HT_A_PROGRESS);
 HTAlert_add(ArenaShowMessage, HT_A_MESSAGE);

 ArenaDialogInit();

 ArenaAuthInit();

 /* get atom values,
  * so that we don't have to do strcmp for content types later
  */

 text_atom        = HTAtom_caseFor(TEXT_DOCUMENT);
 html_atom        = HTAtom_caseFor(HTML_DOCUMENT);
 html3_atom       = HTAtom_caseFor(HTML3_DOCUMENT);
 html_level3_atom = HTAtom_caseFor(HTML_LEVEL3_DOCUMENT);
 gif_atom         = HTAtom_caseFor(GIF_DOCUMENT);
 jpeg_atom        = HTAtom_caseFor(JPEG_DOCUMENT);
 png_atom         = HTAtom_caseFor(PNG_DOCUMENT);
 xpm_atom         = HTAtom_caseFor(XPM_DOCUMENT);
 xbm_atom         = HTAtom_caseFor(XBM_DOCUMENT);

 unknown_atom     = WWW_UNKNOWN;	/* QingLong.13-02-97 */
 css_atom         = HTAtom_caseFor(CSS_DOCUMENT);
 css1_atom        = HTAtom_caseFor(CSS1_DOCUMENT);
}


/*
 * Callback function. Called for each chunk of data arrived.
 */
void HTCallClient(HTXParseStruct* eps)
{
 Doc* doc = NULL;
 HTAnchor* anchor = NULL;
 HTRequest* req = eps->request;
#ifdef ARENA_DEBUG
 char Iam[] = "HTCallClient";
#endif

 {
  HTParentAnchor* theParentAnchor;
  HTChildAnchor*  theChildAnchor;

  theChildAnchor = HTRequest_childAnchor(req);
  if (theChildAnchor)
    anchor = (HTAnchor*)theChildAnchor;
   else
    {
     theParentAnchor = HTRequest_anchor(req);
     if (theParentAnchor) anchor = (HTAnchor*)theParentAnchor;
    }
 }

 if (anchor == NULL)
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if(BRIDGE_TRACE)
      Arena_TracePrint(Iam, " NO ANCHOR FOUND, returning\n");
#endif
    return;
   }

 if (HTAnchor_parent(anchor) == NULL) /* Brian Campbell <brianc@qnx.com>
					 16-Mar-96 */
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if(BRIDGE_TRACE)
      Arena_TracePrint(Iam, " NO ANCHOR parent FOUND, returning.\n");
#endif
    return;
   }

 doc = HTRequest_context(req);

 if (doc == NULL)
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam, " NO DOC FOUND IN CONTEXT ???\n");
#endif
    return;
   }
 /* End ``if (doc == NULL)'' */


 /* The document should be associated with it's anchor
  * as soon as the first chunk of data arrives.
  */
 if ((doc->loaded_length == 0) && (eps->used))
   {
    HTParentAnchor* theParentAnchor = HTAnchor_parent(anchor);

#  ifdef ARENA_DEBUG
    {
     Doc* theDoc = (Doc*)HTAnchor_document(theParentAnchor);

     if (theDoc == NULL)
       {
	if (BRIDGE_TRACE)
	  Arena_TracePrint(Iam,
			   "\n\tThe doc "POINTER_FORMAT
			   " anchor ("POINTER_FORMAT") document is NULL."
			   "\n\tSetting it to the doc.\n",
			   doc, theParentAnchor);
       }
    }
#  endif

    HTAnchor_setDocument(theParentAnchor, (void*)doc);
   }

 doc->url = HTAnchor_address((HTAnchor*)HTAnchor_parent(anchor)); /* What this line is intended for?
					      QingLong.17-12-96. */

 if (eps->finished)
   {
#if 0	/* null-termination now done by library */
   /* doc->request = eps->request; */
   /* NULL-terminating this seems necessary.. do we cross boundaries?? */
    eps->buffer[eps->used] = 0;
#endif

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (BRIDGE_TRACE)
      Arena_TracePrint(Iam,
		       "\n\t\"%s\" ("POINTER_FORMAT")\n"
		       "\tfinishing, got %d bytes\n",
		       doc->url, doc, eps->used);
#endif

    doc->content_buffer = eps->buffer;
   }
  else
   {
    unsigned long this_chunk_begin  = doc->loaded_length;
    unsigned long this_chunk_length = eps->used - doc->loaded_length;

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (BRIDGE_TRACE && VERBOSE_TRACE)
      {
       Arena_TracePrint(Iam,
			"\n\t\"%s\" ("POINTER_FORMAT")\n"
			"\tread %d bytes,"
			" anchor "POINTER_FORMAT"\n",
			doc->url, doc, eps->used, doc->anchor);
      }
#endif
    doc->state = DOC_PENDING;
    doc->loaded_length = eps->used;

    if (initialised)
      {
#ifdef PROGRESSIVE_PNG
       HTParentAnchor* theParentAnchor = HTAnchor_parent(anchor);
       Bool ReparseBuffer = False;

       if ((HTAnchor_format(theParentAnchor) == png_atom) ||
           (HTAnchor_format(theParentAnchor) == png_exp_atom))
         {
          doc->content_buffer = eps->buffer;

          if (this_chunk_length > 0)
	    {
	     if (this_chunk_begin == 0)
	       {
#  ifdef ARENA_DEBUG
		if (BRIDGE_TRACE && VERBOSE_TRACE)
		  Arena_TracePrint(Iam,
				   "\n\tprogressive PNG image "
				   "(doc="POINTER_FORMAT")"
				   " is empty. Initializing...\n",
				   doc);
#  endif
		InitProgressiveImage(doc);
	       }

#  ifdef ARENA_DEBUG
	     if (BRIDGE_TRACE && VERBOSE_TRACE)
	       Arena_TracePrint(Iam,
				" processing progressive PNG image "
				POINTER_FORMAT".\n",
				doc);
#  endif
	     ReparseBuffer = (ProcessProgressiveImage(doc,
						      this_chunk_begin,
						      this_chunk_length)
			      == NEWIMAGE);
	    }

          if (doc->main_doc)
            {
             CurrentDoc = doc;
             CurrentDoc->show_raw = False;
	     if (ReparseBuffer)
	       {
		NewBuffer(CurrentDoc);
		HistoryVerify((HTAnchor*)doc->anchor);
		DisplayDoc(CurrentDoc, False);
	       }
            } 
           else
            {
             if (doc->host_documents)
               {
#  ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
                if (BRIDGE_TRACE)
                  {
		   Arena_TracePrint(Iam,
				    " EVENT_OK"
				    " inline progressive PNG image.\n");
                  }
#  endif
		if (ReparseBuffer)
		  {
		   NewBuffer(CurrentDoc);
		   DisplayDoc(CurrentDoc, True);
		  }
	       }
            }
         }
#endif	/*  PROGRESSIVE_PNG */

       if (doc->loaded_length > 1024)
         {
#if 1
          long int len = HTAnchor_length(doc->anchor);
# else
          long int len =
	             HTAnchor_length((HTAnchor*)HTAnchor_parent(doc->anchor));
#endif
          int a, b;

          /* first, compute the k lengths */

          a = (doc->loaded_length + 512) / 1024;
          b = (len + 512) / 1024;

          if (len > 0)     /* if content-length was given */
            {
             int left = len - doc->loaded_length;
             if (a == b && left > 0)
               {
                Announce(_("Read %dk (%d bytes remain) from %s"),
                         a, left, doc->url);
               }
              else
               {
                Announce(_("Read %dk/%dk from %s"), a, b, doc->url);
               }
            }
           else
            Announce(_("Read %dk of %s"),
                     (int)((doc->loaded_length + 512) / 1024), doc->url);
         }
      }
   }
 return;
}


int timeout_handler (SOCKET theSocket, void * param, HTEventType theType)
{
#ifdef ARENA_DEBUG	/* QingLong.08-12-96 */
 char Iam[] = "timeout_handler";

    if (BRIDGE_TRACE) Arena_TracePrint(Iam, " do something cool!\n");
#endif

 if (!HTAlert_interactive())
   {
#if 0
    Doc* thedoc;

    HTRequest * theRequest = (HTRequest*)param;
    thedoc = (Doc*)HTRequest_context(request);
#endif

#ifdef ARENA_DEBUG	/* QingLong.08-12-96 */
    Announce(_("Still waiting for response..."));
#endif
   }

 HideBusy();
 return 0;
}


/*
 * The front end to the WWW library. This is the last `after' filter.
 */
int terminate_handler (HTRequest* request, HTResponse* response, 
                       void* param, int status)
{
 Doc* doc;
 int return_status;
#ifdef ARENA_DEBUG
 char Iam[] = "terminate_handler";
#endif


#ifdef ARENA_DEBUG	/* QingLong.08-12-96 */
 if (BRIDGE_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam, " status = %d.\n", status);
#endif

 if (!context) return HT_OK;

 doc = (Doc*)HTRequest_context(request);

 if (doc == NULL) /* 12-Mar-96 Herman ten Brugge <herman@htbrug.hobby.nl> */
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if(BRIDGE_TRACE) Arena_TracePrint(Iam, " called before Doc set (?!)\n");
#endif
    return EXIT_SUCCESS;
   }
 /* End ``if (doc == NULL)'' */


 /* QingLong.24-04-97 */	/* Or should this be done in HTCallClient()? */
 {
 /* Update anchor content type from the correspondent response object.
  * (from HTTP response "Content-Type" header)
  */
  HTFormat theAnchorFormat = HTResponse_format(response);

#ifdef ARENA_DEBUG
  if (theAnchorFormat)
    {
     if (BRIDGE_TRACE && VERBOSE_TRACE)
       Arena_TracePrint(Iam,
			" the response "POINTER_FORMAT" format is \"%s\".\n",
			response, theAnchorFormat->name);
    }
   else
    {
     if (BRIDGE_TRACE)
       Arena_TracePrint(Iam,
			" the response "POINTER_FORMAT" format is NULL!\n",
			response);
    }
#endif

  return_status = Internal_TerminateHandler(doc, param, status,
					    theAnchorFormat);
 }

 /*
  * howcome 19/2/96: we can't delete the request here, but where??
  */

 return return_status; 
}


/*
 * Does the main set of proper request termination actions.
 */
int Internal_TerminateHandler(Doc* doc, void* param, int status,
			      HTFormat theDocumentFormat)
{
 int ht_status = HT_OK;
#ifdef ARENA_DEBUG
 char Iam[] = "Internal_TerminateHandler";
#endif


 if (doc == NULL)
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if(BRIDGE_TRACE) Arena_TracePrint(Iam, " called before Doc set (?!)\n");
#endif
    return EXIT_SUCCESS;
   }

#ifdef ARENA_DEBUG
  if (theDocumentFormat)
    {
     if (BRIDGE_TRACE && VERBOSE_TRACE)
       Arena_TracePrint(Iam,
			" the doc "POINTER_FORMAT" format is \"%s\".\n",
			doc, theDocumentFormat->name);

     if (theDocumentFormat != WWW_UNKNOWN)
       HTAnchor_setFormat(doc->anchor, theDocumentFormat);
    }
   else
    {
     if (BRIDGE_TRACE)
       Arena_TracePrint(Iam,
			" the doc "POINTER_FORMAT" format is NULL!\n", doc);
    }
# else
  if ((theDocumentFormat) && (theDocumentFormat != WWW_UNKNOWN))
    HTAnchor_setFormat(doc->anchor, theDocumentFormat);
#endif

#ifdef ARENA_DEBUG
  if (BRIDGE_TRACE && VERBOSE_TRACE) debug_PrintDoc(Iam, doc);
#endif

  switch (status)
    {
     case HT_ERROR:
#ifdef ARENA_DEBUG
       if (BRIDGE_TRACE && VERBOSE_TRACE)
	 Arena_TracePrint(Iam, " case HT_ERROR\n");
#endif

       if ((doc->state == DOC_NOTREGISTERED) || (doc->state == DOC_PENDING))
	 doc->state = DOC_REJECTED;

#if 0	/* Henrik F. Nielsen 22-03-98 */
       Announce(_("Error while fetching %s"), doc->url);
#endif
       HistoryDelete((HTAnchor*)(doc->anchor));
       break;

     case HT_INTERRUPTED:
#ifdef ARENA_DEBUG
       if (BRIDGE_TRACE && VERBOSE_TRACE)
	 Arena_TracePrint(Iam, " case HT_INTERRUPTED\n");
#endif
       break;

    case HT_NOT_MODIFIED:
#ifdef ARENA_DEBUG
      if (BRIDGE_TRACE && VERBOSE_TRACE)
	Arena_TracePrint(Iam, " case HT_NOT_MODIFIED\n");
#endif


#if 0
      if ((doc->state == DOC_NOTREGISTERED) || (doc->state == DOC_PENDING))
	doc->state = DOC_LOADED;

      HideBusy();
      break;
#endif

    case HT_LOADED:
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
      if (BRIDGE_TRACE && VERBOSE_TRACE)
	Arena_TracePrint(Iam, " case HT_LOADED\n");
#endif

      if ((doc->state == DOC_NOTREGISTERED) || (doc->state == DOC_PENDING))
	doc->state = DOC_LOADED;

      if (doc->main_doc)
        {
        /* we will get to this point even when the document was
         * handled by an external application registered through mailcap.
         * The only (?) way to detect this is if content_buffer is NULL.
         */

         if (doc->content_buffer == NULL)
           {
            HideBusy();
            HistoryDelete((HTAnchor*)(doc->anchor));
	    if (!RemoveDocument_from_DocumentList(doc))
	      {
#  ifdef ARENA_DEBUG
	       if (BRIDGE_TRACE)
		 Arena_TracePrint(Iam,
				  " Cannot free document "POINTER_FORMAT").\n",
				  doc);
#  endif
	       ht_status = HT_INTERNAL;	/* Cannot free the document */
               break;
	      }
           }

         if (doc != CurrentDoc)
           {
	   /*
	    * Can the CurrentDoc be NULL?
	    */
	    if (CurrentDoc != NULL)
	      {
	       if (doc->anchor == CurrentDoc->anchor)
		 {
		  doc->pending_reload = CurrentDoc->pending_reload;
		  doc->nofree = CurrentDoc->nofree;
		  doc->user_style = CurrentDoc->user_style;
		 }

#ifdef ARENA_ALWAYS_FREE_PAST_DOCS	/* QingLong.01-08-97 */
	       if (!RemoveDocument_from_DocumentList(doc))
		 {/* Cannot free the document */
#  ifdef ARENA_DEBUG
		  if (BRIDGE_TRACE)
		    Arena_TracePrint(Iam,
				     " Cannot free document (CurrentDoc = "
				     POINTER_FORMAT").\n",
				     CurrentDoc);
#  endif
		 }
#endif
	      }
#ifdef ARENA_DEBUG
	    else
	      {
	       if (BRIDGE_TRACE)
		 Arena_TracePrint(Iam, " ERROR? CurrentDoc is NULL!\n");
	      }
#endif
	    CurrentDoc = doc;
            if (CurrentDoc->anchor == context->home_anchor)
              CurrentDoc->user_style = True;
           }
         /* at this point, we should verify the history
          * registration done in libGetDocument
          */
	 /* glk.7.12.97
	  * But make sure we use the correct anchor!  If the user gets
	  * .../doc.html#tag, we have a child anchor, but the doc->anchor
	  * is always a parent anchor.  If we don't verify the child, our
	  * history is a mess!  I've also altered the HistoryVerify routine
	  * to "verify" ALL entries that have this same "parent anchor"...
	  * so the child verify also cleans up the parents.
          * (as is usually the case late in life).
          */
	  if ((HTChildAnchor*)(HTRequest_childAnchor(doc->request)) != NULL)
	    HistoryVerify((HTAnchor*)(HTRequest_childAnchor(doc->request)));
	   else
	    HistoryVerify((HTAnchor*)(doc->anchor));

         /* if an external view shows an image,
          * the content_buffer will be NULL
          */
         if (!CurrentDoc->content_buffer) CurrentDoc->loaded_length = 0;

         NewBuffer(doc);

         /* if the document hasn't been processed by now, it never will be.
          * Since we've already changed CurrentDoc, we need to change it back.
          * This is easiest with BackDoc().
          */
         if (doc->state != DOC_PROCESSED)
           {
            BackDoc(DocDispGC, GetDefaultDocViewWindow());
            HistoryDelete((HTAnchor*)(doc->anchor));
            Announce(_("Unable to display %s"), doc->url);
            HideBusy();
#ifdef ARENA_DEBUG
	    if (BRIDGE_TRACE)
	      debug_PrintDoc(Iam, doc);
#endif
            break;
           }

        /* if the document has been reloaded,
         * turn off flag to avoid looping with inlines
         */
         CurrentDoc->pending_reload = False;

	 /* Sanity must prevail! */
	 if (context->current_history == NULL)
	   {
#  ifdef ARENA_DEBUG
	    Arena_TracePrint(Iam, " context->current_history= NULL (error)\n");
#  endif
	    exit(33);
	   }

         HideBusy();
        }
       else
        {
	 /* i.e. !main_doc
	  * this is an inline finishing
	  * --- first check to see that the host_documents exists,
	  * in it the hostimg document should be found.
	  */
         if (doc->host_documents)
           {
#ifdef ARENA_DEBUG	/* QingLong.11-12-96 */
            if (BRIDGE_TRACE) Arena_TracePrint(Iam, " EVENT_OK inline\n");
#endif
	    CurrentDoc->pending_reload = False; /* fixes looping inline gifs */
            NewBuffer(CurrentDoc);
           }
          else
	   {
#ifdef ARENA_DEBUG	/* QingLong.11-12-96 */
	    if (BRIDGE_TRACE)
	      Arena_TracePrint(Iam, " WHOOPS no host_doc specified.\n");
#endif
	   }

         if (CurrentDoc) DisplayUrl(CurrentDoc);
        }
      break;

   default:
#ifdef ARENA_DEBUG	/* QingLong.08-12-96 */
     if (BRIDGE_TRACE && VERBOSE_TRACE)
       Arena_TracePrint(Iam,
			" unexpected status %i (?)\n", status);
#endif
     break;
   }

 Arena_anchor_pending_stop(doc->anchor);

 return ht_status;
}



/* libEventLoop is the final call from the application.
 * It should only exit when the application is being exited.
 */
void libEventLoop(int theConnexionNumber, char *href)
{
 char * ref = NULL;
 Doc* doc;
 HTAnchor* anchor;
 /* init_doc must be visiable to style.c
  * to determine if initial doc is being loaded
  */
 Doc* init_doc;
 HTRequest* request = HTRequest_new();
#ifdef ARENA_DEBUG
 char Iam[] = "libEventLoop";
#endif


 /*
  * QingLong.09-12-96: Register default event manager
  */
 HTEventInit();

 /*
  * Remember X connection socket number for future.
  */
 ArenaXsocket = theConnexionNumber;

 {
  /* Set timeout on sockets */
  HTEvent* theArenaTimeoutEvent;

  theArenaTimeoutEvent = HTEvent_new(timeout_handler, (void*)request,
				     HT_PRIORITY_MAX,
				     ARENA_SOCKETS_TIMEOUT_SEC*1000);

  HTEvent_register(theConnexionNumber, HTEvent_TIMEOUT, theArenaTimeoutEvent);
 }


 /*
  * register socket for X events
  */
 {
  HTEvent* theArenaReadEvent;

  theArenaReadEvent = HTEvent_new(EventHandler, NULL,
				  HT_PRIORITY_MAX,
				  ARENA_SOCKETS_TIMEOUT_SEC*1000);
  HTEvent_setParam(theArenaReadEvent, (void*)theArenaReadEvent);

  HTEvent_register(theConnexionNumber,
		   HTEvent_READ, theArenaReadEvent);
 }

 /* display initial page */

 init_doc = DocNew(); /* free this at some point */
 init_doc->main_doc = True;
 init_doc->url = NULL;
 init_doc->nofree = True;
 init_doc->content_buffer = INITIAL_HTML;
 init_doc->tag = NULL;
 init_doc->Fragment = NULL;
 init_doc->already_displayed = False;
 init_doc->field_editor = NULL;
 init_doc->edited_field = NULL;

 /* get atom values, so that we don't have to do strcmp for content types later
  */

 text_atom        = HTAtom_caseFor(TEXT_DOCUMENT);
 html_atom        = HTAtom_caseFor(HTML_DOCUMENT);
 html3_atom       = HTAtom_caseFor(HTML3_DOCUMENT);
 html_level3_atom = HTAtom_caseFor(HTML_LEVEL3_DOCUMENT);
 gif_atom         = HTAtom_caseFor(GIF_DOCUMENT);
 jpeg_atom        = HTAtom_caseFor(JPEG_DOCUMENT);
 png_atom         = HTAtom_caseFor(PNG_DOCUMENT);
 png_exp_atom     = HTAtom_caseFor(PNG_EXP_DOCUMENT);
 xpm_atom         = HTAtom_caseFor(XPM_DOCUMENT);
 xbm_atom         = HTAtom_caseFor(XBM_DOCUMENT);

 unknown_atom     = WWW_UNKNOWN;	/* QingLong.13-02-97 */
 css_atom         = HTAtom_caseFor(CSS_DOCUMENT);
 css1_atom        = HTAtom_caseFor(CSS1_DOCUMENT);

 CurrentDoc = init_doc;
 NewBuffer(init_doc);
 DisplayAll(init_doc, True);

 GuiEvents(0, NULL, 0);
 GuiEvents(0, NULL, 0);
 GuiEvents(0, NULL, 0);


 /*
  * Init "pro" and "epi" filters, register our own "epi" filter.
  */
 HTBeforeInit();
 HTNet_deleteBefore(HTMemoryCacheFilter);
 HTAfterInit();
 HTNet_addAfter(terminate_handler, NULL, NULL, HT_ALL, HT_FILTER_LAST);

 HTDNS_setTimeout(300);

 /* now do home page */
 anchor = anchorExpandnHref(href, Arena_StrLen(href));
 context->home_anchor = HTAnchor_parent(anchor);
 doc = libLoadAnchor(anchor, NULL, True, False, True, False, False, True);

 if (doc == NULL)
   {
#ifdef ARENA_DEBUG	/* QingLong.11-05-97 */
    Arena_TracePrint(Iam, " failed to load initial document. Exiting.\n");
# else
    Arena_PrintError(_("Failed to load initial document.\n"));
#endif
    Exit(-1);
   }

 /* Enter infinite loop...
  * HTEventrg_loop() has its own infinite loop... However, spawning with
  * ASYNC mode seem to create problems for it!  What's happening
  * is that we are interrupting the select() loop when the
  * SIGCHLD signal is received.  Just reenter HTEventrg_loop().
  */
 for (;;)
   {
    HTEventList_loop(doc->request);      /* libwww-5.1 API QingLong.25-03-97 */

#if 1	/* Greg L. Kaup. 12-10-97  Thanks to Steffen Zahn for 5.01  testing  */
    if (errno != EINTR && errno != SIGINT) break; /* not really SIGINT! == 2 */
#endif

#ifdef ARENA_DEBUG
    if (SPAWN_TRACE || BRIDGE_TRACE)
      Arena_TracePrint(Iam, " HTEventrg_loop() broken. errno %d.\n", errno);
#endif
   }

#ifdef ARENA_DEBUG
 Arena_TracePrint(Iam, " HTEventrg_loop exited, why??? errno %d.\n", errno);
#endif
 Free(ref);
}


void libExit(int pCache)
{
#ifdef ARENA_DEBUG	/* QingLong.11-12-96 */
 if (BRIDGE_TRACE) Arena_TracePrint("libExit", " cleaning up...\n");
#endif

 Arena_deleteDocuments();
 Arena_deleteDocumentLeaks();

 if (HTCacheMode_getRoot() != NULL) HTCacheTerminate();
 if (HTLib_isInitialized()) HTLibTerminate();
}
