/*
 * xbm.c - XBM stuff.
 */


#include <stdio.h>
#include <stdlib.h>

#include <string.h>
#include <ctype.h>
#include <math.h>

#include "arena.h"
#include "defs.h"
#include "bridge.h"
#include "colour.h"
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
#  include "debug.h"
# else
#  include "error.h"
#endif
#include "dither.h"
#include "image.h"
#include "types.h"
#include "main.h"
#include "util.h"


/*
 * Process loaded XBM.
 */
ImageXyoke* ProcessXBM(char* xbm,
		       unsigned int width, unsigned int height,
		       unsigned int depth)
{
 int i, j, k;
 unsigned int byte = 0;
 unsigned char* xbm_p = NULL;
 ImageData* theImageData = NULL;
 ImageXyoke* theImageXcouple = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "ProcessXBM";
#endif


 if (width && height && depth)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       " processing %d by %d XBM image.\n", width, height);
#endif
   }
  else
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! Got zero size for the XBM image.\n");
#endif
    return NULL;
   }

 if (xbm)
   {
    xbm_p = (unsigned char*)xbm;
   }
  else
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE) Arena_TracePrint(Iam, " ERROR! The XBM is NULL.\n");
#endif
    return NULL;
   }

 switch (depth)
   {
    case 1:
    case 2:
    case 4:
    case 8:
    case 16:
    case 24:
      break;

    default:
#ifdef ARENA_DEBUG
      Arena_TracePrint(Iam, " Display depth %d unsupported.\n", depth);
# else
      Arena_PrintError(_("XBM for display depth %d unsupported\n"), depth);
#endif
      return NULL;
      break;
   }
 /* End ``switch (depth)'' */

 {
  unsigned int m;
  unsigned char WhiteColourL[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };
  unsigned char BlackColourL[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };

#ifdef ARENA_DEBUG
  if (!ArenaExactColour(textColour))
    {
     if (IMAGE_TRACE)
       Arena_TracePrint(Iam, " WEIRD! Failed to get exact text colour.\n");
    }

  if (!ArenaExactColour(windowColour))
    {
     if (IMAGE_TRACE)
       Arena_TracePrint(Iam, " WEIRD! Failed to get exact window colour.\n");
    }
#endif

  if (!ArenaPackColour_to_chars(  textColour, depth, imaging, BlackColourL))
    {
#ifdef ARENA_DEBUG
     if (IMAGE_TRACE)
       Arena_TracePrint(Iam, " WEIRD! Failed to pack text colour.\n");
#endif
     return NULL;
    }
 
  if (!ArenaPackColour_to_chars(windowColour, depth, imaging, WhiteColourL))
    {
#ifdef ARENA_DEBUG
     if (IMAGE_TRACE)
       Arena_TracePrint(Iam, " WEIRD! Failed to pack window colour.\n");
#endif
     return NULL;
    }
 
  {
   ImageAssembly* theImageAssembly = NewImageAssembly(imaging, width, height,
						      depth, 1);

   if (theImageAssembly)
     {
      for (i = 0; i < height; i++)
	{
	 for (j = 0, k = 8; j < width; j++)
	   {
	    if (++k > 8)  /* need to read next 8 pixel values */
	      {
	       byte = *xbm_p++;
	       k = 1;
	      }

	    m = byte & 0x01;

	    if (!NextPixel_to_ImageAssembly(theImageAssembly,
					    (m ? BlackColourL : WhiteColourL),
					    m))
	      {
#ifdef ARENA_DEBUG
	       if (IMAGE_TRACE)
		 Arena_TracePrint(Iam, " ERROR! Failed to put pixel.\n");
#endif
	       FreeImageData(FinishImageAssembly(theImageAssembly));
	       return NULL;
	      }

	    byte = byte >> 1;
	   }
	}

      theImageData = FinishImageAssembly(theImageAssembly);
     }
    else
     {
#ifdef ARENA_DEBUG
      if (IMAGE_TRACE)
	Arena_TracePrint(Iam, " ERROR! Failed to create image assembly.\n");
#endif
      return NULL;
     }
  }
 }

 theImageXcouple = processImage_data2image(theImageData, width, height, depth);
 theImageData->image = theImageData->mask = NULL;
 FreeImageData(theImageData);

 return theImageXcouple;
}



/*
 * This is an XBM image egg-sample.
 */
#if 0
#define back_width  20
#define back_height 23
#define back_x_hot  6
#define back_y_hot  6
static char back_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x0f,
   0x00, 0x80, 0x0f, 0x00, 0x80, 0x0f, 0x00, 0x80, 0x0f, 0x00, 0x80, 0x0f,
   0x00, 0x80, 0x0f, 0x60, 0x80, 0x0f, 0x70, 0x80, 0x0f, 0x78, 0x00, 0x00,
   0xfc, 0xff, 0x0f, 0xfe, 0xff, 0x07, 0xff, 0xff, 0x03, 0xfe, 0xff, 0x01,
   0xfc, 0xff, 0x00, 0x78, 0x00, 0x00, 0x70, 0x00, 0x00, 0x60, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
#endif
/*
 * So find line start with #define and look for `width'/`height'/`[xy]_hot'
 * in the line, extract prefix of the found word (xxx_width) --- XBM name. 
 * Extract the value (number which must follow the found parameter).
 * Repeat for other parameters.
 *
 * Number of bytes of data is:   ((width + 7)/8) * height
 *              i.e. equal to:   (width/8 + (width%8 ? 1 : 0)) * height
 *
 * Then skip to:
 *      "static short XBMname_bits[] = {"                (X10 format)
 *      "static unsigned char XBMname_bits[] = {"        (X11 format)
 *      "static char XBMname_bits[] = {"                 (X11 format)
 *
 * and then start reading data:
 *   whitespace or ',' followed by hex number
 */

/*
 * Load data from an XBM file, then call ProcessXBM() to process it.
 */
ImageXyoke* LoadXBMimage(Block *theBlock, unsigned int depth)
{
#ifdef ARENA_DEBUG
 char Iam[] = "LoadXBMimage";
#endif


 if (theBlock == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The block structure is NULL.\n");
#endif
    return NULL;
   }

 if (theBlock->buffer == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The block buffer is NULL.\n");
#endif
    return NULL;
   }

 if (*(theBlock->buffer + theBlock->size) == '\0')
   {
    char* xbm_b;
    int xbm_b_length;
    unsigned int width = 0, height = 0, x_hot = 0, y_hot = 0;
    unsigned int* theParameter = NULL;
    char *p, *p0;
    char* xbm_name = NULL;   /* Let XBM name include trailing underscore! */
    int xbm_name_length = -1, xbm_name_length_x = -2;
    char* xbm_bits = NULL;
    char* xbm_bits_end = NULL;
    char str_define[] = "#define";
    int str_define_length = 7;   /* sizeof(str_define[]) - 1 */
    Bool start_define;


    p = p0 = (char *)theBlock->buffer + theBlock->next;

    while ((p = strstr(p, str_define))) /* Find "#define" */
      {
       if (p == p0)
         start_define = True;
        else
         {
	  char cpx = *(p - 1);

	  if ((cpx == '\n') || (cpx == '\r') || (cpx == '\f'))
	    start_define = True;
	   else
	    start_define = False;
         }

       if (start_define)  /* This "#define" begins the line, i.e. it's valid */
         {
	  char cp;
	  char* px;

	  /* Jump over the "#define", remember the position */
	  px = (p += str_define_length);

	  /* Squeeze horizontal space */
	  while ((cp = *p++)) if (!((cp == ' ') || (cp == '\t'))) break;
	  p--;

	  if (cp)
	    {
	     if (px == p) /* Corrupted: no white space after `define'. Skip. */
	       continue;
	      else
	       {
		if (isalpha(cp))
		  {
		  /* This is parameter beginning. */
		   px = p;

		   /* Parse parameter name, extract XBM name. */

		   /* First find next space on the line. */
		   while ((cp = *p++)) if (isspace(cp)) break;
		   p--;

		   if (cp)
		     {
		      if ((cp == ' ') || (cp == '\t'))
			{
			/* This is parameter name end.
			 * Try if it ends on `width'/`height'/`[xy]_hot'
			 */
			 if ((p - px) >= 5)
			   {
			    if (strncmp(p - 4, "_hot", 4) == 0)
			      {
			       cp = *(p - 5);
			       if (cp == 'x')
				 {
				  xbm_name_length_x = p - px - 5;
				  theParameter = &x_hot;
				 }
			        else
				 {
				  if (cp == 'y')
				    {
				     xbm_name_length_x = p - px - 5;
				     theParameter = &y_hot;
				    }
				   else /* Malformed parameter name. Skip. */
				    continue;
				 }
			      }
			     else
			      {
			       if (strncmp(p - 5, "width", 5) == 0)
				 {
				  xbm_name_length_x = p - px - 5;
				  theParameter = &width;
				 }
			        else
				 {
				  if ((p - px) >= 6)
				    {
				     if (strncmp(p - 6, "height", 6) == 0)
				       {
					xbm_name_length_x = p - px - 6;
					theParameter = &height;
				       }
				      else /* Corrupted: short name. Skip. */
				       continue;
				    }
				 }
			      }

			    /* If the XBM name isn't dummy,
			     * it should end on '_'.
			     */
			    if ((xbm_name_length_x > 0) &&
				(px[xbm_name_length_x - 1] != '_'))
			      break;

			    /* Compare XBM name got from this line
			     * with the one already known (if available).
			     */
			    if (xbm_name_length >= 0)
			      {
			      /* Already know XBM name */
			       if (xbm_name_length != xbm_name_length_x)
				 continue; /* Invalid parameter name. Skip. */

			       if (strncmp(xbm_name, px, xbm_name_length))
				 continue; /* Invalid parameter name. Skip. */
			      }
			     else
			      {
			      /* Store XBM name in malloc'ed buffer */
			       if ((xbm_name = strndup(px, xbm_name_length_x)))
				 {
				  xbm_name_length = xbm_name_length_x;
				 }
			        else /* Memory exhausted, => exit the loop. */
				 break;
			      }

			    /* Squeeze horizontal space */
			    while ((cp = *p++))
			      if (!((cp == ' ') || (cp == '\t'))) break;

			    p--;

			    if (cp)
			      {
			      /* This is the parameter value beginning.
			       * Must be digit.
			       */
			       if (isdigit(cp))
				 {
				 /* Parse parameter value. */
				  int param = 0;

				  if (sscanf(p, "%i", &param) == 1)
				    if (param >= 0)
				      *theParameter = param;

				  /* Skip the rest of the line. */
				  while ((cp = *p++))
				    {
				     if ((cp == '\n') ||
					 (cp == '\r') ||
					 (cp == '\f'))
				       break;
				    }

				  p--;

				  if (cp)
				    continue;
				   else /* Buffer end, => exit the loop. */
				    break;
				 }
			       else /* Premature line end, => exit the loop. */
				 break;
			      }
			     else /* Premature end, => exit the loop. */
			      break;
			   }
			  else /* Corrupted: short parameter name. Skip. */
			   continue;
			}
		       else /* Corrupted: premature line end. Skip. */
			continue;
		     }
		    else /* Premature end, => exit the loop. */
		     break;
		  }
		 else /* Corrupted "#define ..." line: invalid name. Skip. */
		  continue;
	       }
	    }
	  else /* Premature end, => exit the loop. */
	    break;
	 }
      }

    if ((xbm_name_length < 0) || (xbm_name == NULL) ||
        (height == 0) || (width == 0))
      {
       Free(xbm_name);
       return NULL;
      }

#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		     " reading %dx%d XBM image \"%s\".\n",
		     width, height, xbm_name);
#endif

    p = p0;

    {
     char* str_xbm_bits;
     char str_bits[] = "bits[] =";
     int str_bits_length = 8;

     if (xbm_name_length > 0)
       {
        if ((str_xbm_bits = Arena_MAlloc(xbm_name_length + str_bits_length + 1,
				       False)))
	{
	 strcpy(str_xbm_bits, xbm_name);
	 strcat(str_xbm_bits, str_bits);
	}
       }
      else
       str_xbm_bits = strdup(str_bits);

     p = strstr(p, str_xbm_bits);

     Free(xbm_name);
     Free(str_xbm_bits);

     if (p)
       {
        Bool typeOk = False;
        int pX = p - p0;

        if (pX >= 12)
	  {
	   if (strncmp(p - 12, "static char ", 12) == 0)
	     typeOk = True;
	    else
	     if (pX >= 13)
	       {
		if (strncmp(p - 13, "static short ", 13) == 0)
		  typeOk = True;
		 else
		  if (pX >= 21)
		    {
		     if (strncmp(p - 21, "static unsigned char ", 21) == 0)
		       typeOk = True;
		    }
	       }
	  }

        if (typeOk)
	  {
	   char cp;

	  /* Move to the position just after "...bits[] =" */
	   p += (xbm_name_length + str_bits_length);

	   /* Squeeze space */
	   while ((cp = *p++)) if (!isspace(cp)) break;

	   /* Must be '{' --- data block beginning. */
	   if (cp == '{') xbm_bits = p;
	  }
       }
    }

    if (xbm_bits)
      {
       /* find closing brace for data */
       xbm_bits_end = ARENA_Index(xbm_bits, '}');
       if (xbm_bits_end == NULL)
         xbm_bits_end = theBlock->buffer + theBlock->size;

       /* output data length (from XBM geometry) */
       xbm_b_length = ((width + 7)/8) * height;

       if ((xbm_b = (char*)Arena_CAlloc(xbm_b_length, sizeof(char), False)))
         {
	  unsigned int byte;
	  int r, xbm_data_length;
	  char* xbm_p = xbm_b;
	  ImageXyoke* theImageXcouple = NULL;


	  for (xbm_data_length = 0 ;
	       (0 < xbm_bits_end - p) && (xbm_data_length < xbm_b_length);
	       p++)
	    {
	     /* Find `0' (first character of `0xXX') */
	     p =  ARENA_Index(p, '0');
	     if (p == NULL) break;

	     r = sscanf(p, "0x%2x", &byte);
	     if (r == EOF || r == 0) break;
	     *(xbm_p++) = byte;
	     xbm_data_length++;
	     p += 3;	/* Move to the last character of the current hexbyte */

	     /* Find the comma separator */
	     p =  ARENA_Index(p, ',');
	     if (p == NULL) break;
	    }

#ifdef ARENA_DEBUG
	  if (IMAGE_TRACE)
	    Arena_TracePrint(Iam,
			     " Read %d bytes from the XBM.\n",
			     xbm_data_length);
#endif

	  theImageXcouple = ProcessXBM(xbm_b, width, height, depth);
	  Free(xbm_b);
	  return theImageXcouple;
         }
        else
         {
#ifdef ARENA_DEBUG
	  if (IMAGE_TRACE)
	    Arena_TracePrint(Iam, " Failed to alloc buffer for XBM.\n");
#endif
	  return NULL;
         }
      }
     else
      {
#ifdef ARENA_DEBUG
       if (IMAGE_TRACE)
	 Arena_TracePrint(Iam, " ERROR! No data (`bits' block) in XBM?\n");
#endif
       return NULL;
      }
   }
  else
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The block buffer isn't ASCIIZ.\n");
#endif
    return NULL;
   }
}
