/*
 * "$Id: ps-pdf.cpp,v 1.79 1999/01/07 17:28:57 mike Exp $"
 *
 *   PostScript + PDF output routines for HTMLDOC, a HTML document processing
 *   program.
 *
 *   Just in case you didn't notice it, this file is too big; it will be
 *   broken into more manageable pieces once we make all of the output
 *   "drivers" classes...
 *
 *   Copyright 1997-1999 by Michael Sweet.
 *
 *   HTMLDOC is distributed under the terms of the GNU General Public License
 *   which is described in the file "COPYING-2.0".
 *
 * Contents:
 *
 *   pdf_export()            - Export PDF file(s)...
 *   ps_export_level1()      - Export level 1 PostScript file(s)...
 *   ps_export_level2()      - Export level 2 PostScript file(s)...
 *   pspdf_export()          - Export PostScript/PDF file(s)...
 *   pspdf_prepare_page()    - Add headers/footers to page before writing...
 *   pspdf_prepare_heading() - Add header/footer to page before writing...
 *   ps_write_document()     - Write all render entities to PostScript file(s).
 *   ps_write_page()         - Write all render entities on a page to a
 *                             PostScript file.
 *   ps_write_background()   - Write a background image...
 *   pdf_write_document()    - Write all render entities to a PDF file.
 *   pdf_write_page()        - Write a page to a PDF file.
 *   pdf_write_contents()    - Write the table of contents as outline records to
 *                             a PDF file.
 *   pdf_count_headings()    - Count the number of headings under this TOC
 *                             entry.
 *   pdf_write_background()  - Write a background image...
 *   pdf_write_links()       - Write annotation link objects for each page in the
 *                             document.
 *   pdf_write_names()       - Write named destinations for each link.
 *   parse_contents()        - Parse the table of contents and produce a
 *                             rendering list...
 *   parse_doc()             - Parse a document tree and produce rendering list
 *                             output.
 *   parse_heading()         - Parse a heading tree and produce rendering list
 *                             output.
 *   parse_paragraph()       - Parse a paragraph tree and produce rendering list
 *                             output.
 *   parse_pre()             - Parse preformatted text and produce rendering list
 *                             output.
 *   parse_table()           - Parse a table and produce rendering output.
 *   parse_list()            - Parse a list entry and produce rendering output.
 *   init_list()             - Initialize the list type and value as necessary.
 *   new_render()            - Allocate memory for a new rendering structure.
 *   add_link()              - Add a named link...
 *   find_link()             - Find a named link...
 *   compare_links()         - Compare two named links.
 *   copy_tree()             - Copy a markup tree...
 *   flatten_tree()          - Flatten an HTML tree to only include the text,
 *                             image, link, and break markups.
 *   update_image_size()     - Update the size of an image based upon the
 *                             printable width.
 *   get_width()             - Get the width of a string in points.
 *   get_title()             - Get the title string for a document.
 *   open_file()             - Open an output file for the current chapter.
 *   set_color()             - Set the current text color...
 *   set_font()              - Set the current text font.
 *   set_pos()               - Set the current text position.
 *   ps_hex()                - Print binary data as a series of hexadecimal
 *                             numbers.
 *   ps_ascii85()            - Print binary data as a series of base-85 numbers.
 *   jpg_init()              - Initialize the JPEG destination.
 *   jpg_empty()             - Empty the JPEG output buffer.
 *   jpg_term()              - Write the last JPEG data to the file.
 *   jpg_setup()             - Setup the JPEG compressor for writing an image.
 *   compare_rgb()           - Compare two RGB colors...
 *   write_image()           - Write an image to the given output file...
 *   write_prolog()          - Write the file prolog...
 *   write_string()          - Write a text string.
 *   write_text()            - Write a text entity.
 *   write_trailer()         - Write the file trailer.
 *   pdf_open_stream()       - Open a deflated output stream in a PDF file.
 *   pdf_close_stream()      - Close a deflated output string in a PDF file.
 *   pdf_puts()              - Write a character string to a compressed stream in
 *                             a PDF file.
 *   pdf_printf()            - Write a formatted character string to a compressed
 *                             stream in a PDF file.
 *   pdf_write()             - Write data to a compressed stream in a PDF file.
 */

/*
 * Include necessary headers.
 */

/*#define DEBUG*/
#include "htmldoc.h"
#include "widths.h"
#include <stdarg.h>
#include <ctype.h>
#include <time.h>
#include <math.h>

#ifdef MAC		// MacOS-specific header file...
#  include <Files.h>
#endif // MAC

#ifdef HAVE_LIBZ
#  include <zlib.h>
#endif /* HAVE_LIBZ */

#ifdef HAVE_LIBJPEG
extern "C" {		/* Workaround for JPEG header problems... */
#  include <jpeglib.h>	/* JPEG/JFIF image definitions */
}
#endif /* HAVE_LIBJPEG */


/*
 * Constants...
 */

#define CONTENTS	"Table of Contents"	/* Change as necessary */

#define RENDER_TEXT	0
#define RENDER_IMAGE	1
#define RENDER_BOX	2
#define RENDER_FBOX	3
#define RENDER_LINK	4


/*
 * Structures...
 */

typedef struct render_str	/**** Render entity structure ****/
{
  struct render_str	*next;	/* Next rendering entity */
  int	type;			/* Type of entity */
  float	x,			/* Position in points */
	y,			/* ... */
	width,			/* Size in points */
	height;			/* ... */
  union
  {
    struct
    {
      int	typeface,	/* Typeface for text */
		style;		/* Style of text */
      float	size;		/* Size of text in points */
      float	rgb[3];		/* Color of text */
      uchar	buffer[1];	/* String buffer */
    }   	text;
    image_t	*image;		/* Image pointer */
    float	box[3];		/* Box color */
    float	fbox[3];	/* Filled box color */
    uchar	link[1];	/* Link URL */
  }	data;
} render_t;

typedef struct			/**** Named link position structure */
{
  short		page,		/* Page # */
		top;		/* Top position */
  uchar		name[124];	/* Reference name */
} link_t;


/*
 * Local globals...
 */

static int	pslevel;

static int	chapter,
		chapter_starts[MAX_CHAPTERS],
		chapter_ends[MAX_CHAPTERS];

static int	num_headings,
		heading_pages[MAX_HEADINGS],
		heading_tops[MAX_HEADINGS];

static int	num_pages;
static render_t	*pages[MAX_PAGES],
		*endpages[MAX_PAGES];
static uchar	*page_headings[MAX_PAGES];
static tree_t	*current_heading;

static int	num_links;
static link_t	links[MAX_LINKS];

static uchar	list_types[16];
static int	list_values[16];

static int	num_objects,
		objects[MAX_OBJECTS],
		root_object,
		info_object,
		outline_object,
		pages_object,
		names_object,
		annots_objects[MAX_PAGES],
		background_object = 0,
		font_objects[16];

static image_t	*logo_image = NULL;

static image_t	*background_image = NULL;
static float	background_color[3] = { 1.0, 1.0, 1.0 };

static int	render_typeface,
		render_style;
static float	render_size,
		render_rgb[3],
		render_x,
		render_y;

#ifdef HAVE_LIBZ
static z_stream	compressor;
static uchar	comp_buffer[64 * 1024];
#endif /* HAVE_LIBZ */

static char	*font_names[] =	/* Short font names */
		{
		  "FC",
		  "FCB",
		  "FCI",
		  "FCBI",
		  "FT",
		  "FTB",
		  "FTI",
		  "FTBI",
		  "FH",
		  "FHB",
		  "FHI",
		  "FHBI",
		  "FS",
		  "FSB",
		  "FSI",
		  "FSBI"
		};


/*
 * Local functions...
 */

static int	pspdf_export(tree_t *document, tree_t *toc);
static char	*pspdf_prepare_page(int page, int *file_page, uchar *title,
		                    float title_width, uchar **page_heading);
static void	pspdf_prepare_heading(int page, int print_page, uchar *title,
		        	      float title_width, uchar *heading,
				      float heading_width, char *format, int y);
static void	ps_write_document(uchar *title, uchar *author, uchar *creator,
		                  uchar *copyright);
static void	ps_write_page(FILE *out, int page, uchar *title,
		              float title_width, uchar **page_heading);
static void	ps_write_background(FILE *out);
static void	pdf_write_document(uchar *title, uchar *author, uchar *creator,
		                   uchar *copyright, tree_t *toc);
static void	pdf_write_page(FILE *out, int page, uchar *title,
		               float title_width, uchar **page_heading);
static void	pdf_write_contents(FILE *out, tree_t *toc, int parent,
		                   int prev, int next, int *heading);
static void	pdf_write_background(FILE *out);
static void	pdf_write_links(FILE *out);
static void	pdf_write_names(FILE *out);
static int	pdf_count_headings(tree_t *toc);

static void	pdf_open_stream(FILE *out);
static void	pdf_close_stream(FILE *out);
static void	pdf_puts(char *s, FILE *out);
static void	pdf_printf(FILE *out, char *format, ...);
static void	pdf_write(FILE *out, uchar *inbuf, int length);	

static void	parse_contents(tree_t *t, int left, int width, int bottom,
		               int length, float *y, int *page, int *heading);
static void	parse_doc(tree_t *t, int left, int width, int bottom, int length,
		          float *x, float *y, int *page, tree_t *cpara);
static void	parse_heading(tree_t *t, int left, int width, int bottom,
		              int length, float *x, float *y, int *page);
static void	parse_paragraph(tree_t *t, int left, int width, int bottom,
		                int length, float *x, float *y, int *page);
static void	parse_pre(tree_t *t, int left, int width, int bottom,
		          int length, float *x, float *y, int *page);
static void	parse_table(tree_t *t, int left, int width, int bottom,
		            int length, float *x, float *y, int *page);
static void	parse_list(tree_t *t, int left, int width, int bottom,
		           int length, float *x, float *y, int *page);
static void	init_list(tree_t *t);

static void	add_link(uchar *name, int page, int top);
static link_t	*find_link(uchar *name);
static int	compare_links(link_t *n1, link_t *n2);

static void	find_background(tree_t *t);
static void	write_background(FILE *out);

static render_t	*new_render(int page, int type, float x, float y,
		            float width, float height, void *data);
static void	copy_tree(tree_t *parent, tree_t *t);
static tree_t	*flatten_tree(tree_t *t);
static float	get_width(uchar *s, int typeface, int style, int size);
static void	update_image_size(tree_t *t);
static uchar	*get_title(tree_t *doc);
static FILE	*open_file(void);
static void	set_color(FILE *out, float *rgb);
static void	set_font(FILE *out, int typeface, int style, float size);
static void	set_pos(FILE *out, float x, float y);
static void	write_prolog(FILE *out, int pages, uchar *title, uchar *author,
		             uchar *creator, uchar *copyright);
static void	ps_hex(FILE *out, uchar *data, int length);
static void	ps_ascii85(FILE *out, uchar *data, int length);
#ifdef HAVE_LIBJPEG
static void	jpg_init(j_compress_ptr cinfo);
static boolean	jpg_empty(j_compress_ptr cinfo);
static void	jpg_term(j_compress_ptr cinfo);
static void	jpg_setup(FILE *out, image_t *img, j_compress_ptr cinfo);
#endif // HAVE_LIBJPEG
static int	compare_rgb(uchar *rgb1, uchar *rgb2);
static void	write_image(FILE *out, render_t *r);
static void	write_string(FILE *out, uchar *s, int compress);
static void	write_text(FILE *out, render_t *r);
static void	write_trailer(FILE *out, int pages);


/*
 * 'pdf_export()' - Export PDF file(s)...
 */

int
pdf_export(tree_t *document,	/* I - Document to export */
           tree_t *toc)		/* I - Table of contents for document */
{
  pslevel = 0;

  return (pspdf_export(document, toc));
}


/*
 * 'ps_export_level1()' - Export level 1 PostScript file(s)...
 */

int
ps_export_level1(tree_t *document,	/* I - Document to export */
                 tree_t *toc)		/* I - Table of contents for document */
{
  pslevel = 1;

  return (pspdf_export(document, toc));
}


/*
 * 'ps_export_level2()' - Export level 2 PostScript file(s)...
 */

int
ps_export_level2(tree_t *document, tree_t *toc)
{
  pslevel = 2;

  return (pspdf_export(document, toc));
}


/*
 * 'pspdf_export()' - Export PostScript/PDF file(s)...
 */

static int
pspdf_export(tree_t *document,	/* I - Document to export */
             tree_t *toc)	/* I - Table of contents for document */
{
  uchar		*title,		/* Title text */
		*author,	/* Author of document */
		*creator,	/* HTML file creator (Netscape, etc) */
		*copyright,	/* File copyright */
		*docnumber;	/* Document number */
  float		x, y,		/* Current page position */
		width,		/* Width of title, author, etc */
		height;		/* Height of title area */
  int		page,		/* Current page # */
		heading;	/* Current heading # */
  int		top, bottom;	/* Top and bottom margins... */
  image_t	*timage;	/* Title image */
  float		timage_width,	/* Title image width */
		timage_height;	/* Title image height */
  render_t	*r;		/* Rendering structure... */


 /*
  * Flush the image cache if we're not outputting color - we'll need to reload
  * some images...
  */

  if (!OutputColor)
    image_flush_cache();

 /*
  * Get the document title, author, etc...
  */

  title      = get_title(document);
  author     = htmlGetMeta(document, (uchar *)"author");
  creator    = htmlGetMeta(document, (uchar *)"generator");
  copyright  = htmlGetMeta(document, (uchar *)"copyright");
  docnumber  = htmlGetMeta(document, (uchar *)"docnumber");
  timage     = image_load(TitleImage, !OutputColor);
  logo_image = image_load(LogoImage, !OutputColor);

  if (timage != NULL)
  {
    timage_width  = timage->width * PagePrintWidth / 680.0f;
    timage_height = timage_width * timage->height / timage->width;
  };

  find_background(document);

 /*
  * Initialize page rendering variables...
  */

  memset(pages, 0, sizeof(pages));
  memset(page_headings, 0, sizeof(page_headings));
  memset(endpages, 0, sizeof(pages));
  memset(list_types, 0267, sizeof(list_types));
  memset(list_values, 0, sizeof(list_values));
  memset(chapter_starts, -1, sizeof(chapter_starts));
  memset(chapter_ends, -1, sizeof(chapter_starts));

  if (TitlePage)
    num_pages = PageDuplex ? 2 : 1;
  else
    num_pages = 0;

  if (TocLevels == 0)
  {
    chapter           = 1;
    TocDocCount       = 1;
    chapter_starts[1] = num_pages;
  }
  else
    chapter = 0;

  page            = num_pages;
  current_heading = NULL;
  num_headings    = 0;
  num_links       = 0;
  x               = 0.0;
  y               = PagePrintLength - 2 * _htmlSpacings[SIZE_P];

 /*
  * Parse the document...
  */

  bottom = 0;
  top    = PagePrintLength;

  if (strncmp(Header, "...", 3) != 0)
    top -= (int)(2.0f * HeadFootSize + 0.5f);

  if (strncmp(Footer, "...", 3) != 0)
    bottom += (int)(2.0f * HeadFootSize + 0.5f);

  parse_doc(document, 0, PagePrintWidth, bottom, top, &x, &y, &page, NULL);

  if (PageDuplex && (num_pages & 1))
    num_pages ++;
  chapter_ends[chapter] = num_pages - 1;

 /*
  * Parse the table-of-contents if necessary...
  */

  if (TocLevels > 0)
  {
    y                 = 0.0;
    page              = num_pages - 1;
    heading           = 0;
    chapter_starts[0] = num_pages;
    bottom            = 0;
    top               = PagePrintLength;

    if (strncmp(TocHeader, "...", 3) != 0)
      top -= (int)(2.0f * HeadFootSize + 0.5f);

    if (strncmp(TocFooter, "...", 3) != 0)
      bottom += (int)(2.0f * HeadFootSize + 0.5f);

    parse_contents(toc, 0, PagePrintWidth, bottom, top, &y, &page, &heading);
    if (PageDuplex && (num_pages & 1))
      num_pages ++;
    chapter_ends[0] = num_pages - 1;
  };

  if (TitlePage)
  {
   /*
    * Create a title page...
    */

    height = 0.0;

    if (timage != NULL)
      height += timage_height + _htmlSpacings[SIZE_P];
    if (title != NULL)
      height += _htmlSpacings[SIZE_H1] + _htmlSpacings[SIZE_P];
    if (author != NULL)
      height += _htmlSpacings[SIZE_P];
    if (docnumber != NULL)
      height += _htmlSpacings[SIZE_P];
    if (copyright != NULL)
      height += _htmlSpacings[SIZE_P];

    y = 0.5f * (PagePrintLength + height);

    if (timage != NULL)
    {
      r = new_render(0, RENDER_IMAGE, 0.5f * (PagePrintWidth - timage_width),
                     y - timage_height, timage_width, timage_height, timage);
      y -= timage->height + _htmlSpacings[SIZE_P];
    };

    if (title != NULL)
    {
      width = get_width(title, _htmlHeadingFont, STYLE_BOLD, SIZE_H1);
      r     = new_render(0, RENDER_TEXT, (PagePrintWidth - width) * 0.5f,
                	 y - _htmlSpacings[SIZE_H1], width,
			 _htmlSizes[SIZE_H1], title);

      r->data.text.typeface = _htmlHeadingFont;
      r->data.text.style    = STYLE_BOLD;
      r->data.text.size     = _htmlSizes[SIZE_H1];

      y -= _htmlSpacings[SIZE_H1];

      if (docnumber != NULL)
      {
	width = get_width(docnumber, _htmlBodyFont, STYLE_NORMAL, SIZE_P);
	r     = new_render(0, RENDER_TEXT, (PagePrintWidth - width) * 0.5f,
                           y - _htmlSpacings[SIZE_P], width,
			   _htmlSizes[SIZE_P], docnumber);

	r->data.text.typeface = _htmlBodyFont;
	r->data.text.style    = STYLE_NORMAL;
	r->data.text.size     = _htmlSizes[SIZE_P];

	y -= _htmlSpacings[SIZE_P];
      };

      y -= _htmlSpacings[SIZE_P];
    };

    if (author != NULL)
    {
      width = get_width(author, _htmlBodyFont, STYLE_NORMAL, SIZE_P);
      r     = new_render(0, RENDER_TEXT, (PagePrintWidth - width) * 0.5f,
                	 y - _htmlSpacings[SIZE_P], width, _htmlSizes[SIZE_P],
			 author);

      r->data.text.typeface = _htmlBodyFont;
      r->data.text.style    = STYLE_NORMAL;
      r->data.text.size     = _htmlSizes[SIZE_P];

      y -= _htmlSpacings[SIZE_P];
    };

    if (copyright != NULL)
    {
      width = get_width(copyright, _htmlBodyFont, STYLE_NORMAL, SIZE_P);
      r     = new_render(0, RENDER_TEXT, (PagePrintWidth - width) * 0.5f,
                	 y - _htmlSpacings[SIZE_P], width, _htmlSizes[SIZE_P],
			 copyright);

      r->data.text.typeface = _htmlBodyFont;
      r->data.text.style    = STYLE_NORMAL;
      r->data.text.size     = _htmlSizes[SIZE_P];
    };
  };

 /*
  * Write the document to disk...
  */

  if (pslevel > 0)
    ps_write_document(title, author, creator, copyright);
  else
    pdf_write_document(title, author, creator, copyright, toc);

  if (title != NULL)
    free(title);

  return (0);
}


/*
 * 'pspdf_prepare_page()' - Add headers/footers to page before writing...
 */

static char *
pspdf_prepare_page(int   page,			/* I - Page number */
                   int   *file_page,		/* O - File page number */
        	   uchar *title,		/* I - Title string */
        	   float title_width,		/* I - Width of title string */
        	   uchar **page_heading)	/* IO - Page heading string */
{
  int		ord_page;	/* Ordinal page # in file */
  int		print_page;	/* Printed page # */
  float		width;		/* Width of page heading */
  static char	*page_text;	/* Page number text */


  if (chapter == 0 || OutputFiles)
  {
    *file_page = page - chapter_starts[chapter] + 1;

    if (chapter == 0)
      ord_page = *file_page + PageDuplex + 1;
    else
      ord_page = *file_page;
  }
  else if (chapter == -1 || TocLevels <= 0)
    *file_page = ord_page = page + 1;
  else if (PageDuplex)
  {
    *file_page = page - 1;
    ord_page   = *file_page + chapter_ends[0] - chapter_starts[0] + 2;
  }
  else
  {
    *file_page = page;
    ord_page   = *file_page + chapter_ends[0] - chapter_starts[0] + 2;
  };

  if (Verbosity)
  {
    progress_show("Writing page %d...", ord_page);
    progress_update(100 * page / num_pages);
  };

 /*
  * Get the new heading if necessary...
  */

  if (page_headings[page] != NULL)
    *page_heading = page_headings[page];

 /*
  * Make a page number; use roman numerals for the table of contents
  * and arabic numbers for all others...
  */

  if (chapter == 0)
  {
    print_page = *file_page;
    page_text  = format_number(print_page, 'i');
  }
  else if (chapter < 0)
    page_text = "Title-Page";
  else if (TitlePage)
  {
    print_page = page - PageDuplex;
    page_text  = format_number(print_page, '1');
  }
  else
  {
    print_page = page + 1;
    page_text  = format_number(print_page + 1, '1');
  };

 /*
  * Add page headings...
  */

  width = get_width(*page_heading, TYPE_HELVETICA, STYLE_NORMAL, SIZE_P);

  if (chapter == 0)
  {
   /*
    * Add table-of-contents header & footer...
    */

    pspdf_prepare_heading(page, print_page, title, title_width, *page_heading,
                          width, TocHeader, PagePrintLength);
    pspdf_prepare_heading(page, print_page, title, title_width, *page_heading,
                          width, TocFooter, 0);
  }
  else if (chapter > 0)
  {
   /*
    * Add chapter header & footer...
    */

    if (page > chapter_starts[chapter] || TocLevels == 0)
      pspdf_prepare_heading(page, print_page, title, title_width, *page_heading,
                            width, Header, PagePrintLength);
    pspdf_prepare_heading(page, print_page, title, title_width, *page_heading,
                          width, Footer, 0);
  };

  return (page_text);
}


/*
 * 'pspdf_prepare_heading()' - Add headers/footers to page before writing...
 */

static void
pspdf_prepare_heading(int   page,		/* I - Page number */
                      int   print_page,         /* I - Printed page number */
        	      uchar *title,		/* I - Title string */
        	      float title_width,	/* I - Width of title string */
        	      uchar *heading,		/* I - Page heading string */
		      float heading_width,	/* I - Width of heading */
		      char  *format,		/* I - Format of heading */
		      int   y)			/* I - Baseline of heading */
{
  int		pos,		/* Position in heading */
		dir;		/* Direction of page */
  char		*number;	/* Page number */
  render_t	*temp;		/* Render structure for titles, etc. */


 /*
  * Return right away if there is nothing to do...
  */

  if (strncmp(format, "...", 3) == 0)
    return;

 /*
  * Add page headings...
  */

  if (PageDuplex && (page & 1))
  {
    dir    = -1;
    format += 2;
  }
  else
    dir = 1;

  for (pos = 0; pos < 3; pos ++, format += dir)
  {
   /*
    * Add the appropriate object...
    */

    switch (*format)
    {
      case '.' :
      default :
          temp = NULL;
	  break;

      case '1' :
      case 'i' :
      case 'I' :
      case 'a' :
      case 'A' :
          number = format_number(print_page, *format);
	  temp   = new_render(page, RENDER_TEXT, 0, y,
                              HeadFootSize / _htmlSizes[SIZE_P] *
			      get_width((uchar *)number, HeadFootFont,
			                STYLE_NORMAL, SIZE_P),
			      HeadFootSize, number);
          break;

      case 't' :
          if (title != NULL)
	    temp = new_render(page, RENDER_TEXT, 0, y, title_width,
	                      HeadFootSize, title);
          else
	    temp = NULL;
          break;

      case 'h' :
          if (heading != NULL)
	    temp = new_render(page, RENDER_TEXT, 0, y, heading_width,
	                      HeadFootSize, heading);
          else
	    temp = NULL;
          break;

      case 'l' :
          if (logo_image != NULL)
	    temp = new_render(page, RENDER_IMAGE, 0, y,
	                      HeadFootSize * logo_image->width / logo_image->height,
	                      HeadFootSize, logo_image);
	  else
	    temp = NULL;
	  break;
    };	

    if (temp == NULL)
      continue;

   /*
    * Justify the object...
    */

    switch (pos)
    {
      case 0 : /* Left justified */
          break;
      case 1 : /* Centered */
          temp->x = (PagePrintWidth - temp->width) * 0.5;
          break;
      case 2 : /* Right justified */
          temp->x = PagePrintWidth - temp->width;
          break;
    };

   /*
    * Set the text font...
    */

    if (temp->type == RENDER_TEXT)
    {
      temp->data.text.typeface = HeadFootFont;
      temp->data.text.style    = STYLE_NORMAL;
      temp->data.text.size     = HeadFootSize;
    };
  };
}


/*
 * 'ps_write_document()' - Write all render entities to PostScript file(s).
 */

static void
ps_write_document(uchar *title,		/* I - Title on all pages */
        	  uchar *author,		/* I - Author of document */
        	  uchar *creator,	/* I - Application that generated the HTML file */
        	  uchar *copyright)	/* I - Copyright (if any) on the document */
{
  uchar		*page_heading;	/* Current heading text */
  FILE		*out;		/* Output file */
  int		page;		/* Current page # */
  float		title_width;	/* Width of title string */


 /*
  * Get the document title width...
  */

  if (title != NULL)
    title_width = HeadFootSize / _htmlSizes[SIZE_P] *
                  get_width(title, HeadFootFont, STYLE_NORMAL, SIZE_P);

 /*
  * Write the title page(s)...
  */

  chapter      = -1;
  page_heading = NULL;
  out          = open_file();

  if (out == NULL)
    return; /**** NEED TO ADD ERROR MESSAGE HOOK ****/

  write_prolog(out, 0, title, author, creator, copyright);
  if (TitlePage)
  {
    ps_write_page(out, 0, NULL, 0.0, &page_heading);
    if (PageDuplex)
      ps_write_page(out, 1, NULL, 0.0, &page_heading);
  };

  if (TocLevels > 0)
    chapter = 0;
  else
    chapter = 1;

  for (; chapter <= TocDocCount; chapter ++)
  {
    if (chapter_starts[chapter] < 0)
      continue;

    if (OutputFiles)
    {
      out = open_file();
      if (out == NULL)
        return; /**** NEED TO ADD ERROR MESSAGE HOOK ****/

      write_prolog(out, 0, title, author, creator, copyright);
    };

    for (page = chapter_starts[chapter], page_heading = NULL;
         page <= chapter_ends[chapter];
         page ++)
      ps_write_page(out, page, title, title_width, &page_heading);

   /*
    * Close the output file as necessary...
    */

    if (OutputFiles)
    {
      write_trailer(out, chapter_ends[chapter] - chapter_starts[chapter] + 1);
      fclose(out);
    };
  };

 /*
  * Close the output file as necessary...
  */

  if (!OutputFiles)
  {
    write_trailer(out, num_pages);
    if (out != stdout)
      fclose(out);
  };

  if (Verbosity)
    progress_hide();
}


/*
 * 'ps_write_page()' - Write all render entities on a page to a PostScript file.
 */

static void
ps_write_page(FILE  *out,		/* I - Output file */
              int   page,		/* I - Page number */
              uchar *title,		/* I - Title string */
              float title_width,	/* I - Width of title string */
              uchar **page_heading)	/* IO - Page heading string */
{
  int		file_page;	/* Current page # in document */
  char		*page_text;	/* Page number text */
  render_t	*r,		/* Render pointer */
		*next;		/* Next render */


  if (page < 0 || page >= MAX_PAGES)
    return;

 /*
  * Add headers/footers as needed...
  */

  page_text = pspdf_prepare_page(page, &file_page, title, title_width,
                                 page_heading);

 /*
  * Clear the render cache...
  */

  render_typeface = -1;
  render_style    = -1;
  render_size     = -1;
  render_rgb[0]   = 0.0;
  render_rgb[1]   = 0.0;
  render_rgb[2]   = 0.0;
  render_x        = -1.0;
  render_y        = -1.0;

 /*
  * Output the page prolog...
  */

  fprintf(out, "%%%%Page: %s %d\n", page_text, file_page);
  fputs("GS\n", out);

  write_background(out);

  if (PageDuplex && (page & 1))
    fprintf(out, "%d %d T\n", PageRight, PageBottom);
  else
    fprintf(out, "%d %d T\n", PageLeft, PageBottom);

 /*
  * Render all page elements, freeing used memory as we go...
  */

  for (r = pages[page], next = NULL; r != NULL; r = next)
  {
    switch (r->type)
    {
      case RENDER_IMAGE :
          write_image(out, r);
          break;
      case RENDER_TEXT :
          write_text(out, r);
          break;
      case RENDER_BOX :
          render_x = -1.0;
          render_y = -1.0;
          set_color(out, r->data.box);
          set_pos(out, r->x, r->y);
          if (r->height > 0.0)
            fprintf(out, " %.1f %.1f B\n", r->width, r->height);
          else
            fprintf(out, " %.1f L\n", r->width);
          render_x += r->width;
          break;
      case RENDER_FBOX :
          render_x = -1.0;
          render_y = -1.0;
          set_color(out, r->data.box);
          set_pos(out, r->x, r->y);
          if (r->height > 0.0)
            fprintf(out, " %.1f %.1f F\n", r->width, r->height);
          else
            fprintf(out, " %.1f L\n", r->width);
          render_x += r->width;
          break;
    };

    next = r->next;
    free(r);
  };

 /*
  * Output the page trailer...
  */

  fputs("GR\n", out);
  fputs("SP\n", out);
  fputs("%%EndPage\n", out);
}


/*
 * 'ps_write_background()' - Write a background image...
 */

static void
ps_write_background(FILE *out)		/* I - Output file */
{
  int	y,				/* Current line */
	pwidth;				/* Pixel width */


  pwidth = background_image->width * background_image->depth;

  fputs("/BG[", out);
  for (y = 0; y < background_image->height; y ++)
  {
    putc('<', out);
    ps_hex(out, background_image->pixels + y * pwidth, pwidth);
    putc('>', out);
  };
  fputs("]def", out);
}


/*
 * 'pdf_write_document()' - Write all render entities to a PDF file.
 */

static void
pdf_write_document(uchar   *title,	/* I - Title for all pages */
        	   uchar   *author,	/* I - Author of document */
        	   uchar   *creator,	/* I - Application that generated the HTML file */
        	   uchar   *copyright,	/* I - Copyright (if any) on the document */
                   tree_t *toc)		/* I - Table of contents tree */
{
  uchar		*page_heading;	/* Current heading text */
  FILE		*out;		/* Output file */
  int		i,		/* Looping var */
		page,		/* Current page # */
		heading;	/* Current heading # */
  float		title_width;	/* Width of title string */


  if (title != NULL)
    title_width = HeadFootSize / _htmlSizes[SIZE_P] *
                  get_width(title, HeadFootFont, STYLE_NORMAL, SIZE_P);

  out = open_file();
  if (out == NULL)
    return; /**** NEED TO ADD ERROR MESSAGE HOOK ****/

  write_prolog(out, 0, title, author, creator, copyright);

  pdf_write_links(out);
  pdf_write_names(out);

  num_objects ++;
  if (pages_object != num_objects)
    puts("ERROR pages_object != num_objects");

  objects[num_objects] = ftell(out);
  fprintf(out, "%d 0 obj", num_objects);
  fputs("<<", out);
  fputs("/Type/Pages", out);
  fprintf(out, "/MediaBox[0 0 %d %d]", PageWidth, PageLength);

  fputs("/Resources<<", out);
  if (OutputColor)
    fputs("/ProcSet[/PDF/Text/ImageB/ImageC]", out);
  else
    fputs("/ProcSet[/PDF/Text/ImageB]", out);
  fputs("/Font<<", out);
  for (i = 0; i < 16; i ++)
    fprintf(out, "/%s %d 0 R", font_names[i], font_objects[i]);
  fputs(">>", out);
  if (background_object > 0)
    fprintf(out, "/XObject<</BG %d 0 R>>", background_object);
  fputs(">>", out);

  fprintf(out, "/Count %d", num_pages);
  fputs("/Kids[", out);

  if (TitlePage)
  {
    fprintf(out, "%d 0 R\n", pages_object + 1);
    if (PageDuplex)
      fprintf(out, "%d 0 R\n", pages_object + 4);
  };

  if (TocLevels > 0)
    chapter = 0;
  else
    chapter = 1;

  for (; chapter <= TocDocCount; chapter ++)
    for (page = chapter_starts[chapter]; page <= chapter_ends[chapter]; page ++)
      if (page < MAX_PAGES)
        fprintf(out, "%d 0 R\n", pages_object + 3 * page + 1);
  fputs("]", out);
  fputs(">>", out);
  fputs("endobj\n", out);

  page_heading = NULL;
  chapter      = -1;

  if (TitlePage)
  {
    pdf_write_page(out, 0, NULL, 0.0, &page_heading);
    if (PageDuplex)
      pdf_write_page(out, 1, NULL, 0.0, &page_heading);
  };

  for (chapter = 1; chapter <= TocDocCount; chapter ++)
  {
    if (chapter_starts[chapter] < 0)
      continue;

    for (page = chapter_starts[chapter], page_heading = NULL;
         page <= chapter_ends[chapter];
         page ++)
      pdf_write_page(out, page, title, title_width, &page_heading);
  };

  if (TocLevels > 0)
  {
    for (chapter = 0, page = chapter_starts[0], page_heading = NULL;
	 page <= chapter_ends[0];
	 page ++)
      pdf_write_page(out, page, title, title_width, &page_heading);

   /*
    * Write the outline tree...
    */

    heading = 0;
    pdf_write_contents(out, toc, 0, 0, 0, &heading);
  }
  else
    outline_object = 0;

 /*
  * Write the trailer and close the output file...
  */

  write_trailer(out, num_pages);

#ifdef MAC
  if (out != stdout)
  {
    //
    // On the MacOS, files are not associated with applications by extensions.
    // Instead, it uses a pair of values called the type & creator.  
    // This block of code sets the those values for PDF files.
    //

    FCBPBRec    fcbInfo;	// File control block information
    Str32	name;		// Name of file
    FInfo	fInfo;		// File type/creator information
    FSSpec	fSpec;		// File specification


    memset(&fcbInfo, 0, sizeof(FCBPBRec));
    fcbInfo.ioRefNum  = out->handle;
    fcbInfo.ioNamePtr = name;
    if (!PBGetFCBInfoSync(&fcbInfo))
      if (FSMakeFSSpec(fcbInfo.ioFCBVRefNum, fcbInfo.ioFCBParID, name, &fSpec) == noErr)
      {
	FSpGetFInfo(&fSpec, &fInfo);
	fInfo.fdType    = 'PDF ';
	fInfo.fdCreator = 'CARO';
	FSpSetFInfo(&fSpec, &fInfo);
      };

    //
    // Now that the PDF file is associated with that type, close the file.
    //

    fclose(out);
  };
#else
  if (out != stdout)
    fclose(out);
#endif // MAC

  if (Verbosity)
    progress_hide();
}


/*
 * 'pdf_write_page()' - Write a page to a PDF file.
 */

static void
pdf_write_page(FILE  *out,		/* I - Output file */
               int   page,		/* I - Page number */
               uchar  *title,		/* I - Title string */
               float title_width,	/* I - Width of title string */
               uchar  **page_heading)	/* IO - Page heading string */
{
  int		file_page,	/* Current page # in file */
		length,		/* Stream length */
		last_render;	/* Last type of render */
  render_t	*r,		/* Render pointer */
		*next;		/* Next render */


  if (page < 0 || page >= MAX_PAGES)
    return;

 /*
  * Add headers/footers as needed...
  */

  pspdf_prepare_page(page, &file_page, title, title_width, page_heading);

 /*
  * Clear the render cache...
  */

  render_typeface = -1;
  render_style    = -1;
  render_size     = -1;
  render_rgb[0]   = 0.0;
  render_rgb[1]   = 0.0;
  render_rgb[2]   = 0.0;
  render_x        = -1.0;
  render_y        = -1.0;

 /*
  * Output the page prolog...
  */

  num_objects ++;
  objects[num_objects] = ftell(out);

  fprintf(out, "%d 0 obj", num_objects);
  fputs("<<", out);
  fputs("/Type/Page", out);
  fprintf(out, "/Parent %d 0 R", pages_object);
  fprintf(out, "/Contents %d 0 R", num_objects + 1);

 /*
  * Actions (links)...
  */

  if (annots_objects[page] > 0)
    fprintf(out, "/Annots %d 0 R", annots_objects[page]);

  fputs(">>", out);
  fputs("endobj\n", out);

  num_objects ++;
  objects[num_objects] = ftell(out);

  fprintf(out, "%d 0 obj", num_objects);
  fputs("<<", out);
  fprintf(out, "/Length %d 0 R", num_objects + 1);
#ifdef HAVE_LIBZ
  if (Compression)
    fputs("/Filter/Fl", out);
#endif /* HAVE_LIBZ */
  fputs(">>", out);
  fputs("stream\n", out);

  length = ftell(out);

  pdf_open_stream(out);

  pdf_puts("q\n", out);
  write_background(out);

  if (PageDuplex && (page & 1))
    pdf_printf(out, "1 0 0 1 %d %d cm\n", PageRight, PageBottom);
  else
    pdf_printf(out, "1 0 0 1 %d %d cm\n", PageLeft, PageBottom);

  pdf_puts("BT\n", out);
  last_render = RENDER_TEXT;

 /*
  * Render all page elements, freeing used memory as we go...
  */

  for (r = pages[page], next = NULL; r != NULL; r = next)
  {
    if (r->type != last_render)
    {
      if (r->type == RENDER_TEXT)
      {
	render_x = -1.0;
	render_y = -1.0;
        pdf_puts("BT\n", out);
      }
      else if (last_render == RENDER_TEXT)
        pdf_puts("ET\n", out);

      last_render = r->type;
    };

    switch (r->type)
    {
      case RENDER_IMAGE :
          write_image(out, r);
          break;
      case RENDER_TEXT :
          write_text(out, r);
          break;
      case RENDER_BOX :
          if (r->height == 0.0)
            pdf_printf(out, "%.2f %.2f %.2f RG %.1f %.1f m %.1f %.1f l S\n",
                    r->data.box[0], r->data.box[1], r->data.box[2],
                    r->x, r->y, r->x + r->width, r->y);
          else
            pdf_printf(out, "%.2f %.2f %.2f RG %.1f %.1f %.1f %.1f re S\n",
                    r->data.box[0], r->data.box[1], r->data.box[2],
                    r->x, r->y, r->width, r->height);
          break;
      case RENDER_FBOX :
          if (r->height == 0.0)
            pdf_printf(out, "%.2f %.2f %.2f RG %.1f %.1f m %.1f %.1f l S\n",
                    r->data.box[0], r->data.box[1], r->data.box[2],
                    r->x, r->y, r->x + r->width, r->y);
          else
          {
            set_color(out, r->data.fbox);
            pdf_printf(out, "%.1f %.1f %.1f %.1f re f\n",
                    r->x, r->y, r->width, r->height);
          };
          break;
    };

    next = r->next;
    free(r);
  };

 /*
  * Output the page trailer...
  */

  if (last_render == RENDER_TEXT)
   pdf_puts("ET\n", out);

  pdf_puts("Q\n", out);
  pdf_close_stream(out);
  length = ftell(out) - length;
  fputs("endstream\n", out);
  fputs("endobj\n", out);

  num_objects ++;
  objects[num_objects] = ftell(out);

  fprintf(out, "%d 0 obj\n", num_objects);
  fprintf(out, "%d\n", length);
  fputs("endobj\n", out);
}


/*
 * 'pdf_write_contents()' - Write the table of contents as outline records to
 *                          a PDF file.
 */

static void
pdf_write_contents(FILE   *out,			/* I - Output file */
                   tree_t *toc,			/* I - Table of contents tree */
                   int    parent,		/* I - Parent outline object */
                   int    prev,			/* I - Previous outline object */
                   int    next,			/* I - Next outline object */
                   int    *heading)		/* IO - Current heading # */
{
  int		i,				/* Looping var */
		thisobj,			/* This object */
		entry,				/* TOC entry object */
		count;				/* Number of entries at this level */
  uchar		*text;				/* Entry text */
  tree_t	*temp;				/* Looping var */
  int		entry_counts[MAX_HEADINGS],	/* Number of sub-entries for this entry */
		entry_objects[MAX_HEADINGS];	/* Objects for each entry */
  tree_t	*entries[MAX_HEADINGS];		/* Pointers to each entry */


 /*
  * Make an object for this entry...
  */

  num_objects ++;
  thisobj = num_objects;

  if (toc == NULL)
  {
   /*
    * This is for the Table of Contents page...
    */

    objects[thisobj] = ftell(out);
    fprintf(out, "%d 0 obj", thisobj);
    fputs("<<", out);
    fprintf(out, "/Parent %d 0 R", parent);

    fputs("/Title(Table of Contents)", out);
    fprintf(out, "/Dest[%d 0 R/XYZ null %d null]",
            pages_object + 3 * chapter_starts[0] + 1,
            PagePrintLength + PageBottom);

    if (prev > 0)
      fprintf(out, "/Prev %d 0 R", prev);

    if (next > 0)
      fprintf(out, "/Next %d 0 R", next);

    fputs(">>", out);
    fputs("endobj\n", out);
  }
  else
  {
   /*
    * Find and count the children (entries)...
    */

    if (toc->markup == MARKUP_B || toc->markup == MARKUP_LI)
    {
      if (toc->next != NULL && toc->next->markup == MARKUP_UL)
	temp = toc->next->child;
      else
	temp = NULL;
    }
    else
      temp = toc->child;

    if (parent == 0 && TocLevels > 0)
    {
     /*
      * Add the table of contents to the top-level contents...
      */

      entries[0]       = NULL;
      entry_objects[0] = thisobj + 1;
      entry            = thisobj + 2;
      count            = 1;
    }
    else
    {
      entry = thisobj + 1;
      count = 0;
    };

    for (; temp != NULL && count < MAX_HEADINGS; temp = temp->next)
      if (temp->markup == MARKUP_B || temp->markup == MARKUP_LI)
      {
	entries[count]       = temp;
	entry_objects[count] = entry;
	if (temp->next != NULL && temp->next->markup == MARKUP_UL)
          entry_counts[count] = pdf_count_headings(temp->next->child);
	else
          entry_counts[count] = 0;
	entry += entry_counts[count] + 1;
	count ++;
      };

   /*
    * Output the top-level object...
    */

    objects[thisobj] = ftell(out);
    fprintf(out, "%d 0 obj", thisobj);
    fputs("<<", out);
    if (parent == 0)
      outline_object = thisobj;
    else
      fprintf(out, "/Parent %d 0 R", parent);

    if (count > 0)
    {
      fprintf(out, "/Count %d", parent == 0 ? count : -count);
      fprintf(out, "/First %d 0 R", entry_objects[0]);
      fprintf(out, "/Last %d 0 R", entry_objects[count - 1]);
    };

    if (toc->markup == MARKUP_B || toc->markup == MARKUP_LI)
    {
      if ((text = htmlGetText(toc->child)) != NULL)
      {
	fputs("/Title", out);
	write_string(out, text, 0);
	free(text);
      };

      if (heading_pages[*heading] > 0)
	fprintf(out, "/Dest[%d 0 R/XYZ null %d null]",
        	pages_object + 3 * heading_pages[*heading] + ((PageDuplex && TitlePage) ? 4 : 1),
        	heading_tops[*heading]);

      (*heading) ++;
    };

    if (prev > 0)
      fprintf(out, "/Prev %d 0 R", prev);

    if (next > 0)
      fprintf(out, "/Next %d 0 R", next);

    fputs(">>", out);
    fputs("endobj\n", out);

    for (i = 0; i < count ; i ++)
      pdf_write_contents(out, entries[i], thisobj, i > 0 ? entry_objects[i - 1] : 0,
                	 i < (count - 1) ? entry_objects[i + 1] : 0,
                	 heading);
  };
}


/*
 * 'pdf_count_headings()' - Count the number of headings under this TOC
 *                          entry.
 */

static int			/* O - Number of headings found */
pdf_count_headings(tree_t *toc)	/* I - TOC entry */
{
  int	headings;		/* Number of headings */


  for (headings = 0; toc != NULL; toc = toc->next)
    if (toc->markup == MARKUP_B || toc->markup == MARKUP_LI)
      headings ++;
    else if (toc->markup == MARKUP_UL && toc->child != NULL)
      headings += pdf_count_headings(toc->child);

  return (headings);
}


/*
 * 'pdf_write_background()' - Write a background image...
 */

static void
pdf_write_background(FILE *out)		/* I - Output file */
{
  int	length;				/* Length of image */


  num_objects ++;
  background_object = num_objects;
  objects[num_objects] = ftell(out);

  fprintf(out, "%d 0 obj", num_objects);
  fputs("<<", out);
  fputs("/Type/XObject", out);
  fputs("/Subtype/Image", out);
  fputs("/Name/BG", out);
  if (background_image->depth == 1)
    fputs("/ColorSpace/DeviceGray", out);
  else
    fputs("/ColorSpace/DeviceRGB", out);
  fputs("/Interpolate true", out);
  fprintf(out, "/Width %d/Height %d/BitsPerComponent 8",
      	  background_image->width, background_image->height); 
  fprintf(out, "/Length %d 0 R", num_objects + 1);
#ifdef HAVE_LIBZ
  if (Compression)
    fputs("/Filter/Fl", out);
#endif /* HAVE_LIBZ */
  fputs(">>", out);
  fputs("stream\n", out);

  length = ftell(out);

  pdf_open_stream(out);
  pdf_write(out, background_image->pixels,
            background_image->width * background_image->height *
	    background_image->depth);
  pdf_close_stream(out);

  length = ftell(out) - length;
  fputs("endstream\n", out);
  fputs("endobj\n", out);

  num_objects ++;
  objects[num_objects] = ftell(out);

  fprintf(out, "%d 0 obj\n", num_objects);
  fprintf(out, "%d\n", length);
  fputs("endobj\n", out);

}


/*
 * 'pdf_write_links()' - Write annotation link objects for each page in the
 *                       document.
 */

static void
pdf_write_links(FILE *out)		/* I - Output file */
{
  int		page,			/* Current page */
		lobj,			/* Current link */
		num_lobjs,		/* Number of links on this page */
		lobjs[2 * MAX_LINKS];	/* Link objects */
  uchar		*filename;		/* Filename */
  render_t	*r;			/* Current render primitive */
  link_t	*link;			/* Local link */


 /*
  * First figure out how many link objects we'll have...
  */

  pages_object = num_objects + 1;

  for (page = 0; page < num_pages; page ++)
  {
    num_lobjs = 0;

    for (r = pages[page]; r != NULL; r = r->next)
      if (r->type == RENDER_LINK)
      {
        if (r->data.link[0] == '#')
        {
          if (find_link(r->data.link + 1) != NULL)
            num_lobjs ++;
        }
        else
          num_lobjs += 2;
      };

    if (num_lobjs > 0)
      pages_object += num_lobjs + 1;
  };

  pages_object += num_links + 3;

 /*
  * Then generate annotation objects for all the links...
  */

  memset(annots_objects, 0, sizeof(annots_objects));

  for (page = 0; page < num_pages; page ++)
  {
    num_lobjs = 0;

    for (r = pages[page]; r != NULL; r = r->next)
      if (r->type == RENDER_LINK)
      {
	if (r->data.link[0] == '#')
	{
	 /*
          * Local link...
          */

          if ((link = find_link(r->data.link + 1)) == NULL)
            continue;

          num_objects ++;
          lobjs[num_lobjs ++] = num_objects;
          objects[num_objects] = ftell(out);

          fprintf(out, "%d 0 obj", num_objects);
          fputs("<<", out);
          fputs("/Subtype/Link", out);
          if (PageDuplex && (page & 1))
            fprintf(out, "/Rect[%.1f %.1f %.1f %.1f]",
                    r->x + PageRight, r->y + PageBottom - 2,
                    r->x + r->width + PageRight, r->y + r->height + PageBottom);
          else
            fprintf(out, "/Rect[%.1f %.1f %.1f %.1f]",
                    r->x + PageLeft, r->y + PageBottom - 2,
                    r->x + r->width + PageLeft, r->y + r->height + PageBottom);
          fputs("/Border[0 0 0]", out);
	  fprintf(out, "/Dest[%d 0 R/XYZ null %d 0]",
        	  pages_object + 3 * link->page + 4,
        	  link->top);
          fputs(">>", out);
          fputs("endobj\n", out);
	}
	else
	{
	 /*
          * Remote link...
          */

          num_objects ++;
          objects[num_objects] = ftell(out);

          fprintf(out, "%d 0 obj", num_objects);
          fputs("<<", out);
	  if (strncmp((char *)r->data.link, "file:", 5) == 0 &&
	      strlen((char *)r->data.link) > 4 &&
	      strcasecmp((char *)r->data.link + strlen((char *)r->data.link) - 4,
	                 ".pdf") == 0)
	  {
	   /*
	    * Link to external PDF file...
	    */

            fputs("/S/GoToR", out);
            fputs("/D[0/XYZ null null 0]", out);
            fputs("/F(", out);

	    for (filename = r->data.link + 5; *filename != '\0'; filename ++)
	      switch (*filename)
	      {
	        case '\\' :
		    putc('/', out);
		    break;
		case ')' :
		    putc('\\', out);
		default :
		    putc(*filename, out);
		    break;
	      };

            putc(')', out);
	  }
	  else
	  {
	   /*
	    * Link to web file...
	    */

            fputs("/S/URI", out);
            fputs("/URI(", out);

	    for (filename = r->data.link; *filename != '\0'; filename ++)
	      switch (*filename)
	      {
	        case '\\' :
		    putc('/', out);
		    break;
		case ')' :
		    putc('\\', out);
		default :
		    putc(*filename, out);
		    break;
	      };

            putc(')', out);
	  };

          fputs(">>", out);
          fputs("endobj\n", out);

          num_objects ++;
          lobjs[num_lobjs ++] = num_objects;
          objects[num_objects] = ftell(out);

          fprintf(out, "%d 0 obj\n", num_objects);
          fputs("<<", out);
          fputs("/Subtype/Link", out);
          if (PageDuplex && (page & 1))
            fprintf(out, "/Rect[%.1f %.1f %.1f %.1f]",
                    r->x + PageRight, r->y + PageBottom,
                    r->x + r->width + PageRight, r->y + r->height + PageBottom);
          else
            fprintf(out, "/Rect[%.1f %.1f %.1f %.1f]",
                    r->x + PageLeft, r->y + PageBottom - 2,
                    r->x + r->width + PageLeft, r->y + r->height + PageBottom);
          fputs("/Border[0 0 0]", out);
	  fprintf(out, "/A %d 0 R", num_objects - 1);
          fputs(">>", out);
          fputs("endobj\n", out);
	};
      };

    if (num_lobjs > 0)
    {
      num_objects ++;
      annots_objects[page] = num_objects;
      objects[num_objects] = ftell(out);

      fprintf(out, "%d 0 obj", num_objects);
      fputs("[", out);
      for (lobj = 0; lobj < num_lobjs; lobj ++)
        fprintf(out, "%d 0 R\n", lobjs[lobj]);
      fputs("]", out);
      fputs("endobj\n", out);
    };
  };
}


/*
 * 'pdf_write_names()' - Write named destinations for each link.
 */

static void
pdf_write_names(FILE *out)		/* I - Output file */
{
  int		i;			/* Looping var */
  uchar		*s;			/* Current character in name */
  link_t	*link;			/* Local link */


 /*
  * Convert all link names to lowercase...
  */

  for (i = num_links, link = links; i > 0; i --, link ++)
    for (s = link->name; *s != '\0'; s ++)
      *s = tolower(*s);

 /*
  * Write the root name tree entry...
  */

  num_objects ++;
  names_object = num_objects;
  objects[num_objects] = ftell(out);

  fprintf(out, "%d 0 obj", num_objects);
  fputs("<<", out);
  fprintf(out, "/Dests %d 0 R", num_objects + 1);
  fputs(">>", out);
  fputs("endobj\n", out);

 /*
  * Write the name tree child list...
  */

  num_objects ++;
  objects[num_objects] = ftell(out);

  fprintf(out, "%d 0 obj", num_objects);
  fputs("<<", out);
  fprintf(out, "/Kids[%d 0 R]", num_objects + 1);
  fputs(">>", out);
  fputs("endobj\n", out);

 /*
  * Write the leaf node for the name tree...
  */

  num_objects ++;
  objects[num_objects] = ftell(out);

  fprintf(out, "%d 0 obj", num_objects);
  fputs("<<", out);

  fputs("/Limits[", out);
  write_string(out, links[0].name, 0);
  write_string(out, links[num_links - 1].name, 0);
  fputs("]", out);

  fputs("/Names[", out);
  for (i = 1, link = links; i <= num_links; i ++, link ++)
  {
    write_string(out, link->name, 0);
    fprintf(out, "%d 0 R", num_objects + i);
  };
  fputs("]", out);

  fputs(">>", out);
  fputs("endobj\n", out);

  for (i = num_links, link = links; i > 0; i --, link ++)
  {
    num_objects ++;
    objects[num_objects] = ftell(out);

    fprintf(out, "%d 0 obj", num_objects);
    fputs("<<", out);
    fprintf(out, "/D[%d 0 R/XYZ null %d null]", 
            pages_object + 3 * link->page + ((TocLevels > 0 && PageDuplex) ? 4 : 1),
            link->top);
    fputs(">>", out);
    fputs("endobj\n", out);
  };

}


/*
 * 'parse_contents()' - Parse the table of contents and produce a
 *                      rendering list...
 */

static void
parse_contents(tree_t *t,		/* I - Tree to parse */
               int    left,		/* I - Left margin */
               int    right,		/* I - Printable width */
               int    bottom,		/* I - Bottom margin */
               int    top,		/* I - Printable top */
               float  *y,		/* IO - Y position */
               int    *page,		/* IO - Page # */
               int    *heading)		/* IO - Heading # */
{
  float		x,
		width,
		height,
		rgb[3];
  uchar		number[255],
		*nptr,
		*link;
  tree_t	*flat,
		*temp,
		*next;
  render_t	*r;
  static char	*contents = CONTENTS;
#define dot_width  (_htmlSizes[SIZE_P] * char_widths[t->typeface][t->style]['.'])


  DEBUG_printf(("parse_contents(t=%08x, y=%.1f, page=%d, heading=%d)\n",
                t, *y, *page, *heading));

  while (t != NULL)
  {
    switch (t->markup)
    {
      case MARKUP_B :	/* Top-level TOC */
          if (t->prev != NULL)	/* Advance one line prior to top-levels... */
            *y -= _htmlSpacings[SIZE_P];

      case MARKUP_LI :	/* Lower-level TOC */
          DEBUG_printf(("parse_contents: heading=%d, page = %d\n", *heading,
                        heading_pages[*heading]));

         /*
          * Put the text...
          */

          flat = flatten_tree(t->child);

	  for (height = 0.0, temp = flat; temp != NULL; temp = temp->next)
	    if (temp->height > height)
              height = temp->height;

          height *= 1.2f;

          if (*y < (bottom + height))
          {
            (*page) ++;
	    if (Verbosity)
	      progress_show("Formatting page %d", *page);
            width = get_width((uchar *)contents, TYPE_HELVETICA, STYLE_BOLD, SIZE_H1);
            *y = top - _htmlSpacings[SIZE_H1];
            x  = left + 0.5f * (right - left - width);
            r = new_render(*page, RENDER_TEXT, x, *y, 0, 0, contents);
            r->data.text.typeface = TYPE_HELVETICA;
            r->data.text.style    = STYLE_BOLD;
            r->data.text.size     = _htmlSizes[SIZE_H1];

            *y -= _htmlSpacings[SIZE_H1];
          };

          x  = left + 36.0f * t->indent;
	  *y -= height;

          for (temp = flat; temp != NULL; temp = next)
          {
	    if (temp->link != NULL)
	    {
              link = htmlGetVariable(temp->link, (uchar *)"HREF");

	     /*
	      * Add a page link...
	      */

	      if (strncmp((char *)link, "http:", 5) != 0 &&
        	  strncmp((char *)link, "ftp:", 4) != 0 &&
        	  strncmp((char *)link, "file:", 5) != 0 &&
        	  strrchr((char *)link, '#') != NULL)
        	link = (uchar *)strrchr((char *)link, '#');

	      r = new_render(*page, RENDER_LINK, x, *y, temp->width,
	        	     temp->height, link);

	      if (pslevel == 0)
	      {
		temp->red   = 0;
		temp->green = 0;
		temp->blue  = 255;

		rgb[0] = 0.0f;
		rgb[1] = 0.0f;
		rgb[2] = 1.0f;
		r = new_render(*page, RENDER_BOX, x, *y - 1, temp->width, 0, rgb);
	      };
	    };

	    switch (temp->markup)
	    {
              case MARKUP_A :
        	  if ((link = htmlGetVariable(temp, (uchar *)"NAME")) != NULL)
        	  {
        	   /*
        	    * Add a target link...
        	    */

        	    add_link(link, *page, (int)(*y + 6 * height));
        	  };
        	  break;

              case MARKUP_NONE :
        	  if (temp->data == NULL)
        	    break;

		  if (temp->underline)
		  {
		    rgb[0] = temp->red / 255.0f;
		    rgb[1] = temp->green / 255.0f;
		    rgb[2] = temp->blue / 255.0f;
		    r = new_render(*page, RENDER_BOX, x, *y - 1, temp->width, 0, rgb);
		  };

		  if (temp->strikethrough)
		  {
		    rgb[0] = temp->red / 255.0f;
		    rgb[1] = temp->green / 255.0f;
		    rgb[2] = temp->blue / 255.0f;
		    r = new_render(*page, RENDER_BOX, x, *y + t->height * 0.5f, temp->width, 0, rgb);
		  };

        	  r = new_render(*page, RENDER_TEXT, x, *y, 0, 0, temp->data);
        	  r->data.text.typeface = temp->typeface;
        	  r->data.text.style    = temp->style;
        	  r->data.text.size     = _htmlSizes[temp->size];
        	  r->data.text.rgb[0]   = temp->red / 255.0f;
        	  r->data.text.rgb[1]   = temp->green / 255.0f;
        	  r->data.text.rgb[2]   = temp->blue / 255.0f;

        	  if (temp->superscript)
        	    r->y += height / 1.2 - temp->height * 1.2;
		  break;

	      case MARKUP_IMG :
	          update_image_size(temp);
		  r = new_render(*page, RENDER_IMAGE, x, *y, temp->width,
	                	 temp->height,
				 image_load((char *)htmlGetVariable(temp, (uchar *)"SRC"), !OutputColor));
		  break;
	    };

	    x += temp->width;
	    next = temp->next;
	    free(temp);
	  };

         /*
          * Draw dots leading up to the page number...
          */

          sprintf((char *)number, "%d", heading_pages[*heading]);
          width = get_width(number, t->typeface, t->style, t->size) + x;

          for (nptr = number; width < right; width += dot_width, nptr ++)
            *nptr = '.';
          nptr --;
          sprintf((char *)nptr, "%d", heading_pages[*heading]);

          r = new_render(*page, RENDER_TEXT, right - width + x, *y, 0, 0, number);
          r->data.text.typeface = t->typeface;
          r->data.text.style    = t->style;
          r->data.text.size     = _htmlSizes[t->size];
          r->data.text.rgb[0]   = t->red / 255.0f;
          r->data.text.rgb[1]   = t->green / 255.0f;
          r->data.text.rgb[2]   = t->blue / 255.0f;

         /*
          * Next heading...
          */

          (*heading) ++;
          break;

      default :
          parse_contents(t->child, left, right, bottom, top, y, page, heading);
          break;
    };

    t = t->next;
  };
}


/*
 * 'parse_doc()' - Parse a document tree and produce rendering list output.
 */

static void
parse_doc(tree_t *t,		/* I - Tree to parse */
          int    left,		/* I - Left margin */
          int    right,		/* I - Printable width */
          int    bottom,	/* I - Bottom margin */
          int    top,		/* I - Printable top */
          float  *x,		/* IO - X position */
          float  *y,		/* IO - Y position */
          int    *page,		/* IO - Page # */
	  tree_t *cpara)	/* I - Current paragraph */
{
  int		i;		/* Looping var */
  tree_t	*para,		/* Phoney paragraph tree entry */
		*temp;		/* Paragraph entry */
  var_t		*var;		/* Variable entry */
  uchar		*name;		/* ID name */
  float		width,		/* Width of horizontal rule */
		height,		/* Height of rule */
		rgb[3];		/* RGB color of rule */


  DEBUG_printf(("parse_doc(t=%08x, left=%d, right=%d, x=%.1f, y=%.1f, page=%d, cpara = %08x\n",
                t, left, right, *x, *y, *page, cpara));

  if (cpara == NULL)
    para = htmlNewTree(NULL, MARKUP_P, NULL);
  else
    para = cpara;

  while (t != NULL)
  {
    if ((name = htmlGetVariable(t, (uchar *)"ID")) != NULL)
    {
     /*
      * Add a link target using the ID=name variable...
      */

      add_link(name, *page, (int)(*y + 3 * t->height));
    };

    if (t->markup == MARKUP_H1 && TocLevels > 0)
    {
      if (para->child != NULL)
      {
        parse_paragraph(para, left, right, bottom, top, x, y, page);
        htmlDeleteTree(para->child);
        para->child = para->last_child = NULL;
      };

      if (chapter > 0)
      {
        (*page) ++;
        if (PageDuplex && (*page & 1))
          (*page) ++;

        if (Verbosity)
          progress_show("Formatting page %d", *page);

        chapter_ends[chapter] = *page - 1;
      };

      chapter ++;
      chapter_starts[chapter] = *page;

      *y = (float)top;
      *x = (float)left;

      if (page_headings[*page] == NULL)
        page_headings[*page] = htmlGetText(current_heading);
    };

    if (chapter == 0)
    {
      if (t->child != NULL)
        parse_doc(t->child, left, right, bottom, top, x, y, page, para);

      t = t->next;
      continue;
    };

    switch (t->markup)
    {
      case MARKUP_IMG :
          update_image_size(t);
      case MARKUP_NONE :
      case MARKUP_BR :
          if (para->child == NULL)
          {
            para->halignment = t->halignment;
            para->indent     = t->indent;
          };

          if ((temp = htmlAddTree(para, t->markup, t->data)) != NULL)
          {
	    temp->link          = t->link;
            temp->width         = t->width;
            temp->height        = t->height;
            temp->typeface      = t->typeface;
            temp->style         = t->style;
            temp->size          = t->size;
            temp->underline     = t->underline;
            temp->strikethrough = t->strikethrough;
            temp->superscript   = t->superscript;
            temp->subscript     = t->subscript;
            temp->halignment    = t->halignment;
            temp->valignment    = t->valignment;
            temp->red           = t->red;
            temp->green         = t->green;
            temp->blue          = t->blue;
            for (i = 0, var = t->vars; i < t->nvars; i ++, var ++)
              htmlSetVariable(temp, var->name, var->value);

            copy_tree(temp, t->child);
          };
          break;

      case MARKUP_TABLE :
          if (para->child != NULL)
          {
            parse_paragraph(para, left, right, bottom, top, x, y, page);
            htmlDeleteTree(para->child);
            para->child = para->last_child = NULL;
          };

          parse_table(t, left, right, bottom, top, x, y, page);
          break;

      case MARKUP_H1 :
      case MARKUP_H2 :
      case MARKUP_H3 :
      case MARKUP_H4 :
      case MARKUP_H5 :
      case MARKUP_H6 :
      case MARKUP_H7 :
          if (para->child != NULL)
          {
            parse_paragraph(para, left, right, bottom, top, x, y, page);
            htmlDeleteTree(para->child);
            para->child = para->last_child = NULL;
          };

          parse_heading(t, left, right, bottom, top, x, y, page);
          break;

      case MARKUP_CENTER :
          if (para->child != NULL)
          {
            parse_paragraph(para, left, right, bottom, top, x, y, page);
            htmlDeleteTree(para->child);
            para->child = para->last_child = NULL;
          };

          if (t->prev != NULL && *y < top)
	    *y -= _htmlSpacings[t->size];

          parse_doc(t->child, left, right, bottom, top, x, y, page, NULL);

          if (t->next != NULL && *y < top)
	    *y -= _htmlSpacings[t->size];

          *x = (float)left;
          break;

      case MARKUP_P :
          if (para->child != NULL)
          {
            parse_paragraph(para, left, right, bottom, top, x, y, page);
            htmlDeleteTree(para->child);
            para->child = para->last_child = NULL;
          };

          if (t->prev != NULL && *y < top)
	    *y -= _htmlSpacings[t->size];

          parse_doc(t->child, left, right, bottom, top, x, y, page, NULL);
          break;

      case MARKUP_PRE :
          if (para->child != NULL)
          {
            parse_paragraph(para, left, right, bottom, top, x, y, page);
            htmlDeleteTree(para->child);
            para->child = para->last_child = NULL;
          };

          parse_pre(t, left, right, bottom, top, x, y, page);
          break;

      case MARKUP_UL :
      case MARKUP_OL :
          init_list(t);
      case MARKUP_DL :
          if (para->child != NULL)
          {
            parse_paragraph(para, left, right, bottom, top, x, y, page);
            htmlDeleteTree(para->child);
            para->child = para->last_child = NULL;
          };

          *x = (float)(left + 36);

          if (t->prev != NULL && t->prev->markup != MARKUP_LI && *y < top)
	    *y -= _htmlSpacings[t->size];

          parse_doc(t->child, left + 36, right, bottom, top, x, y, page, para);

          if (t->next != NULL &&
	      t->next->markup != MARKUP_LI &&
	      t->next->markup != MARKUP_UL &&
	      t->next->markup != MARKUP_OL &&
	      t->next->markup != MARKUP_DL &&
	      t->next->markup != MARKUP_HR &&
	      (t->next->markup < MARKUP_H1 || t->next->markup > MARKUP_H7) &&
	      *y < top)
	    *y -= _htmlSpacings[t->size];
          break;

      case MARKUP_LI :
          if (para->child != NULL)
          {
            parse_paragraph(para, left, right, bottom, top, x, y, page);
            htmlDeleteTree(para->child);
            para->child = para->last_child = NULL;
          };

          parse_list(t, left, right, bottom, top, x, y, page);
          break;

      case MARKUP_DT :
          if (para->child != NULL)
          {
            parse_paragraph(para, left, right, bottom, top, x, y, page);
            htmlDeleteTree(para->child);
            para->child = para->last_child = NULL;
          };

          parse_paragraph(t, left, right, bottom, top, x, y, page);
          break;

      case MARKUP_DD :
          if (para->child != NULL)
          {
            parse_paragraph(para, left, right, bottom, top, x, y, page);
            htmlDeleteTree(para->child);
            para->child = para->last_child = NULL;
          };

          *x = (float)(left + 36);

          parse_paragraph(t, left + 36, right, bottom, top, x, y, page);
          break;

      case MARKUP_HR :
          if (para->child != NULL)
          {
            parse_paragraph(para, left, right, bottom, top, x, y, page);
            htmlDeleteTree(para->child);
            para->child = para->last_child = NULL;

	    if (*y < top)
	      *y -= _htmlSpacings[para->size];
          };

          if (htmlGetVariable(t, (uchar *)"BREAK") == NULL)
	  {
	   /*
	    * Generate a horizontal rule...
	    */

            if ((name = htmlGetVariable(t, (uchar *)"WIDTH")) == NULL)
	      width = right - left;
	    else
	    {
	      if (strchr((char *)name, '%') != NULL)
	        width = atoi((char *)name) * (right - left) / 100;
	      else
                width = atoi((char *)name) * PagePrintWidth / 680.0f;
            };

            if ((name = htmlGetVariable(t, (uchar *)"SIZE")) == NULL)
	      height = 2;
	    else
	      height = atoi((char *)name) * PagePrintWidth / 680.0f;

            switch (t->halignment)
	    {
	      case ALIGN_LEFT :
	          *x = (float)left;
		  break;
	      case ALIGN_CENTER :
	          *x = (float)left + ((float)(right - left) - width) * 0.5;
		  break;
	      case ALIGN_RIGHT :
	          *x = (float)right - width;
		  break;
	    };

            if (*y < ((float)bottom + height + _htmlSpacings[SIZE_P]))
	    {
	     /*
	      * Won't fit on this page...
	      */

              (*page) ++;
	      if (Verbosity)
	        progress_show("Formatting page %d", *page);
              *y = (float)top;

              if (page_headings[*page] == NULL)
        	page_headings[*page] = htmlGetText(current_heading);
            };

            (*y)   -= height + _htmlSpacings[SIZE_P];
            rgb[0] = t->red / 255.0f;
            rgb[1] = t->green / 255.0f;
            rgb[2] = t->blue / 255.0f;

            new_render(*page, RENDER_FBOX, *x, *y + _htmlSpacings[SIZE_P] * 0.5,
	               width, height, rgb);
	  }
	  else
	  {
	   /*
	    * <HR BREAK> generates a page break...
	    */

            (*page) ++;
	    if (Verbosity)
	      progress_show("Formatting page %d", *page);
            *y = (float)top;

            if (page_headings[*page] == NULL)
              page_headings[*page] = htmlGetText(current_heading);
	  };

          *x = (float)left;
          break;

      case MARKUP_TITLE :
      case MARKUP_META :
          break;

      case MARKUP_A :
          if (htmlGetVariable(t, (uchar *)"NAME") != NULL)
	  {
	   /*
	    * Add this named destination to the paragraph tree...
	    */

            if (para->child == NULL)
            {
              para->halignment = t->halignment;
              para->indent     = t->indent;
            };

            if ((temp = htmlAddTree(para, t->markup, t->data)) != NULL)
            {
	      temp->link          = t->link;
              temp->width         = t->width;
              temp->height        = t->height;
              temp->typeface      = t->typeface;
              temp->style         = t->style;
              temp->size          = t->size;
              temp->underline     = t->underline;
              temp->strikethrough = t->strikethrough;
              temp->superscript   = t->superscript;
              temp->subscript     = t->subscript;
              temp->halignment    = t->halignment;
              temp->valignment    = t->valignment;
              temp->red           = t->red;
              temp->green         = t->green;
              temp->blue          = t->blue;
              for (i = 0, var = t->vars; i < t->nvars; i ++, var ++)
        	htmlSetVariable(temp, var->name, var->value);
            };
	  };

      default :
	  if (t->child != NULL)
            parse_doc(t->child, left, right, bottom, top, x, y, page, para);
          break;
    };

    t = t->next;
  };

  if (para->child != NULL && cpara != para)
  {
    parse_paragraph(para, left, right, bottom, top, x, y, page);
    htmlDeleteTree(para->child);
    para->child = para->last_child = NULL;
  };

  if (cpara != para)
    htmlDeleteTree(para);
}


/*
 * 'parse_heading()' - Parse a heading tree and produce rendering list output.
 */

static void
parse_heading(tree_t *t,	/* I - Tree to parse */
              int    left,	/* I - Left margin */
              int    right,	/* I - Printable width */
              int    bottom,	/* I - Bottom margin */
              int    top,	/* I - Printable top */
              float  *x,	/* IO - X position */
              float  *y,	/* IO - Y position */
              int    *page)	/* IO - Page # */
{
  DEBUG_printf(("parse_heading(t=%08x, left=%d, right=%d, x=%.1f, y=%.1f, page=%d\n",
                t, left, right, *x, *y, *page));

  if ((t->markup - MARKUP_H1) < TocLevels || TocLevels == 0)
    current_heading = t->child;

  if (*y < (3 * _htmlSpacings[t->size] + bottom))
  {
    (*page) ++;
    *y = (float)top;
    if (Verbosity)
      progress_show("Formatting page %d", *page);
  };

  if (page_headings[*page] == NULL || t->markup == MARKUP_H1)
    page_headings[*page] = htmlGetText(current_heading);

  if ((t->markup - MARKUP_H1) < TocLevels)
  {
    if (PageDuplex)
    {
      DEBUG_printf(("H%d: heading_pages[%d] = %d\n", t->markup - MARKUP_H1 + 1,
                    num_headings, *page - 1));
      heading_pages[num_headings] = *page - 1;
    }
    else
    {
      DEBUG_printf(("H%d: heading_pages[%d] = %d\n", t->markup - MARKUP_H1 + 1,
                    num_headings, *page - 1));
      heading_pages[num_headings] = *page;
    };

    heading_tops[num_headings] = (int)(*y + 2 * _htmlSpacings[t->size]);
    num_headings ++;
  };

  parse_paragraph(t, left, right, bottom, top, x, y, page);

  if (t->halignment == ALIGN_RIGHT && t->markup == MARKUP_H1 && TocLevels > 0)
  {
   /*
    * Special case - chapter heading for user's manual...
    */

    *y = bottom + 0.5f * (top - bottom);
  }
  else if (t->next != NULL &&
           t->next->markup != MARKUP_HR &&
           t->next->markup != MARKUP_UL &&
	   t->next->markup != MARKUP_DL &&
	   t->next->markup != MARKUP_OL)
    *y -= _htmlSpacings[t->next->size];
}


/*
 * 'parse_paragraph()' - Parse a paragraph tree and produce rendering list
 *                       output.
 */

static void
parse_paragraph(tree_t *t,	/* I - Tree to parse */
        	int    left,	/* I - Left margin */
        	int    right,	/* I - Printable width */
        	int    bottom,	/* I - Bottom margin */
        	int    top,	/* I - Printable top */
        	float  *x,	/* IO - X position */
        	float  *y,	/* IO - Y position */
                int    *page)	/* IO - Page # */
{
  int		whitespace;	/* Non-zero if a fragment ends in whitespace */
  tree_t	*flat,
		*start,
		*end,
		*prev,
		*temp;
  float		width,
		height,
		temp_width,
		temp_height;
  float		format_width, image_y, image_left, image_right;
  render_t	*r;
  uchar		*align,
		*link;
  float		rgb[3];


  if (*page > MAX_PAGES)
    return;

  DEBUG_printf(("parse_paragraph(t=%08x, left=%d, right=%d, x=%.1f, y=%.1f, page=%d\n",
                t, left, right, *x, *y, *page));

  flat        = flatten_tree(t->child);
  image_left  = (float)left;
  image_right = (float)right;
  image_y     = 0;

  if (flat == NULL)
    DEBUG_puts("parse_paragraph: flat == NULL!");

  if (*y < top && t->prev != NULL)
    *y -= _htmlSpacings[t->size];

 /*
  * First scan for images with left/right alignment tags...
  */

  for (temp = flat, prev = NULL; temp != NULL;)
  {
    if (temp->markup == MARKUP_IMG)
      update_image_size(temp);

    if (temp->markup == MARKUP_IMG &&
        (align = htmlGetVariable(temp, (uchar *)"ALIGN")))
    {
      if (strcasecmp((char *)align, "LEFT") == 0)
      {
        if (*y < (bottom + temp->height))
        {
	  (*page) ++;
	  *y = (float)top;

	  if (Verbosity)
	    progress_show("Formatting page %d", *page);

          if (page_headings[*page] == NULL)
            page_headings[*page] = htmlGetText(current_heading);
        };

        new_render(*page, RENDER_IMAGE, (float)left, *y - temp->height, temp->width,
                   temp->height,
		   image_load((char *)htmlGetVariable(temp, (uchar *)"SRC"), !OutputColor));
        image_left = left + temp->width;
        image_y    = *y - temp->height;

        if (prev != NULL)
          prev->next = temp->next;
        else
          flat = temp->next;

        free(temp);
        temp = prev;
      }
      else if (strcasecmp((char *)align, "RIGHT") == 0)
      {
        if (*y < (bottom + temp->height))
        {
	  (*page) ++;
	  *y = (float)top;

	  if (Verbosity)
	    progress_show("Formatting page %d", *page);

          if (page_headings[*page] == NULL)
            page_headings[*page] = htmlGetText(current_heading);
        };

        new_render(*page, RENDER_IMAGE, right - temp->width, *y - temp->height,
                   temp->width, temp->height,
		   image_load((char *)htmlGetVariable(temp, (uchar *)"SRC"), !OutputColor));
        image_right = right - temp->width;
        image_y     = *y - temp->height;

        if (prev != NULL)
          prev->next = temp->next;
        else
          flat = temp->next;

        free(temp);
        temp = prev;
      };
    };

    if (temp != NULL)
    {
      prev = temp;
      temp = temp->next;
    }
    else
      temp = flat;
  };

 /*
  * Then format the text and inline images...
  */

  format_width = image_right - image_left;

  while (flat != NULL)
  {
    start = flat;
    end   = flat;
    width = 0.0;

    while (flat != NULL)
    {
      temp_width = 0.0;
      temp       = flat;
      whitespace = 0;

      do
      {
        if ((temp_width == 0.0 || whitespace) &&
            temp->markup == MARKUP_NONE && temp->data[0] == ' ')
          temp_width -= char_widths[temp->typeface][temp->style][' '] *
                        _htmlSizes[temp->size];

        if (temp->markup == MARKUP_NONE && temp->data[strlen((char *)temp->data) - 1] == ' ')
          whitespace = 1;
        else
          whitespace = 0;

        prev       = temp;
        temp       = temp->next;
        temp_width += prev->width;
      }
      while (temp != NULL && !whitespace && prev->markup != MARKUP_BR);

      if ((width + temp_width) <= format_width)
      {
        width += temp_width;
        end  = temp;
        flat = temp;

        if (prev->markup == MARKUP_BR && width > 0.0)
          break;
      }
      else if (width == 0.0)
      {
        width += temp_width;
        end  = temp;
        flat = temp;
        break;
      }
      else
        break;
    };

    if (start == end)
    {
      end   = start->next;
      flat  = start->next;
      width = start->width;
    };

    for (height = 0.0, temp = prev = start; temp != end; temp = temp->next)
    {
      prev = temp;
      if (temp->markup == MARKUP_IMG)
        temp_height = temp->height;
      else
        temp_height = temp->height * 1.2f;

      if (temp_height > height)
        height = temp_height;
    };

    if (prev != NULL && prev->markup == MARKUP_NONE &&
        prev->data[strlen((char *)prev->data) - 1] == ' ')
    {
     /*
      * Remove trailing space...
      */

      prev->data[strlen((char *)prev->data) - 1] = '\0';
      temp_width = char_widths[prev->typeface][prev->style][' '] *
                   _htmlSizes[prev->size];
      prev->width -= temp_width;
      width       -= temp_width;
    };

    if (*y < (height + bottom))
    {
      (*page) ++;
      *y = (float)top;

      if (Verbosity)
        progress_show("Formatting page %d", *page);

      if (page_headings[*page] == NULL)
        page_headings[*page] = htmlGetText(current_heading);
    };

    *y -= height;

    if (Verbosity)
      progress_update(100 - 100 * (*y) / PagePrintLength);

    if (t->halignment == ALIGN_LEFT)
      *x = image_left;
    else if (t->halignment == ALIGN_CENTER)
      *x = image_left + 0.5f * (format_width - width);
    else
      *x = image_right - width;

    whitespace = 0;
    temp       = start;
    while (temp != end)
    {
      if (temp->link != NULL)
      {
        link = htmlGetVariable(temp->link, (uchar *)"HREF");

       /*
	* Add a page link...
	*/

	if (strncmp((char *)link, "http:", 5) != 0 &&
            strncmp((char *)link, "ftp:", 4) != 0 &&
            strncmp((char *)link, "file:", 5) != 0 &&
            strrchr((char *)link, '#') != NULL)
          link = (uchar *)strrchr((char *)link, '#');

	r = new_render(*page, RENDER_LINK, *x, *y, temp->width,
	               temp->height, link);

	if (pslevel == 0)
	{
	  temp->red   = 0;
	  temp->green = 0;
	  temp->blue  = 255;

	  rgb[0] = 0.0f;
	  rgb[1] = 0.0f;
	  rgb[2] = 1.0f;
	  r = new_render(*page, RENDER_BOX, *x, *y - 1, temp->width, 0, rgb);
	};
      };

      switch (temp->markup)
      {
        case MARKUP_A :
            if ((link = htmlGetVariable(temp, (uchar *)"NAME")) != NULL)
            {
             /*
              * Add a target link...
              */

              add_link(link, *page, (int)(*y + 6 * height));
            };
            break;

        case MARKUP_NONE :
            if (temp->data == NULL)
              break;

	    if (temp->underline)
	    {
	      rgb[0] = temp->red / 255.0f;
	      rgb[1] = temp->green / 255.0f;
	      rgb[2] = temp->blue / 255.0f;
	      r = new_render(*page, RENDER_BOX, *x, *y - 1, temp->width, 0, rgb);
	    };

	    if (temp->strikethrough)
	    {
	      rgb[0] = temp->red / 255.0f;
	      rgb[1] = temp->green / 255.0f;
	      rgb[2] = temp->blue / 255.0f;
	      r = new_render(*page, RENDER_BOX, *x, *y + t->height * 0.5f, temp->width, 0, rgb);
	    };

            if ((temp == start || whitespace) && temp->data[0] == ' ')
              r = new_render(*page, RENDER_TEXT, *x, *y, temp->width,
                             temp->height, temp->data + 1);
            else
              r = new_render(*page, RENDER_TEXT, *x, *y, temp->width,
                             temp->height, temp->data);
            r->data.text.typeface = temp->typeface;
            r->data.text.style    = temp->style;
            r->data.text.size     = _htmlSizes[temp->size];
            r->data.text.rgb[0]   = temp->red / 255.0f;
            r->data.text.rgb[1]   = temp->green / 255.0f;
            r->data.text.rgb[2]   = temp->blue / 255.0f;

            if (temp->superscript)
              r->y += height / 1.2 - temp->height * 1.2;

            if (temp->data[strlen((char *)temp->data) - 1] == ' ')
              whitespace = 1;
            else
              whitespace = 0;
	    break;

	case MARKUP_IMG :
	    r = new_render(*page, RENDER_IMAGE, *x, *y, temp->width,
	                   temp->height,
			   image_load((char *)htmlGetVariable(temp, (uchar *)"SRC"), !OutputColor));
            whitespace = 0;
	    break;
      };

      *x += temp->width;
      prev = temp;
      temp = temp->next;
      free(prev);
    };

    if (*y < image_y)
    {
      image_left   = (float)left;
      image_right  = (float)right;
      format_width = image_right - image_left;
    };
  };

  *x = (float)left;
  if (*y > image_y && image_y > 0.0)
    *y = image_y;
}


/*
 * 'parse_pre()' - Parse preformatted text and produce rendering list output.
 */

static void
parse_pre(tree_t *t,		/* I - Tree to parse */
          int    left,		/* I - Left margin */
          int    right,		/* I - Printable width */
          int    bottom,	/* I - Bottom margin */
          int    top,		/* I - Printable top */
          float  *x,		/* IO - X position */
          float  *y,		/* IO - Y position */
          int    *page)		/* IO - Page # */
{
  tree_t	*flat, *next;
  uchar		*link,
		line[10240],
		*lineptr,
		*dataptr;
  int		col;
  float		width,
		rgb[3];
  render_t	*r;


  REF(right);

  DEBUG_printf(("parse_pre(t=%08x, left=%d, right=%d, x=%.1f, y=%.1f, page=%d\n",
                t, left, right, *x, *y, *page));

  if (t->child == NULL)
    return;

  col  = 0;
  flat = flatten_tree(t->child);

  *y += _htmlSpacings[t->size];

  while (flat != NULL)
  {
    if (col == 0)
    {
      if (*y < (_htmlSpacings[t->size] + bottom))
      {
        (*page) ++;
        *y = (float)top;

	if (Verbosity)
	  progress_show("Formatting page %d", *page);

	if (page_headings[*page] == NULL)
          page_headings[*page] = htmlGetText(current_heading);
      };
    
      *x = (float)left;
      *y -= _htmlSpacings[t->size];

      if (Verbosity)
        progress_update(100 - 100 * (*y) / PagePrintLength);
    };

    if (flat->link != NULL)
    {
      link = htmlGetVariable(flat->link, (uchar *)"HREF");

     /*
      * Add a page link...
      */

      if (strncmp((char *)link, "http:", 5) != 0 &&
          strncmp((char *)link, "ftp:", 4) != 0 &&
          strncmp((char *)link, "file:", 5) != 0 &&
          strrchr((char *)link, '#') != NULL)
        link = (uchar *)strrchr((char *)link, '#');

      r = new_render(*page, RENDER_LINK, *x, *y, flat->width,
	             flat->height, link);

      if (pslevel == 0)
      {
	flat->red   = 0;
	flat->green = 0;
	flat->blue  = 255;

	rgb[0] = 0.0f;
	rgb[1] = 0.0f;
	rgb[2] = 1.0f;
	r = new_render(*page, RENDER_BOX, *x, *y - 1, flat->width, 0, rgb);
      };
    };

    switch (flat->markup)
    {
      case MARKUP_A :
          if ((link = htmlGetVariable(flat, (uchar *)"NAME")) != NULL)
          {
           /*
            * Add a target link...
            */

            add_link(link, *page, (int)(*y + 6 * t->height));
          };
          break;

      case MARKUP_BR :
          col = 0;
          break;

      case MARKUP_NONE :
          for (lineptr = line, dataptr = flat->data;
	       *dataptr != '\0' && lineptr < (line + sizeof(line) - 1);
	       dataptr ++)
            if (*dataptr == '\n')
            {
              col = 0;
              break;
            }
            else if (*dataptr == '\t')
            {
              do
              {
                *lineptr++ = ' ';
                col ++;
              }
              while (col & 7);
            }
            else if (*dataptr != '\r')
            {
              *lineptr++ = *dataptr;
              col ++;
            };

          *lineptr = '\0';

          width = get_width(line, flat->typeface, flat->style, flat->size);
          r = new_render(*page, RENDER_TEXT, *x, *y, width, 0, line);
          r->data.text.typeface = flat->typeface;
          r->data.text.style    = flat->style;
          r->data.text.size     = _htmlSizes[flat->size];
          r->data.text.rgb[0]   = flat->red / 255.0f;
          r->data.text.rgb[1]   = flat->green / 255.0f;
          r->data.text.rgb[2]   = flat->blue / 255.0f;

	  if (flat->underline)
	  {
	    rgb[0] = flat->red / 255.0f;
	    rgb[1] = flat->green / 255.0f;
	    rgb[2] = flat->blue / 255.0f;
	    r = new_render(*page, RENDER_BOX, *x, *y - 1, flat->width, 0, rgb);
	  };

	  if (flat->strikethrough)
	  {
	    rgb[0] = flat->red / 255.0f;
	    rgb[1] = flat->green / 255.0f;
	    rgb[2] = flat->blue / 255.0f;
	    r = new_render(*page, RENDER_BOX, *x, *y + t->height * 0.5f, flat->width, 0, rgb);
	  };

          *x += flat->width;
          break;

      case MARKUP_IMG :
	  r = new_render(*page, RENDER_IMAGE, *x, *y, flat->width,
	                 flat->height,
			 image_load((char *)htmlGetVariable(flat, (uchar *)"SRC"), !OutputColor));

          *x += flat->width;
          col ++;
	  break;
    };

    next = flat->next;
    free(flat);
    flat = next;
  };

  *x = (float)left;
}



/*
 * 'parse_table()' - Parse a table and produce rendering output.
 */

static void
parse_table(tree_t *t,		/* I - Tree to parse */
            int    left,	/* I - Left margin */
            int    right,	/* I - Printable width */
            int    bottom,	/* I - Bottom margin */
            int    top,		/* I - Printable top */
            float  *x,		/* IO - X position */
            float  *y,		/* IO - Y position */
            int    *page)	/* IO - Page # */
{
  int		col,
		row,
		colspan,
		cellpadding,
		cellspacing,
		border,
		num_cols,
		num_rows,
		regular_cols,
		col_lefts[MAX_COLUMNS],
		col_rights[MAX_COLUMNS];
  float		col_widths[MAX_COLUMNS],
		col_mins[MAX_COLUMNS],
		width,
		regular_width,
		actual_width,
		row_y, temp_y;
  int		row_page, temp_page;
  uchar		*var;
  tree_t	*temprow,
		*tempcol,
		*flat,
		*next,
		*cells[MAX_ROWS][MAX_COLUMNS];
  static float	black[3] = { 0.0, 0.0, 0.0 };


  DEBUG_printf(("parse_table(t=%08x, left=%d, right=%d, x=%.1f, y=%.1f, page=%d\n",
                t, left, right, *x, *y, *page));

  if (t->child == NULL)
    return;   /* Empty table... */

 /*
  * Figure out the # of rows, columns, and the desired widths...
  */

  memset(col_widths, 0, sizeof(col_widths));
  memset(col_mins, 0, sizeof(col_mins));
  memset(cells, 0, sizeof(cells));

  for (temprow = t->child, num_cols = 0, num_rows = 0;
       temprow != NULL && num_rows < MAX_ROWS;
       temprow = temprow->next)
    if (temprow->markup == MARKUP_TR || temprow->markup == MARKUP_THEAD)
    {
      for (tempcol = temprow->child, col = 0;
           tempcol != NULL && col < MAX_COLUMNS;
           tempcol = tempcol->next)
        if (tempcol->markup == MARKUP_TD || tempcol->markup == MARKUP_TH)
        {
          if ((var = htmlGetVariable(tempcol, (uchar *)"COLSPAN")) != NULL)
            colspan = atoi((char *)var);
          else
            colspan = 1;

          while (colspan > 0 && col < MAX_COLUMNS)
          {
            cells[num_rows][col] = tempcol;
            col ++;
            colspan --;
          };

          if ((var = htmlGetVariable(tempcol, (uchar *)"WIDTH")) != NULL)
	  {
            if (var[strlen((char *)var) - 1] == '%')
              col_widths[col - 1] = atoi((char *)var) * PagePrintWidth / 100.0f;
            else
              col_widths[col - 1] = atoi((char *)var) * PagePrintWidth / 680.0f;

            col_mins[col - 1] = col_widths[col - 1];
	  }
	  else
	  {
            flat  = flatten_tree(tempcol->child);
            width = 0.0;
            while (flat != NULL)
            {
              if (flat->markup == MARKUP_BR ||
                  (flat->preformatted &&
                   flat->data != NULL &&
                   flat->data[strlen((char *)flat->data) - 1] == '\n'))
              {
                width += flat->width;

                if (width > col_widths[col - 1])
                  col_widths[col - 1] = width;

                width = 0.0;
              }
              else
                width += flat->width;

              if (flat->width > col_mins[col - 1])
	        col_mins[col - 1] = flat->width;

              next = flat->next;
              free(flat);
              flat = next;
            };

            if (width > col_widths[col - 1] &&
	        (col <= 1 || cells[num_rows][col - 1] != cells[num_rows][col - 2]))
              col_widths[col - 1] = width;
	  };
        };

      if (col > num_cols)
        num_cols = col;

      num_rows ++;
    };

 /*
  * Now figure out the width of the table...
  */

  if ((var = htmlGetVariable(t, (uchar *)"CELLPADDING")) != NULL)
    cellpadding = atoi((char *)var);
  else
    cellpadding = 4;

  if ((var = htmlGetVariable(t, (uchar *)"CELLSPACING")) != NULL)
    cellspacing = atoi((char *)var);
  else
    cellspacing = 0;

  if ((var = htmlGetVariable(t, (uchar *)"BORDER")) != NULL)
  {
    if ((border = atoi((char *)var)) == 0 && var[0] != '0')
      border = 1;
  }
  else
    border = 0;

  if ((var = htmlGetVariable(t, (uchar *)"WIDTH")) != NULL)
  {
    if (var[strlen((char *)var) - 1] == '%')
      width = atoi((char *)var) * (right - left) / 100.0f;
    else
      width = atoi((char *)var) * 0.9f;
  }
  else
  {
    for (col = 0, width = 0.0; col < num_cols; col ++)
      width += col_widths[col];
    width += 2 * (border + cellpadding + cellspacing) * num_cols;

    if (width > (right - left))
      width = (float)(right - left);
  };

 /*
  * Compute the width of each column based on the printable width.
  */

  actual_width  = (float)(2 * (border + cellpadding + cellspacing) * num_cols);
  regular_width = (width - actual_width) / num_cols;

  for (col = 0, regular_cols = 0; col < num_cols; col ++)
  {
    if ((actual_width + col_widths[col]) < width &&
        (col_mins[col] > regular_width ||
	 col_widths[col] < regular_width ||
         (cells[0][col] != NULL &&
	  htmlGetVariable(cells[0][col], (uchar *)"WIDTH") != NULL)))
      actual_width += col_widths[col];
    else
    {
      regular_cols ++;
      col_widths[col] = 0.0;
    };
  };

  if (regular_cols > 0)
  {
    regular_width = (width - actual_width) / regular_cols;

    for (col = 0; col < num_cols; col ++)
    {
      if (col_widths[col] == 0.0)
	col_widths[col] = regular_width;
    };
  }
  else if (width > actual_width)
  {
    regular_width = (width - actual_width) / num_cols;

    for (col = 0; col < num_cols; col ++)
      col_widths[col] += regular_width;
  };

  switch (t->halignment)
  {
    case ALIGN_LEFT :
        *x = (float)(left + border + cellpadding + cellspacing);
        break;
    case ALIGN_CENTER :
        *x = left + 0.5f * (right - left - width) + border + cellpadding + cellspacing;
        break;
    case ALIGN_RIGHT :
        *x = right - width + border + cellpadding + cellspacing;
        break;
  };

  for (col = 0; col < num_cols; col ++)
  {
    col_lefts[col]  = (int)*x;
    col_rights[col] = (int)(*x + col_widths[col] + 0.5);
    *x = (float)col_rights[col] + 2 * (border + cellpadding + cellspacing);
  };

  *y -= (border + cellpadding + cellspacing);

  for (row = 0; row < num_rows; row ++)
  {
    if (*y < (bottom + 2 * (border + cellpadding + cellspacing) + _htmlSpacings[SIZE_P]))
    {
      *y = (float)top;
      (*page) ++;

      if (Verbosity)
        progress_show("Formatting page %d", *page);

      if (page_headings[*page] == NULL)
        page_headings[*page] = htmlGetText(current_heading);
    };

    row_y    = *y;
    row_page = *page;

    for (col = 0; col < num_cols; col += colspan + 1)
    {
      for (colspan = 1; (col + colspan) < num_cols; colspan ++)
        if (cells[row][col] != cells[row][col + colspan])
          break;
      colspan --;

      if (cells[row][col] == NULL)
        continue;

      *x        = (float)col_lefts[col];
      temp_y    = *y - (border + cellspacing);
      temp_page = *page;

      parse_doc(cells[row][col]->child,
                col_lefts[col], col_rights[col + colspan],
                bottom + border + cellspacing + cellpadding,
                top - border - cellspacing - cellpadding,
                x, &temp_y, &temp_page, NULL);

      if (temp_page > row_page)
      {
        row_page = temp_page;
        row_y    = temp_y;
      }
      else if (temp_y < row_y && temp_page == row_page)
        row_y = temp_y;
    };

    row_y -= 2 * (border + cellpadding + cellspacing);

    if (border > 0)
    {
      for (col = 0; col < num_cols; col ++)
      {
	for (colspan = 1; (col + colspan) < num_cols; colspan ++)
          if (cells[row][col] != cells[row][col + colspan])
            break;
	colspan --;

        width = (float)(col_rights[col + colspan] - col_lefts[col] +
                        2 * cellpadding + 2 * border);

        if (row_page != *page)
        {
         /*
          * Crossing a page boundary...
          */

          new_render(*page, RENDER_BOX, (float)(col_lefts[col] - cellpadding - border),
                     (float)(bottom + cellspacing), width,
                     *y - bottom - 2 * cellspacing,
                     black);

          for (temp_page = *page + 1; temp_page != row_page; temp_page ++)
            new_render(temp_page, RENDER_BOX, (float)(col_lefts[col] - cellpadding - border),
                       (float)(bottom + cellspacing), width,
                       (float)(top - bottom - 2 * cellspacing), black);

          new_render(row_page, RENDER_BOX, (float)(col_lefts[col] - cellpadding - border),
                     row_y + cellspacing, width,
                     top - row_y - 2 * cellspacing, black);
        }
        else
          new_render(*page, RENDER_BOX, (float)(col_lefts[col] - cellpadding - border),
                     row_y + cellspacing, width,
                     *y - row_y - 2 * cellspacing, black);

	col += colspan;
      };
    };

    *page = row_page;
    *y    = row_y;
  };

  *x = (float)left;
  *y -= _htmlSpacings[SIZE_P] - (border + cellpadding + cellspacing);
  if (*y < bottom)
  {
    *y = (float)top;
    (*page) ++;

    if (Verbosity)
      progress_show("Formatting page %d", *page);

    if (page_headings[*page] == NULL)
      page_headings[*page] = htmlGetText(current_heading);
  };
}


/*
 * 'parse_list()' - Parse a list entry and produce rendering output.
 */

static void
parse_list(tree_t *t,		/* I - Tree to parse */
           int    left,		/* I - Left margin */
           int    right,	/* I - Printable width */
           int    bottom,	/* I - Bottom margin */
           int    top,		/* I - Printable top */
           float  *x,		/* IO - X position */
           float  *y,		/* IO - Y position */
           int    *page)	/* IO - Page # */
{
  uchar		number[255];	/* List number (for numbered types) */
  int		typeface;	/* Typeface of list number */
  float		width;		/* Width of list number */
  render_t	*r;		/* Render primitive */


  DEBUG_printf(("parse_list(t=%08x, left=%d, right=%d, x=%.1f, y=%.1f, page=%d\n",
                t, left, right, *x, *y, *page));

  if (*y < (_htmlSpacings[t->size] + bottom))
  {
    (*page) ++;
    *y = (float)top;

    if (Verbosity)
      progress_show("Formatting page %d", *page);

    if (page_headings[*page] == NULL)
      page_headings[*page] = htmlGetText(current_heading);
  };

  switch (list_types[t->indent])
  {
    case 'a' :
    case 'A' :
    case '1' :
    case 'i' :
    case 'I' :
        strcpy((char *)number, format_number(list_values[t->indent],
	                                     list_types[t->indent]));
        strcat((char *)number, ". ");
        typeface = t->typeface;
        break;

    default :
        sprintf((char *)number, "%c ", list_types[t->indent]);
        typeface = TYPE_SYMBOL;
        break;
  };

  width = get_width(number, typeface, t->style, t->size);

  r = new_render(*page, RENDER_TEXT, left - width, *y - _htmlSpacings[t->size],
                 width, _htmlSpacings[t->size], number);
  r->data.text.typeface = typeface;
  r->data.text.style    = t->style;
  r->data.text.size     = _htmlSizes[t->size];
  r->data.text.rgb[0]   = t->red / 255.0f;
  r->data.text.rgb[1]   = t->green / 255.0f;
  r->data.text.rgb[2]   = t->blue / 255.0f;

  parse_doc(t->child, left, right, bottom, top, x, y, page, NULL);

  list_values[t->indent] ++;
}


/*
 * 'init_list()' - Initialize the list type and value as necessary.
 */

static void
init_list(tree_t *t)		/* I - List entry */
{
  uchar		*type,		/* TYPE= variable */
		*value;		/* VALUE= variable */
  static uchar	*symbols = (uchar *)"\327\267\250\340";


  if ((type = htmlGetVariable(t, (uchar *)"TYPE")) != NULL)
  {
    if (strlen((char *)type) == 1)
      list_types[t->indent] = type[0];
    else if (strcasecmp((char *)type, "disc") == 0 ||
             strcasecmp((char *)type, "circle") == 0)
      list_types[t->indent] = symbols[1];
    else
      list_types[t->indent] = symbols[2];
  }
  else if (t->markup == MARKUP_UL)
    list_types[t->indent] = symbols[t->indent & 3];
  else if (t->markup == MARKUP_OL)
    list_types[t->indent] = '1';

  if ((value = htmlGetVariable(t, (uchar *)"VALUE")) == NULL)
    value = htmlGetVariable(t, (uchar *)"START");

  if (value != NULL)
  {
    if (isdigit(value[0]))
      list_values[t->indent] = atoi((char *)value);
    else if (isupper(value[0]))
      list_values[t->indent] = value[0] - 'A' + 1;
    else
      list_values[t->indent] = value[0] - 'a' + 1;
  }
  else if (t->markup == MARKUP_OL)
    list_values[t->indent] = 1;
}


/*
 * 'find_background()' - Find the background image/color for the given document.
 */

static void
find_background(tree_t *t)	/* I - Document to search */
{
  int		i;		/* Looping vars */
  uchar		*var;		/* BGCOLOR/BACKGROUND variable */
  static struct
  {
    char	*name;		/* Color name */
    uchar	red,		/* Red value */
		green,		/* Green value */
		blue;		/* Blue value */
  }		colors[] =	/* Color "database" */
  {
    { "black",		0,   0,   0 },
    { "red",		255, 0,   0 },
    { "green",		0,   255, 0 },
    { "yellow",		255, 255, 0 },
    { "blue",		0,   0,   255 },
    { "magenta",	255, 0,   255 },
    { "cyan",		0,   255, 255 },
    { "white",		255, 255, 255 }
  };


 /*
  * First see if the --bodycolor or --bodyimage options have been
  * specified...
  */

  if (BodyImage[0] != '\0')
  {
    background_image = image_load(BodyImage, !OutputColor);
    return;
  }
  else if (BodyColor[0] != '\0')
  {
    if (BodyColor[0] == '#')
    {
     /*
      * RGB value in hex...
      */

      i = strtol(BodyColor + 1, NULL, 16);

      background_color[0] = (i >> 16) / 255.0f;
      background_color[1] = ((i >> 8) & 255) / 255.0f;
      background_color[2] = (i & 255) / 255.0f;
    }
    else
    {
      for (i = 0; i < (sizeof(colors) / sizeof(colors[0])); i ++)
	if (strcasecmp(colors[i].name, BodyColor) == 0)
	{
          background_color[0] = colors[i].red / 255.0f;
          background_color[1] = colors[i].green / 255.0f;
          background_color[2] = colors[i].blue / 255.0f;
	};
    };

    return;
  };

 /*
  * If not, search the document tree...
  */

  while (t != NULL && background_image == NULL &&
         background_color[0] == 1.0 && background_color[1] == 1.0 &&
	 background_color[2] == 1.0)
  {
    if (t->markup == MARKUP_BODY)
    {
      if ((var = htmlGetVariable(t, (uchar *)"BACKGROUND")) != NULL)
        background_image = image_load((char *)var, !OutputColor);

      if ((var = htmlGetVariable(t, (uchar *)"BGCOLOR")) != NULL)
      {
        if (var[0] == '#')
        {
         /*
          * RGB value in hex...
          */

          i = strtol((char *)var + 1, NULL, 16);

          background_color[0] = (i >> 16) / 255.0f;
          background_color[1] = ((i >> 8) & 255) / 255.0f;
          background_color[2] = (i & 255) / 255.0f;
        }
	else
	{
          for (i = 0; i < (sizeof(colors) / sizeof(colors[0])); i ++)
            if (strcasecmp(colors[i].name, (char *)var) == 0)
            {
              background_color[0] = colors[i].red / 255.0f;
              background_color[1] = colors[i].green / 255.0f;
              background_color[2] = colors[i].blue / 255.0f;
            };
	};
      };
    };

    if (t->child != NULL)
      find_background(t->child);

    t = t->next;
  };
}


/*
 * 'write_background()' - Write the background image/color for to the current
 *                        page.
 */

static void
write_background(FILE *out)	/* I - File to write to */
{
  float	    x, y;
  float	    width, height;


  if (background_image != NULL)
  {
    width  = background_image->width * PagePrintWidth / 680.0f;
    height = background_image->height * PagePrintWidth / 680.0f;

    switch (pslevel)
    {
      case 0 :
          for (x = 0.0; x < PageWidth; x += width)
            for (y = 0.0; y < PageLength; y += height)
            {
  	      pdf_printf(out, "q %.1f 0 0 %.1f %.1f %.1f cm", width, height, x, y);
              pdf_puts("/BG Do\n", out);
	      pdf_puts("Q\n", out);
            };
	  break;

      case 1 :
      case 2 :
          fprintf(out, "0 %.1f %d{/y exch def 0 %.1f %d{/x exch def\n",
	          height, PageLength + (int)height - 1, width, PageWidth);
          fprintf(out, "GS[%.1f 0 0 %.1f x y]CM/iy -1 def\n", width, height);
	  fprintf(out, "%d %d 8[%d 0 0 %d 0 %d]",
	          background_image->width, background_image->height,
                  background_image->width, -background_image->height,
		  background_image->height);
          fputs("{/iy iy 1 add def BG iy get}", out);
	  if (background_image->depth == 1)
	    fputs("image\n", out);
	  else
	    fputs("false 3 colorimage\n", out);
	  fputs("GR}for}for\n", out);
          break;
    };
  }
  else if (background_color[0] != 1.0 ||
           background_color[1] != 1.0 ||
           background_color[2] != 1.0)
  {
    if (pslevel > 0)
    {
      render_x = -1.0;
      render_y = -1.0;
      set_color(out, background_color);
      fprintf(out, "0 0 M %d %d F\n", PageWidth, PageLength);
    }
    else
    {
      set_color(out, background_color);
      pdf_printf(out, "0 0 %d %d re f\n", PageWidth, PageLength);
    };
  };
}


/*
 * 'new_render()' - Allocate memory for a new rendering structure.
 */

static render_t *		/* O - New render structure */
new_render(int   page,		/* I - Page number (0-n) */
           int   type,		/* I - Type of render primitive */
           float x,		/* I - Horizontal position */
           float y,		/* I - Vertical position */
           float width,		/* I - Width */
           float height,	/* I - Height */
           void  *data)		/* I - Data */
{
  render_t		*r;	/* New render primitive */
  static render_t	dummy;	/* Dummy var for errors... */


  if (page >= MAX_PAGES)
    return (&dummy);

  if (type == RENDER_IMAGE || data == NULL)
    r = (render_t *)calloc(sizeof(render_t), 1);
  else
    r = (render_t *)calloc(sizeof(render_t) + strlen((char *)data), 1);

  if (r == NULL)
    return (&dummy);

  r->type   = type;
  r->x      = x;
  r->y      = y;
  r->width  = width;
  r->height = height;

  switch (type)
  {
    case RENDER_TEXT :
        if (data == NULL)
        {
          free(r);
          return (NULL);
        };
        strcpy((char *)r->data.text.buffer, (char *)data);
        break;
    case RENDER_IMAGE :
        if (data == NULL)
        {
          free(r);
          return (NULL);
        };
        r->data.image = (image_t *)data;
        break;
    case RENDER_BOX :
        memcpy(r->data.box, data, sizeof(r->data.box));
        break;
    case RENDER_FBOX :
        memcpy(r->data.fbox, data, sizeof(r->data.fbox));
        break;
    case RENDER_LINK :
        if (data == NULL)
        {
          free(r);
          return (NULL);
        };
        strcpy((char *)r->data.link, (char *)data);
        break;
  };

  if (endpages[page] != NULL)
    endpages[page]->next = r;
  else
    pages[page] = r;

  r->next        = NULL;
  endpages[page] = r;

  if (page >= num_pages)
    num_pages = page + 1;

  return (r);
}


/*
 * 'add_link()' - Add a named link...
 */

static void
add_link(uchar *name,	/* I - Name of link */
         int   page,	/* I - Page # */
         int   top)	/* I - Y position */
{
  link_t	*temp;	/* New name */


  if ((temp = find_link(name)) != NULL)
  {
    temp->page = page - 1;
    temp->top  = top;
  }
  else if (num_links < MAX_LINKS)
  {
    temp = links + num_links;
    num_links ++;

    strncpy((char *)temp->name, (char *)name, sizeof(temp->name) - 1);
    temp->name[sizeof(temp->name) - 1] = '\0';
    temp->page = page - 1;
    temp->top  = top;

    if (num_links > 1)
      qsort(links, num_links, sizeof(link_t),
            (int (*)(const void *, const void *))compare_links);
  };
}


/*
 * 'find_link()' - Find a named link...
 */

static link_t *
find_link(uchar *name)	/* I - Name to find */
{
  link_t	key,	/* Search key */
		*match;	/* Matching name entry */


  if (name == NULL || num_links == 0)
    return (NULL);

  strncpy((char *)key.name, (char *)name, sizeof(key.name) - 1);
  key.name[sizeof(key.name) - 1] = '\0';
  match = (link_t *)bsearch(&key, links, num_links, sizeof(link_t),
                            (int (*)(const void *, const void *))compare_links);

  return (match);
}


/*
 * 'compare_links()' - Compare two named links.
 */

static int			/* O - 0 = equal, -1 or 1 = not equal */
compare_links(link_t *n1,	/* I - First name */
              link_t *n2)	/* I - Second name */
{
  return (strcasecmp((char *)n1->name, (char *)n2->name));
}


/*
 * 'copy_tree()' - Copy a markup tree...
 */

static void
copy_tree(tree_t *parent,	/* I - Source tree */
          tree_t *t)		/* I - Destination tree */
{
  int		i;		/* I - Looping var */
  tree_t	*temp;		/* I - New tree entry */
  var_t		*var;		/* I - Current markup variable */


  while (t != NULL)
  {
    if ((temp = htmlAddTree(parent, t->markup, t->data)) != NULL)
    {
      temp->link          = t->link;
      temp->typeface      = t->typeface;
      temp->style         = t->style;
      temp->size          = t->size;
      temp->halignment    = t->halignment;
      temp->valignment    = t->valignment;
      temp->red           = t->red;
      temp->green         = t->green;
      temp->blue          = t->blue;
      temp->underline     = t->underline;
      temp->strikethrough = t->strikethrough;
      temp->superscript   = t->superscript;
      temp->subscript     = t->subscript;
      for (i = 0, var = t->vars; i < t->nvars; i ++, var ++)
        htmlSetVariable(temp, var->name, var->value);

      copy_tree(temp, t->child);
    };

    t = t->next;
  };
}


/*
 * 'flatten_tree()' - Flatten an HTML tree to only include the text, image,
 *                    link, and break markups.
 */

static tree_t *			/* O - Flattened markup tree */
flatten_tree(tree_t *t)		/* I - Markup tree to flatten */
{
  tree_t	*temp,
		*flat;


  flat = NULL;

  while (t != NULL)
  {
    switch (t->markup)
    {
      case MARKUP_NONE :
      case MARKUP_BR :
      case MARKUP_IMG :
	  temp = (tree_t *)calloc(sizeof(tree_t), 1);
	  memcpy(temp, t, sizeof(tree_t));
	  temp->parent = NULL;
	  temp->child  = NULL;
	  temp->prev   = flat;
	  temp->next   = NULL;
	  if (flat != NULL)
            flat->next = temp;
          flat = temp;

          if (temp->markup == MARKUP_IMG)
            update_image_size(temp);
          break;

      case MARKUP_A :
          if (htmlGetVariable(t, (uchar *)"NAME") != NULL)
          {
	    temp = (tree_t *)calloc(sizeof(tree_t), 1);
	    memcpy(temp, t, sizeof(tree_t));
	    temp->parent = NULL;
	    temp->child  = NULL;
	    temp->prev   = flat;
	    temp->next   = NULL;
	    if (flat != NULL)
              flat->next = temp;
            flat = temp;
          };
	  break;

      case MARKUP_P :
      case MARKUP_PRE :
      case MARKUP_H1 :
      case MARKUP_H2 :
      case MARKUP_H3 :
      case MARKUP_H4 :
      case MARKUP_H5 :
      case MARKUP_H6 :
      case MARKUP_H7 :
      case MARKUP_UL :
      case MARKUP_OL :
      case MARKUP_DL :
      case MARKUP_LI :
      case MARKUP_DD :
      case MARKUP_DT :
	  temp = (tree_t *)calloc(sizeof(tree_t), 1);
	  temp->markup = MARKUP_BR;
	  temp->parent = NULL;
	  temp->child  = NULL;
	  temp->prev   = flat;
	  temp->next   = NULL;
	  if (flat != NULL)
            flat->next = temp;
          flat = temp;
          break;
    };

    if (t->child != NULL)
    {
      temp = flatten_tree(t->child);

      if (temp != NULL)
        temp->prev = flat;
      if (flat != NULL)
        flat->next = temp;
      else
        flat = temp;
    };

    if (flat != NULL)
      while (flat->next != NULL)
        flat = flat->next;

    t = t->next;
  };

  if (flat == NULL)
    return (NULL);

  while (flat->prev != NULL)
    flat = flat->prev;

  return (flat);
}


/*
 * 'update_image_size()' - Update the size of an image based upon the
 *                         printable width.
 */

static void
update_image_size(tree_t *t)	/* I - Tree entry */
{
  image_t	*img;		/* Image file */
  uchar		*width,		/* Width string */
		*height;	/* Height string */


  width  = htmlGetVariable(t, (uchar *)"WIDTH");
  height = htmlGetVariable(t, (uchar *)"HEIGHT");

  if (width != NULL && height != NULL)
  {
    if (width[strlen((char *)width) - 1] == '%')
      t->width = atoi((char *)width) * PagePrintWidth / 100.0f;
    else
      t->width = atoi((char *)width) * PagePrintWidth / 680.0f;

    if (height[strlen((char *)height) - 1] == '%')
      t->height = atoi((char *)height) * PagePrintWidth / 100.0f;
    else
      t->height = atoi((char *)height) * PagePrintWidth / 680.0f;
  };

  img = image_load((char *)htmlGetVariable(t, (uchar *)"SRC"), !OutputColor);

  if (img == NULL)
    return;

  if (width != NULL)
  {
    if (width[strlen((char *)width) - 1] == '%')
      t->width = atoi((char *)width) * PagePrintWidth / 100.0f;
    else
      t->width = atoi((char *)width) * PagePrintWidth / 680.0f;

    t->height = t->width * img->height / img->width;
  }
  else if (height != NULL)
  {
    if (height[strlen((char *)height) - 1] == '%')
      t->height = atoi((char *)height) * PagePrintWidth / 100.0f;
    else
      t->height = atoi((char *)height) * PagePrintWidth / 680.0f;

    t->width = t->height * img->width / img->height;
  }
  else
  {
    t->width  = img->width * PagePrintWidth / 680.0f;
    t->height = img->height * PagePrintWidth / 680.0f;
  };
}


/*
 * 'get_width()' - Get the width of a string in points.
 */

static float			/* O - Width in points */
get_width(uchar *s,		/* I - String to scan */
          int   typeface,	/* I - Typeface code */
          int   style,		/* I - Style code */
          int   size)		/* I - Size */
{
  uchar	*ptr;			/* Current character */
  float	width;			/* Current width */


  if (s == NULL)
    return (0.0);

  for (width = 0.0, ptr = s; *ptr != '\0'; ptr ++)
    width += char_widths[typeface][style][*ptr];

  return (width * _htmlSizes[size]);
}


/*
 * 'get_title()' - Get the title string for a document.
 */

static uchar *		/* O - Title string */
get_title(tree_t *doc)	/* I - Document */
{
  uchar	*temp;


  while (doc != NULL)
  {
    if (doc->markup == MARKUP_TITLE)
      return (htmlGetText(doc->child));
    else if (doc->child != NULL)
      if ((temp = get_title(doc->child)) != NULL)
        return (temp);
    doc = doc->next;
  };

  return (NULL);
}


/*
 * 'open_file()' - Open an output file for the current chapter.
 */

static FILE *		/* O - File pointer */
open_file(void)
{
  char	filename[255];	/* Filename */


  if (OutputFiles && pslevel > 0)
  {
    if (chapter == -1)
      sprintf(filename, "%s/cover.ps", OutputPath);
    else if (chapter == 0)
      sprintf(filename, "%s/contents.ps", OutputPath);
    else
      sprintf(filename, "%s/doc%d.ps", OutputPath, chapter);

    return (fopen(filename, "wb"));
  }
  else if (OutputFiles)
  {
    sprintf(filename, "%s/doc.pdf", OutputPath);

    return (fopen(filename, "wb"));
  }
  else if (OutputPath[0] != '\0')
    return (fopen(OutputPath, "wb"));
  else
    return (stdout);
}


/*
 * 'set_color()' - Set the current text color...
 */

static void
set_color(FILE  *out,	/* I - File to write to */
          float *rgb)	/* I - RGB color */
{
  if (rgb[0] == render_rgb[0] &&
      rgb[1] == render_rgb[1] &&
      rgb[2] == render_rgb[2])
    return;

  render_rgb[0] = rgb[0];
  render_rgb[1] = rgb[1];
  render_rgb[2] = rgb[2];

  if (pslevel > 0)
    fprintf(out, "%.2f %.2f %.2f C ", rgb[0], rgb[1], rgb[2]);
  else
    pdf_printf(out, "%.2f %.2f %.2f rg ", rgb[0], rgb[1], rgb[2]);
}


/*
 * 'set_font()' - Set the current text font.
 */

static void
set_font(FILE  *out,		/* I - File to write to */
         int   typeface,	/* I - Typeface code */
         int   style,		/* I - Style code */
         float size)		/* I - Size */
{
  static char	*typefaces = "CTHS",
		*styles[] = { "", "B", "I", "BI" };


  if (typeface == render_typeface &&
      style == render_style &&
      size == render_size)
    return;

  render_typeface = typeface;
  render_style    = style;
  render_size     = size;

  if (pslevel > 0)
    fprintf(out, "%.1f/F%c%s SF ", size, typefaces[typeface], styles[style]);
  else
    pdf_printf(out, "/F%c%s %.1f Tf ", typefaces[typeface], styles[style], size);
}


/*
 * 'set_pos()' - Set the current text position.
 */

static void
set_pos(FILE  *out,	/* I - File to write to */
        float x,	/* I - X position */
        float y)	/* I - Y position */
{
  if (fabs(render_x - x) < 0.1 && fabs(render_y - y) < 0.1)
    return;

  if (pslevel > 0)
    fprintf(out, "%.1f %.1f M", x, y);
  else if (render_x == -1.0)
    pdf_printf(out, "%.1f %.1f Td", x, y);
  else
    pdf_printf(out, "ET BT %.1f %.1f Td", x, y);

  render_x = x;
  render_y = y;
}


/*
 * 'ps_hex()' - Print binary data as a series of hexadecimal numbers.
 */

static void
ps_hex(FILE  *out,	/* I - File to print to */
       uchar *data,	/* I - Data to print */
       int   length)	/* I - Number of bytes to print */
{
  int		col;
  static char	*hex = "0123456789ABCDEF";


  col = 0;
  while (length > 0)
  {
   /*
    * Put the hex uchars out to the file; note that we don't use fprintf()
    * for speed reasons...
    */

    putc(hex[*data >> 4], out);
    putc(hex[*data & 15], out);

    data ++;
    length --;

    col = (col + 1) % 40;
    if (col == 0)
      putc('\n', out);
  };

  if (col > 0)
    putc('\n', out);
}


/*
 * 'ps_ascii85()' - Print binary data as a series of base-85 numbers.
 */

static void
ps_ascii85(FILE  *out,		/* I - File to print to */
	   uchar *data,		/* I - Data to print */
	   int   length)	/* I - Number of bytes to print */
{
  int		i;
  unsigned	b;
  uchar		c[5];


  while (length > 3)
  {
#if defined(i386) || defined(WIN32)	/* little-endian */
    b = (((((data[0] << 8) | data[1]) << 8) | data[2]) << 8) | data[3];
#else		/* big-endian */
    b = *((unsigned *)data);
#endif /* i386 || WIN32 */

    if (b == 0)
      putc('z', out);
    else
    {
      c[4] = (b % 85) + '!';
      b /= 85;
      c[3] = (b % 85) + '!';
      b /= 85;
      c[2] = (b % 85) + '!';
      b /= 85;
      c[1] = (b % 85) + '!';
      b /= 85;
      c[0] = b + '!';

      fwrite(c, 5, 1, out);
    };

    data += 4;
    length -= 4;
  };

  if (length > 0)
  {
    for (b = 0, i = length; i > 0; b = (b << 8) | data[0], data ++, i --);

    c[4] = (b % 85) + '!';
    b /= 85;
    c[3] = (b % 85) + '!';
    b /= 85;
    c[2] = (b % 85) + '!';
    b /= 85;
    c[1] = (b % 85) + '!';
    b /= 85;
    c[0] = b + '!';

    fwrite(c, length + 1, 1, out);
  };
}


#ifdef HAVE_LIBJPEG
/*
 * JPEG library destination data manager.  These routines direct
 * compressed data from libjpeg into the PDF or PostScript file.
 */

static FILE			*jpg_file;	/* JPEG file */
static uchar			jpg_buf[8192];	/* JPEG buffer */
static jpeg_destination_mgr	jpg_dest;	/* JPEG destination manager */
static struct jpeg_error_mgr	jerr;		/* JPEG error handler */


/*
 * 'jpg_init()' - Initialize the JPEG destination.
 */

static void
jpg_init(j_compress_ptr cinfo)	/* I - Compressor info */
{
  (void)cinfo;

  jpg_dest.next_output_byte = jpg_buf;
  jpg_dest.free_in_buffer   = sizeof(jpg_buf);
}


/*
 * 'jpg_empty()' - Empty the JPEG output buffer.
 */

static boolean			/* O - True if buffer written OK */
jpg_empty(j_compress_ptr cinfo)	/* I - Compressor info */
{
  (void)cinfo;

  if (pslevel > 0)
    ps_ascii85(jpg_file, jpg_buf, sizeof(jpg_buf));
  else
    pdf_write(jpg_file, jpg_buf, sizeof(jpg_buf));

  jpg_dest.next_output_byte = jpg_buf;
  jpg_dest.free_in_buffer   = sizeof(jpg_buf);

  return (TRUE);
}


/*
 * 'jpg_term()' - Write the last JPEG data to the file.
 */

static void
jpg_term(j_compress_ptr cinfo)	/* I - Compressor info */
{
  int nbytes;			/* Number of bytes to write */


  (void)cinfo;

  nbytes = sizeof(jpg_buf) - jpg_dest.free_in_buffer;

  if (pslevel > 0)
    ps_ascii85(jpg_file, jpg_buf, nbytes);
  else
    pdf_write(jpg_file, jpg_buf, nbytes);
}


/*
 * 'jpg_setup()' - Setup the JPEG compressor for writing an image.
 */

static void
jpg_setup(FILE           *out,	/* I - Output file */
          image_t        *img,	/* I - Output image */
          j_compress_ptr cinfo)	/* I - Compressor info */
{
  jpg_file    = out;

  cinfo->err  = jpeg_std_error(&jerr);

  jpeg_create_compress(cinfo);

  cinfo->dest = &jpg_dest;
  jpg_dest.init_destination    = jpg_init;
  jpg_dest.empty_output_buffer = jpg_empty;
  jpg_dest.term_destination    = jpg_term;

  cinfo->image_width      = img->width;
  cinfo->image_height     = img->height;
  cinfo->input_components = img->depth;
  cinfo->in_color_space   = img->depth == 1 ? JCS_GRAYSCALE : JCS_RGB;

  jpeg_set_defaults(cinfo);

  jpeg_set_quality(cinfo, OutputJPEG, TRUE);

  cinfo->write_JFIF_header  = FALSE;
  cinfo->write_Adobe_marker = TRUE;

  jpeg_start_compress(cinfo, FALSE);
}
#endif /* HAVE_LIBJPEG */


/*
 * 'compare_rgb()' - Compare two RGB colors...
 */

static int			/* O - -1 if rgb1<rgb2, etc. */
compare_rgb(uchar *rgb1,	/* I - First color */
            uchar *rgb2)	/* I - Second color */
{
  if (rgb1[0] < rgb2[0])
    return (-1);
  else if (rgb1[0] > rgb2[0])
    return (1);
  else if (rgb1[1] < rgb2[1])
    return (-1);
  else if (rgb1[1] > rgb2[1])
    return (1);
  else if (rgb1[2] < rgb2[2])
    return (-1);
  else if (rgb1[2] > rgb2[2])
    return (1);
  else
    return (0);
}


/*
 * 'write_image()' - Write an image to the given output file...
 */

static void
write_image(FILE     *out,	/* I - Output file */
            render_t *r)	/* I - Image to write */
{
  int		i, j, k, m,	/* Looping vars */
		ncolors;	/* Number of colors */
  uchar		*pixel,		/* Current pixel */
		*indices,	/* New indexed pixel array */
		*indptr;	/* Current index */
  int		indwidth,	/* Width of indexed line */
		indbits;	/* Bits per index */
  uchar		colors[256][3],	/* Colormap values */
		grays[256],	/* Grayscale usage */
		*match;		/* Matching color value */
  image_t 	*img;		/* Image */
#ifdef HAVE_LIBJPEG
  struct jpeg_compress_struct	cinfo;	/* JPEG compressor */
#endif /* HAVE_LIBJPEG */


 /*
  * See if we can optimize the image as indexed without color loss...
  */

  img     = r->data.image;
  ncolors = 0;

  if (pslevel != 1)
  {
    if (img->depth == 1)
    {
     /*
      * Greyscale image...
      */

      memset(grays, 0, sizeof(grays));

      for (i = img->width * img->height, pixel = img->pixels;
	   i > 0;
	   i --, pixel ++)
	if (!grays[*pixel])
	{
	  grays[*pixel] = 1;
	  ncolors ++;
	};

      if (ncolors <= 16)
      {
	for (i = 0, j = 0; i < 256; i ++)
	  if (grays[i])
	  {
	    colors[j][0] = i;
	    colors[j][1] = i;
	    colors[j][2] = i;
	    grays[i]   = j;
	    j ++;
	  };
      }
      else
        ncolors = 0;
    }
    else
    {
     /*
      * Color image...
      */

      for (i = img->width * img->height, pixel = img->pixels;
	   i > 0;
	   i --, pixel += 3)
      {
        if (ncolors > 0)
          match = (uchar *)bsearch(pixel, colors[0], ncolors, 3,
                                   (int (*)(const void *, const void *))compare_rgb);
        else
          match = NULL;

        if (match == NULL)
        {
          if (ncolors >= 256)
            break;

          colors[ncolors][0] = pixel[0];
          colors[ncolors][1] = pixel[1];
          colors[ncolors][2] = pixel[2];
          ncolors ++;

          if (ncolors > 1)
            qsort(colors[0], ncolors, 3,
                  (int (*)(const void *, const void *))compare_rgb);
        };
      };

      if (i > 0)
        ncolors = 0;
    };
  };

  if (ncolors > 0)
  {
    if (ncolors <= 2)
      indbits = 1;
    else if (ncolors <= 4)
      indbits = 2;
    else if (ncolors <= 16)
      indbits = 4;
    else
      indbits = 8;

    indwidth = (img->width * indbits + 7) / 8;
    indices  = (uchar *)malloc(indwidth * img->height);

    if (img->depth == 1)
    {
     /*
      * Convert a grayscale image...
      */

      switch (indbits)
      {
        case 1 :
	    for (i = img->height, pixel = img->pixels, indptr = indices;
		 i > 0;
		 i --)
	    {
	      for (j = img->width, k = 7; j > 0; j --, k = (k + 7) & 7, pixel ++)
		switch (k)
		{
		  case 7 :
	              *indptr = grays[*pixel] << 7;
		      break;
		  default :
	              *indptr |= grays[*pixel] << k;
		      break;
		  case 0 :
	              *indptr++ |= grays[*pixel];
		      break;
        	};

	      if (k != 7)
		indptr ++;
	    };
	    break;

        case 2 :
	    for (i = img->height, pixel = img->pixels, indptr = indices;
		 i > 0;
		 i --)
	    {
	      for (j = img->width, k = 0; j > 0; j --, k = (k + 1) & 3, pixel ++)
		switch (k)
		{
		  case 0 :
	              *indptr = grays[*pixel] << 6;
		      break;
		  case 1 :
	              *indptr |= grays[*pixel] << 4;
		      break;
		  case 2 :
	              *indptr |= grays[*pixel] << 2;
		      break;
		  case 3 :
	              *indptr++ |= grays[*pixel];
		      break;
        	};

	      if (k)
		indptr ++;
	    };
	    break;

        case 4 :
	    for (i = img->height, pixel = img->pixels, indptr = indices;
		 i > 0;
		 i --)
	    {
	      for (j = img->width, k = 0; j > 0; j --, k ^= 1, pixel ++)
		if (k)
		  *indptr++ |= grays[*pixel];
		else
		  *indptr = grays[*pixel] << 4;

	      if (k)
		indptr ++;
	    };
	    break;
      };
    }
    else
    {
     /*
      * Convert a color image...
      */

      switch (indbits)
      {
        case 1 :
	    for (i = img->height, pixel = img->pixels, indptr = indices;
		 i > 0;
		 i --)
	    {
	      for (j = img->width, k = 7; j > 0; j --, k = (k + 7) & 7, pixel += 3)
	      {
        	match = (uchar *)bsearch(pixel, colors[0], ncolors, 3,
                            	         (int (*)(const void *, const void *))compare_rgb);
	        m = (match - colors[0]) / 3;

		switch (k)
		{
		  case 7 :
	              *indptr = m << 7;
		      break;
		  default :
	              *indptr |= m << k;
		      break;
		  case 0 :
	              *indptr++ |= m;
		      break;
        	};
	      };

	      if (k != 7)
	        indptr ++;
	    };
	    break;

        case 2 :
	    for (i = img->height, pixel = img->pixels, indptr = indices;
		 i > 0;
		 i --)
	    {
	      for (j = img->width, k = 0; j > 0; j --, k = (k + 1) & 3, pixel += 3)
	      {
        	match = (uchar *)bsearch(pixel, colors[0], ncolors, 3,
                           	         (int (*)(const void *, const void *))compare_rgb);
	        m = (match - colors[0]) / 3;

		switch (k)
		{
		  case 0 :
	              *indptr = m << 6;
		      break;
		  case 1 :
	              *indptr |= m << 4;
		      break;
		  case 2 :
	              *indptr |= m << 2;
		      break;
		  case 3 :
	              *indptr++ |= m;
		      break;
        	};
	      };

	      if (k)
	        indptr ++;
	    };
	    break;

        case 4 :
	    for (i = img->height, pixel = img->pixels, indptr = indices;
		 i > 0;
		 i --)
	    {
	      for (j = img->width, k = 0; j > 0; j --, k ^= 1, pixel += 3)
	      {
        	match = (uchar *)bsearch(pixel, colors[0], ncolors, 3,
                        	         (int (*)(const void *, const void *))compare_rgb);
	        m = (match - colors[0]) / 3;

		if (k)
		  *indptr++ |= m;
		else
		  *indptr = m << 4;
	      };

	      if (k)
	        indptr ++;
	    };
	    break;

        case 8 :
	    for (i = img->height, pixel = img->pixels, indptr = indices;
		 i > 0;
		 i --)
	    {
	      for (j = img->width; j > 0; j --, pixel += 3, indptr ++)
	      {
        	match = (uchar *)bsearch(pixel, colors[0], ncolors, 3,
                        	         (int (*)(const void *, const void *))compare_rgb);
	        *indptr = (match - colors[0]) / 3;
	      };
	    };
	    break;
      };
    };
  };

 /*
  * Now write the image...
  */

  switch (pslevel)
  {
    case 0 : /* PDF */
	pdf_printf(out, "q %.1f 0 0 %.1f %.1f %.1f cm\n", r->width, r->height,
	           r->x, r->y);
        pdf_puts("BI", out);

	if (ncolors > 0)
	{
	  pdf_printf(out, "/CS[/I/RGB %d<", ncolors - 1);
	  for (i = 0; i < ncolors; i ++)
	    pdf_printf(out, "%02X%02X%02X", colors[i][0], colors[i][1], colors[i][2]);
	  pdf_puts(">]", out);
        }
	else if (img->depth == 1)
          pdf_puts("/CS/G", out);
        else
          pdf_puts("/CS/RGB", out);

        pdf_puts("/I true", out);

        if (ncolors > 0)
	{
  	  pdf_printf(out, "/W %d/H %d/BPC %d ID\n",
               	     img->width, img->height, indbits); 

  	  pdf_write(out, indices, indwidth * img->height);
	}
#ifdef HAVE_LIBJPEG
	else if (OutputJPEG)
	{
  	  pdf_printf(out, "/W %d/H %d/BPC 8/F/DCT ID\n",
               	     img->width, img->height); 

	  jpg_setup(out, img, &cinfo);

	  for (i = img->height, pixel = img->pixels;
	       i > 0;
	       i --, pixel += img->width * img->depth)
	    jpeg_write_scanlines(&cinfo, &pixel, 1);

	  jpeg_finish_compress(&cinfo);
	  jpeg_destroy_compress(&cinfo);
        }
#endif /* HAVE_LIBJPEG */
	else
	{
  	  pdf_printf(out, "/W %d/H %d/BPC 8 ID\n",
               	     img->width, img->height); 

  	  pdf_write(out, img->pixels, img->width * img->height * img->depth);
        };

	pdf_puts("EI\nQ\n", out);
        break;

    case 1 : /* PostScript, Level 1 */
        fputs("GS", out);
	fprintf(out, "[%.1f 0 0 %.1f %.1f %.1f]CM", r->width, r->height,
	        r->x, r->y);

	fprintf(out, "/picture %d string def\n", img->width * img->depth);

	if (img->depth == 1)
	  fprintf(out, "%d %d 8 [%d 0 0 %d 0 %d] {currentfile picture readhexstring pop} image\n",
        	  img->width, img->height,
        	  img->width, -img->height,
        	  img->height); 
	else
	  fprintf(out, "%d %d 8 [%d 0 0 %d 0 %d] {currentfile picture readhexstring pop} false 3 colorimage\n",
        	  img->width, img->height,
        	  img->width, -img->height,
        	  img->height); 

	ps_hex(out, img->pixels, img->width * img->height * img->depth);

	fputs("GR\n", out);
        break;

    case 2 : /* PostScript, Level 2 */
        fputs("GS", out);
	fprintf(out, "[%.1f 0 0 %.1f %.1f %.1f]CM", r->width, r->height,
	        r->x, r->y);

        if (ncolors > 0)
        {
	  fprintf(out, "[/Indexed/DeviceRGB %d<", ncolors - 1);
	  for (i = 0; i < ncolors; i ++)
	    fprintf(out, "%02X%02X%02X", colors[i][0], colors[i][1], colors[i][2]);
	  fputs(">]setcolorspace", out);

	  fprintf(out, "<<"
	               "/ImageType 1"
	               "/Width %d"
	               "/Height %d"
	               "/BitsPerComponent %d"
	               "/ImageMatrix[%d 0 0 %d 0 %d]"
	               "/Decode[0 %d]"
		       "/Interpolate true"
	               "/DataSource currentfile/ASCII85Decode filter"
	               ">>image\n",
	          img->width, img->height, indbits,
        	  img->width, -img->height, img->height,
        	  (1 << indbits) - 1);

	  ps_ascii85(out, indices, indwidth * img->height);
          fputs("~>\n", out);
        }
#ifdef HAVE_LIBJPEG
	else if (OutputJPEG)
	{
	  if (img->depth == 1)
	    fputs("/DeviceGray setcolorspace", out);
	  else
	    fputs("/DeviceRGB setcolorspace", out);

	  fprintf(out, "<<"
	               "/ImageType 1"
	               "/Width %d"
	               "/Height %d"
	               "/BitsPerComponent 8"
	               "/ImageMatrix[%d 0 0 %d 0 %d]"
	               "/Decode[%s]"
		       "/Interpolate true"
	               "/DataSource currentfile/ASCII85Decode filter/DCTDecode filter"
	               ">>image\n",
	          img->width, img->height,
        	  img->width, -img->height, img->height,
        	  img->depth == 1 ? "0 1" : "0 1 0 1 0 1");

	  jpg_setup(out, img, &cinfo);

	  for (i = img->height, pixel = img->pixels;
	       i > 0;
	       i --, pixel += img->width * img->depth)
	    jpeg_write_scanlines(&cinfo, &pixel, 1);

	  jpeg_finish_compress(&cinfo);
	  jpeg_destroy_compress(&cinfo);

          fputs("~>\n", out);
        }
#endif /* HAVE_LIBJPEG */
        else
        {
	  if (img->depth == 1)
	    fputs("/DeviceGray setcolorspace", out);
	  else
	    fputs("/DeviceRGB setcolorspace", out);

	  fprintf(out, "<<"
	               "/ImageType 1"
	               "/Width %d"
	               "/Height %d"
	               "/BitsPerComponent 8"
	               "/ImageMatrix[%d 0 0 %d 0 %d]"
	               "/Decode[%s]"
		       "/Interpolate true"
	               "/DataSource currentfile/ASCII85Decode filter"
	               ">>image\n",
	          img->width, img->height,
        	  img->width, -img->height, img->height,
        	  img->depth == 1 ? "0 1" : "0 1 0 1 0 1");

	  ps_ascii85(out, img->pixels, img->width * img->height * img->depth);
          fputs("~>\n", out);
        };

	fputs("GR\n", out);
        break;
  };

  if (ncolors > 0)
    free(indices);
}


/*
 * 'write_prolog()' - Write the file prolog...
 */

static void
write_prolog(FILE *out,		/* I - Output file */
             int  pages,	/* I - Number of pages (0 if not known) */
             uchar *title,	/* I - Title of document */
             uchar *author,	/* I - Author of document */
             uchar *creator,	/* I - Application that generated the HTML file */
             uchar *copyright)	/* I - Copyright (if any) on the document */
{
  int		i,		/* Looping var */
		encoding_object;/* Font encoding object */
  time_t	curtime;	/* Current time */
  struct tm	*curdate;	/* Current date */
  static char	*typefaces[] =	/* Long font names */
  {
    "Courier",
    "Courier-Bold",
    "Courier-Oblique",
    "Courier-BoldOblique",
    "Times-Roman",
    "Times-Bold",
    "Times-Italic",
    "Times-BoldItalic",
    "Helvetica",
    "Helvetica-Bold",
    "Helvetica-Oblique",
    "Helvetica-BoldOblique",
    "Symbol",
    "Symbol",
    "Symbol",
    "Symbol"
  };


  curtime = time(NULL);
  curdate = gmtime(&curtime);

  if (pslevel > 0)
  {
   /*
    * Write PostScript prolog stuff...
    */

    fputs("%!PS-Adobe-3.0\n", out);
    fprintf(out, "%%%%BoundingBox: 0 0 %d %d\n", PageWidth, PageLength);
    fprintf(out,"%%%%LanguageLevel: %d\n", pslevel);
    fputs("%%Creator: htmldoc " SVERSION " Copyright 1997-1998 Michael Sweet, All Rights Reserved.\n", out);
    fprintf(out, "%%%%CreationDate: D:%04d%02d%02d%02d%02d%02dZ\n",
            curdate->tm_year + 1900, curdate->tm_mon + 1, curdate->tm_mday,
            curdate->tm_hour, curdate->tm_min, curdate->tm_sec);
    if (title != NULL)
      fprintf(out, "%%%%docTitle: %s\n", title);
    if (author != NULL)
      fprintf(out, "%%%%docAuthor: %s\n", author);
    if (creator != NULL)
      fprintf(out, "%%%%docGenerator: %s\n", creator);
    if (copyright != NULL)
      fprintf(out, "%%%%docCopyright: %s\n", copyright);
    if (pages > 0)
      fprintf(out, "%%Pages: %d\n", pages);
    else
      fputs("%%Pages: (atend)\n", out);
    fputs("%%DocumentNeededResources: font Symbol\n", out);
    fputs("%%+ font Times-Roman Times-Bold Times-Italic Times-BoldItalic\n", out);
    fputs("%%+ font Helvetica Helvetica-Bold Helvetica-Oblique Helvetica-BoldOblique\n", out);
    fputs("%%+ font Courier Courier-Bold Courier-Oblique Courier-BoldOblique\n", out);
    fputs("%%DocumentData: Clean7bit\n", out);
    fputs("%%EndComments\n", out);

    fputs("%%BeginProlog\n", out);

   /*
    * Oh, joy, another font encoding.  For now assume that all documents use
    * ISO-8859-1 encoding (this covers about 1/2 of the human population)...
    */

    fputs("/iso8859encoding[", out);
    fputs("/.notdef/.notdef/.notdef/.notdef", out);
    fputs("/.notdef/.notdef/.notdef/.notdef", out);
    fputs("/.notdef/.notdef/.notdef/.notdef", out);
    fputs("/.notdef/.notdef/.notdef/.notdef", out);
    fputs("/.notdef/.notdef/.notdef/.notdef", out);
    fputs("/.notdef/.notdef/.notdef/.notdef", out);
    fputs("/.notdef/.notdef/.notdef/.notdef", out);
    fputs("/.notdef/.notdef/.notdef/.notdef", out);
    fputs("/space/exclam/quotedbl/numbersign", out);
    fputs("/dollar/percent/ampersand/quoteright", out);
    fputs("/parenleft/parenright/asterisk/plus", out);
    fputs("/comma/hyphen/period/slash", out);
    fputs("/zero/one/two/three/four/five/six/seven", out);
    fputs("/eight/nine/colon/semicolon", out);
    fputs("/less/equal/greater/question", out);
    fputs("/at/A/B/C/D/E/F/G", out);
    fputs("/H/I/J/K/L/M/N/O", out);
    fputs("/P/Q/R/S/T/U/V/W", out);
    fputs("/X/Y/Z/bracketleft", out);
    fputs("/backslash/bracketright/asciicircum/underscore", out);
    fputs("/quoteleft/a/b/c/d/e/f/g", out);
    fputs("/h/i/j/k/l/m/n/o", out);
    fputs("/p/q/r/s/t/u/v/w", out);
    fputs("/x/y/z/braceleft", out);
    fputs("/bar/braceright/asciitilde/guilsinglright", out);
    fputs("/fraction/florin/quotesingle/quotedblleft", out);
    fputs("/guilsinglleft/fi/fl/endash", out);
    fputs("/dagger/daggerdbl/bullet/quotesinglbase", out);
    fputs("/quotedblbase/quotedblright/ellipsis/trademark", out);
    fputs("/dotlessi/grave/acute/circumflex", out);
    fputs("/tilde/macron/breve/dotaccent", out);
    fputs("/dieresis/perthousand/ring/cedilla", out);
    fputs("/Ydieresis/hungarumlaut/ogonek/caron", out);
    fputs("/emdash/exclamdown/cent/sterling", out);
    fputs("/currency/yen/brokenbar/section", out);
    fputs("/dieresis/copyright/ordfeminine/guillemotleft", out);
    fputs("/logicalnot/hyphen/registered/macron", out);
    fputs("/degree/plusminus/twosuperior/threesuperior", out);
    fputs("/acute/mu/paragraph/periodcentered", out);
    fputs("/cedilla/onesuperior/ordmasculine/guillemotright", out);
    fputs("/onequarter/onehalf/threequarters/questiondown", out);
    fputs("/Agrave/Aacute/Acircumflex/Atilde", out);
    fputs("/Adieresis/Aring/AE/Ccedilla", out);
    fputs("/Egrave/Eacute/Ecircumflex/Edieresis", out);
    fputs("/Igrave/Iacute/Icircumflex/Idieresis", out);
    fputs("/Eth/Ntilde/Ograve/Oacute", out);
    fputs("/Ocircumflex/Otilde/Odieresis/multiply", out);
    fputs("/Oslash/Ugrave/Uacute/Ucircumflex", out);
    fputs("/Udieresis/Yacute/Thorn/germandbls", out);
    fputs("/agrave/aacute/acircumflex/atilde", out);
    fputs("/adieresis/aring/ae/ccedilla", out);
    fputs("/egrave/eacute/ecircumflex/edieresis", out);
    fputs("/igrave/iacute/icircumflex/idieresis", out);
    fputs("/eth/ntilde/ograve/oacute", out);
    fputs("/ocircumflex/otilde/odieresis/divide", out);
    fputs("/oslash/ugrave/uacute/ucircumflex", out);
    fputs("/udieresis/yacute/thorn/ydieresis]def", out);

   /*
    * Fonts...
    */

    for (i = 0; i < 16; i ++)
    {
      fprintf(out, "/%s findfont\n", typefaces[i]);
      if (i < 12)
	fputs("dup length dict begin"
              "{1 index/FID ne{def}{pop pop}ifelse}forall"
              "/Encoding iso8859encoding def"
              "	currentdict end\n", out);
      fprintf(out, "/%s exch definefont pop\n", font_names[i]);
    };

   /*
    * Now for the macros...
    */

    fputs("/BD{bind def}bind def", out);
    fputs("/B{dup 0 exch rlineto exch 0 rlineto neg 0 exch rlineto closepath stroke}BD", out);
    if (!OutputColor)
      fputs("/C{0.08 mul exch 0.61 mul add exch 0.31 mul add setgray}BD", out);
    else
      fputs("/C{setrgbcolor} BD", out);
    fputs("/CM{concat}BD", out);
    fputs("/F{dup 0 exch rlineto exch 0 rlineto neg 0 exch rlineto closepath fill}BD", out);
    fputs("/GS{gsave}BD", out);
    fputs("/GR{grestore}BD", out);
    fputs("/L{0 rlineto stroke}BD", out);
    fputs("/M{moveto}BD", out);
    fputs("/S{show}BD", out);
    fputs("/SF{findfont exch scalefont setfont}BD", out);
    fputs("/SP{showpage}BD", out);
    fputs("/T{translate}BD\n", out);

    if (background_image != NULL)
      ps_write_background(out);

    fputs("%%EndProlog\n", out);
  }
  else
  {
   /*
    * Write PDF prolog stuff...
    */

    fputs("%PDF-1.2\n", out);
    fputs("%\342\343\317\323\n", out);
    num_objects = 1;

    info_object = num_objects;
    objects[num_objects] = ftell(out);
    fprintf(out, "%d 0 obj", num_objects);
    fputs("<<", out);
    fputs("/Producer(htmldoc " SVERSION " Copyright 1997-1998 Michael Sweet, All Rights Reserved.)", out);
    fprintf(out, "/CreationDate(D:%04d%02d%02d%02d%02d%02dZ)",
            curdate->tm_year + 1900, curdate->tm_mon + 1, curdate->tm_mday,
            curdate->tm_hour, curdate->tm_min, curdate->tm_sec);

    if (title != NULL)
    {
      fputs("/Title", out);
      write_string(out, title, 0);
    };

    if (author != NULL)
    {
      fputs("/Author", out);
      write_string(out, author, 0);
    };

    if (creator != NULL)
    {
      fputs("/Creator", out);
      write_string(out, creator, 0);
    };

    fputs(">>", out);
    fputs("endobj\n", out);

    num_objects ++;
    encoding_object = num_objects;
    objects[num_objects] = ftell(out);

    fprintf(out, "%d 0 obj", encoding_object);
    fputs("<<", out);
    fputs("/Type/Encoding", out);
    fputs("/BaseEncoding/WinAnsiEncoding", out);
    fputs(">>", out);
    fputs("endobj\n", out);

    for (i = 0; i < 16; i ++)
    {
      num_objects ++;
      font_objects[i] = num_objects;
      objects[num_objects] = ftell(out);

      fprintf(out, "%d 0 obj", font_objects[i]);
      fputs("<<", out);
      fputs("/Type/Font", out);
      fputs("/Subtype/Type1", out);
      fprintf(out, "/BaseFont/%s", typefaces[i]);
      if (i < 12) /* Use native encoding for symbols */
	fprintf(out, "/Encoding %d 0 R", encoding_object);
      fputs(">>", out);
      fputs("endobj\n", out);
    };

    if (background_image != NULL)
      pdf_write_background(out);
  };
}


/*
 * 'write_string()' - Write a text entity.
 */

static void
write_string(FILE  *out,	/* I - Output file */
             uchar *s,		/* I - String */
	     int   compress)	/* I - Compress output? */
{
  if (compress)
    pdf_write(out, (uchar *)"(", 1);
  else
    putc('(', out);

  while (*s != '\0')
  {
    if (*s < 32 || *s > 126)
    {
      if (compress)
        pdf_printf(out, "\\%o", *s);
      else
        fprintf(out, "\\%o", *s);
    }
    else if (compress)
    {
      if (*s == '(' || *s == ')' || *s == '\\')
        pdf_write(out, (uchar *)"\\", 1);

      pdf_write(out, s, 1);
    }
    else
    {
      if (*s == '(' || *s == ')' || *s == '\\')
        putc('\\', out);

      putc(*s, out);
    };

    s ++;
  };

  if (compress)
    pdf_write(out, (uchar *)")", 1);
  else
    putc(')', out);
}


/*
 * 'write_text()' - Write a text entity.
 */

static void
write_text(FILE     *out,	/* I - Output file */
           render_t *r)		/* I - Text entity */
{
  set_color(out, r->data.text.rgb);
  set_font(out, r->data.text.typeface, r->data.text.style, r->data.text.size);
  set_pos(out, r->x, r->y);

  write_string(out, r->data.text.buffer, pslevel == 0);

  if (pslevel > 0)
    fputs("S\n", out);
  else
    pdf_puts("Tj\n", out);

  render_x += r->width;
}


/*
 * 'write_trailer()' - Write the file trailer.
 */

static void
write_trailer(FILE *out,	/* I - Output file */
              int  pages)	/* I - Number of pages in file */
{
  int	i,			/* Looping var */
	offset;			/* Offset to xref table in PDF file */


  if (pslevel > 0)
  {
   /*
    * PostScript...
    */

    if (pages > 0)
      fprintf(out, "%%%%Pages: %d\n", pages);

    fputs("%%EOF\n", out);
  }
  else
  {
   /*
    * PDF...
    */

    num_objects ++;
    root_object = num_objects;
    objects[num_objects] = ftell(out);
    fprintf(out, "%d 0 obj", num_objects);
    fputs("<<", out);
    fputs("/Type/Catalog", out);
    fprintf(out, "/Pages %d 0 R", pages_object);
    fprintf(out, "/Names %d 0 R", names_object);
    if (outline_object > 0)
    {
      fprintf(out, "/Outlines %d 0 R", outline_object);
      fputs("/PageMode/UseOutlines", out);
      fprintf(out, "/OpenAction[%d 0 R/XYZ null null null]",
              pages_object + 3 * chapter_starts[1] + 1);
    }
    else
      fputs("/PageMode/UseNone", out);

    fputs(">>", out);
    fputs("endobj\n", out);

    offset = ftell(out);

    fputs("xref\n", out);
    fprintf(out, "0 %d \n", num_objects + 1);
    fputs("0000000000 65535 f \n", out);
    for (i = 1; i <= num_objects; i ++)
      fprintf(out, "%010d 00000 n \n", objects[i]);

    fputs("trailer\n", out);
    fputs("<<", out);
    fprintf(out, "/Size %d", num_objects + 1);
    fprintf(out, "/Root %d 0 R", root_object);
    fprintf(out, "/Info %d 0 R", info_object);
    fputs(">>\n", out);
    fputs("startxref\n", out);
    fprintf(out, "%d\n", offset);
    fputs("%%EOF\n", out);
  };
}


/*
 * 'pdf_open_stream()' - Open a deflated output stream in a PDF file.
 */

static void
pdf_open_stream(FILE *out)	/* I - Output file */
{
  REF(out);

#ifdef HAVE_LIBZ
  if (!Compression)
    return;

  compressor.zalloc = (alloc_func)0;
  compressor.zfree  = (free_func)0;
  compressor.opaque = (voidpf)0;

  deflateInit(&compressor, Compression);

  compressor.next_out  = (Bytef *)comp_buffer;
  compressor.avail_out = sizeof(comp_buffer);
#endif /* HAVE_LIBZ */
}


/*
 * 'pdf_close_stream()' - Close a deflated output string in a PDF file.
 */

static void
pdf_close_stream(FILE *out)	/* I - Output file */
{
#ifdef HAVE_LIBZ
  if (!Compression)
    return;

  while (deflate(&compressor, Z_FINISH) != Z_STREAM_END)
  {
    fwrite(comp_buffer, (uchar *)compressor.next_out - (uchar *)comp_buffer, 1, out);
    compressor.next_out  = (Bytef *)comp_buffer;
    compressor.avail_out = sizeof(comp_buffer);
  };

  if ((uchar *)compressor.next_out > (uchar *)comp_buffer)
    fwrite(comp_buffer, (uchar *)compressor.next_out - (uchar *)comp_buffer, 1, out);

  deflateEnd(&compressor);
#endif /* HAVE_LIBZ */
}


/*
 * 'pdf_puts()' - Write a character string to a compressed stream in a PDF file.
 */

static void
pdf_puts(char *s,	/* I - String to write */
         FILE *out)	/* I - Output file */
{
  pdf_write(out, (uchar *)s, strlen(s));
}


/*
 * 'pdf_printf()' - Write a formatted character string to a compressed stream
 *                  in a PDF file.
 */

static void
pdf_printf(FILE *out,		/* I - Output file */
           char *format,	/* I - Format string */
           ...)			/* I - Additional args as necessary */
{
  int		length;		/* Length of output string */
  char		buf[10240];	/* Output buffer */
  va_list	ap;		/* Argument pointer */


  va_start(ap, format);
  length = vsprintf(buf, format, ap);
  va_end(ap);

  pdf_write(out, (uchar *)buf, length);
}


/*
 * 'pdf_write()' - Write data to a compressed stream in a PDF file.
 */

static void
pdf_write(FILE *out,	/* I - Output file */
          uchar *buf,	/* I - Buffer */
          int  length)	/* I - Number of bytes to write */
{
#ifdef HAVE_LIBZ
  if (Compression)
  {
    compressor.next_in  = buf;
    compressor.avail_in = length;

    while (compressor.avail_in > 0)
    {
      if (compressor.avail_out < (sizeof(comp_buffer) / 8))
      {
	fwrite(comp_buffer, (uchar *)compressor.next_out - (uchar *)comp_buffer, 1, out);
	compressor.next_out  = (Bytef *)comp_buffer;
	compressor.avail_out = sizeof(comp_buffer);
      };

      deflate(&compressor, Z_NO_FLUSH);
    };
  }
  else
#endif /* HAVE_LIBZ */
    fwrite(buf, length, 1, out);
}


/*
 * End of "$Id: ps-pdf.cpp,v 1.79 1999/01/07 17:28:57 mike Exp $".
 */
