/* 
   sitecopy, for managing remote web sites.
   Copyright (C) 1998, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* Protocol driver interface, second attempt.
 * The actual driver handles the interface to the remote site.
 */

#ifndef PROTO_H
#define PROTO_H

#include <sys/types.h>

#include "common.h"

#define PROTO_OK 0
/* Misc error */
#define PROTO_ERROR -1
/* Hostname lookup failed */
#define PROTO_LOOKUP -2
/* Local hostname could not be found */
#define PROTO_LOCALHOST -3
/* Could not connect to remote host */
#define PROTO_CONNECT -4
/* Could not find local file */
#define PROTO_FILE -5
/* The operation is not supported by the driver */
#define PROTO_UNSUPPORTED -6
/* Could not authorize user on server */
#define PROTO_AUTH -7

struct proto_file_t {
    char *filename;
    char *directory;
    bool isdir;
    size_t size;
    time_t modtime;
    int depth;
    struct proto_file_t *next;
};

struct proto_driver {

    /* Driver initialization.
     *
     * Returns:
     *  PROTO_OK       if initialized okay
     *  PROTO_CONNECT  if could not connect to the remote server
     *  PROTO_AUTH     if could not authorise user on the server
     *  PROTO_LOOKUP   could not lookup remote host name
     */
    int (*init) ( const char *remote_root,
		  const char *hostname, const int port,
		  const char *username, const char *password );

    /* Called when the driver has been finished with */
    int (*finish) ( void );

    /* Perform the file operations - these should return one of 
     * the PROTO_ codes */
    int (*file_move) ( const char *from, const char *to );
    int (*file_upload) ( const char *local, const char *remote,
			 const bool ascii );
    int (*file_download) ( const char *local, const char *remote,
			   const int remotesize,
			   const bool ascii );
    int (*file_delete) ( const char *filename );
    int (*file_chmod) ( const char *filename, const mode_t mode );
    /* Perform the directory operations */
    int (*dir_create) ( const char *dirname );
    int (*dir_remove) ( const char *dirname );
    /* Creates a link with given target */
    int (*link_create) ( const char *link, const char *target );
    /* Changes a link to point to a different target */
    int (*link_change) ( const char *link, const char *target );
    /* Deletes a link */
    int (*link_delete) ( const char *link );
    
    /* Fetches the list of files.
     * Returns:
     *   PROTO_OK     if the fetch was successful
     *   PROTO_ERROR  if the fetch failed
     * The files list must be returned in dynamically allocated 
     * memory, which will be freed by the caller.
     */
    int (*fetch_list) ( const char *dirname, struct proto_file_t **files );
    char *service_name; /* The name of the port which this protocol
			 * will use by default */
    char *protocol_name; /* The user-visible name for this protocol */
    char *last_error; /* The last error used. */
};

#endif /* PROTO_H */
