#include <config.h>
#include <gnome.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "editor.h"
#include "helpers.h"
#include "page.h"
#include "preferences.h"
#include "site.h"

extern Site *current_site;
extern Page *current_page;
extern Preferences *cfg;

#ifndef STDIN
#define STDIN 0
#endif

#ifndef STDOUT
#define STDOUT 1
#endif

static void screem_helper_execute_all( void );

/*
 * screem_helper_new:
 *
 * Creates and initialises a new helper struct
 *
 * return values: a Helper
 */
Helper* screem_helper_new()
{
	Helper *helper;

	helper = (Helper*)g_malloc( sizeof( Helper ) );

	helper->path = NULL;
	helper->mime_type = NULL;

	return helper;
}

/*
 * screem_helper_destroy:
 * @helper: the helper to destroy
 *
 * Destroys the given helper freeing up any memory used by it.
 *
 * return values: none
 */
void screem_helper_destroy( Helper *helper )
{
	g_return_if_fail( helper != NULL );

	g_free( helper->path );
	g_free( helper->mime_type );

	g_free( helper );
}

/*
 * screem_helper_locate:
 * @path: the pathname of the helper to locate
 *
 * locates the helper with the given pathname
 *
 * return values: a Helper pointer
 */
Helper* screem_helper_locate( gchar *path )
{
	GList *list;
	Helper *helper = NULL;

	g_return_val_if_fail( path != NULL, NULL );

	for( list = cfg->helpers; ( list ) && ( !helper ); list = list->next ){
		helper = (Helper*)list->data;
		if( strcmp( path, helper->path ) )
			helper = NULL;
	}

	return helper;
}

/*
 * screem_helper_add:
 * @helper: the helper to add to the list
 *
 * adds the given helper to the list of helpers
 *
 * return values: none
 */
void screem_helper_add( Helper *helper )
{
	g_return_if_fail( helper != NULL );

	cfg->helpers = g_list_append( cfg->helpers, helper );
}

/*
 * screem_helper_remove:
 * @path: the pathname of the helper to remove
 *
 * removes the helper with the given pathname from the list of helpers
 *
 * return values: a Helper pointer
 */
Helper* screem_helper_remove( gchar *path )
{
	Helper *helper;

	helper = screem_helper_locate( path );

	if( helper )
		cfg->helpers = g_list_remove( cfg->helpers, helper );

	return helper;
}

/*
 * screem_helper_execute:
 * @path: the pathname of the helper to execute
 *
 * executes the helper
 *
 * return values: none
 */
void screem_helper_execute( gchar *path )
{
	Helper *helper;
	Page *page;
	int p[ 2 ];
	int q[ 2 ]; 
	gchar *backup;
	gint status;
	FILE *stream;

	g_return_if_fail( path != NULL );

	helper = screem_helper_locate( path );

	g_return_if_fail( helper != NULL );

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_if_fail( page != NULL );

	screem_editor_buffer_text();

	backup = page->data;

	pipe( p );
	pipe( q );
	switch( fork() ) {
		/* error */
	case -1:
		break;
		/* child */
	case 0:
		close( STDIN );
                dup( p[ 0 ] );
                close( p[ 1 ] );
                close( STDOUT );
                dup( q[ 1 ] );
                close( q[ 0 ] );
                execl( "/bin/sh", "/bin/sh", "-c", path, NULL );
                _exit( 0 );
		break;
		/* parent */
	default:
		close( p[ 0 ] );
                close( q[ 1 ] );
                stream = fdopen( p[ 1 ], "w" );
                fprintf( stream, "%s", page->data );
                fclose( stream );
                screem_page_read( page, q[ 0 ] );
                waitpid( -1, &status, 0 );

		if( ! status ) {
                        g_free( backup );
			screem_editor_clear();
			screem_editor_insert( 0, page->data );
                } else {
                        g_free( page->data );
                        page->data = backup;
                }
		break;
	}
}

/*
 * screem_helper_menu:
 *
 * builds the menu heirachy for the helpers menu(s)
 *
 * return values: none
 */
GtkWidget *screem_helper_menu()
{
	GtkWidget *menu_item;
	GtkWidget *menu;

	GtkWidget *item;
	GtkWidget *menu2;

	GList *list;
	Helper *helper;

	menu_item = gtk_menu_item_new_with_label( _( "Exec Helpers" ) );
	gtk_widget_show( menu_item );

	menu = gtk_menu_new();
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( menu_item ), menu );

	item = gtk_menu_item_new_with_label( _( "All Helpers" ) );
	gtk_signal_connect( GTK_OBJECT( item ), "activate",
			    GTK_SIGNAL_FUNC( screem_helper_execute_all ), 0 );
	gtk_menu_append( GTK_MENU( menu ), item );
	gtk_widget_show( item );

	for( list = cfg->helpers; list; list = list->next ) {
		helper = list->data;
		if( helper ) {
			item = gtk_menu_item_new_with_label(helper->path);
			gtk_signal_connect_object( GTK_OBJECT( item ), "activate", GTK_SIGNAL_FUNC( screem_helper_execute ), (gpointer)helper->path );
			gtk_menu_append( GTK_MENU( menu ), item );
			gtk_widget_show( item );
		}
	}

	return menu_item;
}

static void screem_helper_execute_all()
{
	GList *list;
	Helper *helper;

	for( list = cfg->helpers; list; list = list->next ) {
		helper = (Helper*)list->data;
		if( helper )
			screem_helper_execute( helper->path );
	}
}
