/*  Screem:  pageUI.c,
 *  a tidied up page.c, separating ui from backend, this is the GUI file
 * 
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>

#include "editor.h"
#include "page.h"
#include "pageUI.h"
#include "preferences.h"
#include "preview.h"
#include "site.h"
#include "siteUI.h"
#include "siteTreeUI.h"
#include "support.h"

extern GtkWidget *app;
extern Site *current_site;
extern Page *current_page;
extern Preferences *cfg;

extern GList *loaded_pages;

gint open_pages = -1;

static void screem_page_wizard_clicked_cb( GtkWidget *widget, gint button,
					   Page *page );

static void recent_proxy( GtkWidget *widget, gchar *filename );

/** 
 * screem_page_insert:
 * @page: the page to insert into the editor
 *
 * inserts the page into the editor
 *
 * return values: boolean
 */
gboolean screem_page_insert( Page *page )
{
	const gchar *page_path;
	gchar *pname;

	GtkWidget *notebook = gtk_object_get_data( GTK_OBJECT( app ), 
						   "notebook" );
	GtkWidget *pagebook = gtk_object_get_data( GTK_OBJECT( app ),
						   "pagebook" );
	GtkWidget *label;
	GtkWidget *junk;

	gint num;

	g_return_val_if_fail( page != NULL, FALSE );

	if( ! screem_page_load( page ) ) {
		g_print( "load failed\n" );
		return FALSE;
	}

	/* if we are working on a site then we don't want a page open
	   twice */

	if( ( num = screem_page_get_tab( page ) ) != -1 ) {
		/* already open, switch to it */
		gtk_notebook_set_page( GTK_NOTEBOOK( pagebook ), num );
		return TRUE;
	}
	
	if( current_site )
		screem_site_set_current_page( current_site, page );
	else
		current_page = page;

	page_path = screem_page_get_pathname( page );
	pname = g_dirname( page_path );
	chdir( pname );

	/* display in the editor */
	screem_editor_clear();
	screem_editor_insert( 0, page->data );
	screem_editor_enable();

	gtk_notebook_set_page( GTK_NOTEBOOK( notebook ), 0 );

	if( ! current_site ) {
		/* add to recent pages list */
		cfg->recent_pages = g_list_append( cfg->recent_pages, 
						   g_strdup( page_path ) );
		build_recent_pages_menu();
	}

	/* create the new page tab */
	label = gtk_label_new( page_path + strlen( pname ) + 1 );
	junk = gtk_hseparator_new();
	gtk_widget_show( junk );
	gtk_widget_show( label );
	gtk_notebook_append_page( GTK_NOTEBOOK( pagebook ), junk, label );
	gtk_widget_set_usize( junk, 0, 0 );

	if( current_site ) {
		screem_site_inc_open_pages( current_site );
		num = screem_site_get_open_pages( current_site );
	} else {
		open_pages ++;
		num = open_pages;
	}

	screem_page_set_tab( page, num );

	gtk_notebook_set_page( GTK_NOTEBOOK( pagebook ), num );

	g_free( pname );

	return TRUE;
}

/**
 * screem_page_open:
 *
 * opens the fileselector to select a page to open
 *
 * return values: boolean
 */
gboolean screem_page_open()
{
	Page *page;
	gchar *filename;

	/* do we already have a site loaded? */
	if( current_site ) {
		screem_site_save_confirmation( current_site );
		screem_site_destroy( current_site );
		current_site = NULL;
	}

	/* select the page to open */
	filename = file_select( _( "Select page to open" ) );

	/* we didn't select one */
	if( ! filename )
		return FALSE;

	return screem_page_open_with_filename( filename );
}


/**
 * screem_page_open_with_filename:
 * @filename: the filename of the page to open
 *
 * opens a page with the given filename
 *
 * return values: boolean
 */
gboolean screem_page_open_with_filename( const gchar *filename )
{
	Page *page;
	gchar *path;
	gchar *pname;
	gchar cwd[ 16384 ];

	const gchar *pathname;

	g_return_val_if_fail( filename != NULL, FALSE );

	/* is the filename absolute? */
	if( filename[ 0 ] != G_DIR_SEPARATOR ) {
		/* no */
		getcwd( cwd, 16384 );
		path = g_strconcat( cwd, G_DIR_SEPARATOR_S, filename, NULL );
	} else {
		path = g_strdup( filename );
	}

	page = screem_page_new();
	screem_page_set_pathname( page, path );

	g_free( path );

	if( ! current_site ) {
		/* add to list of opened pages */
		loaded_pages = g_list_append( loaded_pages, page );
	}

	return screem_page_insert( page );
}

/**
 * screem_page_save_as:
 *
 * opens the fileselector to select the filename to save as
 *
 * return values: boolean
 */
gboolean screem_page_save_as()
{
	gchar *filename;
	Page *page;

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_val_if_fail( page, FALSE );

	filename = file_select( _( "Select filename to save as" ) );

	/* we didn't select one */
	if( ! filename )
		return FALSE;

	screem_page_set_pathname( page, filename );
	return screem_page_save( page );
}

/**
 * screem_page_save_proxy:
 *
 * handles the menu selection of save page
 *
 * return values: boolean
 */
void screem_page_save_proxy()
{
	Page *page;

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_if_fail( page != NULL );

	screem_editor_buffer_text();

	screem_page_save( page );
}

/**
 * screem_page_revert_proxy:
 *
 * handles the menu selection of revert page
 *
 * return values: boolean
 */
void screem_page_revert_proxy()
{
	Page *page;

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_if_fail( page != NULL );

  	if( screem_page_revert( page ) ) {
		screem_editor_clear();
		screem_editor_insert( 0, page->data );
	}
}

/**
 * screem_page_create_page:
 *
 * creates a new page
 *
 * return values: none
 */
void screem_page_create_page()
{
	Page *page;

	page = screem_page_new();
	screem_page_wizard( page );
}

void screem_page_create_blank_page()
{
	Page *page;
	gchar *filename;

	filename = file_select( _( "Select filename for new page" ) );

	if( ! filename )
		return;
	
	if( current_site ) {
		screem_site_add_page( current_site, filename );
		page = screem_site_locate_page( current_site, filename );
	} else {
		page = screem_page_new();
		screem_page_set_pathname( page, filename );
		loaded_pages = g_list_append( loaded_pages, page );
	}

	page->data = g_strdup( "" );

	screem_page_save( page );

	refresh_file_tree();

	screem_page_insert( page );
}

void screem_page_close_page()
{
	Page *page;
	GList *list = NULL;

	gint num;
	GtkWidget *pagebook = gtk_object_get_data( GTK_OBJECT( app ),
						   "pagebook" );

	num = gtk_notebook_get_current_page( GTK_NOTEBOOK( pagebook ) );
	
	g_return_if_fail( num > -1 );

	if( current_site )
		list = screem_site_get_pages( current_site );
	else 
		list = loaded_pages;

	page = screem_page_from_tab_num( list, num );

	g_return_if_fail( page != NULL );

	/* buffer the text */
	screem_editor_buffer_text();

	/* request save confirmation */

	/* destroy */
	if( ! current_site ) {
		current_page = NULL;
		loaded_pages = g_list_remove( loaded_pages, page );
		screem_page_destroy( page );
	} else {
		screem_site_set_current_page( current_site, NULL );
	}

	screem_editor_clear();
	screem_editor_disable();
	screem_preview_display( "" );

	/* remove the notebook tab */
	gtk_notebook_remove_page( GTK_NOTEBOOK( pagebook ), num );
	/* update other tab numbers */
	screem_page_update_tab_nums( list, num );

	if( current_site )
		screem_site_dec_open_pages( current_site );
	else 
		open_pages --;
}

void preview_current_page()
{
	Page *page;
	gchar *text;

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_if_fail( page != NULL );

	screem_editor_buffer_text();
      	screem_preview_update( page->data );
}


void build_recent_pages_menu()
{
	GnomeUIInfo *menu;
	gchar *path = N_( "_File/_Recent Pages/" );
        int i;
     
	GList *list = cfg->recent_pages;
	GList *prev = NULL;

	if( ( i = g_list_length( list ) ) ) {
                gnome_app_remove_menu_range( GNOME_APP( app ), path,  0, i );
        } else {
		return;
	}

	list = g_list_last( list );
	for( i = 3; list && i > 0; i --, list = list->prev )
		prev = list;
 
	if( ! list )
		list = prev;

	cfg->recent_pages = list;
	list->prev = NULL;

	for( list = cfg->recent_pages; list; list = list->next ) {
		menu = g_malloc0 (2 * sizeof (GnomeUIInfo));
		menu->label = escape_path( list->data );
		
		menu->type = GNOME_APP_UI_ITEM;
		menu->hint = NULL;
		
		menu->moreinfo = recent_proxy;
		
		menu->user_data = g_strdup( list->data );
		menu->unused_data = NULL;
		menu->pixmap_type = 0;
		menu->pixmap_info = NULL;
		menu->accelerator_key = 0;
		
		(menu + 1)->type = GNOME_APP_UI_ENDOFINFO;
		
		gnome_app_insert_menus( GNOME_APP(app), path, menu );
	}
	
	save_preferences();
}

static void recent_proxy( GtkWidget *widget, gchar *filename )
{
	screem_site_close_all();

	screem_page_open_with_filename( filename );
}

void page_book_switch( GtkNotebook *notebook, GtkNotebookPage *p,
		       guint page_num )
{
	GList *list = NULL;
	Page *page;
	const gchar *page_path;
	gchar *pname;
   
	gint hack;

	GtkWidget *edbook = gtk_object_get_data( GTK_OBJECT( app ),
						 "notebook" );
	
	hack = (gint)gtk_object_get_data( GTK_OBJECT( notebook ), "hack" );
	if( hack )
		return;

	if( current_site )
		list = screem_site_get_pages( current_site );
	else
		list = loaded_pages;

	page = screem_page_from_tab_num( list, page_num );

	if( page ) {
		if( current_site )
			screem_site_set_current_page( current_site, page );
		else
			current_page = page;

		page_path = screem_page_get_pathname( page );
		pname = g_dirname( page_path );
		chdir( pname );
		
		/* display in the editor */
		screem_editor_clear();
		screem_editor_insert( 0, page->data );
		screem_editor_enable();
		
		gtk_notebook_set_page( GTK_NOTEBOOK( edbook ), 0 );
	
		g_free( pname );
	}
	
}
