/*  Screem:  tagtree.c,
 *  Handles creation of / interaction with the tag tree
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <dirent.h>
#include <gnome.h>
#include <gnome-xml/debugXML.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>

#include "editor.h"
#include "tag_tree.h"
#include "xml.h"

static void add_node( GtkCTree *t, xmlDocPtr doc, xmlNodePtr node, 
		      GtkCTreeNode *parent );

static void tag_tree_clicked( GtkWidget *widget, GdkEventButton *event,
			      gpointer data );

/*
 * screem_tag_tree_build:
 * @tree: the tree
 *
 * build a tag tree from the given xml tree
 *
 * return values: none
 */
GtkWidget* screem_tag_tree_build( xmlDocPtr tree )
{
	xmlNodePtr node;
	GtkWidget *t;

	t = gtk_ctree_new( 1, 0 );

	g_return_val_if_fail( tree != NULL, t );

	node = tree->root;

	/* not interested in the first element */
	node = node->childs;

	add_node( GTK_CTREE( t ), tree, node, NULL );

	add_bluefish_fn( GTK_CTREE( t ) );

	gtk_ctree_set_line_style( GTK_CTREE( t ), GTK_CTREE_LINES_DOTTED );
        gtk_ctree_set_expander_style(GTK_CTREE( t ),GTK_CTREE_EXPANDER_SQUARE);
        gtk_clist_set_column_auto_resize( GTK_CLIST( t ), 0, TRUE );
        gtk_clist_set_selection_mode( GTK_CLIST( t), GTK_SELECTION_BROWSE );

	gtk_signal_connect( GTK_OBJECT( t ), "button_press_event",
			    GTK_SIGNAL_FUNC( tag_tree_clicked ), 0 );

	GTK_WIDGET_UNSET_FLAGS( t, GTK_CAN_FOCUS );

	return t;
}

/*
 * screem_tag_tree_load:
 *
 * load the global and user tag trees
 *
 * return values: an xml tree
 */
xmlDocPtr screem_tag_tree_load()
{
	xmlDocPtr doc;
	xmlDocPtr user_doc;
	xmlNodePtr node;

	const gchar *home;
	gchar *path;
	struct dirent **namelist;
	gint num;
	gint pos;
	gchar *file;
	
	/* load the global file */
	doc = xmlParseFile( TAG_TREE_FILE );

	/* load user tag trees */
	home = g_get_home_dir();
	path = g_strconcat( home, G_DIR_SEPARATOR_S, ".screem", NULL );
	num = scandir( path, &namelist, 0, alphasort );
	g_free( path );

	for( pos = 0; pos < num; pos ++ ) {
		file = namelist[ pos ]->d_name;
		if( ! strcmp( "application/x-screem-tag-tree",
			      gnome_mime_type( file ) ) )	{
			path = g_strconcat( home, G_DIR_SEPARATOR_S, 
					    ".screem", G_DIR_SEPARATOR_S,
					    file, NULL );
			user_doc = xmlParseFile( path );
			g_free( path );
			/* copy the root node */
			node = user_doc->root;
			node = xmlCopyNode( node->childs, TRUE );
			/* add the node to the main tree */
			xmlAddChild( doc->root, node );
			/* delete user_doc */
			xmlFreeDoc( user_doc );
		}
	}

	xmlSaveFile( "/tmp/tree.xml", doc );

	return doc;
}

static void add_node( GtkCTree *t, xmlDocPtr doc, xmlNodePtr node,
		      GtkCTreeNode *parent )
{
	gchar *text[ 2 ] = { NULL, NULL };
	GtkCTreeNode *this;
	gchar *name;
	gchar *open;
	gchar *close;
    	Tag_data *td;

	name = xml_get_value( node, "name" );
	if( ! name )
		name = "Invalid tree";

	/* add the node */
	text[ 0 ] = name;
	this = gtk_ctree_insert_node( t, parent, NULL, text, 3, 
				      NULL, NULL, NULL, NULL, 
				      (gboolean)parent, FALSE );
	/* has children? */
	if( node->childs ) {
		add_node( t, doc, node->childs, this );
	}

	/* move on to the next sibling */
	if( node->next )
		add_node( t, doc, node->next, parent );
	

	/* set up the row data for the node we added if needed */
	open = xml_get_value( node, "open" );
	close = xml_get_value( node, "close" );
	if( open || close ) {
		td = g_new( Tag_data, 1 );
		td->open = open;
		td->close = close;
		gtk_ctree_node_set_row_data( GTK_CTREE( t ), this, td );
	}

	return;
}

static void tag_tree_clicked( GtkWidget *widget, GdkEventButton *event,
			      gpointer data )
{
	GtkCTreeNode *node;
        gint row = -1;
        gint col;
	Tag_data *td;

  	gtk_clist_get_selection_info( GTK_CLIST( widget ),
                                      event->x, event->y, &row, &col );

	if( row == -1 )
		return;

	node = gtk_ctree_node_nth( GTK_CTREE( widget ), ( guint )row );
       
	if( ! node )
		return;

	if( event->button == 1 ) {
		td = gtk_ctree_node_get_row_data( GTK_CTREE( widget ), node );
		if( ! td )
			return;
		screem_editor_insert_markup( td->open, td->close );
	}
}
