package biss.awt.kernel;

import biss.awt.Awt;
import biss.awt.BarMenu;
import java.awt.AWTError;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Insets;
import java.awt.MenuBar;
import java.awt.Toolkit;
import java.awt.image.ColorModel;
import java.awt.image.ImageObserver;
import java.awt.image.ImageProducer;
import java.awt.peer.CanvasPeer;
import java.awt.peer.ComponentPeer;
import java.awt.peer.FramePeer;
import java.awt.peer.PanelPeer;
import java.awt.peer.WindowPeer;

/**
 * This is the only real (invisible) peer class within the biss.awt.kernel.
 * It implements all peer interfaces required to instanciate the generic
 * Components of java.awt (Frame, Window, Panel, Canvas). And since all the
 * real work (except for some menubar handling) is done in the native lib
 * (via NativeLib methods), it seems reasonable to concentrate it all in
 * one class.
 *
 * The main purpose of class Peer is to link the Target (Component) to the
 * native window object and to keep track of the  parent-child relation
 * of native window objects.
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class Peer
  implements PanelPeer, CanvasPeer, WindowPeer, FramePeer
{
	NativeLib Lib;
	Peer Parent;
	Insets Inset;
	int Type = 0;
	final static int WINDOW = 1;
	final static int FRAME = 2;
	final static int MODAL = 8;
	final static int CANVAS = 0;
	final static int PANEL = 4;
	Component Target;
	boolean IsExposed = false;
	boolean IsDisabled = false;
	BarMenu BM;
	static int TBHeight;
	final static int POPUP = 16;
	long PData;
	static Insets FrameInset;
	static Insets ZeroInset;
	Color Background;

Peer () {
	// dummy peer, just used to recycle PeerTable entries
	// don't use biss.awt.Awt here !!
}

Peer ( NativeLib lib, Component target, int type ) {
	Component c;
	ComponentPeer p;

	Lib = lib;
	Target = target;
	Type = type;

	if ( FrameInset == null ) // must be defered because it needs static Awt init
		initStandardInsets();

	if ( (c = target.getParent()) != null ) {
		for ( p=c.getPeer(); p instanceof Component; p = ((Component)p).getPeer());

		if ( p instanceof Peer ){
			Parent = (Peer) p;
		}
		else {
			throw new AWTError( "no NativeWindow peer of: " + c);
		}
	}
	else {
		//--- these are the toplevel defaults java.awt depends on (via getParent().getXX())
		if ( Target.getBackground() == null )
			Target.setBackground( Awt.WindowBackClr);
		if ( Target.getForeground() == null )
			Target.setForeground( Color.black);
		if ( Target.getFont() == null )
			Target.setFont( new Font( "Dialog", Font.PLAIN, 12));
	}

	if ( Type == FRAME && Kernel.ShrinkFrame )
		Inset = (Insets) FrameInset.clone();
	else
		Inset = (Insets) ZeroInset.clone();

	Background = Target.getBackground();

	//	Dbg.log( "NativeWindow: " + this);
}

public int checkImage ( Image img, int w, int h, ImageObserver o ) {
	return 0;
}

public Image createImage ( ImageProducer producer ) {
	return new biss.awt.kernel.Image( Lib, producer);
}

public Image createImage ( int width, int height ) {
	return new biss.awt.kernel.Image( Lib, width, height);
}

public void disable () {
	IsDisabled = true;
}

public void dispose () {
	Lib.destroyWindow( this);
}

public void enable () {
	IsDisabled = false;
}

public ColorModel getColorModel () {
	return null;
}

public FontMetrics getFontMetrics (Font font) {
	return new biss.awt.kernel.FontMetrics( Lib, font);
}

public java.awt.Graphics getGraphics () {
	return (new biss.awt.kernel.Graphics( Lib, this));
}

public Toolkit getToolkit () {
	return Lib.Tk;
}

public boolean handleEvent ( Event evt ) {
	Component c;

	//	Dbg.log( evt.toString());

	if ( evt.id == 0 )
		return true;
	else if ( evt.id == Event.WINDOW_MOVED ) {
		if ( Parent == null && !Target.isValid() ) {
			Target.validate();
			if ( BM != null)
				BM.reshape( 0, 0, Target.size().width, TBHeight);
			// Lib.accelerateReshapes();
		}
	}

	for ( c = Target; c != null; c = c.getParent() ) {
		if ( c.handleEvent( evt) )
			return true;
	}

	return false;
}

public void hide () {
	Lib.hideWindow( this);
}

static void initStandardInsets () {
	int bb = 2 * Kernel.BorderInset;

	TBHeight = BarMenu.getDefaultHeight();
	if ( Kernel.TitleInset == 0 )
		Kernel.TitleInset = TBHeight;

	ZeroInset = new Insets( 0, 0, 0, 0);
	FrameInset = new Insets( 0, 0, bb + Kernel.TitleInset, bb);
}

public Insets insets () {
	return Inset;
}

public Dimension minimumSize () {
	return Target.size();
}

public void nextFocus () {
}

public void paint ( java.awt.Graphics g ) {
	return;
}

public Dimension preferredSize () {
	return Target.size();
}

public boolean prepareImage (Image img, int w, int h, ImageObserver o) {
	return false;
}

public void print ( java.awt.Graphics g ) {
}

public void repaint (long tm, int x, int y, int width, int height) {
	if ( tm == 0 ) {
		Graphics g = getGraphics();
		if ( x != 0 || y != 0 )
			g.clipRect( x, y, width, height);
		Target.paint( g);
		g.dispose();
	}
	else {
		Lib.repaintWindow( this, tm, x, y, width, height);
	}
}

public void requestFocus () {
	Lib.setFocusWindow( this);
}

public void reshape (int x, int y, int width, int height) {

	Lib.reshapeWindow( this, x, y, width, height);

	if ( BM != null)
		BM.reshape( 0, 0, width, TBHeight );
}

public void setBackground ( Color c ) {
	Background = c;
	Lib.setWindowBackground( this, c);
}

public void setCursor ( int cursorType ) {
	Lib.setWindowCursor( this, cursorType);
}

public void setFont (Font f) {
}

public void setForeground (Color c) {
}

public void setIconImage ( Image image ) {
}

public void setMenuBar (MenuBar menu) {
	if ( menu == null )
		return;

	BM = new BarMenu( menu);
	Inset.top = TBHeight;
}

public void setResizable (boolean resizeable) {
}

public void setTitle ( String title ) {
	Lib.setWindowTitle( this, title);
}

public void show () {
	Lib.showWindow( this);

	if ( BM != null)
		BM.reshape( 0, 0, Target.size().width, TBHeight);
}

public void toBack () {
	Lib.lowerWindow( this);
}

public void toFront () {
	Lib.raiseWindow( this);
}

public String toString() {
	String s = "Peer [" + Long.toString( PData, 16); 

	if ( Target != null )
		s += ", target: " + Target.getClass().getName();

	if ( Parent != null )
		s += ", parent: " + Long.toString( Parent.PData, 16);

	return s + ']';
}
}
