#include "xmame.h"
#include <stdarg.h> /* prolly should go in xmame.h */
#include "driver.h"
#include "audit.h"
#include "common.h"
#include "info.h"

/* Mame frontend interface & commandline */
/* parsing rountines by Maurizio Zanello */

static const struct GameDriver *gamedrv;

/* MESS doesn't have a neogeo_bios but to work around a lott
   off #defines later on, we create one here ;) */
#if defined(MESS) || defined(NEOFREE)
static struct GameDriver neogeo_bios;
#else
extern struct GameDriver neogeo_bios;
#endif

/* compare string[8] using standard(?) wildchars ('?' & '*')          */
/* for this to work correctly, the shells internal wildcard expansion */
/* mechanism has to be disabled, use quotes */
int strwildcmp(const char *sp1, const char *sp2)
{
   char s1[9], s2[9];
   int i, l1, l2;
   char *p;

   strncpy(s1, sp1, 8); s1[8] = 0; if (s1[0] == 0) strcpy(s1, "*");

   strncpy(s2, sp2, 8); s2[8] = 0; if (s2[0] == 0) strcpy(s2, "*");

   p = strchr(s1, '*');
   if (p)
   {
      for (i = p - s1; i < 8; i++) s1[i] = '?';
      s1[8] = 0;
   }

   p = strchr(s2, '*');
   if (p)
   {
      for (i = p - s2; i < 8; i++) s2[i] = '?';
      s2[8] = 0;
   }

   l1 = strlen(s1);
   if (l1 < 8)
   {
      for (i = l1 + 1; i < 8; i++) s1[i] = ' ';
      s1[8] = 0;
   }

   l2 = strlen(s2);
   if (l2 < 8)
   {
      for (i = l2 + 1; i < 8; i++) s2[i] = ' ';
      s2[8] = 0;
   }

   for (i = 0; i < 8; i++)
   {
      if (s1[i] == '?' && s2[i] != '?') s1[i] = s2[i];
      if (s2[i] == '?' && s1[i] != '?') s2[i] = s1[i];
   }

   return stricmp(s1, s2);
}

int myprintf(char *fmt, ...) {
  int i;
  va_list args;
  va_start(args, fmt);
  i = vfprintf(stdout_file, fmt, args);
  va_end(args);
  return i;
}

void print_description(int driver, int showclones)
{
   char name[200];
   
   strcpy(name,drivers[driver]->description);
   
   /* Move leading "The" to the end */
   if (strstr(name," (")) *strstr(name," (") = 0;
      if (strncmp(name,"The ",4) == 0)
      {
         fprintf(stdout_file, "%s",name+4);
         fprintf(stdout_file, ", The");
      }
      else
         fprintf(stdout_file, "%s",name);
   
   /* print the additional description only if we are listing clones */
   if (showclones)
   {
      if (strchr(drivers[driver]->description,'('))
         fprintf(stdout_file, " %s",strchr(drivers[driver]->description,'('));
   }
   fprintf(stdout_file, "\n");
}

int frontend_list(int list, char *gamename, int showclones)
{
   int i, j;
   const char *header[] = {
       NAME" currently supports:\n",
       "Name:     Description:\n",
       "Year: Manufacturer:                        Description:\n",
       "", /* listclones */
       " romname driver     cpu 1    cpu 2    cpu 3    cpu 4    sound 1     sound 2     sound 3     sound 4     name\n"
       "-------- ---------- -------- -------- -------- -------- ----------- ----------- ----------- ----------- --------------------------\n",
       "", /* listroms */
       "", /* listsamples */
       "Name:     Samples dir:\n",
       "CRC:     Filename:    Description:\n",
       "", /* crcdup */
       "", /* listinfo */
       "Name:     Colors:\n",
       "", /* lmr */
       "", /* wrongorientation */
       "", /* wrongmerge */
       "", /* verifyroms */
       ""  /* verifysamples */
   };
       
   int matching     = 0;
   int correct      = 0;
   int incorrect    = 0;
   int skipped      = 0;
   int not_found    = 0;
      
   fprintf(stdout_file, header[list-1]);
      
   if (list == LIST_INFO) {
      print_mame_info( stdout_file, drivers ); 
      return OSD_OK;
   }
   
   for (i=0;drivers[i];i++)
   {
         if ((showclones || drivers[i]->clone_of == 0 || drivers[i]->clone_of == &neogeo_bios) &&
            !strwildcmp(gamename, drivers[i]->name))
         {
            matching++;

            
            switch(list)
            {
               case LIST_LIST: /* simple games list */
                  fprintf(stdout_file, "%-8s",drivers[i]->name);
                  if (!(matching % 8)) fprintf(stdout_file, "\n");
                  break;
               case LIST_FULL: /* games list with descriptions */
                  fprintf(stdout_file, "%-10s",drivers[i]->name);
                  print_description(i, showclones);
                  break;
               case LIST_GAMES:
               {
                  fprintf(stdout_file, "%-6s%-36s ",drivers[i]->year,drivers[i]->manufacturer);
                  print_description(i, showclones);
                  break;
               }
               case LIST_DETAILS: /* A detailed MAMELIST.TXT type roms lister */
                  /* First, the rom name */
                  fprintf(stdout_file, "%-8s  ",drivers[i]->name);

                  /* source file (skip the leading "src/drivers/" */
                  fprintf(stdout_file, "%-10s  ",&drivers[i]->source_file[12]);

                  /* Then, cpus */
                  for(j=0;j<MAX_CPU;j++)
                  {
                     const struct MachineCPU *x_cpu = drivers[i]->drv->cpu;
                     if (x_cpu[j].cpu_type & CPU_AUDIO_CPU)
                        fprintf(stdout_file, "[%-6s] ",cputype_name(x_cpu[j].cpu_type));
                      else
                        fprintf(stdout_file, "%-8s ",cputype_name(x_cpu[j].cpu_type));
                  }
               
                  for(j=0;j<MAX_CPU;j++)
                  {
                     const struct MachineSound *x_sound = drivers[i]->drv->sound;
                     if (sound_num(&x_sound[j]))
                     {
                        fprintf(stdout_file, "%dx",sound_num(&x_sound[j]));
                        fprintf(stdout_file, "%-9s ",sound_name(&x_sound[j]));
                     }
                     else
                        fprintf(stdout_file, "%-11s ",sound_name(&x_sound[j]));
                  }
                  
                  /* Lastly, the name of the game and a \newline */
                  fprintf(stdout_file, "%s\n",drivers[i]->description);
                  break;
               case LIST_ROMS: /* game roms list */
                  printromlist(drivers[i]->rom,drivers[i]->name);
                  fprintf(stdout_file, "\n");
                  break;
               case LIST_SAMPLES: /* game samples list */
                  gamedrv = drivers[i];
                  if (gamedrv->samplenames != 0 && gamedrv->samplenames[0] != 0)
                  {
                     j=0;
                     fprintf(stdout_file, "Samples need by %s:\n", gamedrv->name);
                     if(gamedrv->samplenames[0][0]=='*')
                     {
                        fprintf(stdout_file, "Uses samples from: %s\n", gamedrv->samplenames[0]+1);
                        j++;
                     } 
                     while(gamedrv->samplenames[j])
                     {
                           fprintf(stdout_file, "%s\n",gamedrv->samplenames[j]);
                           j++;
                     }
                  } else skipped++;
                  break;
               case LIST_SAMDIR: /* games list with samples directories */
                  if (drivers[i]->samplenames != 0 && drivers[i]->samplenames[0] != 0)
                  {
                        fprintf(stdout_file, "%-10s",drivers[i]->name);
                        if (drivers[i]->samplenames[0][0] == '*')
                              fprintf(stdout_file, "%s\n",drivers[i]->samplenames[0]+1);
                        else
                              fprintf(stdout_file, "%s\n",drivers[i]->name);
                  } else skipped++;
                  break;
               case LIST_CRC: /* list all crc-32 */
                  {
                     const struct RomModule *romp = drivers[i]->rom;

                     while (romp->name || romp->offset || romp->length)
                     {
                        if (romp->name && romp->name != (char *)-1)
                           fprintf(stdout_file, "%08x %-12s %s\n",romp->crc,romp->name,drivers[i]->description);
                        romp++;
                     }
                  }
                  break;
               case LIST_DUPCRC:
                  {
                     const struct RomModule *romp = drivers[i]->rom;

                     while (romp->name || romp->offset || romp->length)
                     {
                        if (romp->name && romp->name != (char *)-1 && romp->crc)
                        {
                           j = i+1;
                           while (drivers[j])
                           {
                              const struct RomModule *romp1 = drivers[j]->rom;

                              while (romp1->name || romp1->offset || romp1->length)
                              {
                                 if (romp1->name && romp1->name != (char *)-1 &&
                                    strcmp(romp->name,romp1->name) &&
                                    romp1->crc == romp->crc)
                                 {
                                    fprintf(stdout_file, "%08x %-12s %-8s <-> %-12s %-8s\n",romp->crc,
                                       romp->name,drivers[i]->name, romp1->name,drivers[j]->name);
                                 }
                                 romp1++;
                              }
                              j++;
                           }
                        }
                        romp++;
                     }                  
                  }
                  break;
               case LIST_COLORS:
                  fprintf(stdout_file, "%-8s  %d\n", drivers[i]->name,
                     drivers[i]->drv->total_colors);
                  break;
               case LIST_LMR: /* missing romsets list */
                  if (RomsetMissing (i))
                  {
                     if (!not_found)
                     {
                        printf ("game      clone of  description\n");
                        printf ("--------  --------  -----------\n");
                     }
                     printf ("%-10s%-10s%s\n", drivers[i]->name,
                        (drivers[i]->clone_of) ? drivers[i]->clone_of->name : "",
                        drivers[i]->description);
                     not_found++;
                  }
                  break;
               case LIST_WRONGORIENTATION: /* list drivers which incorrectly use the orientation and visible area fields */
                  if ((drivers[i]->drv->video_attributes & VIDEO_TYPE_VECTOR) == 0 &&
                              drivers[i]->drv->visible_area.max_x - drivers[i]->drv->visible_area.min_x + 1 <=
                              drivers[i]->drv->visible_area.max_y - drivers[i]->drv->visible_area.min_y + 1)
                  {
                      fprintf(stdout_file, "%s %dx%d\n",drivers[i]->name,
                         drivers[i]->drv->visible_area.max_x - drivers[i]->drv->visible_area.min_x + 1,
                         drivers[i]->drv->visible_area.max_y - drivers[i]->drv->visible_area.min_y + 1);
                      incorrect++;
                  } else correct++;
                  break;
               case LIST_WRONGMERGE: /* list duplicate crc-32 with different ROM name in clone sets */
                  {
                     const struct RomModule *romp = drivers[i]->rom;
                     
                     while (romp->name || romp->offset || romp->length)
                     {
                        if (romp->name && romp->name != (char *)-1 && romp->crc)
                        {
                           j = 0;
                           while (drivers[j])
                           {
                              if(j != i &&
                                 drivers[j]->clone_of != &neogeo_bios &&
                                 drivers[j]->clone_of &&
                                 (drivers[j]->clone_of == drivers[i] ||
                                    (i < j &&
                                       drivers[j]->clone_of == drivers[i]->clone_of)))
                              {
                                 int match = 0;
                                 const struct RomModule *romp1 = drivers[j]->rom;
                                 
                                 while (romp1->name || romp1->offset || romp1->length)
                                 {
                                    if(romp1->name && romp1->name != (char *)-1 &&
                                       !strcmp(romp->name,romp1->name))
                                    {
                                       match = 1;
                                       break;
                                    }
                                    romp1++;
                                 }
                                 if (match == 0)
                                 {
                                    romp1 = drivers[j]->rom;
                                    
                                    while (romp1->name || romp1->offset || romp1->length)
                                    {
                                       if(romp1->name && romp1->name != (char *)-1 &&
                                          strcmp(romp->name,romp1->name) &&
                                          romp1->crc == romp->crc)
                                       {
                                          printf("%08x %-12s %-8s <-> %-12s %-8s\n",
                                             romp->crc,	romp->name,drivers[i]->name,
                                             romp1->name,drivers[j]->name);
                                       }
                                       romp1++;
                                    }
                                 }
                              }
                              j++;
                           }
                        }
                        romp++;
                     }
                  }
                  break;
               case LIST_WRONGFPS: /* list drivers with too high frame rate */
                  if ((drivers[i]->drv->video_attributes & VIDEO_TYPE_VECTOR) == 0 &&
                     drivers[i]->clone_of == 0 &&
                     drivers[i]->drv->frames_per_second > 57 &&
                     drivers[i]->drv->visible_area.max_y - drivers[i]->drv->visible_area.min_y + 1 > 244 &&
                     drivers[i]->drv->visible_area.max_y - drivers[i]->drv->visible_area.min_y + 1 <= 256)
                  {
                     printf("%s %dx%d %dHz\n",drivers[i]->name,
                        drivers[i]->drv->visible_area.max_x - drivers[i]->drv->visible_area.min_x + 1,
                        drivers[i]->drv->visible_area.max_y - drivers[i]->drv->visible_area.min_y + 1,
                        drivers[i]->drv->frames_per_second);
                  }
                  break;
               case VERIFY_ROMS: /* verify options */
               case VERIFY_SAMPLES:
                  if(list==VERIFY_ROMS)
                  {
                      /* some drivers don't use roms,
                        this happens in MESS */
                     if (drivers[i]->rom == NULL)
                     {
                        skipped++;
                        continue;
                     }
                     j = VerifyRomSet (i, (verify_printf_proc)myprintf);
                     fprintf(stdout_file, "romset %s ", drivers[i]->name);
                  }
                  else
                  {
                     /* ignore games that need no samples */
                        if (drivers[i]->samplenames == 0 ||
                        drivers[i]->samplenames[0] == 0)
                     {
                        skipped++;
                        continue;
                     }
                     j = VerifySampleSet (i,(verify_printf_proc)myprintf);
                     fprintf(stdout_file, "sampleset %s ", drivers[i]->name);
                  }
                  
                  switch (j)
                  {
                     case CORRECT:
                        fprintf(stdout_file, "correct\n");
                        correct++;
                        break;
                     case NOTFOUND:
                     case CLONE_NOTFOUND:
                        fprintf(stdout_file, "not found\n");
                        not_found++;
                        break;
                     case INCORRECT:
                        fprintf(stdout_file, "incorrect\n");
                        incorrect++;
                        break;
                  }
                  break;
            }
         }
   }
   if (matching == 0)
   {
      fprintf(stderr_file, "Error: \"%s\" is not supported!\n", gamename);
      return 1;
   }
      
   fprintf(stdout_file, "\n\n");
   fprintf(stdout_file, "Total Supported: %d", i);
   if (matching  != i)
   {
      fprintf(stdout_file, ", Matching \"%s\": %d\n", gamename, matching);
   }
   else
   {
      fprintf(stdout_file, "\n");
   }
   if (skipped) fprintf(stdout_file, "Displayed: %d, Skipped: %d, because they don't use any roms/samples\n", matching-skipped, skipped);
   if (correct+incorrect) fprintf(stdout_file, "Found: %d, of which %d correct and %d incorrect\n", correct+incorrect, correct, incorrect);
   if (not_found) fprintf(stdout_file, "Not found: %d\n", not_found);
      
   if (incorrect > 0)
      return 2;
   else
      return 0;
}

int frontend_list_clones(char *gamename)
{
   /* listclones is a special case since the strwildcmp */
   /* also has to be done on clone_of. */
   int i;
   
   fprintf(stdout_file, "Name:    Clone of:\n");
   for (i=0;drivers[i];i++)
   {
      if (drivers[i]->clone_of &&
            (!strwildcmp(gamename,drivers[i]->name) || !strwildcmp(gamename,drivers[i]->clone_of->name)))
         fprintf(stdout_file, "%-8s %-8s\n",drivers[i]->name,drivers[i]->clone_of->name);
   }
   return 0;
}
