/*         _____ ___  _   _ _____ 
**    __ _|  ___/ _ \| \ | |_   _|
**   / _` | |_ | | | |  \| | | |  
**  | (_| |  _|| |_| | |\  | | |  
**   \__, |_|   \___/|_| \_| |_|  
**   |___/                        
**  
**  gFONT -- Create GIF image rendered with TeX-available Font
**
**  The gFONT program creates a GIF image for a given ASCII string
**  by the use of an arbitrary TeX-available font (Postscript or
**  METAFONT). The used font is converted from TeX's PK format to
**  gFONT's own GdF format (Gd Font) and rendered into the
**  resulting GIF image by the use of its own enhanced Gd library.
**  The result is intended to be included into HTML pages with an
**  IMG tag.
**
**  ======================================================================
**
**  Copyright (c) 1997 Ralf S. Engelschall, All rights reserved.
**
**  This program is free software; it may be redistributed and/or modified
**  only under the terms of either the Artistic License or the GNU General
**  Public License, which may be found in the ePerl source distribution.
**  Look at the files ARTISTIC and COPYING or run ``eperl -l'' to receive
**  a built-in copy of both license files.
**
**  This program is distributed in the hope that it will be useful, but
**  WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See either the
**  Artistic License or the GNU General Public License for more details.
**
**  ======================================================================
**
**  gfont_pxtogdf.c -- PX (pixel) to GdF (Gd Font) converter
*/

/*
**  pxtogdf -- Convert PX fontfile into GDF fontfile format
**  Copyright (c) Ralf S. Engelschall, All Rights Reserved.
**
**  Usage: pxtogdf pxfile.px gdffile.gdf gdifile.gdi
**
**      METAFONT  
**  .mf ----------+ 
** (vector)        \            gftopk         pktopx         pxtogdf            
**                  +--> .gf -----------> .pk ---------> .px ----------> .gdf    
**      Ghostscript/  (rendered)      (packed)        (as pixels)     (for gd)
**  .pfb ---------+
** (vector)     
*/

#include "config_ac.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <time.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#else
#include <string.h>
#endif

/* OK and FAIL exits should ALLWAYS exists */
#undef  EX_OK
#define EX_OK   0
#undef  EX_FAIL
#define EX_FAIL 1

/*
 *  CU() -- CleanUp Makro
 *  implemented in a safety way
 */
#define DECL_EXRC int rc
#define ZERO 0
#define STMT(stuff) do { stuff } while (ZERO)
#define CU(returncode) \
            STMT( \
            rc = returncode; goto CUS; )
#define CU_E1(returncode, errstr) \
            STMT( \
            fprintf(stderr, "ERROR: "); \
            fprintf(stderr, errstr); \
            fprintf(stderr, "\n"); \
            rc = returncode; goto CUS; )
#define CU_E2(returncode, errstr1, errstr2) \
            STMT( \
            fprintf(stderr, "ERROR: "); \
            fprintf(stderr, errstr1, errstr2); \
            fprintf(stderr, "\n"); \
            rc = returncode; goto CUS; )
#define CU_E3(returncode, errstr1, errstr2, errstr3) \
            STMT( \
            fprintf(stderr, "ERROR: "); \
            fprintf(stderr, errstr1, errstr2, errstr3); \
            fprintf(stderr, "\n"); \
            rc = returncode; goto CUS; )
#define START_OF_CUS CUS:
#define RETURN_EXRC return (rc)


/*
 *  Yeah, here it comes...all ASCII control codes.
 *  Defined as their correct acronyms.
 */

#define ASC_NUL '\x00'
#define ASC_SOH '\x01'
#define ASC_STX '\x02'
#define ASC_ETX '\x03'
#define ASC_EOT '\x04'
#define ASC_ENQ '\x05'
#define ASC_ACK '\x06'
#define ASC_BEL '\x07'
#define ASC_BS  '\x08'
#define ASC_HT  '\x09'
#define ASC_LF  '\x0a'
#define ASC_VT  '\x0b'
#define ASC_FF  '\x0c'
#define ASC_CR  '\x0d'
#define ASC_SO  '\x0e'
#define ASC_SI  '\x0f'
#define ASC_DLE '\x10'
#define ASC_DC1 '\x11'
#define ASC_DC2 '\x12'
#define ASC_DC3 '\x13'
#define ASC_DC4 '\x14'
#define ASC_NAK '\x15'
#define ASC_SYN '\x16'
#define ASC_ETB '\x17'
#define ASC_CAN '\x18'
#define ASC_EM  '\x19'
#define ASC_SUB '\x1a'
#define ASC_ESC '\x1b'
#define ASC_FS  '\x1c'
#define ASC_GS  '\x1d'
#define ASC_RS  '\x1e'
#define ASC_US  '\x1f'
#define ASC_SP  '\x20'
#define ASC_DEL '\x7f'
#ifndef NUL
#define NUL ASC_NUL
#endif

#define ASC_QUOTE '\x22'
#define ASC_NL    ASC_LF


/*   define the boolean values in a general and
 *   portable way.
 */
#define TRUE  (0 || !(0))
#define FALSE (!(TRUE))
typedef enum { false = FALSE, true = TRUE } bool;

/*   some other names for true and false */
#define YES   TRUE
#define NO    FALSE

#define GOOD  TRUE
#define WRONG FALSE

#define OK    TRUE
#define BAD   FALSE

#define SOME  TRUE
#define NONE  FALSE

/*
 *   define NULL if not already defined
 */
#ifndef NULL
#define NULL (void *)0
#endif


/*   define some handy str* functions
 */
#define   strNE(s1, s2)          (strcmp(s1, s2))
#define   strEQ(s1, s2)         (!strcmp(s1, s2))
#define  strnNE(s1, s2, l)      (strncmp(s1, s2, l))
#define  strnEQ(s1, s2, l)     (!strncmp(s1, s2, l))
#define strniNE(s1, s2, l)  (strncasecmp(s1, s2, l))
#define strniEQ(s1, s2, l) (!strncasecmp(s1, s2, l))


int fDebug = 0;

void Debug(char *str, ...)
{
    va_list ap;
    char buf[1024];

    va_start(ap, str);
    if (fDebug) {
        vsprintf(buf, str, ap);
        fprintf(stderr, "Debug: %s\n", buf);
    }
    va_end(ap);
    return;
}

char int2binstrBuf[32];

char *int2binstr(unsigned int c)
{
    int i;

    for (i = 0; i < 8*4; i++) {
        if ((c & 0x80000000) == 0)
            int2binstrBuf[i] = '0';
        else
            int2binstrBuf[i] = '1';
        c = c << 1;
    }
    int2binstrBuf[i] = 0;
    return int2binstrBuf;
}

unsigned int Get32(FILE *fp)
{
    unsigned int i;
    unsigned char c;

    i = 0;
    c = fgetc(fp);
    i = (i | c) << 8;
    c = fgetc(fp);
    i = (i | c) << 8;
    c = fgetc(fp);
    i = (i | c) << 8;
    c = fgetc(fp);
    i = (i | c);
  
    return i;
}

void Put32(FILE *fp, int i)
{
    fwrite(&i, sizeof(int), 1, fp);
    return;
}

void Put8(FILE *fp, unsigned char c)
{
    fwrite(&c, sizeof(unsigned char), 1, fp);
    return;
}

#define NUM_CHARS_IN_PK 256

int 
main(int argc, char *argv[])
{
    DECL_EXRC;
    time_t tm;

    char pxfile[512];   /* font data in PX format */
    char gdffile[512];  /* font data in GDF format */
    char gdifile[512];  /* font data in GDF format */
    char fontname[512]; /* font name */

    FILE *fppx  = NULL;
    FILE *fpgdf = NULL;
    FILE *fpgdi = NULL;

    char *gdfiobuf;
    char *gdiiobuf;

    int int32;

    int sizes_w[NUM_CHARS_IN_PK];
    int sizes_h[NUM_CHARS_IN_PK];
    int offsets_x[NUM_CHARS_IN_PK];
    int offsets_y[NUM_CHARS_IN_PK];
    int rasterpointer[NUM_CHARS_IN_PK];
    int tfmwidth_w[NUM_CHARS_IN_PK];
    int tfmwidth_h[NUM_CHARS_IN_PK];
    int sizes_wMax;
    int sizes_hMax;
    int abs_wMax;
    int abs_wMin;
    int abs_hMax;
    int abs_hMin;

    int xdim, ydim, ybase;
    int xoffs;

    int checksum;
    int magnification;
    int designsize;
    int designsize_w;
    int designsize_h;
    int dirptr;

    int i, k, l;
    char buf[1000];
    char *cp;


    if (argc != 4) {
        fprintf(stderr, "Usage: pxtogdf <pxfile>.px <gdfile>.gdf <gdifile>.gdi\n");
        exit(1);
    }
    strcpy(pxfile,   argv[1]);
    strcpy(gdffile,  argv[2]);
    strcpy(gdifile,  argv[3]);
    strcpy(fontname, argv[2]);
    cp = index(fontname, '.');
    if (cp != NULL)
        *cp = '\0';

    tm = time(NULL);

    if ((fppx = fopen(pxfile, "r")) == NULL)
        CU_E2(EX_FAIL, "cannot open input file '%s'", pxfile);
    if ((fpgdf = fopen(gdffile, "w")) == NULL)
        CU_E2(EX_FAIL, "cannot open output file '%s'", gdffile);
    if ((fpgdi = fopen(gdifile, "w")) == NULL)
        CU_E2(EX_FAIL, "cannot open output file '%s'", gdifile);

    gdfiobuf = malloc(16384);
    gdiiobuf = malloc(16384);
    setvbuf(fpgdf, gdfiobuf, _IOFBF, 16384);
    setvbuf(fpgdi, gdiiobuf, _IOFBF, 16384);

    /*
     *
     *  READ PART 1: HEADER
     *
     */

    int32 = Get32(fppx);
    if (int32 != 1001)
        CU_E2(EX_FAIL, "input file '%s' not in PX format", pxfile);
    Debug("read: START_ID=1001");


    /*
     *
     *  READ PART 3: CHARACTER INFORMATION
     *
     */

    fseek(fppx, -(sizeof(int)*(NUM_CHARS_IN_PK*4+5)), SEEK_END);

    sizes_hMax = 0;
    sizes_wMax = 0;
    abs_wMin = abs_wMax = abs_hMin = abs_hMax = 0;

    for (i = 0; i < NUM_CHARS_IN_PK; i++) {

        /*  read size information of char and decode
            it into a width (=X) and height (=Y) parts          */
        int32 = Get32(fppx);
        sizes_w[i] = int32 / 65536;
        sizes_h[i] = int32 % 65536;
        /*  ajust max sizes  */
        sizes_wMax = sizes_w[i] > sizes_wMax ? sizes_w[i] : sizes_wMax;
        sizes_hMax = sizes_h[i] > sizes_hMax ? sizes_h[i] : sizes_hMax;
      
        /*  read offset information of char and decode
            it into a X and Y parts          */
        int32 = Get32(fppx);
        offsets_x[i] = int32 / 65536;
        if (offsets_x[i] >= 128) 
            offsets_x[i] -= 256;
        offsets_y[i] = int32 % 65536;
        if (offsets_y[i] >= 128) 
            offsets_y[i] -= 256;
        /*  special "if" because there exists offsets = 2 in cmr...  */
        /*
        if (offsets_y[i] < -100) {
            offsets_y[i] = offsets_y[i] & 0x0000ffff;
        }
        */

        /*  ajust absolute max sizes  */
        abs_wMin = -offsets_x[i] < abs_wMin ? -offsets_x[i] : abs_wMin;
        abs_wMax = -offsets_x[i]+sizes_w[i] > abs_wMax ?  -offsets_x[i]+sizes_w[i] : abs_wMax;
        abs_hMin = -offsets_y[i] < abs_hMin ? -offsets_y[i] : abs_hMin;
        abs_hMax = -offsets_y[i]+sizes_h[i] > abs_hMax ?  -offsets_y[i]+sizes_h[i] : abs_hMax;

        /*  read in direct fseek() offset of character,
            calculated as sizeof(int) blocks, i.e. 4 chars usually */
        rasterpointer[i] = Get32(fppx);

        /*  read in TeX Font Metirc width and decode it  */
        int32 = Get32(fppx);
        tfmwidth_w[i] = int32 / 65536 - 1;
        tfmwidth_h[i] = int32 % 65536;

        Debug("read: INFO of char #%03d: size = %2dx%2d, offset = (x=%3d,y=%3d), rasterpointer=%6d",
            i, sizes_w[i], sizes_h[i], offsets_x[i], offsets_y[i], rasterpointer[i]);
    }
    Debug("sizes_wMax=%d, sizes_hMax=%d", sizes_wMax, sizes_hMax);
    Debug("abs_wMin = %d, abs_wMax = %d", abs_wMin, abs_wMax);
    Debug("abs_hMin = %d, abs_hMax = %d", abs_hMin, abs_hMax);

    xdim  = abs(abs_wMin)+abs_wMax;
    ybase = abs(abs_hMin);
    ydim  = abs(abs_hMin)+abs_hMax;

    Debug("Char Block = %d x %d (%d base)", xdim, ydim, ybase); 


    /*
     *
     *  READ PART 4: TOTAL INFORMATION
     *
     */

    checksum = Get32(fppx);
    Debug("checksum=%d", checksum);

    magnification = Get32(fppx);
    Debug("magnification=%d", magnification);

    designsize = Get32(fppx);
    designsize_w = designsize / 65536;
    designsize_h = designsize % 65536;
    Debug("designsize=%d (%dx%d)", designsize, designsize_w, designsize_h);

    dirptr = Get32(fppx);
    Debug("dirptr=%d", dirptr);


    /*
     *
     *  READ PART 5: FOOTER
     *
     */

    Debug("END_ID = %d", Get32(fppx));


    /*
     *
     *  WRITE HEADER
     *
     */

    Put8(fpgdf, 'G');
    Put8(fpgdf, 'D');
    Put8(fpgdf, 'F');
    Put8(fpgdf, '\n');

    Put32(fpgdf, NUM_CHARS_IN_PK); /* num of chars */
    Put32(fpgdf, 0);               /* offset of first char */
    Put32(fpgdf, xdim);            /* character width */
    Put32(fpgdf, ydim);            /* character height */

    for (i = 0; i < NUM_CHARS_IN_PK; i++) {
        Put32(fpgdf, -offsets_x[i]+sizes_w[i]);
    }

    fprintf(fpgdi, "______________________________________________________________\n");
    fprintf(fpgdi, "\n");
    fprintf(fpgdi, "  Gd Font Information (GDI)\n");
    fprintf(fpgdi, "\n");
    fprintf(fpgdi, "  Font Name: %s\n", fontname);
    fprintf(fpgdi, "  Font Dims: %d x %d (%d base)\n", xdim, ydim, ybase); 
    fprintf(fpgdi, "\n");
    fprintf(fpgdi, "  Created on %s", ctime(&tm));
    fprintf(fpgdi, "  via ``pxtogdf %s %s %s''\n", pxfile, gdffile, gdifile);
    fprintf(fpgdi, "______________________________________________________________\n");
    fprintf(fpgdi, "\n");
    fprintf(fpgdi, "\n");


    /*
     *
     *  READ PART 2: CHARACTER DATA
     *  WRITE GDI CHARACTER INFO
     *  WRITE GDF CHARACTER DATA
     *
     */


    for (i = 0; i <= 127; i++) {
        fseek(fppx, sizeof(int)*rasterpointer[i], SEEK_SET);

        fprintf(fpgdi, " char: # %-3d    design size: %2d x %2d\n", i, sizes_w[i], sizes_h[i]);
        fprintf(fpgdi, " width: %-3d          offset: x = %d, y = %d\n", -offsets_x[i]+sizes_w[i], offsets_x[i], offsets_y[i]);
        fprintf(fpgdi, "\n");
        fprintf(fpgdi, "     ");
        for (k = 1; k <= xdim; k++) {
            if (k % 10 == 0)
                fprintf(fpgdi, "%d", k / 10);
            else
                fprintf(fpgdi, " ");
        }
        fprintf(fpgdi, "\n");
        fprintf(fpgdi, "     ");
        for (k = 1; k <= xdim; k++) {
            fprintf(fpgdi, "%d", k % 10);
        }
        fprintf(fpgdi, "\n");
 

        /*  first zero bytes up to line where real data starts  */
        for (l = 0; l < ybase-offsets_y[i]; l++) {
            fprintf(fpgdi, " %3d ", l+1);
            for (k = 0; k < xdim; k++) {
                fprintf(fpgdi, ".");
                Put8(fpgdf, 0);
            }
            fprintf(fpgdi, "\n");
        }

        /*  real data lines  */
        for (l = ybase-offsets_y[i]; l < ybase+(-offsets_y[i])+sizes_h[i]; l++) {
            fprintf(fpgdi, " %3d ", l+1);
            xoffs = (-offsets_x[i] < 0 ? 0 : -offsets_x[i]);

            /* before real data in the line */
            for (k = 0; k < xoffs; k++) {
                fprintf(fpgdi, ".");
                Put8(fpgdf, 0);
            }

            /* the real data in the line */
            strcpy(buf, "");
            for (k = 0; k < (sizes_w[i] + 31) / 32; k++) {
                int32 = Get32(fppx);
                sprintf(buf+strlen(buf), "%s", int2binstr(int32));
            }
            buf[sizes_w[i]] = '\0';
            for (k = 0; buf[k] != '\0'; k++) {
                fprintf(fpgdi, "%c", buf[k] == '0' ? '.' : '*');
                Put8(fpgdf, buf[k] == '0' ? 0 : 1);
            }

            /* after real data in the line */
            for (k = xoffs+sizes_w[i]; k < xdim; k++) {
                fprintf(fpgdi, ".");
                Put8(fpgdf, 0);
            }

            fprintf(fpgdi, "\n");
        }
        
        /*  from the last real data line up to the last line */
        for (l = ybase+-offsets_y[i]+sizes_h[i]; l < ydim; l++) {
            fprintf(fpgdi, " %3d ", l+1);
            for (k = 0; k < xdim; k++) {
                fprintf(fpgdi, ".");
                Put8(fpgdf, 0);
            }
            fprintf(fpgdi, "\n");
        }
        fprintf(fpgdi, "\n\n");

    }
    CU(EX_OK);

    START_OF_CUS
    if (fppx)
        fclose(fppx);
    if (fpgdf)
        fclose(fpgdf);
    if (fpgdi)
        fclose(fpgdi);

    RETURN_EXRC;
}

/*EOF*/
