/*
 * Author:      Magnos Martinello, <magnos@land.ufrj.br> in April, 1999.
 *
 * @(#)$Header: /mm/src/tgif/v4/RCS/wb_buff.c,v 4.3 1999/12/27 15:02:44 william Exp $
 */

#define _INCLUDE_FROM_WB_BUFF_C_

#include "tgifdefs.h"

#include "wb_buff.e"

#ifdef _TGIF_WB2

#ifdef PTHREAD
#include <pthread.h>
#endif

/*
  Structure of each component from the buffer
*/
typedef struct item_s {
   void *data;
   struct item_s *prev;
   struct item_s *next;
} item_t;

/*******************************************************************
  Header:
   - Maximum number of items in the buffer
   - Size of each item
   - Number of items in the buffer
   - SORTED/UNSORTED insertion
   - Pointer to first item of the queue
   - Pointer to last item of the queue
     (Item is inserted in the start pointer and removed from the end pointer)
   - Function to compare two items of the buffer
*******************************************************************/
typedef struct {
   int nitems;
   int size; 
   int n;
   int option;
#ifdef PTHREAD
   pthread_mutex_t *mp;
#endif
   item_t *start;
   item_t *end;
   int (*compar)(const void *, const void *);
} header_t;

header_t *table[MAXTABLE];
#ifdef PTHREAD
pthread_mutex_t table_mp;
#endif

/***********************************************************************
 * This function creates a new buffer and returns its descriptor as    *
 * an integer                                                          *
 **********************************************************************/
int buff_init (int n, int size, int option, int (*compar)(const void *,
const void *))
{
  header_t *header;
  item_t *item, *auxitem;
  int aux;
  static int last_used_desc = MAXTABLE - 1;

  /* If option is invalid then return error */
  if (((option & SORTED)==SORTED) && ((option & UNSORTED)==UNSORTED))
    return(-1);

  /* If insertion is sorted and no comparison function has been assigned,
     then return error */
  if ((option == SORTED) && (compar == NULL))
    return(-1);

  /* Check for valid size and number of items */
  if ((n <= 0) || (size <= 0))
    return(-1);

  /* Looks for an empty position in the Descriptor Table */
#ifdef PTHREAD
  pthread_mutex_lock(&table_mp);
#endif
  aux = last_used_desc;
  last_used_desc = (last_used_desc + 1) % MAXTABLE;
  while ((table[last_used_desc] != NULL) && (aux != last_used_desc))
    last_used_desc = (last_used_desc + 1) % MAXTABLE;
  if (!(table[last_used_desc] == NULL)){
#ifdef PTHREAD
    pthread_mutex_unlock(&table_mp);
#endif
    return(-1);
  }
#ifdef PTHREAD
  pthread_mutex_unlock(&table_mp);
#endif

  /* Create a new header and store its pointer in the descriptor table */
  header = (header_t *) malloc (sizeof(header_t));
  if (header == NULL)
    return(-1);
  memset(header, 0, sizeof(header_t));
  table[last_used_desc] = header;

  /* Set values for attributes in the header */
  header->nitems = n;
  header->size = size;
  header->n = 0;
#ifdef PTHREAD
  header->mp = malloc(sizeof(pthread_mutex_t));
  if (header->mp == NULL)
    return(-1);
  memset(header->mp, 0, sizeof(pthread_mutex_t));
  pthread_mutex_init(header->mp, NULL);
#endif
  header->option = option;
  header->compar = compar;
  header->end = (item_t *) malloc (sizeof(item_t));
  if (header->end == NULL)
    return(-1);
  memset(header->end, 0, sizeof(item_t));

  header->end->data = malloc (size);
  if (header->end->data == NULL)
    return(-1);
  memset(header->end->data, 0, size);
  auxitem = header->end;
  for (aux = 1; aux < n; aux++) {
    item = (item_t *) malloc (sizeof(item_t));
    if (item == NULL)
      return(-1);
    memset(item, 0, sizeof(item_t));
    item->data = malloc (size);
    memset(item->data, 0, size);
    if (item->data == NULL)
      return(-1);
    item->prev = auxitem;
    auxitem->next = item;
    auxitem = item;
  }
  item->next = header->end;
  header->end->prev = item;

  header->start = header->end;

  return last_used_desc;
}

/***********************************************************************
 * This function inserts a new item in the buffer especified by its    *
 * descriptor. The insertion may be sorted or not, depending on the    *
 * option set during the buffer initialization. It returns a negative  *
 * integer if the insertion wasn't successful.                         *
 **********************************************************************/
int buff_ins (int bd, const void *buffer)
{
  header_t *header;
  item_t *item, *newitem, *auxitem;

  if (table[bd] == NULL)
    return(-1);

  header = table[bd];

#ifdef PTHREAD
  pthread_mutex_lock(header->mp);
#endif

  if (header->n == header->nitems){
#ifdef PTHREAD
    pthread_mutex_unlock(header->mp);
#endif
    return(-1);
  }

  /* Unsorted insertion or empty buffer */
  if (((header->option & UNSORTED) == UNSORTED) || (header->n == 0)){
    memcpy(header->start->data, buffer, header->size);
    header->start = header->start->next;
  }
  else /* Sorted buffer */
  if ((header->option & SORTED) == SORTED){

    /* Search position where new item will be inserted according to
       comparison function previously specified                     */
    /* Search is made from begining to end and form end to begining */
    item = header->end;
    auxitem = header->start->prev;
    while (((item != header->start) &&
(header->compar(item->data,buffer)<= 0)) &&
    ((auxitem != header->end) &&
(header->compar(auxitem->data,buffer)>0))){
      item = item->next;
      auxitem = auxitem->prev;
    }

    /* Decide wich pointer points to the correct position */
    if (header->compar(auxitem->data, buffer)<=0)
      item = auxitem->next;

    if ((header->option & DUPLICATED) != DUPLICATED)
      if (header->compar(item->prev->data, buffer)==0){
#ifdef PTHREAD
        pthread_mutex_unlock(header->mp);
#endif
	return(-1);
    }

    /* If new item is the greatest then simply copy it to the fisrt
       avaiable position in the buffer                             */
    if (item == header->start) {
      memcpy (item->data, buffer, header->size);
      header->start = header->start->next;
      header->n++;
#ifdef PTHREAD
      pthread_mutex_unlock(header->mp);
#endif
      return(0);
    }

    /* Create the new item */
    newitem = (item_t *) malloc (sizeof(item_t));
    if (newitem == NULL)
      return(-1);
    memset(newitem, 0, sizeof(item_t));
    newitem->data = malloc (header->size);
    if (newitem->data == NULL)
      return(-1);
    memset(newitem->data, 0, header->size);
    memcpy(newitem->data, buffer, header->size);

    /* Set pointers */
    item->prev->next = newitem;
    newitem->next = item;
    newitem->prev = item->prev;
    item->prev = newitem;

    if (header->compar(buffer, header->end->data) < 0)
      header->end = newitem;

    newitem = header->start;
    newitem->prev->next = newitem->next;
    newitem->next->prev = newitem->prev;
    header->start = header->start->next;

    free(newitem->data);
    free(newitem);
  }

  header->n++;
#ifdef PTHREAD
  pthread_mutex_unlock(header->mp);
#endif
  return(0);
}

/***********************************************************************
 * This function removes an item from the buffer especified by its     *
 * descriptor. The data stored in the item is copied to a pointer      *
 * especified by the user. It returns a negative integer if an error   *
 * occurs                                                              *
 **********************************************************************/
int buff_rem (int bd, void *buffer)
{
  header_t *header;

  if (table[bd] == NULL)
    return(-1);

  header = table[bd];

#ifdef PTHREAD
  pthread_mutex_lock(header->mp);
#endif 

  if (header->n == 0){
#ifdef PTHREAD
    pthread_mutex_unlock(header->mp);
#endif
    return(-1);
  }

  memcpy(buffer, header->end->data, header->size);
  header->end = header->end->next;
  header->n--;

#ifdef PTHREAD
  pthread_mutex_unlock(header->mp);
#endif

  return(0);
}

/***********************************************************************
 * This function disposes memory occupied by the buffer.               *
 **********************************************************************/
int buff_destroy (int bd)
{
  item_t *item=NULL, *next_item=NULL;
  int i;

  if (table[bd] == NULL)
    return(-1);

  item = table[bd]->start;
  for (i = 0; i < table[bd]->nitems; i++, item=next_item) {
    next_item = item->next;
    free(item->data);
    free(item);
  }

#ifdef PTHREAD
  pthread_mutex_destroy(table[bd]->mp);
  free(table[bd]->mp);
#endif
  free(table[bd]);
  table[bd] = NULL;

  return(0);
}

/***********************************************************************
 * This function makes all items avaiable for insertion                *
 **********************************************************************/
int buff_empty (int bd)
{
  header_t *header;

  if (table[bd] == NULL)
    return(-1);

  header = table[bd];
#ifdef PTHREAD
  pthread_mutex_lock(header->mp);
#endif
  header->n = 0;
  header->end = header->start;
#ifdef PTHREAD
  pthread_mutex_unlock(header->mp);
#endif

  return(0);
}

/***********************************************************************
 * This function makes the first n items avaiable for insertion. It    *
 * returns the number of items disposed.                               *
 **********************************************************************/
int buff_emptyn (int bd, int n)
{
  header_t *header;
  int nitems, i;

  if (table[bd] == NULL)
    return(-1);

  header = table[bd];
#ifdef PTHREAD
  pthread_mutex_lock(header->mp);
#endif
  nitems = header->n;
  i = 0;
  while ( (header->n > 0) && (i < n)) {
    i++;
    header->n--;
    header->end = header->end->next;
  }

#ifdef PTHREAD
  pthread_mutex_unlock(header->mp);
#endif

  return((nitems<n?nitems:n));
}

/***********************************************************************
 * This function returns the number of active items in the buffer      *
 **********************************************************************/
int buff_items (int bd)  {

  if (table[bd] == NULL)
    return -1;
  else
    return table[bd]->n;
}

/***********************************************************************
 * This function concatenates two buffers, creating a new unsorted     *
 * insertion buffer                                                    *
 **********************************************************************/
int buff_conc (int bd1, int bd2)
{
  header_t *hd1, *hd2;
  item_t *item;
  int nbd, i;
  char *buffer;

  if ( (table[bd1] == NULL) || (table[bd2] == NULL))
    return(-1);

  hd1 = table[bd1];
  hd2 = table[bd2];

  if (hd1->size != hd2->size)
    return(-1);

  nbd = buff_init (hd1->n + hd2->n, hd1->size, 0, hd1->compar);
  buffer = (char*) malloc (hd1->size);
  if (buffer == NULL)
    return(-1);
  memset(buffer, 0, hd1->size);

  item = hd1->end;
  i = 0;
  while (i < hd1->n)
  {
    i++;
    memcpy(buffer, item->data, hd1->size);
    buff_ins(nbd, buffer);
    item = item->next;
  }

  item = hd2->end;
  i = 0;
  while (i < hd2->n)
  {
    i++;
    memcpy(buffer, item->data, hd1->size);
    buff_ins(nbd, buffer);
    item = item->next;
  }

  free (buffer);

  return(nbd);
}

/***********************************************************************
 * This function sorts a buffer according to a comparison function     *
 * pointed to by compar                                                *
 **********************************************************************/
int buff_sort (int bd, int (*compar)(const void *, const void *))
{
  header_t *header;
  char *buffer;
  int i, n;
  item_t *item;

  if (table[bd] == NULL)
    return(-1);

  header = table[bd];

#ifdef PTHREAD
  pthread_mutex_lock(header->mp);
#endif

  if ((header->option & SORTED) == SORTED){
#ifdef PTHREAD
    pthread_mutex_unlock(header->mp);
#endif
    return(0);
  }

  if (header->compar == NULL){
#ifdef PTHREAD
    pthread_mutex_unlock(header->mp);
#endif
    return(-1);
  }

  n = header->n;
  buffer = (char*) malloc (n * header->size);
  if (buffer == NULL)
    return(-1);
  memset(buffer, 0, n*header->size);
  i = 0;
  item = header->end;
  while (i < n){
    memcpy(buffer + i * header->size, item->data, header->size);
    item = item->next;
    i++;
  }

  qsort (buffer, n, header->size, compar);

  i = 0;
  item = header->end;
  while (i < n){
    memcpy(item->data, buffer + i * header->size, header->size);
    item = item->next;
    i++;
  }

  free (buffer);

#ifdef PTHREAD
  pthread_mutex_unlock(header->mp);
#endif

  return(0);
}

#endif /* _TGIF_WB2 */
