/*
	File:		RTCPRemoteStatusModule.cpp

	Contains:	
					
	Created By: Chris LeCroy
	
	Copyright:	Copyright Apple Computer, Inc. 1999
				All rights reserved

	$Log: RTCPRemoteStatusModule.cpp,v $

	Created: Thu, Mar 4, 1999 @ 2:38 PM
*/

#include <stdio.h>

#include "RTCPRemoteStatusModule.h"

#include "RTPSession.h"
#include "RTPStream.h"
#include "RTPServerInterface.h"

#include "RTSPServerInterface.h"
#include "RTSPPrefs.h"
#include "RTSPModule.h"

#include "TCPSocket.h"
#include "StringFormatter.h"
#include "OS.h"


//Toggles some useful printf's on and off
#define REMOTE_STATUS_SOCKET_DEBUGGING 0


enum
{
	rs_SERVER_STARTED = 42,			// rs_head
	rs_SERVER_SHUTDOWN,				// rs_head
	rs_SERVER_STATS,				// rs_server
	rs_OPEN,						// rs_command
	rs_CLOSE,						// rs_command
	rs_DESCRIBE,					// rs_x_command	(URI data)
	rs_PLAY,						// rs_x_command (j_ints, stream ID's for this session)
	rs_PAUSE,						// rs_command
	rs_TEARDOWN,					// rs_command
	rs_REDIRECT,					// rs_x_command (URI data)
	rs_BUSY_REDIRECT,				// rs_x_command (URI data)
	rs_RND_REDIRECT,				// rs_x_command (URI data)
	rs_SEND_LESS,					// rs_x_command (short rs_REASON, short more_data)
	rs_SEND_MORE,					// rs_x_command (short rs_REASON)
	rs_RTCP,						// rs_rtcp
	rs_SESSION_ERROR,				// rs_error
	rs_SESSION_DESCRIPTIONS,		// rs_data
	rs_JOIN_STATS					// rs_command
};

#pragma mark _REMOTE_STATUS_PACKET_

//Individual packet classes. These classes make the packet generation code much cleaner
class RemoteStatusPacket
{
	public:

		RemoteStatusPacket(UInt8 inPacketType, UInt16 inPacketSize);
		void	WritePacket(StringFormatter* inFormatter);
		void	IncreaseLength(UInt16 amount);
		
	protected:

		static const UInt32 kMaxPacketSize = 512;
		static const UInt32 kPacketHeaderSize = 16;
		
		struct PacketHeader
		{
			UInt8 packetType;
			UInt8 version;
			UInt16 packetSize;
			UInt32 serverAddr;
			SInt64 milliseconds;
		};
		
		char 	fPacketBuffer[kMaxPacketSize];
		char*	fPacketDataPtr;
		UInt32 	fPacketSize;	
};

RemoteStatusPacket::RemoteStatusPacket(UInt8 inPacketType, UInt16 inPacketSize)
: fPacketDataPtr(&fPacketBuffer[0] + kPacketHeaderSize), fPacketSize(inPacketSize)
{
	::memset(fPacketBuffer, 0, inPacketSize);
	PacketHeader* theHeader = (PacketHeader*)&fPacketBuffer[0];
	
	theHeader->packetType = inPacketType;
	theHeader->version = 0;
	theHeader->packetSize = htons(inPacketSize);
	theHeader->serverAddr = htonl(RTSPServerInterface::GetDefaultIPAddr());
	theHeader->milliseconds = htonl(OS::Milliseconds());
}

void RemoteStatusPacket::WritePacket(StringFormatter* inFormatter)
{
	PacketHeader* theHeader = (PacketHeader*)&fPacketBuffer[0];
	UInt16 thePacketLength = ntohs(theHeader->packetSize);
	
	//If there isn't enough space in the buffer, don't write the packet at all.
	//We might really run into this situation if the outgoing socket is continually
	//getting flow controlled. Eventually the buffer will fill up and we'll have to
	//start dropping packets. This is really the best thing to do given the circumstances...
	//due to this protocol design, the server has to deliver a constant data rate in
	//order to keep the stats monitor updated, and that rate may simply not be possible
	//over some links. At some point, we must "thin" the stream.
	if (thePacketLength < inFormatter->GetSpaceLeft())
		inFormatter->Put(&fPacketBuffer[0], thePacketLength);
#if REMOTE_STATUS_SOCKET_DEBUGGING
	else
		printf("Dropping a packet\n");
#endif
}

void RemoteStatusPacket::IncreaseLength(UInt16 amount)
{
	PacketHeader* theHeader = (PacketHeader*)&fPacketBuffer[0];
	UInt16 oldLength = ntohs(theHeader->packetSize);
	oldLength += amount;
	theHeader->packetSize = htons(oldLength);
}


#pragma mark _STARTUP_PACKET_

class StartupPacket : public RemoteStatusPacket
{
	public:

		StartupPacket();
		
	protected:

		static const UInt32 kStartupPacketSize = 24;
};

StartupPacket::StartupPacket() : RemoteStatusPacket(rs_SERVER_STARTED, kStartupPacketSize)
{
	*((UInt64*)fPacketDataPtr) = OS::HostToNetworkSInt64(OS::ConvertToSecondsSince1900(OS::Milliseconds()));
}

#pragma mark _RTCP_PACKET

class RTCPPacket : public RemoteStatusPacket
{
	public:

		RTCPPacket(RTPStream* inStream);
		
	protected:

		static const UInt32 kRTCPPacketSize = 28;

		// note: the following flags are used to determine what data is in the
		//    rs_rtcp block. If the flag is set, then the data is there.
		//    If the flag is unset, the data isn't there. Scan the bits in
		//    order.
		enum
		{
			rs_HAS_BPS 				= 0x00000001,
			rs_HAS_LATE_MS			= 0x00000002,
			rs_HAS_BUFFER_DELAY		= 0x00000004,
			rs_HAS_BUFFER_LENGTH	= 0x00000008,
			rs_HAS_FRAME_RATE		= 0x00000010,
			rs_HAS_EXPECTED_RATE	= 0x00000020,
			rs_HAS_DRY_NUM			= 0x00000040,
			rs_HAS_LOSS_PERCENT		= 0x00000080,
			rs_HAS_PACKETS_DROPPED	= 0x00000100,
			rs_HAS_PACKETS_LOST		= 0x00000200,
			rs_HAS_PACKETS_RECEIVED	= 0x00000400,
			rs_HAS_EYES				= 0x00000800,
			rs_HAS_EYES_ACTIVE		= 0x00001000,
			rs_HAS_EYES_PAUSED		= 0x00002000,
			rs_HAS_GETTING_WORSE	= 0x00004000,		// no data - boolean
			rs_HAS_GETTING_BETTER	= 0x00008000		// no data - boolean
		};

		struct RTCPInfo
		{
			UInt32	clientAddr;
			UInt32	clientPort;
			UInt32	itemsEnabled;			// flags set from rs_RTCP_HAS
		};
		
		
		//variable length RTCP packet format is as follows:
			
		//UInt32	bps;					// bits received per second
		//UInt16	lateMilSecs;			// ave. milliseconds packets received late
		//UInt16	bufferDelay;			// desired length of buffer on client (10ths/sec)
		//UInt16	bufferLength;			// actual filled length of buffer (10ths/sec)
		//UInt16	frameRate;				// frame rate (byte.byte fixed)
		//UInt16	expectedRate;			// expected frame rate (byte.byte fixed)
		//UInt16	dryNum;					// # times audio runs dry
		//UInt16	lossPercent;			// percent of packets dropped (byte.byte fixed)
		//UInt16	packetsDropped;			// how many packets received and dropped by client
		//UInt32	packetsLost;			// how many packets never received by client
		//UInt32	packetsReceived;		// how many packets received by client
		//UInt32	eyes;					// how many watchers
		//UInt32	eyesActive;				// how many watchers active
		//UInt32	eyesPaused;				// how many watchers paused
};

RTCPPacket::RTCPPacket(RTPStream* inStream)
: RemoteStatusPacket(rs_RTCP, kRTCPPacketSize)
{
	RTCPInfo* theInfo = (RTCPInfo*)fPacketDataPtr;
	theInfo->clientAddr = inStream->GetRemoteAddr();
	theInfo->clientPort = inStream->GetRemoteRTCPPort();
	theInfo->itemsEnabled = 0;
	
	//now begin to append information in the RTCP packet to the end of our packet
	UInt32 rtcpDataLen = 0;
	bool found = false;
	char* thePacketWriter = fPacketDataPtr + sizeof(RTCPInfo);
	StreamDictionary* theDictionary = inStream->GetDictionary();
	
	UInt32 uint32Val = theDictionary->GetStandardUInt32Value(StreamDictionary::kReceiverBitRate, &found);
	if (found)
	{
		*(UInt32*)(thePacketWriter + rtcpDataLen) = uint32Val;
		rtcpDataLen += sizeof(uint32Val);
		theInfo->itemsEnabled |= rs_HAS_BPS;
	}

	UInt16 uint16Val = theDictionary->GetStandardUInt16Value(StreamDictionary::kAvgLateMilliseconds, &found);
	if (found)
	{
		*(UInt16*)(thePacketWriter + rtcpDataLen) = uint16Val;
		rtcpDataLen += sizeof(uint16Val);
		theInfo->itemsEnabled |= rs_HAS_LATE_MS;
	}

	uint16Val = theDictionary->GetStandardUInt16Value(StreamDictionary::kAverageBufferDelayMilliseconds, &found);
	if (found)
	{
		*(UInt16*)(thePacketWriter + rtcpDataLen) = uint16Val;
		rtcpDataLen += sizeof(uint16Val);
		theInfo->itemsEnabled |= rs_HAS_BUFFER_DELAY;
	}

	uint16Val = theDictionary->GetStandardUInt16Value(StreamDictionary::kClientBufferFill, &found);
	if (found)
	{
		*(UInt16*)(thePacketWriter + rtcpDataLen) = uint16Val;
		rtcpDataLen += sizeof(uint16Val);
		theInfo->itemsEnabled |= rs_HAS_BUFFER_LENGTH;
	}

	uint16Val = theDictionary->GetStandardUInt16Value(StreamDictionary::kFrameRate, &found);
	if (found)
	{
		*(UInt16*)(thePacketWriter + rtcpDataLen) = uint16Val;
		rtcpDataLen += sizeof(uint16Val);
		theInfo->itemsEnabled |= rs_HAS_FRAME_RATE;
	}

	uint16Val = theDictionary->GetStandardUInt16Value(StreamDictionary::kExpectedFrameRate, &found);
	if (found)
	{
		*(UInt16*)(thePacketWriter + rtcpDataLen) = uint16Val;
		rtcpDataLen += sizeof(uint16Val);
		theInfo->itemsEnabled |= rs_HAS_EXPECTED_RATE;
	}

	uint16Val = theDictionary->GetStandardUInt16Value(StreamDictionary::kAudioDryCount, &found);
	if (found)
	{
		*(UInt16*)(thePacketWriter + rtcpDataLen) = uint16Val;
		rtcpDataLen += sizeof(uint16Val);
		theInfo->itemsEnabled |= rs_HAS_DRY_NUM;
	}

	uint16Val = theDictionary->GetStandardUInt16Value(StreamDictionary::kPercentPacketsLost, &found);
	if (found)
	{
		*(UInt16*)(thePacketWriter + rtcpDataLen) = uint16Val;
		rtcpDataLen += sizeof(uint16Val);
		theInfo->itemsEnabled |= rs_HAS_LOSS_PERCENT;
	}

	uint16Val = theDictionary->GetStandardUInt16Value(StreamDictionary::kTotalPacketsDropped, &found);
	if (found)
	{
		*(UInt16*)(thePacketWriter + rtcpDataLen) = uint16Val;
		rtcpDataLen += sizeof(uint16Val);
		theInfo->itemsEnabled |= rs_HAS_PACKETS_DROPPED;
	}

	uint32Val = theDictionary->GetStandardUInt32Value(StreamDictionary::kTotalLostPackets, &found);
	if (found)
	{
		*(UInt32*)(thePacketWriter + rtcpDataLen) = uint32Val;
		rtcpDataLen += sizeof(uint32Val);
		theInfo->itemsEnabled |= rs_HAS_PACKETS_LOST;
	}

	uint32Val = theDictionary->GetStandardUInt32Value(StreamDictionary::kTotalPacketsReceived, &found);
	if (found)
	{
		*(UInt32*)(thePacketWriter + rtcpDataLen) = uint32Val;
		rtcpDataLen += sizeof(uint32Val);
		theInfo->itemsEnabled |= rs_HAS_PACKETS_RECEIVED;
	}

	uint32Val = theDictionary->GetStandardUInt32Value(StreamDictionary::kNumEyes, &found);
	if (found)
	{
		*(UInt32*)(thePacketWriter + rtcpDataLen) = uint32Val;
		rtcpDataLen += sizeof(uint32Val);
		theInfo->itemsEnabled |= rs_HAS_EYES;
	}
	uint32Val = theDictionary->GetStandardUInt32Value(StreamDictionary::kNumEyesActive, &found);
	if (found)
	{
		*(UInt32*)(thePacketWriter + rtcpDataLen) = uint32Val;
		rtcpDataLen += sizeof(uint32Val);
		theInfo->itemsEnabled |= rs_HAS_EYES_ACTIVE;
	}
	uint32Val = theDictionary->GetStandardUInt32Value(StreamDictionary::kNumEyesPaused, &found);
	if (found)
	{
		*(UInt32*)(thePacketWriter + rtcpDataLen) = uint32Val;
		rtcpDataLen += sizeof(uint32Val);
		theInfo->itemsEnabled |= rs_HAS_EYES_PAUSED;
	}
	
	//ok, we've appended all the values to the buffer, now update the total length of the packet
	this->IncreaseLength(rtcpDataLen);
}

#pragma mark _SERVER_PACKET_

class ServerPacket : public RemoteStatusPacket
{
	public:

		ServerPacket();
		
	protected:

		static const UInt32 kServerPacketSize = 56;

		struct ServerPacketInfo
		{
			UInt16	refusingNew;			// server is refusing new connections
			UInt16	load;					// percent load on server (byte.byte fixed)
			UInt32	bps;					// bits per second being served
			UInt64	totalBytes;				// total # bytes served since start of server
			UInt32	totalSessions;			// total # sessions served since start of server
			UInt32	activeStreams;			// # of currently active stored movie streams
			UInt32	inactiveStreams;		// # of currently inactive stored movie streams
			UInt32	activeReflections;		// # of currently active reflected streams
			UInt32	inactiveReflections;	// # of currently inactive reflected streams
			UInt32	clients;				// # of currently active clients
		};
};

ServerPacket::ServerPacket() : RemoteStatusPacket(rs_SERVER_STATS, kServerPacketSize)
{
	ServerPacketInfo* theInfo = (ServerPacketInfo*)fPacketDataPtr;
	theInfo->refusingNew = htons(RTPServerInterface::GetServerState() == RTPServerInterface::kRefusingConnectionsState);
	theInfo->load = 0;
	theInfo->bps = htonl(RTPServerInterface::GetCurrentBandwidthInBits());
	theInfo->totalBytes = OS::HostToNetworkSInt64(RTPServerInterface::GetTotalBytes());
	theInfo->totalSessions = htonl(RTPServerInterface::GetTotalSessions());
	theInfo->activeStreams = htonl(RTPServerInterface::GetCurrentSessionCount());
	theInfo->inactiveStreams = 0;
	theInfo->activeReflections = 0;
	theInfo->inactiveReflections = 0;
	theInfo->clients = htonl(RTPServerInterface::GetCurrentSessionCount());
}

#pragma mark _RTSP_REQUEST_PACKET_

class RTSPRequestPacket : public RemoteStatusPacket
{
	public:

		RTSPRequestPacket(UInt32 inCommandType, UInt32 inClientIPAddr, UInt32 inSessionID, char* inAdditionalData, UInt32 inDataLen);
		
	protected:

		static const UInt32 kRTSPRequestPacketSize = 32;

		struct RequestPacketInfo
		{
			UInt32	clientIP;				// 4 byte IP address
			UInt32	id;						// session ID for commands
			SInt64	secondsSince1900;
			UInt16  additionalDataLen;
			UInt8	additionalData[1];		//data starts here		
		};
};

RTSPRequestPacket::RTSPRequestPacket(UInt32 inCommandType, UInt32 inClientIPAddr, UInt32 inSessionID, char* inAdditionalData, UInt32 inDataLen)
: RemoteStatusPacket(inCommandType, kRTSPRequestPacketSize)
{
	RequestPacketInfo* theInfo = (RequestPacketInfo*)fPacketDataPtr;
	theInfo->clientIP = htonl(inClientIPAddr);
	theInfo->id = htonl(inSessionID);
	theInfo->secondsSince1900 = OS::HostToNetworkSInt64(OS::ConvertToSecondsSince1900(OS::Milliseconds()));
	
	if (inDataLen > 0)
	{
		theInfo->additionalDataLen = htons(inDataLen);
		::memcpy(&theInfo->additionalData[0], inAdditionalData, inDataLen);
		this->IncreaseLength(sizeof(UInt16) + inDataLen);
	}
}

#pragma mark _RTSP_PLAY_PACKET_

class RTSPPlayPacket : public RTSPRequestPacket
{
	public:
		RTSPPlayPacket(RTPSession* inSession, UInt32 inClientIPAddr, UInt32 inSessionID);
};

RTSPPlayPacket::RTSPPlayPacket(RTPSession* inSession, UInt32 inClientIPAddr, UInt32 inSessionID)
: RTSPRequestPacket(rs_PLAY, inClientIPAddr, inSessionID, NULL, 0)
{
	RequestPacketInfo* theInfo = (RequestPacketInfo*)fPacketDataPtr;
	UInt32* thePortWriter = (UInt32*)&theInfo->additionalData[0];
	
	//loop through all the streams set up for this session, appending all the RTCP port numbers
	//We have a fixed length max packet size, and of course this packet is variable in length,
	//so we need to make sure we aren't overrunning the buffer...
	//FIXME! Not done yet
	for (OSQueueIter theIter(inSession->GetStreamQueue()); !theIter.IsDone(); theIter.Next())
	{
		UInt32 thePort = (UInt32)((RTPStream*)theIter.GetCurrent()->GetEnclosingObject())->GetRemoteRTCPPort();
		*thePortWriter = htonl(thePort);
		thePortWriter++;
	}

	//write in the length of the additional data, and up the total packet length
	UInt16 portBufferLen = inSession->GetStreamQueue()->GetLength() * sizeof(UInt32);
	theInfo->additionalDataLen = htons(portBufferLen);
	this->IncreaseLength(sizeof(UInt16) + portBufferLen);
}

#pragma mark _REMOTE_STATUS_SOCKET_

RemoteStatusSocket::RemoteStatusSocket() : TCPSocket(kNonBlocking | kWantsEvents, NULL)
{
	RTSPPrefs* thePrefs = RTSPServerInterface::GetRTSPPrefs();
	if (thePrefs->GetRemoteStatsAddr() != 0)
	{
		this->Open();
		fState |= kConnecting;
		(void)this->Connect(thePrefs->GetRemoteStatsAddr(), thePrefs->GetRemoteStatsPort());
	}
}

void RemoteStatusSocket::ProcessEvent(Task::EventFlags theEvent)
{
	//if the socket is connecting, and it gets both a read event and write event, this means
	//there was some connection error that occurred (Stevens, pp 409)
	if (fState & kConnecting)
	{
		//The order of how these things get turned off and on is IMPORTANT, because this activity
		//is happening on a different thread from the rest of the remote status module.

		//A write event by itself means that the socket is now connected sucessfully.
		//A read event AND a write event means that the socket encountered a connection error.
		if (theEvent == Task::kWriteEvent)
			fState |= kConnected;

		//make sure NOT to wait for a write event at this point. We only 
		//do that when first connecting
		this->Modwatch(EV_RE);

		fState ^= kConnecting;
	}
}


#pragma mark _REMOTE_STATUS_MODULE_

RTCPRemoteStatusModule::RTCPRemoteStatusModule() :
	RTPModule(kAllRTCPProcessingModule | kPostProcessingModule | kRTSPSessionClosingModule, "RTCPRemoteStatusModule"),
	fPacketWriter(fPacketBuffer, kPacketBufferSizeInBytes), 
	fLastServerStatsTime(0),
	fMutex('rsmt'),
	fSocket(NULL),
	fConnectTime(0)
{
	//Attempt to connect immediately
#if REMOTE_STATUS_SOCKET_DEBUGGING
	printf("Attempting to connect. Constructing rssocket\n");
#endif
	fConnectTime = OS::Milliseconds();
	fSocket = new RemoteStatusSocket();
}

bool RTCPRemoteStatusModule::CheckConnection()
{
	//This function checks to see if the socket is connected. If it is, it returns true. If not,
	//it may attempt to initiate the connection.
	SInt64 theCurrentTime = OS::Milliseconds();
	
	//If we aren't connected, and it is time to connect, then attempt to connect
	if ((fSocket == NULL) && ((fConnectTime == 0) ||
		((theCurrentTime - fConnectTime) > kConnectRetryIntervalInMilSecs)))
	{
#if REMOTE_STATUS_SOCKET_DEBUGGING
		printf("Attempting to connect. Constructing rssocket\n");
#endif
		fSocket = new RemoteStatusSocket();
		fConnectTime = theCurrentTime;//timestamp this connection attempt
	}
	if (fSocket == NULL)
		return false;
	
	//The order of these checks is very important!
	
	if (fSocket->IsConnected())
	{
		//If we are connected currently, that's great. If this is our first pass after
		//becoming connected, we should send the startup messages to the server now.
	
		if (fConnectTime != 0)
		{
			//We use the fConnectTime to flag the first pass after becoming connected.
			StartupPacket theStartupPacket;
			this->AddPacketToBuffer(&theStartupPacket);
			this->FlushBuffer();
			fConnectTime = 0;
		}
		//We are connected
		return true;
	}
	
	//If we are connecting currently, check to see if we should time out this connection attempt
	if (fSocket->IsConnecting())
	{
		Assert(fConnectTime != 0);
		Assert(fConnectTime <= theCurrentTime);
		if ((theCurrentTime - fConnectTime) > kMaxConnectTimeInMilSecs)
		{
#if REMOTE_STATUS_SOCKET_DEBUGGING
			printf("Connection attempt timed out\n");
#endif
			delete fSocket;
			fSocket = NULL;
		}
	}
	//We need to check this again, becase after checking for IsConnecting, the other thread could have
	//come and set connecting off and isConnected on!
	else if (!fSocket->IsConnected())
	{
#if REMOTE_STATUS_SOCKET_DEBUGGING
		printf("Connection attempt failed\n");
#endif
		//We aren't connecting, or connected, so the connection must be dead.
		delete fSocket;
		fSocket = NULL;
	}
	//We aren't connected
	return false;
}

void RTCPRemoteStatusModule::PostProcessRTSPRequest(RTSPRequestInterface* inRequest, RTPSession* inSession)
{
	OSMutexLocker locker(&fMutex);
	
	if (!this->CheckConnection())
		return;
		
#if REMOTE_STATUS_SOCKET_DEBUGGING
	printf("Doing RTCPRemoteStatusModule::PostProcessRTSPRequest\n");
#endif
	this->CheckAndSendServerStats();
	
	UInt32 theClientIPAddr = inRequest->GetSession()->GetSocket()->GetRemoteAddr();
	UInt32 theClientSessionID = inRequest->GetSession()->GetSessionID();
	
	if (inRequest->GetStatus() == RTSPProtocol::kSuccessOK)
	{
		switch (inRequest->GetMethod())
		{
			case RTSPProtocol::kDescribeMethod:
			{
				StrPtrLen* url = inRequest->GetQTSSParameter(qtssURLParam);
				RTSPRequestPacket theDescribePacket(rs_DESCRIBE, theClientIPAddr, theClientSessionID, url->Ptr, url->Len);
				this->AddPacketToBuffer(&theDescribePacket);
			}
			break;
			
			case RTSPProtocol::kPlayMethod:
			{
				RTSPPlayPacket thePlayPacket(inSession, theClientIPAddr, theClientSessionID);
				this->AddPacketToBuffer(&thePlayPacket);
			}
			break;

			case RTSPProtocol::kPauseMethod:
			{
				RTSPRequestPacket thePausePacket(rs_PAUSE, theClientIPAddr, theClientSessionID, NULL, 0);
				this->AddPacketToBuffer(&thePausePacket);			
			}
			break;

			case RTSPProtocol::kTeardownMethod:
			{
				RTSPRequestPacket theTeardownPacket(rs_TEARDOWN, theClientIPAddr, theClientSessionID, NULL, 0);
				this->AddPacketToBuffer(&theTeardownPacket);						
			}
			break;
		}
	}
}

void RTCPRemoteStatusModule::ProcessRTSPSessionClosing(RTSPSessionInterface* inRTSPSession)
{
	OSMutexLocker locker(&fMutex);

	if (!this->CheckConnection())
		return;

#if REMOTE_STATUS_SOCKET_DEBUGGING
	printf("Doing RTCPRemoteStatusModule::ProcessRTSPSessionClosing\n");
#endif
	this->CheckAndSendServerStats();

	RTSPRequestPacket theSessionClosingPacket(rs_CLOSE, inRTSPSession->GetSocket()->GetRemoteAddr(),
																inRTSPSession->GetSessionID(), NULL, 0);
	
	this->AddPacketToBuffer(&theSessionClosingPacket);						
}

void RTCPRemoteStatusModule::ProcessRTCPPacket(RTPStream* inStream, StrPtrLen*/*inPacket*/)
{
	OSMutexLocker locker(&fMutex);

	if (!this->CheckConnection())
		return;

#if REMOTE_STATUS_SOCKET_DEBUGGING
	printf("Doing RTCPRemoteStatusModule::ProcessRTCPPacket\n");
#endif
	this->CheckAndSendServerStats();
	
	RTCPPacket thePacket(inStream);
	this->AddPacketToBuffer(&thePacket);
}

void RTCPRemoteStatusModule::CheckAndSendServerStats()
{
	SInt64 theCurrentTime = OS::Milliseconds();
	
	//Only send the server stats packet every once in awhile
	if ((fLastServerStatsTime + kServerStatsIntervalInMilSecs) < theCurrentTime)
	{
		ServerPacket theServerStatsPacket;
		this->AddPacketToBuffer(&theServerStatsPacket);
		fLastServerStatsTime = theCurrentTime;
	}
}

void RTCPRemoteStatusModule::AddPacketToBuffer(RemoteStatusPacket* inPacket)
{
	inPacket->WritePacket(&fPacketWriter);

	if ((UInt32)fPacketWriter.GetCurrentOffset() > kSendAmountInBytes)
		this->FlushBuffer();
}

void RTCPRemoteStatusModule::FlushBuffer()
{
	UInt32 theLengthSent = 0;
	QTSS_ErrorCode theErr = fSocket->Send(fPacketBuffer, fPacketWriter.GetCurrentOffset(), &theLengthSent);
	if (theErr == QTSS_NoErr)
		fPacketWriter.Reset(theLengthSent);
}